"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PROPFINDRequestHandler = void 0;
const xml_utils_1 = require("../../utils/xml.utils");
const format_utils_1 = require("../../utils/format.utils");
const node_crypto_1 = require("node:crypto");
const mime_types_1 = __importDefault(require("mime-types"));
const webdav_utils_1 = require("../../utils/webdav.utils");
const logger_utils_1 = require("../../utils/logger.utils");
const usage_service_1 = require("../../services/usage.service");
class PROPFINDRequestHandler {
    dependencies;
    constructor(dependencies) {
        this.dependencies = dependencies;
    }
    handle = async (req, res) => {
        const { driveFolderService, driveFileService } = this.dependencies;
        const resource = await webdav_utils_1.WebDavUtils.getRequestedResource(req.url);
        logger_utils_1.webdavLogger.info(`[PROPFIND] Request received for item at ${resource.url}`);
        const driveItem = await webdav_utils_1.WebDavUtils.getDriveItemFromResource({
            resource,
            driveFolderService,
            driveFileService,
        });
        if (!driveItem) {
            res.status(404).send();
            return;
        }
        switch (driveItem.itemType) {
            case 'file': {
                const fileMetaXML = await this.getFileMetaXML(resource, driveItem);
                res.status(207).send(fileMetaXML);
                break;
            }
            case 'folder': {
                const depth = req.header('depth') ?? '1';
                const folderMetaXML = await this.getFolderContentXML(resource, driveItem, depth);
                res.status(207).send(folderMetaXML);
                break;
            }
        }
    };
    getFileMetaXML = async (resource, driveFileItem) => {
        const driveFile = this.driveFileItemToXMLNode(driveFileItem, resource.url);
        const xml = xml_utils_1.XMLUtils.toWebDavXML([driveFile], {
            arrayNodeName: xml_utils_1.XMLUtils.addDefaultNamespace('response'),
        });
        return xml;
    };
    getFolderContentXML = async (resource, folderItem, depth) => {
        const relativePath = resource.url;
        const isRootFolder = resource.url === '/';
        let XMLNodes = [];
        switch (depth) {
            case '0':
                if (isRootFolder) {
                    XMLNodes.push(await this.driveFolderRootStatsToXMLNode(folderItem, relativePath));
                }
                else {
                    XMLNodes.push(this.driveFolderItemToXMLNode(folderItem, relativePath));
                }
                break;
            case '1':
            default:
                if (isRootFolder) {
                    XMLNodes.push(await this.driveFolderRootStatsToXMLNode(folderItem, relativePath));
                    XMLNodes = XMLNodes.concat(await this.getFolderChildsXMLNode(relativePath, folderItem.uuid));
                }
                else {
                    XMLNodes.push(this.driveFolderItemToXMLNode(folderItem, relativePath));
                    XMLNodes = XMLNodes.concat(await this.getFolderChildsXMLNode(relativePath, folderItem.uuid));
                }
                break;
        }
        const xml = xml_utils_1.XMLUtils.toWebDavXML(XMLNodes, {
            arrayNodeName: xml_utils_1.XMLUtils.addDefaultNamespace('response'),
            ignoreAttributes: false,
            suppressEmptyNode: true,
        });
        return xml;
    };
    getFolderChildsXMLNode = async (relativePath, folderUuid) => {
        const { driveFolderService } = this.dependencies;
        const folderContent = await driveFolderService.getFolderContent(folderUuid);
        const foldersXML = folderContent.folders.map((folder) => {
            const folderRelativePath = webdav_utils_1.WebDavUtils.joinURL(relativePath, folder.plainName, '/');
            return this.driveFolderItemToXMLNode({
                itemType: 'folder',
                name: folder.plainName,
                bucket: folder.bucket,
                status: folder.deleted || folder.removed ? 'TRASHED' : 'EXISTS',
                createdAt: new Date(folder.createdAt),
                updatedAt: new Date(folder.updatedAt),
                id: folder.id,
                encryptedName: folder.name,
                uuid: folder.uuid,
                parentId: null,
                parentUuid: null,
            }, folderRelativePath);
        });
        const filesXML = folderContent.files.map((file) => {
            const fileRelativePath = webdav_utils_1.WebDavUtils.joinURL(relativePath, file.type ? `${file.plainName}.${file.type}` : file.plainName);
            return this.driveFileItemToXMLNode({
                itemType: 'file',
                name: file.plainName,
                bucket: file.bucket,
                id: file.id,
                fileId: file.fileId,
                uuid: file.uuid,
                type: file.type,
                status: file.status,
                folderId: file.folderId,
                folderUuid: file.folderUuid,
                size: Number(file.size),
                creationTime: new Date(file.creationTime),
                modificationTime: new Date(file.modificationTime),
                createdAt: new Date(file.createdAt),
                updatedAt: new Date(file.updatedAt),
            }, fileRelativePath);
        });
        return foldersXML.concat(filesXML);
    };
    driveFolderRootStatsToXMLNode = async (driveFolderItem, relativePath) => {
        const totalUsage = (await usage_service_1.UsageService.instance.fetchUsage()).total;
        const spaceLimit = await usage_service_1.UsageService.instance.fetchSpaceLimit();
        const driveFolderXML = {
            [xml_utils_1.XMLUtils.addDefaultNamespace('href')]: xml_utils_1.XMLUtils.encodeWebDavUri(relativePath),
            [xml_utils_1.XMLUtils.addDefaultNamespace('propstat')]: {
                [xml_utils_1.XMLUtils.addDefaultNamespace('status')]: 'HTTP/1.1 200 OK',
                [xml_utils_1.XMLUtils.addDefaultNamespace('prop')]: {
                    [xml_utils_1.XMLUtils.addDefaultNamespace('getcontenttype')]: 'application/octet-stream',
                    'x1:lastmodified': {
                        '#text': format_utils_1.FormatUtils.formatDateForWebDav(driveFolderItem.updatedAt),
                        '@_xmlns:x1': 'SAR:',
                    },
                    'x2:executable': {
                        '#text': 'F',
                        '@_xmlns:x2': 'http://apache.org/dav/props/',
                    },
                    'x3:Win32FileAttributes': {
                        '#text': '00000030',
                        '@_xmlns:x3': 'urn:schemas-microsoft-com:',
                    },
                    [xml_utils_1.XMLUtils.addDefaultNamespace('quota-available-bytes')]: spaceLimit - totalUsage,
                    [xml_utils_1.XMLUtils.addDefaultNamespace('quota-used-bytes')]: totalUsage,
                    [xml_utils_1.XMLUtils.addDefaultNamespace('resourcetype')]: {
                        [xml_utils_1.XMLUtils.addDefaultNamespace('collection')]: '',
                    },
                },
            },
        };
        return driveFolderXML;
    };
    driveFolderItemToXMLNode = (driveFolderItem, relativePath) => {
        const displayName = `${driveFolderItem.name}`;
        const driveFolderXML = {
            [xml_utils_1.XMLUtils.addDefaultNamespace('href')]: xml_utils_1.XMLUtils.encodeWebDavUri(relativePath),
            [xml_utils_1.XMLUtils.addDefaultNamespace('propstat')]: {
                [xml_utils_1.XMLUtils.addDefaultNamespace('status')]: 'HTTP/1.1 200 OK',
                [xml_utils_1.XMLUtils.addDefaultNamespace('prop')]: {
                    [xml_utils_1.XMLUtils.addDefaultNamespace('displayname')]: displayName,
                    [xml_utils_1.XMLUtils.addDefaultNamespace('getlastmodified')]: format_utils_1.FormatUtils.formatDateForWebDav(driveFolderItem.updatedAt),
                    [xml_utils_1.XMLUtils.addDefaultNamespace('getcontentlength')]: 0,
                    [xml_utils_1.XMLUtils.addDefaultNamespace('resourcetype')]: {
                        [xml_utils_1.XMLUtils.addDefaultNamespace('collection')]: '',
                    },
                },
            },
        };
        return driveFolderXML;
    };
    driveFileItemToXMLNode = (driveFileItem, relativePath) => {
        const displayName = driveFileItem.type ? `${driveFileItem.name}.${driveFileItem.type}` : driveFileItem.name;
        const lastModified = format_utils_1.FormatUtils.formatDateForWebDav(driveFileItem.modificationTime ?? driveFileItem.updatedAt);
        const driveFileXML = {
            [xml_utils_1.XMLUtils.addDefaultNamespace('href')]: xml_utils_1.XMLUtils.encodeWebDavUri(relativePath),
            [xml_utils_1.XMLUtils.addDefaultNamespace('propstat')]: {
                [xml_utils_1.XMLUtils.addDefaultNamespace('status')]: 'HTTP/1.1 200 OK',
                [xml_utils_1.XMLUtils.addDefaultNamespace('prop')]: {
                    [xml_utils_1.XMLUtils.addDefaultNamespace('resourcetype')]: '',
                    [xml_utils_1.XMLUtils.addDefaultNamespace('getetag')]: '"' + (0, node_crypto_1.randomUUID)().replaceAll('-', '') + '"',
                    [xml_utils_1.XMLUtils.addDefaultNamespace('displayname')]: displayName,
                    [xml_utils_1.XMLUtils.addDefaultNamespace('getcontenttype')]: mime_types_1.default.lookup(displayName) || 'application/octet-stream',
                    [xml_utils_1.XMLUtils.addDefaultNamespace('getlastmodified')]: lastModified,
                    [xml_utils_1.XMLUtils.addDefaultNamespace('getcontentlength')]: driveFileItem.size,
                },
            },
        };
        return driveFileXML;
    };
}
exports.PROPFINDRequestHandler = PROPFINDRequestHandler;
