"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MOVERequestHandler = void 0;
const errors_utils_1 = require("../../utils/errors.utils");
const logger_utils_1 = require("../../utils/logger.utils");
const webdav_utils_1 = require("../../utils/webdav.utils");
class MOVERequestHandler {
    dependencies;
    constructor(dependencies) {
        this.dependencies = dependencies;
    }
    handle = async (req, res) => {
        const { driveFolderService, driveFileService, webDavFolderService } = this.dependencies;
        const resource = await webdav_utils_1.WebDavUtils.getRequestedResource(req.url);
        logger_utils_1.webdavLogger.info(`[MOVE] Request received for item at ${resource.url}`);
        const destinationUrl = req.header('destination');
        if (!destinationUrl) {
            throw new errors_utils_1.NotFoundError('[MOVE] Destination folder not received');
        }
        const destinationPath = webdav_utils_1.WebDavUtils.removeHostFromURL(destinationUrl);
        const destinationResource = await webdav_utils_1.WebDavUtils.getRequestedResource(destinationPath);
        logger_utils_1.webdavLogger.info('[MOVE] Destination resource found', { destinationResource });
        const originalDriveItem = await webdav_utils_1.WebDavUtils.getDriveItemFromResource({
            resource,
            driveFolderService,
            driveFileService,
        });
        if (!originalDriveItem) {
            throw new errors_utils_1.NotFoundError(`Resource not found on Internxt Drive at ${resource.url}`);
        }
        if (destinationResource.path.dir === resource.path.dir) {
            logger_utils_1.webdavLogger.info(`[MOVE] Renaming ${originalDriveItem.itemType}
        with UUID ${originalDriveItem.uuid} to ${destinationResource.name}`);
            if (originalDriveItem.itemType === 'folder') {
                const folder = originalDriveItem;
                await driveFolderService.renameFolder({
                    folderUuid: folder.uuid,
                    name: destinationResource.name,
                });
            }
            else if (originalDriveItem.itemType === 'file') {
                const file = originalDriveItem;
                const plainName = destinationResource.path.name;
                const fileType = destinationResource.path.ext.replace('.', '');
                await driveFileService.renameFile(file.uuid, {
                    plainName: plainName,
                    type: fileType,
                });
            }
        }
        else {
            logger_utils_1.webdavLogger.info(`[MOVE] Moving ${originalDriveItem.itemType} with UUID ${originalDriveItem.uuid} to ${destinationPath}`);
            const destinationFolderItem = (await webDavFolderService.getDriveFolderItemFromPath(destinationResource.parentPath)) ??
                (await webDavFolderService.createParentPathOrThrow(destinationResource.parentPath));
            if (originalDriveItem.itemType === 'folder') {
                const folder = originalDriveItem;
                await driveFolderService.moveFolder(folder.uuid, {
                    destinationFolder: destinationFolderItem.uuid,
                    name: destinationResource.name,
                });
            }
            else if (originalDriveItem.itemType === 'file') {
                const file = originalDriveItem;
                const plainName = destinationResource.path.name;
                const fileType = destinationResource.path.ext.replace('.', '');
                await driveFileService.moveFile(file.uuid, {
                    destinationFolder: destinationFolderItem.uuid,
                    name: plainName,
                    type: fileType,
                });
            }
        }
        res.status(204).send();
    };
}
exports.MOVERequestHandler = MOVERequestHandler;
