"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GETRequestHandler = void 0;
const webdav_utils_1 = require("../../utils/webdav.utils");
const errors_utils_1 = require("../../utils/errors.utils");
const logger_utils_1 = require("../../utils/logger.utils");
const network_utils_1 = require("../../utils/network.utils");
class GETRequestHandler {
    dependencies;
    constructor(dependencies) {
        this.dependencies = dependencies;
    }
    handle = async (req, res) => {
        const { driveFileService, authService, networkFacade } = this.dependencies;
        const resource = await webdav_utils_1.WebDavUtils.getRequestedResource(req.url);
        if (resource.name.startsWith('._'))
            throw new errors_utils_1.NotFoundError('File not found');
        logger_utils_1.webdavLogger.info(`[GET] Request received item at ${resource.url}`);
        const driveFile = await webdav_utils_1.WebDavUtils.getDriveFileFromResource({
            url: resource.url,
            driveFileService,
        });
        if (!driveFile) {
            throw new errors_utils_1.NotFoundError(`Resource not found on Internxt Drive at ${resource.url}, if trying to access a folder use PROPFIND instead.`);
        }
        logger_utils_1.webdavLogger.info(`[GET] [${driveFile.uuid}] Found Drive File`);
        const { user } = await authService.getAuthDetails();
        logger_utils_1.webdavLogger.info(`[GET] [${driveFile.uuid}] Network ready for download`);
        const range = req.headers['range'];
        const rangeOptions = network_utils_1.NetworkUtils.parseRangeHeader({
            range,
            totalFileSize: driveFile.size,
        });
        let contentLength = driveFile.size;
        if (rangeOptions) {
            logger_utils_1.webdavLogger.info(`[GET] [${driveFile.uuid}] Range request received:`, { rangeOptions });
            contentLength = rangeOptions.rangeSize;
        }
        res.header('Content-Type', 'application/octet-stream');
        res.header('Content-length', contentLength.toString());
        const writable = new WritableStream({
            write(chunk) {
                res.write(chunk);
            },
            close() {
                res.end();
            },
        });
        const [executeDownload] = await networkFacade.downloadToStream(driveFile.bucket, user.mnemonic, driveFile.fileId, contentLength, writable, rangeOptions);
        logger_utils_1.webdavLogger.info(`[GET] [${driveFile.uuid}] Download prepared, executing...`);
        res.status(200);
        await executeDownload;
        logger_utils_1.webdavLogger.info(`[GET] [${driveFile.uuid}] ✅ Download ready, replying to client`);
    };
}
exports.GETRequestHandler = GETRequestHandler;
