"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebDavUtils = void 0;
const node_path_1 = __importDefault(require("node:path"));
const logger_utils_1 = require("./logger.utils");
class WebDavUtils {
    static joinURL(...pathComponents) {
        return node_path_1.default.posix.join(...pathComponents);
    }
    static removeHostFromURL(completeURL) {
        if (!completeURL.startsWith('/') && !/^https?:\/\//i.test(completeURL)) {
            completeURL = 'https://' + completeURL;
        }
        const parsedUrl = new URL(completeURL);
        let url = parsedUrl.href.replace(parsedUrl.origin + '/', '');
        if (!url.startsWith('/'))
            url = '/'.concat(url);
        return url;
    }
    static decodeUrl(requestUrl, decodeUri = true) {
        return (decodeUri ? decodeURIComponent(requestUrl) : requestUrl).replaceAll('/./', '/');
    }
    static normalizeFolderPath(path) {
        let normalizedPath = path;
        if (!normalizedPath.startsWith('/')) {
            normalizedPath = `/${normalizedPath}`;
        }
        if (!normalizedPath.endsWith('/')) {
            normalizedPath = `${normalizedPath}/`;
        }
        return normalizedPath;
    }
    static async getRequestedResource(requestUrl, decodeUri = true) {
        const decodedUrl = this.decodeUrl(requestUrl, decodeUri);
        const parsedPath = node_path_1.default.parse(decodedUrl);
        const parentPath = this.normalizeFolderPath(node_path_1.default.dirname(decodedUrl));
        return {
            url: decodedUrl,
            name: parsedPath.base,
            path: parsedPath,
            parentPath,
        };
    }
    static async tryGetFileOrFolderMetadata({ url, driveFileService, driveFolderService, }) {
        try {
            return await driveFileService.getFileMetadataByPath(url);
        }
        catch {
            return await driveFolderService.getFolderMetadataByPath(url);
        }
    }
    static async getDriveFileFromResource({ url, driveFileService, }) {
        try {
            return await driveFileService.getFileMetadataByPath(url);
        }
        catch (err) {
            logger_utils_1.webdavLogger.error('Exception while getting the file metadata by path', err);
        }
    }
    static async getDriveFolderFromResource({ url, driveFolderService, }) {
        try {
            return await driveFolderService.getFolderMetadataByPath(url);
        }
        catch (err) {
            logger_utils_1.webdavLogger.error('Exception while getting the folder metadata by path', err);
        }
    }
    static async getDriveItemFromResource({ resource, driveFolderService, driveFileService, }) {
        let item = undefined;
        const isFolder = resource.url.endsWith('/');
        try {
            if (isFolder) {
                item = await driveFolderService.getFolderMetadataByPath(resource.url);
            }
            else {
                item = await this.tryGetFileOrFolderMetadata({
                    url: resource.url,
                    driveFileService,
                    driveFolderService,
                });
            }
        }
        catch {
        }
        return item;
    }
}
exports.WebDavUtils = WebDavUtils;
