"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UniversalLinkService = void 0;
const node_http_1 = __importDefault(require("node:http"));
const open_1 = __importDefault(require("open"));
const config_service_1 = require("./config.service");
const auth_service_1 = require("./auth.service");
const cli_utils_1 = require("../utils/cli.utils");
class UniversalLinkService {
    static instance = new UniversalLinkService();
    getUserCredentials = async (userSession) => {
        const clearMnemonic = Buffer.from(userSession.mnemonic, 'base64').toString('utf-8');
        const clearToken = Buffer.from(userSession.token, 'base64').toString('utf-8');
        const loginCredentials = await auth_service_1.AuthService.instance.refreshUserToken(clearToken, clearMnemonic);
        return {
            user: {
                ...loginCredentials.user,
                mnemonic: clearMnemonic,
            },
            token: clearToken,
        };
    };
    buildLoginUrl = (redirectUri) => {
        const loginURL = `${config_service_1.ConfigService.instance.get('DRIVE_WEB_URL')}/login`;
        const params = new URLSearchParams({
            universalLink: 'true',
            redirectUri: redirectUri,
        });
        return `${loginURL}?${params.toString()}`;
    };
    loginSSO = async (jsonFlag, reporter, hostIp = '127.0.0.1', forcedPort = 0) => {
        return new Promise((resolve, reject) => {
            const server = node_http_1.default.createServer(async (req, res) => {
                if (!req.url)
                    return;
                const parsedUrl = new URL(req.url, `http://${req.headers.host}`);
                const driveUrl = config_service_1.ConfigService.instance.get('DRIVE_WEB_URL');
                try {
                    const mnemonic = parsedUrl.searchParams.get('mnemonic');
                    const token = parsedUrl.searchParams.get('newToken');
                    if (!mnemonic || !token) {
                        throw new Error('Login has failed, please try again');
                    }
                    const loginCredentials = await this.getUserCredentials({ mnemonic, token });
                    res.writeHead(302, {
                        Location: `${driveUrl}/auth-link-ok`,
                    });
                    res.end();
                    resolve(loginCredentials);
                }
                catch (error) {
                    res.writeHead(302, {
                        Location: `${driveUrl}/auth-link-error`,
                    });
                    res.end();
                    reject(error);
                }
                finally {
                    server.closeAllConnections();
                    server.close();
                }
            });
            server.listen(forcedPort, async () => {
                const { port } = server.address();
                const redirectUri = Buffer.from(`http://${hostIp}:${port}/callback`).toString('base64');
                const loginUrl = this.buildLoginUrl(redirectUri);
                cli_utils_1.CLIUtils.log(reporter, 'Opening browser for login...');
                cli_utils_1.CLIUtils.log(reporter, 'If the browser doesn’t open automatically, visit:');
                const printLoginUrl = jsonFlag ? `{ "loginUrl": "${loginUrl}" }` : loginUrl;
                cli_utils_1.CLIUtils.consoleLog(printLoginUrl);
                try {
                    await (0, open_1.default)(loginUrl);
                }
                catch {
                    cli_utils_1.CLIUtils.warning(reporter, 'Could not open browser automatically.');
                }
                cli_utils_1.CLIUtils.log(reporter, 'Waiting for authentication...');
            });
        });
    };
}
exports.UniversalLinkService = UniversalLinkService;
