"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CryptoService = void 0;
const node_crypto_1 = require("node:crypto");
const keys_service_1 = require("./keys.service");
const config_service_1 = require("../services/config.service");
const stream_utils_1 = require("../utils/stream.utils");
class CryptoService {
    static instance = new CryptoService();
    static cryptoProvider = {
        encryptPasswordHash(password, encryptedSalt) {
            const salt = CryptoService.instance.decryptText(encryptedSalt);
            const hashObj = CryptoService.instance.passToHash({ password, salt });
            return CryptoService.instance.encryptText(hashObj.hash);
        },
        async generateKeys(password) {
            const { privateKeyArmoredEncrypted, publicKeyArmored, revocationCertificate } = await keys_service_1.KeysService.instance.generateNewKeysWithEncrypted(password);
            const keys = {
                privateKeyEncrypted: privateKeyArmoredEncrypted,
                publicKey: publicKeyArmored,
                revocationCertificate: revocationCertificate,
                ecc: {
                    privateKeyEncrypted: privateKeyArmoredEncrypted,
                    publicKey: publicKeyArmored,
                },
                kyber: {
                    privateKeyEncrypted: null,
                    publicKey: null,
                },
            };
            return keys;
        },
    };
    passToHash = (passObject) => {
        const salt = passObject.salt ?? (0, node_crypto_1.randomBytes)(128 / 8).toString('hex');
        const hash = (0, node_crypto_1.pbkdf2Sync)(passObject.password, Buffer.from(salt, 'hex'), 10000, 256 / 8, 'sha1').toString('hex');
        const hashedObjetc = {
            salt,
            hash,
        };
        return hashedObjetc;
    };
    encryptText = (textToEncrypt) => {
        const APP_CRYPTO_SECRET = config_service_1.ConfigService.instance.get('APP_CRYPTO_SECRET');
        return this.encryptTextWithKey(textToEncrypt, APP_CRYPTO_SECRET);
    };
    decryptText = (encryptedText) => {
        const APP_CRYPTO_SECRET = config_service_1.ConfigService.instance.get('APP_CRYPTO_SECRET');
        return this.decryptTextWithKey(encryptedText, APP_CRYPTO_SECRET);
    };
    encryptTextWithKey = (textToEncrypt, secret) => {
        const salt = (0, node_crypto_1.randomBytes)(8);
        const { key, iv } = this.getKeyAndIvFrom(secret, salt);
        const cipher = (0, node_crypto_1.createCipheriv)('aes-256-cbc', key, iv);
        const encrypted = Buffer.concat([cipher.update(textToEncrypt, 'utf8'), cipher.final()]);
        const openSSLstart = Buffer.from('Salted__');
        return Buffer.concat([openSSLstart, salt, encrypted]).toString('hex');
    };
    decryptTextWithKey = (encryptedText, secret) => {
        const cypherText = Buffer.from(encryptedText, 'hex');
        const salt = cypherText.subarray(8, 16);
        const { key, iv } = this.getKeyAndIvFrom(secret, salt);
        const decipher = (0, node_crypto_1.createDecipheriv)('aes-256-cbc', key, iv);
        const contentsToDecrypt = cypherText.subarray(16);
        return Buffer.concat([decipher.update(contentsToDecrypt), decipher.final()]).toString('utf8');
    };
    decryptStream = (inputSlices, key, iv, startOffsetByte) => {
        let decipher;
        if (startOffsetByte) {
            const aesBlockSize = 16;
            const startOffset = startOffsetByte % aesBlockSize;
            const startBlockFirstByte = startOffsetByte - startOffset;
            const startBlockNumber = startBlockFirstByte / aesBlockSize;
            const ivForRange = (BigInt('0x' + iv.toString('hex')) + BigInt(startBlockNumber)).toString(16).padStart(32, '0');
            const newIv = Buffer.from(ivForRange, 'hex');
            const skipBuffer = Buffer.alloc(startOffset, 0);
            decipher = (0, node_crypto_1.createDecipheriv)('aes-256-ctr', key, newIv);
            decipher.update(skipBuffer);
        }
        else {
            decipher = (0, node_crypto_1.createDecipheriv)('aes-256-ctr', key, iv);
        }
        const encryptedStream = stream_utils_1.StreamUtils.joinReadableBinaryStreams(inputSlices);
        let keepReading = true;
        const decryptedStream = new ReadableStream({
            async pull(controller) {
                if (!keepReading)
                    return;
                const reader = encryptedStream.getReader();
                const status = await reader.read();
                if (status.done) {
                    controller.close();
                }
                else {
                    controller.enqueue(decipher.update(status.value));
                }
                reader.releaseLock();
            },
            cancel() {
                keepReading = false;
            },
        });
        return decryptedStream;
    };
    getKeyAndIvFrom = (secret, salt) => {
        const TRANSFORM_ROUNDS = 3;
        const password = Buffer.concat([Buffer.from(secret, 'utf8'), salt]);
        const md5Hashes = [];
        let digest = password;
        for (let i = 0; i < TRANSFORM_ROUNDS; i++) {
            md5Hashes[i] = (0, node_crypto_1.createHash)('md5').update(digest).digest();
            digest = Buffer.concat([md5Hashes[i], password]);
        }
        const key = Buffer.concat([md5Hashes[0], md5Hashes[1]]);
        const iv = md5Hashes[2];
        return { key, iv };
    };
}
exports.CryptoService = CryptoService;
