"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
class WebDAVConfig extends core_1.Command {
    static args = {};
    static description = 'Edit the configuration of the Internxt CLI WebDav server as the port or the protocol.';
    static aliases = [];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        host: core_1.Flags.string({
            char: 'l',
            description: 'The listening host for the WebDAV server.',
            required: false,
        }),
        port: core_1.Flags.string({
            char: 'p',
            description: 'The new port for the WebDAV server.',
            required: false,
        }),
        https: core_1.Flags.boolean({
            char: 's',
            description: 'Configures the WebDAV server to use HTTPS with self-signed certificates.',
            required: false,
            exclusive: ['http'],
        }),
        http: core_1.Flags.boolean({
            char: 'h',
            description: 'Configures the WebDAV server to use insecure plain HTTP.',
            required: false,
            exclusive: ['https'],
        }),
        timeout: core_1.Flags.integer({
            char: 't',
            description: 'Configures the WebDAV server to use this timeout in minutes.',
            required: false,
            min: 0,
        }),
        createFullPath: core_1.Flags.boolean({
            char: 'c',
            description: 'Auto-create missing parent directories during file uploads.',
            required: false,
            allowNo: true,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags: { host, port, http, https, timeout, createFullPath }, } = await this.parse(WebDAVConfig);
        const webdavConfig = await config_service_1.ConfigService.instance.readWebdavConfig();
        if (host) {
            webdavConfig['host'] = host;
        }
        if (port) {
            if (validation_service_1.ValidationService.instance.validateTCPIntegerPort(port)) {
                webdavConfig['port'] = port;
            }
            else {
                throw new command_types_1.NotValidPortError();
            }
        }
        if (http) {
            webdavConfig['protocol'] = 'http';
        }
        if (https) {
            webdavConfig['protocol'] = 'https';
        }
        if (timeout !== undefined) {
            webdavConfig['timeoutMinutes'] = timeout;
        }
        if (createFullPath !== undefined) {
            webdavConfig['createFullPath'] = createFullPath;
        }
        await config_service_1.ConfigService.instance.saveWebdavConfig(webdavConfig);
        const message = `On the next start, the WebDAV server will use the next config: ${JSON.stringify(webdavConfig)}`;
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message, config: webdavConfig };
    };
    catch = async (error) => {
        const { flags } = await this.parse(WebDAVConfig);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
}
exports.default = WebDAVConfig;
