"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const format_utils_1 = require("../utils/format.utils");
const trash_service_1 = require("../services/drive/trash.service");
class TrashList extends core_1.Command {
    static args = {};
    static description = 'Lists the content of the trash.';
    static aliases = ['trash:list'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        extended: core_1.Flags.boolean({
            char: 'e',
            description: 'Displays additional information in the trash list.',
            required: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(TrashList);
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        const { folders, files } = await trash_service_1.TrashService.instance.getTrashFolderContent();
        const allItems = [
            ...folders.map((folder) => {
                return {
                    type: 'folder',
                    name: folder.plainName,
                    id: folder.uuid,
                    size: '-',
                    modified: format_utils_1.FormatUtils.formatDate(folder.updatedAt),
                };
            }),
            ...files.map((file) => {
                return {
                    type: 'file',
                    name: file.type && file.type.length > 0 ? `${file.plainName}.${file.type}` : file.plainName,
                    id: file.uuid,
                    size: format_utils_1.FormatUtils.humanFileSize(Number(file.size)),
                    modified: format_utils_1.FormatUtils.formatDate(file.updatedAt),
                };
            }),
        ];
        const headers = [
            { value: 'type', alias: 'Type' },
            { value: 'name', alias: 'Name' },
            { value: 'id', alias: 'Id' },
        ];
        if (flags.extended) {
            headers.push({ value: 'modified', alias: 'Modified' }, { value: 'size', alias: 'Size' });
        }
        cli_utils_1.CLIUtils.table(this.log.bind(this), headers, allItems);
        return { success: true, list: { folders, files } };
    };
    catch = async (error) => {
        const { flags } = await this.parse(TrashList);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
}
exports.default = TrashList;
