"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
const drive_folder_service_1 = require("../services/drive/drive-folder.service");
class RenameFolder extends core_1.Command {
    static args = {};
    static description = 'Rename a folder.';
    static aliases = ['rename:folder'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        id: core_1.Flags.string({
            char: 'i',
            description: 'The ID of the folder to be renamed.',
            required: false,
        }),
        name: core_1.Flags.string({
            char: 'n',
            description: 'The new name for the folder.',
            required: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(RenameFolder);
        const nonInteractive = flags['non-interactive'];
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        const folderUuid = await this.getFolderUuid(flags['id'], nonInteractive);
        const name = await this.getFolderName(flags['name'], nonInteractive);
        await drive_folder_service_1.DriveFolderService.instance.renameFolder({ folderUuid, name });
        const message = `Folder renamed successfully with: ${name}`;
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message, folder: { uuid: folderUuid, plainName: name } };
    };
    catch = async (error) => {
        const { flags } = await this.parse(RenameFolder);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFolderUuid = async (folderUuidFlag, nonInteractive) => {
        const folderUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: folderUuidFlag,
            name: RenameFolder.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the folder id you want to rename?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFolderUuidError(),
        }, this.log.bind(this));
        return folderUuid;
    };
    getFolderName = async (folderNameFlag, nonInteractive) => {
        const folderName = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: folderNameFlag,
            name: RenameFolder.flags['name'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the new name of the folder?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateStringIsNotEmpty,
            error: new command_types_1.EmptyFolderNameError(),
        }, this.log.bind(this));
        return folderName;
    };
}
exports.default = RenameFolder;
