"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
const drive_file_service_1 = require("../services/drive/drive-file.service");
const node_path_1 = __importDefault(require("node:path"));
class RenameFile extends core_1.Command {
    static args = {};
    static description = 'Rename a file.';
    static aliases = ['rename:file'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        id: core_1.Flags.string({
            char: 'i',
            description: 'The ID of the file to be renamed.',
            required: false,
        }),
        name: core_1.Flags.string({
            char: 'n',
            description: 'The new name for the file.',
            required: false,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(RenameFile);
        const nonInteractive = flags['non-interactive'];
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        const fileUuid = await this.getFileUuid(flags['id'], nonInteractive);
        const fileName = await this.getFileName(flags['name'], nonInteractive);
        const pathInfo = node_path_1.default.parse(fileName);
        const newName = pathInfo.name;
        const newType = pathInfo.ext.replace('.', '');
        await drive_file_service_1.DriveFileService.instance.renameFile(fileUuid, { plainName: newName, type: newType });
        const message = `File renamed successfully with: ${newName}${newType ? '.' + newType : ''}`;
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message, file: { uuid: fileUuid, plainName: newName, type: newType } };
    };
    catch = async (error) => {
        const { flags } = await this.parse(RenameFile);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFileUuid = async (fileUuidFlag, nonInteractive) => {
        const fileUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: fileUuidFlag,
            name: RenameFile.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the file id you want to rename?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFileUuidError(),
        }, this.log.bind(this));
        return fileUuid;
    };
    getFileName = async (fileNameFlag, nonInteractive) => {
        const fileName = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: fileNameFlag,
            name: RenameFile.flags['name'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the new name of the file?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateStringIsNotEmpty,
            error: new command_types_1.EmptyFileNameError(),
        }, this.log.bind(this));
        return fileName;
    };
}
exports.default = RenameFile;
