"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const config_service_1 = require("../services/config.service");
const cli_utils_1 = require("../utils/cli.utils");
const command_types_1 = require("../types/command.types");
const validation_service_1 = require("../services/validation.service");
const drive_file_service_1 = require("../services/drive/drive-file.service");
class MoveFile extends core_1.Command {
    static args = {};
    static description = 'Move a file into a destination folder.';
    static aliases = ['move:file'];
    static examples = ['<%= config.bin %> <%= command.id %>'];
    static flags = {
        ...cli_utils_1.CLIUtils.CommonFlags,
        id: core_1.Flags.string({
            char: 'i',
            description: 'The ID of the file to be moved.',
            required: false,
        }),
        destination: core_1.Flags.string({
            char: 'd',
            description: 'The destination folder id where the file is going to be moved. Leave empty for the root folder.',
            required: false,
            parse: cli_utils_1.CLIUtils.parseEmpty,
        }),
    };
    static enableJsonFlag = true;
    run = async () => {
        const { flags } = await this.parse(MoveFile);
        const nonInteractive = flags['non-interactive'];
        const userCredentials = await config_service_1.ConfigService.instance.readUser();
        if (!userCredentials)
            throw new command_types_1.MissingCredentialsError();
        const fileUuid = await this.getFileUuid(flags['id'], nonInteractive);
        let destinationFolderUuid = await this.getDestinationFolderUuid(flags['destination'], nonInteractive);
        if (destinationFolderUuid.trim().length === 0) {
            destinationFolderUuid = userCredentials.user.rootFolderId;
        }
        const newFile = await drive_file_service_1.DriveFileService.instance.moveFile(fileUuid, { destinationFolder: destinationFolderUuid });
        const message = `File moved successfully to: ${destinationFolderUuid}`;
        cli_utils_1.CLIUtils.success(this.log.bind(this), message);
        return { success: true, message, file: newFile };
    };
    catch = async (error) => {
        const { flags } = await this.parse(MoveFile);
        cli_utils_1.CLIUtils.catchError({
            error,
            command: this.id,
            logReporter: this.log.bind(this),
            jsonFlag: flags['json'],
        });
        this.exit(1);
    };
    getFileUuid = async (fileUuidFlag, nonInteractive) => {
        const fileUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: fileUuidFlag,
            name: MoveFile.flags['id'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the file id you want to move?',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFileUuidError(),
        }, this.log.bind(this));
        return fileUuid;
    };
    getDestinationFolderUuid = async (destinationFolderUuidFlag, nonInteractive) => {
        const destinationFolderUuid = await cli_utils_1.CLIUtils.getValueFromFlag({
            value: destinationFolderUuidFlag,
            name: MoveFile.flags['destination'].name,
        }, {
            nonInteractive,
            prompt: {
                message: 'What is the destination folder id? (leave empty for the root folder)',
                options: { type: 'input' },
            },
        }, {
            validate: validation_service_1.ValidationService.instance.validateUUIDv4,
            error: new command_types_1.NotValidFolderUuidError(),
            canBeEmpty: true,
        }, this.log.bind(this));
        return destinationFolderUuid;
    };
}
exports.default = MoveFile;
