import ws from 'ws';
import { hexy as hexdump } from 'hexy';
import { getParsedWsUrlObject } from './ws-url';

/**
 * Safely parse JSON string with error handling
 * @param {string} jsonString - The JSON string to parse
 * @param {string} context - Context for error messages
 * @returns {Object} Parsed object or throws error with context
 * @throws {Error} If JSON parsing fails
 */
const safeParseJSON = (jsonString, context = 'JSON string') => {
  try {
    return JSON.parse(jsonString);
  } catch (error) {
    const errorMessage = `Failed to parse ${context}: ${error.message}`;
    console.error(errorMessage, {
      originalString: jsonString,
      parseError: error
    });
    throw new Error(errorMessage);
  }
};

const normalizeMessageByFormat = (message, format) => {
  if (!message) {
    return '';
  }
  switch (format) {
    case 'json':
      // If it was already stringified, do not double encode
      if (typeof message === 'string') {
        return message;
      }
      return JSON.stringify(message);
    case 'raw':
    case 'xml':
      return message;
    default: {
      if (typeof message === 'string') {
        return message;
      }
      if (typeof message === 'object') {
        return JSON.stringify(message);
      }
      console.warn('Received message of unhandled type.', { type: typeof message });
      return '';
    }
  }
};

const createSequencer = () => {
  const seq = {};

  const nextSeq = (requestId, collectionId) => {
    seq[requestId] ||= {};
    seq[requestId][collectionId] ||= 0;
    return ++seq[requestId][collectionId];
  };

  /**
   * @param {string} requestId
   * @param {string} [collectionId]
   */
  const clean = (requestId, collectionId = undefined) => {
    if (collectionId) {
      delete seq[requestId][collectionId];
    }
    if (!Object.keys(seq[requestId]).length) {
      delete seq[requestId];
    }
  };

  return {
    next: nextSeq,
    clean
  };
};

const seq = createSequencer();

class WsClient {
  messageQueues = {};
  activeConnections = new Map();
  connectionKeepAlive = new Map();

  constructor(eventCallback) {
    this.eventCallback = eventCallback;
  }

  /**
   * Start a WebSocket connection
   * @param {Object} params - Connection parameters
   * @param {Object} params.request - The WebSocket request object
   * @param {Object} params.collection - The collection object
   * @param {Object} params.options - Additional connection options
   */
  async startConnection({ request, collection, options = {} }) {
    const { url, headers } = request;
    const { timeout = 30000, keepAlive = false, keepAliveInterval = 10_000, sslOptions = {} } = options;

    const parsedUrl = getParsedWsUrlObject(url);
    const timeoutAsNumber = Number(timeout);
    const validTimeout = isNaN(timeoutAsNumber) ? 30000 : timeoutAsNumber;

    const requestId = request.uid;
    const collectionUid = collection.uid;

    try {
      // Create WebSocket connection
      // Note: unlike the standard Websocket constructor the `ws` library doesn't support adding Protocols as a single string
      // and instead needs it broken down manually, make sure this tested with multiple protocols again.
      const protocols = []
        .concat([headers['Sec-WebSocket-Protocol'], headers['sec-websocket-protocol']])
        .filter(Boolean)
        .map((d) => d.split(','))
        .flat()
        .map((d) => d.trim());

      const protocolVersion = headers['Sec-WebSocket-Version'] || headers['sec-websocket-version'];

      const wsOptions = {
        headers,
        handshakeTimeout: validTimeout,
        followRedirects: true,
        rejectUnauthorized: sslOptions.rejectUnauthorized,
        ca: sslOptions.ca,
        cert: sslOptions.cert,
        key: sslOptions.key,
        pfx: sslOptions.pfx,
        passphrase: sslOptions.passphrase
      };

      if (protocolVersion) {
        // Force convert to number since `ws` doesn't do it for you
        const asNumber = Number(protocolVersion);
        if (!isNaN(asNumber)) {
          wsOptions.protocolVersion = asNumber;
        }
      }

      const wsConnection = new ws.WebSocket(parsedUrl.fullUrl, protocols, wsOptions);

      // Set up event handlers
      this.#setupWsEventHandlers(wsConnection, requestId, collectionUid, { keepAlive, keepAliveInterval });

      // Store the connection
      this.#addConnection(requestId, collectionUid, wsConnection);

      // Emit connecting event
      this.eventCallback('main:ws:connecting', requestId, collectionUid);

      return wsConnection;
    } catch (error) {
      console.error('Error creating WebSocket connection:', error);
      this.eventCallback('main:ws:error', requestId, collectionUid, {
        error: error.message
      });
      throw error;
    }
  }

  #getMessageQueueId(requestId) {
    return `${requestId}`;
  }

  queueMessage(requestId, collectionUid, message, format = 'raw') {
    const connectionMeta = this.activeConnections.get(requestId);

    const mqKey = this.#getMessageQueueId(requestId);
    this.messageQueues[mqKey] ||= [];
    this.messageQueues[mqKey].push({
      message,
      format
    });

    if (connectionMeta && connectionMeta.connection && connectionMeta.connection.readyState === WebSocket.OPEN) {
      this.#flushQueue(requestId, collectionUid);
      return;
    }
  }

  #flushQueue(requestId, collectionUid) {
    const mqKey = this.#getMessageQueueId(requestId);
    if (!(mqKey in this.messageQueues)) return;
    while (this.messageQueues[mqKey].length > 0) {
      const { message, format } = this.messageQueues[mqKey].shift();
      this.sendMessage(requestId, collectionUid, message, format);
    }
  }

  /**
   * Send a message to an active WebSocket connection
   * @param {string} requestId - The request ID of the active connection
   * @param {string} collectionUid - The collection UID for the request
   * @param {Object|string} message - The message to send
   */
  sendMessage(requestId, collectionUid, message, format = 'raw') {
    const connectionMeta = this.activeConnections.get(requestId);

    if (connectionMeta.connection && connectionMeta.connection.readyState === WebSocket.OPEN) {
      const payload = normalizeMessageByFormat(message, format);

      // Send the message
      connectionMeta.connection.send(payload, (error) => {
        if (error) {
          this.eventCallback('main:ws:error', requestId, collectionUid, { error });
        } else {
          // Emit message sent event
          this.eventCallback('main:ws:message', requestId, collectionUid, {
            message: payload,
            messageHexdump: hexdump(payload),
            type: 'outgoing',
            seq: seq.next(requestId, collectionUid),
            timestamp: Date.now()
          });
        }
      });
    } else {
      const error = new Error('WebSocket connection not available or not open');
      this.eventCallback('main:ws:error', requestId, collectionUid, {
        error: error.message
      });
    }
  }

  /**
   * Close a WebSocket connection
   * @param {string} requestId - The request ID to close
   * @param {number} code - Close code (optional)
   * @param {string} reason - Close reason (optional)
   */
  close(requestId, code = 1000, reason = 'Client initiated close') {
    const connectionMeta = this.activeConnections.get(requestId);
    if (connectionMeta?.connection) {
      connectionMeta.connection.close(code, reason);
      this.#removeConnection(requestId);
      seq.clean(requestId);
    }
  }

  /**
   * Check if a connection is active
   * @param {string} requestId - The request ID to check
   * @returns {boolean} - Whether the connection is active
   */
  isConnectionActive(requestId) {
    const connectionMeta = this.activeConnections.get(requestId);
    return connectionMeta && connectionMeta.connection.readyState === ws.WebSocket.OPEN;
  }

  /**
   * Get all active connection IDs
   * @returns {string[]} Array of active connection IDs
   */
  getActiveConnectionIds() {
    return Array.from(this.activeConnections.keys());
  }

  closeForCollection(collectionUid) {
    [...this.activeConnections.keys()].forEach((k) => {
      const meta = this.activeConnections.get(k);
      if (meta.collectionUid === collectionUid) {
        meta.connection.close();
        this.activeConnections.delete(k);
      }
    });
  }

  /**
   * Clear all active connections
   */
  clearAllConnections() {
    const connectionIds = this.getActiveConnectionIds();

    this.activeConnections.forEach((connection) => {
      if (connection.readyState === WebSocket.OPEN) {
        connection.close(1000, 'Client clearing all connections');
      }
    });

    this.activeConnections.clear();

    // Emit an event with empty active connection IDs
    if (connectionIds.length > 0) {
      this.eventCallback('main:ws:connections-changed', {
        type: 'cleared',
        activeConnectionIds: []
      });
    }
  }

  /**
   * Set up WebSocket event handlers
   * @param {WebSocket} ws - The WebSocket instance
   * @param {string} requestId - The request ID
   * @param {string} collectionUid - The collection UID
   * @param {object} options
   * @param {boolean} options.keepAlive - keep the connection alive
   * @param {number} options.keepAliveInterval - What the interval for keeping interval
   * @private
   */
  #setupWsEventHandlers(ws, requestId, collectionUid, options) {
    ws.on('open', () => {
      this.#flushQueue(requestId, collectionUid);

      if (options.keepAlive) {
        const handle = setInterval(() => {
          ws.isAlive = false;
          ws.ping();
        }, options.keepAliveInterval);

        this.connectionKeepAlive.set(requestId, handle);
      }

      this.eventCallback('main:ws:open', requestId, collectionUid, {
        timestamp: Date.now(),
        url: ws.url,
        seq: seq.next(requestId, collectionUid)
      });
    });

    ws.on('redirect', (url, req) => {
      const headerNames = req.getHeaderNames();
      const headers = Object.fromEntries(headerNames.map((d) => [d, req.getHeader(d)]));
      this.eventCallback('main:ws:redirect', requestId, collectionUid, {
        message: `Redirected to ${url}`,
        type: 'info',
        timestamp: Date.now(),
        headers: headers,
        seq: seq.next(requestId, collectionUid)
      });
    });

    ws.on('upgrade', (response) => {
      this.eventCallback('main:ws:upgrade', requestId, collectionUid, {
        type: 'info',
        timestamp: Date.now(),
        seq: seq.next(requestId, collectionUid),
        headers: { ...response.headers }
      });
    });

    ws.on('message', (data) => {
      try {
        const message = JSON.parse(data.toString());
        this.eventCallback('main:ws:message', requestId, collectionUid, {
          message,
          messageHexdump: hexdump(Buffer.from(data)),
          type: 'incoming',
          seq: seq.next(requestId, collectionUid),
          timestamp: Date.now()
        });
      } catch (error) {
        // If parsing fails, send as raw data
        this.eventCallback('main:ws:message', requestId, collectionUid, {
          message: data.toString(),
          messageHexdump: hexdump(data),
          type: 'incoming',
          seq: seq.next(requestId, collectionUid),
          timestamp: Date.now()
        });
      }
    });

    ws.on('close', (code, reason) => {
      this.eventCallback('main:ws:close', requestId, collectionUid, {
        code,
        reason: Buffer.from(reason).toString(),
        seq: seq.next(requestId, collectionUid),
        timestamp: Date.now()
      });
      seq.clean(requestId, collectionUid);
      this.#removeConnection(requestId);
    });

    ws.on('error', (error) => {
      this.eventCallback('main:ws:error', requestId, collectionUid, {
        error: error.message,
        seq: seq.next(requestId, collectionUid),
        timestamp: Date.now()
      });
    });
  }

  /**
   * Add a connection to the active connections map and emit an event
   * @param {string} requestId - The request ID
   * @param {WebSocket} connection - The WebSocket connection
   * @private
   */
  #addConnection(requestId, collectionUid, connection) {
    this.activeConnections.set(requestId, { collectionUid, connection });

    // Emit an event with all active connection IDs
    this.eventCallback('main:ws:connections-changed', {
      type: 'added',
      requestId,
      seq: seq.next(requestId, collectionUid),
      activeConnectionIds: this.getActiveConnectionIds()
    });
  }

  /**
   * Remove a connection from the active connections map and emit an event
   * @param {string} requestId - The request ID
   * @private
   */
  #removeConnection(requestId) {
    if (this.connectionKeepAlive.has(requestId)) {
      clearInterval(this.connectionKeepAlive.get(requestId));
      this.connectionKeepAlive.delete(requestId);
    }

    const mqId = this.#getMessageQueueId(requestId);
    if (mqId in this.messageQueues) {
      this.messageQueues[mqId] = [];
    }

    if (this.activeConnections.has(requestId)) {
      this.activeConnections.delete(requestId);

      // Emit an event with all active connection IDs
      this.eventCallback('main:ws:connections-changed', {
        type: 'removed',
        requestId,
        activeConnectionIds: this.getActiveConnectionIds()
      });
    }
  }

  /**
   * Get the connection status of a connection
   * @param {string} requestId - The request ID to get the connection status of
   * @returns {string} - The connection status
   */
  // Returns "disconnected", "connecting", "connected"
  connectionStatus(requestId) {
    const connectionMeta = this.activeConnections.get(requestId);
    if (connectionMeta?.connection?.readyState === ws.WebSocket.CONNECTING) return 'connecting';
    if (connectionMeta?.connection?.readyState === ws.WebSocket.OPEN) return 'connected';
    return 'disconnected';
  }
}

export { WsClient };
