const COMMENT_TYPE = {
    COMMENT_BLOCK: "commentBlock",
    COMMENT_LINE : "commentLine",
};

const REGEX_TYPE = "regex";

const firstFound = (str, stringStarters = null) =>
{
    stringStarters = stringStarters || [
        {name: "quote", char: "'"},
        {name: "literal", char: "`"},
        {name: "doubleQuote", char: "\""},
        {name: COMMENT_TYPE.COMMENT_BLOCK, char: "/*"},
        {name: COMMENT_TYPE.COMMENT_LINE, char: "//"},
        {name: REGEX_TYPE, char: "/"},
    ];

    let lastIndex = -1;
    let winner = -1;
    let item = {};
    for (let i = 0; i < stringStarters.length; ++i)
    {
        item = stringStarters[i];
        const index = str.indexOf(item.char);
        if (index > -1 && lastIndex < 0)
        {
            lastIndex = index;
            winner = i;
        }

        if (index > -1 && index < lastIndex)
        {
            lastIndex = index;
            winner = i;
        }

        item.index = index;
    }

    if (winner === -1)
    {
        return {
            index: -1
        };
    }

    return {
        char : stringStarters[winner].char,
        name : stringStarters[winner].name,
        index: lastIndex
    };
};

/**
 * Look for non escaped characters
 * @param str
 * @param chars
 * @param specialCharStart
 * @param specialCharEnd
 * @returns {{index: number}}
 */
const getNextClosingElement = (str, chars, {specialCharStart = null, specialCharEnd = null} = {}) =>
{
    if (!Array.isArray(chars))
    {
        chars = [chars];
    }

    const n = str.length;
    for (let i = 0; i < n; ++i)
    {
        const currentChar = str[i];
        if (currentChar === "\\")
        {
            ++i;
            continue;
        }

        // Special case for regexes with brackets []
        if (specialCharStart && currentChar === specialCharStart)
        {
            const newStr = str.substring(i);
            const stp = getNextClosingElement(newStr, specialCharEnd);
            i += stp.index;
        }

        if (chars.includes(currentChar))
        {
            return {
                index: i
            };
        }
    }

    return {
        index: -1
    };
};

const movePointerIndex = (str, index) =>
{
    str = str.substring(index);
    return str;
};

const parseString = (str) =>
{
    const originalString = str;
    const originalStringLength = originalString.length;

    const detectedString = [];
    const detectedComments = [];
    const detectedRegex = [];

    do
    {
        let item = firstFound(str);
        if (item.index === -1)
        {
            break;
        }

        const enter = {
            item
        };

        /** Parse comment blocks */
        if (item.name === COMMENT_TYPE.COMMENT_BLOCK)
        {
            enter.type = item.name;

            str = movePointerIndex(str, item.index);
            enter.index = originalStringLength - str.length;

            const nextIndex = str.indexOf("*/");
            if (nextIndex === -1)
            {
                throw new Error("Comment Block opened at position ... not enclosed");
            }

            str = movePointerIndex(str, nextIndex + 2);
            enter.indexEnd = originalStringLength - str.length;
            enter.content = originalString.substring(enter.index, enter.indexEnd);

            detectedComments.push(enter);
            continue;
        }
        /** Parse comment lines */
        else if (item.name === COMMENT_TYPE.COMMENT_LINE)
        {
            enter.type = item.name;
            // Beginning of line comment //
            str = movePointerIndex(str, item.index);

            enter.index = originalStringLength - str.length;

            let newLinePos = str.indexOf("\n");
            if (newLinePos === -1)
            {
                enter.indexEnd = originalStringLength;
                enter.content = originalString.substring(enter.index, enter.indexEnd - 1);
                detectedComments.push(enter);
                break;
            }

            str = movePointerIndex(str, newLinePos + 1);
            enter.indexEnd = originalStringLength - str.length - 1;

            enter.content = originalString.substring(enter.index, enter.indexEnd);
            detectedComments.push(enter);

            continue;
        }
        else if (item.name === REGEX_TYPE)
        {
            enter.type = item.name;

            str = movePointerIndex(str, item.index + 1);
            enter.index = originalStringLength - str.length - 1;

            const nextItem = getNextClosingElement(str, ["/", "\n"], {specialCharStart: "[", specialCharEnd: "]"});
            if (nextItem.index === -1)
            {
                throw new Error(`SCT: (1005) Regex opened at position ${enter.index} not enclosed`);
            }

            str = movePointerIndex(str, nextItem.index + 1);
            enter.indexEnd = originalStringLength - str.length;
            enter.content = originalString.substring(enter.index, enter.indexEnd);

            detectedRegex.push(enter);
            continue;

        }

        /** Parse strings */
        str = str.substring(item.index + 1);
        enter.index = originalStringLength - str.length;

        const nextItem = getNextClosingElement(str, item.char);

        // The string in the file is never closed
        if (nextItem.index === -1)
        {
            throw new Error(`SCT: (1001) String opened at position ${enter.index} with a ${item.name} not enclosed`);
        }

        str = movePointerIndex(str, nextItem.index + 1);
        enter.indexEnd = originalStringLength - str.length - 1;
        enter.content = originalString.substring(enter.index, enter.indexEnd);

        detectedString.push(enter);
    }
    while (true);

    return {
        text    : str,
        strings : detectedString,
        comments: detectedComments,
        regexes : detectedRegex
    };
};

function replaceOccurences(strings, str, replacer, {includeDelimiter = true})
{
    const isCallable = typeof replacer === "function";
    const n = strings.length;

    for (let i = n - 1; i >= 0; --i)
    {
        const info = strings[i];

        const replacement = isCallable ? replacer(info, str) : replacer;

        if (includeDelimiter)
        {
            str = str.substring(0, info.index - 1) + replacement + str.substring(info.indexEnd + 1);
        }
        else
        {
            str = str.substring(0, info.index) + replacement + str.substring(info.indexEnd);
        }
    }
    return str;
}

/**
 * Strip comments from source
 * @param str
 * @param replacer
 * @returns {string|*}
 */
const stripComments = (str, replacer = "") =>
{
    const comments = parseString(str).comments;
    str = replaceOccurences(comments, str, replacer, {includeDelimiter: false});
    return str;
};

/**
 * Strip strings from source
 * @param str
 * @param replacer
 * @param includeDelimiter
 * @returns {*}
 */
const stripStrings = (str, replacer = "", {includeDelimiter = true} = {}) =>
{
    const strings = parseString(str).strings;
    str = replaceOccurences(strings, str, replacer, {includeDelimiter});
    return str;
};

/**
 * Remove all string content from source
 * @param str
 * @param replacer
 * @param includeDelimiter
 * @returns {*}
 */
const clearStrings = (str, replacer = "", {includeDelimiter = false} = {}) =>
{
    const strings = parseString(str).strings;
    str = replaceOccurences(strings, str, replacer, {includeDelimiter});
    return str;
};

/**
 * Strip regexes from source
 * @param str
 * @param replacer
 * @param includeDelimiter
 * @returns {string|*}
 */
const stripRegexes = (str, replacer = "", {includeDelimiter = true} = {}) =>
{
    const strings = parseString(str).regexes;
    str = replaceOccurences(strings, str, replacer, {includeDelimiter});
    return str;
};

/**
 * Remove all regex content from source
 * @param str
 * @param replacer
 * @param includeDelimiter
 * @returns {*}
 */
const clearRegexes = (str, replacer = "//", {includeDelimiter = false} = {}) =>
{
    const strings = parseString(str).regexes;
    str = replaceOccurences(strings, str, replacer, {includeDelimiter});
    return str;
};

module.exports = {
    parseString, stripComments, stripStrings, clearStrings, clearRegexes
};

module.exports.parseString = parseString;

module.exports.stripComments = stripComments;
module.exports.stripStrings = stripStrings;
module.exports.stripRegexes = stripRegexes;

module.exports.clearStrings = clearStrings;
module.exports.clearRegexes = clearRegexes;
