/**
 * @typedef {import('mdast').PhrasingContent} PhrasingContent
 * @typedef {import('mdast').Root} Root
 * @typedef {import('mdast-util-find-and-replace').ReplaceFunction} ReplaceFunction
 * @typedef {import('mdast-util-find-and-replace').RegExpMatchObject} RegExpMatchObject
 * @typedef {import('vfile').VFile} VFile
 */

/**
 * @callback BuildUrl
 *   Create a URL.
 * @param {Readonly<BuildUrlValues>} values
 *   Info on the link to build.
 * @returns {string | false}
 *   URL to use or `false` to not link.
 *
 * @typedef BuildUrlCommitValues
 *   Info for commit hash.
 * @property {string} hash
 *   Commit hash value.
 * @property {string} project
 *   Project name.
 * @property {'commit'} type
 *   Kind.
 * @property {string} user
 *   Owner of repo.
 *
 * @typedef BuildUrlCompareValues
 *   Info for commit hash ranges.
 * @property {string} base
 *   SHA of the range start.
 * @property {string} compare
 *   SHA of the range end.
 * @property {string} project
 *   Project name.
 * @property {'compare'} type
 *   Kind.
 * @property {string} user
 *   Owner of repo.
 *
 * @typedef BuildUrlIssueValues
 *   Info for issues.
 * @property {string} no
 *   Issue number.
 * @property {string} project
 *   Project name.
 * @property {'issue'} type
 *   Kind.
 * @property {string} user
 *   Owner of repo.
 *
 * @typedef BuildUrlMentionValues
 *   Info for mentions.
 * @property {'mention'} type
 *   Kind.
 * @property {string} user
 *   User name.
 *
 * @typedef {BuildUrlCommitValues | BuildUrlCompareValues | BuildUrlIssueValues | BuildUrlMentionValues} BuildUrlValues
 *   Info.
 *
 * @typedef Options
 *   Configuration.
 * @property {BuildUrl | null | undefined} [buildUrl]
 *   Change how things are linked (optional).
 * @property {boolean | null | undefined} [mentionStrong=true]
 *   Wrap mentions in `strong` (default: `true`);
 *   this makes them render more like how GitHub styles them, but GH itself
 *   uses CSS instead of `strong`.
 * @property {string | null | undefined} [repository]
 *   Repository to link against (default: `repository` from `packag.json` in CWD in Node);
 *   should point to a GitHub repository (such as `'user/project'`)
 *
 * @typedef RepositoryInfo
 *   Owner and project of repo.
 * @property {string} project
 *   Project name.
 * @property {string} user
 *   User/organization name.
 *
 * @typedef UrlInfo
 *   Info.
 * @property {boolean} comment
 *   Whether the link is to a comment.
 * @property {string} page
 *   Page type.
 * @property {string} project
 *   Project name.
 * @property {string} reference
 *   Reference.
 * @property {string} user
 *   User/organization name.
 */

import {findAndReplace} from 'mdast-util-find-and-replace'
import {toString} from 'mdast-util-to-string'
import {visit} from 'unist-util-visit'
import {getRepoFromPackage} from '#get-repo-from-package'

/** @type {Readonly<Options>} */
const emptyOptions = {}

// Previously, GitHub linked `@mention` and `@mentions` to their blog post about
// mentions (<https://github.com/blog/821>).
// Since June 2019, and possibly earlier, they stopped linking those references.
const denyMention = new Set(['mention', 'mentions'])

// Denylist of SHAs that are also valid words.
//
// GitHub allows abbreviating SHAs up to 7 characters.
// These cases are ignored in text because they might just be ment as normal
// words.
// If you’d like these to link to their SHAs, use more than 7 characters.
//
// Generated by:
//
// ```sh
// egrep -i "^[a-f0-9]{7,}$" /usr/share/dict/words
// ```
//
// Added a couple forms of 6 character words in GH-20:
// <https://github.com/remarkjs/remark-github/issues/20>.
const denyHash = new Set([
  'acceded',
  'deedeed',
  'defaced',
  'effaced',
  'fabaceae'
])

// Constants.
const minShaLength = 7

// Username may only contain alphanumeric characters or single hyphens, and
// cannot begin or end with a hyphen*.
//
// \* That is: until <https://github.com/remarkjs/remark-github/issues/13>.
const userGroup = '[\\da-z][-\\da-z]{0,38}'
const projectGroup = '(?:\\.git[\\w-]|\\.(?!git)|[\\w-])+'
const repoGroup = '(' + userGroup + ')\\/(' + projectGroup + ')'

const linkRegex = new RegExp(
  '^https?:\\/\\/github\\.com\\/' +
    repoGroup +
    '\\/(commit|compare|issues|pull)\\/([a-f\\d]+(?:\\.{3}[a-f\\d]+)?\\/?(?=[#?]|$))',
  'i'
)

const repoRegex = new RegExp(
  '(?:^|/(?:repos/)?)' + repoGroup + '(?=\\.git|[\\/#@]|$)',
  'i'
)

const referenceRegex = new RegExp(
  '(' +
    userGroup +
    ')(?:\\/(' +
    projectGroup +
    '))?(?:#([1-9]\\d*)|@([a-f\\d]{7,40}))',
  'gi'
)

const mentionRegex = new RegExp(
  '@(' + userGroup + '(?:\\/' + userGroup + ')?)',
  'gi'
)

/**
 * Create a URL to GH.
 *
 * @satisfies {BuildUrl}
 * @param {Readonly<BuildUrlValues>} values
 *   Info on the link to build.
 * @returns {string}
 *   URL to use.
 */
export function defaultBuildUrl(values) {
  const base = 'https://github.com'

  if (values.type === 'mention') {
    return [base, values.user].join('/')
  }

  const {project, user} = values

  if (values.type === 'commit') {
    return [base, user, project, 'commit', values.hash].join('/')
  }

  if (values.type === 'issue') {
    return [base, user, project, 'issues', values.no].join('/')
  }

  // `values.type` is `'compare'`
  return [
    base,
    user,
    project,
    'compare',
    values.base + '...' + values.compare
  ].join('/')
}

/**
 * Link references to users, commits, and issues, in the same way that GitHub
 * does in comments, issues, PRs, and releases.
 *
 * @param {Readonly<Options> | null | undefined} [options]
 *   Configuration (optional).
 * @returns
 *   Transform.
 */
export default function remarkGithub(options) {
  const settings = options || emptyOptions
  const buildUrl = settings.buildUrl || defaultBuildUrl

  /**
   * Transform.
   *
   * @param {Root} tree
   *   Tree.
   * @param {VFile} file
   *   File.
   * @returns {undefined}
   *   Nothing.
   */
  return function (tree, file) {
    // To do: when async/await in React is easier, we can do `getRepoFromPackage` async.
    // Then it should throw errors when the repo is incorrect / files are broken / etc.
    const repository = settings.repository || getRepoFromPackage(file.cwd)

    if (!repository) {
      throw new Error('Unexpected missing `repository` in `options`')
    }

    // Parse the URL: See the tests for all possible kinds.
    const repositoryMatch = repoRegex.exec(repository)

    if (!repositoryMatch) {
      throw new Error(
        'Unexpected invalid `repository`, expected for example `user/project`'
      )
    }

    /** @type {Readonly<RepositoryInfo>} */
    const repositoryInfo = {
      project: repositoryMatch[2],
      user: repositoryMatch[1]
    }

    findAndReplace(
      tree,
      [
        [referenceRegex, replaceReference],
        [mentionRegex, replaceMention],
        [/(?:#|\bgh-)([1-9]\d*)/gi, replaceIssue],
        [/\b([a-f\d]{7,40})\.{3}([a-f\d]{7,40})\b/gi, replaceHashRange],
        [/\b[a-f\d]{7,40}\b/gi, replaceHash]
      ],
      {ignore: ['link', 'linkReference']}
    )

    visit(tree, 'link', function (node) {
      /** @type {Readonly<UrlInfo> | undefined} */
      const link = parse(node)

      if (!link) {
        return
      }

      const comment = link.comment ? ' (comment)' : ''
      /** @type {string} */
      let base

      if (
        link.project !== repositoryInfo.project ||
        // Compare page uses full `user/project` for forks.
        (link.page === 'compare' && link.user !== repositoryInfo.user)
      ) {
        base = link.user + '/' + link.project
      } else if (link.user === repositoryInfo.user) {
        base = ''
      } else {
        base = link.user
      }

      /** @type {Array<PhrasingContent>} */
      const children = []

      if (link.page === 'issues' || link.page === 'pull') {
        base += '#'
        children.push({
          type: 'text',
          value: base + link.reference + comment
        })
      } else {
        if (base) {
          children.push({type: 'text', value: base + '@'})
        }

        children.push({type: 'inlineCode', value: link.reference})

        if (link.comment) {
          children.push({type: 'text', value: comment})
        }
      }

      node.children = children
    })

    /**
     * @type {ReplaceFunction}
     * @param {string} value
     * @param {string} username
     * @param {RegExpMatchObject} match
     */
    function replaceMention(value, username, match) {
      if (
        /[\w`]/.test(match.input.charAt(match.index - 1)) ||
        /[/\w`]/.test(match.input.charAt(match.index + value.length)) ||
        denyMention.has(username)
      ) {
        return false
      }

      const url = buildUrl({type: 'mention', user: username})

      if (!url) return false

      /** @type {PhrasingContent} */
      let node = {type: 'text', value}

      if (settings.mentionStrong !== false) {
        node = {type: 'strong', children: [node]}
      }

      return {type: 'link', title: null, url, children: [node]}
    }

    /**
     * @type {ReplaceFunction}
     * @param {string} value
     * @param {string} no
     * @param {RegExpMatchObject} match
     */
    function replaceIssue(value, no, match) {
      if (
        /\w/.test(match.input.charAt(match.index - 1)) ||
        /\w/.test(match.input.charAt(match.index + value.length))
      ) {
        return false
      }

      const url = buildUrl({no, type: 'issue', ...repositoryInfo})

      return url
        ? {type: 'link', title: null, url, children: [{type: 'text', value}]}
        : false
    }

    /**
     * @type {ReplaceFunction}
     * @param {string} value
     * @param {string} a
     * @param {string} b
     * @param {RegExpMatchObject} match
     */
    function replaceHashRange(value, a, b, match) {
      if (
        /[^\t\n\r (@[{]/.test(match.input.charAt(match.index - 1)) ||
        /\w/.test(match.input.charAt(match.index + value.length)) ||
        denyHash.has(value)
      ) {
        return false
      }

      const url = buildUrl({
        base: a,
        compare: b,
        type: 'compare',
        ...repositoryInfo
      })

      return url
        ? {
            type: 'link',
            title: null,
            url,
            children: [{type: 'inlineCode', value: abbr(a) + '...' + abbr(b)}]
          }
        : false
    }

    /**
     * @type {ReplaceFunction}
     * @param {string} value
     * @param {RegExpMatchObject} match
     */
    function replaceHash(value, match) {
      if (
        /[^\t\n\r (@[{.]/.test(match.input.charAt(match.index - 1)) ||
        // For some weird reason GH does link two dots, but not one 🤷‍♂️
        (match.input.charAt(match.index - 1) === '.' &&
          match.input.charAt(match.index - 2) !== '.') ||
        /\w/.test(match.input.charAt(match.index + value.length)) ||
        denyHash.has(value)
      ) {
        return false
      }

      const url = buildUrl({hash: value, type: 'commit', ...repositoryInfo})

      return url
        ? {
            type: 'link',
            title: null,
            url,
            children: [{type: 'inlineCode', value: abbr(value)}]
          }
        : false
    }

    /**
     * @type {ReplaceFunction}
     * @param {string} $0
     * @param {string} user
     * @param {string} specificProject
     * @param {string} no
     * @param {string} hash
     * @param {RegExpMatchObject} match
     */
    // eslint-disable-next-line max-params
    function replaceReference($0, user, specificProject, no, hash, match) {
      if (
        /[^\t\n\r (@[{]/.test(match.input.charAt(match.index - 1)) ||
        /\w/.test(match.input.charAt(match.index + $0.length))
      ) {
        return false
      }

      const project = specificProject || repositoryInfo.project
      /** @type {BuildUrlValues} */
      const values = no
        ? {no, project, type: 'issue', user}
        : {hash, project, type: 'commit', user}
      const url = buildUrl(values)

      if (!url) return false

      /** @type {Array<PhrasingContent>} */
      const nodes = []
      let value = ''

      if (project !== repositoryInfo.project) {
        value += user + '/' + project
      } else if (user !== repositoryInfo.user) {
        value += user
      }

      if (no) {
        value += '#' + no
      } else {
        value += '@'
        nodes.push({type: 'inlineCode', value: abbr(hash)})
      }

      nodes.unshift({type: 'text', value})

      return {type: 'link', title: null, url, children: nodes}
    }
  }
}

/**
 * Abbreviate a SHA.
 *
 * @param {string} sha
 *   SHA.
 * @returns {string}
 *   Abbreivated SHA.
 */
function abbr(sha) {
  return sha.slice(0, minShaLength)
}

/**
 * Parse a link and determine whether it links to GitHub.
 *
 * @param {import('mdast').Link} node
 *   Link node.
 * @returns {UrlInfo | undefined}
 *   Info.
 */
function parse(node) {
  const match = linkRegex.exec(node.url)

  if (
    // Not a proper URL.
    !match ||
    // Looks like formatting.
    node.children.length !== 1 ||
    node.children[0].type !== 'text' ||
    toString(node) !== node.url ||
    // SHAs can be min 4, max 40 characters.
    (match[3] === 'commit' && (match[4].length < 4 || match[4].length > 40)) ||
    // SHAs can be min 4, max 40 characters.
    (match[3] === 'compare' &&
      !/^[a-f\d]{4,40}\.{3}[a-f\d]{4,40}$/.test(match[4])) ||
    // Issues / PRs are decimal only.
    ((match[3] === 'issues' || match[3] === 'pull') &&
      /[a-f]/i.test(match[4])) ||
    // Projects can be at most 99 characters.
    match[2].length >= 100
  ) {
    return
  }

  let reference = match[4]

  if (match[3] === 'compare') {
    const [base, compare] = reference.split('...')
    reference = abbr(base) + '...' + abbr(compare)
  } else {
    reference = abbr(reference)
  }

  return {
    comment:
      node.url.charAt(match[0].length) === '#' &&
      match[0].length + 1 < node.url.length,
    page: match[3],
    project: match[2],
    reference,
    user: match[1]
  }
}
