'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var process$1 = require('node:process');
var os = require('node:os');
var tty = require('node:tty');
var Table = require('cli-table3');
var require$$0 = require('os');
var require$$1 = require('tty');
var emoji = require('node-emoji');
var supportsHyperlinks = require('supports-hyperlinks');

function _interopNamespaceDefault(e) {
	var n = Object.create(null);
	if (e) {
		Object.keys(e).forEach(function (k) {
			if (k !== 'default') {
				var d = Object.getOwnPropertyDescriptor(e, k);
				Object.defineProperty(n, k, d.get ? d : {
					enumerable: true,
					get: function () { return e[k]; }
				});
			}
		});
	}
	n.default = e;
	return Object.freeze(n);
}

var emoji__namespace = /*#__PURE__*/_interopNamespaceDefault(emoji);

const ANSI_BACKGROUND_OFFSET = 10;

const wrapAnsi16 = (offset = 0) => code => `\u001B[${code + offset}m`;

const wrapAnsi256 = (offset = 0) => code => `\u001B[${38 + offset};5;${code}m`;

const wrapAnsi16m = (offset = 0) => (red, green, blue) => `\u001B[${38 + offset};2;${red};${green};${blue}m`;

const styles$1 = {
	modifier: {
		reset: [0, 0],
		// 21 isn't widely supported and 22 does the same thing
		bold: [1, 22],
		dim: [2, 22],
		italic: [3, 23],
		underline: [4, 24],
		overline: [53, 55],
		inverse: [7, 27],
		hidden: [8, 28],
		strikethrough: [9, 29],
	},
	color: {
		black: [30, 39],
		red: [31, 39],
		green: [32, 39],
		yellow: [33, 39],
		blue: [34, 39],
		magenta: [35, 39],
		cyan: [36, 39],
		white: [37, 39],

		// Bright color
		blackBright: [90, 39],
		gray: [90, 39], // Alias of `blackBright`
		grey: [90, 39], // Alias of `blackBright`
		redBright: [91, 39],
		greenBright: [92, 39],
		yellowBright: [93, 39],
		blueBright: [94, 39],
		magentaBright: [95, 39],
		cyanBright: [96, 39],
		whiteBright: [97, 39],
	},
	bgColor: {
		bgBlack: [40, 49],
		bgRed: [41, 49],
		bgGreen: [42, 49],
		bgYellow: [43, 49],
		bgBlue: [44, 49],
		bgMagenta: [45, 49],
		bgCyan: [46, 49],
		bgWhite: [47, 49],

		// Bright color
		bgBlackBright: [100, 49],
		bgGray: [100, 49], // Alias of `bgBlackBright`
		bgGrey: [100, 49], // Alias of `bgBlackBright`
		bgRedBright: [101, 49],
		bgGreenBright: [102, 49],
		bgYellowBright: [103, 49],
		bgBlueBright: [104, 49],
		bgMagentaBright: [105, 49],
		bgCyanBright: [106, 49],
		bgWhiteBright: [107, 49],
	},
};

Object.keys(styles$1.modifier);
const foregroundColorNames = Object.keys(styles$1.color);
const backgroundColorNames = Object.keys(styles$1.bgColor);
[...foregroundColorNames, ...backgroundColorNames];

function assembleStyles() {
	const codes = new Map();

	for (const [groupName, group] of Object.entries(styles$1)) {
		for (const [styleName, style] of Object.entries(group)) {
			styles$1[styleName] = {
				open: `\u001B[${style[0]}m`,
				close: `\u001B[${style[1]}m`,
			};

			group[styleName] = styles$1[styleName];

			codes.set(style[0], style[1]);
		}

		Object.defineProperty(styles$1, groupName, {
			value: group,
			enumerable: false,
		});
	}

	Object.defineProperty(styles$1, 'codes', {
		value: codes,
		enumerable: false,
	});

	styles$1.color.close = '\u001B[39m';
	styles$1.bgColor.close = '\u001B[49m';

	styles$1.color.ansi = wrapAnsi16();
	styles$1.color.ansi256 = wrapAnsi256();
	styles$1.color.ansi16m = wrapAnsi16m();
	styles$1.bgColor.ansi = wrapAnsi16(ANSI_BACKGROUND_OFFSET);
	styles$1.bgColor.ansi256 = wrapAnsi256(ANSI_BACKGROUND_OFFSET);
	styles$1.bgColor.ansi16m = wrapAnsi16m(ANSI_BACKGROUND_OFFSET);

	// From https://github.com/Qix-/color-convert/blob/3f0e0d4e92e235796ccb17f6e85c72094a651f49/conversions.js
	Object.defineProperties(styles$1, {
		rgbToAnsi256: {
			value(red, green, blue) {
				// We use the extended greyscale palette here, with the exception of
				// black and white. normal palette only has 4 greyscale shades.
				if (red === green && green === blue) {
					if (red < 8) {
						return 16;
					}

					if (red > 248) {
						return 231;
					}

					return Math.round(((red - 8) / 247) * 24) + 232;
				}

				return 16
					+ (36 * Math.round(red / 255 * 5))
					+ (6 * Math.round(green / 255 * 5))
					+ Math.round(blue / 255 * 5);
			},
			enumerable: false,
		},
		hexToRgb: {
			value(hex) {
				const matches = /[a-f\d]{6}|[a-f\d]{3}/i.exec(hex.toString(16));
				if (!matches) {
					return [0, 0, 0];
				}

				let [colorString] = matches;

				if (colorString.length === 3) {
					colorString = [...colorString].map(character => character + character).join('');
				}

				const integer = Number.parseInt(colorString, 16);

				return [
					/* eslint-disable no-bitwise */
					(integer >> 16) & 0xFF,
					(integer >> 8) & 0xFF,
					integer & 0xFF,
					/* eslint-enable no-bitwise */
				];
			},
			enumerable: false,
		},
		hexToAnsi256: {
			value: hex => styles$1.rgbToAnsi256(...styles$1.hexToRgb(hex)),
			enumerable: false,
		},
		ansi256ToAnsi: {
			value(code) {
				if (code < 8) {
					return 30 + code;
				}

				if (code < 16) {
					return 90 + (code - 8);
				}

				let red;
				let green;
				let blue;

				if (code >= 232) {
					red = (((code - 232) * 10) + 8) / 255;
					green = red;
					blue = red;
				} else {
					code -= 16;

					const remainder = code % 36;

					red = Math.floor(code / 36) / 5;
					green = Math.floor(remainder / 6) / 5;
					blue = (remainder % 6) / 5;
				}

				const value = Math.max(red, green, blue) * 2;

				if (value === 0) {
					return 30;
				}

				// eslint-disable-next-line no-bitwise
				let result = 30 + ((Math.round(blue) << 2) | (Math.round(green) << 1) | Math.round(red));

				if (value === 2) {
					result += 60;
				}

				return result;
			},
			enumerable: false,
		},
		rgbToAnsi: {
			value: (red, green, blue) => styles$1.ansi256ToAnsi(styles$1.rgbToAnsi256(red, green, blue)),
			enumerable: false,
		},
		hexToAnsi: {
			value: hex => styles$1.ansi256ToAnsi(styles$1.hexToAnsi256(hex)),
			enumerable: false,
		},
	});

	return styles$1;
}

const ansiStyles$1 = assembleStyles();

// From: https://github.com/sindresorhus/has-flag/blob/main/index.js
/// function hasFlag(flag, argv = globalThis.Deno?.args ?? process.argv) {
function hasFlag$1(flag, argv = globalThis.Deno ? globalThis.Deno.args : process$1.argv) {
	const prefix = flag.startsWith('-') ? '' : (flag.length === 1 ? '-' : '--');
	const position = argv.indexOf(prefix + flag);
	const terminatorPosition = argv.indexOf('--');
	return position !== -1 && (terminatorPosition === -1 || position < terminatorPosition);
}

const {env} = process$1;

let flagForceColor;
if (
	hasFlag$1('no-color')
	|| hasFlag$1('no-colors')
	|| hasFlag$1('color=false')
	|| hasFlag$1('color=never')
) {
	flagForceColor = 0;
} else if (
	hasFlag$1('color')
	|| hasFlag$1('colors')
	|| hasFlag$1('color=true')
	|| hasFlag$1('color=always')
) {
	flagForceColor = 1;
}

function envForceColor() {
	if ('FORCE_COLOR' in env) {
		if (env.FORCE_COLOR === 'true') {
			return 1;
		}

		if (env.FORCE_COLOR === 'false') {
			return 0;
		}

		return env.FORCE_COLOR.length === 0 ? 1 : Math.min(Number.parseInt(env.FORCE_COLOR, 10), 3);
	}
}

function translateLevel(level) {
	if (level === 0) {
		return false;
	}

	return {
		level,
		hasBasic: true,
		has256: level >= 2,
		has16m: level >= 3,
	};
}

function _supportsColor(haveStream, {streamIsTTY, sniffFlags = true} = {}) {
	const noFlagForceColor = envForceColor();
	if (noFlagForceColor !== undefined) {
		flagForceColor = noFlagForceColor;
	}

	const forceColor = sniffFlags ? flagForceColor : noFlagForceColor;

	if (forceColor === 0) {
		return 0;
	}

	if (sniffFlags) {
		if (hasFlag$1('color=16m')
			|| hasFlag$1('color=full')
			|| hasFlag$1('color=truecolor')) {
			return 3;
		}

		if (hasFlag$1('color=256')) {
			return 2;
		}
	}

	// Check for Azure DevOps pipelines.
	// Has to be above the `!streamIsTTY` check.
	if ('TF_BUILD' in env && 'AGENT_NAME' in env) {
		return 1;
	}

	if (haveStream && !streamIsTTY && forceColor === undefined) {
		return 0;
	}

	const min = forceColor || 0;

	if (env.TERM === 'dumb') {
		return min;
	}

	if (process$1.platform === 'win32') {
		// Windows 10 build 10586 is the first Windows release that supports 256 colors.
		// Windows 10 build 14931 is the first release that supports 16m/TrueColor.
		const osRelease = os.release().split('.');
		if (
			Number(osRelease[0]) >= 10
			&& Number(osRelease[2]) >= 10_586
		) {
			return Number(osRelease[2]) >= 14_931 ? 3 : 2;
		}

		return 1;
	}

	if ('CI' in env) {
		if (['GITHUB_ACTIONS', 'GITEA_ACTIONS', 'CIRCLECI'].some(key => key in env)) {
			return 3;
		}

		if (['TRAVIS', 'APPVEYOR', 'GITLAB_CI', 'BUILDKITE', 'DRONE'].some(sign => sign in env) || env.CI_NAME === 'codeship') {
			return 1;
		}

		return min;
	}

	if ('TEAMCITY_VERSION' in env) {
		return /^(9\.(0*[1-9]\d*)\.|\d{2,}\.)/.test(env.TEAMCITY_VERSION) ? 1 : 0;
	}

	if (env.COLORTERM === 'truecolor') {
		return 3;
	}

	if (env.TERM === 'xterm-kitty') {
		return 3;
	}

	if ('TERM_PROGRAM' in env) {
		const version = Number.parseInt((env.TERM_PROGRAM_VERSION || '').split('.')[0], 10);

		switch (env.TERM_PROGRAM) {
			case 'iTerm.app': {
				return version >= 3 ? 3 : 2;
			}

			case 'Apple_Terminal': {
				return 2;
			}
			// No default
		}
	}

	if (/-256(color)?$/i.test(env.TERM)) {
		return 2;
	}

	if (/^screen|^xterm|^vt100|^vt220|^rxvt|color|ansi|cygwin|linux/i.test(env.TERM)) {
		return 1;
	}

	if ('COLORTERM' in env) {
		return 1;
	}

	return min;
}

function createSupportsColor(stream, options = {}) {
	const level = _supportsColor(stream, {
		streamIsTTY: stream && stream.isTTY,
		...options,
	});

	return translateLevel(level);
}

const supportsColor = {
	stdout: createSupportsColor({isTTY: tty.isatty(1)}),
	stderr: createSupportsColor({isTTY: tty.isatty(2)}),
};

// TODO: When targeting Node.js 16, use `String.prototype.replaceAll`.
function stringReplaceAll(string, substring, replacer) {
	let index = string.indexOf(substring);
	if (index === -1) {
		return string;
	}

	const substringLength = substring.length;
	let endIndex = 0;
	let returnValue = '';
	do {
		returnValue += string.slice(endIndex, index) + substring + replacer;
		endIndex = index + substringLength;
		index = string.indexOf(substring, endIndex);
	} while (index !== -1);

	returnValue += string.slice(endIndex);
	return returnValue;
}

function stringEncaseCRLFWithFirstIndex(string, prefix, postfix, index) {
	let endIndex = 0;
	let returnValue = '';
	do {
		const gotCR = string[index - 1] === '\r';
		returnValue += string.slice(endIndex, (gotCR ? index - 1 : index)) + prefix + (gotCR ? '\r\n' : '\n') + postfix;
		endIndex = index + 1;
		index = string.indexOf('\n', endIndex);
	} while (index !== -1);

	returnValue += string.slice(endIndex);
	return returnValue;
}

const {stdout: stdoutColor, stderr: stderrColor} = supportsColor;

const GENERATOR = Symbol('GENERATOR');
const STYLER = Symbol('STYLER');
const IS_EMPTY = Symbol('IS_EMPTY');

// `supportsColor.level` → `ansiStyles.color[name]` mapping
const levelMapping = [
	'ansi',
	'ansi',
	'ansi256',
	'ansi16m',
];

const styles = Object.create(null);

const applyOptions = (object, options = {}) => {
	if (options.level && !(Number.isInteger(options.level) && options.level >= 0 && options.level <= 3)) {
		throw new Error('The `level` option should be an integer from 0 to 3');
	}

	// Detect level if not set manually
	const colorLevel = stdoutColor ? stdoutColor.level : 0;
	object.level = options.level === undefined ? colorLevel : options.level;
};

const chalkFactory = options => {
	const chalk = (...strings) => strings.join(' ');
	applyOptions(chalk, options);

	Object.setPrototypeOf(chalk, createChalk.prototype);

	return chalk;
};

function createChalk(options) {
	return chalkFactory(options);
}

Object.setPrototypeOf(createChalk.prototype, Function.prototype);

for (const [styleName, style] of Object.entries(ansiStyles$1)) {
	styles[styleName] = {
		get() {
			const builder = createBuilder(this, createStyler(style.open, style.close, this[STYLER]), this[IS_EMPTY]);
			Object.defineProperty(this, styleName, {value: builder});
			return builder;
		},
	};
}

styles.visible = {
	get() {
		const builder = createBuilder(this, this[STYLER], true);
		Object.defineProperty(this, 'visible', {value: builder});
		return builder;
	},
};

const getModelAnsi = (model, level, type, ...arguments_) => {
	if (model === 'rgb') {
		if (level === 'ansi16m') {
			return ansiStyles$1[type].ansi16m(...arguments_);
		}

		if (level === 'ansi256') {
			return ansiStyles$1[type].ansi256(ansiStyles$1.rgbToAnsi256(...arguments_));
		}

		return ansiStyles$1[type].ansi(ansiStyles$1.rgbToAnsi(...arguments_));
	}

	if (model === 'hex') {
		return getModelAnsi('rgb', level, type, ...ansiStyles$1.hexToRgb(...arguments_));
	}

	return ansiStyles$1[type][model](...arguments_);
};

const usedModels = ['rgb', 'hex', 'ansi256'];

for (const model of usedModels) {
	styles[model] = {
		get() {
			const {level} = this;
			return function (...arguments_) {
				const styler = createStyler(getModelAnsi(model, levelMapping[level], 'color', ...arguments_), ansiStyles$1.color.close, this[STYLER]);
				return createBuilder(this, styler, this[IS_EMPTY]);
			};
		},
	};

	const bgModel = 'bg' + model[0].toUpperCase() + model.slice(1);
	styles[bgModel] = {
		get() {
			const {level} = this;
			return function (...arguments_) {
				const styler = createStyler(getModelAnsi(model, levelMapping[level], 'bgColor', ...arguments_), ansiStyles$1.bgColor.close, this[STYLER]);
				return createBuilder(this, styler, this[IS_EMPTY]);
			};
		},
	};
}

const proto = Object.defineProperties(() => {}, {
	...styles,
	level: {
		enumerable: true,
		get() {
			return this[GENERATOR].level;
		},
		set(level) {
			this[GENERATOR].level = level;
		},
	},
});

const createStyler = (open, close, parent) => {
	let openAll;
	let closeAll;
	if (parent === undefined) {
		openAll = open;
		closeAll = close;
	} else {
		openAll = parent.openAll + open;
		closeAll = close + parent.closeAll;
	}

	return {
		open,
		close,
		openAll,
		closeAll,
		parent,
	};
};

const createBuilder = (self, _styler, _isEmpty) => {
	// Single argument is hot path, implicit coercion is faster than anything
	// eslint-disable-next-line no-implicit-coercion
	const builder = (...arguments_) => applyStyle(builder, (arguments_.length === 1) ? ('' + arguments_[0]) : arguments_.join(' '));

	// We alter the prototype because we must return a function, but there is
	// no way to create a function with a different prototype
	Object.setPrototypeOf(builder, proto);

	builder[GENERATOR] = self;
	builder[STYLER] = _styler;
	builder[IS_EMPTY] = _isEmpty;

	return builder;
};

const applyStyle = (self, string) => {
	if (self.level <= 0 || !string) {
		return self[IS_EMPTY] ? '' : string;
	}

	let styler = self[STYLER];

	if (styler === undefined) {
		return string;
	}

	const {openAll, closeAll} = styler;
	if (string.includes('\u001B')) {
		while (styler !== undefined) {
			// Replace any instances already present with a re-opening code
			// otherwise only the part of the string until said closing code
			// will be colored, and the rest will simply be 'plain'.
			string = stringReplaceAll(string, styler.close, styler.open);

			styler = styler.parent;
		}
	}

	// We can move both next actions out of loop, because remaining actions in loop won't have
	// any/visible effect on parts we add here. Close the styling before a linebreak and reopen
	// after next line to fix a bleed issue on macOS: https://github.com/chalk/chalk/pull/92
	const lfIndex = string.indexOf('\n');
	if (lfIndex !== -1) {
		string = stringEncaseCRLFWithFirstIndex(string, closeAll, openAll, lfIndex);
	}

	return openAll + string + closeAll;
};

Object.defineProperties(createChalk.prototype, styles);

const chalk = createChalk();
createChalk({level: stderrColor ? stderrColor.level : 0});

var dist = {};

var core;
var hasRequiredCore;

function requireCore () {
	if (hasRequiredCore) return core;
	hasRequiredCore = 1;
	function deepFreeze(obj) {
	    if (obj instanceof Map) {
	        obj.clear = obj.delete = obj.set = function () {
	            throw new Error('map is read-only');
	        };
	    } else if (obj instanceof Set) {
	        obj.add = obj.clear = obj.delete = function () {
	            throw new Error('set is read-only');
	        };
	    }

	    // Freeze self
	    Object.freeze(obj);

	    Object.getOwnPropertyNames(obj).forEach(function (name) {
	        var prop = obj[name];

	        // Freeze prop if it is an object
	        if (typeof prop == 'object' && !Object.isFrozen(prop)) {
	            deepFreeze(prop);
	        }
	    });

	    return obj;
	}

	var deepFreezeEs6 = deepFreeze;
	var _default = deepFreeze;
	deepFreezeEs6.default = _default;

	/** @implements CallbackResponse */
	class Response {
	  /**
	   * @param {CompiledMode} mode
	   */
	  constructor(mode) {
	    // eslint-disable-next-line no-undefined
	    if (mode.data === undefined) mode.data = {};

	    this.data = mode.data;
	    this.isMatchIgnored = false;
	  }

	  ignoreMatch() {
	    this.isMatchIgnored = true;
	  }
	}

	/**
	 * @param {string} value
	 * @returns {string}
	 */
	function escapeHTML(value) {
	  return value
	    .replace(/&/g, '&amp;')
	    .replace(/</g, '&lt;')
	    .replace(/>/g, '&gt;')
	    .replace(/"/g, '&quot;')
	    .replace(/'/g, '&#x27;');
	}

	/**
	 * performs a shallow merge of multiple objects into one
	 *
	 * @template T
	 * @param {T} original
	 * @param {Record<string,any>[]} objects
	 * @returns {T} a single new object
	 */
	function inherit(original, ...objects) {
	  /** @type Record<string,any> */
	  const result = Object.create(null);

	  for (const key in original) {
	    result[key] = original[key];
	  }
	  objects.forEach(function(obj) {
	    for (const key in obj) {
	      result[key] = obj[key];
	    }
	  });
	  return /** @type {T} */ (result);
	}

	/**
	 * @typedef {object} Renderer
	 * @property {(text: string) => void} addText
	 * @property {(node: Node) => void} openNode
	 * @property {(node: Node) => void} closeNode
	 * @property {() => string} value
	 */

	/** @typedef {{kind?: string, sublanguage?: boolean}} Node */
	/** @typedef {{walk: (r: Renderer) => void}} Tree */
	/** */

	const SPAN_CLOSE = '</span>';

	/**
	 * Determines if a node needs to be wrapped in <span>
	 *
	 * @param {Node} node */
	const emitsWrappingTags = (node) => {
	  return !!node.kind;
	};

	/** @type {Renderer} */
	class HTMLRenderer {
	  /**
	   * Creates a new HTMLRenderer
	   *
	   * @param {Tree} parseTree - the parse tree (must support `walk` API)
	   * @param {{classPrefix: string}} options
	   */
	  constructor(parseTree, options) {
	    this.buffer = "";
	    this.classPrefix = options.classPrefix;
	    parseTree.walk(this);
	  }

	  /**
	   * Adds texts to the output stream
	   *
	   * @param {string} text */
	  addText(text) {
	    this.buffer += escapeHTML(text);
	  }

	  /**
	   * Adds a node open to the output stream (if needed)
	   *
	   * @param {Node} node */
	  openNode(node) {
	    if (!emitsWrappingTags(node)) return;

	    let className = node.kind;
	    if (!node.sublanguage) {
	      className = `${this.classPrefix}${className}`;
	    }
	    this.span(className);
	  }

	  /**
	   * Adds a node close to the output stream (if needed)
	   *
	   * @param {Node} node */
	  closeNode(node) {
	    if (!emitsWrappingTags(node)) return;

	    this.buffer += SPAN_CLOSE;
	  }

	  /**
	   * returns the accumulated buffer
	  */
	  value() {
	    return this.buffer;
	  }

	  // helpers

	  /**
	   * Builds a span element
	   *
	   * @param {string} className */
	  span(className) {
	    this.buffer += `<span class="${className}">`;
	  }
	}

	/** @typedef {{kind?: string, sublanguage?: boolean, children: Node[]} | string} Node */
	/** @typedef {{kind?: string, sublanguage?: boolean, children: Node[]} } DataNode */
	/**  */

	class TokenTree {
	  constructor() {
	    /** @type DataNode */
	    this.rootNode = { children: [] };
	    this.stack = [this.rootNode];
	  }

	  get top() {
	    return this.stack[this.stack.length - 1];
	  }

	  get root() { return this.rootNode; }

	  /** @param {Node} node */
	  add(node) {
	    this.top.children.push(node);
	  }

	  /** @param {string} kind */
	  openNode(kind) {
	    /** @type Node */
	    const node = { kind, children: [] };
	    this.add(node);
	    this.stack.push(node);
	  }

	  closeNode() {
	    if (this.stack.length > 1) {
	      return this.stack.pop();
	    }
	    // eslint-disable-next-line no-undefined
	    return undefined;
	  }

	  closeAllNodes() {
	    while (this.closeNode());
	  }

	  toJSON() {
	    return JSON.stringify(this.rootNode, null, 4);
	  }

	  /**
	   * @typedef { import("./html_renderer").Renderer } Renderer
	   * @param {Renderer} builder
	   */
	  walk(builder) {
	    // this does not
	    return this.constructor._walk(builder, this.rootNode);
	    // this works
	    // return TokenTree._walk(builder, this.rootNode);
	  }

	  /**
	   * @param {Renderer} builder
	   * @param {Node} node
	   */
	  static _walk(builder, node) {
	    if (typeof node === "string") {
	      builder.addText(node);
	    } else if (node.children) {
	      builder.openNode(node);
	      node.children.forEach((child) => this._walk(builder, child));
	      builder.closeNode(node);
	    }
	    return builder;
	  }

	  /**
	   * @param {Node} node
	   */
	  static _collapse(node) {
	    if (typeof node === "string") return;
	    if (!node.children) return;

	    if (node.children.every(el => typeof el === "string")) {
	      // node.text = node.children.join("");
	      // delete node.children;
	      node.children = [node.children.join("")];
	    } else {
	      node.children.forEach((child) => {
	        TokenTree._collapse(child);
	      });
	    }
	  }
	}

	/**
	  Currently this is all private API, but this is the minimal API necessary
	  that an Emitter must implement to fully support the parser.

	  Minimal interface:

	  - addKeyword(text, kind)
	  - addText(text)
	  - addSublanguage(emitter, subLanguageName)
	  - finalize()
	  - openNode(kind)
	  - closeNode()
	  - closeAllNodes()
	  - toHTML()

	*/

	/**
	 * @implements {Emitter}
	 */
	class TokenTreeEmitter extends TokenTree {
	  /**
	   * @param {*} options
	   */
	  constructor(options) {
	    super();
	    this.options = options;
	  }

	  /**
	   * @param {string} text
	   * @param {string} kind
	   */
	  addKeyword(text, kind) {
	    if (text === "") { return; }

	    this.openNode(kind);
	    this.addText(text);
	    this.closeNode();
	  }

	  /**
	   * @param {string} text
	   */
	  addText(text) {
	    if (text === "") { return; }

	    this.add(text);
	  }

	  /**
	   * @param {Emitter & {root: DataNode}} emitter
	   * @param {string} name
	   */
	  addSublanguage(emitter, name) {
	    /** @type DataNode */
	    const node = emitter.root;
	    node.kind = name;
	    node.sublanguage = true;
	    this.add(node);
	  }

	  toHTML() {
	    const renderer = new HTMLRenderer(this, this.options);
	    return renderer.value();
	  }

	  finalize() {
	    return true;
	  }
	}

	/**
	 * @param {string} value
	 * @returns {RegExp}
	 * */
	function escape(value) {
	  return new RegExp(value.replace(/[-/\\^$*+?.()|[\]{}]/g, '\\$&'), 'm');
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/**
	 * @param {RegExp} re
	 * @returns {number}
	 */
	function countMatchGroups(re) {
	  return (new RegExp(re.toString() + '|')).exec('').length - 1;
	}

	/**
	 * Does lexeme start with a regular expression match at the beginning
	 * @param {RegExp} re
	 * @param {string} lexeme
	 */
	function startsWith(re, lexeme) {
	  const match = re && re.exec(lexeme);
	  return match && match.index === 0;
	}

	// BACKREF_RE matches an open parenthesis or backreference. To avoid
	// an incorrect parse, it additionally matches the following:
	// - [...] elements, where the meaning of parentheses and escapes change
	// - other escape sequences, so we do not misparse escape sequences as
	//   interesting elements
	// - non-matching or lookahead parentheses, which do not capture. These
	//   follow the '(' with a '?'.
	const BACKREF_RE = /\[(?:[^\\\]]|\\.)*\]|\(\??|\\([1-9][0-9]*)|\\./;

	// join logically computes regexps.join(separator), but fixes the
	// backreferences so they continue to match.
	// it also places each individual regular expression into it's own
	// match group, keeping track of the sequencing of those match groups
	// is currently an exercise for the caller. :-)
	/**
	 * @param {(string | RegExp)[]} regexps
	 * @param {string} separator
	 * @returns {string}
	 */
	function join(regexps, separator = "|") {
	  let numCaptures = 0;

	  return regexps.map((regex) => {
	    numCaptures += 1;
	    const offset = numCaptures;
	    let re = source(regex);
	    let out = '';

	    while (re.length > 0) {
	      const match = BACKREF_RE.exec(re);
	      if (!match) {
	        out += re;
	        break;
	      }
	      out += re.substring(0, match.index);
	      re = re.substring(match.index + match[0].length);
	      if (match[0][0] === '\\' && match[1]) {
	        // Adjust the backreference.
	        out += '\\' + String(Number(match[1]) + offset);
	      } else {
	        out += match[0];
	        if (match[0] === '(') {
	          numCaptures++;
	        }
	      }
	    }
	    return out;
	  }).map(re => `(${re})`).join(separator);
	}

	// Common regexps
	const MATCH_NOTHING_RE = /\b\B/;
	const IDENT_RE = '[a-zA-Z]\\w*';
	const UNDERSCORE_IDENT_RE = '[a-zA-Z_]\\w*';
	const NUMBER_RE = '\\b\\d+(\\.\\d+)?';
	const C_NUMBER_RE = '(-?)(\\b0[xX][a-fA-F0-9]+|(\\b\\d+(\\.\\d*)?|\\.\\d+)([eE][-+]?\\d+)?)'; // 0x..., 0..., decimal, float
	const BINARY_NUMBER_RE = '\\b(0b[01]+)'; // 0b...
	const RE_STARTERS_RE = '!|!=|!==|%|%=|&|&&|&=|\\*|\\*=|\\+|\\+=|,|-|-=|/=|/|:|;|<<|<<=|<=|<|===|==|=|>>>=|>>=|>=|>>>|>>|>|\\?|\\[|\\{|\\(|\\^|\\^=|\\||\\|=|\\|\\||~';

	/**
	* @param { Partial<Mode> & {binary?: string | RegExp} } opts
	*/
	const SHEBANG = (opts = {}) => {
	  const beginShebang = /^#![ ]*\//;
	  if (opts.binary) {
	    opts.begin = concat(
	      beginShebang,
	      /.*\b/,
	      opts.binary,
	      /\b.*/);
	  }
	  return inherit({
	    className: 'meta',
	    begin: beginShebang,
	    end: /$/,
	    relevance: 0,
	    /** @type {ModeCallback} */
	    "on:begin": (m, resp) => {
	      if (m.index !== 0) resp.ignoreMatch();
	    }
	  }, opts);
	};

	// Common modes
	const BACKSLASH_ESCAPE = {
	  begin: '\\\\[\\s\\S]', relevance: 0
	};
	const APOS_STRING_MODE = {
	  className: 'string',
	  begin: '\'',
	  end: '\'',
	  illegal: '\\n',
	  contains: [BACKSLASH_ESCAPE]
	};
	const QUOTE_STRING_MODE = {
	  className: 'string',
	  begin: '"',
	  end: '"',
	  illegal: '\\n',
	  contains: [BACKSLASH_ESCAPE]
	};
	const PHRASAL_WORDS_MODE = {
	  begin: /\b(a|an|the|are|I'm|isn't|don't|doesn't|won't|but|just|should|pretty|simply|enough|gonna|going|wtf|so|such|will|you|your|they|like|more)\b/
	};
	/**
	 * Creates a comment mode
	 *
	 * @param {string | RegExp} begin
	 * @param {string | RegExp} end
	 * @param {Mode | {}} [modeOptions]
	 * @returns {Partial<Mode>}
	 */
	const COMMENT = function(begin, end, modeOptions = {}) {
	  const mode = inherit(
	    {
	      className: 'comment',
	      begin,
	      end,
	      contains: []
	    },
	    modeOptions
	  );
	  mode.contains.push(PHRASAL_WORDS_MODE);
	  mode.contains.push({
	    className: 'doctag',
	    begin: '(?:TODO|FIXME|NOTE|BUG|OPTIMIZE|HACK|XXX):',
	    relevance: 0
	  });
	  return mode;
	};
	const C_LINE_COMMENT_MODE = COMMENT('//', '$');
	const C_BLOCK_COMMENT_MODE = COMMENT('/\\*', '\\*/');
	const HASH_COMMENT_MODE = COMMENT('#', '$');
	const NUMBER_MODE = {
	  className: 'number',
	  begin: NUMBER_RE,
	  relevance: 0
	};
	const C_NUMBER_MODE = {
	  className: 'number',
	  begin: C_NUMBER_RE,
	  relevance: 0
	};
	const BINARY_NUMBER_MODE = {
	  className: 'number',
	  begin: BINARY_NUMBER_RE,
	  relevance: 0
	};
	const CSS_NUMBER_MODE = {
	  className: 'number',
	  begin: NUMBER_RE + '(' +
	    '%|em|ex|ch|rem' +
	    '|vw|vh|vmin|vmax' +
	    '|cm|mm|in|pt|pc|px' +
	    '|deg|grad|rad|turn' +
	    '|s|ms' +
	    '|Hz|kHz' +
	    '|dpi|dpcm|dppx' +
	    ')?',
	  relevance: 0
	};
	const REGEXP_MODE = {
	  // this outer rule makes sure we actually have a WHOLE regex and not simply
	  // an expression such as:
	  //
	  //     3 / something
	  //
	  // (which will then blow up when regex's `illegal` sees the newline)
	  begin: /(?=\/[^/\n]*\/)/,
	  contains: [{
	    className: 'regexp',
	    begin: /\//,
	    end: /\/[gimuy]*/,
	    illegal: /\n/,
	    contains: [
	      BACKSLASH_ESCAPE,
	      {
	        begin: /\[/,
	        end: /\]/,
	        relevance: 0,
	        contains: [BACKSLASH_ESCAPE]
	      }
	    ]
	  }]
	};
	const TITLE_MODE = {
	  className: 'title',
	  begin: IDENT_RE,
	  relevance: 0
	};
	const UNDERSCORE_TITLE_MODE = {
	  className: 'title',
	  begin: UNDERSCORE_IDENT_RE,
	  relevance: 0
	};
	const METHOD_GUARD = {
	  // excludes method names from keyword processing
	  begin: '\\.\\s*' + UNDERSCORE_IDENT_RE,
	  relevance: 0
	};

	/**
	 * Adds end same as begin mechanics to a mode
	 *
	 * Your mode must include at least a single () match group as that first match
	 * group is what is used for comparison
	 * @param {Partial<Mode>} mode
	 */
	const END_SAME_AS_BEGIN = function(mode) {
	  return Object.assign(mode,
	    {
	      /** @type {ModeCallback} */
	      'on:begin': (m, resp) => { resp.data._beginMatch = m[1]; },
	      /** @type {ModeCallback} */
	      'on:end': (m, resp) => { if (resp.data._beginMatch !== m[1]) resp.ignoreMatch(); }
	    });
	};

	var MODES = /*#__PURE__*/Object.freeze({
	    __proto__: null,
	    MATCH_NOTHING_RE: MATCH_NOTHING_RE,
	    IDENT_RE: IDENT_RE,
	    UNDERSCORE_IDENT_RE: UNDERSCORE_IDENT_RE,
	    NUMBER_RE: NUMBER_RE,
	    C_NUMBER_RE: C_NUMBER_RE,
	    BINARY_NUMBER_RE: BINARY_NUMBER_RE,
	    RE_STARTERS_RE: RE_STARTERS_RE,
	    SHEBANG: SHEBANG,
	    BACKSLASH_ESCAPE: BACKSLASH_ESCAPE,
	    APOS_STRING_MODE: APOS_STRING_MODE,
	    QUOTE_STRING_MODE: QUOTE_STRING_MODE,
	    PHRASAL_WORDS_MODE: PHRASAL_WORDS_MODE,
	    COMMENT: COMMENT,
	    C_LINE_COMMENT_MODE: C_LINE_COMMENT_MODE,
	    C_BLOCK_COMMENT_MODE: C_BLOCK_COMMENT_MODE,
	    HASH_COMMENT_MODE: HASH_COMMENT_MODE,
	    NUMBER_MODE: NUMBER_MODE,
	    C_NUMBER_MODE: C_NUMBER_MODE,
	    BINARY_NUMBER_MODE: BINARY_NUMBER_MODE,
	    CSS_NUMBER_MODE: CSS_NUMBER_MODE,
	    REGEXP_MODE: REGEXP_MODE,
	    TITLE_MODE: TITLE_MODE,
	    UNDERSCORE_TITLE_MODE: UNDERSCORE_TITLE_MODE,
	    METHOD_GUARD: METHOD_GUARD,
	    END_SAME_AS_BEGIN: END_SAME_AS_BEGIN
	});

	// Grammar extensions / plugins
	// See: https://github.com/highlightjs/highlight.js/issues/2833

	// Grammar extensions allow "syntactic sugar" to be added to the grammar modes
	// without requiring any underlying changes to the compiler internals.

	// `compileMatch` being the perfect small example of now allowing a grammar
	// author to write `match` when they desire to match a single expression rather
	// than being forced to use `begin`.  The extension then just moves `match` into
	// `begin` when it runs.  Ie, no features have been added, but we've just made
	// the experience of writing (and reading grammars) a little bit nicer.

	// ------

	// TODO: We need negative look-behind support to do this properly
	/**
	 * Skip a match if it has a preceding dot
	 *
	 * This is used for `beginKeywords` to prevent matching expressions such as
	 * `bob.keyword.do()`. The mode compiler automatically wires this up as a
	 * special _internal_ 'on:begin' callback for modes with `beginKeywords`
	 * @param {RegExpMatchArray} match
	 * @param {CallbackResponse} response
	 */
	function skipIfhasPrecedingDot(match, response) {
	  const before = match.input[match.index - 1];
	  if (before === ".") {
	    response.ignoreMatch();
	  }
	}


	/**
	 * `beginKeywords` syntactic sugar
	 * @type {CompilerExt}
	 */
	function beginKeywords(mode, parent) {
	  if (!parent) return;
	  if (!mode.beginKeywords) return;

	  // for languages with keywords that include non-word characters checking for
	  // a word boundary is not sufficient, so instead we check for a word boundary
	  // or whitespace - this does no harm in any case since our keyword engine
	  // doesn't allow spaces in keywords anyways and we still check for the boundary
	  // first
	  mode.begin = '\\b(' + mode.beginKeywords.split(' ').join('|') + ')(?!\\.)(?=\\b|\\s)';
	  mode.__beforeBegin = skipIfhasPrecedingDot;
	  mode.keywords = mode.keywords || mode.beginKeywords;
	  delete mode.beginKeywords;

	  // prevents double relevance, the keywords themselves provide
	  // relevance, the mode doesn't need to double it
	  // eslint-disable-next-line no-undefined
	  if (mode.relevance === undefined) mode.relevance = 0;
	}

	/**
	 * Allow `illegal` to contain an array of illegal values
	 * @type {CompilerExt}
	 */
	function compileIllegal(mode, _parent) {
	  if (!Array.isArray(mode.illegal)) return;

	  mode.illegal = either(...mode.illegal);
	}

	/**
	 * `match` to match a single expression for readability
	 * @type {CompilerExt}
	 */
	function compileMatch(mode, _parent) {
	  if (!mode.match) return;
	  if (mode.begin || mode.end) throw new Error("begin & end are not supported with match");

	  mode.begin = mode.match;
	  delete mode.match;
	}

	/**
	 * provides the default 1 relevance to all modes
	 * @type {CompilerExt}
	 */
	function compileRelevance(mode, _parent) {
	  // eslint-disable-next-line no-undefined
	  if (mode.relevance === undefined) mode.relevance = 1;
	}

	// keywords that should have no default relevance value
	const COMMON_KEYWORDS = [
	  'of',
	  'and',
	  'for',
	  'in',
	  'not',
	  'or',
	  'if',
	  'then',
	  'parent', // common variable name
	  'list', // common variable name
	  'value' // common variable name
	];

	const DEFAULT_KEYWORD_CLASSNAME = "keyword";

	/**
	 * Given raw keywords from a language definition, compile them.
	 *
	 * @param {string | Record<string,string|string[]> | Array<string>} rawKeywords
	 * @param {boolean} caseInsensitive
	 */
	function compileKeywords(rawKeywords, caseInsensitive, className = DEFAULT_KEYWORD_CLASSNAME) {
	  /** @type KeywordDict */
	  const compiledKeywords = {};

	  // input can be a string of keywords, an array of keywords, or a object with
	  // named keys representing className (which can then point to a string or array)
	  if (typeof rawKeywords === 'string') {
	    compileList(className, rawKeywords.split(" "));
	  } else if (Array.isArray(rawKeywords)) {
	    compileList(className, rawKeywords);
	  } else {
	    Object.keys(rawKeywords).forEach(function(className) {
	      // collapse all our objects back into the parent object
	      Object.assign(
	        compiledKeywords,
	        compileKeywords(rawKeywords[className], caseInsensitive, className)
	      );
	    });
	  }
	  return compiledKeywords;

	  // ---

	  /**
	   * Compiles an individual list of keywords
	   *
	   * Ex: "for if when while|5"
	   *
	   * @param {string} className
	   * @param {Array<string>} keywordList
	   */
	  function compileList(className, keywordList) {
	    if (caseInsensitive) {
	      keywordList = keywordList.map(x => x.toLowerCase());
	    }
	    keywordList.forEach(function(keyword) {
	      const pair = keyword.split('|');
	      compiledKeywords[pair[0]] = [className, scoreForKeyword(pair[0], pair[1])];
	    });
	  }
	}

	/**
	 * Returns the proper score for a given keyword
	 *
	 * Also takes into account comment keywords, which will be scored 0 UNLESS
	 * another score has been manually assigned.
	 * @param {string} keyword
	 * @param {string} [providedScore]
	 */
	function scoreForKeyword(keyword, providedScore) {
	  // manual scores always win over common keywords
	  // so you can force a score of 1 if you really insist
	  if (providedScore) {
	    return Number(providedScore);
	  }

	  return commonKeyword(keyword) ? 0 : 1;
	}

	/**
	 * Determines if a given keyword is common or not
	 *
	 * @param {string} keyword */
	function commonKeyword(keyword) {
	  return COMMON_KEYWORDS.includes(keyword.toLowerCase());
	}

	// compilation

	/**
	 * Compiles a language definition result
	 *
	 * Given the raw result of a language definition (Language), compiles this so
	 * that it is ready for highlighting code.
	 * @param {Language} language
	 * @param {{plugins: HLJSPlugin[]}} opts
	 * @returns {CompiledLanguage}
	 */
	function compileLanguage(language, { plugins }) {
	  /**
	   * Builds a regex with the case sensativility of the current language
	   *
	   * @param {RegExp | string} value
	   * @param {boolean} [global]
	   */
	  function langRe(value, global) {
	    return new RegExp(
	      source(value),
	      'm' + (language.case_insensitive ? 'i' : '') + (global ? 'g' : '')
	    );
	  }

	  /**
	    Stores multiple regular expressions and allows you to quickly search for
	    them all in a string simultaneously - returning the first match.  It does
	    this by creating a huge (a|b|c) regex - each individual item wrapped with ()
	    and joined by `|` - using match groups to track position.  When a match is
	    found checking which position in the array has content allows us to figure
	    out which of the original regexes / match groups triggered the match.

	    The match object itself (the result of `Regex.exec`) is returned but also
	    enhanced by merging in any meta-data that was registered with the regex.
	    This is how we keep track of which mode matched, and what type of rule
	    (`illegal`, `begin`, end, etc).
	  */
	  class MultiRegex {
	    constructor() {
	      this.matchIndexes = {};
	      // @ts-ignore
	      this.regexes = [];
	      this.matchAt = 1;
	      this.position = 0;
	    }

	    // @ts-ignore
	    addRule(re, opts) {
	      opts.position = this.position++;
	      // @ts-ignore
	      this.matchIndexes[this.matchAt] = opts;
	      this.regexes.push([opts, re]);
	      this.matchAt += countMatchGroups(re) + 1;
	    }

	    compile() {
	      if (this.regexes.length === 0) {
	        // avoids the need to check length every time exec is called
	        // @ts-ignore
	        this.exec = () => null;
	      }
	      const terminators = this.regexes.map(el => el[1]);
	      this.matcherRe = langRe(join(terminators), true);
	      this.lastIndex = 0;
	    }

	    /** @param {string} s */
	    exec(s) {
	      this.matcherRe.lastIndex = this.lastIndex;
	      const match = this.matcherRe.exec(s);
	      if (!match) { return null; }

	      // eslint-disable-next-line no-undefined
	      const i = match.findIndex((el, i) => i > 0 && el !== undefined);
	      // @ts-ignore
	      const matchData = this.matchIndexes[i];
	      // trim off any earlier non-relevant match groups (ie, the other regex
	      // match groups that make up the multi-matcher)
	      match.splice(0, i);

	      return Object.assign(match, matchData);
	    }
	  }

	  /*
	    Created to solve the key deficiently with MultiRegex - there is no way to
	    test for multiple matches at a single location.  Why would we need to do
	    that?  In the future a more dynamic engine will allow certain matches to be
	    ignored.  An example: if we matched say the 3rd regex in a large group but
	    decided to ignore it - we'd need to started testing again at the 4th
	    regex... but MultiRegex itself gives us no real way to do that.

	    So what this class creates MultiRegexs on the fly for whatever search
	    position they are needed.

	    NOTE: These additional MultiRegex objects are created dynamically.  For most
	    grammars most of the time we will never actually need anything more than the
	    first MultiRegex - so this shouldn't have too much overhead.

	    Say this is our search group, and we match regex3, but wish to ignore it.

	      regex1 | regex2 | regex3 | regex4 | regex5    ' ie, startAt = 0

	    What we need is a new MultiRegex that only includes the remaining
	    possibilities:

	      regex4 | regex5                               ' ie, startAt = 3

	    This class wraps all that complexity up in a simple API... `startAt` decides
	    where in the array of expressions to start doing the matching. It
	    auto-increments, so if a match is found at position 2, then startAt will be
	    set to 3.  If the end is reached startAt will return to 0.

	    MOST of the time the parser will be setting startAt manually to 0.
	  */
	  class ResumableMultiRegex {
	    constructor() {
	      // @ts-ignore
	      this.rules = [];
	      // @ts-ignore
	      this.multiRegexes = [];
	      this.count = 0;

	      this.lastIndex = 0;
	      this.regexIndex = 0;
	    }

	    // @ts-ignore
	    getMatcher(index) {
	      if (this.multiRegexes[index]) return this.multiRegexes[index];

	      const matcher = new MultiRegex();
	      this.rules.slice(index).forEach(([re, opts]) => matcher.addRule(re, opts));
	      matcher.compile();
	      this.multiRegexes[index] = matcher;
	      return matcher;
	    }

	    resumingScanAtSamePosition() {
	      return this.regexIndex !== 0;
	    }

	    considerAll() {
	      this.regexIndex = 0;
	    }

	    // @ts-ignore
	    addRule(re, opts) {
	      this.rules.push([re, opts]);
	      if (opts.type === "begin") this.count++;
	    }

	    /** @param {string} s */
	    exec(s) {
	      const m = this.getMatcher(this.regexIndex);
	      m.lastIndex = this.lastIndex;
	      let result = m.exec(s);

	      // The following is because we have no easy way to say "resume scanning at the
	      // existing position but also skip the current rule ONLY". What happens is
	      // all prior rules are also skipped which can result in matching the wrong
	      // thing. Example of matching "booger":

	      // our matcher is [string, "booger", number]
	      //
	      // ....booger....

	      // if "booger" is ignored then we'd really need a regex to scan from the
	      // SAME position for only: [string, number] but ignoring "booger" (if it
	      // was the first match), a simple resume would scan ahead who knows how
	      // far looking only for "number", ignoring potential string matches (or
	      // future "booger" matches that might be valid.)

	      // So what we do: We execute two matchers, one resuming at the same
	      // position, but the second full matcher starting at the position after:

	      //     /--- resume first regex match here (for [number])
	      //     |/---- full match here for [string, "booger", number]
	      //     vv
	      // ....booger....

	      // Which ever results in a match first is then used. So this 3-4 step
	      // process essentially allows us to say "match at this position, excluding
	      // a prior rule that was ignored".
	      //
	      // 1. Match "booger" first, ignore. Also proves that [string] does non match.
	      // 2. Resume matching for [number]
	      // 3. Match at index + 1 for [string, "booger", number]
	      // 4. If #2 and #3 result in matches, which came first?
	      if (this.resumingScanAtSamePosition()) {
	        if (result && result.index === this.lastIndex) ; else { // use the second matcher result
	          const m2 = this.getMatcher(0);
	          m2.lastIndex = this.lastIndex + 1;
	          result = m2.exec(s);
	        }
	      }

	      if (result) {
	        this.regexIndex += result.position + 1;
	        if (this.regexIndex === this.count) {
	          // wrap-around to considering all matches again
	          this.considerAll();
	        }
	      }

	      return result;
	    }
	  }

	  /**
	   * Given a mode, builds a huge ResumableMultiRegex that can be used to walk
	   * the content and find matches.
	   *
	   * @param {CompiledMode} mode
	   * @returns {ResumableMultiRegex}
	   */
	  function buildModeRegex(mode) {
	    const mm = new ResumableMultiRegex();

	    mode.contains.forEach(term => mm.addRule(term.begin, { rule: term, type: "begin" }));

	    if (mode.terminatorEnd) {
	      mm.addRule(mode.terminatorEnd, { type: "end" });
	    }
	    if (mode.illegal) {
	      mm.addRule(mode.illegal, { type: "illegal" });
	    }

	    return mm;
	  }

	  /** skip vs abort vs ignore
	   *
	   * @skip   - The mode is still entered and exited normally (and contains rules apply),
	   *           but all content is held and added to the parent buffer rather than being
	   *           output when the mode ends.  Mostly used with `sublanguage` to build up
	   *           a single large buffer than can be parsed by sublanguage.
	   *
	   *             - The mode begin ands ends normally.
	   *             - Content matched is added to the parent mode buffer.
	   *             - The parser cursor is moved forward normally.
	   *
	   * @abort  - A hack placeholder until we have ignore.  Aborts the mode (as if it
	   *           never matched) but DOES NOT continue to match subsequent `contains`
	   *           modes.  Abort is bad/suboptimal because it can result in modes
	   *           farther down not getting applied because an earlier rule eats the
	   *           content but then aborts.
	   *
	   *             - The mode does not begin.
	   *             - Content matched by `begin` is added to the mode buffer.
	   *             - The parser cursor is moved forward accordingly.
	   *
	   * @ignore - Ignores the mode (as if it never matched) and continues to match any
	   *           subsequent `contains` modes.  Ignore isn't technically possible with
	   *           the current parser implementation.
	   *
	   *             - The mode does not begin.
	   *             - Content matched by `begin` is ignored.
	   *             - The parser cursor is not moved forward.
	   */

	  /**
	   * Compiles an individual mode
	   *
	   * This can raise an error if the mode contains certain detectable known logic
	   * issues.
	   * @param {Mode} mode
	   * @param {CompiledMode | null} [parent]
	   * @returns {CompiledMode | never}
	   */
	  function compileMode(mode, parent) {
	    const cmode = /** @type CompiledMode */ (mode);
	    if (mode.isCompiled) return cmode;

	    [
	      // do this early so compiler extensions generally don't have to worry about
	      // the distinction between match/begin
	      compileMatch
	    ].forEach(ext => ext(mode, parent));

	    language.compilerExtensions.forEach(ext => ext(mode, parent));

	    // __beforeBegin is considered private API, internal use only
	    mode.__beforeBegin = null;

	    [
	      beginKeywords,
	      // do this later so compiler extensions that come earlier have access to the
	      // raw array if they wanted to perhaps manipulate it, etc.
	      compileIllegal,
	      // default to 1 relevance if not specified
	      compileRelevance
	    ].forEach(ext => ext(mode, parent));

	    mode.isCompiled = true;

	    let keywordPattern = null;
	    if (typeof mode.keywords === "object") {
	      keywordPattern = mode.keywords.$pattern;
	      delete mode.keywords.$pattern;
	    }

	    if (mode.keywords) {
	      mode.keywords = compileKeywords(mode.keywords, language.case_insensitive);
	    }

	    // both are not allowed
	    if (mode.lexemes && keywordPattern) {
	      throw new Error("ERR: Prefer `keywords.$pattern` to `mode.lexemes`, BOTH are not allowed. (see mode reference) ");
	    }

	    // `mode.lexemes` was the old standard before we added and now recommend
	    // using `keywords.$pattern` to pass the keyword pattern
	    keywordPattern = keywordPattern || mode.lexemes || /\w+/;
	    cmode.keywordPatternRe = langRe(keywordPattern, true);

	    if (parent) {
	      if (!mode.begin) mode.begin = /\B|\b/;
	      cmode.beginRe = langRe(mode.begin);
	      if (mode.endSameAsBegin) mode.end = mode.begin;
	      if (!mode.end && !mode.endsWithParent) mode.end = /\B|\b/;
	      if (mode.end) cmode.endRe = langRe(mode.end);
	      cmode.terminatorEnd = source(mode.end) || '';
	      if (mode.endsWithParent && parent.terminatorEnd) {
	        cmode.terminatorEnd += (mode.end ? '|' : '') + parent.terminatorEnd;
	      }
	    }
	    if (mode.illegal) cmode.illegalRe = langRe(/** @type {RegExp | string} */ (mode.illegal));
	    if (!mode.contains) mode.contains = [];

	    mode.contains = [].concat(...mode.contains.map(function(c) {
	      return expandOrCloneMode(c === 'self' ? mode : c);
	    }));
	    mode.contains.forEach(function(c) { compileMode(/** @type Mode */ (c), cmode); });

	    if (mode.starts) {
	      compileMode(mode.starts, parent);
	    }

	    cmode.matcher = buildModeRegex(cmode);
	    return cmode;
	  }

	  if (!language.compilerExtensions) language.compilerExtensions = [];

	  // self is not valid at the top-level
	  if (language.contains && language.contains.includes('self')) {
	    throw new Error("ERR: contains `self` is not supported at the top-level of a language.  See documentation.");
	  }

	  // we need a null object, which inherit will guarantee
	  language.classNameAliases = inherit(language.classNameAliases || {});

	  return compileMode(/** @type Mode */ (language));
	}

	/**
	 * Determines if a mode has a dependency on it's parent or not
	 *
	 * If a mode does have a parent dependency then often we need to clone it if
	 * it's used in multiple places so that each copy points to the correct parent,
	 * where-as modes without a parent can often safely be re-used at the bottom of
	 * a mode chain.
	 *
	 * @param {Mode | null} mode
	 * @returns {boolean} - is there a dependency on the parent?
	 * */
	function dependencyOnParent(mode) {
	  if (!mode) return false;

	  return mode.endsWithParent || dependencyOnParent(mode.starts);
	}

	/**
	 * Expands a mode or clones it if necessary
	 *
	 * This is necessary for modes with parental dependenceis (see notes on
	 * `dependencyOnParent`) and for nodes that have `variants` - which must then be
	 * exploded into their own individual modes at compile time.
	 *
	 * @param {Mode} mode
	 * @returns {Mode | Mode[]}
	 * */
	function expandOrCloneMode(mode) {
	  if (mode.variants && !mode.cachedVariants) {
	    mode.cachedVariants = mode.variants.map(function(variant) {
	      return inherit(mode, { variants: null }, variant);
	    });
	  }

	  // EXPAND
	  // if we have variants then essentially "replace" the mode with the variants
	  // this happens in compileMode, where this function is called from
	  if (mode.cachedVariants) {
	    return mode.cachedVariants;
	  }

	  // CLONE
	  // if we have dependencies on parents then we need a unique
	  // instance of ourselves, so we can be reused with many
	  // different parents without issue
	  if (dependencyOnParent(mode)) {
	    return inherit(mode, { starts: mode.starts ? inherit(mode.starts) : null });
	  }

	  if (Object.isFrozen(mode)) {
	    return inherit(mode);
	  }

	  // no special dependency issues, just return ourselves
	  return mode;
	}

	var version = "10.7.3";

	// @ts-nocheck

	function hasValueOrEmptyAttribute(value) {
	  return Boolean(value || value === "");
	}

	function BuildVuePlugin(hljs) {
	  const Component = {
	    props: ["language", "code", "autodetect"],
	    data: function() {
	      return {
	        detectedLanguage: "",
	        unknownLanguage: false
	      };
	    },
	    computed: {
	      className() {
	        if (this.unknownLanguage) return "";

	        return "hljs " + this.detectedLanguage;
	      },
	      highlighted() {
	        // no idea what language to use, return raw code
	        if (!this.autoDetect && !hljs.getLanguage(this.language)) {
	          console.warn(`The language "${this.language}" you specified could not be found.`);
	          this.unknownLanguage = true;
	          return escapeHTML(this.code);
	        }

	        let result = {};
	        if (this.autoDetect) {
	          result = hljs.highlightAuto(this.code);
	          this.detectedLanguage = result.language;
	        } else {
	          result = hljs.highlight(this.language, this.code, this.ignoreIllegals);
	          this.detectedLanguage = this.language;
	        }
	        return result.value;
	      },
	      autoDetect() {
	        return !this.language || hasValueOrEmptyAttribute(this.autodetect);
	      },
	      ignoreIllegals() {
	        return true;
	      }
	    },
	    // this avoids needing to use a whole Vue compilation pipeline just
	    // to build Highlight.js
	    render(createElement) {
	      return createElement("pre", {}, [
	        createElement("code", {
	          class: this.className,
	          domProps: { innerHTML: this.highlighted }
	        })
	      ]);
	    }
	    // template: `<pre><code :class="className" v-html="highlighted"></code></pre>`
	  };

	  const VuePlugin = {
	    install(Vue) {
	      Vue.component('highlightjs', Component);
	    }
	  };

	  return { Component, VuePlugin };
	}

	/* plugin itself */

	/** @type {HLJSPlugin} */
	const mergeHTMLPlugin = {
	  "after:highlightElement": ({ el, result, text }) => {
	    const originalStream = nodeStream(el);
	    if (!originalStream.length) return;

	    const resultNode = document.createElement('div');
	    resultNode.innerHTML = result.value;
	    result.value = mergeStreams(originalStream, nodeStream(resultNode), text);
	  }
	};

	/* Stream merging support functions */

	/**
	 * @typedef Event
	 * @property {'start'|'stop'} event
	 * @property {number} offset
	 * @property {Node} node
	 */

	/**
	 * @param {Node} node
	 */
	function tag(node) {
	  return node.nodeName.toLowerCase();
	}

	/**
	 * @param {Node} node
	 */
	function nodeStream(node) {
	  /** @type Event[] */
	  const result = [];
	  (function _nodeStream(node, offset) {
	    for (let child = node.firstChild; child; child = child.nextSibling) {
	      if (child.nodeType === 3) {
	        offset += child.nodeValue.length;
	      } else if (child.nodeType === 1) {
	        result.push({
	          event: 'start',
	          offset: offset,
	          node: child
	        });
	        offset = _nodeStream(child, offset);
	        // Prevent void elements from having an end tag that would actually
	        // double them in the output. There are more void elements in HTML
	        // but we list only those realistically expected in code display.
	        if (!tag(child).match(/br|hr|img|input/)) {
	          result.push({
	            event: 'stop',
	            offset: offset,
	            node: child
	          });
	        }
	      }
	    }
	    return offset;
	  })(node, 0);
	  return result;
	}

	/**
	 * @param {any} original - the original stream
	 * @param {any} highlighted - stream of the highlighted source
	 * @param {string} value - the original source itself
	 */
	function mergeStreams(original, highlighted, value) {
	  let processed = 0;
	  let result = '';
	  const nodeStack = [];

	  function selectStream() {
	    if (!original.length || !highlighted.length) {
	      return original.length ? original : highlighted;
	    }
	    if (original[0].offset !== highlighted[0].offset) {
	      return (original[0].offset < highlighted[0].offset) ? original : highlighted;
	    }

	    /*
	    To avoid starting the stream just before it should stop the order is
	    ensured that original always starts first and closes last:

	    if (event1 == 'start' && event2 == 'start')
	      return original;
	    if (event1 == 'start' && event2 == 'stop')
	      return highlighted;
	    if (event1 == 'stop' && event2 == 'start')
	      return original;
	    if (event1 == 'stop' && event2 == 'stop')
	      return highlighted;

	    ... which is collapsed to:
	    */
	    return highlighted[0].event === 'start' ? original : highlighted;
	  }

	  /**
	   * @param {Node} node
	   */
	  function open(node) {
	    /** @param {Attr} attr */
	    function attributeString(attr) {
	      return ' ' + attr.nodeName + '="' + escapeHTML(attr.value) + '"';
	    }
	    // @ts-ignore
	    result += '<' + tag(node) + [].map.call(node.attributes, attributeString).join('') + '>';
	  }

	  /**
	   * @param {Node} node
	   */
	  function close(node) {
	    result += '</' + tag(node) + '>';
	  }

	  /**
	   * @param {Event} event
	   */
	  function render(event) {
	    (event.event === 'start' ? open : close)(event.node);
	  }

	  while (original.length || highlighted.length) {
	    let stream = selectStream();
	    result += escapeHTML(value.substring(processed, stream[0].offset));
	    processed = stream[0].offset;
	    if (stream === original) {
	      /*
	      On any opening or closing tag of the original markup we first close
	      the entire highlighted node stack, then render the original tag along
	      with all the following original tags at the same offset and then
	      reopen all the tags on the highlighted stack.
	      */
	      nodeStack.reverse().forEach(close);
	      do {
	        render(stream.splice(0, 1)[0]);
	        stream = selectStream();
	      } while (stream === original && stream.length && stream[0].offset === processed);
	      nodeStack.reverse().forEach(open);
	    } else {
	      if (stream[0].event === 'start') {
	        nodeStack.push(stream[0].node);
	      } else {
	        nodeStack.pop();
	      }
	      render(stream.splice(0, 1)[0]);
	    }
	  }
	  return result + escapeHTML(value.substr(processed));
	}

	/*

	For the reasoning behind this please see:
	https://github.com/highlightjs/highlight.js/issues/2880#issuecomment-747275419

	*/

	/**
	 * @type {Record<string, boolean>}
	 */
	const seenDeprecations = {};

	/**
	 * @param {string} message
	 */
	const error = (message) => {
	  console.error(message);
	};

	/**
	 * @param {string} message
	 * @param {any} args
	 */
	const warn = (message, ...args) => {
	  console.log(`WARN: ${message}`, ...args);
	};

	/**
	 * @param {string} version
	 * @param {string} message
	 */
	const deprecated = (version, message) => {
	  if (seenDeprecations[`${version}/${message}`]) return;

	  console.log(`Deprecated as of ${version}. ${message}`);
	  seenDeprecations[`${version}/${message}`] = true;
	};

	/*
	Syntax highlighting with language autodetection.
	https://highlightjs.org/
	*/

	const escape$1 = escapeHTML;
	const inherit$1 = inherit;
	const NO_MATCH = Symbol("nomatch");

	/**
	 * @param {any} hljs - object that is extended (legacy)
	 * @returns {HLJSApi}
	 */
	const HLJS = function(hljs) {
	  // Global internal variables used within the highlight.js library.
	  /** @type {Record<string, Language>} */
	  const languages = Object.create(null);
	  /** @type {Record<string, string>} */
	  const aliases = Object.create(null);
	  /** @type {HLJSPlugin[]} */
	  const plugins = [];

	  // safe/production mode - swallows more errors, tries to keep running
	  // even if a single syntax or parse hits a fatal error
	  let SAFE_MODE = true;
	  const fixMarkupRe = /(^(<[^>]+>|\t|)+|\n)/gm;
	  const LANGUAGE_NOT_FOUND = "Could not find the language '{}', did you forget to load/include a language module?";
	  /** @type {Language} */
	  const PLAINTEXT_LANGUAGE = { disableAutodetect: true, name: 'Plain text', contains: [] };

	  // Global options used when within external APIs. This is modified when
	  // calling the `hljs.configure` function.
	  /** @type HLJSOptions */
	  let options = {
	    noHighlightRe: /^(no-?highlight)$/i,
	    languageDetectRe: /\blang(?:uage)?-([\w-]+)\b/i,
	    classPrefix: 'hljs-',
	    tabReplace: null,
	    useBR: false,
	    languages: null,
	    // beta configuration options, subject to change, welcome to discuss
	    // https://github.com/highlightjs/highlight.js/issues/1086
	    __emitter: TokenTreeEmitter
	  };

	  /* Utility functions */

	  /**
	   * Tests a language name to see if highlighting should be skipped
	   * @param {string} languageName
	   */
	  function shouldNotHighlight(languageName) {
	    return options.noHighlightRe.test(languageName);
	  }

	  /**
	   * @param {HighlightedHTMLElement} block - the HTML element to determine language for
	   */
	  function blockLanguage(block) {
	    let classes = block.className + ' ';

	    classes += block.parentNode ? block.parentNode.className : '';

	    // language-* takes precedence over non-prefixed class names.
	    const match = options.languageDetectRe.exec(classes);
	    if (match) {
	      const language = getLanguage(match[1]);
	      if (!language) {
	        warn(LANGUAGE_NOT_FOUND.replace("{}", match[1]));
	        warn("Falling back to no-highlight mode for this block.", block);
	      }
	      return language ? match[1] : 'no-highlight';
	    }

	    return classes
	      .split(/\s+/)
	      .find((_class) => shouldNotHighlight(_class) || getLanguage(_class));
	  }

	  /**
	   * Core highlighting function.
	   *
	   * OLD API
	   * highlight(lang, code, ignoreIllegals, continuation)
	   *
	   * NEW API
	   * highlight(code, {lang, ignoreIllegals})
	   *
	   * @param {string} codeOrlanguageName - the language to use for highlighting
	   * @param {string | HighlightOptions} optionsOrCode - the code to highlight
	   * @param {boolean} [ignoreIllegals] - whether to ignore illegal matches, default is to bail
	   * @param {CompiledMode} [continuation] - current continuation mode, if any
	   *
	   * @returns {HighlightResult} Result - an object that represents the result
	   * @property {string} language - the language name
	   * @property {number} relevance - the relevance score
	   * @property {string} value - the highlighted HTML code
	   * @property {string} code - the original raw code
	   * @property {CompiledMode} top - top of the current mode stack
	   * @property {boolean} illegal - indicates whether any illegal matches were found
	  */
	  function highlight(codeOrlanguageName, optionsOrCode, ignoreIllegals, continuation) {
	    let code = "";
	    let languageName = "";
	    if (typeof optionsOrCode === "object") {
	      code = codeOrlanguageName;
	      ignoreIllegals = optionsOrCode.ignoreIllegals;
	      languageName = optionsOrCode.language;
	      // continuation not supported at all via the new API
	      // eslint-disable-next-line no-undefined
	      continuation = undefined;
	    } else {
	      // old API
	      deprecated("10.7.0", "highlight(lang, code, ...args) has been deprecated.");
	      deprecated("10.7.0", "Please use highlight(code, options) instead.\nhttps://github.com/highlightjs/highlight.js/issues/2277");
	      languageName = codeOrlanguageName;
	      code = optionsOrCode;
	    }

	    /** @type {BeforeHighlightContext} */
	    const context = {
	      code,
	      language: languageName
	    };
	    // the plugin can change the desired language or the code to be highlighted
	    // just be changing the object it was passed
	    fire("before:highlight", context);

	    // a before plugin can usurp the result completely by providing it's own
	    // in which case we don't even need to call highlight
	    const result = context.result
	      ? context.result
	      : _highlight(context.language, context.code, ignoreIllegals, continuation);

	    result.code = context.code;
	    // the plugin can change anything in result to suite it
	    fire("after:highlight", result);

	    return result;
	  }

	  /**
	   * private highlight that's used internally and does not fire callbacks
	   *
	   * @param {string} languageName - the language to use for highlighting
	   * @param {string} codeToHighlight - the code to highlight
	   * @param {boolean?} [ignoreIllegals] - whether to ignore illegal matches, default is to bail
	   * @param {CompiledMode?} [continuation] - current continuation mode, if any
	   * @returns {HighlightResult} - result of the highlight operation
	  */
	  function _highlight(languageName, codeToHighlight, ignoreIllegals, continuation) {
	    /**
	     * Return keyword data if a match is a keyword
	     * @param {CompiledMode} mode - current mode
	     * @param {RegExpMatchArray} match - regexp match data
	     * @returns {KeywordData | false}
	     */
	    function keywordData(mode, match) {
	      const matchText = language.case_insensitive ? match[0].toLowerCase() : match[0];
	      return Object.prototype.hasOwnProperty.call(mode.keywords, matchText) && mode.keywords[matchText];
	    }

	    function processKeywords() {
	      if (!top.keywords) {
	        emitter.addText(modeBuffer);
	        return;
	      }

	      let lastIndex = 0;
	      top.keywordPatternRe.lastIndex = 0;
	      let match = top.keywordPatternRe.exec(modeBuffer);
	      let buf = "";

	      while (match) {
	        buf += modeBuffer.substring(lastIndex, match.index);
	        const data = keywordData(top, match);
	        if (data) {
	          const [kind, keywordRelevance] = data;
	          emitter.addText(buf);
	          buf = "";

	          relevance += keywordRelevance;
	          if (kind.startsWith("_")) {
	            // _ implied for relevance only, do not highlight
	            // by applying a class name
	            buf += match[0];
	          } else {
	            const cssClass = language.classNameAliases[kind] || kind;
	            emitter.addKeyword(match[0], cssClass);
	          }
	        } else {
	          buf += match[0];
	        }
	        lastIndex = top.keywordPatternRe.lastIndex;
	        match = top.keywordPatternRe.exec(modeBuffer);
	      }
	      buf += modeBuffer.substr(lastIndex);
	      emitter.addText(buf);
	    }

	    function processSubLanguage() {
	      if (modeBuffer === "") return;
	      /** @type HighlightResult */
	      let result = null;

	      if (typeof top.subLanguage === 'string') {
	        if (!languages[top.subLanguage]) {
	          emitter.addText(modeBuffer);
	          return;
	        }
	        result = _highlight(top.subLanguage, modeBuffer, true, continuations[top.subLanguage]);
	        continuations[top.subLanguage] = /** @type {CompiledMode} */ (result.top);
	      } else {
	        result = highlightAuto(modeBuffer, top.subLanguage.length ? top.subLanguage : null);
	      }

	      // Counting embedded language score towards the host language may be disabled
	      // with zeroing the containing mode relevance. Use case in point is Markdown that
	      // allows XML everywhere and makes every XML snippet to have a much larger Markdown
	      // score.
	      if (top.relevance > 0) {
	        relevance += result.relevance;
	      }
	      emitter.addSublanguage(result.emitter, result.language);
	    }

	    function processBuffer() {
	      if (top.subLanguage != null) {
	        processSubLanguage();
	      } else {
	        processKeywords();
	      }
	      modeBuffer = '';
	    }

	    /**
	     * @param {Mode} mode - new mode to start
	     */
	    function startNewMode(mode) {
	      if (mode.className) {
	        emitter.openNode(language.classNameAliases[mode.className] || mode.className);
	      }
	      top = Object.create(mode, { parent: { value: top } });
	      return top;
	    }

	    /**
	     * @param {CompiledMode } mode - the mode to potentially end
	     * @param {RegExpMatchArray} match - the latest match
	     * @param {string} matchPlusRemainder - match plus remainder of content
	     * @returns {CompiledMode | void} - the next mode, or if void continue on in current mode
	     */
	    function endOfMode(mode, match, matchPlusRemainder) {
	      let matched = startsWith(mode.endRe, matchPlusRemainder);

	      if (matched) {
	        if (mode["on:end"]) {
	          const resp = new Response(mode);
	          mode["on:end"](match, resp);
	          if (resp.isMatchIgnored) matched = false;
	        }

	        if (matched) {
	          while (mode.endsParent && mode.parent) {
	            mode = mode.parent;
	          }
	          return mode;
	        }
	      }
	      // even if on:end fires an `ignore` it's still possible
	      // that we might trigger the end node because of a parent mode
	      if (mode.endsWithParent) {
	        return endOfMode(mode.parent, match, matchPlusRemainder);
	      }
	    }

	    /**
	     * Handle matching but then ignoring a sequence of text
	     *
	     * @param {string} lexeme - string containing full match text
	     */
	    function doIgnore(lexeme) {
	      if (top.matcher.regexIndex === 0) {
	        // no more regexs to potentially match here, so we move the cursor forward one
	        // space
	        modeBuffer += lexeme[0];
	        return 1;
	      } else {
	        // no need to move the cursor, we still have additional regexes to try and
	        // match at this very spot
	        resumeScanAtSamePosition = true;
	        return 0;
	      }
	    }

	    /**
	     * Handle the start of a new potential mode match
	     *
	     * @param {EnhancedMatch} match - the current match
	     * @returns {number} how far to advance the parse cursor
	     */
	    function doBeginMatch(match) {
	      const lexeme = match[0];
	      const newMode = match.rule;

	      const resp = new Response(newMode);
	      // first internal before callbacks, then the public ones
	      const beforeCallbacks = [newMode.__beforeBegin, newMode["on:begin"]];
	      for (const cb of beforeCallbacks) {
	        if (!cb) continue;
	        cb(match, resp);
	        if (resp.isMatchIgnored) return doIgnore(lexeme);
	      }

	      if (newMode && newMode.endSameAsBegin) {
	        newMode.endRe = escape(lexeme);
	      }

	      if (newMode.skip) {
	        modeBuffer += lexeme;
	      } else {
	        if (newMode.excludeBegin) {
	          modeBuffer += lexeme;
	        }
	        processBuffer();
	        if (!newMode.returnBegin && !newMode.excludeBegin) {
	          modeBuffer = lexeme;
	        }
	      }
	      startNewMode(newMode);
	      // if (mode["after:begin"]) {
	      //   let resp = new Response(mode);
	      //   mode["after:begin"](match, resp);
	      // }
	      return newMode.returnBegin ? 0 : lexeme.length;
	    }

	    /**
	     * Handle the potential end of mode
	     *
	     * @param {RegExpMatchArray} match - the current match
	     */
	    function doEndMatch(match) {
	      const lexeme = match[0];
	      const matchPlusRemainder = codeToHighlight.substr(match.index);

	      const endMode = endOfMode(top, match, matchPlusRemainder);
	      if (!endMode) { return NO_MATCH; }

	      const origin = top;
	      if (origin.skip) {
	        modeBuffer += lexeme;
	      } else {
	        if (!(origin.returnEnd || origin.excludeEnd)) {
	          modeBuffer += lexeme;
	        }
	        processBuffer();
	        if (origin.excludeEnd) {
	          modeBuffer = lexeme;
	        }
	      }
	      do {
	        if (top.className) {
	          emitter.closeNode();
	        }
	        if (!top.skip && !top.subLanguage) {
	          relevance += top.relevance;
	        }
	        top = top.parent;
	      } while (top !== endMode.parent);
	      if (endMode.starts) {
	        if (endMode.endSameAsBegin) {
	          endMode.starts.endRe = endMode.endRe;
	        }
	        startNewMode(endMode.starts);
	      }
	      return origin.returnEnd ? 0 : lexeme.length;
	    }

	    function processContinuations() {
	      const list = [];
	      for (let current = top; current !== language; current = current.parent) {
	        if (current.className) {
	          list.unshift(current.className);
	        }
	      }
	      list.forEach(item => emitter.openNode(item));
	    }

	    /** @type {{type?: MatchType, index?: number, rule?: Mode}}} */
	    let lastMatch = {};

	    /**
	     *  Process an individual match
	     *
	     * @param {string} textBeforeMatch - text preceeding the match (since the last match)
	     * @param {EnhancedMatch} [match] - the match itself
	     */
	    function processLexeme(textBeforeMatch, match) {
	      const lexeme = match && match[0];

	      // add non-matched text to the current mode buffer
	      modeBuffer += textBeforeMatch;

	      if (lexeme == null) {
	        processBuffer();
	        return 0;
	      }

	      // we've found a 0 width match and we're stuck, so we need to advance
	      // this happens when we have badly behaved rules that have optional matchers to the degree that
	      // sometimes they can end up matching nothing at all
	      // Ref: https://github.com/highlightjs/highlight.js/issues/2140
	      if (lastMatch.type === "begin" && match.type === "end" && lastMatch.index === match.index && lexeme === "") {
	        // spit the "skipped" character that our regex choked on back into the output sequence
	        modeBuffer += codeToHighlight.slice(match.index, match.index + 1);
	        if (!SAFE_MODE) {
	          /** @type {AnnotatedError} */
	          const err = new Error('0 width match regex');
	          err.languageName = languageName;
	          err.badRule = lastMatch.rule;
	          throw err;
	        }
	        return 1;
	      }
	      lastMatch = match;

	      if (match.type === "begin") {
	        return doBeginMatch(match);
	      } else if (match.type === "illegal" && !ignoreIllegals) {
	        // illegal match, we do not continue processing
	        /** @type {AnnotatedError} */
	        const err = new Error('Illegal lexeme "' + lexeme + '" for mode "' + (top.className || '<unnamed>') + '"');
	        err.mode = top;
	        throw err;
	      } else if (match.type === "end") {
	        const processed = doEndMatch(match);
	        if (processed !== NO_MATCH) {
	          return processed;
	        }
	      }

	      // edge case for when illegal matches $ (end of line) which is technically
	      // a 0 width match but not a begin/end match so it's not caught by the
	      // first handler (when ignoreIllegals is true)
	      if (match.type === "illegal" && lexeme === "") {
	        // advance so we aren't stuck in an infinite loop
	        return 1;
	      }

	      // infinite loops are BAD, this is a last ditch catch all. if we have a
	      // decent number of iterations yet our index (cursor position in our
	      // parsing) still 3x behind our index then something is very wrong
	      // so we bail
	      if (iterations > 100000 && iterations > match.index * 3) {
	        const err = new Error('potential infinite loop, way more iterations than matches');
	        throw err;
	      }

	      /*
	      Why might be find ourselves here?  Only one occasion now.  An end match that was
	      triggered but could not be completed.  When might this happen?  When an `endSameasBegin`
	      rule sets the end rule to a specific match.  Since the overall mode termination rule that's
	      being used to scan the text isn't recompiled that means that any match that LOOKS like
	      the end (but is not, because it is not an exact match to the beginning) will
	      end up here.  A definite end match, but when `doEndMatch` tries to "reapply"
	      the end rule and fails to match, we wind up here, and just silently ignore the end.

	      This causes no real harm other than stopping a few times too many.
	      */

	      modeBuffer += lexeme;
	      return lexeme.length;
	    }

	    const language = getLanguage(languageName);
	    if (!language) {
	      error(LANGUAGE_NOT_FOUND.replace("{}", languageName));
	      throw new Error('Unknown language: "' + languageName + '"');
	    }

	    const md = compileLanguage(language, { plugins });
	    let result = '';
	    /** @type {CompiledMode} */
	    let top = continuation || md;
	    /** @type Record<string,CompiledMode> */
	    const continuations = {}; // keep continuations for sub-languages
	    const emitter = new options.__emitter(options);
	    processContinuations();
	    let modeBuffer = '';
	    let relevance = 0;
	    let index = 0;
	    let iterations = 0;
	    let resumeScanAtSamePosition = false;

	    try {
	      top.matcher.considerAll();

	      for (;;) {
	        iterations++;
	        if (resumeScanAtSamePosition) {
	          // only regexes not matched previously will now be
	          // considered for a potential match
	          resumeScanAtSamePosition = false;
	        } else {
	          top.matcher.considerAll();
	        }
	        top.matcher.lastIndex = index;

	        const match = top.matcher.exec(codeToHighlight);
	        // console.log("match", match[0], match.rule && match.rule.begin)

	        if (!match) break;

	        const beforeMatch = codeToHighlight.substring(index, match.index);
	        const processedCount = processLexeme(beforeMatch, match);
	        index = match.index + processedCount;
	      }
	      processLexeme(codeToHighlight.substr(index));
	      emitter.closeAllNodes();
	      emitter.finalize();
	      result = emitter.toHTML();

	      return {
	        // avoid possible breakage with v10 clients expecting
	        // this to always be an integer
	        relevance: Math.floor(relevance),
	        value: result,
	        language: languageName,
	        illegal: false,
	        emitter: emitter,
	        top: top
	      };
	    } catch (err) {
	      if (err.message && err.message.includes('Illegal')) {
	        return {
	          illegal: true,
	          illegalBy: {
	            msg: err.message,
	            context: codeToHighlight.slice(index - 100, index + 100),
	            mode: err.mode
	          },
	          sofar: result,
	          relevance: 0,
	          value: escape$1(codeToHighlight),
	          emitter: emitter
	        };
	      } else if (SAFE_MODE) {
	        return {
	          illegal: false,
	          relevance: 0,
	          value: escape$1(codeToHighlight),
	          emitter: emitter,
	          language: languageName,
	          top: top,
	          errorRaised: err
	        };
	      } else {
	        throw err;
	      }
	    }
	  }

	  /**
	   * returns a valid highlight result, without actually doing any actual work,
	   * auto highlight starts with this and it's possible for small snippets that
	   * auto-detection may not find a better match
	   * @param {string} code
	   * @returns {HighlightResult}
	   */
	  function justTextHighlightResult(code) {
	    const result = {
	      relevance: 0,
	      emitter: new options.__emitter(options),
	      value: escape$1(code),
	      illegal: false,
	      top: PLAINTEXT_LANGUAGE
	    };
	    result.emitter.addText(code);
	    return result;
	  }

	  /**
	  Highlighting with language detection. Accepts a string with the code to
	  highlight. Returns an object with the following properties:

	  - language (detected language)
	  - relevance (int)
	  - value (an HTML string with highlighting markup)
	  - second_best (object with the same structure for second-best heuristically
	    detected language, may be absent)

	    @param {string} code
	    @param {Array<string>} [languageSubset]
	    @returns {AutoHighlightResult}
	  */
	  function highlightAuto(code, languageSubset) {
	    languageSubset = languageSubset || options.languages || Object.keys(languages);
	    const plaintext = justTextHighlightResult(code);

	    const results = languageSubset.filter(getLanguage).filter(autoDetection).map(name =>
	      _highlight(name, code, false)
	    );
	    results.unshift(plaintext); // plaintext is always an option

	    const sorted = results.sort((a, b) => {
	      // sort base on relevance
	      if (a.relevance !== b.relevance) return b.relevance - a.relevance;

	      // always award the tie to the base language
	      // ie if C++ and Arduino are tied, it's more likely to be C++
	      if (a.language && b.language) {
	        if (getLanguage(a.language).supersetOf === b.language) {
	          return 1;
	        } else if (getLanguage(b.language).supersetOf === a.language) {
	          return -1;
	        }
	      }

	      // otherwise say they are equal, which has the effect of sorting on
	      // relevance while preserving the original ordering - which is how ties
	      // have historically been settled, ie the language that comes first always
	      // wins in the case of a tie
	      return 0;
	    });

	    const [best, secondBest] = sorted;

	    /** @type {AutoHighlightResult} */
	    const result = best;
	    result.second_best = secondBest;

	    return result;
	  }

	  /**
	  Post-processing of the highlighted markup:

	  - replace TABs with something more useful
	  - replace real line-breaks with '<br>' for non-pre containers

	    @param {string} html
	    @returns {string}
	  */
	  function fixMarkup(html) {
	    if (!(options.tabReplace || options.useBR)) {
	      return html;
	    }

	    return html.replace(fixMarkupRe, match => {
	      if (match === '\n') {
	        return options.useBR ? '<br>' : match;
	      } else if (options.tabReplace) {
	        return match.replace(/\t/g, options.tabReplace);
	      }
	      return match;
	    });
	  }

	  /**
	   * Builds new class name for block given the language name
	   *
	   * @param {HTMLElement} element
	   * @param {string} [currentLang]
	   * @param {string} [resultLang]
	   */
	  function updateClassName(element, currentLang, resultLang) {
	    const language = currentLang ? aliases[currentLang] : resultLang;

	    element.classList.add("hljs");
	    if (language) element.classList.add(language);
	  }

	  /** @type {HLJSPlugin} */
	  const brPlugin = {
	    "before:highlightElement": ({ el }) => {
	      if (options.useBR) {
	        el.innerHTML = el.innerHTML.replace(/\n/g, '').replace(/<br[ /]*>/g, '\n');
	      }
	    },
	    "after:highlightElement": ({ result }) => {
	      if (options.useBR) {
	        result.value = result.value.replace(/\n/g, "<br>");
	      }
	    }
	  };

	  const TAB_REPLACE_RE = /^(<[^>]+>|\t)+/gm;
	  /** @type {HLJSPlugin} */
	  const tabReplacePlugin = {
	    "after:highlightElement": ({ result }) => {
	      if (options.tabReplace) {
	        result.value = result.value.replace(TAB_REPLACE_RE, (m) =>
	          m.replace(/\t/g, options.tabReplace)
	        );
	      }
	    }
	  };

	  /**
	   * Applies highlighting to a DOM node containing code. Accepts a DOM node and
	   * two optional parameters for fixMarkup.
	   *
	   * @param {HighlightedHTMLElement} element - the HTML element to highlight
	  */
	  function highlightElement(element) {
	    /** @type HTMLElement */
	    let node = null;
	    const language = blockLanguage(element);

	    if (shouldNotHighlight(language)) return;

	    // support for v10 API
	    fire("before:highlightElement",
	      { el: element, language: language });

	    node = element;
	    const text = node.textContent;
	    const result = language ? highlight(text, { language, ignoreIllegals: true }) : highlightAuto(text);

	    // support for v10 API
	    fire("after:highlightElement", { el: element, result, text });

	    element.innerHTML = result.value;
	    updateClassName(element, language, result.language);
	    element.result = {
	      language: result.language,
	      // TODO: remove with version 11.0
	      re: result.relevance,
	      relavance: result.relevance
	    };
	    if (result.second_best) {
	      element.second_best = {
	        language: result.second_best.language,
	        // TODO: remove with version 11.0
	        re: result.second_best.relevance,
	        relavance: result.second_best.relevance
	      };
	    }
	  }

	  /**
	   * Updates highlight.js global options with the passed options
	   *
	   * @param {Partial<HLJSOptions>} userOptions
	   */
	  function configure(userOptions) {
	    if (userOptions.useBR) {
	      deprecated("10.3.0", "'useBR' will be removed entirely in v11.0");
	      deprecated("10.3.0", "Please see https://github.com/highlightjs/highlight.js/issues/2559");
	    }
	    options = inherit$1(options, userOptions);
	  }

	  /**
	   * Highlights to all <pre><code> blocks on a page
	   *
	   * @type {Function & {called?: boolean}}
	   */
	  // TODO: remove v12, deprecated
	  const initHighlighting = () => {
	    if (initHighlighting.called) return;
	    initHighlighting.called = true;

	    deprecated("10.6.0", "initHighlighting() is deprecated.  Use highlightAll() instead.");

	    const blocks = document.querySelectorAll('pre code');
	    blocks.forEach(highlightElement);
	  };

	  // Higlights all when DOMContentLoaded fires
	  // TODO: remove v12, deprecated
	  function initHighlightingOnLoad() {
	    deprecated("10.6.0", "initHighlightingOnLoad() is deprecated.  Use highlightAll() instead.");
	    wantsHighlight = true;
	  }

	  let wantsHighlight = false;

	  /**
	   * auto-highlights all pre>code elements on the page
	   */
	  function highlightAll() {
	    // if we are called too early in the loading process
	    if (document.readyState === "loading") {
	      wantsHighlight = true;
	      return;
	    }

	    const blocks = document.querySelectorAll('pre code');
	    blocks.forEach(highlightElement);
	  }

	  function boot() {
	    // if a highlight was requested before DOM was loaded, do now
	    if (wantsHighlight) highlightAll();
	  }

	  // make sure we are in the browser environment
	  if (typeof window !== 'undefined' && window.addEventListener) {
	    window.addEventListener('DOMContentLoaded', boot, false);
	  }

	  /**
	   * Register a language grammar module
	   *
	   * @param {string} languageName
	   * @param {LanguageFn} languageDefinition
	   */
	  function registerLanguage(languageName, languageDefinition) {
	    let lang = null;
	    try {
	      lang = languageDefinition(hljs);
	    } catch (error$1) {
	      error("Language definition for '{}' could not be registered.".replace("{}", languageName));
	      // hard or soft error
	      if (!SAFE_MODE) { throw error$1; } else { error(error$1); }
	      // languages that have serious errors are replaced with essentially a
	      // "plaintext" stand-in so that the code blocks will still get normal
	      // css classes applied to them - and one bad language won't break the
	      // entire highlighter
	      lang = PLAINTEXT_LANGUAGE;
	    }
	    // give it a temporary name if it doesn't have one in the meta-data
	    if (!lang.name) lang.name = languageName;
	    languages[languageName] = lang;
	    lang.rawDefinition = languageDefinition.bind(null, hljs);

	    if (lang.aliases) {
	      registerAliases(lang.aliases, { languageName });
	    }
	  }

	  /**
	   * Remove a language grammar module
	   *
	   * @param {string} languageName
	   */
	  function unregisterLanguage(languageName) {
	    delete languages[languageName];
	    for (const alias of Object.keys(aliases)) {
	      if (aliases[alias] === languageName) {
	        delete aliases[alias];
	      }
	    }
	  }

	  /**
	   * @returns {string[]} List of language internal names
	   */
	  function listLanguages() {
	    return Object.keys(languages);
	  }

	  /**
	    intended usage: When one language truly requires another

	    Unlike `getLanguage`, this will throw when the requested language
	    is not available.

	    @param {string} name - name of the language to fetch/require
	    @returns {Language | never}
	  */
	  function requireLanguage(name) {
	    deprecated("10.4.0", "requireLanguage will be removed entirely in v11.");
	    deprecated("10.4.0", "Please see https://github.com/highlightjs/highlight.js/pull/2844");

	    const lang = getLanguage(name);
	    if (lang) { return lang; }

	    const err = new Error('The \'{}\' language is required, but not loaded.'.replace('{}', name));
	    throw err;
	  }

	  /**
	   * @param {string} name - name of the language to retrieve
	   * @returns {Language | undefined}
	   */
	  function getLanguage(name) {
	    name = (name || '').toLowerCase();
	    return languages[name] || languages[aliases[name]];
	  }

	  /**
	   *
	   * @param {string|string[]} aliasList - single alias or list of aliases
	   * @param {{languageName: string}} opts
	   */
	  function registerAliases(aliasList, { languageName }) {
	    if (typeof aliasList === 'string') {
	      aliasList = [aliasList];
	    }
	    aliasList.forEach(alias => { aliases[alias.toLowerCase()] = languageName; });
	  }

	  /**
	   * Determines if a given language has auto-detection enabled
	   * @param {string} name - name of the language
	   */
	  function autoDetection(name) {
	    const lang = getLanguage(name);
	    return lang && !lang.disableAutodetect;
	  }

	  /**
	   * Upgrades the old highlightBlock plugins to the new
	   * highlightElement API
	   * @param {HLJSPlugin} plugin
	   */
	  function upgradePluginAPI(plugin) {
	    // TODO: remove with v12
	    if (plugin["before:highlightBlock"] && !plugin["before:highlightElement"]) {
	      plugin["before:highlightElement"] = (data) => {
	        plugin["before:highlightBlock"](
	          Object.assign({ block: data.el }, data)
	        );
	      };
	    }
	    if (plugin["after:highlightBlock"] && !plugin["after:highlightElement"]) {
	      plugin["after:highlightElement"] = (data) => {
	        plugin["after:highlightBlock"](
	          Object.assign({ block: data.el }, data)
	        );
	      };
	    }
	  }

	  /**
	   * @param {HLJSPlugin} plugin
	   */
	  function addPlugin(plugin) {
	    upgradePluginAPI(plugin);
	    plugins.push(plugin);
	  }

	  /**
	   *
	   * @param {PluginEvent} event
	   * @param {any} args
	   */
	  function fire(event, args) {
	    const cb = event;
	    plugins.forEach(function(plugin) {
	      if (plugin[cb]) {
	        plugin[cb](args);
	      }
	    });
	  }

	  /**
	  Note: fixMarkup is deprecated and will be removed entirely in v11

	  @param {string} arg
	  @returns {string}
	  */
	  function deprecateFixMarkup(arg) {
	    deprecated("10.2.0", "fixMarkup will be removed entirely in v11.0");
	    deprecated("10.2.0", "Please see https://github.com/highlightjs/highlight.js/issues/2534");

	    return fixMarkup(arg);
	  }

	  /**
	   *
	   * @param {HighlightedHTMLElement} el
	   */
	  function deprecateHighlightBlock(el) {
	    deprecated("10.7.0", "highlightBlock will be removed entirely in v12.0");
	    deprecated("10.7.0", "Please use highlightElement now.");

	    return highlightElement(el);
	  }

	  /* Interface definition */
	  Object.assign(hljs, {
	    highlight,
	    highlightAuto,
	    highlightAll,
	    fixMarkup: deprecateFixMarkup,
	    highlightElement,
	    // TODO: Remove with v12 API
	    highlightBlock: deprecateHighlightBlock,
	    configure,
	    initHighlighting,
	    initHighlightingOnLoad,
	    registerLanguage,
	    unregisterLanguage,
	    listLanguages,
	    getLanguage,
	    registerAliases,
	    requireLanguage,
	    autoDetection,
	    inherit: inherit$1,
	    addPlugin,
	    // plugins for frameworks
	    vuePlugin: BuildVuePlugin(hljs).VuePlugin
	  });

	  hljs.debugMode = function() { SAFE_MODE = false; };
	  hljs.safeMode = function() { SAFE_MODE = true; };
	  hljs.versionString = version;

	  for (const key in MODES) {
	    // @ts-ignore
	    if (typeof MODES[key] === "object") {
	      // @ts-ignore
	      deepFreezeEs6(MODES[key]);
	    }
	  }

	  // merge all the modes/regexs into our main object
	  Object.assign(hljs, MODES);

	  // built-in plugins, likely to be moved out of core in the future
	  hljs.addPlugin(brPlugin); // slated to be removed in v11
	  hljs.addPlugin(mergeHTMLPlugin);
	  hljs.addPlugin(tabReplacePlugin);
	  return hljs;
	};

	// export an "instance" of the highlighter
	var highlight = HLJS({});

	core = highlight;
	return core;
}

/*
Language: 1C:Enterprise
Author: Stanislav Belov <stbelov@gmail.com>
Description: built-in language 1C:Enterprise (v7, v8)
Category: enterprise
*/

var _1c_1;
var hasRequired_1c;

function require_1c () {
	if (hasRequired_1c) return _1c_1;
	hasRequired_1c = 1;
	function _1c(hljs) {

	  // общий паттерн для определения идентификаторов
	  var UNDERSCORE_IDENT_RE = '[A-Za-zА-Яа-яёЁ_][A-Za-zА-Яа-яёЁ_0-9]+';

	  // v7 уникальные ключевые слова, отсутствующие в v8 ==> keyword
	  var v7_keywords =
	  'далее ';

	  // v8 ключевые слова ==> keyword
	  var v8_keywords =
	  'возврат вызватьисключение выполнить для если и из или иначе иначеесли исключение каждого конецесли ' +
	  'конецпопытки конеццикла не новый перейти перем по пока попытка прервать продолжить тогда цикл экспорт ';

	  // keyword : ключевые слова
	  var KEYWORD = v7_keywords + v8_keywords;

	  // v7 уникальные директивы, отсутствующие в v8 ==> meta-keyword
	  var v7_meta_keywords =
	  'загрузитьизфайла ';

	  // v8 ключевые слова в инструкциях препроцессора, директивах компиляции, аннотациях ==> meta-keyword
	  var v8_meta_keywords =
	  'вебклиент вместо внешнеесоединение клиент конецобласти мобильноеприложениеклиент мобильноеприложениесервер ' +
	  'наклиенте наклиентенасервере наклиентенасерверебезконтекста насервере насерверебезконтекста область перед ' +
	  'после сервер толстыйклиентобычноеприложение толстыйклиентуправляемоеприложение тонкийклиент ';

	  // meta-keyword : ключевые слова в инструкциях препроцессора, директивах компиляции, аннотациях
	  var METAKEYWORD = v7_meta_keywords + v8_meta_keywords;

	  // v7 системные константы ==> built_in
	  var v7_system_constants =
	  'разделительстраниц разделительстрок символтабуляции ';

	  // v7 уникальные методы глобального контекста, отсутствующие в v8 ==> built_in
	  var v7_global_context_methods =
	  'ansitooem oemtoansi ввестивидсубконто ввестиперечисление ввестипериод ввестиплансчетов выбранныйплансчетов ' +
	  'датагод датамесяц датачисло заголовоксистемы значениевстроку значениеизстроки каталогиб каталогпользователя ' +
	  'кодсимв конгода конецпериодаби конецрассчитанногопериодаби конецстандартногоинтервала конквартала конмесяца ' +
	  'коннедели лог лог10 максимальноеколичествосубконто названиеинтерфейса названиенабораправ назначитьвид ' +
	  'назначитьсчет найтиссылки началопериодаби началостандартногоинтервала начгода начквартала начмесяца ' +
	  'начнедели номерднягода номерднянедели номернеделигода обработкаожидания основнойжурналрасчетов ' +
	  'основнойплансчетов основнойязык очиститьокносообщений периодстр получитьвремята получитьдатута ' +
	  'получитьдокументта получитьзначенияотбора получитьпозициюта получитьпустоезначение получитьта ' +
	  'префиксавтонумерации пропись пустоезначение разм разобратьпозициюдокумента рассчитатьрегистрына ' +
	  'рассчитатьрегистрыпо симв создатьобъект статусвозврата стрколичествострок сформироватьпозициюдокумента ' +
	  'счетпокоду текущеевремя типзначения типзначениястр установитьтана установитьтапо фиксшаблон шаблон ';

	  // v8 методы глобального контекста ==> built_in
	  var v8_global_context_methods =
	  'acos asin atan base64значение base64строка cos exp log log10 pow sin sqrt tan xmlзначение xmlстрока ' +
	  'xmlтип xmlтипзнч активноеокно безопасныйрежим безопасныйрежимразделенияданных булево ввестидату ввестизначение ' +
	  'ввестистроку ввестичисло возможностьчтенияxml вопрос восстановитьзначение врег выгрузитьжурналрегистрации ' +
	  'выполнитьобработкуоповещения выполнитьпроверкуправдоступа вычислить год данныеформывзначение дата день деньгода ' +
	  'деньнедели добавитьмесяц заблокироватьданныедляредактирования заблокироватьработупользователя завершитьработусистемы ' +
	  'загрузитьвнешнююкомпоненту закрытьсправку записатьjson записатьxml записатьдатуjson записьжурналарегистрации ' +
	  'заполнитьзначениясвойств запроситьразрешениепользователя запуститьприложение запуститьсистему зафиксироватьтранзакцию ' +
	  'значениевданныеформы значениевстрокувнутр значениевфайл значениезаполнено значениеизстрокивнутр значениеизфайла ' +
	  'изxmlтипа импортмоделиxdto имякомпьютера имяпользователя инициализироватьпредопределенныеданные информацияобошибке ' +
	  'каталогбиблиотекимобильногоустройства каталогвременныхфайлов каталогдокументов каталогпрограммы кодироватьстроку ' +
	  'кодлокализацииинформационнойбазы кодсимвола командасистемы конецгода конецдня конецквартала конецмесяца конецминуты ' +
	  'конецнедели конецчаса конфигурациябазыданныхизмененадинамически конфигурацияизменена копироватьданныеформы ' +
	  'копироватьфайл краткоепредставлениеошибки лев макс местноевремя месяц мин минута монопольныйрежим найти ' +
	  'найтинедопустимыесимволыxml найтиокнопонавигационнойссылке найтипомеченныенаудаление найтипоссылкам найтифайлы ' +
	  'началогода началодня началоквартала началомесяца началоминуты началонедели началочаса начатьзапросразрешенияпользователя ' +
	  'начатьзапускприложения начатькопированиефайла начатьперемещениефайла начатьподключениевнешнейкомпоненты ' +
	  'начатьподключениерасширенияработыскриптографией начатьподключениерасширенияработысфайлами начатьпоискфайлов ' +
	  'начатьполучениекаталогавременныхфайлов начатьполучениекаталогадокументов начатьполучениерабочегокаталогаданныхпользователя ' +
	  'начатьполучениефайлов начатьпомещениефайла начатьпомещениефайлов начатьсозданиедвоичныхданныхизфайла начатьсозданиекаталога ' +
	  'начатьтранзакцию начатьудалениефайлов начатьустановкувнешнейкомпоненты начатьустановкурасширенияработыскриптографией ' +
	  'начатьустановкурасширенияработысфайлами неделягода необходимостьзавершениясоединения номерсеансаинформационнойбазы ' +
	  'номерсоединенияинформационнойбазы нрег нстр обновитьинтерфейс обновитьнумерациюобъектов обновитьповторноиспользуемыезначения ' +
	  'обработкапрерыванияпользователя объединитьфайлы окр описаниеошибки оповестить оповеститьобизменении ' +
	  'отключитьобработчикзапросанастроекклиенталицензирования отключитьобработчикожидания отключитьобработчикоповещения ' +
	  'открытьзначение открытьиндекссправки открытьсодержаниесправки открытьсправку открытьформу открытьформумодально ' +
	  'отменитьтранзакцию очиститьжурналрегистрации очиститьнастройкипользователя очиститьсообщения параметрыдоступа ' +
	  'перейтипонавигационнойссылке переместитьфайл подключитьвнешнююкомпоненту ' +
	  'подключитьобработчикзапросанастроекклиенталицензирования подключитьобработчикожидания подключитьобработчикоповещения ' +
	  'подключитьрасширениеработыскриптографией подключитьрасширениеработысфайлами подробноепредставлениеошибки ' +
	  'показатьвводдаты показатьвводзначения показатьвводстроки показатьвводчисла показатьвопрос показатьзначение ' +
	  'показатьинформациюобошибке показатьнакарте показатьоповещениепользователя показатьпредупреждение полноеимяпользователя ' +
	  'получитьcomобъект получитьxmlтип получитьадреспоместоположению получитьблокировкусеансов получитьвремязавершенияспящегосеанса ' +
	  'получитьвремязасыпанияпассивногосеанса получитьвремяожиданияблокировкиданных получитьданныевыбора ' +
	  'получитьдополнительныйпараметрклиенталицензирования получитьдопустимыекодылокализации получитьдопустимыечасовыепояса ' +
	  'получитьзаголовокклиентскогоприложения получитьзаголовоксистемы получитьзначенияотборажурналарегистрации ' +
	  'получитьидентификаторконфигурации получитьизвременногохранилища получитьимявременногофайла ' +
	  'получитьимяклиенталицензирования получитьинформациюэкрановклиента получитьиспользованиежурналарегистрации ' +
	  'получитьиспользованиесобытияжурналарегистрации получитькраткийзаголовокприложения получитьмакетоформления ' +
	  'получитьмаскувсефайлы получитьмаскувсефайлыклиента получитьмаскувсефайлысервера получитьместоположениепоадресу ' +
	  'получитьминимальнуюдлинупаролейпользователей получитьнавигационнуюссылку получитьнавигационнуюссылкуинформационнойбазы ' +
	  'получитьобновлениеконфигурациибазыданных получитьобновлениепредопределенныхданныхинформационнойбазы получитьобщиймакет ' +
	  'получитьобщуюформу получитьокна получитьоперативнуюотметкувремени получитьотключениебезопасногорежима ' +
	  'получитьпараметрыфункциональныхопцийинтерфейса получитьполноеимяпредопределенногозначения ' +
	  'получитьпредставлениянавигационныхссылок получитьпроверкусложностипаролейпользователей получитьразделительпути ' +
	  'получитьразделительпутиклиента получитьразделительпутисервера получитьсеансыинформационнойбазы ' +
	  'получитьскоростьклиентскогосоединения получитьсоединенияинформационнойбазы получитьсообщенияпользователю ' +
	  'получитьсоответствиеобъектаиформы получитьсоставстандартногоинтерфейсаodata получитьструктурухранениябазыданных ' +
	  'получитьтекущийсеансинформационнойбазы получитьфайл получитьфайлы получитьформу получитьфункциональнуюопцию ' +
	  'получитьфункциональнуюопциюинтерфейса получитьчасовойпоясинформационнойбазы пользователиос поместитьвовременноехранилище ' +
	  'поместитьфайл поместитьфайлы прав праводоступа предопределенноезначение представлениекодалокализации представлениепериода ' +
	  'представлениеправа представлениеприложения представлениесобытияжурналарегистрации представлениечасовогопояса предупреждение ' +
	  'прекратитьработусистемы привилегированныйрежим продолжитьвызов прочитатьjson прочитатьxml прочитатьдатуjson пустаястрока ' +
	  'рабочийкаталогданныхпользователя разблокироватьданныедляредактирования разделитьфайл разорватьсоединениесвнешнимисточникомданных ' +
	  'раскодироватьстроку рольдоступна секунда сигнал символ скопироватьжурналрегистрации смещениелетнеговремени ' +
	  'смещениестандартноговремени соединитьбуферыдвоичныхданных создатькаталог создатьфабрикуxdto сокрл сокрлп сокрп сообщить ' +
	  'состояние сохранитьзначение сохранитьнастройкипользователя сред стрдлина стрзаканчиваетсяна стрзаменить стрнайти стрначинаетсяс ' +
	  'строка строкасоединенияинформационнойбазы стрполучитьстроку стрразделить стрсоединить стрсравнить стрчисловхождений '+
	  'стрчислострок стршаблон текущаядата текущаядатасеанса текущаяуниверсальнаядата текущаяуниверсальнаядатавмиллисекундах ' +
	  'текущийвариантинтерфейсаклиентскогоприложения текущийвариантосновногошрифтаклиентскогоприложения текущийкодлокализации ' +
	  'текущийрежимзапуска текущийязык текущийязыксистемы тип типзнч транзакцияактивна трег удалитьданныеинформационнойбазы ' +
	  'удалитьизвременногохранилища удалитьобъекты удалитьфайлы универсальноевремя установитьбезопасныйрежим ' +
	  'установитьбезопасныйрежимразделенияданных установитьблокировкусеансов установитьвнешнююкомпоненту ' +
	  'установитьвремязавершенияспящегосеанса установитьвремязасыпанияпассивногосеанса установитьвремяожиданияблокировкиданных ' +
	  'установитьзаголовокклиентскогоприложения установитьзаголовоксистемы установитьиспользованиежурналарегистрации ' +
	  'установитьиспользованиесобытияжурналарегистрации установитькраткийзаголовокприложения ' +
	  'установитьминимальнуюдлинупаролейпользователей установитьмонопольныйрежим установитьнастройкиклиенталицензирования ' +
	  'установитьобновлениепредопределенныхданныхинформационнойбазы установитьотключениебезопасногорежима ' +
	  'установитьпараметрыфункциональныхопцийинтерфейса установитьпривилегированныйрежим ' +
	  'установитьпроверкусложностипаролейпользователей установитьрасширениеработыскриптографией ' +
	  'установитьрасширениеработысфайлами установитьсоединениесвнешнимисточникомданных установитьсоответствиеобъектаиформы ' +
	  'установитьсоставстандартногоинтерфейсаodata установитьчасовойпоясинформационнойбазы установитьчасовойпояссеанса ' +
	  'формат цел час часовойпояс часовойпояссеанса число числопрописью этоадресвременногохранилища ';

	  // v8 свойства глобального контекста ==> built_in
	  var v8_global_context_property =
	  'wsссылки библиотекакартинок библиотекамакетовоформлениякомпоновкиданных библиотекастилей бизнеспроцессы ' +
	  'внешниеисточникиданных внешниеобработки внешниеотчеты встроенныепокупки главныйинтерфейс главныйстиль ' +
	  'документы доставляемыеуведомления журналыдокументов задачи информацияобинтернетсоединении использованиерабочейдаты ' +
	  'историяработыпользователя константы критерииотбора метаданные обработки отображениерекламы отправкадоставляемыхуведомлений ' +
	  'отчеты панельзадачос параметрзапуска параметрысеанса перечисления планывидоврасчета планывидовхарактеристик ' +
	  'планыобмена планысчетов полнотекстовыйпоиск пользователиинформационнойбазы последовательности проверкавстроенныхпокупок ' +
	  'рабочаядата расширенияконфигурации регистрыбухгалтерии регистрынакопления регистрырасчета регистрысведений ' +
	  'регламентныезадания сериализаторxdto справочники средствагеопозиционирования средствакриптографии средствамультимедиа ' +
	  'средстваотображениярекламы средствапочты средствателефонии фабрикаxdto файловыепотоки фоновыезадания хранилищанастроек ' +
	  'хранилищевариантовотчетов хранилищенастроекданныхформ хранилищеобщихнастроек хранилищепользовательскихнастроекдинамическихсписков ' +
	  'хранилищепользовательскихнастроекотчетов хранилищесистемныхнастроек ';

	  // built_in : встроенные или библиотечные объекты (константы, классы, функции)
	  var BUILTIN =
	  v7_system_constants +
	  v7_global_context_methods + v8_global_context_methods +
	  v8_global_context_property;

	  // v8 системные наборы значений ==> class
	  var v8_system_sets_of_values =
	  'webцвета windowsцвета windowsшрифты библиотекакартинок рамкистиля символы цветастиля шрифтыстиля ';

	  // v8 системные перечисления - интерфейсные ==> class
	  var v8_system_enums_interface =
	  'автоматическоесохранениеданныхформывнастройках автонумерациявформе автораздвижениесерий ' +
	  'анимациядиаграммы вариантвыравниванияэлементовизаголовков вариантуправлениявысотойтаблицы ' +
	  'вертикальнаяпрокруткаформы вертикальноеположение вертикальноеположениеэлемента видгруппыформы ' +
	  'виддекорацииформы виддополненияэлементаформы видизмененияданных видкнопкиформы видпереключателя ' +
	  'видподписейкдиаграмме видполяформы видфлажка влияниеразмеранапузырекдиаграммы горизонтальноеположение ' +
	  'горизонтальноеположениеэлемента группировкаколонок группировкаподчиненныхэлементовформы ' +
	  'группыиэлементы действиеперетаскивания дополнительныйрежимотображения допустимыедействияперетаскивания ' +
	  'интервалмеждуэлементамиформы использованиевывода использованиеполосыпрокрутки ' +
	  'используемоезначениеточкибиржевойдиаграммы историявыборапривводе источникзначенийоситочекдиаграммы ' +
	  'источникзначенияразмерапузырькадиаграммы категориягруппыкоманд максимумсерий начальноеотображениедерева ' +
	  'начальноеотображениесписка обновлениетекстаредактирования ориентациядендрограммы ориентациядиаграммы ' +
	  'ориентацияметокдиаграммы ориентацияметоксводнойдиаграммы ориентацияэлементаформы отображениевдиаграмме ' +
	  'отображениевлегендедиаграммы отображениегруппыкнопок отображениезаголовкашкалыдиаграммы ' +
	  'отображениезначенийсводнойдиаграммы отображениезначенияизмерительнойдиаграммы ' +
	  'отображениеинтерваладиаграммыганта отображениекнопки отображениекнопкивыбора отображениеобсужденийформы ' +
	  'отображениеобычнойгруппы отображениеотрицательныхзначенийпузырьковойдиаграммы отображениепанелипоиска ' +
	  'отображениеподсказки отображениепредупрежденияприредактировании отображениеразметкиполосырегулирования ' +
	  'отображениестраницформы отображениетаблицы отображениетекстазначениядиаграммыганта ' +
	  'отображениеуправленияобычнойгруппы отображениефигурыкнопки палитрацветовдиаграммы поведениеобычнойгруппы ' +
	  'поддержкамасштабадендрограммы поддержкамасштабадиаграммыганта поддержкамасштабасводнойдиаграммы ' +
	  'поисквтаблицепривводе положениезаголовкаэлементаформы положениекартинкикнопкиформы ' +
	  'положениекартинкиэлементаграфическойсхемы положениекоманднойпанелиформы положениекоманднойпанелиэлементаформы ' +
	  'положениеопорнойточкиотрисовки положениеподписейкдиаграмме положениеподписейшкалызначенийизмерительнойдиаграммы ' +
	  'положениесостоянияпросмотра положениестрокипоиска положениетекстасоединительнойлинии положениеуправленияпоиском ' +
	  'положениешкалывремени порядокотображенияточекгоризонтальнойгистограммы порядоксерийвлегендедиаграммы ' +
	  'размеркартинки расположениезаголовкашкалыдиаграммы растягиваниеповертикалидиаграммыганта ' +
	  'режимавтоотображениясостояния режимвводастроктаблицы режимвыборанезаполненного режимвыделениядаты ' +
	  'режимвыделениястрокитаблицы режимвыделениятаблицы режимизмененияразмера режимизменениясвязанногозначения ' +
	  'режимиспользованиядиалогапечати режимиспользованияпараметракоманды режиммасштабированияпросмотра ' +
	  'режимосновногоокнаклиентскогоприложения режимоткрытияокнаформы режимотображениявыделения ' +
	  'режимотображениягеографическойсхемы режимотображениязначенийсерии режимотрисовкисеткиграфическойсхемы ' +
	  'режимполупрозрачностидиаграммы режимпробеловдиаграммы режимразмещениянастранице режимредактированияколонки ' +
	  'режимсглаживаниядиаграммы режимсглаживанияиндикатора режимсписказадач сквозноевыравнивание ' +
	  'сохранениеданныхформывнастройках способзаполнениятекстазаголовкашкалыдиаграммы ' +
	  'способопределенияограничивающегозначениядиаграммы стандартнаягруппакоманд стандартноеоформление ' +
	  'статусоповещенияпользователя стильстрелки типаппроксимациилиниитрендадиаграммы типдиаграммы ' +
	  'типединицышкалывремени типимпортасерийслоягеографическойсхемы типлиниигеографическойсхемы типлиниидиаграммы ' +
	  'типмаркерагеографическойсхемы типмаркерадиаграммы типобластиоформления ' +
	  'типорганизацииисточникаданныхгеографическойсхемы типотображениясериислоягеографическойсхемы ' +
	  'типотображенияточечногообъектагеографическойсхемы типотображенияшкалыэлементалегендыгеографическойсхемы ' +
	  'типпоискаобъектовгеографическойсхемы типпроекциигеографическойсхемы типразмещенияизмерений ' +
	  'типразмещенияреквизитовизмерений типрамкиэлементауправления типсводнойдиаграммы ' +
	  'типсвязидиаграммыганта типсоединениязначенийпосериямдиаграммы типсоединенияточекдиаграммы ' +
	  'типсоединительнойлинии типстороныэлементаграфическойсхемы типформыотчета типшкалырадарнойдиаграммы ' +
	  'факторлиниитрендадиаграммы фигуракнопки фигурыграфическойсхемы фиксациявтаблице форматдняшкалывремени ' +
	  'форматкартинки ширинаподчиненныхэлементовформы ';

	  // v8 системные перечисления - свойства прикладных объектов ==> class
	  var v8_system_enums_objects_properties =
	  'виддвижениябухгалтерии виддвижениянакопления видпериодарегистрарасчета видсчета видточкимаршрутабизнеспроцесса ' +
	  'использованиеагрегатарегистранакопления использованиегруппиэлементов использованиережимапроведения ' +
	  'использованиесреза периодичностьагрегатарегистранакопления режимавтовремя режимзаписидокумента режимпроведениядокумента ';

	  // v8 системные перечисления - планы обмена ==> class
	  var v8_system_enums_exchange_plans =
	  'авторегистрацияизменений допустимыйномерсообщения отправкаэлементаданных получениеэлементаданных ';

	  // v8 системные перечисления - табличный документ ==> class
	  var v8_system_enums_tabular_document =
	  'использованиерасшифровкитабличногодокумента ориентациястраницы положениеитоговколоноксводнойтаблицы ' +
	  'положениеитоговстроксводнойтаблицы положениетекстаотносительнокартинки расположениезаголовкагруппировкитабличногодокумента ' +
	  'способчтениязначенийтабличногодокумента типдвустороннейпечати типзаполненияобластитабличногодокумента ' +
	  'типкурсоровтабличногодокумента типлиниирисункатабличногодокумента типлинииячейкитабличногодокумента ' +
	  'типнаправленияпереходатабличногодокумента типотображениявыделениятабличногодокумента типотображениялинийсводнойтаблицы ' +
	  'типразмещениятекстатабличногодокумента типрисункатабличногодокумента типсмещениятабличногодокумента ' +
	  'типузоратабличногодокумента типфайлатабличногодокумента точностьпечати чередованиерасположениястраниц ';

	  // v8 системные перечисления - планировщик ==> class
	  var v8_system_enums_sheduler =
	  'отображениевремениэлементовпланировщика ';

	  // v8 системные перечисления - форматированный документ ==> class
	  var v8_system_enums_formatted_document =
	  'типфайлаформатированногодокумента ';

	  // v8 системные перечисления - запрос ==> class
	  var v8_system_enums_query =
	  'обходрезультатазапроса типзаписизапроса ';

	  // v8 системные перечисления - построитель отчета ==> class
	  var v8_system_enums_report_builder =
	  'видзаполнениярасшифровкипостроителяотчета типдобавленияпредставлений типизмеренияпостроителяотчета типразмещенияитогов ';

	  // v8 системные перечисления - работа с файлами ==> class
	  var v8_system_enums_files =
	  'доступкфайлу режимдиалогавыборафайла режимоткрытияфайла ';

	  // v8 системные перечисления - построитель запроса ==> class
	  var v8_system_enums_query_builder =
	  'типизмеренияпостроителязапроса ';

	  // v8 системные перечисления - анализ данных ==> class
	  var v8_system_enums_data_analysis =
	  'видданныханализа методкластеризации типединицыинтервалавременианализаданных типзаполнениятаблицырезультатаанализаданных ' +
	  'типиспользованиячисловыхзначенийанализаданных типисточникаданныхпоискаассоциаций типколонкианализаданныхдереворешений ' +
	  'типколонкианализаданныхкластеризация типколонкианализаданныхобщаястатистика типколонкианализаданныхпоискассоциаций ' +
	  'типколонкианализаданныхпоискпоследовательностей типколонкимоделипрогноза типмерырасстоянияанализаданных ' +
	  'типотсеченияправилассоциации типполяанализаданных типстандартизациианализаданных типупорядочиванияправилассоциациианализаданных ' +
	  'типупорядочиванияшаблоновпоследовательностейанализаданных типупрощениядереварешений ';

	  // v8 системные перечисления - xml, json, xs, dom, xdto, web-сервисы ==> class
	  var v8_system_enums_xml_json_xs_dom_xdto_ws =
	  'wsнаправлениепараметра вариантxpathxs вариантзаписидатыjson вариантпростоготипаxs видгруппымоделиxs видфасетаxdto ' +
	  'действиепостроителяdom завершенностьпростоготипаxs завершенностьсоставноготипаxs завершенностьсхемыxs запрещенныеподстановкиxs ' +
	  'исключениягруппподстановкиxs категорияиспользованияатрибутаxs категорияограниченияидентичностиxs категорияограниченияпространствименxs ' +
	  'методнаследованияxs модельсодержимогоxs назначениетипаxml недопустимыеподстановкиxs обработкапробельныхсимволовxs обработкасодержимогоxs ' +
	  'ограничениезначенияxs параметрыотбораузловdom переносстрокjson позициявдокументеdom пробельныесимволыxml типатрибутаxml типзначенияjson ' +
	  'типканоническогоxml типкомпонентыxs типпроверкиxml типрезультатаdomxpath типузлаdom типузлаxml формаxml формапредставленияxs ' +
	  'форматдатыjson экранированиесимволовjson ';

	  // v8 системные перечисления - система компоновки данных ==> class
	  var v8_system_enums_data_composition_system =
	  'видсравнениякомпоновкиданных действиеобработкирасшифровкикомпоновкиданных направлениесортировкикомпоновкиданных ' +
	  'расположениевложенныхэлементоврезультатакомпоновкиданных расположениеитоговкомпоновкиданных расположениегруппировкикомпоновкиданных ' +
	  'расположениеполейгруппировкикомпоновкиданных расположениеполякомпоновкиданных расположениереквизитовкомпоновкиданных ' +
	  'расположениересурсовкомпоновкиданных типбухгалтерскогоостаткакомпоновкиданных типвыводатекстакомпоновкиданных ' +
	  'типгруппировкикомпоновкиданных типгруппыэлементовотборакомпоновкиданных типдополненияпериодакомпоновкиданных ' +
	  'типзаголовкаполейкомпоновкиданных типмакетагруппировкикомпоновкиданных типмакетаобластикомпоновкиданных типостаткакомпоновкиданных ' +
	  'типпериодакомпоновкиданных типразмещениятекстакомпоновкиданных типсвязинаборовданныхкомпоновкиданных типэлементарезультатакомпоновкиданных ' +
	  'расположениелегендыдиаграммыкомпоновкиданных типпримененияотборакомпоновкиданных режимотображенияэлементанастройкикомпоновкиданных ' +
	  'режимотображениянастроеккомпоновкиданных состояниеэлементанастройкикомпоновкиданных способвосстановлениянастроеккомпоновкиданных ' +
	  'режимкомпоновкирезультата использованиепараметракомпоновкиданных автопозицияресурсовкомпоновкиданных '+
	  'вариантиспользованиягруппировкикомпоновкиданных расположениересурсоввдиаграммекомпоновкиданных фиксациякомпоновкиданных ' +
	  'использованиеусловногооформлениякомпоновкиданных ';

	  // v8 системные перечисления - почта ==> class
	  var v8_system_enums_email =
	  'важностьинтернетпочтовогосообщения обработкатекстаинтернетпочтовогосообщения способкодированияинтернетпочтовоговложения ' +
	  'способкодированиянеasciiсимволовинтернетпочтовогосообщения типтекстапочтовогосообщения протоколинтернетпочты ' +
	  'статусразборапочтовогосообщения ';

	  // v8 системные перечисления - журнал регистрации ==> class
	  var v8_system_enums_logbook =
	  'режимтранзакциизаписижурналарегистрации статустранзакциизаписижурналарегистрации уровеньжурналарегистрации ';

	  // v8 системные перечисления - криптография ==> class
	  var v8_system_enums_cryptography =
	  'расположениехранилищасертификатовкриптографии режимвключениясертификатовкриптографии режимпроверкисертификатакриптографии ' +
	  'типхранилищасертификатовкриптографии ';

	  // v8 системные перечисления - ZIP ==> class
	  var v8_system_enums_zip =
	  'кодировкаименфайловвzipфайле методсжатияzip методшифрованияzip режимвосстановленияпутейфайловzip режимобработкиподкаталоговzip ' +
	  'режимсохраненияпутейzip уровеньсжатияzip ';

	  // v8 системные перечисления -
	  // Блокировка данных, Фоновые задания, Автоматизированное тестирование,
	  // Доставляемые уведомления, Встроенные покупки, Интернет, Работа с двоичными данными ==> class
	  var v8_system_enums_other =
	  'звуковоеоповещение направлениепереходакстроке позициявпотоке порядокбайтов режимблокировкиданных режимуправленияблокировкойданных ' +
	  'сервисвстроенныхпокупок состояниефоновогозадания типподписчикадоставляемыхуведомлений уровеньиспользованиязащищенногосоединенияftp ';

	  // v8 системные перечисления - схема запроса ==> class
	  var v8_system_enums_request_schema =
	  'направлениепорядкасхемызапроса типдополненияпериодамисхемызапроса типконтрольнойточкисхемызапроса типобъединениясхемызапроса ' +
	  'типпараметрадоступнойтаблицысхемызапроса типсоединениясхемызапроса ';

	  // v8 системные перечисления - свойства объектов метаданных ==> class
	  var v8_system_enums_properties_of_metadata_objects =
	  'httpметод автоиспользованиеобщегореквизита автопрефиксномеразадачи вариантвстроенногоязыка видиерархии видрегистранакопления ' +
	  'видтаблицывнешнегоисточникаданных записьдвиженийприпроведении заполнениепоследовательностей индексирование ' +
	  'использованиебазыпланавидоврасчета использованиебыстроговыбора использованиеобщегореквизита использованиеподчинения ' +
	  'использованиеполнотекстовогопоиска использованиеразделяемыхданныхобщегореквизита использованиереквизита ' +
	  'назначениеиспользованияприложения назначениерасширенияконфигурации направлениепередачи обновлениепредопределенныхданных ' +
	  'оперативноепроведение основноепредставлениевидарасчета основноепредставлениевидахарактеристики основноепредставлениезадачи ' +
	  'основноепредставлениепланаобмена основноепредставлениесправочника основноепредставлениесчета перемещениеграницыприпроведении ' +
	  'периодичностьномерабизнеспроцесса периодичностьномерадокумента периодичностьрегистрарасчета периодичностьрегистрасведений ' +
	  'повторноеиспользованиевозвращаемыхзначений полнотекстовыйпоискпривводепостроке принадлежностьобъекта проведение ' +
	  'разделениеаутентификацииобщегореквизита разделениеданныхобщегореквизита разделениерасширенийконфигурацииобщегореквизита '+
	  'режимавтонумерацииобъектов режимзаписирегистра режимиспользованиямодальности ' +
	  'режимиспользованиясинхронныхвызововрасширенийплатформыивнешнихкомпонент режимповторногоиспользованиясеансов ' +
	  'режимполученияданныхвыборапривводепостроке режимсовместимости режимсовместимостиинтерфейса ' +
	  'режимуправленияблокировкойданныхпоумолчанию сериикодовпланавидовхарактеристик сериикодовпланасчетов ' +
	  'сериикодовсправочника созданиепривводе способвыбора способпоискастрокипривводепостроке способредактирования ' +
	  'типданныхтаблицывнешнегоисточникаданных типкодапланавидоврасчета типкодасправочника типмакета типномерабизнеспроцесса ' +
	  'типномерадокумента типномеразадачи типформы удалениедвижений ';

	  // v8 системные перечисления - разные ==> class
	  var v8_system_enums_differents =
	  'важностьпроблемыприменениярасширенияконфигурации вариантинтерфейсаклиентскогоприложения вариантмасштабаформклиентскогоприложения ' +
	  'вариантосновногошрифтаклиентскогоприложения вариантстандартногопериода вариантстандартнойдатыначала видграницы видкартинки ' +
	  'видотображенияполнотекстовогопоиска видрамки видсравнения видцвета видчисловогозначения видшрифта допустимаядлина допустимыйзнак ' +
	  'использованиеbyteordermark использованиеметаданныхполнотекстовогопоиска источникрасширенийконфигурации клавиша кодвозвратадиалога ' +
	  'кодировкаxbase кодировкатекста направлениепоиска направлениесортировки обновлениепредопределенныхданных обновлениеприизмененииданных ' +
	  'отображениепанелиразделов проверказаполнения режимдиалогавопрос режимзапускаклиентскогоприложения режимокругления режимоткрытияформприложения ' +
	  'режимполнотекстовогопоиска скоростьклиентскогосоединения состояниевнешнегоисточникаданных состояниеобновленияконфигурациибазыданных ' +
	  'способвыборасертификатаwindows способкодированиястроки статуссообщения типвнешнейкомпоненты типплатформы типповеденияклавишиenter ' +
	  'типэлементаинформацииовыполненииобновленияконфигурациибазыданных уровеньизоляциитранзакций хешфункция частидаты';

	  // class: встроенные наборы значений, системные перечисления (содержат дочерние значения, обращения к которым через разыменование)
	  var CLASS =
	  v8_system_sets_of_values +
	  v8_system_enums_interface +
	  v8_system_enums_objects_properties +
	  v8_system_enums_exchange_plans +
	  v8_system_enums_tabular_document +
	  v8_system_enums_sheduler +
	  v8_system_enums_formatted_document +
	  v8_system_enums_query +
	  v8_system_enums_report_builder +
	  v8_system_enums_files +
	  v8_system_enums_query_builder +
	  v8_system_enums_data_analysis +
	  v8_system_enums_xml_json_xs_dom_xdto_ws +
	  v8_system_enums_data_composition_system +
	  v8_system_enums_email +
	  v8_system_enums_logbook +
	  v8_system_enums_cryptography +
	  v8_system_enums_zip +
	  v8_system_enums_other +
	  v8_system_enums_request_schema +
	  v8_system_enums_properties_of_metadata_objects +
	  v8_system_enums_differents;

	  // v8 общие объекты (у объектов есть конструктор, экземпляры создаются методом НОВЫЙ) ==> type
	  var v8_shared_object =
	  'comобъект ftpсоединение httpзапрос httpсервисответ httpсоединение wsопределения wsпрокси xbase анализданных аннотацияxs ' +
	  'блокировкаданных буфердвоичныхданных включениеxs выражениекомпоновкиданных генераторслучайныхчисел географическаясхема ' +
	  'географическиекоординаты графическаясхема группамоделиxs данныерасшифровкикомпоновкиданных двоичныеданные дендрограмма ' +
	  'диаграмма диаграммаганта диалогвыборафайла диалогвыборацвета диалогвыборашрифта диалограсписаниярегламентногозадания ' +
	  'диалогредактированиястандартногопериода диапазон документdom документhtml документацияxs доставляемоеуведомление ' +
	  'записьdom записьfastinfoset записьhtml записьjson записьxml записьzipфайла записьданных записьтекста записьузловdom ' +
	  'запрос защищенноесоединениеopenssl значенияполейрасшифровкикомпоновкиданных извлечениетекста импортxs интернетпочта ' +
	  'интернетпочтовоесообщение интернетпочтовыйпрофиль интернетпрокси интернетсоединение информациядляприложенияxs ' +
	  'использованиеатрибутаxs использованиесобытияжурналарегистрации источникдоступныхнастроеккомпоновкиданных ' +
	  'итераторузловdom картинка квалификаторыдаты квалификаторыдвоичныхданных квалификаторыстроки квалификаторычисла ' +
	  'компоновщикмакетакомпоновкиданных компоновщикнастроеккомпоновкиданных конструктормакетаоформлениякомпоновкиданных ' +
	  'конструкторнастроеккомпоновкиданных конструкторформатнойстроки линия макеткомпоновкиданных макетобластикомпоновкиданных ' +
	  'макетоформлениякомпоновкиданных маскаxs менеджеркриптографии наборсхемxml настройкикомпоновкиданных настройкисериализацииjson ' +
	  'обработкакартинок обработкарасшифровкикомпоновкиданных обходдереваdom объявлениеатрибутаxs объявлениенотацииxs ' +
	  'объявлениеэлементаxs описаниеиспользованиясобытиядоступжурналарегистрации ' +
	  'описаниеиспользованиясобытияотказвдоступежурналарегистрации описаниеобработкирасшифровкикомпоновкиданных ' +
	  'описаниепередаваемогофайла описаниетипов определениегруппыатрибутовxs определениегруппымоделиxs ' +
	  'определениеограниченияидентичностиxs определениепростоготипаxs определениесоставноготипаxs определениетипадокументаdom ' +
	  'определенияxpathxs отборкомпоновкиданных пакетотображаемыхдокументов параметрвыбора параметркомпоновкиданных ' +
	  'параметрызаписиjson параметрызаписиxml параметрычтенияxml переопределениеxs планировщик полеанализаданных ' +
	  'полекомпоновкиданных построительdom построительзапроса построительотчета построительотчетаанализаданных ' +
	  'построительсхемxml поток потоквпамяти почта почтовоесообщение преобразованиеxsl преобразованиекканоническомуxml ' +
	  'процессорвыводарезультатакомпоновкиданныхвколлекциюзначений процессорвыводарезультатакомпоновкиданныхвтабличныйдокумент ' +
	  'процессоркомпоновкиданных разыменовательпространствименdom рамка расписаниерегламентногозадания расширенноеимяxml ' +
	  'результатчтенияданных своднаядиаграмма связьпараметравыбора связьпотипу связьпотипукомпоновкиданных сериализаторxdto ' +
	  'сертификатклиентаwindows сертификатклиентафайл сертификаткриптографии сертификатыудостоверяющихцентровwindows ' +
	  'сертификатыудостоверяющихцентровфайл сжатиеданных системнаяинформация сообщениепользователю сочетаниеклавиш ' +
	  'сравнениезначений стандартнаядатаначала стандартныйпериод схемаxml схемакомпоновкиданных табличныйдокумент ' +
	  'текстовыйдокумент тестируемоеприложение типданныхxml уникальныйидентификатор фабрикаxdto файл файловыйпоток ' +
	  'фасетдлиныxs фасетколичестваразрядовдробнойчастиxs фасетмаксимальноговключающегозначенияxs ' +
	  'фасетмаксимальногоисключающегозначенияxs фасетмаксимальнойдлиныxs фасетминимальноговключающегозначенияxs ' +
	  'фасетминимальногоисключающегозначенияxs фасетминимальнойдлиныxs фасетобразцаxs фасетобщегоколичестваразрядовxs ' +
	  'фасетперечисленияxs фасетпробельныхсимволовxs фильтрузловdom форматированнаястрока форматированныйдокумент ' +
	  'фрагментxs хешированиеданных хранилищезначения цвет чтениеfastinfoset чтениеhtml чтениеjson чтениеxml чтениеzipфайла ' +
	  'чтениеданных чтениетекста чтениеузловdom шрифт элементрезультатакомпоновкиданных ';

	  // v8 универсальные коллекции значений ==> type
	  var v8_universal_collection =
	  'comsafearray деревозначений массив соответствие списокзначений структура таблицазначений фиксированнаяструктура ' +
	  'фиксированноесоответствие фиксированныймассив ';

	  // type : встроенные типы
	  var TYPE =
	  v8_shared_object +
	  v8_universal_collection;

	  // literal : примитивные типы
	  var LITERAL = 'null истина ложь неопределено';

	  // number : числа
	  var NUMBERS = hljs.inherit(hljs.NUMBER_MODE);

	  // string : строки
	  var STRINGS = {
	    className: 'string',
	    begin: '"|\\|', end: '"|$',
	    contains: [{begin: '""'}]
	  };

	  // number : даты
	  var DATE = {
	    begin: "'", end: "'", excludeBegin: true, excludeEnd: true,
	    contains: [
	      {
	        className: 'number',
	        begin: '\\d{4}([\\.\\\\/:-]?\\d{2}){0,5}'
	      }
	    ]
	  };

	  // comment : комментарии
	  var COMMENTS = hljs.inherit(hljs.C_LINE_COMMENT_MODE);

	  // meta : инструкции препроцессора, директивы компиляции
	  var META = {
	    className: 'meta',

	    begin: '#|&', end: '$',
	    keywords: {
	      $pattern: UNDERSCORE_IDENT_RE,
	      'meta-keyword': KEYWORD + METAKEYWORD
	    },
	    contains: [
	      COMMENTS
	    ]
	  };

	  // symbol : метка goto
	  var SYMBOL = {
	    className: 'symbol',
	    begin: '~', end: ';|:', excludeEnd: true
	  };

	  // function : объявление процедур и функций
	  var FUNCTION = {
	    className: 'function',
	    variants: [
	      {begin: 'процедура|функция', end: '\\)', keywords: 'процедура функция'},
	      {begin: 'конецпроцедуры|конецфункции', keywords: 'конецпроцедуры конецфункции'}
	    ],
	    contains: [
	      {
	        begin: '\\(', end: '\\)', endsParent : true,
	        contains: [
	          {
	            className: 'params',
	            begin: UNDERSCORE_IDENT_RE, end: ',', excludeEnd: true, endsWithParent: true,
	            keywords: {
	              $pattern: UNDERSCORE_IDENT_RE,
	              keyword: 'знач',
	              literal: LITERAL
	            },
	            contains: [
	              NUMBERS,
	              STRINGS,
	              DATE
	            ]
	          },
	          COMMENTS
	        ]
	      },
	      hljs.inherit(hljs.TITLE_MODE, {begin: UNDERSCORE_IDENT_RE})
	    ]
	  };

	  return {
	    name: '1C:Enterprise',
	    case_insensitive: true,
	    keywords: {
	      $pattern: UNDERSCORE_IDENT_RE,
	      keyword: KEYWORD,
	      built_in: BUILTIN,
	      class: CLASS,
	      type: TYPE,
	      literal: LITERAL
	    },
	    contains: [
	      META,
	      FUNCTION,
	      COMMENTS,
	      SYMBOL,
	      NUMBERS,
	      STRINGS,
	      DATE
	    ]
	  };
	}

	_1c_1 = _1c;
	return _1c_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var abnf_1;
var hasRequiredAbnf;

function requireAbnf () {
	if (hasRequiredAbnf) return abnf_1;
	hasRequiredAbnf = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: Augmented Backus-Naur Form
	Author: Alex McKibben <alex@nullscope.net>
	Website: https://tools.ietf.org/html/rfc5234
	Audit: 2020
	*/

	/** @type LanguageFn */
	function abnf(hljs) {
	  const regexes = {
	    ruleDeclaration: /^[a-zA-Z][a-zA-Z0-9-]*/,
	    unexpectedChars: /[!@#$^&',?+~`|:]/
	  };

	  const keywords = [
	    "ALPHA",
	    "BIT",
	    "CHAR",
	    "CR",
	    "CRLF",
	    "CTL",
	    "DIGIT",
	    "DQUOTE",
	    "HEXDIG",
	    "HTAB",
	    "LF",
	    "LWSP",
	    "OCTET",
	    "SP",
	    "VCHAR",
	    "WSP"
	  ];

	  const commentMode = hljs.COMMENT(/;/, /$/);

	  const terminalBinaryMode = {
	    className: "symbol",
	    begin: /%b[0-1]+(-[0-1]+|(\.[0-1]+)+){0,1}/
	  };

	  const terminalDecimalMode = {
	    className: "symbol",
	    begin: /%d[0-9]+(-[0-9]+|(\.[0-9]+)+){0,1}/
	  };

	  const terminalHexadecimalMode = {
	    className: "symbol",
	    begin: /%x[0-9A-F]+(-[0-9A-F]+|(\.[0-9A-F]+)+){0,1}/
	  };

	  const caseSensitivityIndicatorMode = {
	    className: "symbol",
	    begin: /%[si]/
	  };

	  const ruleDeclarationMode = {
	    className: "attribute",
	    begin: concat(regexes.ruleDeclaration, /(?=\s*=)/)
	  };

	  return {
	    name: 'Augmented Backus-Naur Form',
	    illegal: regexes.unexpectedChars,
	    keywords: keywords,
	    contains: [
	      ruleDeclarationMode,
	      commentMode,
	      terminalBinaryMode,
	      terminalDecimalMode,
	      terminalHexadecimalMode,
	      caseSensitivityIndicatorMode,
	      hljs.QUOTE_STRING_MODE,
	      hljs.NUMBER_MODE
	    ]
	  };
	}

	abnf_1 = abnf;
	return abnf_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var accesslog_1;
var hasRequiredAccesslog;

function requireAccesslog () {
	if (hasRequiredAccesslog) return accesslog_1;
	hasRequiredAccesslog = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/*
	 Language: Apache Access Log
	 Author: Oleg Efimov <efimovov@gmail.com>
	 Description: Apache/Nginx Access Logs
	 Website: https://httpd.apache.org/docs/2.4/logs.html#accesslog
	 Audit: 2020
	 */

	/** @type LanguageFn */
	function accesslog(_hljs) {
	  // https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods
	  const HTTP_VERBS = [
	    "GET",
	    "POST",
	    "HEAD",
	    "PUT",
	    "DELETE",
	    "CONNECT",
	    "OPTIONS",
	    "PATCH",
	    "TRACE"
	  ];
	  return {
	    name: 'Apache Access Log',
	    contains: [
	      // IP
	      {
	        className: 'number',
	        begin: /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}(:\d{1,5})?\b/,
	        relevance: 5
	      },
	      // Other numbers
	      {
	        className: 'number',
	        begin: /\b\d+\b/,
	        relevance: 0
	      },
	      // Requests
	      {
	        className: 'string',
	        begin: concat(/"/, either(...HTTP_VERBS)),
	        end: /"/,
	        keywords: HTTP_VERBS,
	        illegal: /\n/,
	        relevance: 5,
	        contains: [
	          {
	            begin: /HTTP\/[12]\.\d'/,
	            relevance: 5
	          }
	        ]
	      },
	      // Dates
	      {
	        className: 'string',
	        // dates must have a certain length, this prevents matching
	        // simple array accesses a[123] and [] and other common patterns
	        // found in other languages
	        begin: /\[\d[^\]\n]{8,}\]/,
	        illegal: /\n/,
	        relevance: 1
	      },
	      {
	        className: 'string',
	        begin: /\[/,
	        end: /\]/,
	        illegal: /\n/,
	        relevance: 0
	      },
	      // User agent / relevance boost
	      {
	        className: 'string',
	        begin: /"Mozilla\/\d\.\d \(/,
	        end: /"/,
	        illegal: /\n/,
	        relevance: 3
	      },
	      // Strings
	      {
	        className: 'string',
	        begin: /"/,
	        end: /"/,
	        illegal: /\n/,
	        relevance: 0
	      }
	    ]
	  };
	}

	accesslog_1 = accesslog;
	return accesslog_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var actionscript_1;
var hasRequiredActionscript;

function requireActionscript () {
	if (hasRequiredActionscript) return actionscript_1;
	hasRequiredActionscript = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: ActionScript
	Author: Alexander Myadzel <myadzel@gmail.com>
	Category: scripting
	Audit: 2020
	*/

	/** @type LanguageFn */
	function actionscript(hljs) {
	  const IDENT_RE = /[a-zA-Z_$][a-zA-Z0-9_$]*/;
	  const IDENT_FUNC_RETURN_TYPE_RE = /([*]|[a-zA-Z_$][a-zA-Z0-9_$]*)/;

	  const AS3_REST_ARG_MODE = {
	    className: 'rest_arg',
	    begin: /[.]{3}/,
	    end: IDENT_RE,
	    relevance: 10
	  };

	  return {
	    name: 'ActionScript',
	    aliases: [ 'as' ],
	    keywords: {
	      keyword: 'as break case catch class const continue default delete do dynamic each ' +
	        'else extends final finally for function get if implements import in include ' +
	        'instanceof interface internal is namespace native new override package private ' +
	        'protected public return set static super switch this throw try typeof use var void ' +
	        'while with',
	      literal: 'true false null undefined'
	    },
	    contains: [
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.C_NUMBER_MODE,
	      {
	        className: 'class',
	        beginKeywords: 'package',
	        end: /\{/,
	        contains: [ hljs.TITLE_MODE ]
	      },
	      {
	        className: 'class',
	        beginKeywords: 'class interface',
	        end: /\{/,
	        excludeEnd: true,
	        contains: [
	          { beginKeywords: 'extends implements' },
	          hljs.TITLE_MODE
	        ]
	      },
	      {
	        className: 'meta',
	        beginKeywords: 'import include',
	        end: /;/,
	        keywords: { 'meta-keyword': 'import include' }
	      },
	      {
	        className: 'function',
	        beginKeywords: 'function',
	        end: /[{;]/,
	        excludeEnd: true,
	        illegal: /\S/,
	        contains: [
	          hljs.TITLE_MODE,
	          {
	            className: 'params',
	            begin: /\(/,
	            end: /\)/,
	            contains: [
	              hljs.APOS_STRING_MODE,
	              hljs.QUOTE_STRING_MODE,
	              hljs.C_LINE_COMMENT_MODE,
	              hljs.C_BLOCK_COMMENT_MODE,
	              AS3_REST_ARG_MODE
	            ]
	          },
	          { begin: concat(/:\s*/, IDENT_FUNC_RETURN_TYPE_RE) }
	        ]
	      },
	      hljs.METHOD_GUARD
	    ],
	    illegal: /#/
	  };
	}

	actionscript_1 = actionscript;
	return actionscript_1;
}

/*
Language: Ada
Author: Lars Schulna <kartoffelbrei.mit.muskatnuss@gmail.org>
Description: Ada is a general-purpose programming language that has great support for saftey critical and real-time applications.
             It has been developed by the DoD and thus has been used in military and safety-critical applications (like civil aviation).
             The first version appeared in the 80s, but it's still actively developed today with
             the newest standard being Ada2012.
*/

var ada_1;
var hasRequiredAda;

function requireAda () {
	if (hasRequiredAda) return ada_1;
	hasRequiredAda = 1;
	// We try to support full Ada2012
	//
	// We highlight all appearances of types, keywords, literals (string, char, number, bool)
	// and titles (user defined function/procedure/package)
	// CSS classes are set accordingly
	//
	// Languages causing problems for language detection:
	// xml (broken by Foo : Bar type), elm (broken by Foo : Bar type), vbscript-html (broken by body keyword)
	// sql (ada default.txt has a lot of sql keywords)

	/** @type LanguageFn */
	function ada(hljs) {
	  // Regular expression for Ada numeric literals.
	  // stolen form the VHDL highlighter

	  // Decimal literal:
	  const INTEGER_RE = '\\d(_|\\d)*';
	  const EXPONENT_RE = '[eE][-+]?' + INTEGER_RE;
	  const DECIMAL_LITERAL_RE = INTEGER_RE + '(\\.' + INTEGER_RE + ')?' + '(' + EXPONENT_RE + ')?';

	  // Based literal:
	  const BASED_INTEGER_RE = '\\w+';
	  const BASED_LITERAL_RE = INTEGER_RE + '#' + BASED_INTEGER_RE + '(\\.' + BASED_INTEGER_RE + ')?' + '#' + '(' + EXPONENT_RE + ')?';

	  const NUMBER_RE = '\\b(' + BASED_LITERAL_RE + '|' + DECIMAL_LITERAL_RE + ')';

	  // Identifier regex
	  const ID_REGEX = '[A-Za-z](_?[A-Za-z0-9.])*';

	  // bad chars, only allowed in literals
	  const BAD_CHARS = `[]\\{\\}%#'"`;

	  // Ada doesn't have block comments, only line comments
	  const COMMENTS = hljs.COMMENT('--', '$');

	  // variable declarations of the form
	  // Foo : Bar := Baz;
	  // where only Bar will be highlighted
	  const VAR_DECLS = {
	    // TODO: These spaces are not required by the Ada syntax
	    // however, I have yet to see handwritten Ada code where
	    // someone does not put spaces around :
	    begin: '\\s+:\\s+',
	    end: '\\s*(:=|;|\\)|=>|$)',
	    // endsWithParent: true,
	    // returnBegin: true,
	    illegal: BAD_CHARS,
	    contains: [
	      {
	        // workaround to avoid highlighting
	        // named loops and declare blocks
	        beginKeywords: 'loop for declare others',
	        endsParent: true
	      },
	      {
	        // properly highlight all modifiers
	        className: 'keyword',
	        beginKeywords: 'not null constant access function procedure in out aliased exception'
	      },
	      {
	        className: 'type',
	        begin: ID_REGEX,
	        endsParent: true,
	        relevance: 0
	      }
	    ]
	  };

	  return {
	    name: 'Ada',
	    case_insensitive: true,
	    keywords: {
	      keyword:
	                'abort else new return abs elsif not reverse abstract end ' +
	                'accept entry select access exception of separate aliased exit or some ' +
	                'all others subtype and for out synchronized array function overriding ' +
	                'at tagged generic package task begin goto pragma terminate ' +
	                'body private then if procedure type case in protected constant interface ' +
	                'is raise use declare range delay limited record when delta loop rem while ' +
	                'digits renames with do mod requeue xor',
	      literal:
	                'True False'
	    },
	    contains: [
	      COMMENTS,
	      // strings "foobar"
	      {
	        className: 'string',
	        begin: /"/,
	        end: /"/,
	        contains: [{
	          begin: /""/,
	          relevance: 0
	        }]
	      },
	      // characters ''
	      {
	        // character literals always contain one char
	        className: 'string',
	        begin: /'.'/
	      },
	      {
	        // number literals
	        className: 'number',
	        begin: NUMBER_RE,
	        relevance: 0
	      },
	      {
	        // Attributes
	        className: 'symbol',
	        begin: "'" + ID_REGEX
	      },
	      {
	        // package definition, maybe inside generic
	        className: 'title',
	        begin: '(\\bwith\\s+)?(\\bprivate\\s+)?\\bpackage\\s+(\\bbody\\s+)?',
	        end: '(is|$)',
	        keywords: 'package body',
	        excludeBegin: true,
	        excludeEnd: true,
	        illegal: BAD_CHARS
	      },
	      {
	        // function/procedure declaration/definition
	        // maybe inside generic
	        begin: '(\\b(with|overriding)\\s+)?\\b(function|procedure)\\s+',
	        end: '(\\bis|\\bwith|\\brenames|\\)\\s*;)',
	        keywords: 'overriding function procedure with is renames return',
	        // we need to re-match the 'function' keyword, so that
	        // the title mode below matches only exactly once
	        returnBegin: true,
	        contains:
	                [
	                  COMMENTS,
	                  {
	                    // name of the function/procedure
	                    className: 'title',
	                    begin: '(\\bwith\\s+)?\\b(function|procedure)\\s+',
	                    end: '(\\(|\\s+|$)',
	                    excludeBegin: true,
	                    excludeEnd: true,
	                    illegal: BAD_CHARS
	                  },
	                  // 'self'
	                  // // parameter types
	                  VAR_DECLS,
	                  {
	                    // return type
	                    className: 'type',
	                    begin: '\\breturn\\s+',
	                    end: '(\\s+|;|$)',
	                    keywords: 'return',
	                    excludeBegin: true,
	                    excludeEnd: true,
	                    // we are done with functions
	                    endsParent: true,
	                    illegal: BAD_CHARS

	                  }
	                ]
	      },
	      {
	        // new type declarations
	        // maybe inside generic
	        className: 'type',
	        begin: '\\b(sub)?type\\s+',
	        end: '\\s+',
	        keywords: 'type',
	        excludeBegin: true,
	        illegal: BAD_CHARS
	      },

	      // see comment above the definition
	      VAR_DECLS

	      // no markup
	      // relevance boosters for small snippets
	      // {begin: '\\s*=>\\s*'},
	      // {begin: '\\s*:=\\s*'},
	      // {begin: '\\s+:=\\s+'},
	    ]
	  };
	}

	ada_1 = ada;
	return ada_1;
}

/*
Language: AngelScript
Author: Melissa Geels <melissa@nimble.tools>
Category: scripting
Website: https://www.angelcode.com/angelscript/
*/

var angelscript_1;
var hasRequiredAngelscript;

function requireAngelscript () {
	if (hasRequiredAngelscript) return angelscript_1;
	hasRequiredAngelscript = 1;
	/** @type LanguageFn */
	function angelscript(hljs) {
	  var builtInTypeMode = {
	    className: 'built_in',
	    begin: '\\b(void|bool|int|int8|int16|int32|int64|uint|uint8|uint16|uint32|uint64|string|ref|array|double|float|auto|dictionary)'
	  };

	  var objectHandleMode = {
	    className: 'symbol',
	    begin: '[a-zA-Z0-9_]+@'
	  };

	  var genericMode = {
	    className: 'keyword',
	    begin: '<', end: '>',
	    contains: [ builtInTypeMode, objectHandleMode ]
	  };

	  builtInTypeMode.contains = [ genericMode ];
	  objectHandleMode.contains = [ genericMode ];

	  return {
	    name: 'AngelScript',
	    aliases: ['asc'],

	    keywords:
	      'for in|0 break continue while do|0 return if else case switch namespace is cast ' +
	      'or and xor not get|0 in inout|10 out override set|0 private public const default|0 ' +
	      'final shared external mixin|10 enum typedef funcdef this super import from interface ' +
	      'abstract|0 try catch protected explicit property',

	    // avoid close detection with C# and JS
	    illegal: '(^using\\s+[A-Za-z0-9_\\.]+;$|\\bfunction\\s*[^\\(])',

	    contains: [
	      { // 'strings'
	        className: 'string',
	        begin: '\'', end: '\'',
	        illegal: '\\n',
	        contains: [ hljs.BACKSLASH_ESCAPE ],
	        relevance: 0
	      },

	      // """heredoc strings"""
	      {
	        className: 'string',
	        begin: '"""', end: '"""'
	      },

	      { // "strings"
	        className: 'string',
	        begin: '"', end: '"',
	        illegal: '\\n',
	        contains: [ hljs.BACKSLASH_ESCAPE ],
	        relevance: 0
	      },

	      hljs.C_LINE_COMMENT_MODE, // single-line comments
	      hljs.C_BLOCK_COMMENT_MODE, // comment blocks

	      { // metadata
	        className: 'string',
	        begin: '^\\s*\\[', end: '\\]',
	      },

	      { // interface or namespace declaration
	        beginKeywords: 'interface namespace', end: /\{/,
	        illegal: '[;.\\-]',
	        contains: [
	          { // interface or namespace name
	            className: 'symbol',
	            begin: '[a-zA-Z0-9_]+'
	          }
	        ]
	      },

	      { // class declaration
	        beginKeywords: 'class', end: /\{/,
	        illegal: '[;.\\-]',
	        contains: [
	          { // class name
	            className: 'symbol',
	            begin: '[a-zA-Z0-9_]+',
	            contains: [
	              {
	                begin: '[:,]\\s*',
	                contains: [
	                  {
	                    className: 'symbol',
	                    begin: '[a-zA-Z0-9_]+'
	                  }
	                ]
	              }
	            ]
	          }
	        ]
	      },

	      builtInTypeMode, // built-in types
	      objectHandleMode, // object handles

	      { // literals
	        className: 'literal',
	        begin: '\\b(null|true|false)'
	      },

	      { // numbers
	        className: 'number',
	        relevance: 0,
	        begin: '(-?)(\\b0[xXbBoOdD][a-fA-F0-9]+|(\\b\\d+(\\.\\d*)?f?|\\.\\d+f?)([eE][-+]?\\d+f?)?)'
	      }
	    ]
	  };
	}

	angelscript_1 = angelscript;
	return angelscript_1;
}

/*
Language: Apache config
Author: Ruslan Keba <rukeba@gmail.com>
Contributors: Ivan Sagalaev <maniac@softwaremaniacs.org>
Website: https://httpd.apache.org
Description: language definition for Apache configuration files (httpd.conf & .htaccess)
Category: common, config
Audit: 2020
*/

var apache_1;
var hasRequiredApache;

function requireApache () {
	if (hasRequiredApache) return apache_1;
	hasRequiredApache = 1;
	/** @type LanguageFn */
	function apache(hljs) {
	  const NUMBER_REF = {
	    className: 'number',
	    begin: /[$%]\d+/
	  };
	  const NUMBER = {
	    className: 'number',
	    begin: /\d+/
	  };
	  const IP_ADDRESS = {
	    className: "number",
	    begin: /\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}(:\d{1,5})?/
	  };
	  const PORT_NUMBER = {
	    className: "number",
	    begin: /:\d{1,5}/
	  };
	  return {
	    name: 'Apache config',
	    aliases: [ 'apacheconf' ],
	    case_insensitive: true,
	    contains: [
	      hljs.HASH_COMMENT_MODE,
	      {
	        className: 'section',
	        begin: /<\/?/,
	        end: />/,
	        contains: [
	          IP_ADDRESS,
	          PORT_NUMBER,
	          // low relevance prevents us from claming XML/HTML where this rule would
	          // match strings inside of XML tags
	          hljs.inherit(hljs.QUOTE_STRING_MODE, { relevance: 0 })
	        ]
	      },
	      {
	        className: 'attribute',
	        begin: /\w+/,
	        relevance: 0,
	        // keywords aren’t needed for highlighting per se, they only boost relevance
	        // for a very generally defined mode (starts with a word, ends with line-end
	        keywords: {
	          nomarkup:
	            'order deny allow setenv rewriterule rewriteengine rewritecond documentroot ' +
	            'sethandler errordocument loadmodule options header listen serverroot ' +
	            'servername'
	        },
	        starts: {
	          end: /$/,
	          relevance: 0,
	          keywords: { literal: 'on off all deny allow' },
	          contains: [
	            {
	              className: 'meta',
	              begin: /\s\[/,
	              end: /\]$/
	            },
	            {
	              className: 'variable',
	              begin: /[\$%]\{/,
	              end: /\}/,
	              contains: [
	                'self',
	                NUMBER_REF
	              ]
	            },
	            IP_ADDRESS,
	            NUMBER,
	            hljs.QUOTE_STRING_MODE
	          ]
	        }
	      }
	    ],
	    illegal: /\S/
	  };
	}

	apache_1 = apache;
	return apache_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var applescript_1;
var hasRequiredApplescript;

function requireApplescript () {
	if (hasRequiredApplescript) return applescript_1;
	hasRequiredApplescript = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/*
	Language: AppleScript
	Authors: Nathan Grigg <nathan@nathanamy.org>, Dr. Drang <drdrang@gmail.com>
	Category: scripting
	Website: https://developer.apple.com/library/archive/documentation/AppleScript/Conceptual/AppleScriptLangGuide/introduction/ASLR_intro.html
	Audit: 2020
	*/

	/** @type LanguageFn */
	function applescript(hljs) {
	  const STRING = hljs.inherit(
	    hljs.QUOTE_STRING_MODE, {
	      illegal: null
	    });
	  const PARAMS = {
	    className: 'params',
	    begin: /\(/,
	    end: /\)/,
	    contains: [
	      'self',
	      hljs.C_NUMBER_MODE,
	      STRING
	    ]
	  };
	  const COMMENT_MODE_1 = hljs.COMMENT(/--/, /$/);
	  const COMMENT_MODE_2 = hljs.COMMENT(
	    /\(\*/,
	    /\*\)/,
	    {
	      contains: [
	        'self', // allow nesting
	        COMMENT_MODE_1
	      ]
	    }
	  );
	  const COMMENTS = [
	    COMMENT_MODE_1,
	    COMMENT_MODE_2,
	    hljs.HASH_COMMENT_MODE
	  ];

	  const KEYWORD_PATTERNS = [
	    /apart from/,
	    /aside from/,
	    /instead of/,
	    /out of/,
	    /greater than/,
	    /isn't|(doesn't|does not) (equal|come before|come after|contain)/,
	    /(greater|less) than( or equal)?/,
	    /(starts?|ends|begins?) with/,
	    /contained by/,
	    /comes (before|after)/,
	    /a (ref|reference)/,
	    /POSIX (file|path)/,
	    /(date|time) string/,
	    /quoted form/
	  ];

	  const BUILT_IN_PATTERNS = [
	    /clipboard info/,
	    /the clipboard/,
	    /info for/,
	    /list (disks|folder)/,
	    /mount volume/,
	    /path to/,
	    /(close|open for) access/,
	    /(get|set) eof/,
	    /current date/,
	    /do shell script/,
	    /get volume settings/,
	    /random number/,
	    /set volume/,
	    /system attribute/,
	    /system info/,
	    /time to GMT/,
	    /(load|run|store) script/,
	    /scripting components/,
	    /ASCII (character|number)/,
	    /localized string/,
	    /choose (application|color|file|file name|folder|from list|remote application|URL)/,
	    /display (alert|dialog)/
	  ];

	  return {
	    name: 'AppleScript',
	    aliases: [ 'osascript' ],
	    keywords: {
	      keyword:
	        'about above after against and around as at back before beginning ' +
	        'behind below beneath beside between but by considering ' +
	        'contain contains continue copy div does eighth else end equal ' +
	        'equals error every exit fifth first for fourth from front ' +
	        'get given global if ignoring in into is it its last local me ' +
	        'middle mod my ninth not of on onto or over prop property put ref ' +
	        'reference repeat returning script second set seventh since ' +
	        'sixth some tell tenth that the|0 then third through thru ' +
	        'timeout times to transaction try until where while whose with ' +
	        'without',
	      literal:
	        'AppleScript false linefeed return pi quote result space tab true',
	      built_in:
	        'alias application boolean class constant date file integer list ' +
	        'number real record string text ' +
	        'activate beep count delay launch log offset read round ' +
	        'run say summarize write ' +
	        'character characters contents day frontmost id item length ' +
	        'month name paragraph paragraphs rest reverse running time version ' +
	        'weekday word words year'
	    },
	    contains: [
	      STRING,
	      hljs.C_NUMBER_MODE,
	      {
	        className: 'built_in',
	        begin: concat(
	          /\b/,
	          either(...BUILT_IN_PATTERNS),
	          /\b/
	        )
	      },
	      {
	        className: 'built_in',
	        begin: /^\s*return\b/
	      },
	      {
	        className: 'literal',
	        begin:
	          /\b(text item delimiters|current application|missing value)\b/
	      },
	      {
	        className: 'keyword',
	        begin: concat(
	          /\b/,
	          either(...KEYWORD_PATTERNS),
	          /\b/
	        )
	      },
	      {
	        beginKeywords: 'on',
	        illegal: /[${=;\n]/,
	        contains: [
	          hljs.UNDERSCORE_TITLE_MODE,
	          PARAMS
	        ]
	      },
	      ...COMMENTS
	    ],
	    illegal: /\/\/|->|=>|\[\[/
	  };
	}

	applescript_1 = applescript;
	return applescript_1;
}

/*
 Language: ArcGIS Arcade
 Category: scripting
 Author: John Foster <jfoster@esri.com>
 Website: https://developers.arcgis.com/arcade/
 Description: ArcGIS Arcade is an expression language used in many Esri ArcGIS products such as Pro, Online, Server, Runtime, JavaScript, and Python
*/

var arcade_1;
var hasRequiredArcade;

function requireArcade () {
	if (hasRequiredArcade) return arcade_1;
	hasRequiredArcade = 1;
	/** @type LanguageFn */
	function arcade(hljs) {
	  const IDENT_RE = '[A-Za-z_][0-9A-Za-z_]*';
	  const KEYWORDS = {
	    keyword:
	      'if for while var new function do return void else break',
	    literal:
	      'BackSlash DoubleQuote false ForwardSlash Infinity NaN NewLine null PI SingleQuote Tab TextFormatting true undefined',
	    built_in:
	      'Abs Acos Angle Attachments Area AreaGeodetic Asin Atan Atan2 Average Bearing Boolean Buffer BufferGeodetic ' +
	      'Ceil Centroid Clip Console Constrain Contains Cos Count Crosses Cut Date DateAdd ' +
	      'DateDiff Day Decode DefaultValue Dictionary Difference Disjoint Distance DistanceGeodetic Distinct ' +
	      'DomainCode DomainName Equals Exp Extent Feature FeatureSet FeatureSetByAssociation FeatureSetById FeatureSetByPortalItem ' +
	      'FeatureSetByRelationshipName FeatureSetByTitle FeatureSetByUrl Filter First Floor Geometry GroupBy Guid HasKey Hour IIf IndexOf ' +
	      'Intersection Intersects IsEmpty IsNan IsSelfIntersecting Length LengthGeodetic Log Max Mean Millisecond Min Minute Month ' +
	      'MultiPartToSinglePart Multipoint NextSequenceValue Now Number OrderBy Overlaps Point Polygon ' +
	      'Polyline Portal Pow Random Relate Reverse RingIsClockWise Round Second SetGeometry Sin Sort Sqrt Stdev Sum ' +
	      'SymmetricDifference Tan Text Timestamp Today ToLocal Top Touches ToUTC TrackCurrentTime ' +
	      'TrackGeometryWindow TrackIndex TrackStartTime TrackWindow TypeOf Union UrlEncode Variance ' +
	      'Weekday When Within Year '
	  };
	  const SYMBOL = {
	    className: 'symbol',
	    begin: '\\$[datastore|feature|layer|map|measure|sourcefeature|sourcelayer|targetfeature|targetlayer|value|view]+'
	  };
	  const NUMBER = {
	    className: 'number',
	    variants: [
	      {
	        begin: '\\b(0[bB][01]+)'
	      },
	      {
	        begin: '\\b(0[oO][0-7]+)'
	      },
	      {
	        begin: hljs.C_NUMBER_RE
	      }
	    ],
	    relevance: 0
	  };
	  const SUBST = {
	    className: 'subst',
	    begin: '\\$\\{',
	    end: '\\}',
	    keywords: KEYWORDS,
	    contains: [] // defined later
	  };
	  const TEMPLATE_STRING = {
	    className: 'string',
	    begin: '`',
	    end: '`',
	    contains: [
	      hljs.BACKSLASH_ESCAPE,
	      SUBST
	    ]
	  };
	  SUBST.contains = [
	    hljs.APOS_STRING_MODE,
	    hljs.QUOTE_STRING_MODE,
	    TEMPLATE_STRING,
	    NUMBER,
	    hljs.REGEXP_MODE
	  ];
	  const PARAMS_CONTAINS = SUBST.contains.concat([
	    hljs.C_BLOCK_COMMENT_MODE,
	    hljs.C_LINE_COMMENT_MODE
	  ]);

	  return {
	    name: 'ArcGIS Arcade',
	    keywords: KEYWORDS,
	    contains: [
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      TEMPLATE_STRING,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      SYMBOL,
	      NUMBER,
	      { // object attr container
	        begin: /[{,]\s*/,
	        relevance: 0,
	        contains: [{
	          begin: IDENT_RE + '\\s*:',
	          returnBegin: true,
	          relevance: 0,
	          contains: [{
	            className: 'attr',
	            begin: IDENT_RE,
	            relevance: 0
	          }]
	        }]
	      },
	      { // "value" container
	        begin: '(' + hljs.RE_STARTERS_RE + '|\\b(return)\\b)\\s*',
	        keywords: 'return',
	        contains: [
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE,
	          hljs.REGEXP_MODE,
	          {
	            className: 'function',
	            begin: '(\\(.*?\\)|' + IDENT_RE + ')\\s*=>',
	            returnBegin: true,
	            end: '\\s*=>',
	            contains: [{
	              className: 'params',
	              variants: [
	                {
	                  begin: IDENT_RE
	                },
	                {
	                  begin: /\(\s*\)/
	                },
	                {
	                  begin: /\(/,
	                  end: /\)/,
	                  excludeBegin: true,
	                  excludeEnd: true,
	                  keywords: KEYWORDS,
	                  contains: PARAMS_CONTAINS
	                }
	              ]
	            }]
	          }
	        ],
	        relevance: 0
	      },
	      {
	        className: 'function',
	        beginKeywords: 'function',
	        end: /\{/,
	        excludeEnd: true,
	        contains: [
	          hljs.inherit(hljs.TITLE_MODE, {
	            begin: IDENT_RE
	          }),
	          {
	            className: 'params',
	            begin: /\(/,
	            end: /\)/,
	            excludeBegin: true,
	            excludeEnd: true,
	            contains: PARAMS_CONTAINS
	          }
	        ],
	        illegal: /\[|%/
	      },
	      {
	        begin: /\$[(.]/
	      }
	    ],
	    illegal: /#(?!!)/
	  };
	}

	arcade_1 = arcade;
	return arcade_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var arduino_1;
var hasRequiredArduino;

function requireArduino () {
	if (hasRequiredArduino) return arduino_1;
	hasRequiredArduino = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function optional(re) {
	  return concat('(', re, ')?');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: C++
	Category: common, system
	Website: https://isocpp.org
	*/

	/** @type LanguageFn */
	function cPlusPlus(hljs) {
	  // added for historic reasons because `hljs.C_LINE_COMMENT_MODE` does
	  // not include such support nor can we be sure all the grammars depending
	  // on it would desire this behavior
	  const C_LINE_COMMENT_MODE = hljs.COMMENT('//', '$', {
	    contains: [
	      {
	        begin: /\\\n/
	      }
	    ]
	  });
	  const DECLTYPE_AUTO_RE = 'decltype\\(auto\\)';
	  const NAMESPACE_RE = '[a-zA-Z_]\\w*::';
	  const TEMPLATE_ARGUMENT_RE = '<[^<>]+>';
	  const FUNCTION_TYPE_RE = '(' +
	    DECLTYPE_AUTO_RE + '|' +
	    optional(NAMESPACE_RE) +
	    '[a-zA-Z_]\\w*' + optional(TEMPLATE_ARGUMENT_RE) +
	  ')';
	  const CPP_PRIMITIVE_TYPES = {
	    className: 'keyword',
	    begin: '\\b[a-z\\d_]*_t\\b'
	  };

	  // https://en.cppreference.com/w/cpp/language/escape
	  // \\ \x \xFF \u2837 \u00323747 \374
	  const CHARACTER_ESCAPES = '\\\\(x[0-9A-Fa-f]{2}|u[0-9A-Fa-f]{4,8}|[0-7]{3}|\\S)';
	  const STRINGS = {
	    className: 'string',
	    variants: [
	      {
	        begin: '(u8?|U|L)?"',
	        end: '"',
	        illegal: '\\n',
	        contains: [ hljs.BACKSLASH_ESCAPE ]
	      },
	      {
	        begin: '(u8?|U|L)?\'(' + CHARACTER_ESCAPES + "|.)",
	        end: '\'',
	        illegal: '.'
	      },
	      hljs.END_SAME_AS_BEGIN({
	        begin: /(?:u8?|U|L)?R"([^()\\ ]{0,16})\(/,
	        end: /\)([^()\\ ]{0,16})"/
	      })
	    ]
	  };

	  const NUMBERS = {
	    className: 'number',
	    variants: [
	      {
	        begin: '\\b(0b[01\']+)'
	      },
	      {
	        begin: '(-?)\\b([\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)((ll|LL|l|L)(u|U)?|(u|U)(ll|LL|l|L)?|f|F|b|B)'
	      },
	      {
	        begin: '(-?)(\\b0[xX][a-fA-F0-9\']+|(\\b[\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)([eE][-+]?[\\d\']+)?)'
	      }
	    ],
	    relevance: 0
	  };

	  const PREPROCESSOR = {
	    className: 'meta',
	    begin: /#\s*[a-z]+\b/,
	    end: /$/,
	    keywords: {
	      'meta-keyword':
	        'if else elif endif define undef warning error line ' +
	        'pragma _Pragma ifdef ifndef include'
	    },
	    contains: [
	      {
	        begin: /\\\n/,
	        relevance: 0
	      },
	      hljs.inherit(STRINGS, {
	        className: 'meta-string'
	      }),
	      {
	        className: 'meta-string',
	        begin: /<.*?>/
	      },
	      C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE
	    ]
	  };

	  const TITLE_MODE = {
	    className: 'title',
	    begin: optional(NAMESPACE_RE) + hljs.IDENT_RE,
	    relevance: 0
	  };

	  const FUNCTION_TITLE = optional(NAMESPACE_RE) + hljs.IDENT_RE + '\\s*\\(';

	  const COMMON_CPP_HINTS = [
	    'asin',
	    'atan2',
	    'atan',
	    'calloc',
	    'ceil',
	    'cosh',
	    'cos',
	    'exit',
	    'exp',
	    'fabs',
	    'floor',
	    'fmod',
	    'fprintf',
	    'fputs',
	    'free',
	    'frexp',
	    'auto_ptr',
	    'deque',
	    'list',
	    'queue',
	    'stack',
	    'vector',
	    'map',
	    'set',
	    'pair',
	    'bitset',
	    'multiset',
	    'multimap',
	    'unordered_set',
	    'fscanf',
	    'future',
	    'isalnum',
	    'isalpha',
	    'iscntrl',
	    'isdigit',
	    'isgraph',
	    'islower',
	    'isprint',
	    'ispunct',
	    'isspace',
	    'isupper',
	    'isxdigit',
	    'tolower',
	    'toupper',
	    'labs',
	    'ldexp',
	    'log10',
	    'log',
	    'malloc',
	    'realloc',
	    'memchr',
	    'memcmp',
	    'memcpy',
	    'memset',
	    'modf',
	    'pow',
	    'printf',
	    'putchar',
	    'puts',
	    'scanf',
	    'sinh',
	    'sin',
	    'snprintf',
	    'sprintf',
	    'sqrt',
	    'sscanf',
	    'strcat',
	    'strchr',
	    'strcmp',
	    'strcpy',
	    'strcspn',
	    'strlen',
	    'strncat',
	    'strncmp',
	    'strncpy',
	    'strpbrk',
	    'strrchr',
	    'strspn',
	    'strstr',
	    'tanh',
	    'tan',
	    'unordered_map',
	    'unordered_multiset',
	    'unordered_multimap',
	    'priority_queue',
	    'make_pair',
	    'array',
	    'shared_ptr',
	    'abort',
	    'terminate',
	    'abs',
	    'acos',
	    'vfprintf',
	    'vprintf',
	    'vsprintf',
	    'endl',
	    'initializer_list',
	    'unique_ptr',
	    'complex',
	    'imaginary',
	    'std',
	    'string',
	    'wstring',
	    'cin',
	    'cout',
	    'cerr',
	    'clog',
	    'stdin',
	    'stdout',
	    'stderr',
	    'stringstream',
	    'istringstream',
	    'ostringstream'
	  ];

	  const CPP_KEYWORDS = {
	    keyword: 'int float while private char char8_t char16_t char32_t catch import module export virtual operator sizeof ' +
	      'dynamic_cast|10 typedef const_cast|10 const for static_cast|10 union namespace ' +
	      'unsigned long volatile static protected bool template mutable if public friend ' +
	      'do goto auto void enum else break extern using asm case typeid wchar_t ' +
	      'short reinterpret_cast|10 default double register explicit signed typename try this ' +
	      'switch continue inline delete alignas alignof constexpr consteval constinit decltype ' +
	      'concept co_await co_return co_yield requires ' +
	      'noexcept static_assert thread_local restrict final override ' +
	      'atomic_bool atomic_char atomic_schar ' +
	      'atomic_uchar atomic_short atomic_ushort atomic_int atomic_uint atomic_long atomic_ulong atomic_llong ' +
	      'atomic_ullong new throw return ' +
	      'and and_eq bitand bitor compl not not_eq or or_eq xor xor_eq',
	    built_in: '_Bool _Complex _Imaginary',
	    _relevance_hints: COMMON_CPP_HINTS,
	    literal: 'true false nullptr NULL'
	  };

	  const FUNCTION_DISPATCH = {
	    className: "function.dispatch",
	    relevance: 0,
	    keywords: CPP_KEYWORDS,
	    begin: concat(
	      /\b/,
	      /(?!decltype)/,
	      /(?!if)/,
	      /(?!for)/,
	      /(?!while)/,
	      hljs.IDENT_RE,
	      lookahead(/\s*\(/))
	  };

	  const EXPRESSION_CONTAINS = [
	    FUNCTION_DISPATCH,
	    PREPROCESSOR,
	    CPP_PRIMITIVE_TYPES,
	    C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    NUMBERS,
	    STRINGS
	  ];


	  const EXPRESSION_CONTEXT = {
	    // This mode covers expression context where we can't expect a function
	    // definition and shouldn't highlight anything that looks like one:
	    // `return some()`, `else if()`, `(x*sum(1, 2))`
	    variants: [
	      {
	        begin: /=/,
	        end: /;/
	      },
	      {
	        begin: /\(/,
	        end: /\)/
	      },
	      {
	        beginKeywords: 'new throw return else',
	        end: /;/
	      }
	    ],
	    keywords: CPP_KEYWORDS,
	    contains: EXPRESSION_CONTAINS.concat([
	      {
	        begin: /\(/,
	        end: /\)/,
	        keywords: CPP_KEYWORDS,
	        contains: EXPRESSION_CONTAINS.concat([ 'self' ]),
	        relevance: 0
	      }
	    ]),
	    relevance: 0
	  };

	  const FUNCTION_DECLARATION = {
	    className: 'function',
	    begin: '(' + FUNCTION_TYPE_RE + '[\\*&\\s]+)+' + FUNCTION_TITLE,
	    returnBegin: true,
	    end: /[{;=]/,
	    excludeEnd: true,
	    keywords: CPP_KEYWORDS,
	    illegal: /[^\w\s\*&:<>.]/,
	    contains: [
	      { // to prevent it from being confused as the function title
	        begin: DECLTYPE_AUTO_RE,
	        keywords: CPP_KEYWORDS,
	        relevance: 0
	      },
	      {
	        begin: FUNCTION_TITLE,
	        returnBegin: true,
	        contains: [ TITLE_MODE ],
	        relevance: 0
	      },
	      // needed because we do not have look-behind on the below rule
	      // to prevent it from grabbing the final : in a :: pair
	      {
	        begin: /::/,
	        relevance: 0
	      },
	      // initializers
	      {
	        begin: /:/,
	        endsWithParent: true,
	        contains: [
	          STRINGS,
	          NUMBERS
	        ]
	      },
	      {
	        className: 'params',
	        begin: /\(/,
	        end: /\)/,
	        keywords: CPP_KEYWORDS,
	        relevance: 0,
	        contains: [
	          C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE,
	          STRINGS,
	          NUMBERS,
	          CPP_PRIMITIVE_TYPES,
	          // Count matching parentheses.
	          {
	            begin: /\(/,
	            end: /\)/,
	            keywords: CPP_KEYWORDS,
	            relevance: 0,
	            contains: [
	              'self',
	              C_LINE_COMMENT_MODE,
	              hljs.C_BLOCK_COMMENT_MODE,
	              STRINGS,
	              NUMBERS,
	              CPP_PRIMITIVE_TYPES
	            ]
	          }
	        ]
	      },
	      CPP_PRIMITIVE_TYPES,
	      C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      PREPROCESSOR
	    ]
	  };

	  return {
	    name: 'C++',
	    aliases: [
	      'cc',
	      'c++',
	      'h++',
	      'hpp',
	      'hh',
	      'hxx',
	      'cxx'
	    ],
	    keywords: CPP_KEYWORDS,
	    illegal: '</',
	    classNameAliases: {
	      "function.dispatch": "built_in"
	    },
	    contains: [].concat(
	      EXPRESSION_CONTEXT,
	      FUNCTION_DECLARATION,
	      FUNCTION_DISPATCH,
	      EXPRESSION_CONTAINS,
	      [
	        PREPROCESSOR,
	        { // containers: ie, `vector <int> rooms (9);`
	          begin: '\\b(deque|list|queue|priority_queue|pair|stack|vector|map|set|bitset|multiset|multimap|unordered_map|unordered_set|unordered_multiset|unordered_multimap|array)\\s*<',
	          end: '>',
	          keywords: CPP_KEYWORDS,
	          contains: [
	            'self',
	            CPP_PRIMITIVE_TYPES
	          ]
	        },
	        {
	          begin: hljs.IDENT_RE + '::',
	          keywords: CPP_KEYWORDS
	        },
	        {
	          className: 'class',
	          beginKeywords: 'enum class struct union',
	          end: /[{;:<>=]/,
	          contains: [
	            {
	              beginKeywords: "final class struct"
	            },
	            hljs.TITLE_MODE
	          ]
	        }
	      ]),
	    exports: {
	      preprocessor: PREPROCESSOR,
	      strings: STRINGS,
	      keywords: CPP_KEYWORDS
	    }
	  };
	}

	/*
	Language: Arduino
	Author: Stefania Mellai <s.mellai@arduino.cc>
	Description: The Arduino® Language is a superset of C++. This rules are designed to highlight the Arduino® source code. For info about language see http://www.arduino.cc.
	Website: https://www.arduino.cc
	*/

	/** @type LanguageFn */
	function arduino(hljs) {
	  const ARDUINO_KW = {
	    keyword:
	      'boolean byte word String',
	    built_in:
	      'KeyboardController MouseController SoftwareSerial ' +
	      'EthernetServer EthernetClient LiquidCrystal ' +
	      'RobotControl GSMVoiceCall EthernetUDP EsploraTFT ' +
	      'HttpClient RobotMotor WiFiClient GSMScanner ' +
	      'FileSystem Scheduler GSMServer YunClient YunServer ' +
	      'IPAddress GSMClient GSMModem Keyboard Ethernet ' +
	      'Console GSMBand Esplora Stepper Process ' +
	      'WiFiUDP GSM_SMS Mailbox USBHost Firmata PImage ' +
	      'Client Server GSMPIN FileIO Bridge Serial ' +
	      'EEPROM Stream Mouse Audio Servo File Task ' +
	      'GPRS WiFi Wire TFT GSM SPI SD ',
	    _:
	      'setup loop ' +
	      'runShellCommandAsynchronously analogWriteResolution ' +
	      'retrieveCallingNumber printFirmwareVersion ' +
	      'analogReadResolution sendDigitalPortPair ' +
	      'noListenOnLocalhost readJoystickButton setFirmwareVersion ' +
	      'readJoystickSwitch scrollDisplayRight getVoiceCallStatus ' +
	      'scrollDisplayLeft writeMicroseconds delayMicroseconds ' +
	      'beginTransmission getSignalStrength runAsynchronously ' +
	      'getAsynchronously listenOnLocalhost getCurrentCarrier ' +
	      'readAccelerometer messageAvailable sendDigitalPorts ' +
	      'lineFollowConfig countryNameWrite runShellCommand ' +
	      'readStringUntil rewindDirectory readTemperature ' +
	      'setClockDivider readLightSensor endTransmission ' +
	      'analogReference detachInterrupt countryNameRead ' +
	      'attachInterrupt encryptionType readBytesUntil ' +
	      'robotNameWrite readMicrophone robotNameRead cityNameWrite ' +
	      'userNameWrite readJoystickY readJoystickX mouseReleased ' +
	      'openNextFile scanNetworks noInterrupts digitalWrite ' +
	      'beginSpeaker mousePressed isActionDone mouseDragged ' +
	      'displayLogos noAutoscroll addParameter remoteNumber ' +
	      'getModifiers keyboardRead userNameRead waitContinue ' +
	      'processInput parseCommand printVersion readNetworks ' +
	      'writeMessage blinkVersion cityNameRead readMessage ' +
	      'setDataMode parsePacket isListening setBitOrder ' +
	      'beginPacket isDirectory motorsWrite drawCompass ' +
	      'digitalRead clearScreen serialEvent rightToLeft ' +
	      'setTextSize leftToRight requestFrom keyReleased ' +
	      'compassRead analogWrite interrupts WiFiServer ' +
	      'disconnect playMelody parseFloat autoscroll ' +
	      'getPINUsed setPINUsed setTimeout sendAnalog ' +
	      'readSlider analogRead beginWrite createChar ' +
	      'motorsStop keyPressed tempoWrite readButton ' +
	      'subnetMask debugPrint macAddress writeGreen ' +
	      'randomSeed attachGPRS readString sendString ' +
	      'remotePort releaseAll mouseMoved background ' +
	      'getXChange getYChange answerCall getResult ' +
	      'voiceCall endPacket constrain getSocket writeJSON ' +
	      'getButton available connected findUntil readBytes ' +
	      'exitValue readGreen writeBlue startLoop IPAddress ' +
	      'isPressed sendSysex pauseMode gatewayIP setCursor ' +
	      'getOemKey tuneWrite noDisplay loadImage switchPIN ' +
	      'onRequest onReceive changePIN playFile noBuffer ' +
	      'parseInt overflow checkPIN knobRead beginTFT ' +
	      'bitClear updateIR bitWrite position writeRGB ' +
	      'highByte writeRed setSpeed readBlue noStroke ' +
	      'remoteIP transfer shutdown hangCall beginSMS ' +
	      'endWrite attached maintain noCursor checkReg ' +
	      'checkPUK shiftOut isValid shiftIn pulseIn ' +
	      'connect println localIP pinMode getIMEI ' +
	      'display noBlink process getBand running beginSD ' +
	      'drawBMP lowByte setBand release bitRead prepare ' +
	      'pointTo readRed setMode noFill remove listen ' +
	      'stroke detach attach noTone exists buffer ' +
	      'height bitSet circle config cursor random ' +
	      'IRread setDNS endSMS getKey micros ' +
	      'millis begin print write ready flush width ' +
	      'isPIN blink clear press mkdir rmdir close ' +
	      'point yield image BSSID click delay ' +
	      'read text move peek beep rect line open ' +
	      'seek fill size turn stop home find ' +
	      'step tone sqrt RSSI SSID ' +
	      'end bit tan cos sin pow map abs max ' +
	      'min get run put',
	    literal:
	      'DIGITAL_MESSAGE FIRMATA_STRING ANALOG_MESSAGE ' +
	      'REPORT_DIGITAL REPORT_ANALOG INPUT_PULLUP ' +
	      'SET_PIN_MODE INTERNAL2V56 SYSTEM_RESET LED_BUILTIN ' +
	      'INTERNAL1V1 SYSEX_START INTERNAL EXTERNAL ' +
	      'DEFAULT OUTPUT INPUT HIGH LOW'
	  };

	  const ARDUINO = cPlusPlus(hljs);

	  const kws = /** @type {Record<string,any>} */ (ARDUINO.keywords);

	  kws.keyword += ' ' + ARDUINO_KW.keyword;
	  kws.literal += ' ' + ARDUINO_KW.literal;
	  kws.built_in += ' ' + ARDUINO_KW.built_in;
	  kws._ += ' ' + ARDUINO_KW._;

	  ARDUINO.name = 'Arduino';
	  ARDUINO.aliases = ['ino'];
	  ARDUINO.supersetOf = "cpp";

	  return ARDUINO;
	}

	arduino_1 = arduino;
	return arduino_1;
}

/*
Language: ARM Assembly
Author: Dan Panzarella <alsoelp@gmail.com>
Description: ARM Assembly including Thumb and Thumb2 instructions
Category: assembler
*/

var armasm_1;
var hasRequiredArmasm;

function requireArmasm () {
	if (hasRequiredArmasm) return armasm_1;
	hasRequiredArmasm = 1;
	/** @type LanguageFn */
	function armasm(hljs) {
	  // local labels: %?[FB]?[AT]?\d{1,2}\w+

	  const COMMENT = {
	    variants: [
	      hljs.COMMENT('^[ \\t]*(?=#)', '$', {
	        relevance: 0,
	        excludeBegin: true
	      }),
	      hljs.COMMENT('[;@]', '$', {
	        relevance: 0
	      }),
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE
	    ]
	  };

	  return {
	    name: 'ARM Assembly',
	    case_insensitive: true,
	    aliases: ['arm'],
	    keywords: {
	      $pattern: '\\.?' + hljs.IDENT_RE,
	      meta:
	        // GNU preprocs
	        '.2byte .4byte .align .ascii .asciz .balign .byte .code .data .else .end .endif .endm .endr .equ .err .exitm .extern .global .hword .if .ifdef .ifndef .include .irp .long .macro .rept .req .section .set .skip .space .text .word .arm .thumb .code16 .code32 .force_thumb .thumb_func .ltorg ' +
	        // ARM directives
	        'ALIAS ALIGN ARM AREA ASSERT ATTR CN CODE CODE16 CODE32 COMMON CP DATA DCB DCD DCDU DCDO DCFD DCFDU DCI DCQ DCQU DCW DCWU DN ELIF ELSE END ENDFUNC ENDIF ENDP ENTRY EQU EXPORT EXPORTAS EXTERN FIELD FILL FUNCTION GBLA GBLL GBLS GET GLOBAL IF IMPORT INCBIN INCLUDE INFO KEEP LCLA LCLL LCLS LTORG MACRO MAP MEND MEXIT NOFP OPT PRESERVE8 PROC QN READONLY RELOC REQUIRE REQUIRE8 RLIST FN ROUT SETA SETL SETS SN SPACE SUBT THUMB THUMBX TTL WHILE WEND ',
	      built_in:
	        'r0 r1 r2 r3 r4 r5 r6 r7 r8 r9 r10 r11 r12 r13 r14 r15 ' + // standard registers
	        'pc lr sp ip sl sb fp ' + // typical regs plus backward compatibility
	        'a1 a2 a3 a4 v1 v2 v3 v4 v5 v6 v7 v8 f0 f1 f2 f3 f4 f5 f6 f7 ' + // more regs and fp
	        'p0 p1 p2 p3 p4 p5 p6 p7 p8 p9 p10 p11 p12 p13 p14 p15 ' + // coprocessor regs
	        'c0 c1 c2 c3 c4 c5 c6 c7 c8 c9 c10 c11 c12 c13 c14 c15 ' + // more coproc
	        'q0 q1 q2 q3 q4 q5 q6 q7 q8 q9 q10 q11 q12 q13 q14 q15 ' + // advanced SIMD NEON regs

	        // program status registers
	        'cpsr_c cpsr_x cpsr_s cpsr_f cpsr_cx cpsr_cxs cpsr_xs cpsr_xsf cpsr_sf cpsr_cxsf ' +
	        'spsr_c spsr_x spsr_s spsr_f spsr_cx spsr_cxs spsr_xs spsr_xsf spsr_sf spsr_cxsf ' +

	        // NEON and VFP registers
	        's0 s1 s2 s3 s4 s5 s6 s7 s8 s9 s10 s11 s12 s13 s14 s15 ' +
	        's16 s17 s18 s19 s20 s21 s22 s23 s24 s25 s26 s27 s28 s29 s30 s31 ' +
	        'd0 d1 d2 d3 d4 d5 d6 d7 d8 d9 d10 d11 d12 d13 d14 d15 ' +
	        'd16 d17 d18 d19 d20 d21 d22 d23 d24 d25 d26 d27 d28 d29 d30 d31 ' +

	        '{PC} {VAR} {TRUE} {FALSE} {OPT} {CONFIG} {ENDIAN} {CODESIZE} {CPU} {FPU} {ARCHITECTURE} {PCSTOREOFFSET} {ARMASM_VERSION} {INTER} {ROPI} {RWPI} {SWST} {NOSWST} . @'
	    },
	    contains: [
	      {
	        className: 'keyword',
	        begin: '\\b(' + // mnemonics
	            'adc|' +
	            '(qd?|sh?|u[qh]?)?add(8|16)?|usada?8|(q|sh?|u[qh]?)?(as|sa)x|' +
	            'and|adrl?|sbc|rs[bc]|asr|b[lx]?|blx|bxj|cbn?z|tb[bh]|bic|' +
	            'bfc|bfi|[su]bfx|bkpt|cdp2?|clz|clrex|cmp|cmn|cpsi[ed]|cps|' +
	            'setend|dbg|dmb|dsb|eor|isb|it[te]{0,3}|lsl|lsr|ror|rrx|' +
	            'ldm(([id][ab])|f[ds])?|ldr((s|ex)?[bhd])?|movt?|mvn|mra|mar|' +
	            'mul|[us]mull|smul[bwt][bt]|smu[as]d|smmul|smmla|' +
	            'mla|umlaal|smlal?([wbt][bt]|d)|mls|smlsl?[ds]|smc|svc|sev|' +
	            'mia([bt]{2}|ph)?|mrr?c2?|mcrr2?|mrs|msr|orr|orn|pkh(tb|bt)|rbit|' +
	            'rev(16|sh)?|sel|[su]sat(16)?|nop|pop|push|rfe([id][ab])?|' +
	            'stm([id][ab])?|str(ex)?[bhd]?|(qd?)?sub|(sh?|q|u[qh]?)?sub(8|16)|' +
	            '[su]xt(a?h|a?b(16)?)|srs([id][ab])?|swpb?|swi|smi|tst|teq|' +
	            'wfe|wfi|yield' +
	        ')' +
	        '(eq|ne|cs|cc|mi|pl|vs|vc|hi|ls|ge|lt|gt|le|al|hs|lo)?' + // condition codes
	        '[sptrx]?' + // legal postfixes
	        '(?=\\s)' // followed by space
	      },
	      COMMENT,
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'string',
	        begin: '\'',
	        end: '[^\\\\]\'',
	        relevance: 0
	      },
	      {
	        className: 'title',
	        begin: '\\|',
	        end: '\\|',
	        illegal: '\\n',
	        relevance: 0
	      },
	      {
	        className: 'number',
	        variants: [
	          { // hex
	            begin: '[#$=]?0x[0-9a-f]+'
	          },
	          { // bin
	            begin: '[#$=]?0b[01]+'
	          },
	          { // literal
	            begin: '[#$=]\\d+'
	          },
	          { // bare number
	            begin: '\\b\\d+'
	          }
	        ],
	        relevance: 0
	      },
	      {
	        className: 'symbol',
	        variants: [
	          { // GNU ARM syntax
	            begin: '^[ \\t]*[a-z_\\.\\$][a-z0-9_\\.\\$]+:'
	          },
	          { // ARM syntax
	            begin: '^[a-z_\\.\\$][a-z0-9_\\.\\$]+'
	          },
	          { // label reference
	            begin: '[=#]\\w+'
	          }
	        ],
	        relevance: 0
	      }
	    ]
	  };
	}

	armasm_1 = armasm;
	return armasm_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var xml_1;
var hasRequiredXml;

function requireXml () {
	if (hasRequiredXml) return xml_1;
	hasRequiredXml = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function optional(re) {
	  return concat('(', re, ')?');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/*
	Language: HTML, XML
	Website: https://www.w3.org/XML/
	Category: common
	Audit: 2020
	*/

	/** @type LanguageFn */
	function xml(hljs) {
	  // Element names can contain letters, digits, hyphens, underscores, and periods
	  const TAG_NAME_RE = concat(/[A-Z_]/, optional(/[A-Z0-9_.-]*:/), /[A-Z0-9_.-]*/);
	  const XML_IDENT_RE = /[A-Za-z0-9._:-]+/;
	  const XML_ENTITIES = {
	    className: 'symbol',
	    begin: /&[a-z]+;|&#[0-9]+;|&#x[a-f0-9]+;/
	  };
	  const XML_META_KEYWORDS = {
	    begin: /\s/,
	    contains: [
	      {
	        className: 'meta-keyword',
	        begin: /#?[a-z_][a-z1-9_-]+/,
	        illegal: /\n/
	      }
	    ]
	  };
	  const XML_META_PAR_KEYWORDS = hljs.inherit(XML_META_KEYWORDS, {
	    begin: /\(/,
	    end: /\)/
	  });
	  const APOS_META_STRING_MODE = hljs.inherit(hljs.APOS_STRING_MODE, {
	    className: 'meta-string'
	  });
	  const QUOTE_META_STRING_MODE = hljs.inherit(hljs.QUOTE_STRING_MODE, {
	    className: 'meta-string'
	  });
	  const TAG_INTERNALS = {
	    endsWithParent: true,
	    illegal: /</,
	    relevance: 0,
	    contains: [
	      {
	        className: 'attr',
	        begin: XML_IDENT_RE,
	        relevance: 0
	      },
	      {
	        begin: /=\s*/,
	        relevance: 0,
	        contains: [
	          {
	            className: 'string',
	            endsParent: true,
	            variants: [
	              {
	                begin: /"/,
	                end: /"/,
	                contains: [ XML_ENTITIES ]
	              },
	              {
	                begin: /'/,
	                end: /'/,
	                contains: [ XML_ENTITIES ]
	              },
	              {
	                begin: /[^\s"'=<>`]+/
	              }
	            ]
	          }
	        ]
	      }
	    ]
	  };
	  return {
	    name: 'HTML, XML',
	    aliases: [
	      'html',
	      'xhtml',
	      'rss',
	      'atom',
	      'xjb',
	      'xsd',
	      'xsl',
	      'plist',
	      'wsf',
	      'svg'
	    ],
	    case_insensitive: true,
	    contains: [
	      {
	        className: 'meta',
	        begin: /<![a-z]/,
	        end: />/,
	        relevance: 10,
	        contains: [
	          XML_META_KEYWORDS,
	          QUOTE_META_STRING_MODE,
	          APOS_META_STRING_MODE,
	          XML_META_PAR_KEYWORDS,
	          {
	            begin: /\[/,
	            end: /\]/,
	            contains: [
	              {
	                className: 'meta',
	                begin: /<![a-z]/,
	                end: />/,
	                contains: [
	                  XML_META_KEYWORDS,
	                  XML_META_PAR_KEYWORDS,
	                  QUOTE_META_STRING_MODE,
	                  APOS_META_STRING_MODE
	                ]
	              }
	            ]
	          }
	        ]
	      },
	      hljs.COMMENT(
	        /<!--/,
	        /-->/,
	        {
	          relevance: 10
	        }
	      ),
	      {
	        begin: /<!\[CDATA\[/,
	        end: /\]\]>/,
	        relevance: 10
	      },
	      XML_ENTITIES,
	      {
	        className: 'meta',
	        begin: /<\?xml/,
	        end: /\?>/,
	        relevance: 10
	      },
	      {
	        className: 'tag',
	        /*
	        The lookahead pattern (?=...) ensures that 'begin' only matches
	        '<style' as a single word, followed by a whitespace or an
	        ending braket. The '$' is needed for the lexeme to be recognized
	        by hljs.subMode() that tests lexemes outside the stream.
	        */
	        begin: /<style(?=\s|>)/,
	        end: />/,
	        keywords: {
	          name: 'style'
	        },
	        contains: [ TAG_INTERNALS ],
	        starts: {
	          end: /<\/style>/,
	          returnEnd: true,
	          subLanguage: [
	            'css',
	            'xml'
	          ]
	        }
	      },
	      {
	        className: 'tag',
	        // See the comment in the <style tag about the lookahead pattern
	        begin: /<script(?=\s|>)/,
	        end: />/,
	        keywords: {
	          name: 'script'
	        },
	        contains: [ TAG_INTERNALS ],
	        starts: {
	          end: /<\/script>/,
	          returnEnd: true,
	          subLanguage: [
	            'javascript',
	            'handlebars',
	            'xml'
	          ]
	        }
	      },
	      // we need this for now for jSX
	      {
	        className: 'tag',
	        begin: /<>|<\/>/
	      },
	      // open tag
	      {
	        className: 'tag',
	        begin: concat(
	          /</,
	          lookahead(concat(
	            TAG_NAME_RE,
	            // <tag/>
	            // <tag>
	            // <tag ...
	            either(/\/>/, />/, /\s/)
	          ))
	        ),
	        end: /\/?>/,
	        contains: [
	          {
	            className: 'name',
	            begin: TAG_NAME_RE,
	            relevance: 0,
	            starts: TAG_INTERNALS
	          }
	        ]
	      },
	      // close tag
	      {
	        className: 'tag',
	        begin: concat(
	          /<\//,
	          lookahead(concat(
	            TAG_NAME_RE, />/
	          ))
	        ),
	        contains: [
	          {
	            className: 'name',
	            begin: TAG_NAME_RE,
	            relevance: 0
	          },
	          {
	            begin: />/,
	            relevance: 0,
	            endsParent: true
	          }
	        ]
	      }
	    ]
	  };
	}

	xml_1 = xml;
	return xml_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var asciidoc_1;
var hasRequiredAsciidoc;

function requireAsciidoc () {
	if (hasRequiredAsciidoc) return asciidoc_1;
	hasRequiredAsciidoc = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: AsciiDoc
	Requires: xml.js
	Author: Dan Allen <dan.j.allen@gmail.com>
	Website: http://asciidoc.org
	Description: A semantic, text-based document format that can be exported to HTML, DocBook and other backends.
	Category: markup
	*/

	/** @type LanguageFn */
	function asciidoc(hljs) {
	  const HORIZONTAL_RULE = {
	    begin: '^\'{3,}[ \\t]*$',
	    relevance: 10
	  };
	  const ESCAPED_FORMATTING = [
	    // escaped constrained formatting marks (i.e., \* \_ or \`)
	    {
	      begin: /\\[*_`]/
	    },
	    // escaped unconstrained formatting marks (i.e., \\** \\__ or \\``)
	    // must ignore until the next formatting marks
	    // this rule might not be 100% compliant with Asciidoctor 2.0 but we are entering undefined behavior territory...
	    {
	      begin: /\\\\\*{2}[^\n]*?\*{2}/
	    },
	    {
	      begin: /\\\\_{2}[^\n]*_{2}/
	    },
	    {
	      begin: /\\\\`{2}[^\n]*`{2}/
	    },
	    // guard: constrained formatting mark may not be preceded by ":", ";" or
	    // "}". match these so the constrained rule doesn't see them
	    {
	      begin: /[:;}][*_`](?![*_`])/
	    }
	  ];
	  const STRONG = [
	    // inline unconstrained strong (single line)
	    {
	      className: 'strong',
	      begin: /\*{2}([^\n]+?)\*{2}/
	    },
	    // inline unconstrained strong (multi-line)
	    {
	      className: 'strong',
	      begin: concat(
	        /\*\*/,
	        /((\*(?!\*)|\\[^\n]|[^*\n\\])+\n)+/,
	        /(\*(?!\*)|\\[^\n]|[^*\n\\])*/,
	        /\*\*/
	      ),
	      relevance: 0
	    },
	    // inline constrained strong (single line)
	    {
	      className: 'strong',
	      // must not precede or follow a word character
	      begin: /\B\*(\S|\S[^\n]*?\S)\*(?!\w)/
	    },
	    // inline constrained strong (multi-line)
	    {
	      className: 'strong',
	      // must not precede or follow a word character
	      begin: /\*[^\s]([^\n]+\n)+([^\n]+)\*/
	    }
	  ];
	  const EMPHASIS = [
	    // inline unconstrained emphasis (single line)
	    {
	      className: 'emphasis',
	      begin: /_{2}([^\n]+?)_{2}/
	    },
	    // inline unconstrained emphasis (multi-line)
	    {
	      className: 'emphasis',
	      begin: concat(
	        /__/,
	        /((_(?!_)|\\[^\n]|[^_\n\\])+\n)+/,
	        /(_(?!_)|\\[^\n]|[^_\n\\])*/,
	        /__/
	      ),
	      relevance: 0
	    },
	    // inline constrained emphasis (single line)
	    {
	      className: 'emphasis',
	      // must not precede or follow a word character
	      begin: /\b_(\S|\S[^\n]*?\S)_(?!\w)/
	    },
	    // inline constrained emphasis (multi-line)
	    {
	      className: 'emphasis',
	      // must not precede or follow a word character
	      begin: /_[^\s]([^\n]+\n)+([^\n]+)_/
	    },
	    // inline constrained emphasis using single quote (legacy)
	    {
	      className: 'emphasis',
	      // must not follow a word character or be followed by a single quote or space
	      begin: '\\B\'(?![\'\\s])',
	      end: '(\\n{2}|\')',
	      // allow escaped single quote followed by word char
	      contains: [{
	        begin: '\\\\\'\\w',
	        relevance: 0
	      }],
	      relevance: 0
	    }
	  ];
	  const ADMONITION = {
	    className: 'symbol',
	    begin: '^(NOTE|TIP|IMPORTANT|WARNING|CAUTION):\\s+',
	    relevance: 10
	  };
	  const BULLET_LIST = {
	    className: 'bullet',
	    begin: '^(\\*+|-+|\\.+|[^\\n]+?::)\\s+'
	  };

	  return {
	    name: 'AsciiDoc',
	    aliases: ['adoc'],
	    contains: [
	      // block comment
	      hljs.COMMENT(
	        '^/{4,}\\n',
	        '\\n/{4,}$',
	        // can also be done as...
	        // '^/{4,}$',
	        // '^/{4,}$',
	        {
	          relevance: 10
	        }
	      ),
	      // line comment
	      hljs.COMMENT(
	        '^//',
	        '$',
	        {
	          relevance: 0
	        }
	      ),
	      // title
	      {
	        className: 'title',
	        begin: '^\\.\\w.*$'
	      },
	      // example, admonition & sidebar blocks
	      {
	        begin: '^[=\\*]{4,}\\n',
	        end: '\\n^[=\\*]{4,}$',
	        relevance: 10
	      },
	      // headings
	      {
	        className: 'section',
	        relevance: 10,
	        variants: [
	          {
	            begin: '^(={1,6})[ \t].+?([ \t]\\1)?$'
	          },
	          {
	            begin: '^[^\\[\\]\\n]+?\\n[=\\-~\\^\\+]{2,}$'
	          }
	        ]
	      },
	      // document attributes
	      {
	        className: 'meta',
	        begin: '^:.+?:',
	        end: '\\s',
	        excludeEnd: true,
	        relevance: 10
	      },
	      // block attributes
	      {
	        className: 'meta',
	        begin: '^\\[.+?\\]$',
	        relevance: 0
	      },
	      // quoteblocks
	      {
	        className: 'quote',
	        begin: '^_{4,}\\n',
	        end: '\\n_{4,}$',
	        relevance: 10
	      },
	      // listing and literal blocks
	      {
	        className: 'code',
	        begin: '^[\\-\\.]{4,}\\n',
	        end: '\\n[\\-\\.]{4,}$',
	        relevance: 10
	      },
	      // passthrough blocks
	      {
	        begin: '^\\+{4,}\\n',
	        end: '\\n\\+{4,}$',
	        contains: [{
	          begin: '<',
	          end: '>',
	          subLanguage: 'xml',
	          relevance: 0
	        }],
	        relevance: 10
	      },

	      BULLET_LIST,
	      ADMONITION,
	      ...ESCAPED_FORMATTING,
	      ...STRONG,
	      ...EMPHASIS,

	      // inline smart quotes
	      {
	        className: 'string',
	        variants: [
	          {
	            begin: "``.+?''"
	          },
	          {
	            begin: "`.+?'"
	          }
	        ]
	      },
	      // inline unconstrained emphasis
	      {
	        className: 'code',
	        begin: /`{2}/,
	        end: /(\n{2}|`{2})/
	      },
	      // inline code snippets (TODO should get same treatment as strong and emphasis)
	      {
	        className: 'code',
	        begin: '(`.+?`|\\+.+?\\+)',
	        relevance: 0
	      },
	      // indented literal block
	      {
	        className: 'code',
	        begin: '^[ \\t]',
	        end: '$',
	        relevance: 0
	      },
	      HORIZONTAL_RULE,
	      // images and links
	      {
	        begin: '(link:)?(http|https|ftp|file|irc|image:?):\\S+?\\[[^[]*?\\]',
	        returnBegin: true,
	        contains: [
	          {
	            begin: '(link|image:?):',
	            relevance: 0
	          },
	          {
	            className: 'link',
	            begin: '\\w',
	            end: '[^\\[]+',
	            relevance: 0
	          },
	          {
	            className: 'string',
	            begin: '\\[',
	            end: '\\]',
	            excludeBegin: true,
	            excludeEnd: true,
	            relevance: 0
	          }
	        ],
	        relevance: 10
	      }
	    ]
	  };
	}

	asciidoc_1 = asciidoc;
	return asciidoc_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var aspectj_1;
var hasRequiredAspectj;

function requireAspectj () {
	if (hasRequiredAspectj) return aspectj_1;
	hasRequiredAspectj = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: AspectJ
	Author: Hakan Ozler <ozler.hakan@gmail.com>
	Website: https://www.eclipse.org/aspectj/
	Description: Syntax Highlighting for the AspectJ Language which is a general-purpose aspect-oriented extension to the Java programming language.
	Audit: 2020
	*/

	/** @type LanguageFn */
	function aspectj(hljs) {
	  const KEYWORDS =
	    'false synchronized int abstract float private char boolean static null if const ' +
	    'for true while long throw strictfp finally protected import native final return void ' +
	    'enum else extends implements break transient new catch instanceof byte super volatile case ' +
	    'assert short package default double public try this switch continue throws privileged ' +
	    'aspectOf adviceexecution proceed cflowbelow cflow initialization preinitialization ' +
	    'staticinitialization withincode target within execution getWithinTypeName handler ' +
	    'thisJoinPoint thisJoinPointStaticPart thisEnclosingJoinPointStaticPart declare parents ' +
	    'warning error soft precedence thisAspectInstance';
	  const SHORTKEYS = 'get set args call';

	  return {
	    name: 'AspectJ',
	    keywords: KEYWORDS,
	    illegal: /<\/|#/,
	    contains: [
	      hljs.COMMENT(
	        /\/\*\*/,
	        /\*\//,
	        {
	          relevance: 0,
	          contains: [
	            {
	              // eat up @'s in emails to prevent them to be recognized as doctags
	              begin: /\w+@/,
	              relevance: 0
	            },
	            {
	              className: 'doctag',
	              begin: /@[A-Za-z]+/
	            }
	          ]
	        }
	      ),
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'class',
	        beginKeywords: 'aspect',
	        end: /[{;=]/,
	        excludeEnd: true,
	        illegal: /[:;"\[\]]/,
	        contains: [
	          {
	            beginKeywords: 'extends implements pertypewithin perthis pertarget percflowbelow percflow issingleton'
	          },
	          hljs.UNDERSCORE_TITLE_MODE,
	          {
	            begin: /\([^\)]*/,
	            end: /[)]+/,
	            keywords: KEYWORDS + ' ' + SHORTKEYS,
	            excludeEnd: false
	          }
	        ]
	      },
	      {
	        className: 'class',
	        beginKeywords: 'class interface',
	        end: /[{;=]/,
	        excludeEnd: true,
	        relevance: 0,
	        keywords: 'class interface',
	        illegal: /[:"\[\]]/,
	        contains: [
	          {
	            beginKeywords: 'extends implements'
	          },
	          hljs.UNDERSCORE_TITLE_MODE
	        ]
	      },
	      {
	        // AspectJ Constructs
	        beginKeywords: 'pointcut after before around throwing returning',
	        end: /[)]/,
	        excludeEnd: false,
	        illegal: /["\[\]]/,
	        contains: [
	          {
	            begin: concat(hljs.UNDERSCORE_IDENT_RE, /\s*\(/),
	            returnBegin: true,
	            contains: [ hljs.UNDERSCORE_TITLE_MODE ]
	          }
	        ]
	      },
	      {
	        begin: /[:]/,
	        returnBegin: true,
	        end: /[{;]/,
	        relevance: 0,
	        excludeEnd: false,
	        keywords: KEYWORDS,
	        illegal: /["\[\]]/,
	        contains: [
	          {
	            begin: concat(hljs.UNDERSCORE_IDENT_RE, /\s*\(/),
	            keywords: KEYWORDS + ' ' + SHORTKEYS,
	            relevance: 0
	          },
	          hljs.QUOTE_STRING_MODE
	        ]
	      },
	      {
	        // this prevents 'new Name(...), or throw ...' from being recognized as a function definition
	        beginKeywords: 'new throw',
	        relevance: 0
	      },
	      {
	        // the function class is a bit different for AspectJ compared to the Java language
	        className: 'function',
	        begin: /\w+ +\w+(\.\w+)?\s*\([^\)]*\)\s*((throws)[\w\s,]+)?[\{;]/,
	        returnBegin: true,
	        end: /[{;=]/,
	        keywords: KEYWORDS,
	        excludeEnd: true,
	        contains: [
	          {
	            begin: concat(hljs.UNDERSCORE_IDENT_RE, /\s*\(/),
	            returnBegin: true,
	            relevance: 0,
	            contains: [ hljs.UNDERSCORE_TITLE_MODE ]
	          },
	          {
	            className: 'params',
	            begin: /\(/,
	            end: /\)/,
	            relevance: 0,
	            keywords: KEYWORDS,
	            contains: [
	              hljs.APOS_STRING_MODE,
	              hljs.QUOTE_STRING_MODE,
	              hljs.C_NUMBER_MODE,
	              hljs.C_BLOCK_COMMENT_MODE
	            ]
	          },
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE
	        ]
	      },
	      hljs.C_NUMBER_MODE,
	      {
	        // annotation is also used in this language
	        className: 'meta',
	        begin: /@[A-Za-z]+/
	      }
	    ]
	  };
	}

	aspectj_1 = aspectj;
	return aspectj_1;
}

/*
Language: AutoHotkey
Author: Seongwon Lee <dlimpid@gmail.com>
Description: AutoHotkey language definition
Category: scripting
*/

var autohotkey_1;
var hasRequiredAutohotkey;

function requireAutohotkey () {
	if (hasRequiredAutohotkey) return autohotkey_1;
	hasRequiredAutohotkey = 1;
	/** @type LanguageFn */
	function autohotkey(hljs) {
	  const BACKTICK_ESCAPE = {
	    begin: '`[\\s\\S]'
	  };

	  return {
	    name: 'AutoHotkey',
	    case_insensitive: true,
	    aliases: ['ahk'],
	    keywords: {
	      keyword: 'Break Continue Critical Exit ExitApp Gosub Goto New OnExit Pause return SetBatchLines SetTimer Suspend Thread Throw Until ahk_id ahk_class ahk_pid ahk_exe ahk_group',
	      literal: 'true false NOT AND OR',
	      built_in: 'ComSpec Clipboard ClipboardAll ErrorLevel'
	    },
	    contains: [
	      BACKTICK_ESCAPE,
	      hljs.inherit(hljs.QUOTE_STRING_MODE, {
	        contains: [BACKTICK_ESCAPE]
	      }),
	      hljs.COMMENT(';', '$', {
	        relevance: 0
	      }),
	      hljs.C_BLOCK_COMMENT_MODE,
	      {
	        className: 'number',
	        begin: hljs.NUMBER_RE,
	        relevance: 0
	      },
	      {
	        // subst would be the most accurate however fails the point of
	        // highlighting. variable is comparably the most accurate that actually
	        // has some effect
	        className: 'variable',
	        begin: '%[a-zA-Z0-9#_$@]+%'
	      },
	      {
	        className: 'built_in',
	        begin: '^\\s*\\w+\\s*(,|%)'
	        // I don't really know if this is totally relevant
	      },
	      {
	        // symbol would be most accurate however is highlighted just like
	        // built_in and that makes up a lot of AutoHotkey code meaning that it
	        // would fail to highlight anything
	        className: 'title',
	        variants: [
	          {
	            begin: '^[^\\n";]+::(?!=)'
	          },
	          {
	            begin: '^[^\\n";]+:(?!=)',
	            // zero relevance as it catches a lot of things
	            // followed by a single ':' in many languages
	            relevance: 0
	          }
	        ]
	      },
	      {
	        className: 'meta',
	        begin: '^\\s*#\\w+',
	        end: '$',
	        relevance: 0
	      },
	      {
	        className: 'built_in',
	        begin: 'A_[a-zA-Z0-9]+'
	      },
	      {
	        // consecutive commas, not for highlighting but just for relevance
	        begin: ',\\s*,'
	      }
	    ]
	  };
	}

	autohotkey_1 = autohotkey;
	return autohotkey_1;
}

/*
Language: AutoIt
Author: Manh Tuan <junookyo@gmail.com>
Description: AutoIt language definition
Category: scripting
*/

var autoit_1;
var hasRequiredAutoit;

function requireAutoit () {
	if (hasRequiredAutoit) return autoit_1;
	hasRequiredAutoit = 1;
	/** @type LanguageFn */
	function autoit(hljs) {
	  const KEYWORDS = 'ByRef Case Const ContinueCase ContinueLoop ' +
	        'Dim Do Else ElseIf EndFunc EndIf EndSelect ' +
	        'EndSwitch EndWith Enum Exit ExitLoop For Func ' +
	        'Global If In Local Next ReDim Return Select Static ' +
	        'Step Switch Then To Until Volatile WEnd While With';

	  const DIRECTIVES = [
	    "EndRegion",
	    "forcedef",
	    "forceref",
	    "ignorefunc",
	    "include",
	    "include-once",
	    "NoTrayIcon",
	    "OnAutoItStartRegister",
	    "pragma",
	    "Region",
	    "RequireAdmin",
	    "Tidy_Off",
	    "Tidy_On",
	    "Tidy_Parameters"
	  ];
	  
	  const LITERAL = 'True False And Null Not Or Default';

	  const BUILT_IN
	          = 'Abs ACos AdlibRegister AdlibUnRegister Asc AscW ASin Assign ATan AutoItSetOption AutoItWinGetTitle AutoItWinSetTitle Beep Binary BinaryLen BinaryMid BinaryToString BitAND BitNOT BitOR BitRotate BitShift BitXOR BlockInput Break Call CDTray Ceiling Chr ChrW ClipGet ClipPut ConsoleRead ConsoleWrite ConsoleWriteError ControlClick ControlCommand ControlDisable ControlEnable ControlFocus ControlGetFocus ControlGetHandle ControlGetPos ControlGetText ControlHide ControlListView ControlMove ControlSend ControlSetText ControlShow ControlTreeView Cos Dec DirCopy DirCreate DirGetSize DirMove DirRemove DllCall DllCallAddress DllCallbackFree DllCallbackGetPtr DllCallbackRegister DllClose DllOpen DllStructCreate DllStructGetData DllStructGetPtr DllStructGetSize DllStructSetData DriveGetDrive DriveGetFileSystem DriveGetLabel DriveGetSerial DriveGetType DriveMapAdd DriveMapDel DriveMapGet DriveSetLabel DriveSpaceFree DriveSpaceTotal DriveStatus EnvGet EnvSet EnvUpdate Eval Execute Exp FileChangeDir FileClose FileCopy FileCreateNTFSLink FileCreateShortcut FileDelete FileExists FileFindFirstFile FileFindNextFile FileFlush FileGetAttrib FileGetEncoding FileGetLongName FileGetPos FileGetShortcut FileGetShortName FileGetSize FileGetTime FileGetVersion FileInstall FileMove FileOpen FileOpenDialog FileRead FileReadLine FileReadToArray FileRecycle FileRecycleEmpty FileSaveDialog FileSelectFolder FileSetAttrib FileSetEnd FileSetPos FileSetTime FileWrite FileWriteLine Floor FtpSetProxy FuncName GUICreate GUICtrlCreateAvi GUICtrlCreateButton GUICtrlCreateCheckbox GUICtrlCreateCombo GUICtrlCreateContextMenu GUICtrlCreateDate GUICtrlCreateDummy GUICtrlCreateEdit GUICtrlCreateGraphic GUICtrlCreateGroup GUICtrlCreateIcon GUICtrlCreateInput GUICtrlCreateLabel GUICtrlCreateList GUICtrlCreateListView GUICtrlCreateListViewItem GUICtrlCreateMenu GUICtrlCreateMenuItem GUICtrlCreateMonthCal GUICtrlCreateObj GUICtrlCreatePic GUICtrlCreateProgress GUICtrlCreateRadio GUICtrlCreateSlider GUICtrlCreateTab GUICtrlCreateTabItem GUICtrlCreateTreeView GUICtrlCreateTreeViewItem GUICtrlCreateUpdown GUICtrlDelete GUICtrlGetHandle GUICtrlGetState GUICtrlRead GUICtrlRecvMsg GUICtrlRegisterListViewSort GUICtrlSendMsg GUICtrlSendToDummy GUICtrlSetBkColor GUICtrlSetColor GUICtrlSetCursor GUICtrlSetData GUICtrlSetDefBkColor GUICtrlSetDefColor GUICtrlSetFont GUICtrlSetGraphic GUICtrlSetImage GUICtrlSetLimit GUICtrlSetOnEvent GUICtrlSetPos GUICtrlSetResizing GUICtrlSetState GUICtrlSetStyle GUICtrlSetTip GUIDelete GUIGetCursorInfo GUIGetMsg GUIGetStyle GUIRegisterMsg GUISetAccelerators GUISetBkColor GUISetCoord GUISetCursor GUISetFont GUISetHelp GUISetIcon GUISetOnEvent GUISetState GUISetStyle GUIStartGroup GUISwitch Hex HotKeySet HttpSetProxy HttpSetUserAgent HWnd InetClose InetGet InetGetInfo InetGetSize InetRead IniDelete IniRead IniReadSection IniReadSectionNames IniRenameSection IniWrite IniWriteSection InputBox Int IsAdmin IsArray IsBinary IsBool IsDeclared IsDllStruct IsFloat IsFunc IsHWnd IsInt IsKeyword IsNumber IsObj IsPtr IsString Log MemGetStats Mod MouseClick MouseClickDrag MouseDown MouseGetCursor MouseGetPos MouseMove MouseUp MouseWheel MsgBox Number ObjCreate ObjCreateInterface ObjEvent ObjGet ObjName OnAutoItExitRegister OnAutoItExitUnRegister Ping PixelChecksum PixelGetColor PixelSearch ProcessClose ProcessExists ProcessGetStats ProcessList ProcessSetPriority ProcessWait ProcessWaitClose ProgressOff ProgressOn ProgressSet Ptr Random RegDelete RegEnumKey RegEnumVal RegRead RegWrite Round Run RunAs RunAsWait RunWait Send SendKeepActive SetError SetExtended ShellExecute ShellExecuteWait Shutdown Sin Sleep SoundPlay SoundSetWaveVolume SplashImageOn SplashOff SplashTextOn Sqrt SRandom StatusbarGetText StderrRead StdinWrite StdioClose StdoutRead String StringAddCR StringCompare StringFormat StringFromASCIIArray StringInStr StringIsAlNum StringIsAlpha StringIsASCII StringIsDigit StringIsFloat StringIsInt StringIsLower StringIsSpace StringIsUpper StringIsXDigit StringLeft StringLen StringLower StringMid StringRegExp StringRegExpReplace StringReplace StringReverse StringRight StringSplit StringStripCR StringStripWS StringToASCIIArray StringToBinary StringTrimLeft StringTrimRight StringUpper Tan TCPAccept TCPCloseSocket TCPConnect TCPListen TCPNameToIP TCPRecv TCPSend TCPShutdown, UDPShutdown TCPStartup, UDPStartup TimerDiff TimerInit ToolTip TrayCreateItem TrayCreateMenu TrayGetMsg TrayItemDelete TrayItemGetHandle TrayItemGetState TrayItemGetText TrayItemSetOnEvent TrayItemSetState TrayItemSetText TraySetClick TraySetIcon TraySetOnEvent TraySetPauseIcon TraySetState TraySetToolTip TrayTip UBound UDPBind UDPCloseSocket UDPOpen UDPRecv UDPSend VarGetType WinActivate WinActive WinClose WinExists WinFlash WinGetCaretPos WinGetClassList WinGetClientSize WinGetHandle WinGetPos WinGetProcess WinGetState WinGetText WinGetTitle WinKill WinList WinMenuSelectItem WinMinimizeAll WinMinimizeAllUndo WinMove WinSetOnTop WinSetState WinSetTitle WinSetTrans WinWait WinWaitActive WinWaitClose WinWaitNotActive';

	  const COMMENT = {
	    variants: [
	      hljs.COMMENT(';', '$', {
	        relevance: 0
	      }),
	      hljs.COMMENT('#cs', '#ce'),
	      hljs.COMMENT('#comments-start', '#comments-end')
	    ]
	  };

	  const VARIABLE = {
	    begin: '\\$[A-z0-9_]+'
	  };

	  const STRING = {
	    className: 'string',
	    variants: [
	      {
	        begin: /"/,
	        end: /"/,
	        contains: [{
	          begin: /""/,
	          relevance: 0
	        }]
	      },
	      {
	        begin: /'/,
	        end: /'/,
	        contains: [{
	          begin: /''/,
	          relevance: 0
	        }]
	      }
	    ]
	  };

	  const NUMBER = {
	    variants: [
	      hljs.BINARY_NUMBER_MODE,
	      hljs.C_NUMBER_MODE
	    ]
	  };

	  const PREPROCESSOR = {
	    className: 'meta',
	    begin: '#',
	    end: '$',
	    keywords: {
	      'meta-keyword': DIRECTIVES
	    },
	    contains: [
	      {
	        begin: /\\\n/,
	        relevance: 0
	      },
	      {
	        beginKeywords: 'include',
	        keywords: {
	          'meta-keyword': 'include'
	        },
	        end: '$',
	        contains: [
	          STRING,
	          {
	            className: 'meta-string',
	            variants: [
	              {
	                begin: '<',
	                end: '>'
	              },
	              {
	                begin: /"/,
	                end: /"/,
	                contains: [{
	                  begin: /""/,
	                  relevance: 0
	                }]
	              },
	              {
	                begin: /'/,
	                end: /'/,
	                contains: [{
	                  begin: /''/,
	                  relevance: 0
	                }]
	              }
	            ]
	          }
	        ]
	      },
	      STRING,
	      COMMENT
	    ]
	  };

	  const CONSTANT = {
	    className: 'symbol',
	    // begin: '@',
	    // end: '$',
	    // keywords: 'AppDataCommonDir AppDataDir AutoItExe AutoItPID AutoItVersion AutoItX64 COM_EventObj CommonFilesDir Compiled ComputerName ComSpec CPUArch CR CRLF DesktopCommonDir DesktopDepth DesktopDir DesktopHeight DesktopRefresh DesktopWidth DocumentsCommonDir error exitCode exitMethod extended FavoritesCommonDir FavoritesDir GUI_CtrlHandle GUI_CtrlId GUI_DragFile GUI_DragId GUI_DropId GUI_WinHandle HomeDrive HomePath HomeShare HotKeyPressed HOUR IPAddress1 IPAddress2 IPAddress3 IPAddress4 KBLayout LF LocalAppDataDir LogonDNSDomain LogonDomain LogonServer MDAY MIN MON MSEC MUILang MyDocumentsDir NumParams OSArch OSBuild OSLang OSServicePack OSType OSVersion ProgramFilesDir ProgramsCommonDir ProgramsDir ScriptDir ScriptFullPath ScriptLineNumber ScriptName SEC StartMenuCommonDir StartMenuDir StartupCommonDir StartupDir SW_DISABLE SW_ENABLE SW_HIDE SW_LOCK SW_MAXIMIZE SW_MINIMIZE SW_RESTORE SW_SHOW SW_SHOWDEFAULT SW_SHOWMAXIMIZED SW_SHOWMINIMIZED SW_SHOWMINNOACTIVE SW_SHOWNA SW_SHOWNOACTIVATE SW_SHOWNORMAL SW_UNLOCK SystemDir TAB TempDir TRAY_ID TrayIconFlashing TrayIconVisible UserName UserProfileDir WDAY WindowsDir WorkingDir YDAY YEAR',
	    // relevance: 5
	    begin: '@[A-z0-9_]+'
	  };

	  const FUNCTION = {
	    className: 'function',
	    beginKeywords: 'Func',
	    end: '$',
	    illegal: '\\$|\\[|%',
	    contains: [
	      hljs.UNDERSCORE_TITLE_MODE,
	      {
	        className: 'params',
	        begin: '\\(',
	        end: '\\)',
	        contains: [
	          VARIABLE,
	          STRING,
	          NUMBER
	        ]
	      }
	    ]
	  };

	  return {
	    name: 'AutoIt',
	    case_insensitive: true,
	    illegal: /\/\*/,
	    keywords: {
	      keyword: KEYWORDS,
	      built_in: BUILT_IN,
	      literal: LITERAL
	    },
	    contains: [
	      COMMENT,
	      VARIABLE,
	      STRING,
	      NUMBER,
	      PREPROCESSOR,
	      CONSTANT,
	      FUNCTION
	    ]
	  };
	}

	autoit_1 = autoit;
	return autoit_1;
}

/*
Language: AVR Assembly
Author: Vladimir Ermakov <vooon341@gmail.com>
Category: assembler
Website: https://www.microchip.com/webdoc/avrassembler/avrassembler.wb_instruction_list.html
*/

var avrasm_1;
var hasRequiredAvrasm;

function requireAvrasm () {
	if (hasRequiredAvrasm) return avrasm_1;
	hasRequiredAvrasm = 1;
	/** @type LanguageFn */
	function avrasm(hljs) {
	  return {
	    name: 'AVR Assembly',
	    case_insensitive: true,
	    keywords: {
	      $pattern: '\\.?' + hljs.IDENT_RE,
	      keyword:
	        /* mnemonic */
	        'adc add adiw and andi asr bclr bld brbc brbs brcc brcs break breq brge brhc brhs ' +
	        'brid brie brlo brlt brmi brne brpl brsh brtc brts brvc brvs bset bst call cbi cbr ' +
	        'clc clh cli cln clr cls clt clv clz com cp cpc cpi cpse dec eicall eijmp elpm eor ' +
	        'fmul fmuls fmulsu icall ijmp in inc jmp ld ldd ldi lds lpm lsl lsr mov movw mul ' +
	        'muls mulsu neg nop or ori out pop push rcall ret reti rjmp rol ror sbc sbr sbrc sbrs ' +
	        'sec seh sbi sbci sbic sbis sbiw sei sen ser ses set sev sez sleep spm st std sts sub ' +
	        'subi swap tst wdr',
	      built_in:
	        /* general purpose registers */
	        'r0 r1 r2 r3 r4 r5 r6 r7 r8 r9 r10 r11 r12 r13 r14 r15 r16 r17 r18 r19 r20 r21 r22 ' +
	        'r23 r24 r25 r26 r27 r28 r29 r30 r31 x|0 xh xl y|0 yh yl z|0 zh zl ' +
	        /* IO Registers (ATMega128) */
	        'ucsr1c udr1 ucsr1a ucsr1b ubrr1l ubrr1h ucsr0c ubrr0h tccr3c tccr3a tccr3b tcnt3h ' +
	        'tcnt3l ocr3ah ocr3al ocr3bh ocr3bl ocr3ch ocr3cl icr3h icr3l etimsk etifr tccr1c ' +
	        'ocr1ch ocr1cl twcr twdr twar twsr twbr osccal xmcra xmcrb eicra spmcsr spmcr portg ' +
	        'ddrg ping portf ddrf sreg sph spl xdiv rampz eicrb eimsk gimsk gicr eifr gifr timsk ' +
	        'tifr mcucr mcucsr tccr0 tcnt0 ocr0 assr tccr1a tccr1b tcnt1h tcnt1l ocr1ah ocr1al ' +
	        'ocr1bh ocr1bl icr1h icr1l tccr2 tcnt2 ocr2 ocdr wdtcr sfior eearh eearl eedr eecr ' +
	        'porta ddra pina portb ddrb pinb portc ddrc pinc portd ddrd pind spdr spsr spcr udr0 ' +
	        'ucsr0a ucsr0b ubrr0l acsr admux adcsr adch adcl porte ddre pine pinf',
	      meta:
	        '.byte .cseg .db .def .device .dseg .dw .endmacro .equ .eseg .exit .include .list ' +
	        '.listmac .macro .nolist .org .set'
	    },
	    contains: [
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.COMMENT(
	        ';',
	        '$',
	        {
	          relevance: 0
	        }
	      ),
	      hljs.C_NUMBER_MODE, // 0x..., decimal, float
	      hljs.BINARY_NUMBER_MODE, // 0b...
	      {
	        className: 'number',
	        begin: '\\b(\\$[a-zA-Z0-9]+|0o[0-7]+)' // $..., 0o...
	      },
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'string',
	        begin: '\'',
	        end: '[^\\\\]\'',
	        illegal: '[^\\\\][^\']'
	      },
	      {
	        className: 'symbol',
	        begin: '^[A-Za-z0-9_.$]+:'
	      },
	      {
	        className: 'meta',
	        begin: '#',
	        end: '$'
	      },
	      { // substitution within a macro
	        className: 'subst',
	        begin: '@[0-9]+'
	      }
	    ]
	  };
	}

	avrasm_1 = avrasm;
	return avrasm_1;
}

/*
Language: Awk
Author: Matthew Daly <matthewbdaly@gmail.com>
Website: https://www.gnu.org/software/gawk/manual/gawk.html
Description: language definition for Awk scripts
*/

var awk_1;
var hasRequiredAwk;

function requireAwk () {
	if (hasRequiredAwk) return awk_1;
	hasRequiredAwk = 1;
	/** @type LanguageFn */
	function awk(hljs) {
	  const VARIABLE = {
	    className: 'variable',
	    variants: [
	      {
	        begin: /\$[\w\d#@][\w\d_]*/
	      },
	      {
	        begin: /\$\{(.*?)\}/
	      }
	    ]
	  };
	  const KEYWORDS = 'BEGIN END if else while do for in break continue delete next nextfile function func exit|10';
	  const STRING = {
	    className: 'string',
	    contains: [hljs.BACKSLASH_ESCAPE],
	    variants: [
	      {
	        begin: /(u|b)?r?'''/,
	        end: /'''/,
	        relevance: 10
	      },
	      {
	        begin: /(u|b)?r?"""/,
	        end: /"""/,
	        relevance: 10
	      },
	      {
	        begin: /(u|r|ur)'/,
	        end: /'/,
	        relevance: 10
	      },
	      {
	        begin: /(u|r|ur)"/,
	        end: /"/,
	        relevance: 10
	      },
	      {
	        begin: /(b|br)'/,
	        end: /'/
	      },
	      {
	        begin: /(b|br)"/,
	        end: /"/
	      },
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE
	    ]
	  };
	  return {
	    name: 'Awk',
	    keywords: {
	      keyword: KEYWORDS
	    },
	    contains: [
	      VARIABLE,
	      STRING,
	      hljs.REGEXP_MODE,
	      hljs.HASH_COMMENT_MODE,
	      hljs.NUMBER_MODE
	    ]
	  };
	}

	awk_1 = awk;
	return awk_1;
}

/*
Language: Microsoft X++
Description: X++ is a language used in Microsoft Dynamics 365, Dynamics AX, and Axapta.
Author: Dmitri Roudakov <dmitri@roudakov.ru>
Website: https://dynamics.microsoft.com/en-us/ax-overview/
Category: enterprise
*/

var axapta_1;
var hasRequiredAxapta;

function requireAxapta () {
	if (hasRequiredAxapta) return axapta_1;
	hasRequiredAxapta = 1;
	/** @type LanguageFn */
	function axapta(hljs) {
	  const BUILT_IN_KEYWORDS = [
	    'anytype',
	    'boolean',
	    'byte',
	    'char',
	    'container',
	    'date',
	    'double',
	    'enum',
	    'guid',
	    'int',
	    'int64',
	    'long',
	    'real',
	    'short',
	    'str',
	    'utcdatetime',
	    'var'
	  ];

	  const LITERAL_KEYWORDS = [
	    'default',
	    'false',
	    'null',
	    'true'
	  ];

	  const NORMAL_KEYWORDS = [
	    'abstract',
	    'as',
	    'asc',
	    'avg',
	    'break',
	    'breakpoint',
	    'by',
	    'byref',
	    'case',
	    'catch',
	    'changecompany',
	    'class',
	    'client',
	    'client',
	    'common',
	    'const',
	    'continue',
	    'count',
	    'crosscompany',
	    'delegate',
	    'delete_from',
	    'desc',
	    'display',
	    'div',
	    'do',
	    'edit',
	    'else',
	    'eventhandler',
	    'exists',
	    'extends',
	    'final',
	    'finally',
	    'firstfast',
	    'firstonly',
	    'firstonly1',
	    'firstonly10',
	    'firstonly100',
	    'firstonly1000',
	    'flush',
	    'for',
	    'forceliterals',
	    'forcenestedloop',
	    'forceplaceholders',
	    'forceselectorder',
	    'forupdate',
	    'from',
	    'generateonly',
	    'group',
	    'hint',
	    'if',
	    'implements',
	    'in',
	    'index',
	    'insert_recordset',
	    'interface',
	    'internal',
	    'is',
	    'join',
	    'like',
	    'maxof',
	    'minof',
	    'mod',
	    'namespace',
	    'new',
	    'next',
	    'nofetch',
	    'notexists',
	    'optimisticlock',
	    'order',
	    'outer',
	    'pessimisticlock',
	    'print',
	    'private',
	    'protected',
	    'public',
	    'readonly',
	    'repeatableread',
	    'retry',
	    'return',
	    'reverse',
	    'select',
	    'server',
	    'setting',
	    'static',
	    'sum',
	    'super',
	    'switch',
	    'this',
	    'throw',
	    'try',
	    'ttsabort',
	    'ttsbegin',
	    'ttscommit',
	    'unchecked',
	    'update_recordset',
	    'using',
	    'validtimestate',
	    'void',
	    'where',
	    'while'
	  ];

	  const KEYWORDS = {
	    keyword: NORMAL_KEYWORDS,
	    built_in: BUILT_IN_KEYWORDS,
	    literal: LITERAL_KEYWORDS
	  };

	  return {
	    name: 'X++',
	    aliases: ['x++'],
	    keywords: KEYWORDS,
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.C_NUMBER_MODE,
	      {
	        className: 'meta',
	        begin: '#',
	        end: '$'
	      },
	      {
	        className: 'class',
	        beginKeywords: 'class interface',
	        end: /\{/,
	        excludeEnd: true,
	        illegal: ':',
	        contains: [
	          {
	            beginKeywords: 'extends implements'
	          },
	          hljs.UNDERSCORE_TITLE_MODE
	        ]
	      }
	    ]
	  };
	}

	axapta_1 = axapta;
	return axapta_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var bash_1;
var hasRequiredBash;

function requireBash () {
	if (hasRequiredBash) return bash_1;
	hasRequiredBash = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: Bash
	Author: vah <vahtenberg@gmail.com>
	Contributrors: Benjamin Pannell <contact@sierrasoftworks.com>
	Website: https://www.gnu.org/software/bash/
	Category: common
	*/

	/** @type LanguageFn */
	function bash(hljs) {
	  const VAR = {};
	  const BRACED_VAR = {
	    begin: /\$\{/,
	    end:/\}/,
	    contains: [
	      "self",
	      {
	        begin: /:-/,
	        contains: [ VAR ]
	      } // default values
	    ]
	  };
	  Object.assign(VAR,{
	    className: 'variable',
	    variants: [
	      {begin: concat(/\$[\w\d#@][\w\d_]*/,
	        // negative look-ahead tries to avoid matching patterns that are not
	        // Perl at all like $ident$, @ident@, etc.
	        `(?![\\w\\d])(?![$])`) },
	      BRACED_VAR
	    ]
	  });

	  const SUBST = {
	    className: 'subst',
	    begin: /\$\(/, end: /\)/,
	    contains: [hljs.BACKSLASH_ESCAPE]
	  };
	  const HERE_DOC = {
	    begin: /<<-?\s*(?=\w+)/,
	    starts: {
	      contains: [
	        hljs.END_SAME_AS_BEGIN({
	          begin: /(\w+)/,
	          end: /(\w+)/,
	          className: 'string'
	        })
	      ]
	    }
	  };
	  const QUOTE_STRING = {
	    className: 'string',
	    begin: /"/, end: /"/,
	    contains: [
	      hljs.BACKSLASH_ESCAPE,
	      VAR,
	      SUBST
	    ]
	  };
	  SUBST.contains.push(QUOTE_STRING);
	  const ESCAPED_QUOTE = {
	    className: '',
	    begin: /\\"/

	  };
	  const APOS_STRING = {
	    className: 'string',
	    begin: /'/, end: /'/
	  };
	  const ARITHMETIC = {
	    begin: /\$\(\(/,
	    end: /\)\)/,
	    contains: [
	      { begin: /\d+#[0-9a-f]+/, className: "number" },
	      hljs.NUMBER_MODE,
	      VAR
	    ]
	  };
	  const SH_LIKE_SHELLS = [
	    "fish",
	    "bash",
	    "zsh",
	    "sh",
	    "csh",
	    "ksh",
	    "tcsh",
	    "dash",
	    "scsh",
	  ];
	  const KNOWN_SHEBANG = hljs.SHEBANG({
	    binary: `(${SH_LIKE_SHELLS.join("|")})`,
	    relevance: 10
	  });
	  const FUNCTION = {
	    className: 'function',
	    begin: /\w[\w\d_]*\s*\(\s*\)\s*\{/,
	    returnBegin: true,
	    contains: [hljs.inherit(hljs.TITLE_MODE, {begin: /\w[\w\d_]*/})],
	    relevance: 0
	  };

	  return {
	    name: 'Bash',
	    aliases: ['sh', 'zsh'],
	    keywords: {
	      $pattern: /\b[a-z._-]+\b/,
	      keyword:
	        'if then else elif fi for while in do done case esac function',
	      literal:
	        'true false',
	      built_in:
	        // Shell built-ins
	        // http://www.gnu.org/software/bash/manual/html_node/Shell-Builtin-Commands.html
	        'break cd continue eval exec exit export getopts hash pwd readonly return shift test times ' +
	        'trap umask unset ' +
	        // Bash built-ins
	        'alias bind builtin caller command declare echo enable help let local logout mapfile printf ' +
	        'read readarray source type typeset ulimit unalias ' +
	        // Shell modifiers
	        'set shopt ' +
	        // Zsh built-ins
	        'autoload bg bindkey bye cap chdir clone comparguments compcall compctl compdescribe compfiles ' +
	        'compgroups compquote comptags comptry compvalues dirs disable disown echotc echoti emulate ' +
	        'fc fg float functions getcap getln history integer jobs kill limit log noglob popd print ' +
	        'pushd pushln rehash sched setcap setopt stat suspend ttyctl unfunction unhash unlimit ' +
	        'unsetopt vared wait whence where which zcompile zformat zftp zle zmodload zparseopts zprof ' +
	        'zpty zregexparse zsocket zstyle ztcp'
	    },
	    contains: [
	      KNOWN_SHEBANG, // to catch known shells and boost relevancy
	      hljs.SHEBANG(), // to catch unknown shells but still highlight the shebang
	      FUNCTION,
	      ARITHMETIC,
	      hljs.HASH_COMMENT_MODE,
	      HERE_DOC,
	      QUOTE_STRING,
	      ESCAPED_QUOTE,
	      APOS_STRING,
	      VAR
	    ]
	  };
	}

	bash_1 = bash;
	return bash_1;
}

/*
Language: BASIC
Author: Raphaël Assénat <raph@raphnet.net>
Description: Based on the BASIC reference from the Tandy 1000 guide
Website: https://en.wikipedia.org/wiki/Tandy_1000
*/

var basic_1;
var hasRequiredBasic;

function requireBasic () {
	if (hasRequiredBasic) return basic_1;
	hasRequiredBasic = 1;
	/** @type LanguageFn */
	function basic(hljs) {
	  return {
	    name: 'BASIC',
	    case_insensitive: true,
	    illegal: '^\.',
	    // Support explicitly typed variables that end with $%! or #.
	    keywords: {
	      $pattern: '[a-zA-Z][a-zA-Z0-9_$%!#]*',
	      keyword:
	        'ABS ASC AND ATN AUTO|0 BEEP BLOAD|10 BSAVE|10 CALL CALLS CDBL CHAIN CHDIR CHR$|10 CINT CIRCLE ' +
	        'CLEAR CLOSE CLS COLOR COM COMMON CONT COS CSNG CSRLIN CVD CVI CVS DATA DATE$ ' +
	        'DEFDBL DEFINT DEFSNG DEFSTR DEF|0 SEG USR DELETE DIM DRAW EDIT END ENVIRON ENVIRON$ ' +
	        'EOF EQV ERASE ERDEV ERDEV$ ERL ERR ERROR EXP FIELD FILES FIX FOR|0 FRE GET GOSUB|10 GOTO ' +
	        'HEX$ IF THEN ELSE|0 INKEY$ INP INPUT INPUT# INPUT$ INSTR IMP INT IOCTL IOCTL$ KEY ON ' +
	        'OFF LIST KILL LEFT$ LEN LET LINE LLIST LOAD LOC LOCATE LOF LOG LPRINT USING LSET ' +
	        'MERGE MID$ MKDIR MKD$ MKI$ MKS$ MOD NAME NEW NEXT NOISE NOT OCT$ ON OR PEN PLAY STRIG OPEN OPTION ' +
	        'BASE OUT PAINT PALETTE PCOPY PEEK PMAP POINT POKE POS PRINT PRINT] PSET PRESET ' +
	        'PUT RANDOMIZE READ REM RENUM RESET|0 RESTORE RESUME RETURN|0 RIGHT$ RMDIR RND RSET ' +
	        'RUN SAVE SCREEN SGN SHELL SIN SOUND SPACE$ SPC SQR STEP STICK STOP STR$ STRING$ SWAP ' +
	        'SYSTEM TAB TAN TIME$ TIMER TROFF TRON TO USR VAL VARPTR VARPTR$ VIEW WAIT WHILE ' +
	        'WEND WIDTH WINDOW WRITE XOR'
	    },
	    contains: [
	      hljs.QUOTE_STRING_MODE,
	      hljs.COMMENT('REM', '$', {
	        relevance: 10
	      }),
	      hljs.COMMENT('\'', '$', {
	        relevance: 0
	      }),
	      {
	        // Match line numbers
	        className: 'symbol',
	        begin: '^[0-9]+ ',
	        relevance: 10
	      },
	      {
	        // Match typed numeric constants (1000, 12.34!, 1.2e5, 1.5#, 1.2D2)
	        className: 'number',
	        begin: '\\b\\d+(\\.\\d+)?([edED]\\d+)?[#\!]?',
	        relevance: 0
	      },
	      {
	        // Match hexadecimal numbers (&Hxxxx)
	        className: 'number',
	        begin: '(&[hH][0-9a-fA-F]{1,4})'
	      },
	      {
	        // Match octal numbers (&Oxxxxxx)
	        className: 'number',
	        begin: '(&[oO][0-7]{1,6})'
	      }
	    ]
	  };
	}

	basic_1 = basic;
	return basic_1;
}

/*
Language: Backus–Naur Form
Website: https://en.wikipedia.org/wiki/Backus–Naur_form
Author: Oleg Efimov <efimovov@gmail.com>
*/

var bnf_1;
var hasRequiredBnf;

function requireBnf () {
	if (hasRequiredBnf) return bnf_1;
	hasRequiredBnf = 1;
	/** @type LanguageFn */
	function bnf(hljs) {
	  return {
	    name: 'Backus–Naur Form',
	    contains: [
	      // Attribute
	      {
	        className: 'attribute',
	        begin: /</,
	        end: />/
	      },
	      // Specific
	      {
	        begin: /::=/,
	        end: /$/,
	        contains: [
	          {
	            begin: /</,
	            end: />/
	          },
	          // Common
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE,
	          hljs.APOS_STRING_MODE,
	          hljs.QUOTE_STRING_MODE
	        ]
	      }
	    ]
	  };
	}

	bnf_1 = bnf;
	return bnf_1;
}

/*
Language: Brainfuck
Author: Evgeny Stepanischev <imbolk@gmail.com>
Website: https://esolangs.org/wiki/Brainfuck
*/

var brainfuck_1;
var hasRequiredBrainfuck;

function requireBrainfuck () {
	if (hasRequiredBrainfuck) return brainfuck_1;
	hasRequiredBrainfuck = 1;
	/** @type LanguageFn */
	function brainfuck(hljs) {
	  const LITERAL = {
	    className: 'literal',
	    begin: /[+-]/,
	    relevance: 0
	  };
	  return {
	    name: 'Brainfuck',
	    aliases: ['bf'],
	    contains: [
	      hljs.COMMENT(
	        '[^\\[\\]\\.,\\+\\-<> \r\n]',
	        '[\\[\\]\\.,\\+\\-<> \r\n]',
	        {
	          returnEnd: true,
	          relevance: 0
	        }
	      ),
	      {
	        className: 'title',
	        begin: '[\\[\\]]',
	        relevance: 0
	      },
	      {
	        className: 'string',
	        begin: '[\\.,]',
	        relevance: 0
	      },
	      {
	        // this mode works as the only relevance counter
	        begin: /(?:\+\+|--)/,
	        contains: [LITERAL]
	      },
	      LITERAL
	    ]
	  };
	}

	brainfuck_1 = brainfuck;
	return brainfuck_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var cLike_1;
var hasRequiredCLike;

function requireCLike () {
	if (hasRequiredCLike) return cLike_1;
	hasRequiredCLike = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function optional(re) {
	  return concat('(', re, ')?');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: C++
	Category: common, system
	Website: https://isocpp.org
	*/

	/** @type LanguageFn */
	function cPlusPlus(hljs) {
	  // added for historic reasons because `hljs.C_LINE_COMMENT_MODE` does
	  // not include such support nor can we be sure all the grammars depending
	  // on it would desire this behavior
	  const C_LINE_COMMENT_MODE = hljs.COMMENT('//', '$', {
	    contains: [
	      {
	        begin: /\\\n/
	      }
	    ]
	  });
	  const DECLTYPE_AUTO_RE = 'decltype\\(auto\\)';
	  const NAMESPACE_RE = '[a-zA-Z_]\\w*::';
	  const TEMPLATE_ARGUMENT_RE = '<[^<>]+>';
	  const FUNCTION_TYPE_RE = '(' +
	    DECLTYPE_AUTO_RE + '|' +
	    optional(NAMESPACE_RE) +
	    '[a-zA-Z_]\\w*' + optional(TEMPLATE_ARGUMENT_RE) +
	  ')';
	  const CPP_PRIMITIVE_TYPES = {
	    className: 'keyword',
	    begin: '\\b[a-z\\d_]*_t\\b'
	  };

	  // https://en.cppreference.com/w/cpp/language/escape
	  // \\ \x \xFF \u2837 \u00323747 \374
	  const CHARACTER_ESCAPES = '\\\\(x[0-9A-Fa-f]{2}|u[0-9A-Fa-f]{4,8}|[0-7]{3}|\\S)';
	  const STRINGS = {
	    className: 'string',
	    variants: [
	      {
	        begin: '(u8?|U|L)?"',
	        end: '"',
	        illegal: '\\n',
	        contains: [ hljs.BACKSLASH_ESCAPE ]
	      },
	      {
	        begin: '(u8?|U|L)?\'(' + CHARACTER_ESCAPES + "|.)",
	        end: '\'',
	        illegal: '.'
	      },
	      hljs.END_SAME_AS_BEGIN({
	        begin: /(?:u8?|U|L)?R"([^()\\ ]{0,16})\(/,
	        end: /\)([^()\\ ]{0,16})"/
	      })
	    ]
	  };

	  const NUMBERS = {
	    className: 'number',
	    variants: [
	      {
	        begin: '\\b(0b[01\']+)'
	      },
	      {
	        begin: '(-?)\\b([\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)((ll|LL|l|L)(u|U)?|(u|U)(ll|LL|l|L)?|f|F|b|B)'
	      },
	      {
	        begin: '(-?)(\\b0[xX][a-fA-F0-9\']+|(\\b[\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)([eE][-+]?[\\d\']+)?)'
	      }
	    ],
	    relevance: 0
	  };

	  const PREPROCESSOR = {
	    className: 'meta',
	    begin: /#\s*[a-z]+\b/,
	    end: /$/,
	    keywords: {
	      'meta-keyword':
	        'if else elif endif define undef warning error line ' +
	        'pragma _Pragma ifdef ifndef include'
	    },
	    contains: [
	      {
	        begin: /\\\n/,
	        relevance: 0
	      },
	      hljs.inherit(STRINGS, {
	        className: 'meta-string'
	      }),
	      {
	        className: 'meta-string',
	        begin: /<.*?>/
	      },
	      C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE
	    ]
	  };

	  const TITLE_MODE = {
	    className: 'title',
	    begin: optional(NAMESPACE_RE) + hljs.IDENT_RE,
	    relevance: 0
	  };

	  const FUNCTION_TITLE = optional(NAMESPACE_RE) + hljs.IDENT_RE + '\\s*\\(';

	  const COMMON_CPP_HINTS = [
	    'asin',
	    'atan2',
	    'atan',
	    'calloc',
	    'ceil',
	    'cosh',
	    'cos',
	    'exit',
	    'exp',
	    'fabs',
	    'floor',
	    'fmod',
	    'fprintf',
	    'fputs',
	    'free',
	    'frexp',
	    'auto_ptr',
	    'deque',
	    'list',
	    'queue',
	    'stack',
	    'vector',
	    'map',
	    'set',
	    'pair',
	    'bitset',
	    'multiset',
	    'multimap',
	    'unordered_set',
	    'fscanf',
	    'future',
	    'isalnum',
	    'isalpha',
	    'iscntrl',
	    'isdigit',
	    'isgraph',
	    'islower',
	    'isprint',
	    'ispunct',
	    'isspace',
	    'isupper',
	    'isxdigit',
	    'tolower',
	    'toupper',
	    'labs',
	    'ldexp',
	    'log10',
	    'log',
	    'malloc',
	    'realloc',
	    'memchr',
	    'memcmp',
	    'memcpy',
	    'memset',
	    'modf',
	    'pow',
	    'printf',
	    'putchar',
	    'puts',
	    'scanf',
	    'sinh',
	    'sin',
	    'snprintf',
	    'sprintf',
	    'sqrt',
	    'sscanf',
	    'strcat',
	    'strchr',
	    'strcmp',
	    'strcpy',
	    'strcspn',
	    'strlen',
	    'strncat',
	    'strncmp',
	    'strncpy',
	    'strpbrk',
	    'strrchr',
	    'strspn',
	    'strstr',
	    'tanh',
	    'tan',
	    'unordered_map',
	    'unordered_multiset',
	    'unordered_multimap',
	    'priority_queue',
	    'make_pair',
	    'array',
	    'shared_ptr',
	    'abort',
	    'terminate',
	    'abs',
	    'acos',
	    'vfprintf',
	    'vprintf',
	    'vsprintf',
	    'endl',
	    'initializer_list',
	    'unique_ptr',
	    'complex',
	    'imaginary',
	    'std',
	    'string',
	    'wstring',
	    'cin',
	    'cout',
	    'cerr',
	    'clog',
	    'stdin',
	    'stdout',
	    'stderr',
	    'stringstream',
	    'istringstream',
	    'ostringstream'
	  ];

	  const CPP_KEYWORDS = {
	    keyword: 'int float while private char char8_t char16_t char32_t catch import module export virtual operator sizeof ' +
	      'dynamic_cast|10 typedef const_cast|10 const for static_cast|10 union namespace ' +
	      'unsigned long volatile static protected bool template mutable if public friend ' +
	      'do goto auto void enum else break extern using asm case typeid wchar_t ' +
	      'short reinterpret_cast|10 default double register explicit signed typename try this ' +
	      'switch continue inline delete alignas alignof constexpr consteval constinit decltype ' +
	      'concept co_await co_return co_yield requires ' +
	      'noexcept static_assert thread_local restrict final override ' +
	      'atomic_bool atomic_char atomic_schar ' +
	      'atomic_uchar atomic_short atomic_ushort atomic_int atomic_uint atomic_long atomic_ulong atomic_llong ' +
	      'atomic_ullong new throw return ' +
	      'and and_eq bitand bitor compl not not_eq or or_eq xor xor_eq',
	    built_in: '_Bool _Complex _Imaginary',
	    _relevance_hints: COMMON_CPP_HINTS,
	    literal: 'true false nullptr NULL'
	  };

	  const FUNCTION_DISPATCH = {
	    className: "function.dispatch",
	    relevance: 0,
	    keywords: CPP_KEYWORDS,
	    begin: concat(
	      /\b/,
	      /(?!decltype)/,
	      /(?!if)/,
	      /(?!for)/,
	      /(?!while)/,
	      hljs.IDENT_RE,
	      lookahead(/\s*\(/))
	  };

	  const EXPRESSION_CONTAINS = [
	    FUNCTION_DISPATCH,
	    PREPROCESSOR,
	    CPP_PRIMITIVE_TYPES,
	    C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    NUMBERS,
	    STRINGS
	  ];


	  const EXPRESSION_CONTEXT = {
	    // This mode covers expression context where we can't expect a function
	    // definition and shouldn't highlight anything that looks like one:
	    // `return some()`, `else if()`, `(x*sum(1, 2))`
	    variants: [
	      {
	        begin: /=/,
	        end: /;/
	      },
	      {
	        begin: /\(/,
	        end: /\)/
	      },
	      {
	        beginKeywords: 'new throw return else',
	        end: /;/
	      }
	    ],
	    keywords: CPP_KEYWORDS,
	    contains: EXPRESSION_CONTAINS.concat([
	      {
	        begin: /\(/,
	        end: /\)/,
	        keywords: CPP_KEYWORDS,
	        contains: EXPRESSION_CONTAINS.concat([ 'self' ]),
	        relevance: 0
	      }
	    ]),
	    relevance: 0
	  };

	  const FUNCTION_DECLARATION = {
	    className: 'function',
	    begin: '(' + FUNCTION_TYPE_RE + '[\\*&\\s]+)+' + FUNCTION_TITLE,
	    returnBegin: true,
	    end: /[{;=]/,
	    excludeEnd: true,
	    keywords: CPP_KEYWORDS,
	    illegal: /[^\w\s\*&:<>.]/,
	    contains: [
	      { // to prevent it from being confused as the function title
	        begin: DECLTYPE_AUTO_RE,
	        keywords: CPP_KEYWORDS,
	        relevance: 0
	      },
	      {
	        begin: FUNCTION_TITLE,
	        returnBegin: true,
	        contains: [ TITLE_MODE ],
	        relevance: 0
	      },
	      // needed because we do not have look-behind on the below rule
	      // to prevent it from grabbing the final : in a :: pair
	      {
	        begin: /::/,
	        relevance: 0
	      },
	      // initializers
	      {
	        begin: /:/,
	        endsWithParent: true,
	        contains: [
	          STRINGS,
	          NUMBERS
	        ]
	      },
	      {
	        className: 'params',
	        begin: /\(/,
	        end: /\)/,
	        keywords: CPP_KEYWORDS,
	        relevance: 0,
	        contains: [
	          C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE,
	          STRINGS,
	          NUMBERS,
	          CPP_PRIMITIVE_TYPES,
	          // Count matching parentheses.
	          {
	            begin: /\(/,
	            end: /\)/,
	            keywords: CPP_KEYWORDS,
	            relevance: 0,
	            contains: [
	              'self',
	              C_LINE_COMMENT_MODE,
	              hljs.C_BLOCK_COMMENT_MODE,
	              STRINGS,
	              NUMBERS,
	              CPP_PRIMITIVE_TYPES
	            ]
	          }
	        ]
	      },
	      CPP_PRIMITIVE_TYPES,
	      C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      PREPROCESSOR
	    ]
	  };

	  return {
	    name: 'C++',
	    aliases: [
	      'cc',
	      'c++',
	      'h++',
	      'hpp',
	      'hh',
	      'hxx',
	      'cxx'
	    ],
	    keywords: CPP_KEYWORDS,
	    illegal: '</',
	    classNameAliases: {
	      "function.dispatch": "built_in"
	    },
	    contains: [].concat(
	      EXPRESSION_CONTEXT,
	      FUNCTION_DECLARATION,
	      FUNCTION_DISPATCH,
	      EXPRESSION_CONTAINS,
	      [
	        PREPROCESSOR,
	        { // containers: ie, `vector <int> rooms (9);`
	          begin: '\\b(deque|list|queue|priority_queue|pair|stack|vector|map|set|bitset|multiset|multimap|unordered_map|unordered_set|unordered_multiset|unordered_multimap|array)\\s*<',
	          end: '>',
	          keywords: CPP_KEYWORDS,
	          contains: [
	            'self',
	            CPP_PRIMITIVE_TYPES
	          ]
	        },
	        {
	          begin: hljs.IDENT_RE + '::',
	          keywords: CPP_KEYWORDS
	        },
	        {
	          className: 'class',
	          beginKeywords: 'enum class struct union',
	          end: /[{;:<>=]/,
	          contains: [
	            {
	              beginKeywords: "final class struct"
	            },
	            hljs.TITLE_MODE
	          ]
	        }
	      ]),
	    exports: {
	      preprocessor: PREPROCESSOR,
	      strings: STRINGS,
	      keywords: CPP_KEYWORDS
	    }
	  };
	}

	/*
	Language: C-like (deprecated, use C and C++ instead)
	Author: Ivan Sagalaev <maniac@softwaremaniacs.org>
	Contributors: Evgeny Stepanischev <imbolk@gmail.com>, Zaven Muradyan <megalivoithos@gmail.com>, Roel Deckers <admin@codingcat.nl>, Sam Wu <samsam2310@gmail.com>, Jordi Petit <jordi.petit@gmail.com>, Pieter Vantorre <pietervantorre@gmail.com>, Google Inc. (David Benjamin) <davidben@google.com>
	*/

	/** @type LanguageFn */
	function cLike(hljs) {
	  const lang = cPlusPlus(hljs);

	  const C_ALIASES = [
	    "c",
	    "h"
	  ];

	  const CPP_ALIASES = [
	    'cc',
	    'c++',
	    'h++',
	    'hpp',
	    'hh',
	    'hxx',
	    'cxx'
	  ];

	  lang.disableAutodetect = true;
	  lang.aliases = [];
	  // support users only loading c-like (legacy)
	  if (!hljs.getLanguage("c")) lang.aliases.push(...C_ALIASES);
	  if (!hljs.getLanguage("cpp")) lang.aliases.push(...CPP_ALIASES);

	  // if c and cpp are loaded after then they will reclaim these
	  // aliases for themselves

	  return lang;
	}

	cLike_1 = cLike;
	return cLike_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var c_1;
var hasRequiredC;

function requireC () {
	if (hasRequiredC) return c_1;
	hasRequiredC = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function optional(re) {
	  return concat('(', re, ')?');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: C
	Category: common, system
	Website: https://en.wikipedia.org/wiki/C_(programming_language)
	*/

	/** @type LanguageFn */
	function c(hljs) {
	  // added for historic reasons because `hljs.C_LINE_COMMENT_MODE` does
	  // not include such support nor can we be sure all the grammars depending
	  // on it would desire this behavior
	  const C_LINE_COMMENT_MODE = hljs.COMMENT('//', '$', {
	    contains: [
	      {
	        begin: /\\\n/
	      }
	    ]
	  });
	  const DECLTYPE_AUTO_RE = 'decltype\\(auto\\)';
	  const NAMESPACE_RE = '[a-zA-Z_]\\w*::';
	  const TEMPLATE_ARGUMENT_RE = '<[^<>]+>';
	  const FUNCTION_TYPE_RE = '(' +
	    DECLTYPE_AUTO_RE + '|' +
	    optional(NAMESPACE_RE) +
	    '[a-zA-Z_]\\w*' + optional(TEMPLATE_ARGUMENT_RE) +
	  ')';
	  const CPP_PRIMITIVE_TYPES = {
	    className: 'keyword',
	    begin: '\\b[a-z\\d_]*_t\\b'
	  };

	  // https://en.cppreference.com/w/cpp/language/escape
	  // \\ \x \xFF \u2837 \u00323747 \374
	  const CHARACTER_ESCAPES = '\\\\(x[0-9A-Fa-f]{2}|u[0-9A-Fa-f]{4,8}|[0-7]{3}|\\S)';
	  const STRINGS = {
	    className: 'string',
	    variants: [
	      {
	        begin: '(u8?|U|L)?"',
	        end: '"',
	        illegal: '\\n',
	        contains: [ hljs.BACKSLASH_ESCAPE ]
	      },
	      {
	        begin: '(u8?|U|L)?\'(' + CHARACTER_ESCAPES + "|.)",
	        end: '\'',
	        illegal: '.'
	      },
	      hljs.END_SAME_AS_BEGIN({
	        begin: /(?:u8?|U|L)?R"([^()\\ ]{0,16})\(/,
	        end: /\)([^()\\ ]{0,16})"/
	      })
	    ]
	  };

	  const NUMBERS = {
	    className: 'number',
	    variants: [
	      {
	        begin: '\\b(0b[01\']+)'
	      },
	      {
	        begin: '(-?)\\b([\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)((ll|LL|l|L)(u|U)?|(u|U)(ll|LL|l|L)?|f|F|b|B)'
	      },
	      {
	        begin: '(-?)(\\b0[xX][a-fA-F0-9\']+|(\\b[\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)([eE][-+]?[\\d\']+)?)'
	      }
	    ],
	    relevance: 0
	  };

	  const PREPROCESSOR = {
	    className: 'meta',
	    begin: /#\s*[a-z]+\b/,
	    end: /$/,
	    keywords: {
	      'meta-keyword':
	        'if else elif endif define undef warning error line ' +
	        'pragma _Pragma ifdef ifndef include'
	    },
	    contains: [
	      {
	        begin: /\\\n/,
	        relevance: 0
	      },
	      hljs.inherit(STRINGS, {
	        className: 'meta-string'
	      }),
	      {
	        className: 'meta-string',
	        begin: /<.*?>/
	      },
	      C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE
	    ]
	  };

	  const TITLE_MODE = {
	    className: 'title',
	    begin: optional(NAMESPACE_RE) + hljs.IDENT_RE,
	    relevance: 0
	  };

	  const FUNCTION_TITLE = optional(NAMESPACE_RE) + hljs.IDENT_RE + '\\s*\\(';

	  const CPP_KEYWORDS = {
	    keyword: 'int float while private char char8_t char16_t char32_t catch import module export virtual operator sizeof ' +
	      'dynamic_cast|10 typedef const_cast|10 const for static_cast|10 union namespace ' +
	      'unsigned long volatile static protected bool template mutable if public friend ' +
	      'do goto auto void enum else break extern using asm case typeid wchar_t ' +
	      'short reinterpret_cast|10 default double register explicit signed typename try this ' +
	      'switch continue inline delete alignas alignof constexpr consteval constinit decltype ' +
	      'concept co_await co_return co_yield requires ' +
	      'noexcept static_assert thread_local restrict final override ' +
	      'atomic_bool atomic_char atomic_schar ' +
	      'atomic_uchar atomic_short atomic_ushort atomic_int atomic_uint atomic_long atomic_ulong atomic_llong ' +
	      'atomic_ullong new throw return ' +
	      'and and_eq bitand bitor compl not not_eq or or_eq xor xor_eq',
	    built_in: 'std string wstring cin cout cerr clog stdin stdout stderr stringstream istringstream ostringstream ' +
	      'auto_ptr deque list queue stack vector map set pair bitset multiset multimap unordered_set ' +
	      'unordered_map unordered_multiset unordered_multimap priority_queue make_pair array shared_ptr abort terminate abs acos ' +
	      'asin atan2 atan calloc ceil cosh cos exit exp fabs floor fmod fprintf fputs free frexp ' +
	      'fscanf future isalnum isalpha iscntrl isdigit isgraph islower isprint ispunct isspace isupper ' +
	      'isxdigit tolower toupper labs ldexp log10 log malloc realloc memchr memcmp memcpy memset modf pow ' +
	      'printf putchar puts scanf sinh sin snprintf sprintf sqrt sscanf strcat strchr strcmp ' +
	      'strcpy strcspn strlen strncat strncmp strncpy strpbrk strrchr strspn strstr tanh tan ' +
	      'vfprintf vprintf vsprintf endl initializer_list unique_ptr _Bool complex _Complex imaginary _Imaginary',
	    literal: 'true false nullptr NULL'
	  };

	  const EXPRESSION_CONTAINS = [
	    PREPROCESSOR,
	    CPP_PRIMITIVE_TYPES,
	    C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    NUMBERS,
	    STRINGS
	  ];

	  const EXPRESSION_CONTEXT = {
	    // This mode covers expression context where we can't expect a function
	    // definition and shouldn't highlight anything that looks like one:
	    // `return some()`, `else if()`, `(x*sum(1, 2))`
	    variants: [
	      {
	        begin: /=/,
	        end: /;/
	      },
	      {
	        begin: /\(/,
	        end: /\)/
	      },
	      {
	        beginKeywords: 'new throw return else',
	        end: /;/
	      }
	    ],
	    keywords: CPP_KEYWORDS,
	    contains: EXPRESSION_CONTAINS.concat([
	      {
	        begin: /\(/,
	        end: /\)/,
	        keywords: CPP_KEYWORDS,
	        contains: EXPRESSION_CONTAINS.concat([ 'self' ]),
	        relevance: 0
	      }
	    ]),
	    relevance: 0
	  };

	  const FUNCTION_DECLARATION = {
	    className: 'function',
	    begin: '(' + FUNCTION_TYPE_RE + '[\\*&\\s]+)+' + FUNCTION_TITLE,
	    returnBegin: true,
	    end: /[{;=]/,
	    excludeEnd: true,
	    keywords: CPP_KEYWORDS,
	    illegal: /[^\w\s\*&:<>.]/,
	    contains: [
	      { // to prevent it from being confused as the function title
	        begin: DECLTYPE_AUTO_RE,
	        keywords: CPP_KEYWORDS,
	        relevance: 0
	      },
	      {
	        begin: FUNCTION_TITLE,
	        returnBegin: true,
	        contains: [ TITLE_MODE ],
	        relevance: 0
	      },
	      {
	        className: 'params',
	        begin: /\(/,
	        end: /\)/,
	        keywords: CPP_KEYWORDS,
	        relevance: 0,
	        contains: [
	          C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE,
	          STRINGS,
	          NUMBERS,
	          CPP_PRIMITIVE_TYPES,
	          // Count matching parentheses.
	          {
	            begin: /\(/,
	            end: /\)/,
	            keywords: CPP_KEYWORDS,
	            relevance: 0,
	            contains: [
	              'self',
	              C_LINE_COMMENT_MODE,
	              hljs.C_BLOCK_COMMENT_MODE,
	              STRINGS,
	              NUMBERS,
	              CPP_PRIMITIVE_TYPES
	            ]
	          }
	        ]
	      },
	      CPP_PRIMITIVE_TYPES,
	      C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      PREPROCESSOR
	    ]
	  };

	  return {
	    name: "C",
	    aliases: [
	      'h'
	    ],
	    keywords: CPP_KEYWORDS,
	    // Until differentiations are added between `c` and `cpp`, `c` will
	    // not be auto-detected to avoid auto-detect conflicts between C and C++
	    disableAutodetect: true,
	    illegal: '</',
	    contains: [].concat(
	      EXPRESSION_CONTEXT,
	      FUNCTION_DECLARATION,
	      EXPRESSION_CONTAINS,
	      [
	        PREPROCESSOR,
	        { // containers: ie, `vector <int> rooms (9);`
	          begin: '\\b(deque|list|queue|priority_queue|pair|stack|vector|map|set|bitset|multiset|multimap|unordered_map|unordered_set|unordered_multiset|unordered_multimap|array)\\s*<',
	          end: '>',
	          keywords: CPP_KEYWORDS,
	          contains: [
	            'self',
	            CPP_PRIMITIVE_TYPES
	          ]
	        },
	        {
	          begin: hljs.IDENT_RE + '::',
	          keywords: CPP_KEYWORDS
	        },
	        {
	          className: 'class',
	          beginKeywords: 'enum class struct union',
	          end: /[{;:<>=]/,
	          contains: [
	            {
	              beginKeywords: "final class struct"
	            },
	            hljs.TITLE_MODE
	          ]
	        }
	      ]),
	    exports: {
	      preprocessor: PREPROCESSOR,
	      strings: STRINGS,
	      keywords: CPP_KEYWORDS
	    }
	  };
	}

	c_1 = c;
	return c_1;
}

/*
Language: C/AL
Author: Kenneth Fuglsang Christensen <kfuglsang@gmail.com>
Description: Provides highlighting of Microsoft Dynamics NAV C/AL code files
Website: https://docs.microsoft.com/en-us/dynamics-nav/programming-in-c-al
*/

var cal_1;
var hasRequiredCal;

function requireCal () {
	if (hasRequiredCal) return cal_1;
	hasRequiredCal = 1;
	/** @type LanguageFn */
	function cal(hljs) {
	  const KEYWORDS =
	    'div mod in and or not xor asserterror begin case do downto else end exit for if of repeat then to ' +
	    'until while with var';
	  const LITERALS = 'false true';
	  const COMMENT_MODES = [
	    hljs.C_LINE_COMMENT_MODE,
	    hljs.COMMENT(
	      /\{/,
	      /\}/,
	      {
	        relevance: 0
	      }
	    ),
	    hljs.COMMENT(
	      /\(\*/,
	      /\*\)/,
	      {
	        relevance: 10
	      }
	    )
	  ];
	  const STRING = {
	    className: 'string',
	    begin: /'/,
	    end: /'/,
	    contains: [{
	      begin: /''/
	    }]
	  };
	  const CHAR_STRING = {
	    className: 'string',
	    begin: /(#\d+)+/
	  };
	  const DATE = {
	    className: 'number',
	    begin: '\\b\\d+(\\.\\d+)?(DT|D|T)',
	    relevance: 0
	  };
	  const DBL_QUOTED_VARIABLE = {
	    className: 'string', // not a string technically but makes sense to be highlighted in the same style
	    begin: '"',
	    end: '"'
	  };

	  const PROCEDURE = {
	    className: 'function',
	    beginKeywords: 'procedure',
	    end: /[:;]/,
	    keywords: 'procedure|10',
	    contains: [
	      hljs.TITLE_MODE,
	      {
	        className: 'params',
	        begin: /\(/,
	        end: /\)/,
	        keywords: KEYWORDS,
	        contains: [
	          STRING,
	          CHAR_STRING
	        ]
	      }
	    ].concat(COMMENT_MODES)
	  };

	  const OBJECT = {
	    className: 'class',
	    begin: 'OBJECT (Table|Form|Report|Dataport|Codeunit|XMLport|MenuSuite|Page|Query) (\\d+) ([^\\r\\n]+)',
	    returnBegin: true,
	    contains: [
	      hljs.TITLE_MODE,
	      PROCEDURE
	    ]
	  };

	  return {
	    name: 'C/AL',
	    case_insensitive: true,
	    keywords: {
	      keyword: KEYWORDS,
	      literal: LITERALS
	    },
	    illegal: /\/\*/,
	    contains: [
	      STRING,
	      CHAR_STRING,
	      DATE,
	      DBL_QUOTED_VARIABLE,
	      hljs.NUMBER_MODE,
	      OBJECT,
	      PROCEDURE
	    ]
	  };
	}

	cal_1 = cal;
	return cal_1;
}

/*
Language: Cap’n Proto
Author: Oleg Efimov <efimovov@gmail.com>
Description: Cap’n Proto message definition format
Website: https://capnproto.org/capnp-tool.html
Category: protocols
*/

var capnproto_1;
var hasRequiredCapnproto;

function requireCapnproto () {
	if (hasRequiredCapnproto) return capnproto_1;
	hasRequiredCapnproto = 1;
	/** @type LanguageFn */
	function capnproto(hljs) {
	  return {
	    name: 'Cap’n Proto',
	    aliases: ['capnp'],
	    keywords: {
	      keyword:
	        'struct enum interface union group import using const annotation extends in of on as with from fixed',
	      built_in:
	        'Void Bool Int8 Int16 Int32 Int64 UInt8 UInt16 UInt32 UInt64 Float32 Float64 ' +
	        'Text Data AnyPointer AnyStruct Capability List',
	      literal:
	        'true false'
	    },
	    contains: [
	      hljs.QUOTE_STRING_MODE,
	      hljs.NUMBER_MODE,
	      hljs.HASH_COMMENT_MODE,
	      {
	        className: 'meta',
	        begin: /@0x[\w\d]{16};/,
	        illegal: /\n/
	      },
	      {
	        className: 'symbol',
	        begin: /@\d+\b/
	      },
	      {
	        className: 'class',
	        beginKeywords: 'struct enum',
	        end: /\{/,
	        illegal: /\n/,
	        contains: [hljs.inherit(hljs.TITLE_MODE, {
	          starts: {
	            endsWithParent: true,
	            excludeEnd: true
	          } // hack: eating everything after the first title
	        })]
	      },
	      {
	        className: 'class',
	        beginKeywords: 'interface',
	        end: /\{/,
	        illegal: /\n/,
	        contains: [hljs.inherit(hljs.TITLE_MODE, {
	          starts: {
	            endsWithParent: true,
	            excludeEnd: true
	          } // hack: eating everything after the first title
	        })]
	      }
	    ]
	  };
	}

	capnproto_1 = capnproto;
	return capnproto_1;
}

/*
Language: Ceylon
Author: Lucas Werkmeister <mail@lucaswerkmeister.de>
Website: https://ceylon-lang.org
*/

var ceylon_1;
var hasRequiredCeylon;

function requireCeylon () {
	if (hasRequiredCeylon) return ceylon_1;
	hasRequiredCeylon = 1;
	/** @type LanguageFn */
	function ceylon(hljs) {
	  // 2.3. Identifiers and keywords
	  const KEYWORDS =
	    'assembly module package import alias class interface object given value ' +
	    'assign void function new of extends satisfies abstracts in out return ' +
	    'break continue throw assert dynamic if else switch case for while try ' +
	    'catch finally then let this outer super is exists nonempty';
	  // 7.4.1 Declaration Modifiers
	  const DECLARATION_MODIFIERS =
	    'shared abstract formal default actual variable late native deprecated ' +
	    'final sealed annotation suppressWarnings small';
	  // 7.4.2 Documentation
	  const DOCUMENTATION =
	    'doc by license see throws tagged';
	  const SUBST = {
	    className: 'subst',
	    excludeBegin: true,
	    excludeEnd: true,
	    begin: /``/,
	    end: /``/,
	    keywords: KEYWORDS,
	    relevance: 10
	  };
	  const EXPRESSIONS = [
	    {
	      // verbatim string
	      className: 'string',
	      begin: '"""',
	      end: '"""',
	      relevance: 10
	    },
	    {
	      // string literal or template
	      className: 'string',
	      begin: '"',
	      end: '"',
	      contains: [SUBST]
	    },
	    {
	      // character literal
	      className: 'string',
	      begin: "'",
	      end: "'"
	    },
	    {
	      // numeric literal
	      className: 'number',
	      begin: '#[0-9a-fA-F_]+|\\$[01_]+|[0-9_]+(?:\\.[0-9_](?:[eE][+-]?\\d+)?)?[kMGTPmunpf]?',
	      relevance: 0
	    }
	  ];
	  SUBST.contains = EXPRESSIONS;

	  return {
	    name: 'Ceylon',
	    keywords: {
	      keyword: KEYWORDS + ' ' + DECLARATION_MODIFIERS,
	      meta: DOCUMENTATION
	    },
	    illegal: '\\$[^01]|#[^0-9a-fA-F]',
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.COMMENT('/\\*', '\\*/', {
	        contains: ['self']
	      }),
	      {
	        // compiler annotation
	        className: 'meta',
	        begin: '@[a-z]\\w*(?::"[^"]*")?'
	      }
	    ].concat(EXPRESSIONS)
	  };
	}

	ceylon_1 = ceylon;
	return ceylon_1;
}

/*
Language: Clean
Author: Camil Staps <info@camilstaps.nl>
Category: functional
Website: http://clean.cs.ru.nl
*/

var clean_1;
var hasRequiredClean;

function requireClean () {
	if (hasRequiredClean) return clean_1;
	hasRequiredClean = 1;
	/** @type LanguageFn */
	function clean(hljs) {
	  return {
	    name: 'Clean',
	    aliases: [
	      'icl',
	      'dcl'
	    ],
	    keywords: {
	      keyword:
	        'if let in with where case of class instance otherwise ' +
	        'implementation definition system module from import qualified as ' +
	        'special code inline foreign export ccall stdcall generic derive ' +
	        'infix infixl infixr',
	      built_in:
	        'Int Real Char Bool',
	      literal:
	        'True False'
	    },
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.C_NUMBER_MODE,
	      { // relevance booster
	        begin: '->|<-[|:]?|#!?|>>=|\\{\\||\\|\\}|:==|=:|<>'
	      }
	    ]
	  };
	}

	clean_1 = clean;
	return clean_1;
}

/*
Language: Clojure
Description: Clojure syntax (based on lisp.js)
Author: mfornos
Website: https://clojure.org
Category: lisp
*/

var clojure_1;
var hasRequiredClojure;

function requireClojure () {
	if (hasRequiredClojure) return clojure_1;
	hasRequiredClojure = 1;
	/** @type LanguageFn */
	function clojure(hljs) {
	  const SYMBOLSTART = 'a-zA-Z_\\-!.?+*=<>&#\'';
	  const SYMBOL_RE = '[' + SYMBOLSTART + '][' + SYMBOLSTART + '0-9/;:]*';
	  const globals = 'def defonce defprotocol defstruct defmulti defmethod defn- defn defmacro deftype defrecord';
	  const keywords = {
	    $pattern: SYMBOL_RE,
	    'builtin-name':
	      // Clojure keywords
	      globals + ' ' +
	      'cond apply if-not if-let if not not= =|0 <|0 >|0 <=|0 >=|0 ==|0 +|0 /|0 *|0 -|0 rem ' +
	      'quot neg? pos? delay? symbol? keyword? true? false? integer? empty? coll? list? ' +
	      'set? ifn? fn? associative? sequential? sorted? counted? reversible? number? decimal? ' +
	      'class? distinct? isa? float? rational? reduced? ratio? odd? even? char? seq? vector? ' +
	      'string? map? nil? contains? zero? instance? not-every? not-any? libspec? -> ->> .. . ' +
	      'inc compare do dotimes mapcat take remove take-while drop letfn drop-last take-last ' +
	      'drop-while while intern condp case reduced cycle split-at split-with repeat replicate ' +
	      'iterate range merge zipmap declare line-seq sort comparator sort-by dorun doall nthnext ' +
	      'nthrest partition eval doseq await await-for let agent atom send send-off release-pending-sends ' +
	      'add-watch mapv filterv remove-watch agent-error restart-agent set-error-handler error-handler ' +
	      'set-error-mode! error-mode shutdown-agents quote var fn loop recur throw try monitor-enter ' +
	      'monitor-exit macroexpand macroexpand-1 for dosync and or ' +
	      'when when-not when-let comp juxt partial sequence memoize constantly complement identity assert ' +
	      'peek pop doto proxy first rest cons cast coll last butlast ' +
	      'sigs reify second ffirst fnext nfirst nnext meta with-meta ns in-ns create-ns import ' +
	      'refer keys select-keys vals key val rseq name namespace promise into transient persistent! conj! ' +
	      'assoc! dissoc! pop! disj! use class type num float double short byte boolean bigint biginteger ' +
	      'bigdec print-method print-dup throw-if printf format load compile get-in update-in pr pr-on newline ' +
	      'flush read slurp read-line subvec with-open memfn time re-find re-groups rand-int rand mod locking ' +
	      'assert-valid-fdecl alias resolve ref deref refset swap! reset! set-validator! compare-and-set! alter-meta! ' +
	      'reset-meta! commute get-validator alter ref-set ref-history-count ref-min-history ref-max-history ensure sync io! ' +
	      'new next conj set! to-array future future-call into-array aset gen-class reduce map filter find empty ' +
	      'hash-map hash-set sorted-map sorted-map-by sorted-set sorted-set-by vec vector seq flatten reverse assoc dissoc list ' +
	      'disj get union difference intersection extend extend-type extend-protocol int nth delay count concat chunk chunk-buffer ' +
	      'chunk-append chunk-first chunk-rest max min dec unchecked-inc-int unchecked-inc unchecked-dec-inc unchecked-dec unchecked-negate ' +
	      'unchecked-add-int unchecked-add unchecked-subtract-int unchecked-subtract chunk-next chunk-cons chunked-seq? prn vary-meta ' +
	      'lazy-seq spread list* str find-keyword keyword symbol gensym force rationalize'
	  };

	  const SIMPLE_NUMBER_RE = '[-+]?\\d+(\\.\\d+)?';

	  const SYMBOL = {
	    begin: SYMBOL_RE,
	    relevance: 0
	  };
	  const NUMBER = {
	    className: 'number',
	    begin: SIMPLE_NUMBER_RE,
	    relevance: 0
	  };
	  const STRING = hljs.inherit(hljs.QUOTE_STRING_MODE, {
	    illegal: null
	  });
	  const COMMENT = hljs.COMMENT(
	    ';',
	    '$',
	    {
	      relevance: 0
	    }
	  );
	  const LITERAL = {
	    className: 'literal',
	    begin: /\b(true|false|nil)\b/
	  };
	  const COLLECTION = {
	    begin: '[\\[\\{]',
	    end: '[\\]\\}]'
	  };
	  const HINT = {
	    className: 'comment',
	    begin: '\\^' + SYMBOL_RE
	  };
	  const HINT_COL = hljs.COMMENT('\\^\\{', '\\}');
	  const KEY = {
	    className: 'symbol',
	    begin: '[:]{1,2}' + SYMBOL_RE
	  };
	  const LIST = {
	    begin: '\\(',
	    end: '\\)'
	  };
	  const BODY = {
	    endsWithParent: true,
	    relevance: 0
	  };
	  const NAME = {
	    keywords: keywords,
	    className: 'name',
	    begin: SYMBOL_RE,
	    relevance: 0,
	    starts: BODY
	  };
	  const DEFAULT_CONTAINS = [
	    LIST,
	    STRING,
	    HINT,
	    HINT_COL,
	    COMMENT,
	    KEY,
	    COLLECTION,
	    NUMBER,
	    LITERAL,
	    SYMBOL
	  ];

	  const GLOBAL = {
	    beginKeywords: globals,
	    lexemes: SYMBOL_RE,
	    end: '(\\[|#|\\d|"|:|\\{|\\)|\\(|$)',
	    contains: [
	      {
	        className: 'title',
	        begin: SYMBOL_RE,
	        relevance: 0,
	        excludeEnd: true,
	        // we can only have a single title
	        endsParent: true
	      }
	    ].concat(DEFAULT_CONTAINS)
	  };

	  LIST.contains = [
	    hljs.COMMENT('comment', ''),
	    GLOBAL,
	    NAME,
	    BODY
	  ];
	  BODY.contains = DEFAULT_CONTAINS;
	  COLLECTION.contains = DEFAULT_CONTAINS;
	  HINT_COL.contains = [ COLLECTION ];

	  return {
	    name: 'Clojure',
	    aliases: [ 'clj' ],
	    illegal: /\S/,
	    contains: [
	      LIST,
	      STRING,
	      HINT,
	      HINT_COL,
	      COMMENT,
	      KEY,
	      COLLECTION,
	      NUMBER,
	      LITERAL
	    ]
	  };
	}

	clojure_1 = clojure;
	return clojure_1;
}

/*
Language: Clojure REPL
Description: Clojure REPL sessions
Author: Ivan Sagalaev <maniac@softwaremaniacs.org>
Requires: clojure.js
Website: https://clojure.org
Category: lisp
*/

var clojureRepl_1;
var hasRequiredClojureRepl;

function requireClojureRepl () {
	if (hasRequiredClojureRepl) return clojureRepl_1;
	hasRequiredClojureRepl = 1;
	/** @type LanguageFn */
	function clojureRepl(hljs) {
	  return {
	    name: 'Clojure REPL',
	    contains: [
	      {
	        className: 'meta',
	        begin: /^([\w.-]+|\s*#_)?=>/,
	        starts: {
	          end: /$/,
	          subLanguage: 'clojure'
	        }
	      }
	    ]
	  };
	}

	clojureRepl_1 = clojureRepl;
	return clojureRepl_1;
}

/*
Language: CMake
Description: CMake is an open-source cross-platform system for build automation.
Author: Igor Kalnitsky <igor@kalnitsky.org>
Website: https://cmake.org
*/

var cmake_1;
var hasRequiredCmake;

function requireCmake () {
	if (hasRequiredCmake) return cmake_1;
	hasRequiredCmake = 1;
	/** @type LanguageFn */
	function cmake(hljs) {
	  return {
	    name: 'CMake',
	    aliases: ['cmake.in'],
	    case_insensitive: true,
	    keywords: {
	      keyword:
	        // scripting commands
	        'break cmake_host_system_information cmake_minimum_required cmake_parse_arguments ' +
	        'cmake_policy configure_file continue elseif else endforeach endfunction endif endmacro ' +
	        'endwhile execute_process file find_file find_library find_package find_path ' +
	        'find_program foreach function get_cmake_property get_directory_property ' +
	        'get_filename_component get_property if include include_guard list macro ' +
	        'mark_as_advanced math message option return separate_arguments ' +
	        'set_directory_properties set_property set site_name string unset variable_watch while ' +
	        // project commands
	        'add_compile_definitions add_compile_options add_custom_command add_custom_target ' +
	        'add_definitions add_dependencies add_executable add_library add_link_options ' +
	        'add_subdirectory add_test aux_source_directory build_command create_test_sourcelist ' +
	        'define_property enable_language enable_testing export fltk_wrap_ui ' +
	        'get_source_file_property get_target_property get_test_property include_directories ' +
	        'include_external_msproject include_regular_expression install link_directories ' +
	        'link_libraries load_cache project qt_wrap_cpp qt_wrap_ui remove_definitions ' +
	        'set_source_files_properties set_target_properties set_tests_properties source_group ' +
	        'target_compile_definitions target_compile_features target_compile_options ' +
	        'target_include_directories target_link_directories target_link_libraries ' +
	        'target_link_options target_sources try_compile try_run ' +
	        // CTest commands
	        'ctest_build ctest_configure ctest_coverage ctest_empty_binary_directory ctest_memcheck ' +
	        'ctest_read_custom_files ctest_run_script ctest_sleep ctest_start ctest_submit ' +
	        'ctest_test ctest_update ctest_upload ' +
	        // deprecated commands
	        'build_name exec_program export_library_dependencies install_files install_programs ' +
	        'install_targets load_command make_directory output_required_files remove ' +
	        'subdir_depends subdirs use_mangled_mesa utility_source variable_requires write_file ' +
	        'qt5_use_modules qt5_use_package qt5_wrap_cpp ' +
	        // core keywords
	        'on off true false and or not command policy target test exists is_newer_than ' +
	        'is_directory is_symlink is_absolute matches less greater equal less_equal ' +
	        'greater_equal strless strgreater strequal strless_equal strgreater_equal version_less ' +
	        'version_greater version_equal version_less_equal version_greater_equal in_list defined'
	    },
	    contains: [
	      {
	        className: 'variable',
	        begin: /\$\{/,
	        end: /\}/
	      },
	      hljs.HASH_COMMENT_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.NUMBER_MODE
	    ]
	  };
	}

	cmake_1 = cmake;
	return cmake_1;
}

var coffeescript_1;
var hasRequiredCoffeescript;

function requireCoffeescript () {
	if (hasRequiredCoffeescript) return coffeescript_1;
	hasRequiredCoffeescript = 1;
	const KEYWORDS = [
	  "as", // for exports
	  "in",
	  "of",
	  "if",
	  "for",
	  "while",
	  "finally",
	  "var",
	  "new",
	  "function",
	  "do",
	  "return",
	  "void",
	  "else",
	  "break",
	  "catch",
	  "instanceof",
	  "with",
	  "throw",
	  "case",
	  "default",
	  "try",
	  "switch",
	  "continue",
	  "typeof",
	  "delete",
	  "let",
	  "yield",
	  "const",
	  "class",
	  // JS handles these with a special rule
	  // "get",
	  // "set",
	  "debugger",
	  "async",
	  "await",
	  "static",
	  "import",
	  "from",
	  "export",
	  "extends"
	];
	const LITERALS = [
	  "true",
	  "false",
	  "null",
	  "undefined",
	  "NaN",
	  "Infinity"
	];

	const TYPES = [
	  "Intl",
	  "DataView",
	  "Number",
	  "Math",
	  "Date",
	  "String",
	  "RegExp",
	  "Object",
	  "Function",
	  "Boolean",
	  "Error",
	  "Symbol",
	  "Set",
	  "Map",
	  "WeakSet",
	  "WeakMap",
	  "Proxy",
	  "Reflect",
	  "JSON",
	  "Promise",
	  "Float64Array",
	  "Int16Array",
	  "Int32Array",
	  "Int8Array",
	  "Uint16Array",
	  "Uint32Array",
	  "Float32Array",
	  "Array",
	  "Uint8Array",
	  "Uint8ClampedArray",
	  "ArrayBuffer",
	  "BigInt64Array",
	  "BigUint64Array",
	  "BigInt"
	];

	const ERROR_TYPES = [
	  "EvalError",
	  "InternalError",
	  "RangeError",
	  "ReferenceError",
	  "SyntaxError",
	  "TypeError",
	  "URIError"
	];

	const BUILT_IN_GLOBALS = [
	  "setInterval",
	  "setTimeout",
	  "clearInterval",
	  "clearTimeout",

	  "require",
	  "exports",

	  "eval",
	  "isFinite",
	  "isNaN",
	  "parseFloat",
	  "parseInt",
	  "decodeURI",
	  "decodeURIComponent",
	  "encodeURI",
	  "encodeURIComponent",
	  "escape",
	  "unescape"
	];

	const BUILT_IN_VARIABLES = [
	  "arguments",
	  "this",
	  "super",
	  "console",
	  "window",
	  "document",
	  "localStorage",
	  "module",
	  "global" // Node.js
	];

	const BUILT_INS = [].concat(
	  BUILT_IN_GLOBALS,
	  BUILT_IN_VARIABLES,
	  TYPES,
	  ERROR_TYPES
	);

	/*
	Language: CoffeeScript
	Author: Dmytrii Nagirniak <dnagir@gmail.com>
	Contributors: Oleg Efimov <efimovov@gmail.com>, Cédric Néhémie <cedric.nehemie@gmail.com>
	Description: CoffeeScript is a programming language that transcompiles to JavaScript. For info about language see http://coffeescript.org/
	Category: common, scripting
	Website: https://coffeescript.org
	*/

	/** @type LanguageFn */
	function coffeescript(hljs) {
	  const COFFEE_BUILT_INS = [
	    'npm',
	    'print'
	  ];
	  const COFFEE_LITERALS = [
	    'yes',
	    'no',
	    'on',
	    'off'
	  ];
	  const COFFEE_KEYWORDS = [
	    'then',
	    'unless',
	    'until',
	    'loop',
	    'by',
	    'when',
	    'and',
	    'or',
	    'is',
	    'isnt',
	    'not'
	  ];
	  const NOT_VALID_KEYWORDS = [
	    "var",
	    "const",
	    "let",
	    "function",
	    "static"
	  ];
	  const excluding = (list) =>
	    (kw) => !list.includes(kw);
	  const KEYWORDS$1 = {
	    keyword: KEYWORDS.concat(COFFEE_KEYWORDS).filter(excluding(NOT_VALID_KEYWORDS)),
	    literal: LITERALS.concat(COFFEE_LITERALS),
	    built_in: BUILT_INS.concat(COFFEE_BUILT_INS)
	  };
	  const JS_IDENT_RE = '[A-Za-z$_][0-9A-Za-z$_]*';
	  const SUBST = {
	    className: 'subst',
	    begin: /#\{/,
	    end: /\}/,
	    keywords: KEYWORDS$1
	  };
	  const EXPRESSIONS = [
	    hljs.BINARY_NUMBER_MODE,
	    hljs.inherit(hljs.C_NUMBER_MODE, {
	      starts: {
	        end: '(\\s*/)?',
	        relevance: 0
	      }
	    }), // a number tries to eat the following slash to prevent treating it as a regexp
	    {
	      className: 'string',
	      variants: [
	        {
	          begin: /'''/,
	          end: /'''/,
	          contains: [hljs.BACKSLASH_ESCAPE]
	        },
	        {
	          begin: /'/,
	          end: /'/,
	          contains: [hljs.BACKSLASH_ESCAPE]
	        },
	        {
	          begin: /"""/,
	          end: /"""/,
	          contains: [
	            hljs.BACKSLASH_ESCAPE,
	            SUBST
	          ]
	        },
	        {
	          begin: /"/,
	          end: /"/,
	          contains: [
	            hljs.BACKSLASH_ESCAPE,
	            SUBST
	          ]
	        }
	      ]
	    },
	    {
	      className: 'regexp',
	      variants: [
	        {
	          begin: '///',
	          end: '///',
	          contains: [
	            SUBST,
	            hljs.HASH_COMMENT_MODE
	          ]
	        },
	        {
	          begin: '//[gim]{0,3}(?=\\W)',
	          relevance: 0
	        },
	        {
	          // regex can't start with space to parse x / 2 / 3 as two divisions
	          // regex can't start with *, and it supports an "illegal" in the main mode
	          begin: /\/(?![ *]).*?(?![\\]).\/[gim]{0,3}(?=\W)/
	        }
	      ]
	    },
	    {
	      begin: '@' + JS_IDENT_RE // relevance booster
	    },
	    {
	      subLanguage: 'javascript',
	      excludeBegin: true,
	      excludeEnd: true,
	      variants: [
	        {
	          begin: '```',
	          end: '```'
	        },
	        {
	          begin: '`',
	          end: '`'
	        }
	      ]
	    }
	  ];
	  SUBST.contains = EXPRESSIONS;

	  const TITLE = hljs.inherit(hljs.TITLE_MODE, {
	    begin: JS_IDENT_RE
	  });
	  const POSSIBLE_PARAMS_RE = '(\\(.*\\)\\s*)?\\B[-=]>';
	  const PARAMS = {
	    className: 'params',
	    begin: '\\([^\\(]',
	    returnBegin: true,
	    /* We need another contained nameless mode to not have every nested
	    pair of parens to be called "params" */
	    contains: [{
	      begin: /\(/,
	      end: /\)/,
	      keywords: KEYWORDS$1,
	      contains: ['self'].concat(EXPRESSIONS)
	    }]
	  };

	  return {
	    name: 'CoffeeScript',
	    aliases: [
	      'coffee',
	      'cson',
	      'iced'
	    ],
	    keywords: KEYWORDS$1,
	    illegal: /\/\*/,
	    contains: EXPRESSIONS.concat([
	      hljs.COMMENT('###', '###'),
	      hljs.HASH_COMMENT_MODE,
	      {
	        className: 'function',
	        begin: '^\\s*' + JS_IDENT_RE + '\\s*=\\s*' + POSSIBLE_PARAMS_RE,
	        end: '[-=]>',
	        returnBegin: true,
	        contains: [
	          TITLE,
	          PARAMS
	        ]
	      },
	      {
	        // anonymous function start
	        begin: /[:\(,=]\s*/,
	        relevance: 0,
	        contains: [{
	          className: 'function',
	          begin: POSSIBLE_PARAMS_RE,
	          end: '[-=]>',
	          returnBegin: true,
	          contains: [PARAMS]
	        }]
	      },
	      {
	        className: 'class',
	        beginKeywords: 'class',
	        end: '$',
	        illegal: /[:="\[\]]/,
	        contains: [
	          {
	            beginKeywords: 'extends',
	            endsWithParent: true,
	            illegal: /[:="\[\]]/,
	            contains: [TITLE]
	          },
	          TITLE
	        ]
	      },
	      {
	        begin: JS_IDENT_RE + ':',
	        end: ':',
	        returnBegin: true,
	        returnEnd: true,
	        relevance: 0
	      }
	    ])
	  };
	}

	coffeescript_1 = coffeescript;
	return coffeescript_1;
}

/*
Language: Coq
Author: Stephan Boyer <stephan@stephanboyer.com>
Category: functional
Website: https://coq.inria.fr
*/

var coq_1;
var hasRequiredCoq;

function requireCoq () {
	if (hasRequiredCoq) return coq_1;
	hasRequiredCoq = 1;
	/** @type LanguageFn */
	function coq(hljs) {
	  return {
	    name: 'Coq',
	    keywords: {
	      keyword:
	        '_|0 as at cofix else end exists exists2 fix for forall fun if IF in let ' +
	        'match mod Prop return Set then Type using where with ' +
	        'Abort About Add Admit Admitted All Arguments Assumptions Axiom Back BackTo ' +
	        'Backtrack Bind Blacklist Canonical Cd Check Class Classes Close Coercion ' +
	        'Coercions CoFixpoint CoInductive Collection Combined Compute Conjecture ' +
	        'Conjectures Constant constr Constraint Constructors Context Corollary ' +
	        'CreateHintDb Cut Declare Defined Definition Delimit Dependencies Dependent ' +
	        'Derive Drop eauto End Equality Eval Example Existential Existentials ' +
	        'Existing Export exporting Extern Extract Extraction Fact Field Fields File ' +
	        'Fixpoint Focus for From Function Functional Generalizable Global Goal Grab ' +
	        'Grammar Graph Guarded Heap Hint HintDb Hints Hypotheses Hypothesis ident ' +
	        'Identity If Immediate Implicit Import Include Inductive Infix Info Initial ' +
	        'Inline Inspect Instance Instances Intro Intros Inversion Inversion_clear ' +
	        'Language Left Lemma Let Libraries Library Load LoadPath Local Locate Ltac ML ' +
	        'Mode Module Modules Monomorphic Morphism Next NoInline Notation Obligation ' +
	        'Obligations Opaque Open Optimize Options Parameter Parameters Parametric ' +
	        'Path Paths pattern Polymorphic Preterm Print Printing Program Projections ' +
	        'Proof Proposition Pwd Qed Quit Rec Record Recursive Redirect Relation Remark ' +
	        'Remove Require Reserved Reset Resolve Restart Rewrite Right Ring Rings Save ' +
	        'Scheme Scope Scopes Script Search SearchAbout SearchHead SearchPattern ' +
	        'SearchRewrite Section Separate Set Setoid Show Solve Sorted Step Strategies ' +
	        'Strategy Structure SubClass Table Tables Tactic Term Test Theorem Time ' +
	        'Timeout Transparent Type Typeclasses Types Undelimit Undo Unfocus Unfocused ' +
	        'Unfold Universe Universes Unset Unshelve using Variable Variables Variant ' +
	        'Verbose Visibility where with',
	      built_in:
	        'abstract absurd admit after apply as assert assumption at auto autorewrite ' +
	        'autounfold before bottom btauto by case case_eq cbn cbv change ' +
	        'classical_left classical_right clear clearbody cofix compare compute ' +
	        'congruence constr_eq constructor contradict contradiction cut cutrewrite ' +
	        'cycle decide decompose dependent destruct destruction dintuition ' +
	        'discriminate discrR do double dtauto eapply eassumption eauto ecase ' +
	        'econstructor edestruct ediscriminate eelim eexact eexists einduction ' +
	        'einjection eleft elim elimtype enough equality erewrite eright ' +
	        'esimplify_eq esplit evar exact exactly_once exfalso exists f_equal fail ' +
	        'field field_simplify field_simplify_eq first firstorder fix fold fourier ' +
	        'functional generalize generalizing gfail give_up has_evar hnf idtac in ' +
	        'induction injection instantiate intro intro_pattern intros intuition ' +
	        'inversion inversion_clear is_evar is_var lapply lazy left lia lra move ' +
	        'native_compute nia nsatz omega once pattern pose progress proof psatz quote ' +
	        'record red refine reflexivity remember rename repeat replace revert ' +
	        'revgoals rewrite rewrite_strat right ring ring_simplify rtauto set ' +
	        'setoid_reflexivity setoid_replace setoid_rewrite setoid_symmetry ' +
	        'setoid_transitivity shelve shelve_unifiable simpl simple simplify_eq solve ' +
	        'specialize split split_Rabs split_Rmult stepl stepr subst sum swap ' +
	        'symmetry tactic tauto time timeout top transitivity trivial try tryif ' +
	        'unfold unify until using vm_compute with'
	    },
	    contains: [
	      hljs.QUOTE_STRING_MODE,
	      hljs.COMMENT('\\(\\*', '\\*\\)'),
	      hljs.C_NUMBER_MODE,
	      {
	        className: 'type',
	        excludeBegin: true,
	        begin: '\\|\\s*',
	        end: '\\w+'
	      },
	      { // relevance booster
	        begin: /[-=]>/
	      }
	    ]
	  };
	}

	coq_1 = coq;
	return coq_1;
}

/*
Language: Caché Object Script
Author: Nikita Savchenko <zitros.lab@gmail.com>
Category: enterprise, scripting
Website: https://cedocs.intersystems.com/latest/csp/docbook/DocBook.UI.Page.cls
*/

var cos_1;
var hasRequiredCos;

function requireCos () {
	if (hasRequiredCos) return cos_1;
	hasRequiredCos = 1;
	/** @type LanguageFn */
	function cos(hljs) {
	  const STRINGS = {
	    className: 'string',
	    variants: [{
	      begin: '"',
	      end: '"',
	      contains: [{ // escaped
	        begin: "\"\"",
	        relevance: 0
	      }]
	    }]
	  };

	  const NUMBERS = {
	    className: "number",
	    begin: "\\b(\\d+(\\.\\d*)?|\\.\\d+)",
	    relevance: 0
	  };

	  const COS_KEYWORDS =
	    'property parameter class classmethod clientmethod extends as break ' +
	    'catch close continue do d|0 else elseif for goto halt hang h|0 if job ' +
	    'j|0 kill k|0 lock l|0 merge new open quit q|0 read r|0 return set s|0 ' +
	    'tcommit throw trollback try tstart use view while write w|0 xecute x|0 ' +
	    'zkill znspace zn ztrap zwrite zw zzdump zzwrite print zbreak zinsert ' +
	    'zload zprint zremove zsave zzprint mv mvcall mvcrt mvdim mvprint zquit ' +
	    'zsync ascii';

	  // registered function - no need in them due to all functions are highlighted,
	  // but I'll just leave this here.

	  // "$bit", "$bitcount",
	  // "$bitfind", "$bitlogic", "$case", "$char", "$classmethod", "$classname",
	  // "$compile", "$data", "$decimal", "$double", "$extract", "$factor",
	  // "$find", "$fnumber", "$get", "$increment", "$inumber", "$isobject",
	  // "$isvaliddouble", "$isvalidnum", "$justify", "$length", "$list",
	  // "$listbuild", "$listdata", "$listfind", "$listfromstring", "$listget",
	  // "$listlength", "$listnext", "$listsame", "$listtostring", "$listvalid",
	  // "$locate", "$match", "$method", "$name", "$nconvert", "$next",
	  // "$normalize", "$now", "$number", "$order", "$parameter", "$piece",
	  // "$prefetchoff", "$prefetchon", "$property", "$qlength", "$qsubscript",
	  // "$query", "$random", "$replace", "$reverse", "$sconvert", "$select",
	  // "$sortbegin", "$sortend", "$stack", "$text", "$translate", "$view",
	  // "$wascii", "$wchar", "$wextract", "$wfind", "$wiswide", "$wlength",
	  // "$wreverse", "$xecute", "$zabs", "$zarccos", "$zarcsin", "$zarctan",
	  // "$zcos", "$zcot", "$zcsc", "$zdate", "$zdateh", "$zdatetime",
	  // "$zdatetimeh", "$zexp", "$zhex", "$zln", "$zlog", "$zpower", "$zsec",
	  // "$zsin", "$zsqr", "$ztan", "$ztime", "$ztimeh", "$zboolean",
	  // "$zconvert", "$zcrc", "$zcyc", "$zdascii", "$zdchar", "$zf",
	  // "$ziswide", "$zlascii", "$zlchar", "$zname", "$zposition", "$zqascii",
	  // "$zqchar", "$zsearch", "$zseek", "$zstrip", "$zwascii", "$zwchar",
	  // "$zwidth", "$zwpack", "$zwbpack", "$zwunpack", "$zwbunpack", "$zzenkaku",
	  // "$change", "$mv", "$mvat", "$mvfmt", "$mvfmts", "$mviconv",
	  // "$mviconvs", "$mvinmat", "$mvlover", "$mvoconv", "$mvoconvs", "$mvraise",
	  // "$mvtrans", "$mvv", "$mvname", "$zbitand", "$zbitcount", "$zbitfind",
	  // "$zbitget", "$zbitlen", "$zbitnot", "$zbitor", "$zbitset", "$zbitstr",
	  // "$zbitxor", "$zincrement", "$znext", "$zorder", "$zprevious", "$zsort",
	  // "device", "$ecode", "$estack", "$etrap", "$halt", "$horolog",
	  // "$io", "$job", "$key", "$namespace", "$principal", "$quit", "$roles",
	  // "$storage", "$system", "$test", "$this", "$tlevel", "$username",
	  // "$x", "$y", "$za", "$zb", "$zchild", "$zeof", "$zeos", "$zerror",
	  // "$zhorolog", "$zio", "$zjob", "$zmode", "$znspace", "$zparent", "$zpi",
	  // "$zpos", "$zreference", "$zstorage", "$ztimestamp", "$ztimezone",
	  // "$ztrap", "$zversion"

	  return {
	    name: 'Caché Object Script',
	    case_insensitive: true,
	    aliases: [
	      "cls"
	    ],
	    keywords: COS_KEYWORDS,
	    contains: [
	      NUMBERS,
	      STRINGS,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      {
	        className: "comment",
	        begin: /;/,
	        end: "$",
	        relevance: 0
	      },
	      { // Functions and user-defined functions: write $ztime(60*60*3), $$myFunc(10), $$^Val(1)
	        className: "built_in",
	        begin: /(?:\$\$?|\.\.)\^?[a-zA-Z]+/
	      },
	      { // Macro command: quit $$$OK
	        className: "built_in",
	        begin: /\$\$\$[a-zA-Z]+/
	      },
	      { // Special (global) variables: write %request.Content; Built-in classes: %Library.Integer
	        className: "built_in",
	        begin: /%[a-z]+(?:\.[a-z]+)*/
	      },
	      { // Global variable: set ^globalName = 12 write ^globalName
	        className: "symbol",
	        begin: /\^%?[a-zA-Z][\w]*/
	      },
	      { // Some control constructions: do ##class(Package.ClassName).Method(), ##super()
	        className: "keyword",
	        begin: /##class|##super|#define|#dim/
	      },
	      // sub-languages: are not fully supported by hljs by 11/15/2015
	      // left for the future implementation.
	      {
	        begin: /&sql\(/,
	        end: /\)/,
	        excludeBegin: true,
	        excludeEnd: true,
	        subLanguage: "sql"
	      },
	      {
	        begin: /&(js|jscript|javascript)</,
	        end: />/,
	        excludeBegin: true,
	        excludeEnd: true,
	        subLanguage: "javascript"
	      },
	      {
	        // this brakes first and last tag, but this is the only way to embed a valid html
	        begin: /&html<\s*</,
	        end: />\s*>/,
	        subLanguage: "xml"
	      }
	    ]
	  };
	}

	cos_1 = cos;
	return cos_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var cpp_1;
var hasRequiredCpp;

function requireCpp () {
	if (hasRequiredCpp) return cpp_1;
	hasRequiredCpp = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function optional(re) {
	  return concat('(', re, ')?');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: C++
	Category: common, system
	Website: https://isocpp.org
	*/

	/** @type LanguageFn */
	function cpp(hljs) {
	  // added for historic reasons because `hljs.C_LINE_COMMENT_MODE` does
	  // not include such support nor can we be sure all the grammars depending
	  // on it would desire this behavior
	  const C_LINE_COMMENT_MODE = hljs.COMMENT('//', '$', {
	    contains: [
	      {
	        begin: /\\\n/
	      }
	    ]
	  });
	  const DECLTYPE_AUTO_RE = 'decltype\\(auto\\)';
	  const NAMESPACE_RE = '[a-zA-Z_]\\w*::';
	  const TEMPLATE_ARGUMENT_RE = '<[^<>]+>';
	  const FUNCTION_TYPE_RE = '(' +
	    DECLTYPE_AUTO_RE + '|' +
	    optional(NAMESPACE_RE) +
	    '[a-zA-Z_]\\w*' + optional(TEMPLATE_ARGUMENT_RE) +
	  ')';
	  const CPP_PRIMITIVE_TYPES = {
	    className: 'keyword',
	    begin: '\\b[a-z\\d_]*_t\\b'
	  };

	  // https://en.cppreference.com/w/cpp/language/escape
	  // \\ \x \xFF \u2837 \u00323747 \374
	  const CHARACTER_ESCAPES = '\\\\(x[0-9A-Fa-f]{2}|u[0-9A-Fa-f]{4,8}|[0-7]{3}|\\S)';
	  const STRINGS = {
	    className: 'string',
	    variants: [
	      {
	        begin: '(u8?|U|L)?"',
	        end: '"',
	        illegal: '\\n',
	        contains: [ hljs.BACKSLASH_ESCAPE ]
	      },
	      {
	        begin: '(u8?|U|L)?\'(' + CHARACTER_ESCAPES + "|.)",
	        end: '\'',
	        illegal: '.'
	      },
	      hljs.END_SAME_AS_BEGIN({
	        begin: /(?:u8?|U|L)?R"([^()\\ ]{0,16})\(/,
	        end: /\)([^()\\ ]{0,16})"/
	      })
	    ]
	  };

	  const NUMBERS = {
	    className: 'number',
	    variants: [
	      {
	        begin: '\\b(0b[01\']+)'
	      },
	      {
	        begin: '(-?)\\b([\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)((ll|LL|l|L)(u|U)?|(u|U)(ll|LL|l|L)?|f|F|b|B)'
	      },
	      {
	        begin: '(-?)(\\b0[xX][a-fA-F0-9\']+|(\\b[\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)([eE][-+]?[\\d\']+)?)'
	      }
	    ],
	    relevance: 0
	  };

	  const PREPROCESSOR = {
	    className: 'meta',
	    begin: /#\s*[a-z]+\b/,
	    end: /$/,
	    keywords: {
	      'meta-keyword':
	        'if else elif endif define undef warning error line ' +
	        'pragma _Pragma ifdef ifndef include'
	    },
	    contains: [
	      {
	        begin: /\\\n/,
	        relevance: 0
	      },
	      hljs.inherit(STRINGS, {
	        className: 'meta-string'
	      }),
	      {
	        className: 'meta-string',
	        begin: /<.*?>/
	      },
	      C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE
	    ]
	  };

	  const TITLE_MODE = {
	    className: 'title',
	    begin: optional(NAMESPACE_RE) + hljs.IDENT_RE,
	    relevance: 0
	  };

	  const FUNCTION_TITLE = optional(NAMESPACE_RE) + hljs.IDENT_RE + '\\s*\\(';

	  const COMMON_CPP_HINTS = [
	    'asin',
	    'atan2',
	    'atan',
	    'calloc',
	    'ceil',
	    'cosh',
	    'cos',
	    'exit',
	    'exp',
	    'fabs',
	    'floor',
	    'fmod',
	    'fprintf',
	    'fputs',
	    'free',
	    'frexp',
	    'auto_ptr',
	    'deque',
	    'list',
	    'queue',
	    'stack',
	    'vector',
	    'map',
	    'set',
	    'pair',
	    'bitset',
	    'multiset',
	    'multimap',
	    'unordered_set',
	    'fscanf',
	    'future',
	    'isalnum',
	    'isalpha',
	    'iscntrl',
	    'isdigit',
	    'isgraph',
	    'islower',
	    'isprint',
	    'ispunct',
	    'isspace',
	    'isupper',
	    'isxdigit',
	    'tolower',
	    'toupper',
	    'labs',
	    'ldexp',
	    'log10',
	    'log',
	    'malloc',
	    'realloc',
	    'memchr',
	    'memcmp',
	    'memcpy',
	    'memset',
	    'modf',
	    'pow',
	    'printf',
	    'putchar',
	    'puts',
	    'scanf',
	    'sinh',
	    'sin',
	    'snprintf',
	    'sprintf',
	    'sqrt',
	    'sscanf',
	    'strcat',
	    'strchr',
	    'strcmp',
	    'strcpy',
	    'strcspn',
	    'strlen',
	    'strncat',
	    'strncmp',
	    'strncpy',
	    'strpbrk',
	    'strrchr',
	    'strspn',
	    'strstr',
	    'tanh',
	    'tan',
	    'unordered_map',
	    'unordered_multiset',
	    'unordered_multimap',
	    'priority_queue',
	    'make_pair',
	    'array',
	    'shared_ptr',
	    'abort',
	    'terminate',
	    'abs',
	    'acos',
	    'vfprintf',
	    'vprintf',
	    'vsprintf',
	    'endl',
	    'initializer_list',
	    'unique_ptr',
	    'complex',
	    'imaginary',
	    'std',
	    'string',
	    'wstring',
	    'cin',
	    'cout',
	    'cerr',
	    'clog',
	    'stdin',
	    'stdout',
	    'stderr',
	    'stringstream',
	    'istringstream',
	    'ostringstream'
	  ];

	  const CPP_KEYWORDS = {
	    keyword: 'int float while private char char8_t char16_t char32_t catch import module export virtual operator sizeof ' +
	      'dynamic_cast|10 typedef const_cast|10 const for static_cast|10 union namespace ' +
	      'unsigned long volatile static protected bool template mutable if public friend ' +
	      'do goto auto void enum else break extern using asm case typeid wchar_t ' +
	      'short reinterpret_cast|10 default double register explicit signed typename try this ' +
	      'switch continue inline delete alignas alignof constexpr consteval constinit decltype ' +
	      'concept co_await co_return co_yield requires ' +
	      'noexcept static_assert thread_local restrict final override ' +
	      'atomic_bool atomic_char atomic_schar ' +
	      'atomic_uchar atomic_short atomic_ushort atomic_int atomic_uint atomic_long atomic_ulong atomic_llong ' +
	      'atomic_ullong new throw return ' +
	      'and and_eq bitand bitor compl not not_eq or or_eq xor xor_eq',
	    built_in: '_Bool _Complex _Imaginary',
	    _relevance_hints: COMMON_CPP_HINTS,
	    literal: 'true false nullptr NULL'
	  };

	  const FUNCTION_DISPATCH = {
	    className: "function.dispatch",
	    relevance: 0,
	    keywords: CPP_KEYWORDS,
	    begin: concat(
	      /\b/,
	      /(?!decltype)/,
	      /(?!if)/,
	      /(?!for)/,
	      /(?!while)/,
	      hljs.IDENT_RE,
	      lookahead(/\s*\(/))
	  };

	  const EXPRESSION_CONTAINS = [
	    FUNCTION_DISPATCH,
	    PREPROCESSOR,
	    CPP_PRIMITIVE_TYPES,
	    C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    NUMBERS,
	    STRINGS
	  ];


	  const EXPRESSION_CONTEXT = {
	    // This mode covers expression context where we can't expect a function
	    // definition and shouldn't highlight anything that looks like one:
	    // `return some()`, `else if()`, `(x*sum(1, 2))`
	    variants: [
	      {
	        begin: /=/,
	        end: /;/
	      },
	      {
	        begin: /\(/,
	        end: /\)/
	      },
	      {
	        beginKeywords: 'new throw return else',
	        end: /;/
	      }
	    ],
	    keywords: CPP_KEYWORDS,
	    contains: EXPRESSION_CONTAINS.concat([
	      {
	        begin: /\(/,
	        end: /\)/,
	        keywords: CPP_KEYWORDS,
	        contains: EXPRESSION_CONTAINS.concat([ 'self' ]),
	        relevance: 0
	      }
	    ]),
	    relevance: 0
	  };

	  const FUNCTION_DECLARATION = {
	    className: 'function',
	    begin: '(' + FUNCTION_TYPE_RE + '[\\*&\\s]+)+' + FUNCTION_TITLE,
	    returnBegin: true,
	    end: /[{;=]/,
	    excludeEnd: true,
	    keywords: CPP_KEYWORDS,
	    illegal: /[^\w\s\*&:<>.]/,
	    contains: [
	      { // to prevent it from being confused as the function title
	        begin: DECLTYPE_AUTO_RE,
	        keywords: CPP_KEYWORDS,
	        relevance: 0
	      },
	      {
	        begin: FUNCTION_TITLE,
	        returnBegin: true,
	        contains: [ TITLE_MODE ],
	        relevance: 0
	      },
	      // needed because we do not have look-behind on the below rule
	      // to prevent it from grabbing the final : in a :: pair
	      {
	        begin: /::/,
	        relevance: 0
	      },
	      // initializers
	      {
	        begin: /:/,
	        endsWithParent: true,
	        contains: [
	          STRINGS,
	          NUMBERS
	        ]
	      },
	      {
	        className: 'params',
	        begin: /\(/,
	        end: /\)/,
	        keywords: CPP_KEYWORDS,
	        relevance: 0,
	        contains: [
	          C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE,
	          STRINGS,
	          NUMBERS,
	          CPP_PRIMITIVE_TYPES,
	          // Count matching parentheses.
	          {
	            begin: /\(/,
	            end: /\)/,
	            keywords: CPP_KEYWORDS,
	            relevance: 0,
	            contains: [
	              'self',
	              C_LINE_COMMENT_MODE,
	              hljs.C_BLOCK_COMMENT_MODE,
	              STRINGS,
	              NUMBERS,
	              CPP_PRIMITIVE_TYPES
	            ]
	          }
	        ]
	      },
	      CPP_PRIMITIVE_TYPES,
	      C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      PREPROCESSOR
	    ]
	  };

	  return {
	    name: 'C++',
	    aliases: [
	      'cc',
	      'c++',
	      'h++',
	      'hpp',
	      'hh',
	      'hxx',
	      'cxx'
	    ],
	    keywords: CPP_KEYWORDS,
	    illegal: '</',
	    classNameAliases: {
	      "function.dispatch": "built_in"
	    },
	    contains: [].concat(
	      EXPRESSION_CONTEXT,
	      FUNCTION_DECLARATION,
	      FUNCTION_DISPATCH,
	      EXPRESSION_CONTAINS,
	      [
	        PREPROCESSOR,
	        { // containers: ie, `vector <int> rooms (9);`
	          begin: '\\b(deque|list|queue|priority_queue|pair|stack|vector|map|set|bitset|multiset|multimap|unordered_map|unordered_set|unordered_multiset|unordered_multimap|array)\\s*<',
	          end: '>',
	          keywords: CPP_KEYWORDS,
	          contains: [
	            'self',
	            CPP_PRIMITIVE_TYPES
	          ]
	        },
	        {
	          begin: hljs.IDENT_RE + '::',
	          keywords: CPP_KEYWORDS
	        },
	        {
	          className: 'class',
	          beginKeywords: 'enum class struct union',
	          end: /[{;:<>=]/,
	          contains: [
	            {
	              beginKeywords: "final class struct"
	            },
	            hljs.TITLE_MODE
	          ]
	        }
	      ]),
	    exports: {
	      preprocessor: PREPROCESSOR,
	      strings: STRINGS,
	      keywords: CPP_KEYWORDS
	    }
	  };
	}

	cpp_1 = cpp;
	return cpp_1;
}

/*
Language: crmsh
Author: Kristoffer Gronlund <kgronlund@suse.com>
Website: http://crmsh.github.io
Description: Syntax Highlighting for the crmsh DSL
Category: config
*/

var crmsh_1;
var hasRequiredCrmsh;

function requireCrmsh () {
	if (hasRequiredCrmsh) return crmsh_1;
	hasRequiredCrmsh = 1;
	/** @type LanguageFn */
	function crmsh(hljs) {
	  const RESOURCES = 'primitive rsc_template';
	  const COMMANDS = 'group clone ms master location colocation order fencing_topology ' +
	      'rsc_ticket acl_target acl_group user role ' +
	      'tag xml';
	  const PROPERTY_SETS = 'property rsc_defaults op_defaults';
	  const KEYWORDS = 'params meta operations op rule attributes utilization';
	  const OPERATORS = 'read write deny defined not_defined in_range date spec in ' +
	      'ref reference attribute type xpath version and or lt gt tag ' +
	      'lte gte eq ne \\';
	  const TYPES = 'number string';
	  const LITERALS = 'Master Started Slave Stopped start promote demote stop monitor true false';

	  return {
	    name: 'crmsh',
	    aliases: [
	      'crm',
	      'pcmk'
	    ],
	    case_insensitive: true,
	    keywords: {
	      keyword: KEYWORDS + ' ' + OPERATORS + ' ' + TYPES,
	      literal: LITERALS
	    },
	    contains: [
	      hljs.HASH_COMMENT_MODE,
	      {
	        beginKeywords: 'node',
	        starts: {
	          end: '\\s*([\\w_-]+:)?',
	          starts: {
	            className: 'title',
	            end: '\\s*[\\$\\w_][\\w_-]*'
	          }
	        }
	      },
	      {
	        beginKeywords: RESOURCES,
	        starts: {
	          className: 'title',
	          end: '\\s*[\\$\\w_][\\w_-]*',
	          starts: {
	            end: '\\s*@?[\\w_][\\w_\\.:-]*'
	          }
	        }
	      },
	      {
	        begin: '\\b(' + COMMANDS.split(' ').join('|') + ')\\s+',
	        keywords: COMMANDS,
	        starts: {
	          className: 'title',
	          end: '[\\$\\w_][\\w_-]*'
	        }
	      },
	      {
	        beginKeywords: PROPERTY_SETS,
	        starts: {
	          className: 'title',
	          end: '\\s*([\\w_-]+:)?'
	        }
	      },
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'meta',
	        begin: '(ocf|systemd|service|lsb):[\\w_:-]+',
	        relevance: 0
	      },
	      {
	        className: 'number',
	        begin: '\\b\\d+(\\.\\d+)?(ms|s|h|m)?',
	        relevance: 0
	      },
	      {
	        className: 'literal',
	        begin: '[-]?(infinity|inf)',
	        relevance: 0
	      },
	      {
	        className: 'attr',
	        begin: /([A-Za-z$_#][\w_-]+)=/,
	        relevance: 0
	      },
	      {
	        className: 'tag',
	        begin: '</?',
	        end: '/?>',
	        relevance: 0
	      }
	    ]
	  };
	}

	crmsh_1 = crmsh;
	return crmsh_1;
}

/*
Language: Crystal
Author: TSUYUSATO Kitsune <make.just.on@gmail.com>
Website: https://crystal-lang.org
*/

var crystal_1;
var hasRequiredCrystal;

function requireCrystal () {
	if (hasRequiredCrystal) return crystal_1;
	hasRequiredCrystal = 1;
	/** @type LanguageFn */
	function crystal(hljs) {
	  const INT_SUFFIX = '(_?[ui](8|16|32|64|128))?';
	  const FLOAT_SUFFIX = '(_?f(32|64))?';
	  const CRYSTAL_IDENT_RE = '[a-zA-Z_]\\w*[!?=]?';
	  const CRYSTAL_METHOD_RE = '[a-zA-Z_]\\w*[!?=]?|[-+~]@|<<|>>|[=!]~|===?|<=>|[<>]=?|\\*\\*|[-/+%^&*~|]|//|//=|&[-+*]=?|&\\*\\*|\\[\\][=?]?';
	  const CRYSTAL_PATH_RE = '[A-Za-z_]\\w*(::\\w+)*(\\?|!)?';
	  const CRYSTAL_KEYWORDS = {
	    $pattern: CRYSTAL_IDENT_RE,
	    keyword:
	      'abstract alias annotation as as? asm begin break case class def do else elsif end ensure enum extend for fun if ' +
	      'include instance_sizeof is_a? lib macro module next nil? of out pointerof private protected rescue responds_to? ' +
	      'return require select self sizeof struct super then type typeof union uninitialized unless until verbatim when while with yield ' +
	      '__DIR__ __END_LINE__ __FILE__ __LINE__',
	    literal: 'false nil true'
	  };
	  const SUBST = {
	    className: 'subst',
	    begin: /#\{/,
	    end: /\}/,
	    keywords: CRYSTAL_KEYWORDS
	  };
	  const EXPANSION = {
	    className: 'template-variable',
	    variants: [
	      {
	        begin: '\\{\\{',
	        end: '\\}\\}'
	      },
	      {
	        begin: '\\{%',
	        end: '%\\}'
	      }
	    ],
	    keywords: CRYSTAL_KEYWORDS
	  };

	  function recursiveParen(begin, end) {
	    const
	        contains = [
	          {
	            begin: begin,
	            end: end
	          }
	        ];
	    contains[0].contains = contains;
	    return contains;
	  }
	  const STRING = {
	    className: 'string',
	    contains: [
	      hljs.BACKSLASH_ESCAPE,
	      SUBST
	    ],
	    variants: [
	      {
	        begin: /'/,
	        end: /'/
	      },
	      {
	        begin: /"/,
	        end: /"/
	      },
	      {
	        begin: /`/,
	        end: /`/
	      },
	      {
	        begin: '%[Qwi]?\\(',
	        end: '\\)',
	        contains: recursiveParen('\\(', '\\)')
	      },
	      {
	        begin: '%[Qwi]?\\[',
	        end: '\\]',
	        contains: recursiveParen('\\[', '\\]')
	      },
	      {
	        begin: '%[Qwi]?\\{',
	        end: /\}/,
	        contains: recursiveParen(/\{/, /\}/)
	      },
	      {
	        begin: '%[Qwi]?<',
	        end: '>',
	        contains: recursiveParen('<', '>')
	      },
	      {
	        begin: '%[Qwi]?\\|',
	        end: '\\|'
	      },
	      {
	        begin: /<<-\w+$/,
	        end: /^\s*\w+$/
	      }
	    ],
	    relevance: 0
	  };
	  const Q_STRING = {
	    className: 'string',
	    variants: [
	      {
	        begin: '%q\\(',
	        end: '\\)',
	        contains: recursiveParen('\\(', '\\)')
	      },
	      {
	        begin: '%q\\[',
	        end: '\\]',
	        contains: recursiveParen('\\[', '\\]')
	      },
	      {
	        begin: '%q\\{',
	        end: /\}/,
	        contains: recursiveParen(/\{/, /\}/)
	      },
	      {
	        begin: '%q<',
	        end: '>',
	        contains: recursiveParen('<', '>')
	      },
	      {
	        begin: '%q\\|',
	        end: '\\|'
	      },
	      {
	        begin: /<<-'\w+'$/,
	        end: /^\s*\w+$/
	      }
	    ],
	    relevance: 0
	  };
	  const REGEXP = {
	    begin: '(?!%\\})(' + hljs.RE_STARTERS_RE + '|\\n|\\b(case|if|select|unless|until|when|while)\\b)\\s*',
	    keywords: 'case if select unless until when while',
	    contains: [
	      {
	        className: 'regexp',
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          SUBST
	        ],
	        variants: [
	          {
	            begin: '//[a-z]*',
	            relevance: 0
	          },
	          {
	            begin: '/(?!\\/)',
	            end: '/[a-z]*'
	          }
	        ]
	      }
	    ],
	    relevance: 0
	  };
	  const REGEXP2 = {
	    className: 'regexp',
	    contains: [
	      hljs.BACKSLASH_ESCAPE,
	      SUBST
	    ],
	    variants: [
	      {
	        begin: '%r\\(',
	        end: '\\)',
	        contains: recursiveParen('\\(', '\\)')
	      },
	      {
	        begin: '%r\\[',
	        end: '\\]',
	        contains: recursiveParen('\\[', '\\]')
	      },
	      {
	        begin: '%r\\{',
	        end: /\}/,
	        contains: recursiveParen(/\{/, /\}/)
	      },
	      {
	        begin: '%r<',
	        end: '>',
	        contains: recursiveParen('<', '>')
	      },
	      {
	        begin: '%r\\|',
	        end: '\\|'
	      }
	    ],
	    relevance: 0
	  };
	  const ATTRIBUTE = {
	    className: 'meta',
	    begin: '@\\[',
	    end: '\\]',
	    contains: [
	      hljs.inherit(hljs.QUOTE_STRING_MODE, {
	        className: 'meta-string'
	      })
	    ]
	  };
	  const CRYSTAL_DEFAULT_CONTAINS = [
	    EXPANSION,
	    STRING,
	    Q_STRING,
	    REGEXP2,
	    REGEXP,
	    ATTRIBUTE,
	    hljs.HASH_COMMENT_MODE,
	    {
	      className: 'class',
	      beginKeywords: 'class module struct',
	      end: '$|;',
	      illegal: /=/,
	      contains: [
	        hljs.HASH_COMMENT_MODE,
	        hljs.inherit(hljs.TITLE_MODE, {
	          begin: CRYSTAL_PATH_RE
	        }),
	        { // relevance booster for inheritance
	          begin: '<'
	        }
	      ]
	    },
	    {
	      className: 'class',
	      beginKeywords: 'lib enum union',
	      end: '$|;',
	      illegal: /=/,
	      contains: [
	        hljs.HASH_COMMENT_MODE,
	        hljs.inherit(hljs.TITLE_MODE, {
	          begin: CRYSTAL_PATH_RE
	        })
	      ]
	    },
	    {
	      beginKeywords: 'annotation',
	      end: '$|;',
	      illegal: /=/,
	      contains: [
	        hljs.HASH_COMMENT_MODE,
	        hljs.inherit(hljs.TITLE_MODE, {
	          begin: CRYSTAL_PATH_RE
	        })
	      ],
	      relevance: 2
	    },
	    {
	      className: 'function',
	      beginKeywords: 'def',
	      end: /\B\b/,
	      contains: [
	        hljs.inherit(hljs.TITLE_MODE, {
	          begin: CRYSTAL_METHOD_RE,
	          endsParent: true
	        })
	      ]
	    },
	    {
	      className: 'function',
	      beginKeywords: 'fun macro',
	      end: /\B\b/,
	      contains: [
	        hljs.inherit(hljs.TITLE_MODE, {
	          begin: CRYSTAL_METHOD_RE,
	          endsParent: true
	        })
	      ],
	      relevance: 2
	    },
	    {
	      className: 'symbol',
	      begin: hljs.UNDERSCORE_IDENT_RE + '(!|\\?)?:',
	      relevance: 0
	    },
	    {
	      className: 'symbol',
	      begin: ':',
	      contains: [
	        STRING,
	        {
	          begin: CRYSTAL_METHOD_RE
	        }
	      ],
	      relevance: 0
	    },
	    {
	      className: 'number',
	      variants: [
	        {
	          begin: '\\b0b([01_]+)' + INT_SUFFIX
	        },
	        {
	          begin: '\\b0o([0-7_]+)' + INT_SUFFIX
	        },
	        {
	          begin: '\\b0x([A-Fa-f0-9_]+)' + INT_SUFFIX
	        },
	        {
	          begin: '\\b([1-9][0-9_]*[0-9]|[0-9])(\\.[0-9][0-9_]*)?([eE]_?[-+]?[0-9_]*)?' + FLOAT_SUFFIX + '(?!_)'
	        },
	        {
	          begin: '\\b([1-9][0-9_]*|0)' + INT_SUFFIX
	        }
	      ],
	      relevance: 0
	    }
	  ];
	  SUBST.contains = CRYSTAL_DEFAULT_CONTAINS;
	  EXPANSION.contains = CRYSTAL_DEFAULT_CONTAINS.slice(1); // without EXPANSION

	  return {
	    name: 'Crystal',
	    aliases: [ 'cr' ],
	    keywords: CRYSTAL_KEYWORDS,
	    contains: CRYSTAL_DEFAULT_CONTAINS
	  };
	}

	crystal_1 = crystal;
	return crystal_1;
}

/*
Language: C#
Author: Jason Diamond <jason@diamond.name>
Contributor: Nicolas LLOBERA <nllobera@gmail.com>, Pieter Vantorre <pietervantorre@gmail.com>, David Pine <david.pine@microsoft.com>
Website: https://docs.microsoft.com/en-us/dotnet/csharp/
Category: common
*/

var csharp_1;
var hasRequiredCsharp;

function requireCsharp () {
	if (hasRequiredCsharp) return csharp_1;
	hasRequiredCsharp = 1;
	/** @type LanguageFn */
	function csharp(hljs) {
	  const BUILT_IN_KEYWORDS = [
	    'bool',
	    'byte',
	    'char',
	    'decimal',
	    'delegate',
	    'double',
	    'dynamic',
	    'enum',
	    'float',
	    'int',
	    'long',
	    'nint',
	    'nuint',
	    'object',
	    'sbyte',
	    'short',
	    'string',
	    'ulong',
	    'uint',
	    'ushort'
	  ];
	  const FUNCTION_MODIFIERS = [
	    'public',
	    'private',
	    'protected',
	    'static',
	    'internal',
	    'protected',
	    'abstract',
	    'async',
	    'extern',
	    'override',
	    'unsafe',
	    'virtual',
	    'new',
	    'sealed',
	    'partial'
	  ];
	  const LITERAL_KEYWORDS = [
	    'default',
	    'false',
	    'null',
	    'true'
	  ];
	  const NORMAL_KEYWORDS = [
	    'abstract',
	    'as',
	    'base',
	    'break',
	    'case',
	    'class',
	    'const',
	    'continue',
	    'do',
	    'else',
	    'event',
	    'explicit',
	    'extern',
	    'finally',
	    'fixed',
	    'for',
	    'foreach',
	    'goto',
	    'if',
	    'implicit',
	    'in',
	    'interface',
	    'internal',
	    'is',
	    'lock',
	    'namespace',
	    'new',
	    'operator',
	    'out',
	    'override',
	    'params',
	    'private',
	    'protected',
	    'public',
	    'readonly',
	    'record',
	    'ref',
	    'return',
	    'sealed',
	    'sizeof',
	    'stackalloc',
	    'static',
	    'struct',
	    'switch',
	    'this',
	    'throw',
	    'try',
	    'typeof',
	    'unchecked',
	    'unsafe',
	    'using',
	    'virtual',
	    'void',
	    'volatile',
	    'while'
	  ];
	  const CONTEXTUAL_KEYWORDS = [
	    'add',
	    'alias',
	    'and',
	    'ascending',
	    'async',
	    'await',
	    'by',
	    'descending',
	    'equals',
	    'from',
	    'get',
	    'global',
	    'group',
	    'init',
	    'into',
	    'join',
	    'let',
	    'nameof',
	    'not',
	    'notnull',
	    'on',
	    'or',
	    'orderby',
	    'partial',
	    'remove',
	    'select',
	    'set',
	    'unmanaged',
	    'value|0',
	    'var',
	    'when',
	    'where',
	    'with',
	    'yield'
	  ];

	  const KEYWORDS = {
	    keyword: NORMAL_KEYWORDS.concat(CONTEXTUAL_KEYWORDS),
	    built_in: BUILT_IN_KEYWORDS,
	    literal: LITERAL_KEYWORDS
	  };
	  const TITLE_MODE = hljs.inherit(hljs.TITLE_MODE, {
	    begin: '[a-zA-Z](\\.?\\w)*'
	  });
	  const NUMBERS = {
	    className: 'number',
	    variants: [
	      {
	        begin: '\\b(0b[01\']+)'
	      },
	      {
	        begin: '(-?)\\b([\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)(u|U|l|L|ul|UL|f|F|b|B)'
	      },
	      {
	        begin: '(-?)(\\b0[xX][a-fA-F0-9\']+|(\\b[\\d\']+(\\.[\\d\']*)?|\\.[\\d\']+)([eE][-+]?[\\d\']+)?)'
	      }
	    ],
	    relevance: 0
	  };
	  const VERBATIM_STRING = {
	    className: 'string',
	    begin: '@"',
	    end: '"',
	    contains: [
	      {
	        begin: '""'
	      }
	    ]
	  };
	  const VERBATIM_STRING_NO_LF = hljs.inherit(VERBATIM_STRING, {
	    illegal: /\n/
	  });
	  const SUBST = {
	    className: 'subst',
	    begin: /\{/,
	    end: /\}/,
	    keywords: KEYWORDS
	  };
	  const SUBST_NO_LF = hljs.inherit(SUBST, {
	    illegal: /\n/
	  });
	  const INTERPOLATED_STRING = {
	    className: 'string',
	    begin: /\$"/,
	    end: '"',
	    illegal: /\n/,
	    contains: [
	      {
	        begin: /\{\{/
	      },
	      {
	        begin: /\}\}/
	      },
	      hljs.BACKSLASH_ESCAPE,
	      SUBST_NO_LF
	    ]
	  };
	  const INTERPOLATED_VERBATIM_STRING = {
	    className: 'string',
	    begin: /\$@"/,
	    end: '"',
	    contains: [
	      {
	        begin: /\{\{/
	      },
	      {
	        begin: /\}\}/
	      },
	      {
	        begin: '""'
	      },
	      SUBST
	    ]
	  };
	  const INTERPOLATED_VERBATIM_STRING_NO_LF = hljs.inherit(INTERPOLATED_VERBATIM_STRING, {
	    illegal: /\n/,
	    contains: [
	      {
	        begin: /\{\{/
	      },
	      {
	        begin: /\}\}/
	      },
	      {
	        begin: '""'
	      },
	      SUBST_NO_LF
	    ]
	  });
	  SUBST.contains = [
	    INTERPOLATED_VERBATIM_STRING,
	    INTERPOLATED_STRING,
	    VERBATIM_STRING,
	    hljs.APOS_STRING_MODE,
	    hljs.QUOTE_STRING_MODE,
	    NUMBERS,
	    hljs.C_BLOCK_COMMENT_MODE
	  ];
	  SUBST_NO_LF.contains = [
	    INTERPOLATED_VERBATIM_STRING_NO_LF,
	    INTERPOLATED_STRING,
	    VERBATIM_STRING_NO_LF,
	    hljs.APOS_STRING_MODE,
	    hljs.QUOTE_STRING_MODE,
	    NUMBERS,
	    hljs.inherit(hljs.C_BLOCK_COMMENT_MODE, {
	      illegal: /\n/
	    })
	  ];
	  const STRING = {
	    variants: [
	      INTERPOLATED_VERBATIM_STRING,
	      INTERPOLATED_STRING,
	      VERBATIM_STRING,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE
	    ]
	  };

	  const GENERIC_MODIFIER = {
	    begin: "<",
	    end: ">",
	    contains: [
	      {
	        beginKeywords: "in out"
	      },
	      TITLE_MODE
	    ]
	  };
	  const TYPE_IDENT_RE = hljs.IDENT_RE + '(<' + hljs.IDENT_RE + '(\\s*,\\s*' + hljs.IDENT_RE + ')*>)?(\\[\\])?';
	  const AT_IDENTIFIER = {
	    // prevents expressions like `@class` from incorrect flagging
	    // `class` as a keyword
	    begin: "@" + hljs.IDENT_RE,
	    relevance: 0
	  };

	  return {
	    name: 'C#',
	    aliases: [
	      'cs',
	      'c#'
	    ],
	    keywords: KEYWORDS,
	    illegal: /::/,
	    contains: [
	      hljs.COMMENT(
	        '///',
	        '$',
	        {
	          returnBegin: true,
	          contains: [
	            {
	              className: 'doctag',
	              variants: [
	                {
	                  begin: '///',
	                  relevance: 0
	                },
	                {
	                  begin: '<!--|-->'
	                },
	                {
	                  begin: '</?',
	                  end: '>'
	                }
	              ]
	            }
	          ]
	        }
	      ),
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      {
	        className: 'meta',
	        begin: '#',
	        end: '$',
	        keywords: {
	          'meta-keyword': 'if else elif endif define undef warning error line region endregion pragma checksum'
	        }
	      },
	      STRING,
	      NUMBERS,
	      {
	        beginKeywords: 'class interface',
	        relevance: 0,
	        end: /[{;=]/,
	        illegal: /[^\s:,]/,
	        contains: [
	          {
	            beginKeywords: "where class"
	          },
	          TITLE_MODE,
	          GENERIC_MODIFIER,
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE
	        ]
	      },
	      {
	        beginKeywords: 'namespace',
	        relevance: 0,
	        end: /[{;=]/,
	        illegal: /[^\s:]/,
	        contains: [
	          TITLE_MODE,
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE
	        ]
	      },
	      {
	        beginKeywords: 'record',
	        relevance: 0,
	        end: /[{;=]/,
	        illegal: /[^\s:]/,
	        contains: [
	          TITLE_MODE,
	          GENERIC_MODIFIER,
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE
	        ]
	      },
	      {
	        // [Attributes("")]
	        className: 'meta',
	        begin: '^\\s*\\[',
	        excludeBegin: true,
	        end: '\\]',
	        excludeEnd: true,
	        contains: [
	          {
	            className: 'meta-string',
	            begin: /"/,
	            end: /"/
	          }
	        ]
	      },
	      {
	        // Expression keywords prevent 'keyword Name(...)' from being
	        // recognized as a function definition
	        beginKeywords: 'new return throw await else',
	        relevance: 0
	      },
	      {
	        className: 'function',
	        begin: '(' + TYPE_IDENT_RE + '\\s+)+' + hljs.IDENT_RE + '\\s*(<.+>\\s*)?\\(',
	        returnBegin: true,
	        end: /\s*[{;=]/,
	        excludeEnd: true,
	        keywords: KEYWORDS,
	        contains: [
	          // prevents these from being highlighted `title`
	          {
	            beginKeywords: FUNCTION_MODIFIERS.join(" "),
	            relevance: 0
	          },
	          {
	            begin: hljs.IDENT_RE + '\\s*(<.+>\\s*)?\\(',
	            returnBegin: true,
	            contains: [
	              hljs.TITLE_MODE,
	              GENERIC_MODIFIER
	            ],
	            relevance: 0
	          },
	          {
	            className: 'params',
	            begin: /\(/,
	            end: /\)/,
	            excludeBegin: true,
	            excludeEnd: true,
	            keywords: KEYWORDS,
	            relevance: 0,
	            contains: [
	              STRING,
	              NUMBERS,
	              hljs.C_BLOCK_COMMENT_MODE
	            ]
	          },
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE
	        ]
	      },
	      AT_IDENTIFIER
	    ]
	  };
	}

	csharp_1 = csharp;
	return csharp_1;
}

/*
Language: CSP
Description: Content Security Policy definition highlighting
Author: Taras <oxdef@oxdef.info>
Website: https://developer.mozilla.org/en-US/docs/Web/HTTP/CSP

vim: ts=2 sw=2 st=2
*/

var csp_1;
var hasRequiredCsp;

function requireCsp () {
	if (hasRequiredCsp) return csp_1;
	hasRequiredCsp = 1;
	/** @type LanguageFn */
	function csp(hljs) {
	  return {
	    name: 'CSP',
	    case_insensitive: false,
	    keywords: {
	      $pattern: '[a-zA-Z][a-zA-Z0-9_-]*',
	      keyword: 'base-uri child-src connect-src default-src font-src form-action ' +
	        'frame-ancestors frame-src img-src media-src object-src plugin-types ' +
	        'report-uri sandbox script-src style-src'
	    },
	    contains: [
	      {
	        className: 'string',
	        begin: "'",
	        end: "'"
	      },
	      {
	        className: 'attribute',
	        begin: '^Content',
	        end: ':',
	        excludeEnd: true
	      }
	    ]
	  };
	}

	csp_1 = csp;
	return csp_1;
}

var css_1;
var hasRequiredCss;

function requireCss () {
	if (hasRequiredCss) return css_1;
	hasRequiredCss = 1;
	const MODES = (hljs) => {
	  return {
	    IMPORTANT: {
	      className: 'meta',
	      begin: '!important'
	    },
	    HEXCOLOR: {
	      className: 'number',
	      begin: '#([a-fA-F0-9]{6}|[a-fA-F0-9]{3})'
	    },
	    ATTRIBUTE_SELECTOR_MODE: {
	      className: 'selector-attr',
	      begin: /\[/,
	      end: /\]/,
	      illegal: '$',
	      contains: [
	        hljs.APOS_STRING_MODE,
	        hljs.QUOTE_STRING_MODE
	      ]
	    }
	  };
	};

	const TAGS = [
	  'a',
	  'abbr',
	  'address',
	  'article',
	  'aside',
	  'audio',
	  'b',
	  'blockquote',
	  'body',
	  'button',
	  'canvas',
	  'caption',
	  'cite',
	  'code',
	  'dd',
	  'del',
	  'details',
	  'dfn',
	  'div',
	  'dl',
	  'dt',
	  'em',
	  'fieldset',
	  'figcaption',
	  'figure',
	  'footer',
	  'form',
	  'h1',
	  'h2',
	  'h3',
	  'h4',
	  'h5',
	  'h6',
	  'header',
	  'hgroup',
	  'html',
	  'i',
	  'iframe',
	  'img',
	  'input',
	  'ins',
	  'kbd',
	  'label',
	  'legend',
	  'li',
	  'main',
	  'mark',
	  'menu',
	  'nav',
	  'object',
	  'ol',
	  'p',
	  'q',
	  'quote',
	  'samp',
	  'section',
	  'span',
	  'strong',
	  'summary',
	  'sup',
	  'table',
	  'tbody',
	  'td',
	  'textarea',
	  'tfoot',
	  'th',
	  'thead',
	  'time',
	  'tr',
	  'ul',
	  'var',
	  'video'
	];

	const MEDIA_FEATURES = [
	  'any-hover',
	  'any-pointer',
	  'aspect-ratio',
	  'color',
	  'color-gamut',
	  'color-index',
	  'device-aspect-ratio',
	  'device-height',
	  'device-width',
	  'display-mode',
	  'forced-colors',
	  'grid',
	  'height',
	  'hover',
	  'inverted-colors',
	  'monochrome',
	  'orientation',
	  'overflow-block',
	  'overflow-inline',
	  'pointer',
	  'prefers-color-scheme',
	  'prefers-contrast',
	  'prefers-reduced-motion',
	  'prefers-reduced-transparency',
	  'resolution',
	  'scan',
	  'scripting',
	  'update',
	  'width',
	  // TODO: find a better solution?
	  'min-width',
	  'max-width',
	  'min-height',
	  'max-height'
	];

	// https://developer.mozilla.org/en-US/docs/Web/CSS/Pseudo-classes
	const PSEUDO_CLASSES = [
	  'active',
	  'any-link',
	  'blank',
	  'checked',
	  'current',
	  'default',
	  'defined',
	  'dir', // dir()
	  'disabled',
	  'drop',
	  'empty',
	  'enabled',
	  'first',
	  'first-child',
	  'first-of-type',
	  'fullscreen',
	  'future',
	  'focus',
	  'focus-visible',
	  'focus-within',
	  'has', // has()
	  'host', // host or host()
	  'host-context', // host-context()
	  'hover',
	  'indeterminate',
	  'in-range',
	  'invalid',
	  'is', // is()
	  'lang', // lang()
	  'last-child',
	  'last-of-type',
	  'left',
	  'link',
	  'local-link',
	  'not', // not()
	  'nth-child', // nth-child()
	  'nth-col', // nth-col()
	  'nth-last-child', // nth-last-child()
	  'nth-last-col', // nth-last-col()
	  'nth-last-of-type', //nth-last-of-type()
	  'nth-of-type', //nth-of-type()
	  'only-child',
	  'only-of-type',
	  'optional',
	  'out-of-range',
	  'past',
	  'placeholder-shown',
	  'read-only',
	  'read-write',
	  'required',
	  'right',
	  'root',
	  'scope',
	  'target',
	  'target-within',
	  'user-invalid',
	  'valid',
	  'visited',
	  'where' // where()
	];

	// https://developer.mozilla.org/en-US/docs/Web/CSS/Pseudo-elements
	const PSEUDO_ELEMENTS = [
	  'after',
	  'backdrop',
	  'before',
	  'cue',
	  'cue-region',
	  'first-letter',
	  'first-line',
	  'grammar-error',
	  'marker',
	  'part',
	  'placeholder',
	  'selection',
	  'slotted',
	  'spelling-error'
	];

	const ATTRIBUTES = [
	  'align-content',
	  'align-items',
	  'align-self',
	  'animation',
	  'animation-delay',
	  'animation-direction',
	  'animation-duration',
	  'animation-fill-mode',
	  'animation-iteration-count',
	  'animation-name',
	  'animation-play-state',
	  'animation-timing-function',
	  'auto',
	  'backface-visibility',
	  'background',
	  'background-attachment',
	  'background-clip',
	  'background-color',
	  'background-image',
	  'background-origin',
	  'background-position',
	  'background-repeat',
	  'background-size',
	  'border',
	  'border-bottom',
	  'border-bottom-color',
	  'border-bottom-left-radius',
	  'border-bottom-right-radius',
	  'border-bottom-style',
	  'border-bottom-width',
	  'border-collapse',
	  'border-color',
	  'border-image',
	  'border-image-outset',
	  'border-image-repeat',
	  'border-image-slice',
	  'border-image-source',
	  'border-image-width',
	  'border-left',
	  'border-left-color',
	  'border-left-style',
	  'border-left-width',
	  'border-radius',
	  'border-right',
	  'border-right-color',
	  'border-right-style',
	  'border-right-width',
	  'border-spacing',
	  'border-style',
	  'border-top',
	  'border-top-color',
	  'border-top-left-radius',
	  'border-top-right-radius',
	  'border-top-style',
	  'border-top-width',
	  'border-width',
	  'bottom',
	  'box-decoration-break',
	  'box-shadow',
	  'box-sizing',
	  'break-after',
	  'break-before',
	  'break-inside',
	  'caption-side',
	  'clear',
	  'clip',
	  'clip-path',
	  'color',
	  'column-count',
	  'column-fill',
	  'column-gap',
	  'column-rule',
	  'column-rule-color',
	  'column-rule-style',
	  'column-rule-width',
	  'column-span',
	  'column-width',
	  'columns',
	  'content',
	  'counter-increment',
	  'counter-reset',
	  'cursor',
	  'direction',
	  'display',
	  'empty-cells',
	  'filter',
	  'flex',
	  'flex-basis',
	  'flex-direction',
	  'flex-flow',
	  'flex-grow',
	  'flex-shrink',
	  'flex-wrap',
	  'float',
	  'font',
	  'font-display',
	  'font-family',
	  'font-feature-settings',
	  'font-kerning',
	  'font-language-override',
	  'font-size',
	  'font-size-adjust',
	  'font-smoothing',
	  'font-stretch',
	  'font-style',
	  'font-variant',
	  'font-variant-ligatures',
	  'font-variation-settings',
	  'font-weight',
	  'height',
	  'hyphens',
	  'icon',
	  'image-orientation',
	  'image-rendering',
	  'image-resolution',
	  'ime-mode',
	  'inherit',
	  'initial',
	  'justify-content',
	  'left',
	  'letter-spacing',
	  'line-height',
	  'list-style',
	  'list-style-image',
	  'list-style-position',
	  'list-style-type',
	  'margin',
	  'margin-bottom',
	  'margin-left',
	  'margin-right',
	  'margin-top',
	  'marks',
	  'mask',
	  'max-height',
	  'max-width',
	  'min-height',
	  'min-width',
	  'nav-down',
	  'nav-index',
	  'nav-left',
	  'nav-right',
	  'nav-up',
	  'none',
	  'normal',
	  'object-fit',
	  'object-position',
	  'opacity',
	  'order',
	  'orphans',
	  'outline',
	  'outline-color',
	  'outline-offset',
	  'outline-style',
	  'outline-width',
	  'overflow',
	  'overflow-wrap',
	  'overflow-x',
	  'overflow-y',
	  'padding',
	  'padding-bottom',
	  'padding-left',
	  'padding-right',
	  'padding-top',
	  'page-break-after',
	  'page-break-before',
	  'page-break-inside',
	  'perspective',
	  'perspective-origin',
	  'pointer-events',
	  'position',
	  'quotes',
	  'resize',
	  'right',
	  'src', // @font-face
	  'tab-size',
	  'table-layout',
	  'text-align',
	  'text-align-last',
	  'text-decoration',
	  'text-decoration-color',
	  'text-decoration-line',
	  'text-decoration-style',
	  'text-indent',
	  'text-overflow',
	  'text-rendering',
	  'text-shadow',
	  'text-transform',
	  'text-underline-position',
	  'top',
	  'transform',
	  'transform-origin',
	  'transform-style',
	  'transition',
	  'transition-delay',
	  'transition-duration',
	  'transition-property',
	  'transition-timing-function',
	  'unicode-bidi',
	  'vertical-align',
	  'visibility',
	  'white-space',
	  'widows',
	  'width',
	  'word-break',
	  'word-spacing',
	  'word-wrap',
	  'z-index'
	  // reverse makes sure longer attributes `font-weight` are matched fully
	  // instead of getting false positives on say `font`
	].reverse();

	/**
	 * @param {string} value
	 * @returns {RegExp}
	 * */

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: CSS
	Category: common, css
	Website: https://developer.mozilla.org/en-US/docs/Web/CSS
	*/

	/** @type LanguageFn */
	function css(hljs) {
	  const modes = MODES(hljs);
	  const FUNCTION_DISPATCH = {
	    className: "built_in",
	    begin: /[\w-]+(?=\()/
	  };
	  const VENDOR_PREFIX = {
	    begin: /-(webkit|moz|ms|o)-(?=[a-z])/
	  };
	  const AT_MODIFIERS = "and or not only";
	  const AT_PROPERTY_RE = /@-?\w[\w]*(-\w+)*/; // @-webkit-keyframes
	  const IDENT_RE = '[a-zA-Z-][a-zA-Z0-9_-]*';
	  const STRINGS = [
	    hljs.APOS_STRING_MODE,
	    hljs.QUOTE_STRING_MODE
	  ];

	  return {
	    name: 'CSS',
	    case_insensitive: true,
	    illegal: /[=|'\$]/,
	    keywords: {
	      keyframePosition: "from to"
	    },
	    classNameAliases: {
	      // for visual continuity with `tag {}` and because we
	      // don't have a great class for this?
	      keyframePosition: "selector-tag"
	    },
	    contains: [
	      hljs.C_BLOCK_COMMENT_MODE,
	      VENDOR_PREFIX,
	      // to recognize keyframe 40% etc which are outside the scope of our
	      // attribute value mode
	      hljs.CSS_NUMBER_MODE,
	      {
	        className: 'selector-id',
	        begin: /#[A-Za-z0-9_-]+/,
	        relevance: 0
	      },
	      {
	        className: 'selector-class',
	        begin: '\\.' + IDENT_RE,
	        relevance: 0
	      },
	      modes.ATTRIBUTE_SELECTOR_MODE,
	      {
	        className: 'selector-pseudo',
	        variants: [
	          {
	            begin: ':(' + PSEUDO_CLASSES.join('|') + ')'
	          },
	          {
	            begin: '::(' + PSEUDO_ELEMENTS.join('|') + ')'
	          }
	        ]
	      },
	      // we may actually need this (12/2020)
	      // { // pseudo-selector params
	      //   begin: /\(/,
	      //   end: /\)/,
	      //   contains: [ hljs.CSS_NUMBER_MODE ]
	      // },
	      {
	        className: 'attribute',
	        begin: '\\b(' + ATTRIBUTES.join('|') + ')\\b'
	      },
	      // attribute values
	      {
	        begin: ':',
	        end: '[;}]',
	        contains: [
	          modes.HEXCOLOR,
	          modes.IMPORTANT,
	          hljs.CSS_NUMBER_MODE,
	          ...STRINGS,
	          // needed to highlight these as strings and to avoid issues with
	          // illegal characters that might be inside urls that would tigger the
	          // languages illegal stack
	          {
	            begin: /(url|data-uri)\(/,
	            end: /\)/,
	            relevance: 0, // from keywords
	            keywords: {
	              built_in: "url data-uri"
	            },
	            contains: [
	              {
	                className: "string",
	                // any character other than `)` as in `url()` will be the start
	                // of a string, which ends with `)` (from the parent mode)
	                begin: /[^)]/,
	                endsWithParent: true,
	                excludeEnd: true
	              }
	            ]
	          },
	          FUNCTION_DISPATCH
	        ]
	      },
	      {
	        begin: lookahead(/@/),
	        end: '[{;]',
	        relevance: 0,
	        illegal: /:/, // break on Less variables @var: ...
	        contains: [
	          {
	            className: 'keyword',
	            begin: AT_PROPERTY_RE
	          },
	          {
	            begin: /\s/,
	            endsWithParent: true,
	            excludeEnd: true,
	            relevance: 0,
	            keywords: {
	              $pattern: /[a-z-]+/,
	              keyword: AT_MODIFIERS,
	              attribute: MEDIA_FEATURES.join(" ")
	            },
	            contains: [
	              {
	                begin: /[a-z-]+(?=:)/,
	                className: "attribute"
	              },
	              ...STRINGS,
	              hljs.CSS_NUMBER_MODE
	            ]
	          }
	        ]
	      },
	      {
	        className: 'selector-tag',
	        begin: '\\b(' + TAGS.join('|') + ')\\b'
	      }
	    ]
	  };
	}

	css_1 = css;
	return css_1;
}

/*
Language: D
Author: Aleksandar Ruzicic <aleksandar@ruzicic.info>
Description: D is a language with C-like syntax and static typing. It pragmatically combines efficiency, control, and modeling power, with safety and programmer productivity.
Version: 1.0a
Website: https://dlang.org
Date: 2012-04-08
*/

var d_1;
var hasRequiredD;

function requireD () {
	if (hasRequiredD) return d_1;
	hasRequiredD = 1;
	/**
	 * Known issues:
	 *
	 * - invalid hex string literals will be recognized as a double quoted strings
	 *   but 'x' at the beginning of string will not be matched
	 *
	 * - delimited string literals are not checked for matching end delimiter
	 *   (not possible to do with js regexp)
	 *
	 * - content of token string is colored as a string (i.e. no keyword coloring inside a token string)
	 *   also, content of token string is not validated to contain only valid D tokens
	 *
	 * - special token sequence rule is not strictly following D grammar (anything following #line
	 *   up to the end of line is matched as special token sequence)
	 */

	/** @type LanguageFn */
	function d(hljs) {
	  /**
	   * Language keywords
	   *
	   * @type {Object}
	   */
	  const D_KEYWORDS = {
	    $pattern: hljs.UNDERSCORE_IDENT_RE,
	    keyword:
	      'abstract alias align asm assert auto body break byte case cast catch class ' +
	      'const continue debug default delete deprecated do else enum export extern final ' +
	      'finally for foreach foreach_reverse|10 goto if immutable import in inout int ' +
	      'interface invariant is lazy macro mixin module new nothrow out override package ' +
	      'pragma private protected public pure ref return scope shared static struct ' +
	      'super switch synchronized template this throw try typedef typeid typeof union ' +
	      'unittest version void volatile while with __FILE__ __LINE__ __gshared|10 ' +
	      '__thread __traits __DATE__ __EOF__ __TIME__ __TIMESTAMP__ __VENDOR__ __VERSION__',
	    built_in:
	      'bool cdouble cent cfloat char creal dchar delegate double dstring float function ' +
	      'idouble ifloat ireal long real short string ubyte ucent uint ulong ushort wchar ' +
	      'wstring',
	    literal:
	      'false null true'
	  };

	  /**
	   * Number literal regexps
	   *
	   * @type {String}
	   */
	  const decimal_integer_re = '(0|[1-9][\\d_]*)';
	  const decimal_integer_nosus_re = '(0|[1-9][\\d_]*|\\d[\\d_]*|[\\d_]+?\\d)';
	  const binary_integer_re = '0[bB][01_]+';
	  const hexadecimal_digits_re = '([\\da-fA-F][\\da-fA-F_]*|_[\\da-fA-F][\\da-fA-F_]*)';
	  const hexadecimal_integer_re = '0[xX]' + hexadecimal_digits_re;

	  const decimal_exponent_re = '([eE][+-]?' + decimal_integer_nosus_re + ')';
	  const decimal_float_re = '(' + decimal_integer_nosus_re + '(\\.\\d*|' + decimal_exponent_re + ')|' +
	                '\\d+\\.' + decimal_integer_nosus_re + '|' +
	                '\\.' + decimal_integer_re + decimal_exponent_re + '?' +
	              ')';
	  const hexadecimal_float_re = '(0[xX](' +
	                  hexadecimal_digits_re + '\\.' + hexadecimal_digits_re + '|' +
	                  '\\.?' + hexadecimal_digits_re +
	                 ')[pP][+-]?' + decimal_integer_nosus_re + ')';

	  const integer_re = '(' +
	      decimal_integer_re + '|' +
	      binary_integer_re + '|' +
	       hexadecimal_integer_re +
	    ')';

	  const float_re = '(' +
	      hexadecimal_float_re + '|' +
	      decimal_float_re +
	    ')';

	  /**
	   * Escape sequence supported in D string and character literals
	   *
	   * @type {String}
	   */
	  const escape_sequence_re = '\\\\(' +
	              '[\'"\\?\\\\abfnrtv]|' + // common escapes
	              'u[\\dA-Fa-f]{4}|' + // four hex digit unicode codepoint
	              '[0-7]{1,3}|' + // one to three octal digit ascii char code
	              'x[\\dA-Fa-f]{2}|' + // two hex digit ascii char code
	              'U[\\dA-Fa-f]{8}' + // eight hex digit unicode codepoint
	              ')|' +
	              '&[a-zA-Z\\d]{2,};'; // named character entity

	  /**
	   * D integer number literals
	   *
	   * @type {Object}
	   */
	  const D_INTEGER_MODE = {
	    className: 'number',
	    begin: '\\b' + integer_re + '(L|u|U|Lu|LU|uL|UL)?',
	    relevance: 0
	  };

	  /**
	   * [D_FLOAT_MODE description]
	   * @type {Object}
	   */
	  const D_FLOAT_MODE = {
	    className: 'number',
	    begin: '\\b(' +
	        float_re + '([fF]|L|i|[fF]i|Li)?|' +
	        integer_re + '(i|[fF]i|Li)' +
	      ')',
	    relevance: 0
	  };

	  /**
	   * D character literal
	   *
	   * @type {Object}
	   */
	  const D_CHARACTER_MODE = {
	    className: 'string',
	    begin: '\'(' + escape_sequence_re + '|.)',
	    end: '\'',
	    illegal: '.'
	  };

	  /**
	   * D string escape sequence
	   *
	   * @type {Object}
	   */
	  const D_ESCAPE_SEQUENCE = {
	    begin: escape_sequence_re,
	    relevance: 0
	  };

	  /**
	   * D double quoted string literal
	   *
	   * @type {Object}
	   */
	  const D_STRING_MODE = {
	    className: 'string',
	    begin: '"',
	    contains: [D_ESCAPE_SEQUENCE],
	    end: '"[cwd]?'
	  };

	  /**
	   * D wysiwyg and delimited string literals
	   *
	   * @type {Object}
	   */
	  const D_WYSIWYG_DELIMITED_STRING_MODE = {
	    className: 'string',
	    begin: '[rq]"',
	    end: '"[cwd]?',
	    relevance: 5
	  };

	  /**
	   * D alternate wysiwyg string literal
	   *
	   * @type {Object}
	   */
	  const D_ALTERNATE_WYSIWYG_STRING_MODE = {
	    className: 'string',
	    begin: '`',
	    end: '`[cwd]?'
	  };

	  /**
	   * D hexadecimal string literal
	   *
	   * @type {Object}
	   */
	  const D_HEX_STRING_MODE = {
	    className: 'string',
	    begin: 'x"[\\da-fA-F\\s\\n\\r]*"[cwd]?',
	    relevance: 10
	  };

	  /**
	   * D delimited string literal
	   *
	   * @type {Object}
	   */
	  const D_TOKEN_STRING_MODE = {
	    className: 'string',
	    begin: 'q"\\{',
	    end: '\\}"'
	  };

	  /**
	   * Hashbang support
	   *
	   * @type {Object}
	   */
	  const D_HASHBANG_MODE = {
	    className: 'meta',
	    begin: '^#!',
	    end: '$',
	    relevance: 5
	  };

	  /**
	   * D special token sequence
	   *
	   * @type {Object}
	   */
	  const D_SPECIAL_TOKEN_SEQUENCE_MODE = {
	    className: 'meta',
	    begin: '#(line)',
	    end: '$',
	    relevance: 5
	  };

	  /**
	   * D attributes
	   *
	   * @type {Object}
	   */
	  const D_ATTRIBUTE_MODE = {
	    className: 'keyword',
	    begin: '@[a-zA-Z_][a-zA-Z_\\d]*'
	  };

	  /**
	   * D nesting comment
	   *
	   * @type {Object}
	   */
	  const D_NESTING_COMMENT_MODE = hljs.COMMENT(
	    '\\/\\+',
	    '\\+\\/',
	    {
	      contains: ['self'],
	      relevance: 10
	    }
	  );

	  return {
	    name: 'D',
	    keywords: D_KEYWORDS,
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      D_NESTING_COMMENT_MODE,
	      D_HEX_STRING_MODE,
	      D_STRING_MODE,
	      D_WYSIWYG_DELIMITED_STRING_MODE,
	      D_ALTERNATE_WYSIWYG_STRING_MODE,
	      D_TOKEN_STRING_MODE,
	      D_FLOAT_MODE,
	      D_INTEGER_MODE,
	      D_CHARACTER_MODE,
	      D_HASHBANG_MODE,
	      D_SPECIAL_TOKEN_SEQUENCE_MODE,
	      D_ATTRIBUTE_MODE
	    ]
	  };
	}

	d_1 = d;
	return d_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var markdown_1;
var hasRequiredMarkdown;

function requireMarkdown () {
	if (hasRequiredMarkdown) return markdown_1;
	hasRequiredMarkdown = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: Markdown
	Requires: xml.js
	Author: John Crepezzi <john.crepezzi@gmail.com>
	Website: https://daringfireball.net/projects/markdown/
	Category: common, markup
	*/

	function markdown(hljs) {
	  const INLINE_HTML = {
	    begin: /<\/?[A-Za-z_]/,
	    end: '>',
	    subLanguage: 'xml',
	    relevance: 0
	  };
	  const HORIZONTAL_RULE = {
	    begin: '^[-\\*]{3,}',
	    end: '$'
	  };
	  const CODE = {
	    className: 'code',
	    variants: [
	      // TODO: fix to allow these to work with sublanguage also
	      {
	        begin: '(`{3,})[^`](.|\\n)*?\\1`*[ ]*'
	      },
	      {
	        begin: '(~{3,})[^~](.|\\n)*?\\1~*[ ]*'
	      },
	      // needed to allow markdown as a sublanguage to work
	      {
	        begin: '```',
	        end: '```+[ ]*$'
	      },
	      {
	        begin: '~~~',
	        end: '~~~+[ ]*$'
	      },
	      {
	        begin: '`.+?`'
	      },
	      {
	        begin: '(?=^( {4}|\\t))',
	        // use contains to gobble up multiple lines to allow the block to be whatever size
	        // but only have a single open/close tag vs one per line
	        contains: [
	          {
	            begin: '^( {4}|\\t)',
	            end: '(\\n)$'
	          }
	        ],
	        relevance: 0
	      }
	    ]
	  };
	  const LIST = {
	    className: 'bullet',
	    begin: '^[ \t]*([*+-]|(\\d+\\.))(?=\\s+)',
	    end: '\\s+',
	    excludeEnd: true
	  };
	  const LINK_REFERENCE = {
	    begin: /^\[[^\n]+\]:/,
	    returnBegin: true,
	    contains: [
	      {
	        className: 'symbol',
	        begin: /\[/,
	        end: /\]/,
	        excludeBegin: true,
	        excludeEnd: true
	      },
	      {
	        className: 'link',
	        begin: /:\s*/,
	        end: /$/,
	        excludeBegin: true
	      }
	    ]
	  };
	  const URL_SCHEME = /[A-Za-z][A-Za-z0-9+.-]*/;
	  const LINK = {
	    variants: [
	      // too much like nested array access in so many languages
	      // to have any real relevance
	      {
	        begin: /\[.+?\]\[.*?\]/,
	        relevance: 0
	      },
	      // popular internet URLs
	      {
	        begin: /\[.+?\]\(((data|javascript|mailto):|(?:http|ftp)s?:\/\/).*?\)/,
	        relevance: 2
	      },
	      {
	        begin: concat(/\[.+?\]\(/, URL_SCHEME, /:\/\/.*?\)/),
	        relevance: 2
	      },
	      // relative urls
	      {
	        begin: /\[.+?\]\([./?&#].*?\)/,
	        relevance: 1
	      },
	      // whatever else, lower relevance (might not be a link at all)
	      {
	        begin: /\[.+?\]\(.*?\)/,
	        relevance: 0
	      }
	    ],
	    returnBegin: true,
	    contains: [
	      {
	        className: 'string',
	        relevance: 0,
	        begin: '\\[',
	        end: '\\]',
	        excludeBegin: true,
	        returnEnd: true
	      },
	      {
	        className: 'link',
	        relevance: 0,
	        begin: '\\]\\(',
	        end: '\\)',
	        excludeBegin: true,
	        excludeEnd: true
	      },
	      {
	        className: 'symbol',
	        relevance: 0,
	        begin: '\\]\\[',
	        end: '\\]',
	        excludeBegin: true,
	        excludeEnd: true
	      }
	    ]
	  };
	  const BOLD = {
	    className: 'strong',
	    contains: [], // defined later
	    variants: [
	      {
	        begin: /_{2}/,
	        end: /_{2}/
	      },
	      {
	        begin: /\*{2}/,
	        end: /\*{2}/
	      }
	    ]
	  };
	  const ITALIC = {
	    className: 'emphasis',
	    contains: [], // defined later
	    variants: [
	      {
	        begin: /\*(?!\*)/,
	        end: /\*/
	      },
	      {
	        begin: /_(?!_)/,
	        end: /_/,
	        relevance: 0
	      }
	    ]
	  };
	  BOLD.contains.push(ITALIC);
	  ITALIC.contains.push(BOLD);

	  let CONTAINABLE = [
	    INLINE_HTML,
	    LINK
	  ];

	  BOLD.contains = BOLD.contains.concat(CONTAINABLE);
	  ITALIC.contains = ITALIC.contains.concat(CONTAINABLE);

	  CONTAINABLE = CONTAINABLE.concat(BOLD, ITALIC);

	  const HEADER = {
	    className: 'section',
	    variants: [
	      {
	        begin: '^#{1,6}',
	        end: '$',
	        contains: CONTAINABLE
	      },
	      {
	        begin: '(?=^.+?\\n[=-]{2,}$)',
	        contains: [
	          {
	            begin: '^[=-]*$'
	          },
	          {
	            begin: '^',
	            end: "\\n",
	            contains: CONTAINABLE
	          }
	        ]
	      }
	    ]
	  };

	  const BLOCKQUOTE = {
	    className: 'quote',
	    begin: '^>\\s+',
	    contains: CONTAINABLE,
	    end: '$'
	  };

	  return {
	    name: 'Markdown',
	    aliases: [
	      'md',
	      'mkdown',
	      'mkd'
	    ],
	    contains: [
	      HEADER,
	      INLINE_HTML,
	      LIST,
	      BOLD,
	      ITALIC,
	      BLOCKQUOTE,
	      CODE,
	      HORIZONTAL_RULE,
	      LINK,
	      LINK_REFERENCE
	    ]
	  };
	}

	markdown_1 = markdown;
	return markdown_1;
}

/*
Language: Dart
Requires: markdown.js
Author: Maxim Dikun <dikmax@gmail.com>
Description: Dart a modern, object-oriented language developed by Google. For more information see https://www.dartlang.org/
Website: https://dart.dev
Category: scripting
*/

var dart_1;
var hasRequiredDart;

function requireDart () {
	if (hasRequiredDart) return dart_1;
	hasRequiredDart = 1;
	/** @type LanguageFn */
	function dart(hljs) {
	  const SUBST = {
	    className: 'subst',
	    variants: [{
	      begin: '\\$[A-Za-z0-9_]+'
	    }]
	  };

	  const BRACED_SUBST = {
	    className: 'subst',
	    variants: [{
	      begin: /\$\{/,
	      end: /\}/
	    }],
	    keywords: 'true false null this is new super'
	  };

	  const STRING = {
	    className: 'string',
	    variants: [
	      {
	        begin: 'r\'\'\'',
	        end: '\'\'\''
	      },
	      {
	        begin: 'r"""',
	        end: '"""'
	      },
	      {
	        begin: 'r\'',
	        end: '\'',
	        illegal: '\\n'
	      },
	      {
	        begin: 'r"',
	        end: '"',
	        illegal: '\\n'
	      },
	      {
	        begin: '\'\'\'',
	        end: '\'\'\'',
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          SUBST,
	          BRACED_SUBST
	        ]
	      },
	      {
	        begin: '"""',
	        end: '"""',
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          SUBST,
	          BRACED_SUBST
	        ]
	      },
	      {
	        begin: '\'',
	        end: '\'',
	        illegal: '\\n',
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          SUBST,
	          BRACED_SUBST
	        ]
	      },
	      {
	        begin: '"',
	        end: '"',
	        illegal: '\\n',
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          SUBST,
	          BRACED_SUBST
	        ]
	      }
	    ]
	  };
	  BRACED_SUBST.contains = [
	    hljs.C_NUMBER_MODE,
	    STRING
	  ];

	  const BUILT_IN_TYPES = [
	    // dart:core
	    'Comparable',
	    'DateTime',
	    'Duration',
	    'Function',
	    'Iterable',
	    'Iterator',
	    'List',
	    'Map',
	    'Match',
	    'Object',
	    'Pattern',
	    'RegExp',
	    'Set',
	    'Stopwatch',
	    'String',
	    'StringBuffer',
	    'StringSink',
	    'Symbol',
	    'Type',
	    'Uri',
	    'bool',
	    'double',
	    'int',
	    'num',
	    // dart:html
	    'Element',
	    'ElementList'
	  ];
	  const NULLABLE_BUILT_IN_TYPES = BUILT_IN_TYPES.map((e) => `${e}?`);

	  const KEYWORDS = {
	    keyword: 'abstract as assert async await break case catch class const continue covariant default deferred do ' +
	      'dynamic else enum export extends extension external factory false final finally for Function get hide if ' +
	      'implements import in inferface is late library mixin new null on operator part required rethrow return set ' +
	      'show static super switch sync this throw true try typedef var void while with yield',
	    built_in:
	      BUILT_IN_TYPES
	        .concat(NULLABLE_BUILT_IN_TYPES)
	        .concat([
	          // dart:core
	          'Never',
	          'Null',
	          'dynamic',
	          'print',
	          // dart:html
	          'document',
	          'querySelector',
	          'querySelectorAll',
	          'window'
	        ]),
	    $pattern: /[A-Za-z][A-Za-z0-9_]*\??/
	  };

	  return {
	    name: 'Dart',
	    keywords: KEYWORDS,
	    contains: [
	      STRING,
	      hljs.COMMENT(
	        /\/\*\*(?!\/)/,
	        /\*\//,
	        {
	          subLanguage: 'markdown',
	          relevance: 0
	        }
	      ),
	      hljs.COMMENT(
	        /\/{3,} ?/,
	        /$/, {
	          contains: [{
	            subLanguage: 'markdown',
	            begin: '.',
	            end: '$',
	            relevance: 0
	          }]
	        }
	      ),
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      {
	        className: 'class',
	        beginKeywords: 'class interface',
	        end: /\{/,
	        excludeEnd: true,
	        contains: [
	          {
	            beginKeywords: 'extends implements'
	          },
	          hljs.UNDERSCORE_TITLE_MODE
	        ]
	      },
	      hljs.C_NUMBER_MODE,
	      {
	        className: 'meta',
	        begin: '@[A-Za-z]+'
	      },
	      {
	        begin: '=>' // No markup, just a relevance booster
	      }
	    ]
	  };
	}

	dart_1 = dart;
	return dart_1;
}

/*
Language: Delphi
Website: https://www.embarcadero.com/products/delphi
*/

var delphi_1;
var hasRequiredDelphi;

function requireDelphi () {
	if (hasRequiredDelphi) return delphi_1;
	hasRequiredDelphi = 1;
	/** @type LanguageFn */
	function delphi(hljs) {
	  const KEYWORDS =
	    'exports register file shl array record property for mod while set ally label uses raise not ' +
	    'stored class safecall var interface or private static exit index inherited to else stdcall ' +
	    'override shr asm far resourcestring finalization packed virtual out and protected library do ' +
	    'xorwrite goto near function end div overload object unit begin string on inline repeat until ' +
	    'destructor write message program with read initialization except default nil if case cdecl in ' +
	    'downto threadvar of try pascal const external constructor type public then implementation ' +
	    'finally published procedure absolute reintroduce operator as is abstract alias assembler ' +
	    'bitpacked break continue cppdecl cvar enumerator experimental platform deprecated ' +
	    'unimplemented dynamic export far16 forward generic helper implements interrupt iochecks ' +
	    'local name nodefault noreturn nostackframe oldfpccall otherwise saveregisters softfloat ' +
	    'specialize strict unaligned varargs ';
	  const COMMENT_MODES = [
	    hljs.C_LINE_COMMENT_MODE,
	    hljs.COMMENT(/\{/, /\}/, {
	      relevance: 0
	    }),
	    hljs.COMMENT(/\(\*/, /\*\)/, {
	      relevance: 10
	    })
	  ];
	  const DIRECTIVE = {
	    className: 'meta',
	    variants: [
	      {
	        begin: /\{\$/,
	        end: /\}/
	      },
	      {
	        begin: /\(\*\$/,
	        end: /\*\)/
	      }
	    ]
	  };
	  const STRING = {
	    className: 'string',
	    begin: /'/,
	    end: /'/,
	    contains: [{
	      begin: /''/
	    }]
	  };
	  const NUMBER = {
	    className: 'number',
	    relevance: 0,
	    // Source: https://www.freepascal.org/docs-html/ref/refse6.html
	    variants: [
	      {
	        // Hexadecimal notation, e.g., $7F.
	        begin: '\\$[0-9A-Fa-f]+'
	      },
	      {
	        // Octal notation, e.g., &42.
	        begin: '&[0-7]+'
	      },
	      {
	        // Binary notation, e.g., %1010.
	        begin: '%[01]+'
	      }
	    ]
	  };
	  const CHAR_STRING = {
	    className: 'string',
	    begin: /(#\d+)+/
	  };
	  const CLASS = {
	    begin: hljs.IDENT_RE + '\\s*=\\s*class\\s*\\(',
	    returnBegin: true,
	    contains: [hljs.TITLE_MODE]
	  };
	  const FUNCTION = {
	    className: 'function',
	    beginKeywords: 'function constructor destructor procedure',
	    end: /[:;]/,
	    keywords: 'function constructor|10 destructor|10 procedure|10',
	    contains: [
	      hljs.TITLE_MODE,
	      {
	        className: 'params',
	        begin: /\(/,
	        end: /\)/,
	        keywords: KEYWORDS,
	        contains: [
	          STRING,
	          CHAR_STRING,
	          DIRECTIVE
	        ].concat(COMMENT_MODES)
	      },
	      DIRECTIVE
	    ].concat(COMMENT_MODES)
	  };
	  return {
	    name: 'Delphi',
	    aliases: [
	      'dpr',
	      'dfm',
	      'pas',
	      'pascal',
	      'freepascal',
	      'lazarus',
	      'lpr',
	      'lfm'
	    ],
	    case_insensitive: true,
	    keywords: KEYWORDS,
	    illegal: /"|\$[G-Zg-z]|\/\*|<\/|\|/,
	    contains: [
	      STRING,
	      CHAR_STRING,
	      hljs.NUMBER_MODE,
	      NUMBER,
	      CLASS,
	      FUNCTION,
	      DIRECTIVE
	    ].concat(COMMENT_MODES)
	  };
	}

	delphi_1 = delphi;
	return delphi_1;
}

/*
Language: Diff
Description: Unified and context diff
Author: Vasily Polovnyov <vast@whiteants.net>
Website: https://www.gnu.org/software/diffutils/
Category: common
*/

var diff_1;
var hasRequiredDiff;

function requireDiff () {
	if (hasRequiredDiff) return diff_1;
	hasRequiredDiff = 1;
	/** @type LanguageFn */
	function diff(hljs) {
	  return {
	    name: 'Diff',
	    aliases: ['patch'],
	    contains: [
	      {
	        className: 'meta',
	        relevance: 10,
	        variants: [
	          {
	            begin: /^@@ +-\d+,\d+ +\+\d+,\d+ +@@/
	          },
	          {
	            begin: /^\*\*\* +\d+,\d+ +\*\*\*\*$/
	          },
	          {
	            begin: /^--- +\d+,\d+ +----$/
	          }
	        ]
	      },
	      {
	        className: 'comment',
	        variants: [
	          {
	            begin: /Index: /,
	            end: /$/
	          },
	          {
	            begin: /^index/,
	            end: /$/
	          },
	          {
	            begin: /={3,}/,
	            end: /$/
	          },
	          {
	            begin: /^-{3}/,
	            end: /$/
	          },
	          {
	            begin: /^\*{3} /,
	            end: /$/
	          },
	          {
	            begin: /^\+{3}/,
	            end: /$/
	          },
	          {
	            begin: /^\*{15}$/
	          },
	          {
	            begin: /^diff --git/,
	            end: /$/
	          }
	        ]
	      },
	      {
	        className: 'addition',
	        begin: /^\+/,
	        end: /$/
	      },
	      {
	        className: 'deletion',
	        begin: /^-/,
	        end: /$/
	      },
	      {
	        className: 'addition',
	        begin: /^!/,
	        end: /$/
	      }
	    ]
	  };
	}

	diff_1 = diff;
	return diff_1;
}

/*
Language: Django
Description: Django is a high-level Python Web framework that encourages rapid development and clean, pragmatic design.
Requires: xml.js
Author: Ivan Sagalaev <maniac@softwaremaniacs.org>
Contributors: Ilya Baryshev <baryshev@gmail.com>
Website: https://www.djangoproject.com
Category: template
*/

var django_1;
var hasRequiredDjango;

function requireDjango () {
	if (hasRequiredDjango) return django_1;
	hasRequiredDjango = 1;
	/** @type LanguageFn */
	function django(hljs) {
	  const FILTER = {
	    begin: /\|[A-Za-z]+:?/,
	    keywords: {
	      name:
	        'truncatewords removetags linebreaksbr yesno get_digit timesince random striptags ' +
	        'filesizeformat escape linebreaks length_is ljust rjust cut urlize fix_ampersands ' +
	        'title floatformat capfirst pprint divisibleby add make_list unordered_list urlencode ' +
	        'timeuntil urlizetrunc wordcount stringformat linenumbers slice date dictsort ' +
	        'dictsortreversed default_if_none pluralize lower join center default ' +
	        'truncatewords_html upper length phone2numeric wordwrap time addslashes slugify first ' +
	        'escapejs force_escape iriencode last safe safeseq truncatechars localize unlocalize ' +
	        'localtime utc timezone'
	    },
	    contains: [
	      hljs.QUOTE_STRING_MODE,
	      hljs.APOS_STRING_MODE
	    ]
	  };

	  return {
	    name: 'Django',
	    aliases: ['jinja'],
	    case_insensitive: true,
	    subLanguage: 'xml',
	    contains: [
	      hljs.COMMENT(/\{%\s*comment\s*%\}/, /\{%\s*endcomment\s*%\}/),
	      hljs.COMMENT(/\{#/, /#\}/),
	      {
	        className: 'template-tag',
	        begin: /\{%/,
	        end: /%\}/,
	        contains: [{
	          className: 'name',
	          begin: /\w+/,
	          keywords: {
	            name:
	                'comment endcomment load templatetag ifchanged endifchanged if endif firstof for ' +
	                'endfor ifnotequal endifnotequal widthratio extends include spaceless ' +
	                'endspaceless regroup ifequal endifequal ssi now with cycle url filter ' +
	                'endfilter debug block endblock else autoescape endautoescape csrf_token empty elif ' +
	                'endwith static trans blocktrans endblocktrans get_static_prefix get_media_prefix ' +
	                'plural get_current_language language get_available_languages ' +
	                'get_current_language_bidi get_language_info get_language_info_list localize ' +
	                'endlocalize localtime endlocaltime timezone endtimezone get_current_timezone ' +
	                'verbatim'
	          },
	          starts: {
	            endsWithParent: true,
	            keywords: 'in by as',
	            contains: [FILTER],
	            relevance: 0
	          }
	        }]
	      },
	      {
	        className: 'template-variable',
	        begin: /\{\{/,
	        end: /\}\}/,
	        contains: [FILTER]
	      }
	    ]
	  };
	}

	django_1 = django;
	return django_1;
}

/*
Language: DNS Zone
Author: Tim Schumacher <tim@datenknoten.me>
Category: config
Website: https://en.wikipedia.org/wiki/Zone_file
*/

var dns_1;
var hasRequiredDns;

function requireDns () {
	if (hasRequiredDns) return dns_1;
	hasRequiredDns = 1;
	/** @type LanguageFn */
	function dns(hljs) {
	  return {
	    name: 'DNS Zone',
	    aliases: [
	      'bind',
	      'zone'
	    ],
	    keywords: {
	      keyword:
	        'IN A AAAA AFSDB APL CAA CDNSKEY CDS CERT CNAME DHCID DLV DNAME DNSKEY DS HIP IPSECKEY KEY KX ' +
	        'LOC MX NAPTR NS NSEC NSEC3 NSEC3PARAM PTR RRSIG RP SIG SOA SRV SSHFP TA TKEY TLSA TSIG TXT'
	    },
	    contains: [
	      hljs.COMMENT(';', '$', {
	        relevance: 0
	      }),
	      {
	        className: 'meta',
	        begin: /^\$(TTL|GENERATE|INCLUDE|ORIGIN)\b/
	      },
	      // IPv6
	      {
	        className: 'number',
	        begin: '((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)(\\.(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]?\\d)){3}))|:)))\\b'
	      },
	      // IPv4
	      {
	        className: 'number',
	        begin: '((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\\b'
	      },
	      hljs.inherit(hljs.NUMBER_MODE, {
	        begin: /\b\d+[dhwm]?/
	      })
	    ]
	  };
	}

	dns_1 = dns;
	return dns_1;
}

/*
Language: Dockerfile
Requires: bash.js
Author: Alexis Hénaut <alexis@henaut.net>
Description: language definition for Dockerfile files
Website: https://docs.docker.com/engine/reference/builder/
Category: config
*/

var dockerfile_1;
var hasRequiredDockerfile;

function requireDockerfile () {
	if (hasRequiredDockerfile) return dockerfile_1;
	hasRequiredDockerfile = 1;
	/** @type LanguageFn */
	function dockerfile(hljs) {
	  return {
	    name: 'Dockerfile',
	    aliases: ['docker'],
	    case_insensitive: true,
	    keywords: 'from maintainer expose env arg user onbuild stopsignal',
	    contains: [
	      hljs.HASH_COMMENT_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.NUMBER_MODE,
	      {
	        beginKeywords: 'run cmd entrypoint volume add copy workdir label healthcheck shell',
	        starts: {
	          end: /[^\\]$/,
	          subLanguage: 'bash'
	        }
	      }
	    ],
	    illegal: '</'
	  };
	}

	dockerfile_1 = dockerfile;
	return dockerfile_1;
}

/*
Language: Batch file (DOS)
Author: Alexander Makarov <sam@rmcreative.ru>
Contributors: Anton Kochkov <anton.kochkov@gmail.com>
Website: https://en.wikipedia.org/wiki/Batch_file
*/

var dos_1;
var hasRequiredDos;

function requireDos () {
	if (hasRequiredDos) return dos_1;
	hasRequiredDos = 1;
	/** @type LanguageFn */
	function dos(hljs) {
	  const COMMENT = hljs.COMMENT(
	    /^\s*@?rem\b/, /$/,
	    {
	      relevance: 10
	    }
	  );
	  const LABEL = {
	    className: 'symbol',
	    begin: '^\\s*[A-Za-z._?][A-Za-z0-9_$#@~.?]*(:|\\s+label)',
	    relevance: 0
	  };
	  return {
	    name: 'Batch file (DOS)',
	    aliases: [
	      'bat',
	      'cmd'
	    ],
	    case_insensitive: true,
	    illegal: /\/\*/,
	    keywords: {
	      keyword:
	        'if else goto for in do call exit not exist errorlevel defined ' +
	        'equ neq lss leq gtr geq',
	      built_in:
	        'prn nul lpt3 lpt2 lpt1 con com4 com3 com2 com1 aux ' +
	        'shift cd dir echo setlocal endlocal set pause copy ' +
	        'append assoc at attrib break cacls cd chcp chdir chkdsk chkntfs cls cmd color ' +
	        'comp compact convert date dir diskcomp diskcopy doskey erase fs ' +
	        'find findstr format ftype graftabl help keyb label md mkdir mode more move path ' +
	        'pause print popd pushd promt rd recover rem rename replace restore rmdir shift ' +
	        'sort start subst time title tree type ver verify vol ' +
	        // winutils
	        'ping net ipconfig taskkill xcopy ren del'
	    },
	    contains: [
	      {
	        className: 'variable',
	        begin: /%%[^ ]|%[^ ]+?%|![^ ]+?!/
	      },
	      {
	        className: 'function',
	        begin: LABEL.begin,
	        end: 'goto:eof',
	        contains: [
	          hljs.inherit(hljs.TITLE_MODE, {
	            begin: '([_a-zA-Z]\\w*\\.)*([_a-zA-Z]\\w*:)?[_a-zA-Z]\\w*'
	          }),
	          COMMENT
	        ]
	      },
	      {
	        className: 'number',
	        begin: '\\b\\d+',
	        relevance: 0
	      },
	      COMMENT
	    ]
	  };
	}

	dos_1 = dos;
	return dos_1;
}

/*
 Language: dsconfig
 Description: dsconfig batch configuration language for LDAP directory servers
 Contributors: Jacob Childress <jacobc@gmail.com>
 Category: enterprise, config
 */

var dsconfig_1;
var hasRequiredDsconfig;

function requireDsconfig () {
	if (hasRequiredDsconfig) return dsconfig_1;
	hasRequiredDsconfig = 1;
	/** @type LanguageFn */
	function dsconfig(hljs) {
	  const QUOTED_PROPERTY = {
	    className: 'string',
	    begin: /"/,
	    end: /"/
	  };
	  const APOS_PROPERTY = {
	    className: 'string',
	    begin: /'/,
	    end: /'/
	  };
	  const UNQUOTED_PROPERTY = {
	    className: 'string',
	    begin: /[\w\-?]+:\w+/,
	    end: /\W/,
	    relevance: 0
	  };
	  const VALUELESS_PROPERTY = {
	    className: 'string',
	    begin: /\w+(\-\w+)*/,
	    end: /(?=\W)/,
	    relevance: 0
	  };

	  return {
	    keywords: 'dsconfig',
	    contains: [
	      {
	        className: 'keyword',
	        begin: '^dsconfig',
	        end: /\s/,
	        excludeEnd: true,
	        relevance: 10
	      },
	      {
	        className: 'built_in',
	        begin: /(list|create|get|set|delete)-(\w+)/,
	        end: /\s/,
	        excludeEnd: true,
	        illegal: '!@#$%^&*()',
	        relevance: 10
	      },
	      {
	        className: 'built_in',
	        begin: /--(\w+)/,
	        end: /\s/,
	        excludeEnd: true
	      },
	      QUOTED_PROPERTY,
	      APOS_PROPERTY,
	      UNQUOTED_PROPERTY,
	      VALUELESS_PROPERTY,
	      hljs.HASH_COMMENT_MODE
	    ]
	  };
	}

	dsconfig_1 = dsconfig;
	return dsconfig_1;
}

/*
Language: Device Tree
Description: *.dts files used in the Linux kernel
Author: Martin Braun <martin.braun@ettus.com>, Moritz Fischer <moritz.fischer@ettus.com>
Website: https://elinux.org/Device_Tree_Reference
Category: config
*/

var dts_1;
var hasRequiredDts;

function requireDts () {
	if (hasRequiredDts) return dts_1;
	hasRequiredDts = 1;
	/** @type LanguageFn */
	function dts(hljs) {
	  const STRINGS = {
	    className: 'string',
	    variants: [
	      hljs.inherit(hljs.QUOTE_STRING_MODE, {
	        begin: '((u8?|U)|L)?"'
	      }),
	      {
	        begin: '(u8?|U)?R"',
	        end: '"',
	        contains: [hljs.BACKSLASH_ESCAPE]
	      },
	      {
	        begin: '\'\\\\?.',
	        end: '\'',
	        illegal: '.'
	      }
	    ]
	  };

	  const NUMBERS = {
	    className: 'number',
	    variants: [
	      {
	        begin: '\\b(\\d+(\\.\\d*)?|\\.\\d+)(u|U|l|L|ul|UL|f|F)'
	      },
	      {
	        begin: hljs.C_NUMBER_RE
	      }
	    ],
	    relevance: 0
	  };

	  const PREPROCESSOR = {
	    className: 'meta',
	    begin: '#',
	    end: '$',
	    keywords: {
	      'meta-keyword': 'if else elif endif define undef ifdef ifndef'
	    },
	    contains: [
	      {
	        begin: /\\\n/,
	        relevance: 0
	      },
	      {
	        beginKeywords: 'include',
	        end: '$',
	        keywords: {
	          'meta-keyword': 'include'
	        },
	        contains: [
	          hljs.inherit(STRINGS, {
	            className: 'meta-string'
	          }),
	          {
	            className: 'meta-string',
	            begin: '<',
	            end: '>',
	            illegal: '\\n'
	          }
	        ]
	      },
	      STRINGS,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE
	    ]
	  };

	  const DTS_REFERENCE = {
	    className: 'variable',
	    begin: /&[a-z\d_]*\b/
	  };

	  const DTS_KEYWORD = {
	    className: 'meta-keyword',
	    begin: '/[a-z][a-z\\d-]*/'
	  };

	  const DTS_LABEL = {
	    className: 'symbol',
	    begin: '^\\s*[a-zA-Z_][a-zA-Z\\d_]*:'
	  };

	  const DTS_CELL_PROPERTY = {
	    className: 'params',
	    begin: '<',
	    end: '>',
	    contains: [
	      NUMBERS,
	      DTS_REFERENCE
	    ]
	  };

	  const DTS_NODE = {
	    className: 'class',
	    begin: /[a-zA-Z_][a-zA-Z\d_@]*\s\{/,
	    end: /[{;=]/,
	    returnBegin: true,
	    excludeEnd: true
	  };

	  const DTS_ROOT_NODE = {
	    className: 'class',
	    begin: '/\\s*\\{',
	    end: /\};/,
	    relevance: 10,
	    contains: [
	      DTS_REFERENCE,
	      DTS_KEYWORD,
	      DTS_LABEL,
	      DTS_NODE,
	      DTS_CELL_PROPERTY,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      NUMBERS,
	      STRINGS
	    ]
	  };

	  return {
	    name: 'Device Tree',
	    keywords: "",
	    contains: [
	      DTS_ROOT_NODE,
	      DTS_REFERENCE,
	      DTS_KEYWORD,
	      DTS_LABEL,
	      DTS_NODE,
	      DTS_CELL_PROPERTY,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      NUMBERS,
	      STRINGS,
	      PREPROCESSOR,
	      {
	        begin: hljs.IDENT_RE + '::',
	        keywords: ""
	      }
	    ]
	  };
	}

	dts_1 = dts;
	return dts_1;
}

/*
Language: Dust
Requires: xml.js
Author: Michael Allen <michael.allen@benefitfocus.com>
Description: Matcher for dust.js templates.
Website: https://www.dustjs.com
Category: template
*/

var dust_1;
var hasRequiredDust;

function requireDust () {
	if (hasRequiredDust) return dust_1;
	hasRequiredDust = 1;
	/** @type LanguageFn */
	function dust(hljs) {
	  const EXPRESSION_KEYWORDS = 'if eq ne lt lte gt gte select default math sep';
	  return {
	    name: 'Dust',
	    aliases: ['dst'],
	    case_insensitive: true,
	    subLanguage: 'xml',
	    contains: [
	      {
	        className: 'template-tag',
	        begin: /\{[#\/]/,
	        end: /\}/,
	        illegal: /;/,
	        contains: [{
	          className: 'name',
	          begin: /[a-zA-Z\.-]+/,
	          starts: {
	            endsWithParent: true,
	            relevance: 0,
	            contains: [hljs.QUOTE_STRING_MODE]
	          }
	        }]
	      },
	      {
	        className: 'template-variable',
	        begin: /\{/,
	        end: /\}/,
	        illegal: /;/,
	        keywords: EXPRESSION_KEYWORDS
	      }
	    ]
	  };
	}

	dust_1 = dust;
	return dust_1;
}

/*
Language: Extended Backus-Naur Form
Author: Alex McKibben <alex@nullscope.net>
Website: https://en.wikipedia.org/wiki/Extended_Backus–Naur_form
*/

var ebnf_1;
var hasRequiredEbnf;

function requireEbnf () {
	if (hasRequiredEbnf) return ebnf_1;
	hasRequiredEbnf = 1;
	/** @type LanguageFn */
	function ebnf(hljs) {
	  const commentMode = hljs.COMMENT(/\(\*/, /\*\)/);

	  const nonTerminalMode = {
	    className: "attribute",
	    begin: /^[ ]*[a-zA-Z]+([\s_-]+[a-zA-Z]+)*/
	  };

	  const specialSequenceMode = {
	    className: "meta",
	    begin: /\?.*\?/
	  };

	  const ruleBodyMode = {
	    begin: /=/,
	    end: /[.;]/,
	    contains: [
	      commentMode,
	      specialSequenceMode,
	      {
	        // terminals
	        className: 'string',
	        variants: [
	          hljs.APOS_STRING_MODE,
	          hljs.QUOTE_STRING_MODE,
	          {
	            begin: '`',
	            end: '`'
	          }
	        ]
	      }
	    ]
	  };

	  return {
	    name: 'Extended Backus-Naur Form',
	    illegal: /\S/,
	    contains: [
	      commentMode,
	      nonTerminalMode,
	      ruleBodyMode
	    ]
	  };
	}

	ebnf_1 = ebnf;
	return ebnf_1;
}

/*
Language: Elixir
Author: Josh Adams <josh@isotope11.com>
Description: language definition for Elixir source code files (.ex and .exs).  Based on ruby language support.
Category: functional
Website: https://elixir-lang.org
*/

var elixir_1;
var hasRequiredElixir;

function requireElixir () {
	if (hasRequiredElixir) return elixir_1;
	hasRequiredElixir = 1;
	/** @type LanguageFn */
	function elixir(hljs) {
	  const ELIXIR_IDENT_RE = '[a-zA-Z_][a-zA-Z0-9_.]*(!|\\?)?';
	  const ELIXIR_METHOD_RE = '[a-zA-Z_]\\w*[!?=]?|[-+~]@|<<|>>|=~|===?|<=>|[<>]=?|\\*\\*|[-/+%^&*~`|]|\\[\\]=?';
	  const ELIXIR_KEYWORDS = {
	    $pattern: ELIXIR_IDENT_RE,
	    keyword: 'and false then defined module in return redo retry end for true self when ' +
	    'next until do begin unless nil break not case cond alias while ensure or ' +
	    'include use alias fn quote require import with|0'
	  };
	  const SUBST = {
	    className: 'subst',
	    begin: /#\{/,
	    end: /\}/,
	    keywords: ELIXIR_KEYWORDS
	  };
	  const NUMBER = {
	    className: 'number',
	    begin: '(\\b0o[0-7_]+)|(\\b0b[01_]+)|(\\b0x[0-9a-fA-F_]+)|(-?\\b[1-9][0-9_]*(\\.[0-9_]+([eE][-+]?[0-9]+)?)?)',
	    relevance: 0
	  };
	  const SIGIL_DELIMITERS = '[/|([{<"\']';
	  const LOWERCASE_SIGIL = {
	    className: 'string',
	    begin: '~[a-z]' + '(?=' + SIGIL_DELIMITERS + ')',
	    contains: [
	      {
	        endsParent: true,
	        contains: [
	          {
	            contains: [
	              hljs.BACKSLASH_ESCAPE,
	              SUBST
	            ],
	            variants: [
	              {
	                begin: /"/,
	                end: /"/
	              },
	              {
	                begin: /'/,
	                end: /'/
	              },
	              {
	                begin: /\//,
	                end: /\//
	              },
	              {
	                begin: /\|/,
	                end: /\|/
	              },
	              {
	                begin: /\(/,
	                end: /\)/
	              },
	              {
	                begin: /\[/,
	                end: /\]/
	              },
	              {
	                begin: /\{/,
	                end: /\}/
	              },
	              {
	                begin: /</,
	                end: />/
	              }
	            ]
	          }
	        ]
	      }
	    ]
	  };

	  const UPCASE_SIGIL = {
	    className: 'string',
	    begin: '~[A-Z]' + '(?=' + SIGIL_DELIMITERS + ')',
	    contains: [
	      {
	        begin: /"/,
	        end: /"/
	      },
	      {
	        begin: /'/,
	        end: /'/
	      },
	      {
	        begin: /\//,
	        end: /\//
	      },
	      {
	        begin: /\|/,
	        end: /\|/
	      },
	      {
	        begin: /\(/,
	        end: /\)/
	      },
	      {
	        begin: /\[/,
	        end: /\]/
	      },
	      {
	        begin: /\{/,
	        end: /\}/
	      },
	      {
	        begin: /</,
	        end: />/
	      }
	    ]
	  };

	  const STRING = {
	    className: 'string',
	    contains: [
	      hljs.BACKSLASH_ESCAPE,
	      SUBST
	    ],
	    variants: [
	      {
	        begin: /"""/,
	        end: /"""/
	      },
	      {
	        begin: /'''/,
	        end: /'''/
	      },
	      {
	        begin: /~S"""/,
	        end: /"""/,
	        contains: [] // override default
	      },
	      {
	        begin: /~S"/,
	        end: /"/,
	        contains: [] // override default
	      },
	      {
	        begin: /~S'''/,
	        end: /'''/,
	        contains: [] // override default
	      },
	      {
	        begin: /~S'/,
	        end: /'/,
	        contains: [] // override default
	      },
	      {
	        begin: /'/,
	        end: /'/
	      },
	      {
	        begin: /"/,
	        end: /"/
	      }
	    ]
	  };
	  const FUNCTION = {
	    className: 'function',
	    beginKeywords: 'def defp defmacro',
	    end: /\B\b/, // the mode is ended by the title
	    contains: [
	      hljs.inherit(hljs.TITLE_MODE, {
	        begin: ELIXIR_IDENT_RE,
	        endsParent: true
	      })
	    ]
	  };
	  const CLASS = hljs.inherit(FUNCTION, {
	    className: 'class',
	    beginKeywords: 'defimpl defmodule defprotocol defrecord',
	    end: /\bdo\b|$|;/
	  });
	  const ELIXIR_DEFAULT_CONTAINS = [
	    STRING,
	    UPCASE_SIGIL,
	    LOWERCASE_SIGIL,
	    hljs.HASH_COMMENT_MODE,
	    CLASS,
	    FUNCTION,
	    {
	      begin: '::'
	    },
	    {
	      className: 'symbol',
	      begin: ':(?![\\s:])',
	      contains: [
	        STRING,
	        {
	          begin: ELIXIR_METHOD_RE
	        }
	      ],
	      relevance: 0
	    },
	    {
	      className: 'symbol',
	      begin: ELIXIR_IDENT_RE + ':(?!:)',
	      relevance: 0
	    },
	    NUMBER,
	    {
	      className: 'variable',
	      begin: '(\\$\\W)|((\\$|@@?)(\\w+))'
	    },
	    {
	      begin: '->'
	    },
	    { // regexp container
	      begin: '(' + hljs.RE_STARTERS_RE + ')\\s*',
	      contains: [
	        hljs.HASH_COMMENT_MODE,
	        {
	          // to prevent false regex triggers for the division function:
	          // /:
	          begin: /\/: (?=\d+\s*[,\]])/,
	          relevance: 0,
	          contains: [NUMBER]
	        },
	        {
	          className: 'regexp',
	          illegal: '\\n',
	          contains: [
	            hljs.BACKSLASH_ESCAPE,
	            SUBST
	          ],
	          variants: [
	            {
	              begin: '/',
	              end: '/[a-z]*'
	            },
	            {
	              begin: '%r\\[',
	              end: '\\][a-z]*'
	            }
	          ]
	        }
	      ],
	      relevance: 0
	    }
	  ];
	  SUBST.contains = ELIXIR_DEFAULT_CONTAINS;

	  return {
	    name: 'Elixir',
	    keywords: ELIXIR_KEYWORDS,
	    contains: ELIXIR_DEFAULT_CONTAINS
	  };
	}

	elixir_1 = elixir;
	return elixir_1;
}

/*
Language: Elm
Author: Janis Voigtlaender <janis.voigtlaender@gmail.com>
Website: https://elm-lang.org
Category: functional
*/

var elm_1;
var hasRequiredElm;

function requireElm () {
	if (hasRequiredElm) return elm_1;
	hasRequiredElm = 1;
	/** @type LanguageFn */
	function elm(hljs) {
	  const COMMENT = {
	    variants: [
	      hljs.COMMENT('--', '$'),
	      hljs.COMMENT(
	        /\{-/,
	        /-\}/,
	        {
	          contains: ['self']
	        }
	      )
	    ]
	  };

	  const CONSTRUCTOR = {
	    className: 'type',
	    begin: '\\b[A-Z][\\w\']*', // TODO: other constructors (built-in, infix).
	    relevance: 0
	  };

	  const LIST = {
	    begin: '\\(',
	    end: '\\)',
	    illegal: '"',
	    contains: [
	      {
	        className: 'type',
	        begin: '\\b[A-Z][\\w]*(\\((\\.\\.|,|\\w+)\\))?'
	      },
	      COMMENT
	    ]
	  };

	  const RECORD = {
	    begin: /\{/,
	    end: /\}/,
	    contains: LIST.contains
	  };

	  const CHARACTER = {
	    className: 'string',
	    begin: '\'\\\\?.',
	    end: '\'',
	    illegal: '.'
	  };

	  return {
	    name: 'Elm',
	    keywords:
	      'let in if then else case of where module import exposing ' +
	      'type alias as infix infixl infixr port effect command subscription',
	    contains: [

	      // Top-level constructions.

	      {
	        beginKeywords: 'port effect module',
	        end: 'exposing',
	        keywords: 'port effect module where command subscription exposing',
	        contains: [
	          LIST,
	          COMMENT
	        ],
	        illegal: '\\W\\.|;'
	      },
	      {
	        begin: 'import',
	        end: '$',
	        keywords: 'import as exposing',
	        contains: [
	          LIST,
	          COMMENT
	        ],
	        illegal: '\\W\\.|;'
	      },
	      {
	        begin: 'type',
	        end: '$',
	        keywords: 'type alias',
	        contains: [
	          CONSTRUCTOR,
	          LIST,
	          RECORD,
	          COMMENT
	        ]
	      },
	      {
	        beginKeywords: 'infix infixl infixr',
	        end: '$',
	        contains: [
	          hljs.C_NUMBER_MODE,
	          COMMENT
	        ]
	      },
	      {
	        begin: 'port',
	        end: '$',
	        keywords: 'port',
	        contains: [COMMENT]
	      },

	      // Literals and names.

	      CHARACTER,
	      hljs.QUOTE_STRING_MODE,
	      hljs.C_NUMBER_MODE,
	      CONSTRUCTOR,
	      hljs.inherit(hljs.TITLE_MODE, {
	        begin: '^[_a-z][\\w\']*'
	      }),
	      COMMENT,

	      {
	        begin: '->|<-'
	      } // No markup, relevance booster
	    ],
	    illegal: /;/
	  };
	}

	elm_1 = elm;
	return elm_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var ruby_1;
var hasRequiredRuby;

function requireRuby () {
	if (hasRequiredRuby) return ruby_1;
	hasRequiredRuby = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: Ruby
	Description: Ruby is a dynamic, open source programming language with a focus on simplicity and productivity.
	Website: https://www.ruby-lang.org/
	Author: Anton Kovalyov <anton@kovalyov.net>
	Contributors: Peter Leonov <gojpeg@yandex.ru>, Vasily Polovnyov <vast@whiteants.net>, Loren Segal <lsegal@soen.ca>, Pascal Hurni <phi@ruby-reactive.org>, Cedric Sohrauer <sohrauer@googlemail.com>
	Category: common
	*/

	function ruby(hljs) {
	  const RUBY_METHOD_RE = '([a-zA-Z_]\\w*[!?=]?|[-+~]@|<<|>>|=~|===?|<=>|[<>]=?|\\*\\*|[-/+%^&*~`|]|\\[\\]=?)';
	  const RUBY_KEYWORDS = {
	    keyword:
	      'and then defined module in return redo if BEGIN retry end for self when ' +
	      'next until do begin unless END rescue else break undef not super class case ' +
	      'require yield alias while ensure elsif or include attr_reader attr_writer attr_accessor ' +
	      '__FILE__',
	    built_in: 'proc lambda',
	    literal:
	      'true false nil'
	  };
	  const YARDOCTAG = {
	    className: 'doctag',
	    begin: '@[A-Za-z]+'
	  };
	  const IRB_OBJECT = {
	    begin: '#<',
	    end: '>'
	  };
	  const COMMENT_MODES = [
	    hljs.COMMENT(
	      '#',
	      '$',
	      {
	        contains: [ YARDOCTAG ]
	      }
	    ),
	    hljs.COMMENT(
	      '^=begin',
	      '^=end',
	      {
	        contains: [ YARDOCTAG ],
	        relevance: 10
	      }
	    ),
	    hljs.COMMENT('^__END__', '\\n$')
	  ];
	  const SUBST = {
	    className: 'subst',
	    begin: /#\{/,
	    end: /\}/,
	    keywords: RUBY_KEYWORDS
	  };
	  const STRING = {
	    className: 'string',
	    contains: [
	      hljs.BACKSLASH_ESCAPE,
	      SUBST
	    ],
	    variants: [
	      {
	        begin: /'/,
	        end: /'/
	      },
	      {
	        begin: /"/,
	        end: /"/
	      },
	      {
	        begin: /`/,
	        end: /`/
	      },
	      {
	        begin: /%[qQwWx]?\(/,
	        end: /\)/
	      },
	      {
	        begin: /%[qQwWx]?\[/,
	        end: /\]/
	      },
	      {
	        begin: /%[qQwWx]?\{/,
	        end: /\}/
	      },
	      {
	        begin: /%[qQwWx]?</,
	        end: />/
	      },
	      {
	        begin: /%[qQwWx]?\//,
	        end: /\//
	      },
	      {
	        begin: /%[qQwWx]?%/,
	        end: /%/
	      },
	      {
	        begin: /%[qQwWx]?-/,
	        end: /-/
	      },
	      {
	        begin: /%[qQwWx]?\|/,
	        end: /\|/
	      },
	      // in the following expressions, \B in the beginning suppresses recognition of ?-sequences
	      // where ? is the last character of a preceding identifier, as in: `func?4`
	      {
	        begin: /\B\?(\\\d{1,3})/
	      },
	      {
	        begin: /\B\?(\\x[A-Fa-f0-9]{1,2})/
	      },
	      {
	        begin: /\B\?(\\u\{?[A-Fa-f0-9]{1,6}\}?)/
	      },
	      {
	        begin: /\B\?(\\M-\\C-|\\M-\\c|\\c\\M-|\\M-|\\C-\\M-)[\x20-\x7e]/
	      },
	      {
	        begin: /\B\?\\(c|C-)[\x20-\x7e]/
	      },
	      {
	        begin: /\B\?\\?\S/
	      },
	      { // heredocs
	        begin: /<<[-~]?'?(\w+)\n(?:[^\n]*\n)*?\s*\1\b/,
	        returnBegin: true,
	        contains: [
	          {
	            begin: /<<[-~]?'?/
	          },
	          hljs.END_SAME_AS_BEGIN({
	            begin: /(\w+)/,
	            end: /(\w+)/,
	            contains: [
	              hljs.BACKSLASH_ESCAPE,
	              SUBST
	            ]
	          })
	        ]
	      }
	    ]
	  };

	  // Ruby syntax is underdocumented, but this grammar seems to be accurate
	  // as of version 2.7.2 (confirmed with (irb and `Ripper.sexp(...)`)
	  // https://docs.ruby-lang.org/en/2.7.0/doc/syntax/literals_rdoc.html#label-Numbers
	  const decimal = '[1-9](_?[0-9])*|0';
	  const digits = '[0-9](_?[0-9])*';
	  const NUMBER = {
	    className: 'number',
	    relevance: 0,
	    variants: [
	      // decimal integer/float, optionally exponential or rational, optionally imaginary
	      {
	        begin: `\\b(${decimal})(\\.(${digits}))?([eE][+-]?(${digits})|r)?i?\\b`
	      },

	      // explicit decimal/binary/octal/hexadecimal integer,
	      // optionally rational and/or imaginary
	      {
	        begin: "\\b0[dD][0-9](_?[0-9])*r?i?\\b"
	      },
	      {
	        begin: "\\b0[bB][0-1](_?[0-1])*r?i?\\b"
	      },
	      {
	        begin: "\\b0[oO][0-7](_?[0-7])*r?i?\\b"
	      },
	      {
	        begin: "\\b0[xX][0-9a-fA-F](_?[0-9a-fA-F])*r?i?\\b"
	      },

	      // 0-prefixed implicit octal integer, optionally rational and/or imaginary
	      {
	        begin: "\\b0(_?[0-7])+r?i?\\b"
	      }
	    ]
	  };

	  const PARAMS = {
	    className: 'params',
	    begin: '\\(',
	    end: '\\)',
	    endsParent: true,
	    keywords: RUBY_KEYWORDS
	  };

	  const RUBY_DEFAULT_CONTAINS = [
	    STRING,
	    {
	      className: 'class',
	      beginKeywords: 'class module',
	      end: '$|;',
	      illegal: /=/,
	      contains: [
	        hljs.inherit(hljs.TITLE_MODE, {
	          begin: '[A-Za-z_]\\w*(::\\w+)*(\\?|!)?'
	        }),
	        {
	          begin: '<\\s*',
	          contains: [
	            {
	              begin: '(' + hljs.IDENT_RE + '::)?' + hljs.IDENT_RE,
	              // we already get points for <, we don't need poitns
	              // for the name also
	              relevance: 0
	            }
	          ]
	        }
	      ].concat(COMMENT_MODES)
	    },
	    {
	      className: 'function',
	      // def method_name(
	      // def method_name;
	      // def method_name (end of line)
	      begin: concat(/def\s+/, lookahead(RUBY_METHOD_RE + "\\s*(\\(|;|$)")),
	      relevance: 0, // relevance comes from kewords
	      keywords: "def",
	      end: '$|;',
	      contains: [
	        hljs.inherit(hljs.TITLE_MODE, {
	          begin: RUBY_METHOD_RE
	        }),
	        PARAMS
	      ].concat(COMMENT_MODES)
	    },
	    {
	      // swallow namespace qualifiers before symbols
	      begin: hljs.IDENT_RE + '::'
	    },
	    {
	      className: 'symbol',
	      begin: hljs.UNDERSCORE_IDENT_RE + '(!|\\?)?:',
	      relevance: 0
	    },
	    {
	      className: 'symbol',
	      begin: ':(?!\\s)',
	      contains: [
	        STRING,
	        {
	          begin: RUBY_METHOD_RE
	        }
	      ],
	      relevance: 0
	    },
	    NUMBER,
	    {
	      // negative-look forward attemps to prevent false matches like:
	      // @ident@ or $ident$ that might indicate this is not ruby at all
	      className: "variable",
	      begin: '(\\$\\W)|((\\$|@@?)(\\w+))(?=[^@$?])' + `(?![A-Za-z])(?![@$?'])`
	    },
	    {
	      className: 'params',
	      begin: /\|/,
	      end: /\|/,
	      relevance: 0, // this could be a lot of things (in other languages) other than params
	      keywords: RUBY_KEYWORDS
	    },
	    { // regexp container
	      begin: '(' + hljs.RE_STARTERS_RE + '|unless)\\s*',
	      keywords: 'unless',
	      contains: [
	        {
	          className: 'regexp',
	          contains: [
	            hljs.BACKSLASH_ESCAPE,
	            SUBST
	          ],
	          illegal: /\n/,
	          variants: [
	            {
	              begin: '/',
	              end: '/[a-z]*'
	            },
	            {
	              begin: /%r\{/,
	              end: /\}[a-z]*/
	            },
	            {
	              begin: '%r\\(',
	              end: '\\)[a-z]*'
	            },
	            {
	              begin: '%r!',
	              end: '![a-z]*'
	            },
	            {
	              begin: '%r\\[',
	              end: '\\][a-z]*'
	            }
	          ]
	        }
	      ].concat(IRB_OBJECT, COMMENT_MODES),
	      relevance: 0
	    }
	  ].concat(IRB_OBJECT, COMMENT_MODES);

	  SUBST.contains = RUBY_DEFAULT_CONTAINS;
	  PARAMS.contains = RUBY_DEFAULT_CONTAINS;

	  // >>
	  // ?>
	  const SIMPLE_PROMPT = "[>?]>";
	  // irb(main):001:0>
	  const DEFAULT_PROMPT = "[\\w#]+\\(\\w+\\):\\d+:\\d+>";
	  const RVM_PROMPT = "(\\w+-)?\\d+\\.\\d+\\.\\d+(p\\d+)?[^\\d][^>]+>";

	  const IRB_DEFAULT = [
	    {
	      begin: /^\s*=>/,
	      starts: {
	        end: '$',
	        contains: RUBY_DEFAULT_CONTAINS
	      }
	    },
	    {
	      className: 'meta',
	      begin: '^(' + SIMPLE_PROMPT + "|" + DEFAULT_PROMPT + '|' + RVM_PROMPT + ')(?=[ ])',
	      starts: {
	        end: '$',
	        contains: RUBY_DEFAULT_CONTAINS
	      }
	    }
	  ];

	  COMMENT_MODES.unshift(IRB_OBJECT);

	  return {
	    name: 'Ruby',
	    aliases: [
	      'rb',
	      'gemspec',
	      'podspec',
	      'thor',
	      'irb'
	    ],
	    keywords: RUBY_KEYWORDS,
	    illegal: /\/\*/,
	    contains: [
	      hljs.SHEBANG({
	        binary: "ruby"
	      })
	    ]
	      .concat(IRB_DEFAULT)
	      .concat(COMMENT_MODES)
	      .concat(RUBY_DEFAULT_CONTAINS)
	  };
	}

	ruby_1 = ruby;
	return ruby_1;
}

/*
Language: ERB (Embedded Ruby)
Requires: xml.js, ruby.js
Author: Lucas Mazza <lucastmazza@gmail.com>
Contributors: Kassio Borges <kassioborgesm@gmail.com>
Description: "Bridge" language defining fragments of Ruby in HTML within <% .. %>
Website: https://ruby-doc.org/stdlib-2.6.5/libdoc/erb/rdoc/ERB.html
Category: template
*/

var erb_1;
var hasRequiredErb;

function requireErb () {
	if (hasRequiredErb) return erb_1;
	hasRequiredErb = 1;
	/** @type LanguageFn */
	function erb(hljs) {
	  return {
	    name: 'ERB',
	    subLanguage: 'xml',
	    contains: [
	      hljs.COMMENT('<%#', '%>'),
	      {
	        begin: '<%[%=-]?',
	        end: '[%-]?%>',
	        subLanguage: 'ruby',
	        excludeBegin: true,
	        excludeEnd: true
	      }
	    ]
	  };
	}

	erb_1 = erb;
	return erb_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var erlangRepl_1;
var hasRequiredErlangRepl;

function requireErlangRepl () {
	if (hasRequiredErlangRepl) return erlangRepl_1;
	hasRequiredErlangRepl = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: Erlang REPL
	Author: Sergey Ignatov <sergey@ignatov.spb.su>
	Website: https://www.erlang.org
	Category: functional
	*/

	/** @type LanguageFn */
	function erlangRepl(hljs) {
	  return {
	    name: 'Erlang REPL',
	    keywords: {
	      built_in:
	        'spawn spawn_link self',
	      keyword:
	        'after and andalso|10 band begin bnot bor bsl bsr bxor case catch cond div end fun if ' +
	        'let not of or orelse|10 query receive rem try when xor'
	    },
	    contains: [
	      {
	        className: 'meta',
	        begin: '^[0-9]+> ',
	        relevance: 10
	      },
	      hljs.COMMENT('%', '$'),
	      {
	        className: 'number',
	        begin: '\\b(\\d+(_\\d+)*#[a-fA-F0-9]+(_[a-fA-F0-9]+)*|\\d+(_\\d+)*(\\.\\d+(_\\d+)*)?([eE][-+]?\\d+)?)',
	        relevance: 0
	      },
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      {
	        begin: concat(
	          /\?(::)?/,
	          /([A-Z]\w*)/, // at least one identifier
	          /((::)[A-Z]\w*)*/ // perhaps more
	        )
	      },
	      {
	        begin: '->'
	      },
	      {
	        begin: 'ok'
	      },
	      {
	        begin: '!'
	      },
	      {
	        begin: '(\\b[a-z\'][a-zA-Z0-9_\']*:[a-z\'][a-zA-Z0-9_\']*)|(\\b[a-z\'][a-zA-Z0-9_\']*)',
	        relevance: 0
	      },
	      {
	        begin: '[A-Z][a-zA-Z0-9_\']*',
	        relevance: 0
	      }
	    ]
	  };
	}

	erlangRepl_1 = erlangRepl;
	return erlangRepl_1;
}

/*
Language: Erlang
Description: Erlang is a general-purpose functional language, with strict evaluation, single assignment, and dynamic typing.
Author: Nikolay Zakharov <nikolay.desh@gmail.com>, Dmitry Kovega <arhibot@gmail.com>
Website: https://www.erlang.org
Category: functional
*/

var erlang_1;
var hasRequiredErlang;

function requireErlang () {
	if (hasRequiredErlang) return erlang_1;
	hasRequiredErlang = 1;
	/** @type LanguageFn */
	function erlang(hljs) {
	  const BASIC_ATOM_RE = '[a-z\'][a-zA-Z0-9_\']*';
	  const FUNCTION_NAME_RE = '(' + BASIC_ATOM_RE + ':' + BASIC_ATOM_RE + '|' + BASIC_ATOM_RE + ')';
	  const ERLANG_RESERVED = {
	    keyword:
	      'after and andalso|10 band begin bnot bor bsl bzr bxor case catch cond div end fun if ' +
	      'let not of orelse|10 query receive rem try when xor',
	    literal:
	      'false true'
	  };

	  const COMMENT = hljs.COMMENT('%', '$');
	  const NUMBER = {
	    className: 'number',
	    begin: '\\b(\\d+(_\\d+)*#[a-fA-F0-9]+(_[a-fA-F0-9]+)*|\\d+(_\\d+)*(\\.\\d+(_\\d+)*)?([eE][-+]?\\d+)?)',
	    relevance: 0
	  };
	  const NAMED_FUN = {
	    begin: 'fun\\s+' + BASIC_ATOM_RE + '/\\d+'
	  };
	  const FUNCTION_CALL = {
	    begin: FUNCTION_NAME_RE + '\\(',
	    end: '\\)',
	    returnBegin: true,
	    relevance: 0,
	    contains: [
	      {
	        begin: FUNCTION_NAME_RE,
	        relevance: 0
	      },
	      {
	        begin: '\\(',
	        end: '\\)',
	        endsWithParent: true,
	        returnEnd: true,
	        relevance: 0
	        // "contains" defined later
	      }
	    ]
	  };
	  const TUPLE = {
	    begin: /\{/,
	    end: /\}/,
	    relevance: 0
	    // "contains" defined later
	  };
	  const VAR1 = {
	    begin: '\\b_([A-Z][A-Za-z0-9_]*)?',
	    relevance: 0
	  };
	  const VAR2 = {
	    begin: '[A-Z][a-zA-Z0-9_]*',
	    relevance: 0
	  };
	  const RECORD_ACCESS = {
	    begin: '#' + hljs.UNDERSCORE_IDENT_RE,
	    relevance: 0,
	    returnBegin: true,
	    contains: [
	      {
	        begin: '#' + hljs.UNDERSCORE_IDENT_RE,
	        relevance: 0
	      },
	      {
	        begin: /\{/,
	        end: /\}/,
	        relevance: 0
	        // "contains" defined later
	      }
	    ]
	  };

	  const BLOCK_STATEMENTS = {
	    beginKeywords: 'fun receive if try case',
	    end: 'end',
	    keywords: ERLANG_RESERVED
	  };
	  BLOCK_STATEMENTS.contains = [
	    COMMENT,
	    NAMED_FUN,
	    hljs.inherit(hljs.APOS_STRING_MODE, {
	      className: ''
	    }),
	    BLOCK_STATEMENTS,
	    FUNCTION_CALL,
	    hljs.QUOTE_STRING_MODE,
	    NUMBER,
	    TUPLE,
	    VAR1,
	    VAR2,
	    RECORD_ACCESS
	  ];

	  const BASIC_MODES = [
	    COMMENT,
	    NAMED_FUN,
	    BLOCK_STATEMENTS,
	    FUNCTION_CALL,
	    hljs.QUOTE_STRING_MODE,
	    NUMBER,
	    TUPLE,
	    VAR1,
	    VAR2,
	    RECORD_ACCESS
	  ];
	  FUNCTION_CALL.contains[1].contains = BASIC_MODES;
	  TUPLE.contains = BASIC_MODES;
	  RECORD_ACCESS.contains[1].contains = BASIC_MODES;

	  const DIRECTIVES = [
	    "-module",
	    "-record",
	    "-undef",
	    "-export",
	    "-ifdef",
	    "-ifndef",
	    "-author",
	    "-copyright",
	    "-doc",
	    "-vsn",
	    "-import",
	    "-include",
	    "-include_lib",
	    "-compile",
	    "-define",
	    "-else",
	    "-endif",
	    "-file",
	    "-behaviour",
	    "-behavior",
	    "-spec"
	  ];

	  const PARAMS = {
	    className: 'params',
	    begin: '\\(',
	    end: '\\)',
	    contains: BASIC_MODES
	  };
	  return {
	    name: 'Erlang',
	    aliases: ['erl'],
	    keywords: ERLANG_RESERVED,
	    illegal: '(</|\\*=|\\+=|-=|/\\*|\\*/|\\(\\*|\\*\\))',
	    contains: [
	      {
	        className: 'function',
	        begin: '^' + BASIC_ATOM_RE + '\\s*\\(',
	        end: '->',
	        returnBegin: true,
	        illegal: '\\(|#|//|/\\*|\\\\|:|;',
	        contains: [
	          PARAMS,
	          hljs.inherit(hljs.TITLE_MODE, {
	            begin: BASIC_ATOM_RE
	          })
	        ],
	        starts: {
	          end: ';|\\.',
	          keywords: ERLANG_RESERVED,
	          contains: BASIC_MODES
	        }
	      },
	      COMMENT,
	      {
	        begin: '^-',
	        end: '\\.',
	        relevance: 0,
	        excludeEnd: true,
	        returnBegin: true,
	        keywords: {
	          $pattern: '-' + hljs.IDENT_RE,
	          keyword: DIRECTIVES.map(x => `${x}|1.5`).join(" ")
	        },
	        contains: [PARAMS]
	      },
	      NUMBER,
	      hljs.QUOTE_STRING_MODE,
	      RECORD_ACCESS,
	      VAR1,
	      VAR2,
	      TUPLE,
	      {
	        begin: /\.$/
	      } // relevance booster
	    ]
	  };
	}

	erlang_1 = erlang;
	return erlang_1;
}

/*
Language: Excel formulae
Author: Victor Zhou <OiCMudkips@users.noreply.github.com>
Description: Excel formulae
Website: https://products.office.com/en-us/excel/
*/

var excel_1;
var hasRequiredExcel;

function requireExcel () {
	if (hasRequiredExcel) return excel_1;
	hasRequiredExcel = 1;
	/** @type LanguageFn */
	function excel(hljs) {
	  return {
	    name: 'Excel formulae',
	    aliases: [
	      'xlsx',
	      'xls'
	    ],
	    case_insensitive: true,
	    // built-in functions imported from https://web.archive.org/web/20160513042710/https://support.office.com/en-us/article/Excel-functions-alphabetical-b3944572-255d-4efb-bb96-c6d90033e188
	    keywords: {
	      $pattern: /[a-zA-Z][\w\.]*/,
	      built_in: 'ABS ACCRINT ACCRINTM ACOS ACOSH ACOT ACOTH AGGREGATE ADDRESS AMORDEGRC AMORLINC AND ARABIC AREAS ASC ASIN ASINH ATAN ATAN2 ATANH AVEDEV AVERAGE AVERAGEA AVERAGEIF AVERAGEIFS BAHTTEXT BASE BESSELI BESSELJ BESSELK BESSELY BETADIST BETA.DIST BETAINV BETA.INV BIN2DEC BIN2HEX BIN2OCT BINOMDIST BINOM.DIST BINOM.DIST.RANGE BINOM.INV BITAND BITLSHIFT BITOR BITRSHIFT BITXOR CALL CEILING CEILING.MATH CEILING.PRECISE CELL CHAR CHIDIST CHIINV CHITEST CHISQ.DIST CHISQ.DIST.RT CHISQ.INV CHISQ.INV.RT CHISQ.TEST CHOOSE CLEAN CODE COLUMN COLUMNS COMBIN COMBINA COMPLEX CONCAT CONCATENATE CONFIDENCE CONFIDENCE.NORM CONFIDENCE.T CONVERT CORREL COS COSH COT COTH COUNT COUNTA COUNTBLANK COUNTIF COUNTIFS COUPDAYBS COUPDAYS COUPDAYSNC COUPNCD COUPNUM COUPPCD COVAR COVARIANCE.P COVARIANCE.S CRITBINOM CSC CSCH CUBEKPIMEMBER CUBEMEMBER CUBEMEMBERPROPERTY CUBERANKEDMEMBER CUBESET CUBESETCOUNT CUBEVALUE CUMIPMT CUMPRINC DATE DATEDIF DATEVALUE DAVERAGE DAY DAYS DAYS360 DB DBCS DCOUNT DCOUNTA DDB DEC2BIN DEC2HEX DEC2OCT DECIMAL DEGREES DELTA DEVSQ DGET DISC DMAX DMIN DOLLAR DOLLARDE DOLLARFR DPRODUCT DSTDEV DSTDEVP DSUM DURATION DVAR DVARP EDATE EFFECT ENCODEURL EOMONTH ERF ERF.PRECISE ERFC ERFC.PRECISE ERROR.TYPE EUROCONVERT EVEN EXACT EXP EXPON.DIST EXPONDIST FACT FACTDOUBLE FALSE|0 F.DIST FDIST F.DIST.RT FILTERXML FIND FINDB F.INV F.INV.RT FINV FISHER FISHERINV FIXED FLOOR FLOOR.MATH FLOOR.PRECISE FORECAST FORECAST.ETS FORECAST.ETS.CONFINT FORECAST.ETS.SEASONALITY FORECAST.ETS.STAT FORECAST.LINEAR FORMULATEXT FREQUENCY F.TEST FTEST FV FVSCHEDULE GAMMA GAMMA.DIST GAMMADIST GAMMA.INV GAMMAINV GAMMALN GAMMALN.PRECISE GAUSS GCD GEOMEAN GESTEP GETPIVOTDATA GROWTH HARMEAN HEX2BIN HEX2DEC HEX2OCT HLOOKUP HOUR HYPERLINK HYPGEOM.DIST HYPGEOMDIST IF IFERROR IFNA IFS IMABS IMAGINARY IMARGUMENT IMCONJUGATE IMCOS IMCOSH IMCOT IMCSC IMCSCH IMDIV IMEXP IMLN IMLOG10 IMLOG2 IMPOWER IMPRODUCT IMREAL IMSEC IMSECH IMSIN IMSINH IMSQRT IMSUB IMSUM IMTAN INDEX INDIRECT INFO INT INTERCEPT INTRATE IPMT IRR ISBLANK ISERR ISERROR ISEVEN ISFORMULA ISLOGICAL ISNA ISNONTEXT ISNUMBER ISODD ISREF ISTEXT ISO.CEILING ISOWEEKNUM ISPMT JIS KURT LARGE LCM LEFT LEFTB LEN LENB LINEST LN LOG LOG10 LOGEST LOGINV LOGNORM.DIST LOGNORMDIST LOGNORM.INV LOOKUP LOWER MATCH MAX MAXA MAXIFS MDETERM MDURATION MEDIAN MID MIDBs MIN MINIFS MINA MINUTE MINVERSE MIRR MMULT MOD MODE MODE.MULT MODE.SNGL MONTH MROUND MULTINOMIAL MUNIT N NA NEGBINOM.DIST NEGBINOMDIST NETWORKDAYS NETWORKDAYS.INTL NOMINAL NORM.DIST NORMDIST NORMINV NORM.INV NORM.S.DIST NORMSDIST NORM.S.INV NORMSINV NOT NOW NPER NPV NUMBERVALUE OCT2BIN OCT2DEC OCT2HEX ODD ODDFPRICE ODDFYIELD ODDLPRICE ODDLYIELD OFFSET OR PDURATION PEARSON PERCENTILE.EXC PERCENTILE.INC PERCENTILE PERCENTRANK.EXC PERCENTRANK.INC PERCENTRANK PERMUT PERMUTATIONA PHI PHONETIC PI PMT POISSON.DIST POISSON POWER PPMT PRICE PRICEDISC PRICEMAT PROB PRODUCT PROPER PV QUARTILE QUARTILE.EXC QUARTILE.INC QUOTIENT RADIANS RAND RANDBETWEEN RANK.AVG RANK.EQ RANK RATE RECEIVED REGISTER.ID REPLACE REPLACEB REPT RIGHT RIGHTB ROMAN ROUND ROUNDDOWN ROUNDUP ROW ROWS RRI RSQ RTD SEARCH SEARCHB SEC SECH SECOND SERIESSUM SHEET SHEETS SIGN SIN SINH SKEW SKEW.P SLN SLOPE SMALL SQL.REQUEST SQRT SQRTPI STANDARDIZE STDEV STDEV.P STDEV.S STDEVA STDEVP STDEVPA STEYX SUBSTITUTE SUBTOTAL SUM SUMIF SUMIFS SUMPRODUCT SUMSQ SUMX2MY2 SUMX2PY2 SUMXMY2 SWITCH SYD T TAN TANH TBILLEQ TBILLPRICE TBILLYIELD T.DIST T.DIST.2T T.DIST.RT TDIST TEXT TEXTJOIN TIME TIMEVALUE T.INV T.INV.2T TINV TODAY TRANSPOSE TREND TRIM TRIMMEAN TRUE|0 TRUNC T.TEST TTEST TYPE UNICHAR UNICODE UPPER VALUE VAR VAR.P VAR.S VARA VARP VARPA VDB VLOOKUP WEBSERVICE WEEKDAY WEEKNUM WEIBULL WEIBULL.DIST WORKDAY WORKDAY.INTL XIRR XNPV XOR YEAR YEARFRAC YIELD YIELDDISC YIELDMAT Z.TEST ZTEST'
	    },
	    contains: [
	      {
	        /* matches a beginning equal sign found in Excel formula examples */
	        begin: /^=/,
	        end: /[^=]/,
	        returnEnd: true,
	        illegal: /=/, /* only allow single equal sign at front of line */
	        relevance: 10
	      },
	      /* technically, there can be more than 2 letters in column names, but this prevents conflict with some keywords */
	      {
	        /* matches a reference to a single cell */
	        className: 'symbol',
	        begin: /\b[A-Z]{1,2}\d+\b/,
	        end: /[^\d]/,
	        excludeEnd: true,
	        relevance: 0
	      },
	      {
	        /* matches a reference to a range of cells */
	        className: 'symbol',
	        begin: /[A-Z]{0,2}\d*:[A-Z]{0,2}\d*/,
	        relevance: 0
	      },
	      hljs.BACKSLASH_ESCAPE,
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'number',
	        begin: hljs.NUMBER_RE + '(%)?',
	        relevance: 0
	      },
	      /* Excel formula comments are done by putting the comment in a function call to N() */
	      hljs.COMMENT(/\bN\(/, /\)/,
	        {
	          excludeBegin: true,
	          excludeEnd: true,
	          illegal: /\n/
	        })
	    ]
	  };
	}

	excel_1 = excel;
	return excel_1;
}

/*
Language: FIX
Author: Brent Bradbury <brent@brentium.com>
*/

var fix_1;
var hasRequiredFix;

function requireFix () {
	if (hasRequiredFix) return fix_1;
	hasRequiredFix = 1;
	/** @type LanguageFn */
	function fix(hljs) {
	  return {
	    name: 'FIX',
	    contains: [{
	      begin: /[^\u2401\u0001]+/,
	      end: /[\u2401\u0001]/,
	      excludeEnd: true,
	      returnBegin: true,
	      returnEnd: false,
	      contains: [
	        {
	          begin: /([^\u2401\u0001=]+)/,
	          end: /=([^\u2401\u0001=]+)/,
	          returnEnd: true,
	          returnBegin: false,
	          className: 'attr'
	        },
	        {
	          begin: /=/,
	          end: /([\u2401\u0001])/,
	          excludeEnd: true,
	          excludeBegin: true,
	          className: 'string'
	        }
	      ]
	    }],
	    case_insensitive: true
	  };
	}

	fix_1 = fix;
	return fix_1;
}

/*
 Language: Flix
 Category: functional
 Author: Magnus Madsen <mmadsen@uwaterloo.ca>
 Website: https://flix.dev/
 */

var flix_1;
var hasRequiredFlix;

function requireFlix () {
	if (hasRequiredFlix) return flix_1;
	hasRequiredFlix = 1;
	/** @type LanguageFn */
	function flix(hljs) {
	  const CHAR = {
	    className: 'string',
	    begin: /'(.|\\[xXuU][a-zA-Z0-9]+)'/
	  };

	  const STRING = {
	    className: 'string',
	    variants: [{
	      begin: '"',
	      end: '"'
	    }]
	  };

	  const NAME = {
	    className: 'title',
	    relevance: 0,
	    begin: /[^0-9\n\t "'(),.`{}\[\]:;][^\n\t "'(),.`{}\[\]:;]+|[^0-9\n\t "'(),.`{}\[\]:;=]/
	  };

	  const METHOD = {
	    className: 'function',
	    beginKeywords: 'def',
	    end: /[:={\[(\n;]/,
	    excludeEnd: true,
	    contains: [NAME]
	  };

	  return {
	    name: 'Flix',
	    keywords: {
	      literal: 'true false',
	      keyword: 'case class def else enum if impl import in lat rel index let match namespace switch type yield with'
	    },
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      CHAR,
	      STRING,
	      METHOD,
	      hljs.C_NUMBER_MODE
	    ]
	  };
	}

	flix_1 = flix;
	return flix_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var fortran_1;
var hasRequiredFortran;

function requireFortran () {
	if (hasRequiredFortran) return fortran_1;
	hasRequiredFortran = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: Fortran
	Author: Anthony Scemama <scemama@irsamc.ups-tlse.fr>
	Website: https://en.wikipedia.org/wiki/Fortran
	Category: scientific
	*/

	/** @type LanguageFn */
	function fortran(hljs) {
	  const PARAMS = {
	    className: 'params',
	    begin: '\\(',
	    end: '\\)'
	  };

	  const COMMENT = {
	    variants: [
	      hljs.COMMENT('!', '$', {
	        relevance: 0
	      }),
	      // allow FORTRAN 77 style comments
	      hljs.COMMENT('^C[ ]', '$', {
	        relevance: 0
	      }),
	      hljs.COMMENT('^C$', '$', {
	        relevance: 0
	      })
	    ]
	  };

	  // regex in both fortran and irpf90 should match
	  const OPTIONAL_NUMBER_SUFFIX = /(_[a-z_\d]+)?/;
	  const OPTIONAL_NUMBER_EXP = /([de][+-]?\d+)?/;
	  const NUMBER = {
	    className: 'number',
	    variants: [
	      {
	        begin: concat(/\b\d+/, /\.(\d*)/, OPTIONAL_NUMBER_EXP, OPTIONAL_NUMBER_SUFFIX)
	      },
	      {
	        begin: concat(/\b\d+/, OPTIONAL_NUMBER_EXP, OPTIONAL_NUMBER_SUFFIX)
	      },
	      {
	        begin: concat(/\.\d+/, OPTIONAL_NUMBER_EXP, OPTIONAL_NUMBER_SUFFIX)
	      }
	    ],
	    relevance: 0
	  };

	  const FUNCTION_DEF = {
	    className: 'function',
	    beginKeywords: 'subroutine function program',
	    illegal: '[${=\\n]',
	    contains: [
	      hljs.UNDERSCORE_TITLE_MODE,
	      PARAMS
	    ]
	  };

	  const STRING = {
	    className: 'string',
	    relevance: 0,
	    variants: [
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE
	    ]
	  };

	  const KEYWORDS = {
	    literal: '.False. .True.',
	    keyword: 'kind do concurrent local shared while private call intrinsic where elsewhere ' +
	      'type endtype endmodule endselect endinterface end enddo endif if forall endforall only contains default return stop then block endblock endassociate ' +
	      'public subroutine|10 function program .and. .or. .not. .le. .eq. .ge. .gt. .lt. ' +
	      'goto save else use module select case ' +
	      'access blank direct exist file fmt form formatted iostat name named nextrec number opened rec recl sequential status unformatted unit ' +
	      'continue format pause cycle exit ' +
	      'c_null_char c_alert c_backspace c_form_feed flush wait decimal round iomsg ' +
	      'synchronous nopass non_overridable pass protected volatile abstract extends import ' +
	      'non_intrinsic value deferred generic final enumerator class associate bind enum ' +
	      'c_int c_short c_long c_long_long c_signed_char c_size_t c_int8_t c_int16_t c_int32_t c_int64_t c_int_least8_t c_int_least16_t ' +
	      'c_int_least32_t c_int_least64_t c_int_fast8_t c_int_fast16_t c_int_fast32_t c_int_fast64_t c_intmax_t C_intptr_t c_float c_double ' +
	      'c_long_double c_float_complex c_double_complex c_long_double_complex c_bool c_char c_null_ptr c_null_funptr ' +
	      'c_new_line c_carriage_return c_horizontal_tab c_vertical_tab iso_c_binding c_loc c_funloc c_associated  c_f_pointer ' +
	      'c_ptr c_funptr iso_fortran_env character_storage_size error_unit file_storage_size input_unit iostat_end iostat_eor ' +
	      'numeric_storage_size output_unit c_f_procpointer ieee_arithmetic ieee_support_underflow_control ' +
	      'ieee_get_underflow_mode ieee_set_underflow_mode newunit contiguous recursive ' +
	      'pad position action delim readwrite eor advance nml interface procedure namelist include sequence elemental pure impure ' +
	      'integer real character complex logical codimension dimension allocatable|10 parameter ' +
	      'external implicit|10 none double precision assign intent optional pointer ' +
	      'target in out common equivalence data',
	    built_in: 'alog alog10 amax0 amax1 amin0 amin1 amod cabs ccos cexp clog csin csqrt dabs dacos dasin datan datan2 dcos dcosh ddim dexp dint ' +
	      'dlog dlog10 dmax1 dmin1 dmod dnint dsign dsin dsinh dsqrt dtan dtanh float iabs idim idint idnint ifix isign max0 max1 min0 min1 sngl ' +
	      'algama cdabs cdcos cdexp cdlog cdsin cdsqrt cqabs cqcos cqexp cqlog cqsin cqsqrt dcmplx dconjg derf derfc dfloat dgamma dimag dlgama ' +
	      'iqint qabs qacos qasin qatan qatan2 qcmplx qconjg qcos qcosh qdim qerf qerfc qexp qgamma qimag qlgama qlog qlog10 qmax1 qmin1 qmod ' +
	      'qnint qsign qsin qsinh qsqrt qtan qtanh abs acos aimag aint anint asin atan atan2 char cmplx conjg cos cosh exp ichar index int log ' +
	      'log10 max min nint sign sin sinh sqrt tan tanh print write dim lge lgt lle llt mod nullify allocate deallocate ' +
	      'adjustl adjustr all allocated any associated bit_size btest ceiling count cshift date_and_time digits dot_product ' +
	      'eoshift epsilon exponent floor fraction huge iand ibclr ibits ibset ieor ior ishft ishftc lbound len_trim matmul ' +
	      'maxexponent maxloc maxval merge minexponent minloc minval modulo mvbits nearest pack present product ' +
	      'radix random_number random_seed range repeat reshape rrspacing scale scan selected_int_kind selected_real_kind ' +
	      'set_exponent shape size spacing spread sum system_clock tiny transpose trim ubound unpack verify achar iachar transfer ' +
	      'dble entry dprod cpu_time command_argument_count get_command get_command_argument get_environment_variable is_iostat_end ' +
	      'ieee_arithmetic ieee_support_underflow_control ieee_get_underflow_mode ieee_set_underflow_mode ' +
	      'is_iostat_eor move_alloc new_line selected_char_kind same_type_as extends_type_of ' +
	      'acosh asinh atanh bessel_j0 bessel_j1 bessel_jn bessel_y0 bessel_y1 bessel_yn erf erfc erfc_scaled gamma log_gamma hypot norm2 ' +
	      'atomic_define atomic_ref execute_command_line leadz trailz storage_size merge_bits ' +
	      'bge bgt ble blt dshiftl dshiftr findloc iall iany iparity image_index lcobound ucobound maskl maskr ' +
	      'num_images parity popcnt poppar shifta shiftl shiftr this_image sync change team co_broadcast co_max co_min co_sum co_reduce'
	  };
	  return {
	    name: 'Fortran',
	    case_insensitive: true,
	    aliases: [
	      'f90',
	      'f95'
	    ],
	    keywords: KEYWORDS,
	    illegal: /\/\*/,
	    contains: [
	      STRING,
	      FUNCTION_DEF,
	      // allow `C = value` for assignments so they aren't misdetected
	      // as Fortran 77 style comments
	      {
	        begin: /^C\s*=(?!=)/,
	        relevance: 0
	      },
	      COMMENT,
	      NUMBER
	    ]
	  };
	}

	fortran_1 = fortran;
	return fortran_1;
}

/*
Language: F#
Author: Jonas Follesø <jonas@follesoe.no>
Contributors: Troy Kershaw <hello@troykershaw.com>, Henrik Feldt <henrik@haf.se>
Website: https://docs.microsoft.com/en-us/dotnet/fsharp/
Category: functional
*/

var fsharp_1;
var hasRequiredFsharp;

function requireFsharp () {
	if (hasRequiredFsharp) return fsharp_1;
	hasRequiredFsharp = 1;
	/** @type LanguageFn */
	function fsharp(hljs) {
	  const TYPEPARAM = {
	    begin: '<',
	    end: '>',
	    contains: [
	      hljs.inherit(hljs.TITLE_MODE, {
	        begin: /'[a-zA-Z0-9_]+/
	      })
	    ]
	  };

	  return {
	    name: 'F#',
	    aliases: ['fs'],
	    keywords:
	      'abstract and as assert base begin class default delegate do done ' +
	      'downcast downto elif else end exception extern false finally for ' +
	      'fun function global if in inherit inline interface internal lazy let ' +
	      'match member module mutable namespace new null of open or ' +
	      'override private public rec return sig static struct then to ' +
	      'true try type upcast use val void when while with yield',
	    illegal: /\/\*/,
	    contains: [
	      {
	        // monad builder keywords (matches before non-bang kws)
	        className: 'keyword',
	        begin: /\b(yield|return|let|do)!/
	      },
	      {
	        className: 'string',
	        begin: '@"',
	        end: '"',
	        contains: [
	          {
	            begin: '""'
	          }
	        ]
	      },
	      {
	        className: 'string',
	        begin: '"""',
	        end: '"""'
	      },
	      hljs.COMMENT('\\(\\*(\\s)', '\\*\\)', {
	        contains: ["self"]
	      }),
	      {
	        className: 'class',
	        beginKeywords: 'type',
	        end: '\\(|=|$',
	        excludeEnd: true,
	        contains: [
	          hljs.UNDERSCORE_TITLE_MODE,
	          TYPEPARAM
	        ]
	      },
	      {
	        className: 'meta',
	        begin: '\\[<',
	        end: '>\\]',
	        relevance: 10
	      },
	      {
	        className: 'symbol',
	        begin: '\\B(\'[A-Za-z])\\b',
	        contains: [hljs.BACKSLASH_ESCAPE]
	      },
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.inherit(hljs.QUOTE_STRING_MODE, {
	        illegal: null
	      }),
	      hljs.C_NUMBER_MODE
	    ]
	  };
	}

	fsharp_1 = fsharp;
	return fsharp_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var gams_1;
var hasRequiredGams;

function requireGams () {
	if (hasRequiredGams) return gams_1;
	hasRequiredGams = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function anyNumberOfTimes(re) {
	  return concat('(', re, ')*');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/** @type LanguageFn */
	function gams(hljs) {
	  const KEYWORDS = {
	    keyword:
	      'abort acronym acronyms alias all and assign binary card diag display ' +
	      'else eq file files for free ge gt if integer le loop lt maximizing ' +
	      'minimizing model models ne negative no not option options or ord ' +
	      'positive prod put putpage puttl repeat sameas semicont semiint smax ' +
	      'smin solve sos1 sos2 sum system table then until using while xor yes',
	    literal:
	      'eps inf na',
	    built_in:
	      'abs arccos arcsin arctan arctan2 Beta betaReg binomial ceil centropy ' +
	      'cos cosh cvPower div div0 eDist entropy errorf execSeed exp fact ' +
	      'floor frac gamma gammaReg log logBeta logGamma log10 log2 mapVal max ' +
	      'min mod ncpCM ncpF ncpVUpow ncpVUsin normal pi poly power ' +
	      'randBinomial randLinear randTriangle round rPower sigmoid sign ' +
	      'signPower sin sinh slexp sllog10 slrec sqexp sqlog10 sqr sqrec sqrt ' +
	      'tan tanh trunc uniform uniformInt vcPower bool_and bool_eqv bool_imp ' +
	      'bool_not bool_or bool_xor ifThen rel_eq rel_ge rel_gt rel_le rel_lt ' +
	      'rel_ne gday gdow ghour gleap gmillisec gminute gmonth gsecond gyear ' +
	      'jdate jnow jstart jtime errorLevel execError gamsRelease gamsVersion ' +
	      'handleCollect handleDelete handleStatus handleSubmit heapFree ' +
	      'heapLimit heapSize jobHandle jobKill jobStatus jobTerminate ' +
	      'licenseLevel licenseStatus maxExecError sleep timeClose timeComp ' +
	      'timeElapsed timeExec timeStart'
	  };
	  const PARAMS = {
	    className: 'params',
	    begin: /\(/,
	    end: /\)/,
	    excludeBegin: true,
	    excludeEnd: true
	  };
	  const SYMBOLS = {
	    className: 'symbol',
	    variants: [
	      {
	        begin: /=[lgenxc]=/
	      },
	      {
	        begin: /\$/
	      }
	    ]
	  };
	  const QSTR = { // One-line quoted comment string
	    className: 'comment',
	    variants: [
	      {
	        begin: '\'',
	        end: '\''
	      },
	      {
	        begin: '"',
	        end: '"'
	      }
	    ],
	    illegal: '\\n',
	    contains: [hljs.BACKSLASH_ESCAPE]
	  };
	  const ASSIGNMENT = {
	    begin: '/',
	    end: '/',
	    keywords: KEYWORDS,
	    contains: [
	      QSTR,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.C_NUMBER_MODE
	    ]
	  };
	  const COMMENT_WORD = /[a-z0-9&#*=?@\\><:,()$[\]_.{}!+%^-]+/;
	  const DESCTEXT = { // Parameter/set/variable description text
	    begin: /[a-z][a-z0-9_]*(\([a-z0-9_, ]*\))?[ \t]+/,
	    excludeBegin: true,
	    end: '$',
	    endsWithParent: true,
	    contains: [
	      QSTR,
	      ASSIGNMENT,
	      {
	        className: 'comment',
	        // one comment word, then possibly more
	        begin: concat(
	          COMMENT_WORD,
	          // [ ] because \s would be too broad (matching newlines)
	          anyNumberOfTimes(concat(/[ ]+/, COMMENT_WORD))
	        ),
	        relevance: 0
	      }
	    ]
	  };

	  return {
	    name: 'GAMS',
	    aliases: ['gms'],
	    case_insensitive: true,
	    keywords: KEYWORDS,
	    contains: [
	      hljs.COMMENT(/^\$ontext/, /^\$offtext/),
	      {
	        className: 'meta',
	        begin: '^\\$[a-z0-9]+',
	        end: '$',
	        returnBegin: true,
	        contains: [
	          {
	            className: 'meta-keyword',
	            begin: '^\\$[a-z0-9]+'
	          }
	        ]
	      },
	      hljs.COMMENT('^\\*', '$'),
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.APOS_STRING_MODE,
	      // Declarations
	      {
	        beginKeywords:
	          'set sets parameter parameters variable variables ' +
	          'scalar scalars equation equations',
	        end: ';',
	        contains: [
	          hljs.COMMENT('^\\*', '$'),
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE,
	          hljs.QUOTE_STRING_MODE,
	          hljs.APOS_STRING_MODE,
	          ASSIGNMENT,
	          DESCTEXT
	        ]
	      },
	      { // table environment
	        beginKeywords: 'table',
	        end: ';',
	        returnBegin: true,
	        contains: [
	          { // table header row
	            beginKeywords: 'table',
	            end: '$',
	            contains: [DESCTEXT]
	          },
	          hljs.COMMENT('^\\*', '$'),
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE,
	          hljs.QUOTE_STRING_MODE,
	          hljs.APOS_STRING_MODE,
	          hljs.C_NUMBER_MODE
	          // Table does not contain DESCTEXT or ASSIGNMENT
	        ]
	      },
	      // Function definitions
	      {
	        className: 'function',
	        begin: /^[a-z][a-z0-9_,\-+' ()$]+\.{2}/,
	        returnBegin: true,
	        contains: [
	          { // Function title
	            className: 'title',
	            begin: /^[a-z0-9_]+/
	          },
	          PARAMS,
	          SYMBOLS
	        ]
	      },
	      hljs.C_NUMBER_MODE,
	      SYMBOLS
	    ]
	  };
	}

	gams_1 = gams;
	return gams_1;
}

/*
Language: GAUSS
Author: Matt Evans <matt@aptech.com>
Description: GAUSS Mathematical and Statistical language
Website: https://www.aptech.com
Category: scientific
*/

var gauss_1;
var hasRequiredGauss;

function requireGauss () {
	if (hasRequiredGauss) return gauss_1;
	hasRequiredGauss = 1;
	function gauss(hljs) {
	  const KEYWORDS = {
	    keyword: 'bool break call callexe checkinterrupt clear clearg closeall cls comlog compile ' +
	              'continue create debug declare delete disable dlibrary dllcall do dos ed edit else ' +
	              'elseif enable end endfor endif endp endo errorlog errorlogat expr external fn ' +
	              'for format goto gosub graph if keyword let lib library line load loadarray loadexe ' +
	              'loadf loadk loadm loadp loads loadx local locate loopnextindex lprint lpwidth lshow ' +
	              'matrix msym ndpclex new open output outwidth plot plotsym pop prcsn print ' +
	              'printdos proc push retp return rndcon rndmod rndmult rndseed run save saveall screen ' +
	              'scroll setarray show sparse stop string struct system trace trap threadfor ' +
	              'threadendfor threadbegin threadjoin threadstat threadend until use while winprint ' +
	              'ne ge le gt lt and xor or not eq eqv',
	    built_in: 'abs acf aconcat aeye amax amean AmericanBinomCall AmericanBinomCall_Greeks AmericanBinomCall_ImpVol ' +
	              'AmericanBinomPut AmericanBinomPut_Greeks AmericanBinomPut_ImpVol AmericanBSCall AmericanBSCall_Greeks ' +
	              'AmericanBSCall_ImpVol AmericanBSPut AmericanBSPut_Greeks AmericanBSPut_ImpVol amin amult annotationGetDefaults ' +
	              'annotationSetBkd annotationSetFont annotationSetLineColor annotationSetLineStyle annotationSetLineThickness ' +
	              'annualTradingDays arccos arcsin areshape arrayalloc arrayindex arrayinit arraytomat asciiload asclabel astd ' +
	              'astds asum atan atan2 atranspose axmargin balance band bandchol bandcholsol bandltsol bandrv bandsolpd bar ' +
	              'base10 begwind besselj bessely beta box boxcox cdfBeta cdfBetaInv cdfBinomial cdfBinomialInv cdfBvn cdfBvn2 ' +
	              'cdfBvn2e cdfCauchy cdfCauchyInv cdfChic cdfChii cdfChinc cdfChincInv cdfExp cdfExpInv cdfFc cdfFnc cdfFncInv ' +
	              'cdfGam cdfGenPareto cdfHyperGeo cdfLaplace cdfLaplaceInv cdfLogistic cdfLogisticInv cdfmControlCreate cdfMvn ' +
	              'cdfMvn2e cdfMvnce cdfMvne cdfMvt2e cdfMvtce cdfMvte cdfN cdfN2 cdfNc cdfNegBinomial cdfNegBinomialInv cdfNi ' +
	              'cdfPoisson cdfPoissonInv cdfRayleigh cdfRayleighInv cdfTc cdfTci cdfTnc cdfTvn cdfWeibull cdfWeibullInv cdir ' +
	              'ceil ChangeDir chdir chiBarSquare chol choldn cholsol cholup chrs close code cols colsf combinate combinated ' +
	              'complex con cond conj cons ConScore contour conv convertsatostr convertstrtosa corrm corrms corrvc corrx corrxs ' +
	              'cos cosh counts countwts crossprd crout croutp csrcol csrlin csvReadM csvReadSA cumprodc cumsumc curve cvtos ' +
	              'datacreate datacreatecomplex datalist dataload dataloop dataopen datasave date datestr datestring datestrymd ' +
	              'dayinyr dayofweek dbAddDatabase dbClose dbCommit dbCreateQuery dbExecQuery dbGetConnectOptions dbGetDatabaseName ' +
	              'dbGetDriverName dbGetDrivers dbGetHostName dbGetLastErrorNum dbGetLastErrorText dbGetNumericalPrecPolicy ' +
	              'dbGetPassword dbGetPort dbGetTableHeaders dbGetTables dbGetUserName dbHasFeature dbIsDriverAvailable dbIsOpen ' +
	              'dbIsOpenError dbOpen dbQueryBindValue dbQueryClear dbQueryCols dbQueryExecPrepared dbQueryFetchAllM dbQueryFetchAllSA ' +
	              'dbQueryFetchOneM dbQueryFetchOneSA dbQueryFinish dbQueryGetBoundValue dbQueryGetBoundValues dbQueryGetField ' +
	              'dbQueryGetLastErrorNum dbQueryGetLastErrorText dbQueryGetLastInsertID dbQueryGetLastQuery dbQueryGetPosition ' +
	              'dbQueryIsActive dbQueryIsForwardOnly dbQueryIsNull dbQueryIsSelect dbQueryIsValid dbQueryPrepare dbQueryRows ' +
	              'dbQuerySeek dbQuerySeekFirst dbQuerySeekLast dbQuerySeekNext dbQuerySeekPrevious dbQuerySetForwardOnly ' +
	              'dbRemoveDatabase dbRollback dbSetConnectOptions dbSetDatabaseName dbSetHostName dbSetNumericalPrecPolicy ' +
	              'dbSetPort dbSetUserName dbTransaction DeleteFile delif delrows denseToSp denseToSpRE denToZero design det detl ' +
	              'dfft dffti diag diagrv digamma doswin DOSWinCloseall DOSWinOpen dotfeq dotfeqmt dotfge dotfgemt dotfgt dotfgtmt ' +
	              'dotfle dotflemt dotflt dotfltmt dotfne dotfnemt draw drop dsCreate dstat dstatmt dstatmtControlCreate dtdate dtday ' +
	              'dttime dttodtv dttostr dttoutc dtvnormal dtvtodt dtvtoutc dummy dummybr dummydn eig eigh eighv eigv elapsedTradingDays ' +
	              'endwind envget eof eqSolve eqSolvemt eqSolvemtControlCreate eqSolvemtOutCreate eqSolveset erf erfc erfccplx erfcplx error ' +
	              'etdays ethsec etstr EuropeanBinomCall EuropeanBinomCall_Greeks EuropeanBinomCall_ImpVol EuropeanBinomPut ' +
	              'EuropeanBinomPut_Greeks EuropeanBinomPut_ImpVol EuropeanBSCall EuropeanBSCall_Greeks EuropeanBSCall_ImpVol ' +
	              'EuropeanBSPut EuropeanBSPut_Greeks EuropeanBSPut_ImpVol exctsmpl exec execbg exp extern eye fcheckerr fclearerr feq ' +
	              'feqmt fflush fft ffti fftm fftmi fftn fge fgemt fgets fgetsa fgetsat fgetst fgt fgtmt fileinfo filesa fle flemt ' +
	              'floor flt fltmt fmod fne fnemt fonts fopen formatcv formatnv fputs fputst fseek fstrerror ftell ftocv ftos ftostrC ' +
	              'gamma gammacplx gammaii gausset gdaAppend gdaCreate gdaDStat gdaDStatMat gdaGetIndex gdaGetName gdaGetNames gdaGetOrders ' +
	              'gdaGetType gdaGetTypes gdaGetVarInfo gdaIsCplx gdaLoad gdaPack gdaRead gdaReadByIndex gdaReadSome gdaReadSparse ' +
	              'gdaReadStruct gdaReportVarInfo gdaSave gdaUpdate gdaUpdateAndPack gdaVars gdaWrite gdaWrite32 gdaWriteSome getarray ' +
	              'getdims getf getGAUSShome getmatrix getmatrix4D getname getnamef getNextTradingDay getNextWeekDay getnr getorders ' +
	              'getpath getPreviousTradingDay getPreviousWeekDay getRow getscalar3D getscalar4D getTrRow getwind glm gradcplx gradMT ' +
	              'gradMTm gradMTT gradMTTm gradp graphprt graphset hasimag header headermt hess hessMT hessMTg hessMTgw hessMTm ' +
	              'hessMTmw hessMTT hessMTTg hessMTTgw hessMTTm hessMTw hessp hist histf histp hsec imag indcv indexcat indices indices2 ' +
	              'indicesf indicesfn indnv indsav integrate1d integrateControlCreate intgrat2 intgrat3 inthp1 inthp2 inthp3 inthp4 ' +
	              'inthpControlCreate intquad1 intquad2 intquad3 intrleav intrleavsa intrsect intsimp inv invpd invswp iscplx iscplxf ' +
	              'isden isinfnanmiss ismiss key keyav keyw lag lag1 lagn lapEighb lapEighi lapEighvb lapEighvi lapgEig lapgEigh lapgEighv ' +
	              'lapgEigv lapgSchur lapgSvdcst lapgSvds lapgSvdst lapSvdcusv lapSvds lapSvdusv ldlp ldlsol linSolve listwise ln lncdfbvn ' +
	              'lncdfbvn2 lncdfmvn lncdfn lncdfn2 lncdfnc lnfact lngammacplx lnpdfmvn lnpdfmvt lnpdfn lnpdft loadd loadstruct loadwind ' +
	              'loess loessmt loessmtControlCreate log loglog logx logy lower lowmat lowmat1 ltrisol lu lusol machEpsilon make makevars ' +
	              'makewind margin matalloc matinit mattoarray maxbytes maxc maxindc maxv maxvec mbesselei mbesselei0 mbesselei1 mbesseli ' +
	              'mbesseli0 mbesseli1 meanc median mergeby mergevar minc minindc minv miss missex missrv moment momentd movingave ' +
	              'movingaveExpwgt movingaveWgt nextindex nextn nextnevn nextwind ntos null null1 numCombinations ols olsmt olsmtControlCreate ' +
	              'olsqr olsqr2 olsqrmt ones optn optnevn orth outtyp pacf packedToSp packr parse pause pdfCauchy pdfChi pdfExp pdfGenPareto ' +
	              'pdfHyperGeo pdfLaplace pdfLogistic pdfn pdfPoisson pdfRayleigh pdfWeibull pi pinv pinvmt plotAddArrow plotAddBar plotAddBox ' +
	              'plotAddHist plotAddHistF plotAddHistP plotAddPolar plotAddScatter plotAddShape plotAddTextbox plotAddTS plotAddXY plotArea ' +
	              'plotBar plotBox plotClearLayout plotContour plotCustomLayout plotGetDefaults plotHist plotHistF plotHistP plotLayout ' +
	              'plotLogLog plotLogX plotLogY plotOpenWindow plotPolar plotSave plotScatter plotSetAxesPen plotSetBar plotSetBarFill ' +
	              'plotSetBarStacked plotSetBkdColor plotSetFill plotSetGrid plotSetLegend plotSetLineColor plotSetLineStyle plotSetLineSymbol ' +
	              'plotSetLineThickness plotSetNewWindow plotSetTitle plotSetWhichYAxis plotSetXAxisShow plotSetXLabel plotSetXRange ' +
	              'plotSetXTicInterval plotSetXTicLabel plotSetYAxisShow plotSetYLabel plotSetYRange plotSetZAxisShow plotSetZLabel ' +
	              'plotSurface plotTS plotXY polar polychar polyeval polygamma polyint polymake polymat polymroot polymult polyroot ' +
	              'pqgwin previousindex princomp printfm printfmt prodc psi putarray putf putvals pvCreate pvGetIndex pvGetParNames ' +
	              'pvGetParVector pvLength pvList pvPack pvPacki pvPackm pvPackmi pvPacks pvPacksi pvPacksm pvPacksmi pvPutParVector ' +
	              'pvTest pvUnpack QNewton QNewtonmt QNewtonmtControlCreate QNewtonmtOutCreate QNewtonSet QProg QProgmt QProgmtInCreate ' +
	              'qqr qqre qqrep qr qre qrep qrsol qrtsol qtyr qtyre qtyrep quantile quantiled qyr qyre qyrep qz rank rankindx readr ' +
	              'real reclassify reclassifyCuts recode recserar recsercp recserrc rerun rescale reshape rets rev rfft rffti rfftip rfftn ' +
	              'rfftnp rfftp rndBernoulli rndBeta rndBinomial rndCauchy rndChiSquare rndCon rndCreateState rndExp rndGamma rndGeo rndGumbel ' +
	              'rndHyperGeo rndi rndKMbeta rndKMgam rndKMi rndKMn rndKMnb rndKMp rndKMu rndKMvm rndLaplace rndLCbeta rndLCgam rndLCi rndLCn ' +
	              'rndLCnb rndLCp rndLCu rndLCvm rndLogNorm rndMTu rndMVn rndMVt rndn rndnb rndNegBinomial rndp rndPoisson rndRayleigh ' +
	              'rndStateSkip rndu rndvm rndWeibull rndWishart rotater round rows rowsf rref sampleData satostrC saved saveStruct savewind ' +
	              'scale scale3d scalerr scalinfnanmiss scalmiss schtoc schur searchsourcepath seekr select selif seqa seqm setdif setdifsa ' +
	              'setvars setvwrmode setwind shell shiftr sin singleindex sinh sleep solpd sortc sortcc sortd sorthc sorthcc sortind ' +
	              'sortindc sortmc sortr sortrc spBiconjGradSol spChol spConjGradSol spCreate spDenseSubmat spDiagRvMat spEigv spEye spLDL ' +
	              'spline spLU spNumNZE spOnes spreadSheetReadM spreadSheetReadSA spreadSheetWrite spScale spSubmat spToDense spTrTDense ' +
	              'spTScalar spZeros sqpSolve sqpSolveMT sqpSolveMTControlCreate sqpSolveMTlagrangeCreate sqpSolveMToutCreate sqpSolveSet ' +
	              'sqrt statements stdc stdsc stocv stof strcombine strindx strlen strput strrindx strsect strsplit strsplitPad strtodt ' +
	              'strtof strtofcplx strtriml strtrimr strtrunc strtruncl strtruncpad strtruncr submat subscat substute subvec sumc sumr ' +
	              'surface svd svd1 svd2 svdcusv svds svdusv sysstate tab tan tanh tempname ' +
	              'time timedt timestr timeutc title tkf2eps tkf2ps tocart todaydt toeplitz token topolar trapchk ' +
	              'trigamma trimr trunc type typecv typef union unionsa uniqindx uniqindxsa unique uniquesa upmat upmat1 upper utctodt ' +
	              'utctodtv utrisol vals varCovMS varCovXS varget vargetl varmall varmares varput varputl vartypef vcm vcms vcx vcxs ' +
	              'vec vech vecr vector vget view viewxyz vlist vnamecv volume vput vread vtypecv wait waitc walkindex where window ' +
	              'writer xlabel xlsGetSheetCount xlsGetSheetSize xlsGetSheetTypes xlsMakeRange xlsReadM xlsReadSA xlsWrite xlsWriteM ' +
	              'xlsWriteSA xpnd xtics xy xyz ylabel ytics zeros zeta zlabel ztics cdfEmpirical dot h5create h5open h5read h5readAttribute ' +
	              'h5write h5writeAttribute ldl plotAddErrorBar plotAddSurface plotCDFEmpirical plotSetColormap plotSetContourLabels ' +
	              'plotSetLegendFont plotSetTextInterpreter plotSetXTicCount plotSetYTicCount plotSetZLevels powerm strjoin sylvester ' +
	              'strtrim',
	    literal: 'DB_AFTER_LAST_ROW DB_ALL_TABLES DB_BATCH_OPERATIONS DB_BEFORE_FIRST_ROW DB_BLOB DB_EVENT_NOTIFICATIONS ' +
	             'DB_FINISH_QUERY DB_HIGH_PRECISION DB_LAST_INSERT_ID DB_LOW_PRECISION_DOUBLE DB_LOW_PRECISION_INT32 ' +
	             'DB_LOW_PRECISION_INT64 DB_LOW_PRECISION_NUMBERS DB_MULTIPLE_RESULT_SETS DB_NAMED_PLACEHOLDERS ' +
	             'DB_POSITIONAL_PLACEHOLDERS DB_PREPARED_QUERIES DB_QUERY_SIZE DB_SIMPLE_LOCKING DB_SYSTEM_TABLES DB_TABLES ' +
	             'DB_TRANSACTIONS DB_UNICODE DB_VIEWS __STDIN __STDOUT __STDERR __FILE_DIR'
	  };

	  const AT_COMMENT_MODE = hljs.COMMENT('@', '@');

	  const PREPROCESSOR =
	  {
	    className: 'meta',
	    begin: '#',
	    end: '$',
	    keywords: {
	      'meta-keyword': 'define definecs|10 undef ifdef ifndef iflight ifdllcall ifmac ifos2win ifunix else endif lineson linesoff srcfile srcline'
	    },
	    contains: [
	      {
	        begin: /\\\n/,
	        relevance: 0
	      },
	      {
	        beginKeywords: 'include',
	        end: '$',
	        keywords: {
	          'meta-keyword': 'include'
	        },
	        contains: [
	          {
	            className: 'meta-string',
	            begin: '"',
	            end: '"',
	            illegal: '\\n'
	          }
	        ]
	      },
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      AT_COMMENT_MODE
	    ]
	  };

	  const STRUCT_TYPE =
	  {
	    begin: /\bstruct\s+/,
	    end: /\s/,
	    keywords: "struct",
	    contains: [
	      {
	        className: "type",
	        begin: hljs.UNDERSCORE_IDENT_RE,
	        relevance: 0
	      }
	    ]
	  };

	  // only for definitions
	  const PARSE_PARAMS = [
	    {
	      className: 'params',
	      begin: /\(/,
	      end: /\)/,
	      excludeBegin: true,
	      excludeEnd: true,
	      endsWithParent: true,
	      relevance: 0,
	      contains: [
	        { // dots
	          className: 'literal',
	          begin: /\.\.\./
	        },
	        hljs.C_NUMBER_MODE,
	        hljs.C_BLOCK_COMMENT_MODE,
	        AT_COMMENT_MODE,
	        STRUCT_TYPE
	      ]
	    }
	  ];

	  const FUNCTION_DEF =
	  {
	    className: "title",
	    begin: hljs.UNDERSCORE_IDENT_RE,
	    relevance: 0
	  };

	  const DEFINITION = function(beginKeywords, end, inherits) {
	    const mode = hljs.inherit(
	      {
	        className: "function",
	        beginKeywords: beginKeywords,
	        end: end,
	        excludeEnd: true,
	        contains: [].concat(PARSE_PARAMS)
	      },
	      {}
	    );
	    mode.contains.push(FUNCTION_DEF);
	    mode.contains.push(hljs.C_NUMBER_MODE);
	    mode.contains.push(hljs.C_BLOCK_COMMENT_MODE);
	    mode.contains.push(AT_COMMENT_MODE);
	    return mode;
	  };

	  const BUILT_IN_REF =
	  { // these are explicitly named internal function calls
	    className: 'built_in',
	    begin: '\\b(' + KEYWORDS.built_in.split(' ').join('|') + ')\\b'
	  };

	  const STRING_REF =
	  {
	    className: 'string',
	    begin: '"',
	    end: '"',
	    contains: [hljs.BACKSLASH_ESCAPE],
	    relevance: 0
	  };

	  const FUNCTION_REF =
	  {
	    // className: "fn_ref",
	    begin: hljs.UNDERSCORE_IDENT_RE + '\\s*\\(',
	    returnBegin: true,
	    keywords: KEYWORDS,
	    relevance: 0,
	    contains: [
	      {
	        beginKeywords: KEYWORDS.keyword
	      },
	      BUILT_IN_REF,
	      { // ambiguously named function calls get a relevance of 0
	        className: 'built_in',
	        begin: hljs.UNDERSCORE_IDENT_RE,
	        relevance: 0
	      }
	    ]
	  };

	  const FUNCTION_REF_PARAMS =
	  {
	    // className: "fn_ref_params",
	    begin: /\(/,
	    end: /\)/,
	    relevance: 0,
	    keywords: {
	      built_in: KEYWORDS.built_in,
	      literal: KEYWORDS.literal
	    },
	    contains: [
	      hljs.C_NUMBER_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      AT_COMMENT_MODE,
	      BUILT_IN_REF,
	      FUNCTION_REF,
	      STRING_REF,
	      'self'
	    ]
	  };

	  FUNCTION_REF.contains.push(FUNCTION_REF_PARAMS);

	  return {
	    name: 'GAUSS',
	    aliases: ['gss'],
	    case_insensitive: true, // language is case-insensitive
	    keywords: KEYWORDS,
	    illegal: /(\{[%#]|[%#]\}| <- )/,
	    contains: [
	      hljs.C_NUMBER_MODE,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      AT_COMMENT_MODE,
	      STRING_REF,
	      PREPROCESSOR,
	      {
	        className: 'keyword',
	        begin: /\bexternal (matrix|string|array|sparse matrix|struct|proc|keyword|fn)/
	      },
	      DEFINITION('proc keyword', ';'),
	      DEFINITION('fn', '='),
	      {
	        beginKeywords: 'for threadfor',
	        end: /;/,
	        // end: /\(/,
	        relevance: 0,
	        contains: [
	          hljs.C_BLOCK_COMMENT_MODE,
	          AT_COMMENT_MODE,
	          FUNCTION_REF_PARAMS
	        ]
	      },
	      { // custom method guard
	        // excludes method names from keyword processing
	        variants: [
	          {
	            begin: hljs.UNDERSCORE_IDENT_RE + '\\.' + hljs.UNDERSCORE_IDENT_RE
	          },
	          {
	            begin: hljs.UNDERSCORE_IDENT_RE + '\\s*='
	          }
	        ],
	        relevance: 0
	      },
	      FUNCTION_REF,
	      STRUCT_TYPE
	    ]
	  };
	}

	gauss_1 = gauss;
	return gauss_1;
}

/*
 Language: G-code (ISO 6983)
 Contributors: Adam Joseph Cook <adam.joseph.cook@gmail.com>
 Description: G-code syntax highlighter for Fanuc and other common CNC machine tool controls.
 Website: https://www.sis.se/api/document/preview/911952/
 */

var gcode_1;
var hasRequiredGcode;

function requireGcode () {
	if (hasRequiredGcode) return gcode_1;
	hasRequiredGcode = 1;
	function gcode(hljs) {
	  const GCODE_IDENT_RE = '[A-Z_][A-Z0-9_.]*';
	  const GCODE_CLOSE_RE = '%';
	  const GCODE_KEYWORDS = {
	    $pattern: GCODE_IDENT_RE,
	    keyword: 'IF DO WHILE ENDWHILE CALL ENDIF SUB ENDSUB GOTO REPEAT ENDREPEAT ' +
	      'EQ LT GT NE GE LE OR XOR'
	  };
	  const GCODE_START = {
	    className: 'meta',
	    begin: '([O])([0-9]+)'
	  };
	  const NUMBER = hljs.inherit(hljs.C_NUMBER_MODE, {
	    begin: '([-+]?((\\.\\d+)|(\\d+)(\\.\\d*)?))|' + hljs.C_NUMBER_RE
	  });
	  const GCODE_CODE = [
	    hljs.C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    hljs.COMMENT(/\(/, /\)/),
	    NUMBER,
	    hljs.inherit(hljs.APOS_STRING_MODE, {
	      illegal: null
	    }),
	    hljs.inherit(hljs.QUOTE_STRING_MODE, {
	      illegal: null
	    }),
	    {
	      className: 'name',
	      begin: '([G])([0-9]+\\.?[0-9]?)'
	    },
	    {
	      className: 'name',
	      begin: '([M])([0-9]+\\.?[0-9]?)'
	    },
	    {
	      className: 'attr',
	      begin: '(VC|VS|#)',
	      end: '(\\d+)'
	    },
	    {
	      className: 'attr',
	      begin: '(VZOFX|VZOFY|VZOFZ)'
	    },
	    {
	      className: 'built_in',
	      begin: '(ATAN|ABS|ACOS|ASIN|SIN|COS|EXP|FIX|FUP|ROUND|LN|TAN)(\\[)',
	      contains: [
	        NUMBER
	      ],
	      end: '\\]'
	    },
	    {
	      className: 'symbol',
	      variants: [
	        {
	          begin: 'N',
	          end: '\\d+',
	          illegal: '\\W'
	        }
	      ]
	    }
	  ];

	  return {
	    name: 'G-code (ISO 6983)',
	    aliases: ['nc'],
	    // Some implementations (CNC controls) of G-code are interoperable with uppercase and lowercase letters seamlessly.
	    // However, most prefer all uppercase and uppercase is customary.
	    case_insensitive: true,
	    keywords: GCODE_KEYWORDS,
	    contains: [
	      {
	        className: 'meta',
	        begin: GCODE_CLOSE_RE
	      },
	      GCODE_START
	    ].concat(GCODE_CODE)
	  };
	}

	gcode_1 = gcode;
	return gcode_1;
}

/*
 Language: Gherkin
 Author: Sam Pikesley (@pikesley) <sam.pikesley@theodi.org>
 Description: Gherkin is the format for cucumber specifications. It is a domain specific language which helps you to describe business behavior without the need to go into detail of implementation.
 Website: https://cucumber.io/docs/gherkin/
 */

var gherkin_1;
var hasRequiredGherkin;

function requireGherkin () {
	if (hasRequiredGherkin) return gherkin_1;
	hasRequiredGherkin = 1;
	function gherkin(hljs) {
	  return {
	    name: 'Gherkin',
	    aliases: ['feature'],
	    keywords: 'Feature Background Ability Business\ Need Scenario Scenarios Scenario\ Outline Scenario\ Template Examples Given And Then But When',
	    contains: [
	      {
	        className: 'symbol',
	        begin: '\\*',
	        relevance: 0
	      },
	      {
	        className: 'meta',
	        begin: '@[^@\\s]+'
	      },
	      {
	        begin: '\\|',
	        end: '\\|\\w*$',
	        contains: [
	          {
	            className: 'string',
	            begin: '[^|]+'
	          }
	        ]
	      },
	      {
	        className: 'variable',
	        begin: '<',
	        end: '>'
	      },
	      hljs.HASH_COMMENT_MODE,
	      {
	        className: 'string',
	        begin: '"""',
	        end: '"""'
	      },
	      hljs.QUOTE_STRING_MODE
	    ]
	  };
	}

	gherkin_1 = gherkin;
	return gherkin_1;
}

/*
Language: GLSL
Description: OpenGL Shading Language
Author: Sergey Tikhomirov <sergey@tikhomirov.io>
Website: https://en.wikipedia.org/wiki/OpenGL_Shading_Language
Category: graphics
*/

var glsl_1;
var hasRequiredGlsl;

function requireGlsl () {
	if (hasRequiredGlsl) return glsl_1;
	hasRequiredGlsl = 1;
	function glsl(hljs) {
	  return {
	    name: 'GLSL',
	    keywords: {
	      keyword:
	        // Statements
	        'break continue discard do else for if return while switch case default ' +
	        // Qualifiers
	        'attribute binding buffer ccw centroid centroid varying coherent column_major const cw ' +
	        'depth_any depth_greater depth_less depth_unchanged early_fragment_tests equal_spacing ' +
	        'flat fractional_even_spacing fractional_odd_spacing highp in index inout invariant ' +
	        'invocations isolines layout line_strip lines lines_adjacency local_size_x local_size_y ' +
	        'local_size_z location lowp max_vertices mediump noperspective offset origin_upper_left ' +
	        'out packed patch pixel_center_integer point_mode points precise precision quads r11f_g11f_b10f ' +
	        'r16 r16_snorm r16f r16i r16ui r32f r32i r32ui r8 r8_snorm r8i r8ui readonly restrict ' +
	        'rg16 rg16_snorm rg16f rg16i rg16ui rg32f rg32i rg32ui rg8 rg8_snorm rg8i rg8ui rgb10_a2 ' +
	        'rgb10_a2ui rgba16 rgba16_snorm rgba16f rgba16i rgba16ui rgba32f rgba32i rgba32ui rgba8 ' +
	        'rgba8_snorm rgba8i rgba8ui row_major sample shared smooth std140 std430 stream triangle_strip ' +
	        'triangles triangles_adjacency uniform varying vertices volatile writeonly',
	      type:
	        'atomic_uint bool bvec2 bvec3 bvec4 dmat2 dmat2x2 dmat2x3 dmat2x4 dmat3 dmat3x2 dmat3x3 ' +
	        'dmat3x4 dmat4 dmat4x2 dmat4x3 dmat4x4 double dvec2 dvec3 dvec4 float iimage1D iimage1DArray ' +
	        'iimage2D iimage2DArray iimage2DMS iimage2DMSArray iimage2DRect iimage3D iimageBuffer ' +
	        'iimageCube iimageCubeArray image1D image1DArray image2D image2DArray image2DMS image2DMSArray ' +
	        'image2DRect image3D imageBuffer imageCube imageCubeArray int isampler1D isampler1DArray ' +
	        'isampler2D isampler2DArray isampler2DMS isampler2DMSArray isampler2DRect isampler3D ' +
	        'isamplerBuffer isamplerCube isamplerCubeArray ivec2 ivec3 ivec4 mat2 mat2x2 mat2x3 ' +
	        'mat2x4 mat3 mat3x2 mat3x3 mat3x4 mat4 mat4x2 mat4x3 mat4x4 sampler1D sampler1DArray ' +
	        'sampler1DArrayShadow sampler1DShadow sampler2D sampler2DArray sampler2DArrayShadow ' +
	        'sampler2DMS sampler2DMSArray sampler2DRect sampler2DRectShadow sampler2DShadow sampler3D ' +
	        'samplerBuffer samplerCube samplerCubeArray samplerCubeArrayShadow samplerCubeShadow ' +
	        'image1D uimage1DArray uimage2D uimage2DArray uimage2DMS uimage2DMSArray uimage2DRect ' +
	        'uimage3D uimageBuffer uimageCube uimageCubeArray uint usampler1D usampler1DArray ' +
	        'usampler2D usampler2DArray usampler2DMS usampler2DMSArray usampler2DRect usampler3D ' +
	        'samplerBuffer usamplerCube usamplerCubeArray uvec2 uvec3 uvec4 vec2 vec3 vec4 void',
	      built_in:
	        // Constants
	        'gl_MaxAtomicCounterBindings gl_MaxAtomicCounterBufferSize gl_MaxClipDistances gl_MaxClipPlanes ' +
	        'gl_MaxCombinedAtomicCounterBuffers gl_MaxCombinedAtomicCounters gl_MaxCombinedImageUniforms ' +
	        'gl_MaxCombinedImageUnitsAndFragmentOutputs gl_MaxCombinedTextureImageUnits gl_MaxComputeAtomicCounterBuffers ' +
	        'gl_MaxComputeAtomicCounters gl_MaxComputeImageUniforms gl_MaxComputeTextureImageUnits ' +
	        'gl_MaxComputeUniformComponents gl_MaxComputeWorkGroupCount gl_MaxComputeWorkGroupSize ' +
	        'gl_MaxDrawBuffers gl_MaxFragmentAtomicCounterBuffers gl_MaxFragmentAtomicCounters ' +
	        'gl_MaxFragmentImageUniforms gl_MaxFragmentInputComponents gl_MaxFragmentInputVectors ' +
	        'gl_MaxFragmentUniformComponents gl_MaxFragmentUniformVectors gl_MaxGeometryAtomicCounterBuffers ' +
	        'gl_MaxGeometryAtomicCounters gl_MaxGeometryImageUniforms gl_MaxGeometryInputComponents ' +
	        'gl_MaxGeometryOutputComponents gl_MaxGeometryOutputVertices gl_MaxGeometryTextureImageUnits ' +
	        'gl_MaxGeometryTotalOutputComponents gl_MaxGeometryUniformComponents gl_MaxGeometryVaryingComponents ' +
	        'gl_MaxImageSamples gl_MaxImageUnits gl_MaxLights gl_MaxPatchVertices gl_MaxProgramTexelOffset ' +
	        'gl_MaxTessControlAtomicCounterBuffers gl_MaxTessControlAtomicCounters gl_MaxTessControlImageUniforms ' +
	        'gl_MaxTessControlInputComponents gl_MaxTessControlOutputComponents gl_MaxTessControlTextureImageUnits ' +
	        'gl_MaxTessControlTotalOutputComponents gl_MaxTessControlUniformComponents ' +
	        'gl_MaxTessEvaluationAtomicCounterBuffers gl_MaxTessEvaluationAtomicCounters ' +
	        'gl_MaxTessEvaluationImageUniforms gl_MaxTessEvaluationInputComponents gl_MaxTessEvaluationOutputComponents ' +
	        'gl_MaxTessEvaluationTextureImageUnits gl_MaxTessEvaluationUniformComponents ' +
	        'gl_MaxTessGenLevel gl_MaxTessPatchComponents gl_MaxTextureCoords gl_MaxTextureImageUnits ' +
	        'gl_MaxTextureUnits gl_MaxVaryingComponents gl_MaxVaryingFloats gl_MaxVaryingVectors ' +
	        'gl_MaxVertexAtomicCounterBuffers gl_MaxVertexAtomicCounters gl_MaxVertexAttribs gl_MaxVertexImageUniforms ' +
	        'gl_MaxVertexOutputComponents gl_MaxVertexOutputVectors gl_MaxVertexTextureImageUnits ' +
	        'gl_MaxVertexUniformComponents gl_MaxVertexUniformVectors gl_MaxViewports gl_MinProgramTexelOffset ' +
	        // Variables
	        'gl_BackColor gl_BackLightModelProduct gl_BackLightProduct gl_BackMaterial ' +
	        'gl_BackSecondaryColor gl_ClipDistance gl_ClipPlane gl_ClipVertex gl_Color ' +
	        'gl_DepthRange gl_EyePlaneQ gl_EyePlaneR gl_EyePlaneS gl_EyePlaneT gl_Fog gl_FogCoord ' +
	        'gl_FogFragCoord gl_FragColor gl_FragCoord gl_FragData gl_FragDepth gl_FrontColor ' +
	        'gl_FrontFacing gl_FrontLightModelProduct gl_FrontLightProduct gl_FrontMaterial ' +
	        'gl_FrontSecondaryColor gl_GlobalInvocationID gl_InstanceID gl_InvocationID gl_Layer gl_LightModel ' +
	        'gl_LightSource gl_LocalInvocationID gl_LocalInvocationIndex gl_ModelViewMatrix ' +
	        'gl_ModelViewMatrixInverse gl_ModelViewMatrixInverseTranspose gl_ModelViewMatrixTranspose ' +
	        'gl_ModelViewProjectionMatrix gl_ModelViewProjectionMatrixInverse gl_ModelViewProjectionMatrixInverseTranspose ' +
	        'gl_ModelViewProjectionMatrixTranspose gl_MultiTexCoord0 gl_MultiTexCoord1 gl_MultiTexCoord2 ' +
	        'gl_MultiTexCoord3 gl_MultiTexCoord4 gl_MultiTexCoord5 gl_MultiTexCoord6 gl_MultiTexCoord7 ' +
	        'gl_Normal gl_NormalMatrix gl_NormalScale gl_NumSamples gl_NumWorkGroups gl_ObjectPlaneQ ' +
	        'gl_ObjectPlaneR gl_ObjectPlaneS gl_ObjectPlaneT gl_PatchVerticesIn gl_Point gl_PointCoord ' +
	        'gl_PointSize gl_Position gl_PrimitiveID gl_PrimitiveIDIn gl_ProjectionMatrix gl_ProjectionMatrixInverse ' +
	        'gl_ProjectionMatrixInverseTranspose gl_ProjectionMatrixTranspose gl_SampleID gl_SampleMask ' +
	        'gl_SampleMaskIn gl_SamplePosition gl_SecondaryColor gl_TessCoord gl_TessLevelInner gl_TessLevelOuter ' +
	        'gl_TexCoord gl_TextureEnvColor gl_TextureMatrix gl_TextureMatrixInverse gl_TextureMatrixInverseTranspose ' +
	        'gl_TextureMatrixTranspose gl_Vertex gl_VertexID gl_ViewportIndex gl_WorkGroupID gl_WorkGroupSize gl_in gl_out ' +
	        // Functions
	        'EmitStreamVertex EmitVertex EndPrimitive EndStreamPrimitive abs acos acosh all any asin ' +
	        'asinh atan atanh atomicAdd atomicAnd atomicCompSwap atomicCounter atomicCounterDecrement ' +
	        'atomicCounterIncrement atomicExchange atomicMax atomicMin atomicOr atomicXor barrier ' +
	        'bitCount bitfieldExtract bitfieldInsert bitfieldReverse ceil clamp cos cosh cross ' +
	        'dFdx dFdy degrees determinant distance dot equal exp exp2 faceforward findLSB findMSB ' +
	        'floatBitsToInt floatBitsToUint floor fma fract frexp ftransform fwidth greaterThan ' +
	        'greaterThanEqual groupMemoryBarrier imageAtomicAdd imageAtomicAnd imageAtomicCompSwap ' +
	        'imageAtomicExchange imageAtomicMax imageAtomicMin imageAtomicOr imageAtomicXor imageLoad ' +
	        'imageSize imageStore imulExtended intBitsToFloat interpolateAtCentroid interpolateAtOffset ' +
	        'interpolateAtSample inverse inversesqrt isinf isnan ldexp length lessThan lessThanEqual log ' +
	        'log2 matrixCompMult max memoryBarrier memoryBarrierAtomicCounter memoryBarrierBuffer ' +
	        'memoryBarrierImage memoryBarrierShared min mix mod modf noise1 noise2 noise3 noise4 ' +
	        'normalize not notEqual outerProduct packDouble2x32 packHalf2x16 packSnorm2x16 packSnorm4x8 ' +
	        'packUnorm2x16 packUnorm4x8 pow radians reflect refract round roundEven shadow1D shadow1DLod ' +
	        'shadow1DProj shadow1DProjLod shadow2D shadow2DLod shadow2DProj shadow2DProjLod sign sin sinh ' +
	        'smoothstep sqrt step tan tanh texelFetch texelFetchOffset texture texture1D texture1DLod ' +
	        'texture1DProj texture1DProjLod texture2D texture2DLod texture2DProj texture2DProjLod ' +
	        'texture3D texture3DLod texture3DProj texture3DProjLod textureCube textureCubeLod ' +
	        'textureGather textureGatherOffset textureGatherOffsets textureGrad textureGradOffset ' +
	        'textureLod textureLodOffset textureOffset textureProj textureProjGrad textureProjGradOffset ' +
	        'textureProjLod textureProjLodOffset textureProjOffset textureQueryLevels textureQueryLod ' +
	        'textureSize transpose trunc uaddCarry uintBitsToFloat umulExtended unpackDouble2x32 ' +
	        'unpackHalf2x16 unpackSnorm2x16 unpackSnorm4x8 unpackUnorm2x16 unpackUnorm4x8 usubBorrow',
	      literal: 'true false'
	    },
	    illegal: '"',
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.C_NUMBER_MODE,
	      {
	        className: 'meta',
	        begin: '#',
	        end: '$'
	      }
	    ]
	  };
	}

	glsl_1 = glsl;
	return glsl_1;
}

/*
Language: GML
Author: Meseta <meseta@gmail.com>
Description: Game Maker Language for GameMaker Studio 2
Website: https://docs2.yoyogames.com
Category: scripting
*/

var gml_1;
var hasRequiredGml;

function requireGml () {
	if (hasRequiredGml) return gml_1;
	hasRequiredGml = 1;
	function gml(hljs) {
	  const GML_KEYWORDS = {
	    keyword: 'begin end if then else while do for break continue with until ' +
	      'repeat exit and or xor not return mod div switch case default var ' +
	      'globalvar enum function constructor delete #macro #region #endregion',
	    built_in: 'is_real is_string is_array is_undefined is_int32 is_int64 is_ptr ' +
	      'is_vec3 is_vec4 is_matrix is_bool is_method is_struct is_infinity is_nan ' +
	      'is_numeric typeof variable_global_exists variable_global_get variable_global_set ' +
	      'variable_instance_exists variable_instance_get variable_instance_set ' +
	      'variable_instance_get_names variable_struct_exists variable_struct_get ' +
	      'variable_struct_get_names variable_struct_names_count variable_struct_remove ' +
	      'variable_struct_set array_delete array_insert array_length array_length_1d ' +
	      'array_length_2d array_height_2d array_equals array_create ' +
	      'array_copy array_pop array_push array_resize array_sort ' +
	      'random random_range irandom irandom_range random_set_seed random_get_seed ' +
	      'randomize randomise choose abs round floor ceil sign frac sqrt sqr ' +
	      'exp ln log2 log10 sin cos tan arcsin arccos arctan arctan2 dsin dcos ' +
	      'dtan darcsin darccos darctan darctan2 degtorad radtodeg power logn ' +
	      'min max mean median clamp lerp dot_product dot_product_3d ' +
	      'dot_product_normalised dot_product_3d_normalised ' +
	      'dot_product_normalized dot_product_3d_normalized math_set_epsilon ' +
	      'math_get_epsilon angle_difference point_distance_3d point_distance ' +
	      'point_direction lengthdir_x lengthdir_y real string int64 ptr ' +
	      'string_format chr ansi_char ord string_length string_byte_length ' +
	      'string_pos string_copy string_char_at string_ord_at string_byte_at ' +
	      'string_set_byte_at string_delete string_insert string_lower ' +
	      'string_upper string_repeat string_letters string_digits ' +
	      'string_lettersdigits string_replace string_replace_all string_count ' +
	      'string_hash_to_newline clipboard_has_text clipboard_set_text ' +
	      'clipboard_get_text date_current_datetime date_create_datetime ' +
	      'date_valid_datetime date_inc_year date_inc_month date_inc_week ' +
	      'date_inc_day date_inc_hour date_inc_minute date_inc_second ' +
	      'date_get_year date_get_month date_get_week date_get_day ' +
	      'date_get_hour date_get_minute date_get_second date_get_weekday ' +
	      'date_get_day_of_year date_get_hour_of_year date_get_minute_of_year ' +
	      'date_get_second_of_year date_year_span date_month_span ' +
	      'date_week_span date_day_span date_hour_span date_minute_span ' +
	      'date_second_span date_compare_datetime date_compare_date ' +
	      'date_compare_time date_date_of date_time_of date_datetime_string ' +
	      'date_date_string date_time_string date_days_in_month ' +
	      'date_days_in_year date_leap_year date_is_today date_set_timezone ' +
	      'date_get_timezone game_set_speed game_get_speed motion_set ' +
	      'motion_add place_free place_empty place_meeting place_snapped ' +
	      'move_random move_snap move_towards_point move_contact_solid ' +
	      'move_contact_all move_outside_solid move_outside_all ' +
	      'move_bounce_solid move_bounce_all move_wrap distance_to_point ' +
	      'distance_to_object position_empty position_meeting path_start ' +
	      'path_end mp_linear_step mp_potential_step mp_linear_step_object ' +
	      'mp_potential_step_object mp_potential_settings mp_linear_path ' +
	      'mp_potential_path mp_linear_path_object mp_potential_path_object ' +
	      'mp_grid_create mp_grid_destroy mp_grid_clear_all mp_grid_clear_cell ' +
	      'mp_grid_clear_rectangle mp_grid_add_cell mp_grid_get_cell ' +
	      'mp_grid_add_rectangle mp_grid_add_instances mp_grid_path ' +
	      'mp_grid_draw mp_grid_to_ds_grid collision_point collision_rectangle ' +
	      'collision_circle collision_ellipse collision_line ' +
	      'collision_point_list collision_rectangle_list collision_circle_list ' +
	      'collision_ellipse_list collision_line_list instance_position_list ' +
	      'instance_place_list point_in_rectangle ' +
	      'point_in_triangle point_in_circle rectangle_in_rectangle ' +
	      'rectangle_in_triangle rectangle_in_circle instance_find ' +
	      'instance_exists instance_number instance_position instance_nearest ' +
	      'instance_furthest instance_place instance_create_depth ' +
	      'instance_create_layer instance_copy instance_change instance_destroy ' +
	      'position_destroy position_change instance_id_get ' +
	      'instance_deactivate_all instance_deactivate_object ' +
	      'instance_deactivate_region instance_activate_all ' +
	      'instance_activate_object instance_activate_region room_goto ' +
	      'room_goto_previous room_goto_next room_previous room_next ' +
	      'room_restart game_end game_restart game_load game_save ' +
	      'game_save_buffer game_load_buffer event_perform event_user ' +
	      'event_perform_object event_inherited show_debug_message ' +
	      'show_debug_overlay debug_event debug_get_callstack alarm_get ' +
	      'alarm_set font_texture_page_size keyboard_set_map keyboard_get_map ' +
	      'keyboard_unset_map keyboard_check keyboard_check_pressed ' +
	      'keyboard_check_released keyboard_check_direct keyboard_get_numlock ' +
	      'keyboard_set_numlock keyboard_key_press keyboard_key_release ' +
	      'keyboard_clear io_clear mouse_check_button ' +
	      'mouse_check_button_pressed mouse_check_button_released ' +
	      'mouse_wheel_up mouse_wheel_down mouse_clear draw_self draw_sprite ' +
	      'draw_sprite_pos draw_sprite_ext draw_sprite_stretched ' +
	      'draw_sprite_stretched_ext draw_sprite_tiled draw_sprite_tiled_ext ' +
	      'draw_sprite_part draw_sprite_part_ext draw_sprite_general draw_clear ' +
	      'draw_clear_alpha draw_point draw_line draw_line_width draw_rectangle ' +
	      'draw_roundrect draw_roundrect_ext draw_triangle draw_circle ' +
	      'draw_ellipse draw_set_circle_precision draw_arrow draw_button ' +
	      'draw_path draw_healthbar draw_getpixel draw_getpixel_ext ' +
	      'draw_set_colour draw_set_color draw_set_alpha draw_get_colour ' +
	      'draw_get_color draw_get_alpha merge_colour make_colour_rgb ' +
	      'make_colour_hsv colour_get_red colour_get_green colour_get_blue ' +
	      'colour_get_hue colour_get_saturation colour_get_value merge_color ' +
	      'make_color_rgb make_color_hsv color_get_red color_get_green ' +
	      'color_get_blue color_get_hue color_get_saturation color_get_value ' +
	      'merge_color screen_save screen_save_part draw_set_font ' +
	      'draw_set_halign draw_set_valign draw_text draw_text_ext string_width ' +
	      'string_height string_width_ext string_height_ext ' +
	      'draw_text_transformed draw_text_ext_transformed draw_text_colour ' +
	      'draw_text_ext_colour draw_text_transformed_colour ' +
	      'draw_text_ext_transformed_colour draw_text_color draw_text_ext_color ' +
	      'draw_text_transformed_color draw_text_ext_transformed_color ' +
	      'draw_point_colour draw_line_colour draw_line_width_colour ' +
	      'draw_rectangle_colour draw_roundrect_colour ' +
	      'draw_roundrect_colour_ext draw_triangle_colour draw_circle_colour ' +
	      'draw_ellipse_colour draw_point_color draw_line_color ' +
	      'draw_line_width_color draw_rectangle_color draw_roundrect_color ' +
	      'draw_roundrect_color_ext draw_triangle_color draw_circle_color ' +
	      'draw_ellipse_color draw_primitive_begin draw_vertex ' +
	      'draw_vertex_colour draw_vertex_color draw_primitive_end ' +
	      'sprite_get_uvs font_get_uvs sprite_get_texture font_get_texture ' +
	      'texture_get_width texture_get_height texture_get_uvs ' +
	      'draw_primitive_begin_texture draw_vertex_texture ' +
	      'draw_vertex_texture_colour draw_vertex_texture_color ' +
	      'texture_global_scale surface_create surface_create_ext ' +
	      'surface_resize surface_free surface_exists surface_get_width ' +
	      'surface_get_height surface_get_texture surface_set_target ' +
	      'surface_set_target_ext surface_reset_target surface_depth_disable ' +
	      'surface_get_depth_disable draw_surface draw_surface_stretched ' +
	      'draw_surface_tiled draw_surface_part draw_surface_ext ' +
	      'draw_surface_stretched_ext draw_surface_tiled_ext ' +
	      'draw_surface_part_ext draw_surface_general surface_getpixel ' +
	      'surface_getpixel_ext surface_save surface_save_part surface_copy ' +
	      'surface_copy_part application_surface_draw_enable ' +
	      'application_get_position application_surface_enable ' +
	      'application_surface_is_enabled display_get_width display_get_height ' +
	      'display_get_orientation display_get_gui_width display_get_gui_height ' +
	      'display_reset display_mouse_get_x display_mouse_get_y ' +
	      'display_mouse_set display_set_ui_visibility ' +
	      'window_set_fullscreen window_get_fullscreen ' +
	      'window_set_caption window_set_min_width window_set_max_width ' +
	      'window_set_min_height window_set_max_height window_get_visible_rects ' +
	      'window_get_caption window_set_cursor window_get_cursor ' +
	      'window_set_colour window_get_colour window_set_color ' +
	      'window_get_color window_set_position window_set_size ' +
	      'window_set_rectangle window_center window_get_x window_get_y ' +
	      'window_get_width window_get_height window_mouse_get_x ' +
	      'window_mouse_get_y window_mouse_set window_view_mouse_get_x ' +
	      'window_view_mouse_get_y window_views_mouse_get_x ' +
	      'window_views_mouse_get_y audio_listener_position ' +
	      'audio_listener_velocity audio_listener_orientation ' +
	      'audio_emitter_position audio_emitter_create audio_emitter_free ' +
	      'audio_emitter_exists audio_emitter_pitch audio_emitter_velocity ' +
	      'audio_emitter_falloff audio_emitter_gain audio_play_sound ' +
	      'audio_play_sound_on audio_play_sound_at audio_stop_sound ' +
	      'audio_resume_music audio_music_is_playing audio_resume_sound ' +
	      'audio_pause_sound audio_pause_music audio_channel_num ' +
	      'audio_sound_length audio_get_type audio_falloff_set_model ' +
	      'audio_play_music audio_stop_music audio_master_gain audio_music_gain ' +
	      'audio_sound_gain audio_sound_pitch audio_stop_all audio_resume_all ' +
	      'audio_pause_all audio_is_playing audio_is_paused audio_exists ' +
	      'audio_sound_set_track_position audio_sound_get_track_position ' +
	      'audio_emitter_get_gain audio_emitter_get_pitch audio_emitter_get_x ' +
	      'audio_emitter_get_y audio_emitter_get_z audio_emitter_get_vx ' +
	      'audio_emitter_get_vy audio_emitter_get_vz ' +
	      'audio_listener_set_position audio_listener_set_velocity ' +
	      'audio_listener_set_orientation audio_listener_get_data ' +
	      'audio_set_master_gain audio_get_master_gain audio_sound_get_gain ' +
	      'audio_sound_get_pitch audio_get_name audio_sound_set_track_position ' +
	      'audio_sound_get_track_position audio_create_stream ' +
	      'audio_destroy_stream audio_create_sync_group ' +
	      'audio_destroy_sync_group audio_play_in_sync_group ' +
	      'audio_start_sync_group audio_stop_sync_group audio_pause_sync_group ' +
	      'audio_resume_sync_group audio_sync_group_get_track_pos ' +
	      'audio_sync_group_debug audio_sync_group_is_playing audio_debug ' +
	      'audio_group_load audio_group_unload audio_group_is_loaded ' +
	      'audio_group_load_progress audio_group_name audio_group_stop_all ' +
	      'audio_group_set_gain audio_create_buffer_sound ' +
	      'audio_free_buffer_sound audio_create_play_queue ' +
	      'audio_free_play_queue audio_queue_sound audio_get_recorder_count ' +
	      'audio_get_recorder_info audio_start_recording audio_stop_recording ' +
	      'audio_sound_get_listener_mask audio_emitter_get_listener_mask ' +
	      'audio_get_listener_mask audio_sound_set_listener_mask ' +
	      'audio_emitter_set_listener_mask audio_set_listener_mask ' +
	      'audio_get_listener_count audio_get_listener_info audio_system ' +
	      'show_message show_message_async clickable_add clickable_add_ext ' +
	      'clickable_change clickable_change_ext clickable_delete ' +
	      'clickable_exists clickable_set_style show_question ' +
	      'show_question_async get_integer get_string get_integer_async ' +
	      'get_string_async get_login_async get_open_filename get_save_filename ' +
	      'get_open_filename_ext get_save_filename_ext show_error ' +
	      'highscore_clear highscore_add highscore_value highscore_name ' +
	      'draw_highscore sprite_exists sprite_get_name sprite_get_number ' +
	      'sprite_get_width sprite_get_height sprite_get_xoffset ' +
	      'sprite_get_yoffset sprite_get_bbox_left sprite_get_bbox_right ' +
	      'sprite_get_bbox_top sprite_get_bbox_bottom sprite_save ' +
	      'sprite_save_strip sprite_set_cache_size sprite_set_cache_size_ext ' +
	      'sprite_get_tpe sprite_prefetch sprite_prefetch_multi sprite_flush ' +
	      'sprite_flush_multi sprite_set_speed sprite_get_speed_type ' +
	      'sprite_get_speed font_exists font_get_name font_get_fontname ' +
	      'font_get_bold font_get_italic font_get_first font_get_last ' +
	      'font_get_size font_set_cache_size path_exists path_get_name ' +
	      'path_get_length path_get_time path_get_kind path_get_closed ' +
	      'path_get_precision path_get_number path_get_point_x path_get_point_y ' +
	      'path_get_point_speed path_get_x path_get_y path_get_speed ' +
	      'script_exists script_get_name timeline_add timeline_delete ' +
	      'timeline_clear timeline_exists timeline_get_name ' +
	      'timeline_moment_clear timeline_moment_add_script timeline_size ' +
	      'timeline_max_moment object_exists object_get_name object_get_sprite ' +
	      'object_get_solid object_get_visible object_get_persistent ' +
	      'object_get_mask object_get_parent object_get_physics ' +
	      'object_is_ancestor room_exists room_get_name sprite_set_offset ' +
	      'sprite_duplicate sprite_assign sprite_merge sprite_add ' +
	      'sprite_replace sprite_create_from_surface sprite_add_from_surface ' +
	      'sprite_delete sprite_set_alpha_from_sprite sprite_collision_mask ' +
	      'font_add_enable_aa font_add_get_enable_aa font_add font_add_sprite ' +
	      'font_add_sprite_ext font_replace font_replace_sprite ' +
	      'font_replace_sprite_ext font_delete path_set_kind path_set_closed ' +
	      'path_set_precision path_add path_assign path_duplicate path_append ' +
	      'path_delete path_add_point path_insert_point path_change_point ' +
	      'path_delete_point path_clear_points path_reverse path_mirror ' +
	      'path_flip path_rotate path_rescale path_shift script_execute ' +
	      'object_set_sprite object_set_solid object_set_visible ' +
	      'object_set_persistent object_set_mask room_set_width room_set_height ' +
	      'room_set_persistent room_set_background_colour ' +
	      'room_set_background_color room_set_view room_set_viewport ' +
	      'room_get_viewport room_set_view_enabled room_add room_duplicate ' +
	      'room_assign room_instance_add room_instance_clear room_get_camera ' +
	      'room_set_camera asset_get_index asset_get_type ' +
	      'file_text_open_from_string file_text_open_read file_text_open_write ' +
	      'file_text_open_append file_text_close file_text_write_string ' +
	      'file_text_write_real file_text_writeln file_text_read_string ' +
	      'file_text_read_real file_text_readln file_text_eof file_text_eoln ' +
	      'file_exists file_delete file_rename file_copy directory_exists ' +
	      'directory_create directory_destroy file_find_first file_find_next ' +
	      'file_find_close file_attributes filename_name filename_path ' +
	      'filename_dir filename_drive filename_ext filename_change_ext ' +
	      'file_bin_open file_bin_rewrite file_bin_close file_bin_position ' +
	      'file_bin_size file_bin_seek file_bin_write_byte file_bin_read_byte ' +
	      'parameter_count parameter_string environment_get_variable ' +
	      'ini_open_from_string ini_open ini_close ini_read_string ' +
	      'ini_read_real ini_write_string ini_write_real ini_key_exists ' +
	      'ini_section_exists ini_key_delete ini_section_delete ' +
	      'ds_set_precision ds_exists ds_stack_create ds_stack_destroy ' +
	      'ds_stack_clear ds_stack_copy ds_stack_size ds_stack_empty ' +
	      'ds_stack_push ds_stack_pop ds_stack_top ds_stack_write ds_stack_read ' +
	      'ds_queue_create ds_queue_destroy ds_queue_clear ds_queue_copy ' +
	      'ds_queue_size ds_queue_empty ds_queue_enqueue ds_queue_dequeue ' +
	      'ds_queue_head ds_queue_tail ds_queue_write ds_queue_read ' +
	      'ds_list_create ds_list_destroy ds_list_clear ds_list_copy ' +
	      'ds_list_size ds_list_empty ds_list_add ds_list_insert ' +
	      'ds_list_replace ds_list_delete ds_list_find_index ds_list_find_value ' +
	      'ds_list_mark_as_list ds_list_mark_as_map ds_list_sort ' +
	      'ds_list_shuffle ds_list_write ds_list_read ds_list_set ds_map_create ' +
	      'ds_map_destroy ds_map_clear ds_map_copy ds_map_size ds_map_empty ' +
	      'ds_map_add ds_map_add_list ds_map_add_map ds_map_replace ' +
	      'ds_map_replace_map ds_map_replace_list ds_map_delete ds_map_exists ' +
	      'ds_map_find_value ds_map_find_previous ds_map_find_next ' +
	      'ds_map_find_first ds_map_find_last ds_map_write ds_map_read ' +
	      'ds_map_secure_save ds_map_secure_load ds_map_secure_load_buffer ' +
	      'ds_map_secure_save_buffer ds_map_set ds_priority_create ' +
	      'ds_priority_destroy ds_priority_clear ds_priority_copy ' +
	      'ds_priority_size ds_priority_empty ds_priority_add ' +
	      'ds_priority_change_priority ds_priority_find_priority ' +
	      'ds_priority_delete_value ds_priority_delete_min ds_priority_find_min ' +
	      'ds_priority_delete_max ds_priority_find_max ds_priority_write ' +
	      'ds_priority_read ds_grid_create ds_grid_destroy ds_grid_copy ' +
	      'ds_grid_resize ds_grid_width ds_grid_height ds_grid_clear ' +
	      'ds_grid_set ds_grid_add ds_grid_multiply ds_grid_set_region ' +
	      'ds_grid_add_region ds_grid_multiply_region ds_grid_set_disk ' +
	      'ds_grid_add_disk ds_grid_multiply_disk ds_grid_set_grid_region ' +
	      'ds_grid_add_grid_region ds_grid_multiply_grid_region ds_grid_get ' +
	      'ds_grid_get_sum ds_grid_get_max ds_grid_get_min ds_grid_get_mean ' +
	      'ds_grid_get_disk_sum ds_grid_get_disk_min ds_grid_get_disk_max ' +
	      'ds_grid_get_disk_mean ds_grid_value_exists ds_grid_value_x ' +
	      'ds_grid_value_y ds_grid_value_disk_exists ds_grid_value_disk_x ' +
	      'ds_grid_value_disk_y ds_grid_shuffle ds_grid_write ds_grid_read ' +
	      'ds_grid_sort ds_grid_set ds_grid_get effect_create_below ' +
	      'effect_create_above effect_clear part_type_create part_type_destroy ' +
	      'part_type_exists part_type_clear part_type_shape part_type_sprite ' +
	      'part_type_size part_type_scale part_type_orientation part_type_life ' +
	      'part_type_step part_type_death part_type_speed part_type_direction ' +
	      'part_type_gravity part_type_colour1 part_type_colour2 ' +
	      'part_type_colour3 part_type_colour_mix part_type_colour_rgb ' +
	      'part_type_colour_hsv part_type_color1 part_type_color2 ' +
	      'part_type_color3 part_type_color_mix part_type_color_rgb ' +
	      'part_type_color_hsv part_type_alpha1 part_type_alpha2 ' +
	      'part_type_alpha3 part_type_blend part_system_create ' +
	      'part_system_create_layer part_system_destroy part_system_exists ' +
	      'part_system_clear part_system_draw_order part_system_depth ' +
	      'part_system_position part_system_automatic_update ' +
	      'part_system_automatic_draw part_system_update part_system_drawit ' +
	      'part_system_get_layer part_system_layer part_particles_create ' +
	      'part_particles_create_colour part_particles_create_color ' +
	      'part_particles_clear part_particles_count part_emitter_create ' +
	      'part_emitter_destroy part_emitter_destroy_all part_emitter_exists ' +
	      'part_emitter_clear part_emitter_region part_emitter_burst ' +
	      'part_emitter_stream external_call external_define external_free ' +
	      'window_handle window_device matrix_get matrix_set ' +
	      'matrix_build_identity matrix_build matrix_build_lookat ' +
	      'matrix_build_projection_ortho matrix_build_projection_perspective ' +
	      'matrix_build_projection_perspective_fov matrix_multiply ' +
	      'matrix_transform_vertex matrix_stack_push matrix_stack_pop ' +
	      'matrix_stack_multiply matrix_stack_set matrix_stack_clear ' +
	      'matrix_stack_top matrix_stack_is_empty browser_input_capture ' +
	      'os_get_config os_get_info os_get_language os_get_region ' +
	      'os_lock_orientation display_get_dpi_x display_get_dpi_y ' +
	      'display_set_gui_size display_set_gui_maximise ' +
	      'display_set_gui_maximize device_mouse_dbclick_enable ' +
	      'display_set_timing_method display_get_timing_method ' +
	      'display_set_sleep_margin display_get_sleep_margin virtual_key_add ' +
	      'virtual_key_hide virtual_key_delete virtual_key_show ' +
	      'draw_enable_drawevent draw_enable_swf_aa draw_set_swf_aa_level ' +
	      'draw_get_swf_aa_level draw_texture_flush draw_flush ' +
	      'gpu_set_blendenable gpu_set_ztestenable gpu_set_zfunc ' +
	      'gpu_set_zwriteenable gpu_set_lightingenable gpu_set_fog ' +
	      'gpu_set_cullmode gpu_set_blendmode gpu_set_blendmode_ext ' +
	      'gpu_set_blendmode_ext_sepalpha gpu_set_colorwriteenable ' +
	      'gpu_set_colourwriteenable gpu_set_alphatestenable ' +
	      'gpu_set_alphatestref gpu_set_alphatestfunc gpu_set_texfilter ' +
	      'gpu_set_texfilter_ext gpu_set_texrepeat gpu_set_texrepeat_ext ' +
	      'gpu_set_tex_filter gpu_set_tex_filter_ext gpu_set_tex_repeat ' +
	      'gpu_set_tex_repeat_ext gpu_set_tex_mip_filter ' +
	      'gpu_set_tex_mip_filter_ext gpu_set_tex_mip_bias ' +
	      'gpu_set_tex_mip_bias_ext gpu_set_tex_min_mip gpu_set_tex_min_mip_ext ' +
	      'gpu_set_tex_max_mip gpu_set_tex_max_mip_ext gpu_set_tex_max_aniso ' +
	      'gpu_set_tex_max_aniso_ext gpu_set_tex_mip_enable ' +
	      'gpu_set_tex_mip_enable_ext gpu_get_blendenable gpu_get_ztestenable ' +
	      'gpu_get_zfunc gpu_get_zwriteenable gpu_get_lightingenable ' +
	      'gpu_get_fog gpu_get_cullmode gpu_get_blendmode gpu_get_blendmode_ext ' +
	      'gpu_get_blendmode_ext_sepalpha gpu_get_blendmode_src ' +
	      'gpu_get_blendmode_dest gpu_get_blendmode_srcalpha ' +
	      'gpu_get_blendmode_destalpha gpu_get_colorwriteenable ' +
	      'gpu_get_colourwriteenable gpu_get_alphatestenable ' +
	      'gpu_get_alphatestref gpu_get_alphatestfunc gpu_get_texfilter ' +
	      'gpu_get_texfilter_ext gpu_get_texrepeat gpu_get_texrepeat_ext ' +
	      'gpu_get_tex_filter gpu_get_tex_filter_ext gpu_get_tex_repeat ' +
	      'gpu_get_tex_repeat_ext gpu_get_tex_mip_filter ' +
	      'gpu_get_tex_mip_filter_ext gpu_get_tex_mip_bias ' +
	      'gpu_get_tex_mip_bias_ext gpu_get_tex_min_mip gpu_get_tex_min_mip_ext ' +
	      'gpu_get_tex_max_mip gpu_get_tex_max_mip_ext gpu_get_tex_max_aniso ' +
	      'gpu_get_tex_max_aniso_ext gpu_get_tex_mip_enable ' +
	      'gpu_get_tex_mip_enable_ext gpu_push_state gpu_pop_state ' +
	      'gpu_get_state gpu_set_state draw_light_define_ambient ' +
	      'draw_light_define_direction draw_light_define_point ' +
	      'draw_light_enable draw_set_lighting draw_light_get_ambient ' +
	      'draw_light_get draw_get_lighting shop_leave_rating url_get_domain ' +
	      'url_open url_open_ext url_open_full get_timer achievement_login ' +
	      'achievement_logout achievement_post achievement_increment ' +
	      'achievement_post_score achievement_available ' +
	      'achievement_show_achievements achievement_show_leaderboards ' +
	      'achievement_load_friends achievement_load_leaderboard ' +
	      'achievement_send_challenge achievement_load_progress ' +
	      'achievement_reset achievement_login_status achievement_get_pic ' +
	      'achievement_show_challenge_notifications achievement_get_challenges ' +
	      'achievement_event achievement_show achievement_get_info ' +
	      'cloud_file_save cloud_string_save cloud_synchronise ads_enable ' +
	      'ads_disable ads_setup ads_engagement_launch ads_engagement_available ' +
	      'ads_engagement_active ads_event ads_event_preload ' +
	      'ads_set_reward_callback ads_get_display_height ads_get_display_width ' +
	      'ads_move ads_interstitial_available ads_interstitial_display ' +
	      'device_get_tilt_x device_get_tilt_y device_get_tilt_z ' +
	      'device_is_keypad_open device_mouse_check_button ' +
	      'device_mouse_check_button_pressed device_mouse_check_button_released ' +
	      'device_mouse_x device_mouse_y device_mouse_raw_x device_mouse_raw_y ' +
	      'device_mouse_x_to_gui device_mouse_y_to_gui iap_activate iap_status ' +
	      'iap_enumerate_products iap_restore_all iap_acquire iap_consume ' +
	      'iap_product_details iap_purchase_details facebook_init ' +
	      'facebook_login facebook_status facebook_graph_request ' +
	      'facebook_dialog facebook_logout facebook_launch_offerwall ' +
	      'facebook_post_message facebook_send_invite facebook_user_id ' +
	      'facebook_accesstoken facebook_check_permission ' +
	      'facebook_request_read_permissions ' +
	      'facebook_request_publish_permissions gamepad_is_supported ' +
	      'gamepad_get_device_count gamepad_is_connected ' +
	      'gamepad_get_description gamepad_get_button_threshold ' +
	      'gamepad_set_button_threshold gamepad_get_axis_deadzone ' +
	      'gamepad_set_axis_deadzone gamepad_button_count gamepad_button_check ' +
	      'gamepad_button_check_pressed gamepad_button_check_released ' +
	      'gamepad_button_value gamepad_axis_count gamepad_axis_value ' +
	      'gamepad_set_vibration gamepad_set_colour gamepad_set_color ' +
	      'os_is_paused window_has_focus code_is_compiled http_get ' +
	      'http_get_file http_post_string http_request json_encode json_decode ' +
	      'zip_unzip load_csv base64_encode base64_decode md5_string_unicode ' +
	      'md5_string_utf8 md5_file os_is_network_connected sha1_string_unicode ' +
	      'sha1_string_utf8 sha1_file os_powersave_enable analytics_event ' +
	      'analytics_event_ext win8_livetile_tile_notification ' +
	      'win8_livetile_tile_clear win8_livetile_badge_notification ' +
	      'win8_livetile_badge_clear win8_livetile_queue_enable ' +
	      'win8_secondarytile_pin win8_secondarytile_badge_notification ' +
	      'win8_secondarytile_delete win8_livetile_notification_begin ' +
	      'win8_livetile_notification_secondary_begin ' +
	      'win8_livetile_notification_expiry win8_livetile_notification_tag ' +
	      'win8_livetile_notification_text_add ' +
	      'win8_livetile_notification_image_add win8_livetile_notification_end ' +
	      'win8_appbar_enable win8_appbar_add_element ' +
	      'win8_appbar_remove_element win8_settingscharm_add_entry ' +
	      'win8_settingscharm_add_html_entry win8_settingscharm_add_xaml_entry ' +
	      'win8_settingscharm_set_xaml_property ' +
	      'win8_settingscharm_get_xaml_property win8_settingscharm_remove_entry ' +
	      'win8_share_image win8_share_screenshot win8_share_file ' +
	      'win8_share_url win8_share_text win8_search_enable ' +
	      'win8_search_disable win8_search_add_suggestions ' +
	      'win8_device_touchscreen_available win8_license_initialize_sandbox ' +
	      'win8_license_trial_version winphone_license_trial_version ' +
	      'winphone_tile_title winphone_tile_count winphone_tile_back_title ' +
	      'winphone_tile_back_content winphone_tile_back_content_wide ' +
	      'winphone_tile_front_image winphone_tile_front_image_small ' +
	      'winphone_tile_front_image_wide winphone_tile_back_image ' +
	      'winphone_tile_back_image_wide winphone_tile_background_colour ' +
	      'winphone_tile_background_color winphone_tile_icon_image ' +
	      'winphone_tile_small_icon_image winphone_tile_wide_content ' +
	      'winphone_tile_cycle_images winphone_tile_small_background_image ' +
	      'physics_world_create physics_world_gravity ' +
	      'physics_world_update_speed physics_world_update_iterations ' +
	      'physics_world_draw_debug physics_pause_enable physics_fixture_create ' +
	      'physics_fixture_set_kinematic physics_fixture_set_density ' +
	      'physics_fixture_set_awake physics_fixture_set_restitution ' +
	      'physics_fixture_set_friction physics_fixture_set_collision_group ' +
	      'physics_fixture_set_sensor physics_fixture_set_linear_damping ' +
	      'physics_fixture_set_angular_damping physics_fixture_set_circle_shape ' +
	      'physics_fixture_set_box_shape physics_fixture_set_edge_shape ' +
	      'physics_fixture_set_polygon_shape physics_fixture_set_chain_shape ' +
	      'physics_fixture_add_point physics_fixture_bind ' +
	      'physics_fixture_bind_ext physics_fixture_delete physics_apply_force ' +
	      'physics_apply_impulse physics_apply_angular_impulse ' +
	      'physics_apply_local_force physics_apply_local_impulse ' +
	      'physics_apply_torque physics_mass_properties physics_draw_debug ' +
	      'physics_test_overlap physics_remove_fixture physics_set_friction ' +
	      'physics_set_density physics_set_restitution physics_get_friction ' +
	      'physics_get_density physics_get_restitution ' +
	      'physics_joint_distance_create physics_joint_rope_create ' +
	      'physics_joint_revolute_create physics_joint_prismatic_create ' +
	      'physics_joint_pulley_create physics_joint_wheel_create ' +
	      'physics_joint_weld_create physics_joint_friction_create ' +
	      'physics_joint_gear_create physics_joint_enable_motor ' +
	      'physics_joint_get_value physics_joint_set_value physics_joint_delete ' +
	      'physics_particle_create physics_particle_delete ' +
	      'physics_particle_delete_region_circle ' +
	      'physics_particle_delete_region_box ' +
	      'physics_particle_delete_region_poly physics_particle_set_flags ' +
	      'physics_particle_set_category_flags physics_particle_draw ' +
	      'physics_particle_draw_ext physics_particle_count ' +
	      'physics_particle_get_data physics_particle_get_data_particle ' +
	      'physics_particle_group_begin physics_particle_group_circle ' +
	      'physics_particle_group_box physics_particle_group_polygon ' +
	      'physics_particle_group_add_point physics_particle_group_end ' +
	      'physics_particle_group_join physics_particle_group_delete ' +
	      'physics_particle_group_count physics_particle_group_get_data ' +
	      'physics_particle_group_get_mass physics_particle_group_get_inertia ' +
	      'physics_particle_group_get_centre_x ' +
	      'physics_particle_group_get_centre_y physics_particle_group_get_vel_x ' +
	      'physics_particle_group_get_vel_y physics_particle_group_get_ang_vel ' +
	      'physics_particle_group_get_x physics_particle_group_get_y ' +
	      'physics_particle_group_get_angle physics_particle_set_group_flags ' +
	      'physics_particle_get_group_flags physics_particle_get_max_count ' +
	      'physics_particle_get_radius physics_particle_get_density ' +
	      'physics_particle_get_damping physics_particle_get_gravity_scale ' +
	      'physics_particle_set_max_count physics_particle_set_radius ' +
	      'physics_particle_set_density physics_particle_set_damping ' +
	      'physics_particle_set_gravity_scale network_create_socket ' +
	      'network_create_socket_ext network_create_server ' +
	      'network_create_server_raw network_connect network_connect_raw ' +
	      'network_send_packet network_send_raw network_send_broadcast ' +
	      'network_send_udp network_send_udp_raw network_set_timeout ' +
	      'network_set_config network_resolve network_destroy buffer_create ' +
	      'buffer_write buffer_read buffer_seek buffer_get_surface ' +
	      'buffer_set_surface buffer_delete buffer_exists buffer_get_type ' +
	      'buffer_get_alignment buffer_poke buffer_peek buffer_save ' +
	      'buffer_save_ext buffer_load buffer_load_ext buffer_load_partial ' +
	      'buffer_copy buffer_fill buffer_get_size buffer_tell buffer_resize ' +
	      'buffer_md5 buffer_sha1 buffer_base64_encode buffer_base64_decode ' +
	      'buffer_base64_decode_ext buffer_sizeof buffer_get_address ' +
	      'buffer_create_from_vertex_buffer ' +
	      'buffer_create_from_vertex_buffer_ext buffer_copy_from_vertex_buffer ' +
	      'buffer_async_group_begin buffer_async_group_option ' +
	      'buffer_async_group_end buffer_load_async buffer_save_async ' +
	      'gml_release_mode gml_pragma steam_activate_overlay ' +
	      'steam_is_overlay_enabled steam_is_overlay_activated ' +
	      'steam_get_persona_name steam_initialised ' +
	      'steam_is_cloud_enabled_for_app steam_is_cloud_enabled_for_account ' +
	      'steam_file_persisted steam_get_quota_total steam_get_quota_free ' +
	      'steam_file_write steam_file_write_file steam_file_read ' +
	      'steam_file_delete steam_file_exists steam_file_size steam_file_share ' +
	      'steam_is_screenshot_requested steam_send_screenshot ' +
	      'steam_is_user_logged_on steam_get_user_steam_id steam_user_owns_dlc ' +
	      'steam_user_installed_dlc steam_set_achievement steam_get_achievement ' +
	      'steam_clear_achievement steam_set_stat_int steam_set_stat_float ' +
	      'steam_set_stat_avg_rate steam_get_stat_int steam_get_stat_float ' +
	      'steam_get_stat_avg_rate steam_reset_all_stats ' +
	      'steam_reset_all_stats_achievements steam_stats_ready ' +
	      'steam_create_leaderboard steam_upload_score steam_upload_score_ext ' +
	      'steam_download_scores_around_user steam_download_scores ' +
	      'steam_download_friends_scores steam_upload_score_buffer ' +
	      'steam_upload_score_buffer_ext steam_current_game_language ' +
	      'steam_available_languages steam_activate_overlay_browser ' +
	      'steam_activate_overlay_user steam_activate_overlay_store ' +
	      'steam_get_user_persona_name steam_get_app_id ' +
	      'steam_get_user_account_id steam_ugc_download steam_ugc_create_item ' +
	      'steam_ugc_start_item_update steam_ugc_set_item_title ' +
	      'steam_ugc_set_item_description steam_ugc_set_item_visibility ' +
	      'steam_ugc_set_item_tags steam_ugc_set_item_content ' +
	      'steam_ugc_set_item_preview steam_ugc_submit_item_update ' +
	      'steam_ugc_get_item_update_progress steam_ugc_subscribe_item ' +
	      'steam_ugc_unsubscribe_item steam_ugc_num_subscribed_items ' +
	      'steam_ugc_get_subscribed_items steam_ugc_get_item_install_info ' +
	      'steam_ugc_get_item_update_info steam_ugc_request_item_details ' +
	      'steam_ugc_create_query_user steam_ugc_create_query_user_ex ' +
	      'steam_ugc_create_query_all steam_ugc_create_query_all_ex ' +
	      'steam_ugc_query_set_cloud_filename_filter ' +
	      'steam_ugc_query_set_match_any_tag steam_ugc_query_set_search_text ' +
	      'steam_ugc_query_set_ranked_by_trend_days ' +
	      'steam_ugc_query_add_required_tag steam_ugc_query_add_excluded_tag ' +
	      'steam_ugc_query_set_return_long_description ' +
	      'steam_ugc_query_set_return_total_only ' +
	      'steam_ugc_query_set_allow_cached_response steam_ugc_send_query ' +
	      'shader_set shader_get_name shader_reset shader_current ' +
	      'shader_is_compiled shader_get_sampler_index shader_get_uniform ' +
	      'shader_set_uniform_i shader_set_uniform_i_array shader_set_uniform_f ' +
	      'shader_set_uniform_f_array shader_set_uniform_matrix ' +
	      'shader_set_uniform_matrix_array shader_enable_corner_id ' +
	      'texture_set_stage texture_get_texel_width texture_get_texel_height ' +
	      'shaders_are_supported vertex_format_begin vertex_format_end ' +
	      'vertex_format_delete vertex_format_add_position ' +
	      'vertex_format_add_position_3d vertex_format_add_colour ' +
	      'vertex_format_add_color vertex_format_add_normal ' +
	      'vertex_format_add_texcoord vertex_format_add_textcoord ' +
	      'vertex_format_add_custom vertex_create_buffer ' +
	      'vertex_create_buffer_ext vertex_delete_buffer vertex_begin ' +
	      'vertex_end vertex_position vertex_position_3d vertex_colour ' +
	      'vertex_color vertex_argb vertex_texcoord vertex_normal vertex_float1 ' +
	      'vertex_float2 vertex_float3 vertex_float4 vertex_ubyte4 ' +
	      'vertex_submit vertex_freeze vertex_get_number vertex_get_buffer_size ' +
	      'vertex_create_buffer_from_buffer ' +
	      'vertex_create_buffer_from_buffer_ext push_local_notification ' +
	      'push_get_first_local_notification push_get_next_local_notification ' +
	      'push_cancel_local_notification skeleton_animation_set ' +
	      'skeleton_animation_get skeleton_animation_mix ' +
	      'skeleton_animation_set_ext skeleton_animation_get_ext ' +
	      'skeleton_animation_get_duration skeleton_animation_get_frames ' +
	      'skeleton_animation_clear skeleton_skin_set skeleton_skin_get ' +
	      'skeleton_attachment_set skeleton_attachment_get ' +
	      'skeleton_attachment_create skeleton_collision_draw_set ' +
	      'skeleton_bone_data_get skeleton_bone_data_set ' +
	      'skeleton_bone_state_get skeleton_bone_state_set skeleton_get_minmax ' +
	      'skeleton_get_num_bounds skeleton_get_bounds ' +
	      'skeleton_animation_get_frame skeleton_animation_set_frame ' +
	      'draw_skeleton draw_skeleton_time draw_skeleton_instance ' +
	      'draw_skeleton_collision skeleton_animation_list skeleton_skin_list ' +
	      'skeleton_slot_data layer_get_id layer_get_id_at_depth ' +
	      'layer_get_depth layer_create layer_destroy layer_destroy_instances ' +
	      'layer_add_instance layer_has_instance layer_set_visible ' +
	      'layer_get_visible layer_exists layer_x layer_y layer_get_x ' +
	      'layer_get_y layer_hspeed layer_vspeed layer_get_hspeed ' +
	      'layer_get_vspeed layer_script_begin layer_script_end layer_shader ' +
	      'layer_get_script_begin layer_get_script_end layer_get_shader ' +
	      'layer_set_target_room layer_get_target_room layer_reset_target_room ' +
	      'layer_get_all layer_get_all_elements layer_get_name layer_depth ' +
	      'layer_get_element_layer layer_get_element_type layer_element_move ' +
	      'layer_force_draw_depth layer_is_draw_depth_forced ' +
	      'layer_get_forced_depth layer_background_get_id ' +
	      'layer_background_exists layer_background_create ' +
	      'layer_background_destroy layer_background_visible ' +
	      'layer_background_change layer_background_sprite ' +
	      'layer_background_htiled layer_background_vtiled ' +
	      'layer_background_stretch layer_background_yscale ' +
	      'layer_background_xscale layer_background_blend ' +
	      'layer_background_alpha layer_background_index layer_background_speed ' +
	      'layer_background_get_visible layer_background_get_sprite ' +
	      'layer_background_get_htiled layer_background_get_vtiled ' +
	      'layer_background_get_stretch layer_background_get_yscale ' +
	      'layer_background_get_xscale layer_background_get_blend ' +
	      'layer_background_get_alpha layer_background_get_index ' +
	      'layer_background_get_speed layer_sprite_get_id layer_sprite_exists ' +
	      'layer_sprite_create layer_sprite_destroy layer_sprite_change ' +
	      'layer_sprite_index layer_sprite_speed layer_sprite_xscale ' +
	      'layer_sprite_yscale layer_sprite_angle layer_sprite_blend ' +
	      'layer_sprite_alpha layer_sprite_x layer_sprite_y ' +
	      'layer_sprite_get_sprite layer_sprite_get_index ' +
	      'layer_sprite_get_speed layer_sprite_get_xscale ' +
	      'layer_sprite_get_yscale layer_sprite_get_angle ' +
	      'layer_sprite_get_blend layer_sprite_get_alpha layer_sprite_get_x ' +
	      'layer_sprite_get_y layer_tilemap_get_id layer_tilemap_exists ' +
	      'layer_tilemap_create layer_tilemap_destroy tilemap_tileset tilemap_x ' +
	      'tilemap_y tilemap_set tilemap_set_at_pixel tilemap_get_tileset ' +
	      'tilemap_get_tile_width tilemap_get_tile_height tilemap_get_width ' +
	      'tilemap_get_height tilemap_get_x tilemap_get_y tilemap_get ' +
	      'tilemap_get_at_pixel tilemap_get_cell_x_at_pixel ' +
	      'tilemap_get_cell_y_at_pixel tilemap_clear draw_tilemap draw_tile ' +
	      'tilemap_set_global_mask tilemap_get_global_mask tilemap_set_mask ' +
	      'tilemap_get_mask tilemap_get_frame tile_set_empty tile_set_index ' +
	      'tile_set_flip tile_set_mirror tile_set_rotate tile_get_empty ' +
	      'tile_get_index tile_get_flip tile_get_mirror tile_get_rotate ' +
	      'layer_tile_exists layer_tile_create layer_tile_destroy ' +
	      'layer_tile_change layer_tile_xscale layer_tile_yscale ' +
	      'layer_tile_blend layer_tile_alpha layer_tile_x layer_tile_y ' +
	      'layer_tile_region layer_tile_visible layer_tile_get_sprite ' +
	      'layer_tile_get_xscale layer_tile_get_yscale layer_tile_get_blend ' +
	      'layer_tile_get_alpha layer_tile_get_x layer_tile_get_y ' +
	      'layer_tile_get_region layer_tile_get_visible ' +
	      'layer_instance_get_instance instance_activate_layer ' +
	      'instance_deactivate_layer camera_create camera_create_view ' +
	      'camera_destroy camera_apply camera_get_active camera_get_default ' +
	      'camera_set_default camera_set_view_mat camera_set_proj_mat ' +
	      'camera_set_update_script camera_set_begin_script ' +
	      'camera_set_end_script camera_set_view_pos camera_set_view_size ' +
	      'camera_set_view_speed camera_set_view_border camera_set_view_angle ' +
	      'camera_set_view_target camera_get_view_mat camera_get_proj_mat ' +
	      'camera_get_update_script camera_get_begin_script ' +
	      'camera_get_end_script camera_get_view_x camera_get_view_y ' +
	      'camera_get_view_width camera_get_view_height camera_get_view_speed_x ' +
	      'camera_get_view_speed_y camera_get_view_border_x ' +
	      'camera_get_view_border_y camera_get_view_angle ' +
	      'camera_get_view_target view_get_camera view_get_visible ' +
	      'view_get_xport view_get_yport view_get_wport view_get_hport ' +
	      'view_get_surface_id view_set_camera view_set_visible view_set_xport ' +
	      'view_set_yport view_set_wport view_set_hport view_set_surface_id ' +
	      'gesture_drag_time gesture_drag_distance gesture_flick_speed ' +
	      'gesture_double_tap_time gesture_double_tap_distance ' +
	      'gesture_pinch_distance gesture_pinch_angle_towards ' +
	      'gesture_pinch_angle_away gesture_rotate_time gesture_rotate_angle ' +
	      'gesture_tap_count gesture_get_drag_time gesture_get_drag_distance ' +
	      'gesture_get_flick_speed gesture_get_double_tap_time ' +
	      'gesture_get_double_tap_distance gesture_get_pinch_distance ' +
	      'gesture_get_pinch_angle_towards gesture_get_pinch_angle_away ' +
	      'gesture_get_rotate_time gesture_get_rotate_angle ' +
	      'gesture_get_tap_count keyboard_virtual_show keyboard_virtual_hide ' +
	      'keyboard_virtual_status keyboard_virtual_height',
	    literal: 'self other all noone global local undefined pointer_invalid ' +
	      'pointer_null path_action_stop path_action_restart ' +
	      'path_action_continue path_action_reverse true false pi GM_build_date ' +
	      'GM_version GM_runtime_version  timezone_local timezone_utc ' +
	      'gamespeed_fps gamespeed_microseconds  ev_create ev_destroy ev_step ' +
	      'ev_alarm ev_keyboard ev_mouse ev_collision ev_other ev_draw ' +
	      'ev_draw_begin ev_draw_end ev_draw_pre ev_draw_post ev_keypress ' +
	      'ev_keyrelease ev_trigger ev_left_button ev_right_button ' +
	      'ev_middle_button ev_no_button ev_left_press ev_right_press ' +
	      'ev_middle_press ev_left_release ev_right_release ev_middle_release ' +
	      'ev_mouse_enter ev_mouse_leave ev_mouse_wheel_up ev_mouse_wheel_down ' +
	      'ev_global_left_button ev_global_right_button ev_global_middle_button ' +
	      'ev_global_left_press ev_global_right_press ev_global_middle_press ' +
	      'ev_global_left_release ev_global_right_release ' +
	      'ev_global_middle_release ev_joystick1_left ev_joystick1_right ' +
	      'ev_joystick1_up ev_joystick1_down ev_joystick1_button1 ' +
	      'ev_joystick1_button2 ev_joystick1_button3 ev_joystick1_button4 ' +
	      'ev_joystick1_button5 ev_joystick1_button6 ev_joystick1_button7 ' +
	      'ev_joystick1_button8 ev_joystick2_left ev_joystick2_right ' +
	      'ev_joystick2_up ev_joystick2_down ev_joystick2_button1 ' +
	      'ev_joystick2_button2 ev_joystick2_button3 ev_joystick2_button4 ' +
	      'ev_joystick2_button5 ev_joystick2_button6 ev_joystick2_button7 ' +
	      'ev_joystick2_button8 ev_outside ev_boundary ev_game_start ' +
	      'ev_game_end ev_room_start ev_room_end ev_no_more_lives ' +
	      'ev_animation_end ev_end_of_path ev_no_more_health ev_close_button ' +
	      'ev_user0 ev_user1 ev_user2 ev_user3 ev_user4 ev_user5 ev_user6 ' +
	      'ev_user7 ev_user8 ev_user9 ev_user10 ev_user11 ev_user12 ev_user13 ' +
	      'ev_user14 ev_user15 ev_step_normal ev_step_begin ev_step_end ev_gui ' +
	      'ev_gui_begin ev_gui_end ev_cleanup ev_gesture ev_gesture_tap ' +
	      'ev_gesture_double_tap ev_gesture_drag_start ev_gesture_dragging ' +
	      'ev_gesture_drag_end ev_gesture_flick ev_gesture_pinch_start ' +
	      'ev_gesture_pinch_in ev_gesture_pinch_out ev_gesture_pinch_end ' +
	      'ev_gesture_rotate_start ev_gesture_rotating ev_gesture_rotate_end ' +
	      'ev_global_gesture_tap ev_global_gesture_double_tap ' +
	      'ev_global_gesture_drag_start ev_global_gesture_dragging ' +
	      'ev_global_gesture_drag_end ev_global_gesture_flick ' +
	      'ev_global_gesture_pinch_start ev_global_gesture_pinch_in ' +
	      'ev_global_gesture_pinch_out ev_global_gesture_pinch_end ' +
	      'ev_global_gesture_rotate_start ev_global_gesture_rotating ' +
	      'ev_global_gesture_rotate_end vk_nokey vk_anykey vk_enter vk_return ' +
	      'vk_shift vk_control vk_alt vk_escape vk_space vk_backspace vk_tab ' +
	      'vk_pause vk_printscreen vk_left vk_right vk_up vk_down vk_home ' +
	      'vk_end vk_delete vk_insert vk_pageup vk_pagedown vk_f1 vk_f2 vk_f3 ' +
	      'vk_f4 vk_f5 vk_f6 vk_f7 vk_f8 vk_f9 vk_f10 vk_f11 vk_f12 vk_numpad0 ' +
	      'vk_numpad1 vk_numpad2 vk_numpad3 vk_numpad4 vk_numpad5 vk_numpad6 ' +
	      'vk_numpad7 vk_numpad8 vk_numpad9 vk_divide vk_multiply vk_subtract ' +
	      'vk_add vk_decimal vk_lshift vk_lcontrol vk_lalt vk_rshift ' +
	      'vk_rcontrol vk_ralt  mb_any mb_none mb_left mb_right mb_middle ' +
	      'c_aqua c_black c_blue c_dkgray c_fuchsia c_gray c_green c_lime ' +
	      'c_ltgray c_maroon c_navy c_olive c_purple c_red c_silver c_teal ' +
	      'c_white c_yellow c_orange fa_left fa_center fa_right fa_top ' +
	      'fa_middle fa_bottom pr_pointlist pr_linelist pr_linestrip ' +
	      'pr_trianglelist pr_trianglestrip pr_trianglefan bm_complex bm_normal ' +
	      'bm_add bm_max bm_subtract bm_zero bm_one bm_src_colour ' +
	      'bm_inv_src_colour bm_src_color bm_inv_src_color bm_src_alpha ' +
	      'bm_inv_src_alpha bm_dest_alpha bm_inv_dest_alpha bm_dest_colour ' +
	      'bm_inv_dest_colour bm_dest_color bm_inv_dest_color bm_src_alpha_sat ' +
	      'tf_point tf_linear tf_anisotropic mip_off mip_on mip_markedonly ' +
	      'audio_falloff_none audio_falloff_inverse_distance ' +
	      'audio_falloff_inverse_distance_clamped audio_falloff_linear_distance ' +
	      'audio_falloff_linear_distance_clamped ' +
	      'audio_falloff_exponent_distance ' +
	      'audio_falloff_exponent_distance_clamped audio_old_system ' +
	      'audio_new_system audio_mono audio_stereo audio_3d cr_default cr_none ' +
	      'cr_arrow cr_cross cr_beam cr_size_nesw cr_size_ns cr_size_nwse ' +
	      'cr_size_we cr_uparrow cr_hourglass cr_drag cr_appstart cr_handpoint ' +
	      'cr_size_all spritespeed_framespersecond ' +
	      'spritespeed_framespergameframe asset_object asset_unknown ' +
	      'asset_sprite asset_sound asset_room asset_path asset_script ' +
	      'asset_font asset_timeline asset_tiles asset_shader fa_readonly ' +
	      'fa_hidden fa_sysfile fa_volumeid fa_directory fa_archive  ' +
	      'ds_type_map ds_type_list ds_type_stack ds_type_queue ds_type_grid ' +
	      'ds_type_priority ef_explosion ef_ring ef_ellipse ef_firework ' +
	      'ef_smoke ef_smokeup ef_star ef_spark ef_flare ef_cloud ef_rain ' +
	      'ef_snow pt_shape_pixel pt_shape_disk pt_shape_square pt_shape_line ' +
	      'pt_shape_star pt_shape_circle pt_shape_ring pt_shape_sphere ' +
	      'pt_shape_flare pt_shape_spark pt_shape_explosion pt_shape_cloud ' +
	      'pt_shape_smoke pt_shape_snow ps_distr_linear ps_distr_gaussian ' +
	      'ps_distr_invgaussian ps_shape_rectangle ps_shape_ellipse ' +
	      'ps_shape_diamond ps_shape_line ty_real ty_string dll_cdecl ' +
	      'dll_stdcall matrix_view matrix_projection matrix_world os_win32 ' +
	      'os_windows os_macosx os_ios os_android os_symbian os_linux ' +
	      'os_unknown os_winphone os_tizen os_win8native ' +
	      'os_wiiu os_3ds  os_psvita os_bb10 os_ps4 os_xboxone ' +
	      'os_ps3 os_xbox360 os_uwp os_tvos os_switch ' +
	      'browser_not_a_browser browser_unknown browser_ie browser_firefox ' +
	      'browser_chrome browser_safari browser_safari_mobile browser_opera ' +
	      'browser_tizen browser_edge browser_windows_store browser_ie_mobile  ' +
	      'device_ios_unknown device_ios_iphone device_ios_iphone_retina ' +
	      'device_ios_ipad device_ios_ipad_retina device_ios_iphone5 ' +
	      'device_ios_iphone6 device_ios_iphone6plus device_emulator ' +
	      'device_tablet display_landscape display_landscape_flipped ' +
	      'display_portrait display_portrait_flipped tm_sleep tm_countvsyncs ' +
	      'of_challenge_win of_challen ge_lose of_challenge_tie ' +
	      'leaderboard_type_number leaderboard_type_time_mins_secs ' +
	      'cmpfunc_never cmpfunc_less cmpfunc_equal cmpfunc_lessequal ' +
	      'cmpfunc_greater cmpfunc_notequal cmpfunc_greaterequal cmpfunc_always ' +
	      'cull_noculling cull_clockwise cull_counterclockwise lighttype_dir ' +
	      'lighttype_point iap_ev_storeload iap_ev_product iap_ev_purchase ' +
	      'iap_ev_consume iap_ev_restore iap_storeload_ok iap_storeload_failed ' +
	      'iap_status_uninitialised iap_status_unavailable iap_status_loading ' +
	      'iap_status_available iap_status_processing iap_status_restoring ' +
	      'iap_failed iap_unavailable iap_available iap_purchased iap_canceled ' +
	      'iap_refunded fb_login_default fb_login_fallback_to_webview ' +
	      'fb_login_no_fallback_to_webview fb_login_forcing_webview ' +
	      'fb_login_use_system_account fb_login_forcing_safari  ' +
	      'phy_joint_anchor_1_x phy_joint_anchor_1_y phy_joint_anchor_2_x ' +
	      'phy_joint_anchor_2_y phy_joint_reaction_force_x ' +
	      'phy_joint_reaction_force_y phy_joint_reaction_torque ' +
	      'phy_joint_motor_speed phy_joint_angle phy_joint_motor_torque ' +
	      'phy_joint_max_motor_torque phy_joint_translation phy_joint_speed ' +
	      'phy_joint_motor_force phy_joint_max_motor_force phy_joint_length_1 ' +
	      'phy_joint_length_2 phy_joint_damping_ratio phy_joint_frequency ' +
	      'phy_joint_lower_angle_limit phy_joint_upper_angle_limit ' +
	      'phy_joint_angle_limits phy_joint_max_length phy_joint_max_torque ' +
	      'phy_joint_max_force phy_debug_render_aabb ' +
	      'phy_debug_render_collision_pairs phy_debug_render_coms ' +
	      'phy_debug_render_core_shapes phy_debug_render_joints ' +
	      'phy_debug_render_obb phy_debug_render_shapes  ' +
	      'phy_particle_flag_water phy_particle_flag_zombie ' +
	      'phy_particle_flag_wall phy_particle_flag_spring ' +
	      'phy_particle_flag_elastic phy_particle_flag_viscous ' +
	      'phy_particle_flag_powder phy_particle_flag_tensile ' +
	      'phy_particle_flag_colourmixing phy_particle_flag_colormixing ' +
	      'phy_particle_group_flag_solid phy_particle_group_flag_rigid ' +
	      'phy_particle_data_flag_typeflags phy_particle_data_flag_position ' +
	      'phy_particle_data_flag_velocity phy_particle_data_flag_colour ' +
	      'phy_particle_data_flag_color phy_particle_data_flag_category  ' +
	      'achievement_our_info achievement_friends_info ' +
	      'achievement_leaderboard_info achievement_achievement_info ' +
	      'achievement_filter_all_players achievement_filter_friends_only ' +
	      'achievement_filter_favorites_only ' +
	      'achievement_type_achievement_challenge ' +
	      'achievement_type_score_challenge achievement_pic_loaded  ' +
	      'achievement_show_ui achievement_show_profile ' +
	      'achievement_show_leaderboard achievement_show_achievement ' +
	      'achievement_show_bank achievement_show_friend_picker ' +
	      'achievement_show_purchase_prompt network_socket_tcp ' +
	      'network_socket_udp network_socket_bluetooth network_type_connect ' +
	      'network_type_disconnect network_type_data ' +
	      'network_type_non_blocking_connect network_config_connect_timeout ' +
	      'network_config_use_non_blocking_socket ' +
	      'network_config_enable_reliable_udp ' +
	      'network_config_disable_reliable_udp buffer_fixed buffer_grow ' +
	      'buffer_wrap buffer_fast buffer_vbuffer buffer_network buffer_u8 ' +
	      'buffer_s8 buffer_u16 buffer_s16 buffer_u32 buffer_s32 buffer_u64 ' +
	      'buffer_f16 buffer_f32 buffer_f64 buffer_bool buffer_text ' +
	      'buffer_string buffer_surface_copy buffer_seek_start ' +
	      'buffer_seek_relative buffer_seek_end ' +
	      'buffer_generalerror buffer_outofspace buffer_outofbounds ' +
	      'buffer_invalidtype  text_type button_type input_type ANSI_CHARSET ' +
	      'DEFAULT_CHARSET EASTEUROPE_CHARSET RUSSIAN_CHARSET SYMBOL_CHARSET ' +
	      'SHIFTJIS_CHARSET HANGEUL_CHARSET GB2312_CHARSET CHINESEBIG5_CHARSET ' +
	      'JOHAB_CHARSET HEBREW_CHARSET ARABIC_CHARSET GREEK_CHARSET ' +
	      'TURKISH_CHARSET VIETNAMESE_CHARSET THAI_CHARSET MAC_CHARSET ' +
	      'BALTIC_CHARSET OEM_CHARSET  gp_face1 gp_face2 gp_face3 gp_face4 ' +
	      'gp_shoulderl gp_shoulderr gp_shoulderlb gp_shoulderrb gp_select ' +
	      'gp_start gp_stickl gp_stickr gp_padu gp_padd gp_padl gp_padr ' +
	      'gp_axislh gp_axislv gp_axisrh gp_axisrv ov_friends ov_community ' +
	      'ov_players ov_settings ov_gamegroup ov_achievements lb_sort_none ' +
	      'lb_sort_ascending lb_sort_descending lb_disp_none lb_disp_numeric ' +
	      'lb_disp_time_sec lb_disp_time_ms ugc_result_success ' +
	      'ugc_filetype_community ugc_filetype_microtrans ugc_visibility_public ' +
	      'ugc_visibility_friends_only ugc_visibility_private ' +
	      'ugc_query_RankedByVote ugc_query_RankedByPublicationDate ' +
	      'ugc_query_AcceptedForGameRankedByAcceptanceDate ' +
	      'ugc_query_RankedByTrend ' +
	      'ugc_query_FavoritedByFriendsRankedByPublicationDate ' +
	      'ugc_query_CreatedByFriendsRankedByPublicationDate ' +
	      'ugc_query_RankedByNumTimesReported ' +
	      'ugc_query_CreatedByFollowedUsersRankedByPublicationDate ' +
	      'ugc_query_NotYetRated ugc_query_RankedByTotalVotesAsc ' +
	      'ugc_query_RankedByVotesUp ugc_query_RankedByTextSearch ' +
	      'ugc_sortorder_CreationOrderDesc ugc_sortorder_CreationOrderAsc ' +
	      'ugc_sortorder_TitleAsc ugc_sortorder_LastUpdatedDesc ' +
	      'ugc_sortorder_SubscriptionDateDesc ugc_sortorder_VoteScoreDesc ' +
	      'ugc_sortorder_ForModeration ugc_list_Published ugc_list_VotedOn ' +
	      'ugc_list_VotedUp ugc_list_VotedDown ugc_list_WillVoteLater ' +
	      'ugc_list_Favorited ugc_list_Subscribed ugc_list_UsedOrPlayed ' +
	      'ugc_list_Followed ugc_match_Items ugc_match_Items_Mtx ' +
	      'ugc_match_Items_ReadyToUse ugc_match_Collections ugc_match_Artwork ' +
	      'ugc_match_Videos ugc_match_Screenshots ugc_match_AllGuides ' +
	      'ugc_match_WebGuides ugc_match_IntegratedGuides ' +
	      'ugc_match_UsableInGame ugc_match_ControllerBindings  ' +
	      'vertex_usage_position vertex_usage_colour vertex_usage_color ' +
	      'vertex_usage_normal vertex_usage_texcoord vertex_usage_textcoord ' +
	      'vertex_usage_blendweight vertex_usage_blendindices ' +
	      'vertex_usage_psize vertex_usage_tangent vertex_usage_binormal ' +
	      'vertex_usage_fog vertex_usage_depth vertex_usage_sample ' +
	      'vertex_type_float1 vertex_type_float2 vertex_type_float3 ' +
	      'vertex_type_float4 vertex_type_colour vertex_type_color ' +
	      'vertex_type_ubyte4 layerelementtype_undefined ' +
	      'layerelementtype_background layerelementtype_instance ' +
	      'layerelementtype_oldtilemap layerelementtype_sprite ' +
	      'layerelementtype_tilemap layerelementtype_particlesystem ' +
	      'layerelementtype_tile tile_rotate tile_flip tile_mirror ' +
	      'tile_index_mask kbv_type_default kbv_type_ascii kbv_type_url ' +
	      'kbv_type_email kbv_type_numbers kbv_type_phone kbv_type_phone_name ' +
	      'kbv_returnkey_default kbv_returnkey_go kbv_returnkey_google ' +
	      'kbv_returnkey_join kbv_returnkey_next kbv_returnkey_route ' +
	      'kbv_returnkey_search kbv_returnkey_send kbv_returnkey_yahoo ' +
	      'kbv_returnkey_done kbv_returnkey_continue kbv_returnkey_emergency ' +
	      'kbv_autocapitalize_none kbv_autocapitalize_words ' +
	      'kbv_autocapitalize_sentences kbv_autocapitalize_characters',
	    symbol: 'argument_relative argument argument0 argument1 argument2 ' +
	      'argument3 argument4 argument5 argument6 argument7 argument8 ' +
	      'argument9 argument10 argument11 argument12 argument13 argument14 ' +
	      'argument15 argument_count x|0 y|0 xprevious yprevious xstart ystart ' +
	      'hspeed vspeed direction speed friction gravity gravity_direction ' +
	      'path_index path_position path_positionprevious path_speed ' +
	      'path_scale path_orientation path_endaction object_index id solid ' +
	      'persistent mask_index instance_count instance_id room_speed fps ' +
	      'fps_real current_time current_year current_month current_day ' +
	      'current_weekday current_hour current_minute current_second alarm ' +
	      'timeline_index timeline_position timeline_speed timeline_running ' +
	      'timeline_loop room room_first room_last room_width room_height ' +
	      'room_caption room_persistent score lives health show_score ' +
	      'show_lives show_health caption_score caption_lives caption_health ' +
	      'event_type event_number event_object event_action ' +
	      'application_surface gamemaker_pro gamemaker_registered ' +
	      'gamemaker_version error_occurred error_last debug_mode ' +
	      'keyboard_key keyboard_lastkey keyboard_lastchar keyboard_string ' +
	      'mouse_x mouse_y mouse_button mouse_lastbutton cursor_sprite ' +
	      'visible sprite_index sprite_width sprite_height sprite_xoffset ' +
	      'sprite_yoffset image_number image_index image_speed depth ' +
	      'image_xscale image_yscale image_angle image_alpha image_blend ' +
	      'bbox_left bbox_right bbox_top bbox_bottom layer background_colour  ' +
	      'background_showcolour background_color background_showcolor ' +
	      'view_enabled view_current view_visible view_xview view_yview ' +
	      'view_wview view_hview view_xport view_yport view_wport view_hport ' +
	      'view_angle view_hborder view_vborder view_hspeed view_vspeed ' +
	      'view_object view_surface_id view_camera game_id game_display_name ' +
	      'game_project_name game_save_id working_directory temp_directory ' +
	      'program_directory browser_width browser_height os_type os_device ' +
	      'os_browser os_version display_aa async_load delta_time ' +
	      'webgl_enabled event_data iap_data phy_rotation phy_position_x ' +
	      'phy_position_y phy_angular_velocity phy_linear_velocity_x ' +
	      'phy_linear_velocity_y phy_speed_x phy_speed_y phy_speed ' +
	      'phy_angular_damping phy_linear_damping phy_bullet ' +
	      'phy_fixed_rotation phy_active phy_mass phy_inertia phy_com_x ' +
	      'phy_com_y phy_dynamic phy_kinematic phy_sleeping ' +
	      'phy_collision_points phy_collision_x phy_collision_y ' +
	      'phy_col_normal_x phy_col_normal_y phy_position_xprevious ' +
	      'phy_position_yprevious'
	  };

	  return {
	    name: 'GML',
	    case_insensitive: false, // language is case-insensitive
	    keywords: GML_KEYWORDS,

	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.C_NUMBER_MODE
	    ]
	  };
	}

	gml_1 = gml;
	return gml_1;
}

/*
Language: Go
Author: Stephan Kountso aka StepLg <steplg@gmail.com>
Contributors: Evgeny Stepanischev <imbolk@gmail.com>
Description: Google go language (golang). For info about language
Website: http://golang.org/
Category: common, system
*/

var go_1;
var hasRequiredGo;

function requireGo () {
	if (hasRequiredGo) return go_1;
	hasRequiredGo = 1;
	function go(hljs) {
	  const GO_KEYWORDS = {
	    keyword:
	      'break default func interface select case map struct chan else goto package switch ' +
	      'const fallthrough if range type continue for import return var go defer ' +
	      'bool byte complex64 complex128 float32 float64 int8 int16 int32 int64 string uint8 ' +
	      'uint16 uint32 uint64 int uint uintptr rune',
	    literal:
	       'true false iota nil',
	    built_in:
	      'append cap close complex copy imag len make new panic print println real recover delete'
	  };
	  return {
	    name: 'Go',
	    aliases: ['golang'],
	    keywords: GO_KEYWORDS,
	    illegal: '</',
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      {
	        className: 'string',
	        variants: [
	          hljs.QUOTE_STRING_MODE,
	          hljs.APOS_STRING_MODE,
	          {
	            begin: '`',
	            end: '`'
	          }
	        ]
	      },
	      {
	        className: 'number',
	        variants: [
	          {
	            begin: hljs.C_NUMBER_RE + '[i]',
	            relevance: 1
	          },
	          hljs.C_NUMBER_MODE
	        ]
	      },
	      {
	        begin: /:=/ // relevance booster
	      },
	      {
	        className: 'function',
	        beginKeywords: 'func',
	        end: '\\s*(\\{|$)',
	        excludeEnd: true,
	        contains: [
	          hljs.TITLE_MODE,
	          {
	            className: 'params',
	            begin: /\(/,
	            end: /\)/,
	            keywords: GO_KEYWORDS,
	            illegal: /["']/
	          }
	        ]
	      }
	    ]
	  };
	}

	go_1 = go;
	return go_1;
}

/*
Language: Golo
Author: Philippe Charriere <ph.charriere@gmail.com>
Description: a lightweight dynamic language for the JVM
Website: http://golo-lang.org/
*/

var golo_1;
var hasRequiredGolo;

function requireGolo () {
	if (hasRequiredGolo) return golo_1;
	hasRequiredGolo = 1;
	function golo(hljs) {
	  return {
	    name: 'Golo',
	    keywords: {
	      keyword:
	          'println readln print import module function local return let var ' +
	          'while for foreach times in case when match with break continue ' +
	          'augment augmentation each find filter reduce ' +
	          'if then else otherwise try catch finally raise throw orIfNull ' +
	          'DynamicObject|10 DynamicVariable struct Observable map set vector list array',
	      literal:
	          'true false null'
	    },
	    contains: [
	      hljs.HASH_COMMENT_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.C_NUMBER_MODE,
	      {
	        className: 'meta',
	        begin: '@[A-Za-z]+'
	      }
	    ]
	  };
	}

	golo_1 = golo;
	return golo_1;
}

/*
Language: Gradle
Description: Gradle is an open-source build automation tool focused on flexibility and performance.
Website: https://gradle.org
Author: Damian Mee <mee.damian@gmail.com>
*/

var gradle_1;
var hasRequiredGradle;

function requireGradle () {
	if (hasRequiredGradle) return gradle_1;
	hasRequiredGradle = 1;
	function gradle(hljs) {
	  return {
	    name: 'Gradle',
	    case_insensitive: true,
	    keywords: {
	      keyword:
	        'task project allprojects subprojects artifacts buildscript configurations ' +
	        'dependencies repositories sourceSets description delete from into include ' +
	        'exclude source classpath destinationDir includes options sourceCompatibility ' +
	        'targetCompatibility group flatDir doLast doFirst flatten todir fromdir ant ' +
	        'def abstract break case catch continue default do else extends final finally ' +
	        'for if implements instanceof native new private protected public return static ' +
	        'switch synchronized throw throws transient try volatile while strictfp package ' +
	        'import false null super this true antlrtask checkstyle codenarc copy boolean ' +
	        'byte char class double float int interface long short void compile runTime ' +
	        'file fileTree abs any append asList asWritable call collect compareTo count ' +
	        'div dump each eachByte eachFile eachLine every find findAll flatten getAt ' +
	        'getErr getIn getOut getText grep immutable inject inspect intersect invokeMethods ' +
	        'isCase join leftShift minus multiply newInputStream newOutputStream newPrintWriter ' +
	        'newReader newWriter next plus pop power previous print println push putAt read ' +
	        'readBytes readLines reverse reverseEach round size sort splitEachLine step subMap ' +
	        'times toInteger toList tokenize upto waitForOrKill withPrintWriter withReader ' +
	        'withStream withWriter withWriterAppend write writeLine'
	    },
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.NUMBER_MODE,
	      hljs.REGEXP_MODE

	    ]
	  };
	}

	gradle_1 = gradle;
	return gradle_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var groovy_1;
var hasRequiredGroovy;

function requireGroovy () {
	if (hasRequiredGroovy) return groovy_1;
	hasRequiredGroovy = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	 Language: Groovy
	 Author: Guillaume Laforge <glaforge@gmail.com>
	 Description: Groovy programming language implementation inspired from Vsevolod's Java mode
	 Website: https://groovy-lang.org
	 */

	function variants(variants, obj = {}) {
	  obj.variants = variants;
	  return obj;
	}

	function groovy(hljs) {
	  const IDENT_RE = '[A-Za-z0-9_$]+';
	  const COMMENT = variants([
	    hljs.C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    hljs.COMMENT(
	      '/\\*\\*',
	      '\\*/',
	      {
	        relevance: 0,
	        contains: [
	          {
	            // eat up @'s in emails to prevent them to be recognized as doctags
	            begin: /\w+@/,
	            relevance: 0
	          },
	          {
	            className: 'doctag',
	            begin: '@[A-Za-z]+'
	          }
	        ]
	      }
	    )
	  ]);
	  const REGEXP = {
	    className: 'regexp',
	    begin: /~?\/[^\/\n]+\//,
	    contains: [ hljs.BACKSLASH_ESCAPE ]
	  };
	  const NUMBER = variants([
	    hljs.BINARY_NUMBER_MODE,
	    hljs.C_NUMBER_MODE
	  ]);
	  const STRING = variants([
	    {
	      begin: /"""/,
	      end: /"""/
	    },
	    {
	      begin: /'''/,
	      end: /'''/
	    },
	    {
	      begin: "\\$/",
	      end: "/\\$",
	      relevance: 10
	    },
	    hljs.APOS_STRING_MODE,
	    hljs.QUOTE_STRING_MODE
	  ],
	  {
	    className: "string"
	  }
	  );

	  return {
	    name: 'Groovy',
	    keywords: {
	      built_in: 'this super',
	      literal: 'true false null',
	      keyword:
	            'byte short char int long boolean float double void ' +
	            // groovy specific keywords
	            'def as in assert trait ' +
	            // common keywords with Java
	            'abstract static volatile transient public private protected synchronized final ' +
	            'class interface enum if else for while switch case break default continue ' +
	            'throw throws try catch finally implements extends new import package return instanceof'
	    },
	    contains: [
	      hljs.SHEBANG({
	        binary: "groovy",
	        relevance: 10
	      }),
	      COMMENT,
	      STRING,
	      REGEXP,
	      NUMBER,
	      {
	        className: 'class',
	        beginKeywords: 'class interface trait enum',
	        end: /\{/,
	        illegal: ':',
	        contains: [
	          {
	            beginKeywords: 'extends implements'
	          },
	          hljs.UNDERSCORE_TITLE_MODE
	        ]
	      },
	      {
	        className: 'meta',
	        begin: '@[A-Za-z]+',
	        relevance: 0
	      },
	      {
	        // highlight map keys and named parameters as attrs
	        className: 'attr',
	        begin: IDENT_RE + '[ \t]*:',
	        relevance: 0
	      },
	      {
	        // catch middle element of the ternary operator
	        // to avoid highlight it as a label, named parameter, or map key
	        begin: /\?/,
	        end: /:/,
	        relevance: 0,
	        contains: [
	          COMMENT,
	          STRING,
	          REGEXP,
	          NUMBER,
	          'self'
	        ]
	      },
	      {
	        // highlight labeled statements
	        className: 'symbol',
	        begin: '^[ \t]*' + lookahead(IDENT_RE + ':'),
	        excludeBegin: true,
	        end: IDENT_RE + ':',
	        relevance: 0
	      }
	    ],
	    illegal: /#|<\//
	  };
	}

	groovy_1 = groovy;
	return groovy_1;
}

/*
Language: HAML
Requires: ruby.js
Author: Dan Allen <dan.j.allen@gmail.com>
Website: http://haml.info
Category: template
*/

var haml_1;
var hasRequiredHaml;

function requireHaml () {
	if (hasRequiredHaml) return haml_1;
	hasRequiredHaml = 1;
	// TODO support filter tags like :javascript, support inline HTML
	function haml(hljs) {
	  return {
	    name: 'HAML',
	    case_insensitive: true,
	    contains: [
	      {
	        className: 'meta',
	        begin: '^!!!( (5|1\\.1|Strict|Frameset|Basic|Mobile|RDFa|XML\\b.*))?$',
	        relevance: 10
	      },
	      // FIXME these comments should be allowed to span indented lines
	      hljs.COMMENT(
	        '^\\s*(!=#|=#|-#|/).*$',
	        false,
	        {
	          relevance: 0
	        }
	      ),
	      {
	        begin: '^\\s*(-|=|!=)(?!#)',
	        starts: {
	          end: '\\n',
	          subLanguage: 'ruby'
	        }
	      },
	      {
	        className: 'tag',
	        begin: '^\\s*%',
	        contains: [
	          {
	            className: 'selector-tag',
	            begin: '\\w+'
	          },
	          {
	            className: 'selector-id',
	            begin: '#[\\w-]+'
	          },
	          {
	            className: 'selector-class',
	            begin: '\\.[\\w-]+'
	          },
	          {
	            begin: /\{\s*/,
	            end: /\s*\}/,
	            contains: [
	              {
	                begin: ':\\w+\\s*=>',
	                end: ',\\s+',
	                returnBegin: true,
	                endsWithParent: true,
	                contains: [
	                  {
	                    className: 'attr',
	                    begin: ':\\w+'
	                  },
	                  hljs.APOS_STRING_MODE,
	                  hljs.QUOTE_STRING_MODE,
	                  {
	                    begin: '\\w+',
	                    relevance: 0
	                  }
	                ]
	              }
	            ]
	          },
	          {
	            begin: '\\(\\s*',
	            end: '\\s*\\)',
	            excludeEnd: true,
	            contains: [
	              {
	                begin: '\\w+\\s*=',
	                end: '\\s+',
	                returnBegin: true,
	                endsWithParent: true,
	                contains: [
	                  {
	                    className: 'attr',
	                    begin: '\\w+',
	                    relevance: 0
	                  },
	                  hljs.APOS_STRING_MODE,
	                  hljs.QUOTE_STRING_MODE,
	                  {
	                    begin: '\\w+',
	                    relevance: 0
	                  }
	                ]
	              }
	            ]
	          }
	        ]
	      },
	      {
	        begin: '^\\s*[=~]\\s*'
	      },
	      {
	        begin: /#\{/,
	        starts: {
	          end: /\}/,
	          subLanguage: 'ruby'
	        }
	      }
	    ]
	  };
	}

	haml_1 = haml;
	return haml_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var handlebars_1;
var hasRequiredHandlebars;

function requireHandlebars () {
	if (hasRequiredHandlebars) return handlebars_1;
	hasRequiredHandlebars = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function anyNumberOfTimes(re) {
	  return concat('(', re, ')*');
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function optional(re) {
	  return concat('(', re, ')?');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/*
	Language: Handlebars
	Requires: xml.js
	Author: Robin Ward <robin.ward@gmail.com>
	Description: Matcher for Handlebars as well as EmberJS additions.
	Website: https://handlebarsjs.com
	Category: template
	*/

	function handlebars(hljs) {
	  const BUILT_INS = {
	    'builtin-name': [
	      'action',
	      'bindattr',
	      'collection',
	      'component',
	      'concat',
	      'debugger',
	      'each',
	      'each-in',
	      'get',
	      'hash',
	      'if',
	      'in',
	      'input',
	      'link-to',
	      'loc',
	      'log',
	      'lookup',
	      'mut',
	      'outlet',
	      'partial',
	      'query-params',
	      'render',
	      'template',
	      'textarea',
	      'unbound',
	      'unless',
	      'view',
	      'with',
	      'yield'
	    ]
	  };

	  const LITERALS = {
	    literal: [
	      'true',
	      'false',
	      'undefined',
	      'null'
	    ]
	  };

	  // as defined in https://handlebarsjs.com/guide/expressions.html#literal-segments
	  // this regex matches literal segments like ' abc ' or [ abc ] as well as helpers and paths
	  // like a/b, ./abc/cde, and abc.bcd

	  const DOUBLE_QUOTED_ID_REGEX = /""|"[^"]+"/;
	  const SINGLE_QUOTED_ID_REGEX = /''|'[^']+'/;
	  const BRACKET_QUOTED_ID_REGEX = /\[\]|\[[^\]]+\]/;
	  const PLAIN_ID_REGEX = /[^\s!"#%&'()*+,.\/;<=>@\[\\\]^`{|}~]+/;
	  const PATH_DELIMITER_REGEX = /(\.|\/)/;
	  const ANY_ID = either(
	    DOUBLE_QUOTED_ID_REGEX,
	    SINGLE_QUOTED_ID_REGEX,
	    BRACKET_QUOTED_ID_REGEX,
	    PLAIN_ID_REGEX
	    );

	  const IDENTIFIER_REGEX = concat(
	    optional(/\.|\.\/|\//), // relative or absolute path
	    ANY_ID,
	    anyNumberOfTimes(concat(
	      PATH_DELIMITER_REGEX,
	      ANY_ID
	    ))
	  );

	  // identifier followed by a equal-sign (without the equal sign)
	  const HASH_PARAM_REGEX = concat(
	    '(',
	    BRACKET_QUOTED_ID_REGEX, '|',
	    PLAIN_ID_REGEX,
	    ')(?==)'
	  );

	  const HELPER_NAME_OR_PATH_EXPRESSION = {
	    begin: IDENTIFIER_REGEX,
	    lexemes: /[\w.\/]+/
	  };

	  const HELPER_PARAMETER = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    keywords: LITERALS
	  });

	  const SUB_EXPRESSION = {
	    begin: /\(/,
	    end: /\)/
	    // the "contains" is added below when all necessary sub-modes are defined
	  };

	  const HASH = {
	    // fka "attribute-assignment", parameters of the form 'key=value'
	    className: 'attr',
	    begin: HASH_PARAM_REGEX,
	    relevance: 0,
	    starts: {
	      begin: /=/,
	      end: /=/,
	      starts: {
	        contains: [
	          hljs.NUMBER_MODE,
	          hljs.QUOTE_STRING_MODE,
	          hljs.APOS_STRING_MODE,
	          HELPER_PARAMETER,
	          SUB_EXPRESSION
	        ]
	      }
	    }
	  };

	  const BLOCK_PARAMS = {
	    // parameters of the form '{{#with x as | y |}}...{{/with}}'
	    begin: /as\s+\|/,
	    keywords: {
	      keyword: 'as'
	    },
	    end: /\|/,
	    contains: [
	      {
	        // define sub-mode in order to prevent highlighting of block-parameter named "as"
	        begin: /\w+/
	      }
	    ]
	  };

	  const HELPER_PARAMETERS = {
	    contains: [
	      hljs.NUMBER_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.APOS_STRING_MODE,
	      BLOCK_PARAMS,
	      HASH,
	      HELPER_PARAMETER,
	      SUB_EXPRESSION
	    ],
	    returnEnd: true
	    // the property "end" is defined through inheritance when the mode is used. If depends
	    // on the surrounding mode, but "endsWithParent" does not work here (i.e. it includes the
	    // end-token of the surrounding mode)
	  };

	  const SUB_EXPRESSION_CONTENTS = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    className: 'name',
	    keywords: BUILT_INS,
	    starts: hljs.inherit(HELPER_PARAMETERS, {
	      end: /\)/
	    })
	  });

	  SUB_EXPRESSION.contains = [SUB_EXPRESSION_CONTENTS];

	  const OPENING_BLOCK_MUSTACHE_CONTENTS = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    keywords: BUILT_INS,
	    className: 'name',
	    starts: hljs.inherit(HELPER_PARAMETERS, {
	      end: /\}\}/
	    })
	  });

	  const CLOSING_BLOCK_MUSTACHE_CONTENTS = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    keywords: BUILT_INS,
	    className: 'name'
	  });

	  const BASIC_MUSTACHE_CONTENTS = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    className: 'name',
	    keywords: BUILT_INS,
	    starts: hljs.inherit(HELPER_PARAMETERS, {
	      end: /\}\}/
	    })
	  });

	  const ESCAPE_MUSTACHE_WITH_PRECEEDING_BACKSLASH = {
	    begin: /\\\{\{/,
	    skip: true
	  };
	  const PREVENT_ESCAPE_WITH_ANOTHER_PRECEEDING_BACKSLASH = {
	    begin: /\\\\(?=\{\{)/,
	    skip: true
	  };

	  return {
	    name: 'Handlebars',
	    aliases: [
	      'hbs',
	      'html.hbs',
	      'html.handlebars',
	      'htmlbars'
	    ],
	    case_insensitive: true,
	    subLanguage: 'xml',
	    contains: [
	      ESCAPE_MUSTACHE_WITH_PRECEEDING_BACKSLASH,
	      PREVENT_ESCAPE_WITH_ANOTHER_PRECEEDING_BACKSLASH,
	      hljs.COMMENT(/\{\{!--/, /--\}\}/),
	      hljs.COMMENT(/\{\{!/, /\}\}/),
	      {
	        // open raw block "{{{{raw}}}} content not evaluated {{{{/raw}}}}"
	        className: 'template-tag',
	        begin: /\{\{\{\{(?!\/)/,
	        end: /\}\}\}\}/,
	        contains: [OPENING_BLOCK_MUSTACHE_CONTENTS],
	        starts: {
	          end: /\{\{\{\{\//,
	          returnEnd: true,
	          subLanguage: 'xml'
	        }
	      },
	      {
	        // close raw block
	        className: 'template-tag',
	        begin: /\{\{\{\{\//,
	        end: /\}\}\}\}/,
	        contains: [CLOSING_BLOCK_MUSTACHE_CONTENTS]
	      },
	      {
	        // open block statement
	        className: 'template-tag',
	        begin: /\{\{#/,
	        end: /\}\}/,
	        contains: [OPENING_BLOCK_MUSTACHE_CONTENTS]
	      },
	      {
	        className: 'template-tag',
	        begin: /\{\{(?=else\}\})/,
	        end: /\}\}/,
	        keywords: 'else'
	      },
	      {
	        className: 'template-tag',
	        begin: /\{\{(?=else if)/,
	        end: /\}\}/,
	        keywords: 'else if'
	      },
	      {
	        // closing block statement
	        className: 'template-tag',
	        begin: /\{\{\//,
	        end: /\}\}/,
	        contains: [CLOSING_BLOCK_MUSTACHE_CONTENTS]
	      },
	      {
	        // template variable or helper-call that is NOT html-escaped
	        className: 'template-variable',
	        begin: /\{\{\{/,
	        end: /\}\}\}/,
	        contains: [BASIC_MUSTACHE_CONTENTS]
	      },
	      {
	        // template variable or helper-call that is html-escaped
	        className: 'template-variable',
	        begin: /\{\{/,
	        end: /\}\}/,
	        contains: [BASIC_MUSTACHE_CONTENTS]
	      }
	    ]
	  };
	}

	handlebars_1 = handlebars;
	return handlebars_1;
}

/*
Language: Haskell
Author: Jeremy Hull <sourdrums@gmail.com>
Contributors: Zena Treep <zena.treep@gmail.com>
Website: https://www.haskell.org
Category: functional
*/

var haskell_1;
var hasRequiredHaskell;

function requireHaskell () {
	if (hasRequiredHaskell) return haskell_1;
	hasRequiredHaskell = 1;
	function haskell(hljs) {
	  const COMMENT = {
	    variants: [
	      hljs.COMMENT('--', '$'),
	      hljs.COMMENT(
	        /\{-/,
	        /-\}/,
	        {
	          contains: ['self']
	        }
	      )
	    ]
	  };

	  const PRAGMA = {
	    className: 'meta',
	    begin: /\{-#/,
	    end: /#-\}/
	  };

	  const PREPROCESSOR = {
	    className: 'meta',
	    begin: '^#',
	    end: '$'
	  };

	  const CONSTRUCTOR = {
	    className: 'type',
	    begin: '\\b[A-Z][\\w\']*', // TODO: other constructors (build-in, infix).
	    relevance: 0
	  };

	  const LIST = {
	    begin: '\\(',
	    end: '\\)',
	    illegal: '"',
	    contains: [
	      PRAGMA,
	      PREPROCESSOR,
	      {
	        className: 'type',
	        begin: '\\b[A-Z][\\w]*(\\((\\.\\.|,|\\w+)\\))?'
	      },
	      hljs.inherit(hljs.TITLE_MODE, {
	        begin: '[_a-z][\\w\']*'
	      }),
	      COMMENT
	    ]
	  };

	  const RECORD = {
	    begin: /\{/,
	    end: /\}/,
	    contains: LIST.contains
	  };

	  return {
	    name: 'Haskell',
	    aliases: ['hs'],
	    keywords:
	      'let in if then else case of where do module import hiding ' +
	      'qualified type data newtype deriving class instance as default ' +
	      'infix infixl infixr foreign export ccall stdcall cplusplus ' +
	      'jvm dotnet safe unsafe family forall mdo proc rec',
	    contains: [
	      // Top-level constructions.
	      {
	        beginKeywords: 'module',
	        end: 'where',
	        keywords: 'module where',
	        contains: [
	          LIST,
	          COMMENT
	        ],
	        illegal: '\\W\\.|;'
	      },
	      {
	        begin: '\\bimport\\b',
	        end: '$',
	        keywords: 'import qualified as hiding',
	        contains: [
	          LIST,
	          COMMENT
	        ],
	        illegal: '\\W\\.|;'
	      },
	      {
	        className: 'class',
	        begin: '^(\\s*)?(class|instance)\\b',
	        end: 'where',
	        keywords: 'class family instance where',
	        contains: [
	          CONSTRUCTOR,
	          LIST,
	          COMMENT
	        ]
	      },
	      {
	        className: 'class',
	        begin: '\\b(data|(new)?type)\\b',
	        end: '$',
	        keywords: 'data family type newtype deriving',
	        contains: [
	          PRAGMA,
	          CONSTRUCTOR,
	          LIST,
	          RECORD,
	          COMMENT
	        ]
	      },
	      {
	        beginKeywords: 'default',
	        end: '$',
	        contains: [
	          CONSTRUCTOR,
	          LIST,
	          COMMENT
	        ]
	      },
	      {
	        beginKeywords: 'infix infixl infixr',
	        end: '$',
	        contains: [
	          hljs.C_NUMBER_MODE,
	          COMMENT
	        ]
	      },
	      {
	        begin: '\\bforeign\\b',
	        end: '$',
	        keywords: 'foreign import export ccall stdcall cplusplus jvm ' +
	                  'dotnet safe unsafe',
	        contains: [
	          CONSTRUCTOR,
	          hljs.QUOTE_STRING_MODE,
	          COMMENT
	        ]
	      },
	      {
	        className: 'meta',
	        begin: '#!\\/usr\\/bin\\/env\ runhaskell',
	        end: '$'
	      },
	      // "Whitespaces".
	      PRAGMA,
	      PREPROCESSOR,

	      // Literals and names.

	      // TODO: characters.
	      hljs.QUOTE_STRING_MODE,
	      hljs.C_NUMBER_MODE,
	      CONSTRUCTOR,
	      hljs.inherit(hljs.TITLE_MODE, {
	        begin: '^[_a-z][\\w\']*'
	      }),
	      COMMENT,
	      { // No markup, relevance booster
	        begin: '->|<-'
	      }
	    ]
	  };
	}

	haskell_1 = haskell;
	return haskell_1;
}

/*
Language: Haxe
Description: Haxe is an open source toolkit based on a modern, high level, strictly typed programming language.
Author: Christopher Kaster <ikasoki@gmail.com> (Based on the actionscript.js language file by Alexander Myadzel)
Contributors: Kenton Hamaluik <kentonh@gmail.com>
Website: https://haxe.org
*/

var haxe_1;
var hasRequiredHaxe;

function requireHaxe () {
	if (hasRequiredHaxe) return haxe_1;
	hasRequiredHaxe = 1;
	function haxe(hljs) {

	  const HAXE_BASIC_TYPES = 'Int Float String Bool Dynamic Void Array ';

	  return {
	    name: 'Haxe',
	    aliases: ['hx'],
	    keywords: {
	      keyword: 'break case cast catch continue default do dynamic else enum extern ' +
	               'for function here if import in inline never new override package private get set ' +
	               'public return static super switch this throw trace try typedef untyped using var while ' +
	               HAXE_BASIC_TYPES,
	      built_in:
	        'trace this',
	      literal:
	        'true false null _'
	    },
	    contains: [
	      {
	        className: 'string', // interpolate-able strings
	        begin: '\'',
	        end: '\'',
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          {
	            className: 'subst', // interpolation
	            begin: '\\$\\{',
	            end: '\\}'
	          },
	          {
	            className: 'subst', // interpolation
	            begin: '\\$',
	            end: /\W\}/
	          }
	        ]
	      },
	      hljs.QUOTE_STRING_MODE,
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.C_NUMBER_MODE,
	      {
	        className: 'meta', // compiler meta
	        begin: '@:',
	        end: '$'
	      },
	      {
	        className: 'meta', // compiler conditionals
	        begin: '#',
	        end: '$',
	        keywords: {
	          'meta-keyword': 'if else elseif end error'
	        }
	      },
	      {
	        className: 'type', // function types
	        begin: ':[ \t]*',
	        end: '[^A-Za-z0-9_ \t\\->]',
	        excludeBegin: true,
	        excludeEnd: true,
	        relevance: 0
	      },
	      {
	        className: 'type', // types
	        begin: ':[ \t]*',
	        end: '\\W',
	        excludeBegin: true,
	        excludeEnd: true
	      },
	      {
	        className: 'type', // instantiation
	        begin: 'new *',
	        end: '\\W',
	        excludeBegin: true,
	        excludeEnd: true
	      },
	      {
	        className: 'class', // enums
	        beginKeywords: 'enum',
	        end: '\\{',
	        contains: [hljs.TITLE_MODE]
	      },
	      {
	        className: 'class', // abstracts
	        beginKeywords: 'abstract',
	        end: '[\\{$]',
	        contains: [
	          {
	            className: 'type',
	            begin: '\\(',
	            end: '\\)',
	            excludeBegin: true,
	            excludeEnd: true
	          },
	          {
	            className: 'type',
	            begin: 'from +',
	            end: '\\W',
	            excludeBegin: true,
	            excludeEnd: true
	          },
	          {
	            className: 'type',
	            begin: 'to +',
	            end: '\\W',
	            excludeBegin: true,
	            excludeEnd: true
	          },
	          hljs.TITLE_MODE
	        ],
	        keywords: {
	          keyword: 'abstract from to'
	        }
	      },
	      {
	        className: 'class', // classes
	        begin: '\\b(class|interface) +',
	        end: '[\\{$]',
	        excludeEnd: true,
	        keywords: 'class interface',
	        contains: [
	          {
	            className: 'keyword',
	            begin: '\\b(extends|implements) +',
	            keywords: 'extends implements',
	            contains: [
	              {
	                className: 'type',
	                begin: hljs.IDENT_RE,
	                relevance: 0
	              }
	            ]
	          },
	          hljs.TITLE_MODE
	        ]
	      },
	      {
	        className: 'function',
	        beginKeywords: 'function',
	        end: '\\(',
	        excludeEnd: true,
	        illegal: '\\S',
	        contains: [hljs.TITLE_MODE]
	      }
	    ],
	    illegal: /<\//
	  };
	}

	haxe_1 = haxe;
	return haxe_1;
}

/*
Language: HSP
Author: prince <MC.prince.0203@gmail.com>
Website: https://en.wikipedia.org/wiki/Hot_Soup_Processor
Category: scripting
*/

var hsp_1;
var hasRequiredHsp;

function requireHsp () {
	if (hasRequiredHsp) return hsp_1;
	hasRequiredHsp = 1;
	function hsp(hljs) {
	  return {
	    name: 'HSP',
	    case_insensitive: true,
	    keywords: {
	      $pattern: /[\w._]+/,
	      keyword: 'goto gosub return break repeat loop continue wait await dim sdim foreach dimtype dup dupptr end stop newmod delmod mref run exgoto on mcall assert logmes newlab resume yield onexit onerror onkey onclick oncmd exist delete mkdir chdir dirlist bload bsave bcopy memfile if else poke wpoke lpoke getstr chdpm memexpand memcpy memset notesel noteadd notedel noteload notesave randomize noteunsel noteget split strrep setease button chgdisp exec dialog mmload mmplay mmstop mci pset pget syscolor mes print title pos circle cls font sysfont objsize picload color palcolor palette redraw width gsel gcopy gzoom gmode bmpsave hsvcolor getkey listbox chkbox combox input mesbox buffer screen bgscr mouse objsel groll line clrobj boxf objprm objmode stick grect grotate gsquare gradf objimage objskip objenable celload celdiv celput newcom querycom delcom cnvstow comres axobj winobj sendmsg comevent comevarg sarrayconv callfunc cnvwtos comevdisp libptr system hspstat hspver stat cnt err strsize looplev sublev iparam wparam lparam refstr refdval int rnd strlen length length2 length3 length4 vartype gettime peek wpeek lpeek varptr varuse noteinfo instr abs limit getease str strmid strf getpath strtrim sin cos tan atan sqrt double absf expf logf limitf powf geteasef mousex mousey mousew hwnd hinstance hdc ginfo objinfo dirinfo sysinfo thismod __hspver__ __hsp30__ __date__ __time__ __line__ __file__ _debug __hspdef__ and or xor not screen_normal screen_palette screen_hide screen_fixedsize screen_tool screen_frame gmode_gdi gmode_mem gmode_rgb0 gmode_alpha gmode_rgb0alpha gmode_add gmode_sub gmode_pixela ginfo_mx ginfo_my ginfo_act ginfo_sel ginfo_wx1 ginfo_wy1 ginfo_wx2 ginfo_wy2 ginfo_vx ginfo_vy ginfo_sizex ginfo_sizey ginfo_winx ginfo_winy ginfo_mesx ginfo_mesy ginfo_r ginfo_g ginfo_b ginfo_paluse ginfo_dispx ginfo_dispy ginfo_cx ginfo_cy ginfo_intid ginfo_newid ginfo_sx ginfo_sy objinfo_mode objinfo_bmscr objinfo_hwnd notemax notesize dir_cur dir_exe dir_win dir_sys dir_cmdline dir_desktop dir_mydoc dir_tv font_normal font_bold font_italic font_underline font_strikeout font_antialias objmode_normal objmode_guifont objmode_usefont gsquare_grad msgothic msmincho do until while wend for next _break _continue switch case default swbreak swend ddim ldim alloc m_pi rad2deg deg2rad ease_linear ease_quad_in ease_quad_out ease_quad_inout ease_cubic_in ease_cubic_out ease_cubic_inout ease_quartic_in ease_quartic_out ease_quartic_inout ease_bounce_in ease_bounce_out ease_bounce_inout ease_shake_in ease_shake_out ease_shake_inout ease_loop'
	    },
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.APOS_STRING_MODE,

	      {
	        // multi-line string
	        className: 'string',
	        begin: /\{"/,
	        end: /"\}/,
	        contains: [hljs.BACKSLASH_ESCAPE]
	      },

	      hljs.COMMENT(';', '$', {
	        relevance: 0
	      }),

	      {
	        // pre-processor
	        className: 'meta',
	        begin: '#',
	        end: '$',
	        keywords: {
	          'meta-keyword': 'addion cfunc cmd cmpopt comfunc const defcfunc deffunc define else endif enum epack func global if ifdef ifndef include modcfunc modfunc modinit modterm module pack packopt regcmd runtime undef usecom uselib'
	        },
	        contains: [
	          hljs.inherit(hljs.QUOTE_STRING_MODE, {
	            className: 'meta-string'
	          }),
	          hljs.NUMBER_MODE,
	          hljs.C_NUMBER_MODE,
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE
	        ]
	      },

	      {
	        // label
	        className: 'symbol',
	        begin: '^\\*(\\w+|@)'
	      },

	      hljs.NUMBER_MODE,
	      hljs.C_NUMBER_MODE
	    ]
	  };
	}

	hsp_1 = hsp;
	return hsp_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var htmlbars_1;
var hasRequiredHtmlbars;

function requireHtmlbars () {
	if (hasRequiredHtmlbars) return htmlbars_1;
	hasRequiredHtmlbars = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function anyNumberOfTimes(re) {
	  return concat('(', re, ')*');
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function optional(re) {
	  return concat('(', re, ')?');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/*
	Language: Handlebars
	Requires: xml.js
	Author: Robin Ward <robin.ward@gmail.com>
	Description: Matcher for Handlebars as well as EmberJS additions.
	Website: https://handlebarsjs.com
	Category: template
	*/

	function handlebars(hljs) {
	  const BUILT_INS = {
	    'builtin-name': [
	      'action',
	      'bindattr',
	      'collection',
	      'component',
	      'concat',
	      'debugger',
	      'each',
	      'each-in',
	      'get',
	      'hash',
	      'if',
	      'in',
	      'input',
	      'link-to',
	      'loc',
	      'log',
	      'lookup',
	      'mut',
	      'outlet',
	      'partial',
	      'query-params',
	      'render',
	      'template',
	      'textarea',
	      'unbound',
	      'unless',
	      'view',
	      'with',
	      'yield'
	    ]
	  };

	  const LITERALS = {
	    literal: [
	      'true',
	      'false',
	      'undefined',
	      'null'
	    ]
	  };

	  // as defined in https://handlebarsjs.com/guide/expressions.html#literal-segments
	  // this regex matches literal segments like ' abc ' or [ abc ] as well as helpers and paths
	  // like a/b, ./abc/cde, and abc.bcd

	  const DOUBLE_QUOTED_ID_REGEX = /""|"[^"]+"/;
	  const SINGLE_QUOTED_ID_REGEX = /''|'[^']+'/;
	  const BRACKET_QUOTED_ID_REGEX = /\[\]|\[[^\]]+\]/;
	  const PLAIN_ID_REGEX = /[^\s!"#%&'()*+,.\/;<=>@\[\\\]^`{|}~]+/;
	  const PATH_DELIMITER_REGEX = /(\.|\/)/;
	  const ANY_ID = either(
	    DOUBLE_QUOTED_ID_REGEX,
	    SINGLE_QUOTED_ID_REGEX,
	    BRACKET_QUOTED_ID_REGEX,
	    PLAIN_ID_REGEX
	    );

	  const IDENTIFIER_REGEX = concat(
	    optional(/\.|\.\/|\//), // relative or absolute path
	    ANY_ID,
	    anyNumberOfTimes(concat(
	      PATH_DELIMITER_REGEX,
	      ANY_ID
	    ))
	  );

	  // identifier followed by a equal-sign (without the equal sign)
	  const HASH_PARAM_REGEX = concat(
	    '(',
	    BRACKET_QUOTED_ID_REGEX, '|',
	    PLAIN_ID_REGEX,
	    ')(?==)'
	  );

	  const HELPER_NAME_OR_PATH_EXPRESSION = {
	    begin: IDENTIFIER_REGEX,
	    lexemes: /[\w.\/]+/
	  };

	  const HELPER_PARAMETER = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    keywords: LITERALS
	  });

	  const SUB_EXPRESSION = {
	    begin: /\(/,
	    end: /\)/
	    // the "contains" is added below when all necessary sub-modes are defined
	  };

	  const HASH = {
	    // fka "attribute-assignment", parameters of the form 'key=value'
	    className: 'attr',
	    begin: HASH_PARAM_REGEX,
	    relevance: 0,
	    starts: {
	      begin: /=/,
	      end: /=/,
	      starts: {
	        contains: [
	          hljs.NUMBER_MODE,
	          hljs.QUOTE_STRING_MODE,
	          hljs.APOS_STRING_MODE,
	          HELPER_PARAMETER,
	          SUB_EXPRESSION
	        ]
	      }
	    }
	  };

	  const BLOCK_PARAMS = {
	    // parameters of the form '{{#with x as | y |}}...{{/with}}'
	    begin: /as\s+\|/,
	    keywords: {
	      keyword: 'as'
	    },
	    end: /\|/,
	    contains: [
	      {
	        // define sub-mode in order to prevent highlighting of block-parameter named "as"
	        begin: /\w+/
	      }
	    ]
	  };

	  const HELPER_PARAMETERS = {
	    contains: [
	      hljs.NUMBER_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.APOS_STRING_MODE,
	      BLOCK_PARAMS,
	      HASH,
	      HELPER_PARAMETER,
	      SUB_EXPRESSION
	    ],
	    returnEnd: true
	    // the property "end" is defined through inheritance when the mode is used. If depends
	    // on the surrounding mode, but "endsWithParent" does not work here (i.e. it includes the
	    // end-token of the surrounding mode)
	  };

	  const SUB_EXPRESSION_CONTENTS = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    className: 'name',
	    keywords: BUILT_INS,
	    starts: hljs.inherit(HELPER_PARAMETERS, {
	      end: /\)/
	    })
	  });

	  SUB_EXPRESSION.contains = [SUB_EXPRESSION_CONTENTS];

	  const OPENING_BLOCK_MUSTACHE_CONTENTS = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    keywords: BUILT_INS,
	    className: 'name',
	    starts: hljs.inherit(HELPER_PARAMETERS, {
	      end: /\}\}/
	    })
	  });

	  const CLOSING_BLOCK_MUSTACHE_CONTENTS = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    keywords: BUILT_INS,
	    className: 'name'
	  });

	  const BASIC_MUSTACHE_CONTENTS = hljs.inherit(HELPER_NAME_OR_PATH_EXPRESSION, {
	    className: 'name',
	    keywords: BUILT_INS,
	    starts: hljs.inherit(HELPER_PARAMETERS, {
	      end: /\}\}/
	    })
	  });

	  const ESCAPE_MUSTACHE_WITH_PRECEEDING_BACKSLASH = {
	    begin: /\\\{\{/,
	    skip: true
	  };
	  const PREVENT_ESCAPE_WITH_ANOTHER_PRECEEDING_BACKSLASH = {
	    begin: /\\\\(?=\{\{)/,
	    skip: true
	  };

	  return {
	    name: 'Handlebars',
	    aliases: [
	      'hbs',
	      'html.hbs',
	      'html.handlebars',
	      'htmlbars'
	    ],
	    case_insensitive: true,
	    subLanguage: 'xml',
	    contains: [
	      ESCAPE_MUSTACHE_WITH_PRECEEDING_BACKSLASH,
	      PREVENT_ESCAPE_WITH_ANOTHER_PRECEEDING_BACKSLASH,
	      hljs.COMMENT(/\{\{!--/, /--\}\}/),
	      hljs.COMMENT(/\{\{!/, /\}\}/),
	      {
	        // open raw block "{{{{raw}}}} content not evaluated {{{{/raw}}}}"
	        className: 'template-tag',
	        begin: /\{\{\{\{(?!\/)/,
	        end: /\}\}\}\}/,
	        contains: [OPENING_BLOCK_MUSTACHE_CONTENTS],
	        starts: {
	          end: /\{\{\{\{\//,
	          returnEnd: true,
	          subLanguage: 'xml'
	        }
	      },
	      {
	        // close raw block
	        className: 'template-tag',
	        begin: /\{\{\{\{\//,
	        end: /\}\}\}\}/,
	        contains: [CLOSING_BLOCK_MUSTACHE_CONTENTS]
	      },
	      {
	        // open block statement
	        className: 'template-tag',
	        begin: /\{\{#/,
	        end: /\}\}/,
	        contains: [OPENING_BLOCK_MUSTACHE_CONTENTS]
	      },
	      {
	        className: 'template-tag',
	        begin: /\{\{(?=else\}\})/,
	        end: /\}\}/,
	        keywords: 'else'
	      },
	      {
	        className: 'template-tag',
	        begin: /\{\{(?=else if)/,
	        end: /\}\}/,
	        keywords: 'else if'
	      },
	      {
	        // closing block statement
	        className: 'template-tag',
	        begin: /\{\{\//,
	        end: /\}\}/,
	        contains: [CLOSING_BLOCK_MUSTACHE_CONTENTS]
	      },
	      {
	        // template variable or helper-call that is NOT html-escaped
	        className: 'template-variable',
	        begin: /\{\{\{/,
	        end: /\}\}\}/,
	        contains: [BASIC_MUSTACHE_CONTENTS]
	      },
	      {
	        // template variable or helper-call that is html-escaped
	        className: 'template-variable',
	        begin: /\{\{/,
	        end: /\}\}/,
	        contains: [BASIC_MUSTACHE_CONTENTS]
	      }
	    ]
	  };
	}

	/*
	 Language: HTMLBars (legacy)
	 Requires: xml.js
	 Description: Matcher for Handlebars as well as EmberJS additions.
	 Website: https://github.com/tildeio/htmlbars
	 Category: template
	 */

	function htmlbars(hljs) {
	  const definition = handlebars(hljs);

	  definition.name = "HTMLbars";

	  // HACK: This lets handlebars do the auto-detection if it's been loaded (by
	  // default the build script will load in alphabetical order) and if not (perhaps
	  // an install is only using `htmlbars`, not `handlebars`) then this will still
	  // allow HTMLBars to participate in the auto-detection

	  // worse case someone will have HTMLbars and handlebars competing for the same
	  // content and will need to change their setup to only require handlebars, but
	  // I don't consider this a breaking change
	  if (hljs.getLanguage("handlebars")) {
	    definition.disableAutodetect = true;
	  }

	  return definition;
	}

	htmlbars_1 = htmlbars;
	return htmlbars_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var http_1;
var hasRequiredHttp;

function requireHttp () {
	if (hasRequiredHttp) return http_1;
	hasRequiredHttp = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: HTTP
	Description: HTTP request and response headers with automatic body highlighting
	Author: Ivan Sagalaev <maniac@softwaremaniacs.org>
	Category: common, protocols
	Website: https://developer.mozilla.org/en-US/docs/Web/HTTP/Overview
	*/

	function http(hljs) {
	  const VERSION = 'HTTP/(2|1\\.[01])';
	  const HEADER_NAME = /[A-Za-z][A-Za-z0-9-]*/;
	  const HEADER = {
	    className: 'attribute',
	    begin: concat('^', HEADER_NAME, '(?=\\:\\s)'),
	    starts: {
	      contains: [
	        {
	          className: "punctuation",
	          begin: /: /,
	          relevance: 0,
	          starts: {
	            end: '$',
	            relevance: 0
	          }
	        }
	      ]
	    }
	  };
	  const HEADERS_AND_BODY = [
	    HEADER,
	    {
	      begin: '\\n\\n',
	      starts: { subLanguage: [], endsWithParent: true }
	    }
	  ];

	  return {
	    name: 'HTTP',
	    aliases: ['https'],
	    illegal: /\S/,
	    contains: [
	      // response
	      {
	        begin: '^(?=' + VERSION + " \\d{3})",
	        end: /$/,
	        contains: [
	          {
	            className: "meta",
	            begin: VERSION
	          },
	          {
	            className: 'number', begin: '\\b\\d{3}\\b'
	          }
	        ],
	        starts: {
	          end: /\b\B/,
	          illegal: /\S/,
	          contains: HEADERS_AND_BODY
	        }
	      },
	      // request
	      {
	        begin: '(?=^[A-Z]+ (.*?) ' + VERSION + '$)',
	        end: /$/,
	        contains: [
	          {
	            className: 'string',
	            begin: ' ',
	            end: ' ',
	            excludeBegin: true,
	            excludeEnd: true
	          },
	          {
	            className: "meta",
	            begin: VERSION
	          },
	          {
	            className: 'keyword',
	            begin: '[A-Z]+'
	          }
	        ],
	        starts: {
	          end: /\b\B/,
	          illegal: /\S/,
	          contains: HEADERS_AND_BODY
	        }
	      },
	      // to allow headers to work even without a preamble
	      hljs.inherit(HEADER, {
	        relevance: 0
	      })
	    ]
	  };
	}

	http_1 = http;
	return http_1;
}

/*
Language: Hy
Description: Hy is a wonderful dialect of Lisp that’s embedded in Python.
Author: Sergey Sobko <s.sobko@profitware.ru>
Website: http://docs.hylang.org/en/stable/
Category: lisp
*/

var hy_1;
var hasRequiredHy;

function requireHy () {
	if (hasRequiredHy) return hy_1;
	hasRequiredHy = 1;
	function hy(hljs) {
	  var SYMBOLSTART = 'a-zA-Z_\\-!.?+*=<>&#\'';
	  var SYMBOL_RE = '[' + SYMBOLSTART + '][' + SYMBOLSTART + '0-9/;:]*';
	  var keywords = {
	    $pattern: SYMBOL_RE,
	    'builtin-name':
	      // keywords
	      '!= % %= & &= * ** **= *= *map ' +
	      '+ += , --build-class-- --import-- -= . / // //= ' +
	      '/= < << <<= <= = > >= >> >>= ' +
	      '@ @= ^ ^= abs accumulate all and any ap-compose ' +
	      'ap-dotimes ap-each ap-each-while ap-filter ap-first ap-if ap-last ap-map ap-map-when ap-pipe ' +
	      'ap-reduce ap-reject apply as-> ascii assert assoc bin break butlast ' +
	      'callable calling-module-name car case cdr chain chr coll? combinations compile ' +
	      'compress cond cons cons? continue count curry cut cycle dec ' +
	      'def default-method defclass defmacro defmacro-alias defmacro/g! defmain defmethod defmulti defn ' +
	      'defn-alias defnc defnr defreader defseq del delattr delete-route dict-comp dir ' +
	      'disassemble dispatch-reader-macro distinct divmod do doto drop drop-last drop-while empty? ' +
	      'end-sequence eval eval-and-compile eval-when-compile even? every? except exec filter first ' +
	      'flatten float? fn fnc fnr for for* format fraction genexpr ' +
	      'gensym get getattr global globals group-by hasattr hash hex id ' +
	      'identity if if* if-not if-python2 import in inc input instance? ' +
	      'integer integer-char? integer? interleave interpose is is-coll is-cons is-empty is-even ' +
	      'is-every is-float is-instance is-integer is-integer-char is-iterable is-iterator is-keyword is-neg is-none ' +
	      'is-not is-numeric is-odd is-pos is-string is-symbol is-zero isinstance islice issubclass ' +
	      'iter iterable? iterate iterator? keyword keyword? lambda last len let ' +
	      'lif lif-not list* list-comp locals loop macro-error macroexpand macroexpand-1 macroexpand-all ' +
	      'map max merge-with method-decorator min multi-decorator multicombinations name neg? next ' +
	      'none? nonlocal not not-in not? nth numeric? oct odd? open ' +
	      'or ord partition permutations pos? post-route postwalk pow prewalk print ' +
	      'product profile/calls profile/cpu put-route quasiquote quote raise range read read-str ' +
	      'recursive-replace reduce remove repeat repeatedly repr require rest round route ' +
	      'route-with-methods rwm second seq set-comp setattr setv some sorted string ' +
	      'string? sum switch symbol? take take-nth take-while tee try unless ' +
	      'unquote unquote-splicing vars walk when while with with* with-decorator with-gensyms ' +
	      'xi xor yield yield-from zero? zip zip-longest | |= ~'
	   };

	  var SIMPLE_NUMBER_RE = '[-+]?\\d+(\\.\\d+)?';

	  var SYMBOL = {
	    begin: SYMBOL_RE,
	    relevance: 0
	  };
	  var NUMBER = {
	    className: 'number', begin: SIMPLE_NUMBER_RE,
	    relevance: 0
	  };
	  var STRING = hljs.inherit(hljs.QUOTE_STRING_MODE, {illegal: null});
	  var COMMENT = hljs.COMMENT(
	    ';',
	    '$',
	    {
	      relevance: 0
	    }
	  );
	  var LITERAL = {
	    className: 'literal',
	    begin: /\b([Tt]rue|[Ff]alse|nil|None)\b/
	  };
	  var COLLECTION = {
	    begin: '[\\[\\{]', end: '[\\]\\}]'
	  };
	  var HINT = {
	    className: 'comment',
	    begin: '\\^' + SYMBOL_RE
	  };
	  var HINT_COL = hljs.COMMENT('\\^\\{', '\\}');
	  var KEY = {
	    className: 'symbol',
	    begin: '[:]{1,2}' + SYMBOL_RE
	  };
	  var LIST = {
	    begin: '\\(', end: '\\)'
	  };
	  var BODY = {
	    endsWithParent: true,
	    relevance: 0
	  };
	  var NAME = {
	    className: 'name',
	    relevance: 0,
	    keywords: keywords,
	    begin: SYMBOL_RE,
	    starts: BODY
	  };
	  var DEFAULT_CONTAINS = [LIST, STRING, HINT, HINT_COL, COMMENT, KEY, COLLECTION, NUMBER, LITERAL, SYMBOL];

	  LIST.contains = [hljs.COMMENT('comment', ''), NAME, BODY];
	  BODY.contains = DEFAULT_CONTAINS;
	  COLLECTION.contains = DEFAULT_CONTAINS;

	  return {
	    name: 'Hy',
	    aliases: ['hylang'],
	    illegal: /\S/,
	    contains: [hljs.SHEBANG(), LIST, STRING, HINT, HINT_COL, COMMENT, KEY, COLLECTION, NUMBER, LITERAL]
	  };
	}

	hy_1 = hy;
	return hy_1;
}

/*
Language: Inform 7
Author: Bruno Dias <bruno.r.dias@gmail.com>
Description: Language definition for Inform 7, a DSL for writing parser interactive fiction.
Website: http://inform7.com
*/

var inform7_1;
var hasRequiredInform7;

function requireInform7 () {
	if (hasRequiredInform7) return inform7_1;
	hasRequiredInform7 = 1;
	function inform7(hljs) {
	  const START_BRACKET = '\\[';
	  const END_BRACKET = '\\]';
	  return {
	    name: 'Inform 7',
	    aliases: ['i7'],
	    case_insensitive: true,
	    keywords: {
	      // Some keywords more or less unique to I7, for relevance.
	      keyword:
	        // kind:
	        'thing room person man woman animal container ' +
	        'supporter backdrop door ' +
	        // characteristic:
	        'scenery open closed locked inside gender ' +
	        // verb:
	        'is are say understand ' +
	        // misc keyword:
	        'kind of rule'
	    },
	    contains: [
	      {
	        className: 'string',
	        begin: '"',
	        end: '"',
	        relevance: 0,
	        contains: [
	          {
	            className: 'subst',
	            begin: START_BRACKET,
	            end: END_BRACKET
	          }
	        ]
	      },
	      {
	        className: 'section',
	        begin: /^(Volume|Book|Part|Chapter|Section|Table)\b/,
	        end: '$'
	      },
	      {
	        // Rule definition
	        // This is here for relevance.
	        begin: /^(Check|Carry out|Report|Instead of|To|Rule|When|Before|After)\b/,
	        end: ':',
	        contains: [
	          {
	            // Rule name
	            begin: '\\(This',
	            end: '\\)'
	          }
	        ]
	      },
	      {
	        className: 'comment',
	        begin: START_BRACKET,
	        end: END_BRACKET,
	        contains: ['self']
	      }
	    ]
	  };
	}

	inform7_1 = inform7;
	return inform7_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var ini_1;
var hasRequiredIni;

function requireIni () {
	if (hasRequiredIni) return ini_1;
	hasRequiredIni = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/*
	Language: TOML, also INI
	Description: TOML aims to be a minimal configuration file format that's easy to read due to obvious semantics.
	Contributors: Guillaume Gomez <guillaume1.gomez@gmail.com>
	Category: common, config
	Website: https://github.com/toml-lang/toml
	*/

	function ini(hljs) {
	  const NUMBERS = {
	    className: 'number',
	    relevance: 0,
	    variants: [
	      {
	        begin: /([+-]+)?[\d]+_[\d_]+/
	      },
	      {
	        begin: hljs.NUMBER_RE
	      }
	    ]
	  };
	  const COMMENTS = hljs.COMMENT();
	  COMMENTS.variants = [
	    {
	      begin: /;/,
	      end: /$/
	    },
	    {
	      begin: /#/,
	      end: /$/
	    }
	  ];
	  const VARIABLES = {
	    className: 'variable',
	    variants: [
	      {
	        begin: /\$[\w\d"][\w\d_]*/
	      },
	      {
	        begin: /\$\{(.*?)\}/
	      }
	    ]
	  };
	  const LITERALS = {
	    className: 'literal',
	    begin: /\bon|off|true|false|yes|no\b/
	  };
	  const STRINGS = {
	    className: "string",
	    contains: [hljs.BACKSLASH_ESCAPE],
	    variants: [
	      {
	        begin: "'''",
	        end: "'''",
	        relevance: 10
	      },
	      {
	        begin: '"""',
	        end: '"""',
	        relevance: 10
	      },
	      {
	        begin: '"',
	        end: '"'
	      },
	      {
	        begin: "'",
	        end: "'"
	      }
	    ]
	  };
	  const ARRAY = {
	    begin: /\[/,
	    end: /\]/,
	    contains: [
	      COMMENTS,
	      LITERALS,
	      VARIABLES,
	      STRINGS,
	      NUMBERS,
	      'self'
	    ],
	    relevance: 0
	  };

	  const BARE_KEY = /[A-Za-z0-9_-]+/;
	  const QUOTED_KEY_DOUBLE_QUOTE = /"(\\"|[^"])*"/;
	  const QUOTED_KEY_SINGLE_QUOTE = /'[^']*'/;
	  const ANY_KEY = either(
	    BARE_KEY, QUOTED_KEY_DOUBLE_QUOTE, QUOTED_KEY_SINGLE_QUOTE
	  );
	  const DOTTED_KEY = concat(
	    ANY_KEY, '(\\s*\\.\\s*', ANY_KEY, ')*',
	    lookahead(/\s*=\s*[^#\s]/)
	  );

	  return {
	    name: 'TOML, also INI',
	    aliases: ['toml'],
	    case_insensitive: true,
	    illegal: /\S/,
	    contains: [
	      COMMENTS,
	      {
	        className: 'section',
	        begin: /\[+/,
	        end: /\]+/
	      },
	      {
	        begin: DOTTED_KEY,
	        className: 'attr',
	        starts: {
	          end: /$/,
	          contains: [
	            COMMENTS,
	            ARRAY,
	            LITERALS,
	            VARIABLES,
	            STRINGS,
	            NUMBERS
	          ]
	        }
	      }
	    ]
	  };
	}

	ini_1 = ini;
	return ini_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var irpf90_1;
var hasRequiredIrpf90;

function requireIrpf90 () {
	if (hasRequiredIrpf90) return irpf90_1;
	hasRequiredIrpf90 = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: IRPF90
	Author: Anthony Scemama <scemama@irsamc.ups-tlse.fr>
	Description: IRPF90 is an open-source Fortran code generator
	Website: http://irpf90.ups-tlse.fr
	Category: scientific
	*/

	/** @type LanguageFn */
	function irpf90(hljs) {
	  const PARAMS = {
	    className: 'params',
	    begin: '\\(',
	    end: '\\)'
	  };

	  // regex in both fortran and irpf90 should match
	  const OPTIONAL_NUMBER_SUFFIX = /(_[a-z_\d]+)?/;
	  const OPTIONAL_NUMBER_EXP = /([de][+-]?\d+)?/;
	  const NUMBER = {
	    className: 'number',
	    variants: [
	      {
	        begin: concat(/\b\d+/, /\.(\d*)/, OPTIONAL_NUMBER_EXP, OPTIONAL_NUMBER_SUFFIX)
	      },
	      {
	        begin: concat(/\b\d+/, OPTIONAL_NUMBER_EXP, OPTIONAL_NUMBER_SUFFIX)
	      },
	      {
	        begin: concat(/\.\d+/, OPTIONAL_NUMBER_EXP, OPTIONAL_NUMBER_SUFFIX)
	      }
	    ],
	    relevance: 0
	  };

	  const F_KEYWORDS = {
	    literal: '.False. .True.',
	    keyword: 'kind do while private call intrinsic where elsewhere ' +
	      'type endtype endmodule endselect endinterface end enddo endif if forall endforall only contains default return stop then ' +
	      'public subroutine|10 function program .and. .or. .not. .le. .eq. .ge. .gt. .lt. ' +
	      'goto save else use module select case ' +
	      'access blank direct exist file fmt form formatted iostat name named nextrec number opened rec recl sequential status unformatted unit ' +
	      'continue format pause cycle exit ' +
	      'c_null_char c_alert c_backspace c_form_feed flush wait decimal round iomsg ' +
	      'synchronous nopass non_overridable pass protected volatile abstract extends import ' +
	      'non_intrinsic value deferred generic final enumerator class associate bind enum ' +
	      'c_int c_short c_long c_long_long c_signed_char c_size_t c_int8_t c_int16_t c_int32_t c_int64_t c_int_least8_t c_int_least16_t ' +
	      'c_int_least32_t c_int_least64_t c_int_fast8_t c_int_fast16_t c_int_fast32_t c_int_fast64_t c_intmax_t C_intptr_t c_float c_double ' +
	      'c_long_double c_float_complex c_double_complex c_long_double_complex c_bool c_char c_null_ptr c_null_funptr ' +
	      'c_new_line c_carriage_return c_horizontal_tab c_vertical_tab iso_c_binding c_loc c_funloc c_associated  c_f_pointer ' +
	      'c_ptr c_funptr iso_fortran_env character_storage_size error_unit file_storage_size input_unit iostat_end iostat_eor ' +
	      'numeric_storage_size output_unit c_f_procpointer ieee_arithmetic ieee_support_underflow_control ' +
	      'ieee_get_underflow_mode ieee_set_underflow_mode newunit contiguous recursive ' +
	      'pad position action delim readwrite eor advance nml interface procedure namelist include sequence elemental pure ' +
	      'integer real character complex logical dimension allocatable|10 parameter ' +
	      'external implicit|10 none double precision assign intent optional pointer ' +
	      'target in out common equivalence data ' +
	      // IRPF90 special keywords
	      'begin_provider &begin_provider end_provider begin_shell end_shell begin_template end_template subst assert touch ' +
	      'soft_touch provide no_dep free irp_if irp_else irp_endif irp_write irp_read',
	    built_in: 'alog alog10 amax0 amax1 amin0 amin1 amod cabs ccos cexp clog csin csqrt dabs dacos dasin datan datan2 dcos dcosh ddim dexp dint ' +
	      'dlog dlog10 dmax1 dmin1 dmod dnint dsign dsin dsinh dsqrt dtan dtanh float iabs idim idint idnint ifix isign max0 max1 min0 min1 sngl ' +
	      'algama cdabs cdcos cdexp cdlog cdsin cdsqrt cqabs cqcos cqexp cqlog cqsin cqsqrt dcmplx dconjg derf derfc dfloat dgamma dimag dlgama ' +
	      'iqint qabs qacos qasin qatan qatan2 qcmplx qconjg qcos qcosh qdim qerf qerfc qexp qgamma qimag qlgama qlog qlog10 qmax1 qmin1 qmod ' +
	      'qnint qsign qsin qsinh qsqrt qtan qtanh abs acos aimag aint anint asin atan atan2 char cmplx conjg cos cosh exp ichar index int log ' +
	      'log10 max min nint sign sin sinh sqrt tan tanh print write dim lge lgt lle llt mod nullify allocate deallocate ' +
	      'adjustl adjustr all allocated any associated bit_size btest ceiling count cshift date_and_time digits dot_product ' +
	      'eoshift epsilon exponent floor fraction huge iand ibclr ibits ibset ieor ior ishft ishftc lbound len_trim matmul ' +
	      'maxexponent maxloc maxval merge minexponent minloc minval modulo mvbits nearest pack present product ' +
	      'radix random_number random_seed range repeat reshape rrspacing scale scan selected_int_kind selected_real_kind ' +
	      'set_exponent shape size spacing spread sum system_clock tiny transpose trim ubound unpack verify achar iachar transfer ' +
	      'dble entry dprod cpu_time command_argument_count get_command get_command_argument get_environment_variable is_iostat_end ' +
	      'ieee_arithmetic ieee_support_underflow_control ieee_get_underflow_mode ieee_set_underflow_mode ' +
	      'is_iostat_eor move_alloc new_line selected_char_kind same_type_as extends_type_of ' +
	      'acosh asinh atanh bessel_j0 bessel_j1 bessel_jn bessel_y0 bessel_y1 bessel_yn erf erfc erfc_scaled gamma log_gamma hypot norm2 ' +
	      'atomic_define atomic_ref execute_command_line leadz trailz storage_size merge_bits ' +
	      'bge bgt ble blt dshiftl dshiftr findloc iall iany iparity image_index lcobound ucobound maskl maskr ' +
	      'num_images parity popcnt poppar shifta shiftl shiftr this_image ' +
	      // IRPF90 special built_ins
	      'IRP_ALIGN irp_here'
	  };
	  return {
	    name: 'IRPF90',
	    case_insensitive: true,
	    keywords: F_KEYWORDS,
	    illegal: /\/\*/,
	    contains: [
	      hljs.inherit(hljs.APOS_STRING_MODE, {
	        className: 'string',
	        relevance: 0
	      }),
	      hljs.inherit(hljs.QUOTE_STRING_MODE, {
	        className: 'string',
	        relevance: 0
	      }),
	      {
	        className: 'function',
	        beginKeywords: 'subroutine function program',
	        illegal: '[${=\\n]',
	        contains: [
	          hljs.UNDERSCORE_TITLE_MODE,
	          PARAMS
	        ]
	      },
	      hljs.COMMENT('!', '$', {
	        relevance: 0
	      }),
	      hljs.COMMENT('begin_doc', 'end_doc', {
	        relevance: 10
	      }),
	      NUMBER
	    ]
	  };
	}

	irpf90_1 = irpf90;
	return irpf90_1;
}

/*
Language: ISBL
Author: Dmitriy Tarasov <dimatar@gmail.com>
Description: built-in language DIRECTUM
Category: enterprise
*/

var isbl_1;
var hasRequiredIsbl;

function requireIsbl () {
	if (hasRequiredIsbl) return isbl_1;
	hasRequiredIsbl = 1;
	function isbl(hljs) {
	  // Определение идентификаторов
	  const UNDERSCORE_IDENT_RE = "[A-Za-zА-Яа-яёЁ_!][A-Za-zА-Яа-яёЁ_0-9]*";

	  // Определение имен функций
	  const FUNCTION_NAME_IDENT_RE = "[A-Za-zА-Яа-яёЁ_][A-Za-zА-Яа-яёЁ_0-9]*";

	  // keyword : ключевые слова
	  const KEYWORD =
	    "and и else иначе endexcept endfinally endforeach конецвсе endif конецесли endwhile конецпока " +
	    "except exitfor finally foreach все if если in в not не or или try while пока ";

	  // SYSRES Constants
	  const sysres_constants =
	    "SYSRES_CONST_ACCES_RIGHT_TYPE_EDIT " +
	    "SYSRES_CONST_ACCES_RIGHT_TYPE_FULL " +
	    "SYSRES_CONST_ACCES_RIGHT_TYPE_VIEW " +
	    "SYSRES_CONST_ACCESS_MODE_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_NO_ACCESS_VIEW " +
	    "SYSRES_CONST_ACCESS_NO_ACCESS_VIEW_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_ADD_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_ADD_REQUISITE_YES_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_CHANGE_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_CHANGE_REQUISITE_YES_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_DELETE_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_DELETE_REQUISITE_YES_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_EXECUTE_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_EXECUTE_REQUISITE_YES_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_NO_ACCESS_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_NO_ACCESS_REQUISITE_YES_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_RATIFY_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_RATIFY_REQUISITE_YES_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_VIEW " +
	    "SYSRES_CONST_ACCESS_RIGHTS_VIEW_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_VIEW_REQUISITE_CODE " +
	    "SYSRES_CONST_ACCESS_RIGHTS_VIEW_REQUISITE_YES_CODE " +
	    "SYSRES_CONST_ACCESS_TYPE_CHANGE " +
	    "SYSRES_CONST_ACCESS_TYPE_CHANGE_CODE " +
	    "SYSRES_CONST_ACCESS_TYPE_EXISTS " +
	    "SYSRES_CONST_ACCESS_TYPE_EXISTS_CODE " +
	    "SYSRES_CONST_ACCESS_TYPE_FULL " +
	    "SYSRES_CONST_ACCESS_TYPE_FULL_CODE " +
	    "SYSRES_CONST_ACCESS_TYPE_VIEW " +
	    "SYSRES_CONST_ACCESS_TYPE_VIEW_CODE " +
	    "SYSRES_CONST_ACTION_TYPE_ABORT " +
	    "SYSRES_CONST_ACTION_TYPE_ACCEPT " +
	    "SYSRES_CONST_ACTION_TYPE_ACCESS_RIGHTS " +
	    "SYSRES_CONST_ACTION_TYPE_ADD_ATTACHMENT " +
	    "SYSRES_CONST_ACTION_TYPE_CHANGE_CARD " +
	    "SYSRES_CONST_ACTION_TYPE_CHANGE_KIND " +
	    "SYSRES_CONST_ACTION_TYPE_CHANGE_STORAGE " +
	    "SYSRES_CONST_ACTION_TYPE_CONTINUE " +
	    "SYSRES_CONST_ACTION_TYPE_COPY " +
	    "SYSRES_CONST_ACTION_TYPE_CREATE " +
	    "SYSRES_CONST_ACTION_TYPE_CREATE_VERSION " +
	    "SYSRES_CONST_ACTION_TYPE_DELETE " +
	    "SYSRES_CONST_ACTION_TYPE_DELETE_ATTACHMENT " +
	    "SYSRES_CONST_ACTION_TYPE_DELETE_VERSION " +
	    "SYSRES_CONST_ACTION_TYPE_DISABLE_DELEGATE_ACCESS_RIGHTS " +
	    "SYSRES_CONST_ACTION_TYPE_ENABLE_DELEGATE_ACCESS_RIGHTS " +
	    "SYSRES_CONST_ACTION_TYPE_ENCRYPTION_BY_CERTIFICATE " +
	    "SYSRES_CONST_ACTION_TYPE_ENCRYPTION_BY_CERTIFICATE_AND_PASSWORD " +
	    "SYSRES_CONST_ACTION_TYPE_ENCRYPTION_BY_PASSWORD " +
	    "SYSRES_CONST_ACTION_TYPE_EXPORT_WITH_LOCK " +
	    "SYSRES_CONST_ACTION_TYPE_EXPORT_WITHOUT_LOCK " +
	    "SYSRES_CONST_ACTION_TYPE_IMPORT_WITH_UNLOCK " +
	    "SYSRES_CONST_ACTION_TYPE_IMPORT_WITHOUT_UNLOCK " +
	    "SYSRES_CONST_ACTION_TYPE_LIFE_CYCLE_STAGE " +
	    "SYSRES_CONST_ACTION_TYPE_LOCK " +
	    "SYSRES_CONST_ACTION_TYPE_LOCK_FOR_SERVER " +
	    "SYSRES_CONST_ACTION_TYPE_LOCK_MODIFY " +
	    "SYSRES_CONST_ACTION_TYPE_MARK_AS_READED " +
	    "SYSRES_CONST_ACTION_TYPE_MARK_AS_UNREADED " +
	    "SYSRES_CONST_ACTION_TYPE_MODIFY " +
	    "SYSRES_CONST_ACTION_TYPE_MODIFY_CARD " +
	    "SYSRES_CONST_ACTION_TYPE_MOVE_TO_ARCHIVE " +
	    "SYSRES_CONST_ACTION_TYPE_OFF_ENCRYPTION " +
	    "SYSRES_CONST_ACTION_TYPE_PASSWORD_CHANGE " +
	    "SYSRES_CONST_ACTION_TYPE_PERFORM " +
	    "SYSRES_CONST_ACTION_TYPE_RECOVER_FROM_LOCAL_COPY " +
	    "SYSRES_CONST_ACTION_TYPE_RESTART " +
	    "SYSRES_CONST_ACTION_TYPE_RESTORE_FROM_ARCHIVE " +
	    "SYSRES_CONST_ACTION_TYPE_REVISION " +
	    "SYSRES_CONST_ACTION_TYPE_SEND_BY_MAIL " +
	    "SYSRES_CONST_ACTION_TYPE_SIGN " +
	    "SYSRES_CONST_ACTION_TYPE_START " +
	    "SYSRES_CONST_ACTION_TYPE_UNLOCK " +
	    "SYSRES_CONST_ACTION_TYPE_UNLOCK_FROM_SERVER " +
	    "SYSRES_CONST_ACTION_TYPE_VERSION_STATE " +
	    "SYSRES_CONST_ACTION_TYPE_VERSION_VISIBILITY " +
	    "SYSRES_CONST_ACTION_TYPE_VIEW " +
	    "SYSRES_CONST_ACTION_TYPE_VIEW_SHADOW_COPY " +
	    "SYSRES_CONST_ACTION_TYPE_WORKFLOW_DESCRIPTION_MODIFY " +
	    "SYSRES_CONST_ACTION_TYPE_WRITE_HISTORY " +
	    "SYSRES_CONST_ACTIVE_VERSION_STATE_PICK_VALUE " +
	    "SYSRES_CONST_ADD_REFERENCE_MODE_NAME " +
	    "SYSRES_CONST_ADDITION_REQUISITE_CODE " +
	    "SYSRES_CONST_ADDITIONAL_PARAMS_REQUISITE_CODE " +
	    "SYSRES_CONST_ADITIONAL_JOB_END_DATE_REQUISITE_NAME " +
	    "SYSRES_CONST_ADITIONAL_JOB_READ_REQUISITE_NAME " +
	    "SYSRES_CONST_ADITIONAL_JOB_START_DATE_REQUISITE_NAME " +
	    "SYSRES_CONST_ADITIONAL_JOB_STATE_REQUISITE_NAME " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_ADDING_USER_TO_GROUP_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_ADDING_USER_TO_GROUP_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_CREATION_COMP_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_CREATION_COMP_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_CREATION_GROUP_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_CREATION_GROUP_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_CREATION_USER_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_CREATION_USER_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DATABASE_USER_CREATION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DATABASE_USER_CREATION_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DATABASE_USER_DELETION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DATABASE_USER_DELETION_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DELETION_COMP_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DELETION_COMP_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DELETION_GROUP_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DELETION_GROUP_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DELETION_USER_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DELETION_USER_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DELETION_USER_FROM_GROUP_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_DELETION_USER_FROM_GROUP_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_GRANTING_FILTERER_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_GRANTING_FILTERER_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_GRANTING_FILTERER_RESTRICTION_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_GRANTING_FILTERER_RESTRICTION_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_GRANTING_PRIVILEGE_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_GRANTING_PRIVILEGE_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_GRANTING_RIGHTS_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_GRANTING_RIGHTS_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_IS_MAIN_SERVER_CHANGED_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_IS_MAIN_SERVER_CHANGED_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_IS_PUBLIC_CHANGED_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_IS_PUBLIC_CHANGED_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_REMOVING_FILTERER_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_REMOVING_FILTERER_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_REMOVING_FILTERER_RESTRICTION_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_REMOVING_FILTERER_RESTRICTION_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_REMOVING_PRIVILEGE_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_REMOVING_PRIVILEGE_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_REMOVING_RIGHTS_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_REMOVING_RIGHTS_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_SERVER_LOGIN_CREATION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_SERVER_LOGIN_CREATION_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_SERVER_LOGIN_DELETION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_SERVER_LOGIN_DELETION_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_CATEGORY_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_CATEGORY_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_COMP_TITLE_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_COMP_TITLE_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_FULL_NAME_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_FULL_NAME_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_GROUP_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_GROUP_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_PARENT_GROUP_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_PARENT_GROUP_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_USER_AUTH_TYPE_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_USER_AUTH_TYPE_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_USER_LOGIN_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_USER_LOGIN_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_USER_STATUS_ACTION " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_UPDATING_USER_STATUS_ACTION_CODE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_USER_PASSWORD_CHANGE " +
	    "SYSRES_CONST_ADMINISTRATION_HISTORY_USER_PASSWORD_CHANGE_ACTION " +
	    "SYSRES_CONST_ALL_ACCEPT_CONDITION_RUS " +
	    "SYSRES_CONST_ALL_USERS_GROUP " +
	    "SYSRES_CONST_ALL_USERS_GROUP_NAME " +
	    "SYSRES_CONST_ALL_USERS_SERVER_GROUP_NAME " +
	    "SYSRES_CONST_ALLOWED_ACCESS_TYPE_CODE " +
	    "SYSRES_CONST_ALLOWED_ACCESS_TYPE_NAME " +
	    "SYSRES_CONST_APP_VIEWER_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_APPROVING_SIGNATURE_NAME " +
	    "SYSRES_CONST_APPROVING_SIGNATURE_REQUISITE_CODE " +
	    "SYSRES_CONST_ASSISTANT_SUBSTITUE_TYPE " +
	    "SYSRES_CONST_ASSISTANT_SUBSTITUE_TYPE_CODE " +
	    "SYSRES_CONST_ATTACH_TYPE_COMPONENT_TOKEN " +
	    "SYSRES_CONST_ATTACH_TYPE_DOC " +
	    "SYSRES_CONST_ATTACH_TYPE_EDOC " +
	    "SYSRES_CONST_ATTACH_TYPE_FOLDER " +
	    "SYSRES_CONST_ATTACH_TYPE_JOB " +
	    "SYSRES_CONST_ATTACH_TYPE_REFERENCE " +
	    "SYSRES_CONST_ATTACH_TYPE_TASK " +
	    "SYSRES_CONST_AUTH_ENCODED_PASSWORD " +
	    "SYSRES_CONST_AUTH_ENCODED_PASSWORD_CODE " +
	    "SYSRES_CONST_AUTH_NOVELL " +
	    "SYSRES_CONST_AUTH_PASSWORD " +
	    "SYSRES_CONST_AUTH_PASSWORD_CODE " +
	    "SYSRES_CONST_AUTH_WINDOWS " +
	    "SYSRES_CONST_AUTHENTICATING_SIGNATURE_NAME " +
	    "SYSRES_CONST_AUTHENTICATING_SIGNATURE_REQUISITE_CODE " +
	    "SYSRES_CONST_AUTO_ENUM_METHOD_FLAG " +
	    "SYSRES_CONST_AUTO_NUMERATION_CODE " +
	    "SYSRES_CONST_AUTO_STRONG_ENUM_METHOD_FLAG " +
	    "SYSRES_CONST_AUTOTEXT_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_AUTOTEXT_TEXT_REQUISITE_CODE " +
	    "SYSRES_CONST_AUTOTEXT_USAGE_ALL " +
	    "SYSRES_CONST_AUTOTEXT_USAGE_ALL_CODE " +
	    "SYSRES_CONST_AUTOTEXT_USAGE_SIGN " +
	    "SYSRES_CONST_AUTOTEXT_USAGE_SIGN_CODE " +
	    "SYSRES_CONST_AUTOTEXT_USAGE_WORK " +
	    "SYSRES_CONST_AUTOTEXT_USAGE_WORK_CODE " +
	    "SYSRES_CONST_AUTOTEXT_USE_ANYWHERE_CODE " +
	    "SYSRES_CONST_AUTOTEXT_USE_ON_SIGNING_CODE " +
	    "SYSRES_CONST_AUTOTEXT_USE_ON_WORK_CODE " +
	    "SYSRES_CONST_BEGIN_DATE_REQUISITE_CODE " +
	    "SYSRES_CONST_BLACK_LIFE_CYCLE_STAGE_FONT_COLOR " +
	    "SYSRES_CONST_BLUE_LIFE_CYCLE_STAGE_FONT_COLOR " +
	    "SYSRES_CONST_BTN_PART " +
	    "SYSRES_CONST_CALCULATED_ROLE_TYPE_CODE " +
	    "SYSRES_CONST_CALL_TYPE_VARIABLE_BUTTON_VALUE " +
	    "SYSRES_CONST_CALL_TYPE_VARIABLE_PROGRAM_VALUE " +
	    "SYSRES_CONST_CANCEL_MESSAGE_FUNCTION_RESULT " +
	    "SYSRES_CONST_CARD_PART " +
	    "SYSRES_CONST_CARD_REFERENCE_MODE_NAME " +
	    "SYSRES_CONST_CERTIFICATE_TYPE_REQUISITE_ENCRYPT_VALUE " +
	    "SYSRES_CONST_CERTIFICATE_TYPE_REQUISITE_SIGN_AND_ENCRYPT_VALUE " +
	    "SYSRES_CONST_CERTIFICATE_TYPE_REQUISITE_SIGN_VALUE " +
	    "SYSRES_CONST_CHECK_PARAM_VALUE_DATE_PARAM_TYPE " +
	    "SYSRES_CONST_CHECK_PARAM_VALUE_FLOAT_PARAM_TYPE " +
	    "SYSRES_CONST_CHECK_PARAM_VALUE_INTEGER_PARAM_TYPE " +
	    "SYSRES_CONST_CHECK_PARAM_VALUE_PICK_PARAM_TYPE " +
	    "SYSRES_CONST_CHECK_PARAM_VALUE_REEFRENCE_PARAM_TYPE " +
	    "SYSRES_CONST_CLOSED_RECORD_FLAG_VALUE_FEMININE " +
	    "SYSRES_CONST_CLOSED_RECORD_FLAG_VALUE_MASCULINE " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_ADMIN " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_DEVELOPER " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_DOCS " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_EDOC_CARDS " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_EXTERNAL_EXECUTABLE " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_OTHER " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_REFERENCE " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_REPORT " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_SCRIPT " +
	    "SYSRES_CONST_CODE_COMPONENT_TYPE_URL " +
	    "SYSRES_CONST_CODE_REQUISITE_ACCESS " +
	    "SYSRES_CONST_CODE_REQUISITE_CODE " +
	    "SYSRES_CONST_CODE_REQUISITE_COMPONENT " +
	    "SYSRES_CONST_CODE_REQUISITE_DESCRIPTION " +
	    "SYSRES_CONST_CODE_REQUISITE_EXCLUDE_COMPONENT " +
	    "SYSRES_CONST_CODE_REQUISITE_RECORD " +
	    "SYSRES_CONST_COMMENT_REQ_CODE " +
	    "SYSRES_CONST_COMMON_SETTINGS_REQUISITE_CODE " +
	    "SYSRES_CONST_COMP_CODE_GRD " +
	    "SYSRES_CONST_COMPONENT_GROUP_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_COMPONENT_TYPE_ADMIN_COMPONENTS " +
	    "SYSRES_CONST_COMPONENT_TYPE_DEVELOPER_COMPONENTS " +
	    "SYSRES_CONST_COMPONENT_TYPE_DOCS " +
	    "SYSRES_CONST_COMPONENT_TYPE_EDOC_CARDS " +
	    "SYSRES_CONST_COMPONENT_TYPE_EDOCS " +
	    "SYSRES_CONST_COMPONENT_TYPE_EXTERNAL_EXECUTABLE " +
	    "SYSRES_CONST_COMPONENT_TYPE_OTHER " +
	    "SYSRES_CONST_COMPONENT_TYPE_REFERENCE_TYPES " +
	    "SYSRES_CONST_COMPONENT_TYPE_REFERENCES " +
	    "SYSRES_CONST_COMPONENT_TYPE_REPORTS " +
	    "SYSRES_CONST_COMPONENT_TYPE_SCRIPTS " +
	    "SYSRES_CONST_COMPONENT_TYPE_URL " +
	    "SYSRES_CONST_COMPONENTS_REMOTE_SERVERS_VIEW_CODE " +
	    "SYSRES_CONST_CONDITION_BLOCK_DESCRIPTION " +
	    "SYSRES_CONST_CONST_FIRM_STATUS_COMMON " +
	    "SYSRES_CONST_CONST_FIRM_STATUS_INDIVIDUAL " +
	    "SYSRES_CONST_CONST_NEGATIVE_VALUE " +
	    "SYSRES_CONST_CONST_POSITIVE_VALUE " +
	    "SYSRES_CONST_CONST_SERVER_STATUS_DONT_REPLICATE " +
	    "SYSRES_CONST_CONST_SERVER_STATUS_REPLICATE " +
	    "SYSRES_CONST_CONTENTS_REQUISITE_CODE " +
	    "SYSRES_CONST_DATA_TYPE_BOOLEAN " +
	    "SYSRES_CONST_DATA_TYPE_DATE " +
	    "SYSRES_CONST_DATA_TYPE_FLOAT " +
	    "SYSRES_CONST_DATA_TYPE_INTEGER " +
	    "SYSRES_CONST_DATA_TYPE_PICK " +
	    "SYSRES_CONST_DATA_TYPE_REFERENCE " +
	    "SYSRES_CONST_DATA_TYPE_STRING " +
	    "SYSRES_CONST_DATA_TYPE_TEXT " +
	    "SYSRES_CONST_DATA_TYPE_VARIANT " +
	    "SYSRES_CONST_DATE_CLOSE_REQ_CODE " +
	    "SYSRES_CONST_DATE_FORMAT_DATE_ONLY_CHAR " +
	    "SYSRES_CONST_DATE_OPEN_REQ_CODE " +
	    "SYSRES_CONST_DATE_REQUISITE " +
	    "SYSRES_CONST_DATE_REQUISITE_CODE " +
	    "SYSRES_CONST_DATE_REQUISITE_NAME " +
	    "SYSRES_CONST_DATE_REQUISITE_TYPE " +
	    "SYSRES_CONST_DATE_TYPE_CHAR " +
	    "SYSRES_CONST_DATETIME_FORMAT_VALUE " +
	    "SYSRES_CONST_DEA_ACCESS_RIGHTS_ACTION_CODE " +
	    "SYSRES_CONST_DESCRIPTION_LOCALIZE_ID_REQUISITE_CODE " +
	    "SYSRES_CONST_DESCRIPTION_REQUISITE_CODE " +
	    "SYSRES_CONST_DET1_PART " +
	    "SYSRES_CONST_DET2_PART " +
	    "SYSRES_CONST_DET3_PART " +
	    "SYSRES_CONST_DET4_PART " +
	    "SYSRES_CONST_DET5_PART " +
	    "SYSRES_CONST_DET6_PART " +
	    "SYSRES_CONST_DETAIL_DATASET_KEY_REQUISITE_CODE " +
	    "SYSRES_CONST_DETAIL_PICK_REQUISITE_CODE " +
	    "SYSRES_CONST_DETAIL_REQ_CODE " +
	    "SYSRES_CONST_DO_NOT_USE_ACCESS_TYPE_CODE " +
	    "SYSRES_CONST_DO_NOT_USE_ACCESS_TYPE_NAME " +
	    "SYSRES_CONST_DO_NOT_USE_ON_VIEW_ACCESS_TYPE_CODE " +
	    "SYSRES_CONST_DO_NOT_USE_ON_VIEW_ACCESS_TYPE_NAME " +
	    "SYSRES_CONST_DOCUMENT_STORAGES_CODE " +
	    "SYSRES_CONST_DOCUMENT_TEMPLATES_TYPE_NAME " +
	    "SYSRES_CONST_DOUBLE_REQUISITE_CODE " +
	    "SYSRES_CONST_EDITOR_CLOSE_FILE_OBSERV_TYPE_CODE " +
	    "SYSRES_CONST_EDITOR_CLOSE_PROCESS_OBSERV_TYPE_CODE " +
	    "SYSRES_CONST_EDITOR_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_EDITORS_APPLICATION_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_EDITORS_CREATE_SEVERAL_PROCESSES_REQUISITE_CODE " +
	    "SYSRES_CONST_EDITORS_EXTENSION_REQUISITE_CODE " +
	    "SYSRES_CONST_EDITORS_OBSERVER_BY_PROCESS_TYPE " +
	    "SYSRES_CONST_EDITORS_REFERENCE_CODE " +
	    "SYSRES_CONST_EDITORS_REPLACE_SPEC_CHARS_REQUISITE_CODE " +
	    "SYSRES_CONST_EDITORS_USE_PLUGINS_REQUISITE_CODE " +
	    "SYSRES_CONST_EDITORS_VIEW_DOCUMENT_OPENED_TO_EDIT_CODE " +
	    "SYSRES_CONST_EDOC_CARD_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_CARD_TYPES_LINK_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_CERTIFICATE_AND_PASSWORD_ENCODE_CODE " +
	    "SYSRES_CONST_EDOC_CERTIFICATE_ENCODE_CODE " +
	    "SYSRES_CONST_EDOC_DATE_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_KIND_REFERENCE_CODE " +
	    "SYSRES_CONST_EDOC_KINDS_BY_TEMPLATE_ACTION_CODE " +
	    "SYSRES_CONST_EDOC_MANAGE_ACCESS_CODE " +
	    "SYSRES_CONST_EDOC_NONE_ENCODE_CODE " +
	    "SYSRES_CONST_EDOC_NUMBER_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_PASSWORD_ENCODE_CODE " +
	    "SYSRES_CONST_EDOC_READONLY_ACCESS_CODE " +
	    "SYSRES_CONST_EDOC_SHELL_LIFE_TYPE_VIEW_VALUE " +
	    "SYSRES_CONST_EDOC_SIZE_RESTRICTION_PRIORITY_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_STORAGE_CHECK_ACCESS_RIGHTS_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_STORAGE_COMPUTER_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_STORAGE_DATABASE_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_STORAGE_EDIT_IN_STORAGE_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_STORAGE_LOCAL_PATH_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_STORAGE_SHARED_SOURCE_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_TEMPLATE_REQUISITE_CODE " +
	    "SYSRES_CONST_EDOC_TYPES_REFERENCE_CODE " +
	    "SYSRES_CONST_EDOC_VERSION_ACTIVE_STAGE_CODE " +
	    "SYSRES_CONST_EDOC_VERSION_DESIGN_STAGE_CODE " +
	    "SYSRES_CONST_EDOC_VERSION_OBSOLETE_STAGE_CODE " +
	    "SYSRES_CONST_EDOC_WRITE_ACCES_CODE " +
	    "SYSRES_CONST_EDOCUMENT_CARD_REQUISITES_REFERENCE_CODE_SELECTED_REQUISITE " +
	    "SYSRES_CONST_ENCODE_CERTIFICATE_TYPE_CODE " +
	    "SYSRES_CONST_END_DATE_REQUISITE_CODE " +
	    "SYSRES_CONST_ENUMERATION_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_EXECUTE_ACCESS_RIGHTS_TYPE_CODE " +
	    "SYSRES_CONST_EXECUTIVE_FILE_STORAGE_TYPE " +
	    "SYSRES_CONST_EXIST_CONST " +
	    "SYSRES_CONST_EXIST_VALUE " +
	    "SYSRES_CONST_EXPORT_LOCK_TYPE_ASK " +
	    "SYSRES_CONST_EXPORT_LOCK_TYPE_WITH_LOCK " +
	    "SYSRES_CONST_EXPORT_LOCK_TYPE_WITHOUT_LOCK " +
	    "SYSRES_CONST_EXPORT_VERSION_TYPE_ASK " +
	    "SYSRES_CONST_EXPORT_VERSION_TYPE_LAST " +
	    "SYSRES_CONST_EXPORT_VERSION_TYPE_LAST_ACTIVE " +
	    "SYSRES_CONST_EXTENSION_REQUISITE_CODE " +
	    "SYSRES_CONST_FILTER_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_FILTER_REQUISITE_CODE " +
	    "SYSRES_CONST_FILTER_TYPE_COMMON_CODE " +
	    "SYSRES_CONST_FILTER_TYPE_COMMON_NAME " +
	    "SYSRES_CONST_FILTER_TYPE_USER_CODE " +
	    "SYSRES_CONST_FILTER_TYPE_USER_NAME " +
	    "SYSRES_CONST_FILTER_VALUE_REQUISITE_NAME " +
	    "SYSRES_CONST_FLOAT_NUMBER_FORMAT_CHAR " +
	    "SYSRES_CONST_FLOAT_REQUISITE_TYPE " +
	    "SYSRES_CONST_FOLDER_AUTHOR_VALUE " +
	    "SYSRES_CONST_FOLDER_KIND_ANY_OBJECTS " +
	    "SYSRES_CONST_FOLDER_KIND_COMPONENTS " +
	    "SYSRES_CONST_FOLDER_KIND_EDOCS " +
	    "SYSRES_CONST_FOLDER_KIND_JOBS " +
	    "SYSRES_CONST_FOLDER_KIND_TASKS " +
	    "SYSRES_CONST_FOLDER_TYPE_COMMON " +
	    "SYSRES_CONST_FOLDER_TYPE_COMPONENT " +
	    "SYSRES_CONST_FOLDER_TYPE_FAVORITES " +
	    "SYSRES_CONST_FOLDER_TYPE_INBOX " +
	    "SYSRES_CONST_FOLDER_TYPE_OUTBOX " +
	    "SYSRES_CONST_FOLDER_TYPE_QUICK_LAUNCH " +
	    "SYSRES_CONST_FOLDER_TYPE_SEARCH " +
	    "SYSRES_CONST_FOLDER_TYPE_SHORTCUTS " +
	    "SYSRES_CONST_FOLDER_TYPE_USER " +
	    "SYSRES_CONST_FROM_DICTIONARY_ENUM_METHOD_FLAG " +
	    "SYSRES_CONST_FULL_SUBSTITUTE_TYPE " +
	    "SYSRES_CONST_FULL_SUBSTITUTE_TYPE_CODE " +
	    "SYSRES_CONST_FUNCTION_CANCEL_RESULT " +
	    "SYSRES_CONST_FUNCTION_CATEGORY_SYSTEM " +
	    "SYSRES_CONST_FUNCTION_CATEGORY_USER " +
	    "SYSRES_CONST_FUNCTION_FAILURE_RESULT " +
	    "SYSRES_CONST_FUNCTION_SAVE_RESULT " +
	    "SYSRES_CONST_GENERATED_REQUISITE " +
	    "SYSRES_CONST_GREEN_LIFE_CYCLE_STAGE_FONT_COLOR " +
	    "SYSRES_CONST_GROUP_ACCOUNT_TYPE_VALUE_CODE " +
	    "SYSRES_CONST_GROUP_CATEGORY_NORMAL_CODE " +
	    "SYSRES_CONST_GROUP_CATEGORY_NORMAL_NAME " +
	    "SYSRES_CONST_GROUP_CATEGORY_SERVICE_CODE " +
	    "SYSRES_CONST_GROUP_CATEGORY_SERVICE_NAME " +
	    "SYSRES_CONST_GROUP_COMMON_CATEGORY_FIELD_VALUE " +
	    "SYSRES_CONST_GROUP_FULL_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_GROUP_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_GROUP_RIGHTS_T_REQUISITE_CODE " +
	    "SYSRES_CONST_GROUP_SERVER_CODES_REQUISITE_CODE " +
	    "SYSRES_CONST_GROUP_SERVER_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_GROUP_SERVICE_CATEGORY_FIELD_VALUE " +
	    "SYSRES_CONST_GROUP_USER_REQUISITE_CODE " +
	    "SYSRES_CONST_GROUPS_REFERENCE_CODE " +
	    "SYSRES_CONST_GROUPS_REQUISITE_CODE " +
	    "SYSRES_CONST_HIDDEN_MODE_NAME " +
	    "SYSRES_CONST_HIGH_LVL_REQUISITE_CODE " +
	    "SYSRES_CONST_HISTORY_ACTION_CREATE_CODE " +
	    "SYSRES_CONST_HISTORY_ACTION_DELETE_CODE " +
	    "SYSRES_CONST_HISTORY_ACTION_EDIT_CODE " +
	    "SYSRES_CONST_HOUR_CHAR " +
	    "SYSRES_CONST_ID_REQUISITE_CODE " +
	    "SYSRES_CONST_IDSPS_REQUISITE_CODE " +
	    "SYSRES_CONST_IMAGE_MODE_COLOR " +
	    "SYSRES_CONST_IMAGE_MODE_GREYSCALE " +
	    "SYSRES_CONST_IMAGE_MODE_MONOCHROME " +
	    "SYSRES_CONST_IMPORTANCE_HIGH " +
	    "SYSRES_CONST_IMPORTANCE_LOW " +
	    "SYSRES_CONST_IMPORTANCE_NORMAL " +
	    "SYSRES_CONST_IN_DESIGN_VERSION_STATE_PICK_VALUE " +
	    "SYSRES_CONST_INCOMING_WORK_RULE_TYPE_CODE " +
	    "SYSRES_CONST_INT_REQUISITE " +
	    "SYSRES_CONST_INT_REQUISITE_TYPE " +
	    "SYSRES_CONST_INTEGER_NUMBER_FORMAT_CHAR " +
	    "SYSRES_CONST_INTEGER_TYPE_CHAR " +
	    "SYSRES_CONST_IS_GENERATED_REQUISITE_NEGATIVE_VALUE " +
	    "SYSRES_CONST_IS_PUBLIC_ROLE_REQUISITE_CODE " +
	    "SYSRES_CONST_IS_REMOTE_USER_NEGATIVE_VALUE " +
	    "SYSRES_CONST_IS_REMOTE_USER_POSITIVE_VALUE " +
	    "SYSRES_CONST_IS_STORED_REQUISITE_NEGATIVE_VALUE " +
	    "SYSRES_CONST_IS_STORED_REQUISITE_STORED_VALUE " +
	    "SYSRES_CONST_ITALIC_LIFE_CYCLE_STAGE_DRAW_STYLE " +
	    "SYSRES_CONST_JOB_BLOCK_DESCRIPTION " +
	    "SYSRES_CONST_JOB_KIND_CONTROL_JOB " +
	    "SYSRES_CONST_JOB_KIND_JOB " +
	    "SYSRES_CONST_JOB_KIND_NOTICE " +
	    "SYSRES_CONST_JOB_STATE_ABORTED " +
	    "SYSRES_CONST_JOB_STATE_COMPLETE " +
	    "SYSRES_CONST_JOB_STATE_WORKING " +
	    "SYSRES_CONST_KIND_REQUISITE_CODE " +
	    "SYSRES_CONST_KIND_REQUISITE_NAME " +
	    "SYSRES_CONST_KINDS_CREATE_SHADOW_COPIES_REQUISITE_CODE " +
	    "SYSRES_CONST_KINDS_DEFAULT_EDOC_LIFE_STAGE_REQUISITE_CODE " +
	    "SYSRES_CONST_KINDS_EDOC_ALL_TEPLATES_ALLOWED_REQUISITE_CODE " +
	    "SYSRES_CONST_KINDS_EDOC_ALLOW_LIFE_CYCLE_STAGE_CHANGING_REQUISITE_CODE " +
	    "SYSRES_CONST_KINDS_EDOC_ALLOW_MULTIPLE_ACTIVE_VERSIONS_REQUISITE_CODE " +
	    "SYSRES_CONST_KINDS_EDOC_SHARE_ACCES_RIGHTS_BY_DEFAULT_CODE " +
	    "SYSRES_CONST_KINDS_EDOC_TEMPLATE_REQUISITE_CODE " +
	    "SYSRES_CONST_KINDS_EDOC_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_KINDS_SIGNERS_REQUISITES_CODE " +
	    "SYSRES_CONST_KOD_INPUT_TYPE " +
	    "SYSRES_CONST_LAST_UPDATE_DATE_REQUISITE_CODE " +
	    "SYSRES_CONST_LIFE_CYCLE_START_STAGE_REQUISITE_CODE " +
	    "SYSRES_CONST_LILAC_LIFE_CYCLE_STAGE_FONT_COLOR " +
	    "SYSRES_CONST_LINK_OBJECT_KIND_COMPONENT " +
	    "SYSRES_CONST_LINK_OBJECT_KIND_DOCUMENT " +
	    "SYSRES_CONST_LINK_OBJECT_KIND_EDOC " +
	    "SYSRES_CONST_LINK_OBJECT_KIND_FOLDER " +
	    "SYSRES_CONST_LINK_OBJECT_KIND_JOB " +
	    "SYSRES_CONST_LINK_OBJECT_KIND_REFERENCE " +
	    "SYSRES_CONST_LINK_OBJECT_KIND_TASK " +
	    "SYSRES_CONST_LINK_REF_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_LIST_REFERENCE_MODE_NAME " +
	    "SYSRES_CONST_LOCALIZATION_DICTIONARY_MAIN_VIEW_CODE " +
	    "SYSRES_CONST_MAIN_VIEW_CODE " +
	    "SYSRES_CONST_MANUAL_ENUM_METHOD_FLAG " +
	    "SYSRES_CONST_MASTER_COMP_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_MASTER_TABLE_REC_ID_REQUISITE_CODE " +
	    "SYSRES_CONST_MAXIMIZED_MODE_NAME " +
	    "SYSRES_CONST_ME_VALUE " +
	    "SYSRES_CONST_MESSAGE_ATTENTION_CAPTION " +
	    "SYSRES_CONST_MESSAGE_CONFIRMATION_CAPTION " +
	    "SYSRES_CONST_MESSAGE_ERROR_CAPTION " +
	    "SYSRES_CONST_MESSAGE_INFORMATION_CAPTION " +
	    "SYSRES_CONST_MINIMIZED_MODE_NAME " +
	    "SYSRES_CONST_MINUTE_CHAR " +
	    "SYSRES_CONST_MODULE_REQUISITE_CODE " +
	    "SYSRES_CONST_MONITORING_BLOCK_DESCRIPTION " +
	    "SYSRES_CONST_MONTH_FORMAT_VALUE " +
	    "SYSRES_CONST_NAME_LOCALIZE_ID_REQUISITE_CODE " +
	    "SYSRES_CONST_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_NAME_SINGULAR_REQUISITE_CODE " +
	    "SYSRES_CONST_NAMEAN_INPUT_TYPE " +
	    "SYSRES_CONST_NEGATIVE_PICK_VALUE " +
	    "SYSRES_CONST_NEGATIVE_VALUE " +
	    "SYSRES_CONST_NO " +
	    "SYSRES_CONST_NO_PICK_VALUE " +
	    "SYSRES_CONST_NO_SIGNATURE_REQUISITE_CODE " +
	    "SYSRES_CONST_NO_VALUE " +
	    "SYSRES_CONST_NONE_ACCESS_RIGHTS_TYPE_CODE " +
	    "SYSRES_CONST_NONOPERATING_RECORD_FLAG_VALUE " +
	    "SYSRES_CONST_NONOPERATING_RECORD_FLAG_VALUE_MASCULINE " +
	    "SYSRES_CONST_NORMAL_ACCESS_RIGHTS_TYPE_CODE " +
	    "SYSRES_CONST_NORMAL_LIFE_CYCLE_STAGE_DRAW_STYLE " +
	    "SYSRES_CONST_NORMAL_MODE_NAME " +
	    "SYSRES_CONST_NOT_ALLOWED_ACCESS_TYPE_CODE " +
	    "SYSRES_CONST_NOT_ALLOWED_ACCESS_TYPE_NAME " +
	    "SYSRES_CONST_NOTE_REQUISITE_CODE " +
	    "SYSRES_CONST_NOTICE_BLOCK_DESCRIPTION " +
	    "SYSRES_CONST_NUM_REQUISITE " +
	    "SYSRES_CONST_NUM_STR_REQUISITE_CODE " +
	    "SYSRES_CONST_NUMERATION_AUTO_NOT_STRONG " +
	    "SYSRES_CONST_NUMERATION_AUTO_STRONG " +
	    "SYSRES_CONST_NUMERATION_FROM_DICTONARY " +
	    "SYSRES_CONST_NUMERATION_MANUAL " +
	    "SYSRES_CONST_NUMERIC_TYPE_CHAR " +
	    "SYSRES_CONST_NUMREQ_REQUISITE_CODE " +
	    "SYSRES_CONST_OBSOLETE_VERSION_STATE_PICK_VALUE " +
	    "SYSRES_CONST_OPERATING_RECORD_FLAG_VALUE " +
	    "SYSRES_CONST_OPERATING_RECORD_FLAG_VALUE_CODE " +
	    "SYSRES_CONST_OPERATING_RECORD_FLAG_VALUE_FEMININE " +
	    "SYSRES_CONST_OPERATING_RECORD_FLAG_VALUE_MASCULINE " +
	    "SYSRES_CONST_OPTIONAL_FORM_COMP_REQCODE_PREFIX " +
	    "SYSRES_CONST_ORANGE_LIFE_CYCLE_STAGE_FONT_COLOR " +
	    "SYSRES_CONST_ORIGINALREF_REQUISITE_CODE " +
	    "SYSRES_CONST_OURFIRM_REF_CODE " +
	    "SYSRES_CONST_OURFIRM_REQUISITE_CODE " +
	    "SYSRES_CONST_OURFIRM_VAR " +
	    "SYSRES_CONST_OUTGOING_WORK_RULE_TYPE_CODE " +
	    "SYSRES_CONST_PICK_NEGATIVE_RESULT " +
	    "SYSRES_CONST_PICK_POSITIVE_RESULT " +
	    "SYSRES_CONST_PICK_REQUISITE " +
	    "SYSRES_CONST_PICK_REQUISITE_TYPE " +
	    "SYSRES_CONST_PICK_TYPE_CHAR " +
	    "SYSRES_CONST_PLAN_STATUS_REQUISITE_CODE " +
	    "SYSRES_CONST_PLATFORM_VERSION_COMMENT " +
	    "SYSRES_CONST_PLUGINS_SETTINGS_DESCRIPTION_REQUISITE_CODE " +
	    "SYSRES_CONST_POSITIVE_PICK_VALUE " +
	    "SYSRES_CONST_POWER_TO_CREATE_ACTION_CODE " +
	    "SYSRES_CONST_POWER_TO_SIGN_ACTION_CODE " +
	    "SYSRES_CONST_PRIORITY_REQUISITE_CODE " +
	    "SYSRES_CONST_QUALIFIED_TASK_TYPE " +
	    "SYSRES_CONST_QUALIFIED_TASK_TYPE_CODE " +
	    "SYSRES_CONST_RECSTAT_REQUISITE_CODE " +
	    "SYSRES_CONST_RED_LIFE_CYCLE_STAGE_FONT_COLOR " +
	    "SYSRES_CONST_REF_ID_T_REF_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_REF_REQUISITE " +
	    "SYSRES_CONST_REF_REQUISITE_TYPE " +
	    "SYSRES_CONST_REF_REQUISITES_REFERENCE_CODE_SELECTED_REQUISITE " +
	    "SYSRES_CONST_REFERENCE_RECORD_HISTORY_CREATE_ACTION_CODE " +
	    "SYSRES_CONST_REFERENCE_RECORD_HISTORY_DELETE_ACTION_CODE " +
	    "SYSRES_CONST_REFERENCE_RECORD_HISTORY_MODIFY_ACTION_CODE " +
	    "SYSRES_CONST_REFERENCE_TYPE_CHAR " +
	    "SYSRES_CONST_REFERENCE_TYPE_REQUISITE_NAME " +
	    "SYSRES_CONST_REFERENCES_ADD_PARAMS_REQUISITE_CODE " +
	    "SYSRES_CONST_REFERENCES_DISPLAY_REQUISITE_REQUISITE_CODE " +
	    "SYSRES_CONST_REMOTE_SERVER_STATUS_WORKING " +
	    "SYSRES_CONST_REMOTE_SERVER_TYPE_MAIN " +
	    "SYSRES_CONST_REMOTE_SERVER_TYPE_SECONDARY " +
	    "SYSRES_CONST_REMOTE_USER_FLAG_VALUE_CODE " +
	    "SYSRES_CONST_REPORT_APP_EDITOR_INTERNAL " +
	    "SYSRES_CONST_REPORT_BASE_REPORT_ID_REQUISITE_CODE " +
	    "SYSRES_CONST_REPORT_BASE_REPORT_REQUISITE_CODE " +
	    "SYSRES_CONST_REPORT_SCRIPT_REQUISITE_CODE " +
	    "SYSRES_CONST_REPORT_TEMPLATE_REQUISITE_CODE " +
	    "SYSRES_CONST_REPORT_VIEWER_CODE_REQUISITE_CODE " +
	    "SYSRES_CONST_REQ_ALLOW_COMPONENT_DEFAULT_VALUE " +
	    "SYSRES_CONST_REQ_ALLOW_RECORD_DEFAULT_VALUE " +
	    "SYSRES_CONST_REQ_ALLOW_SERVER_COMPONENT_DEFAULT_VALUE " +
	    "SYSRES_CONST_REQ_MODE_AVAILABLE_CODE " +
	    "SYSRES_CONST_REQ_MODE_EDIT_CODE " +
	    "SYSRES_CONST_REQ_MODE_HIDDEN_CODE " +
	    "SYSRES_CONST_REQ_MODE_NOT_AVAILABLE_CODE " +
	    "SYSRES_CONST_REQ_MODE_VIEW_CODE " +
	    "SYSRES_CONST_REQ_NUMBER_REQUISITE_CODE " +
	    "SYSRES_CONST_REQ_SECTION_VALUE " +
	    "SYSRES_CONST_REQ_TYPE_VALUE " +
	    "SYSRES_CONST_REQUISITE_FORMAT_BY_UNIT " +
	    "SYSRES_CONST_REQUISITE_FORMAT_DATE_FULL " +
	    "SYSRES_CONST_REQUISITE_FORMAT_DATE_TIME " +
	    "SYSRES_CONST_REQUISITE_FORMAT_LEFT " +
	    "SYSRES_CONST_REQUISITE_FORMAT_RIGHT " +
	    "SYSRES_CONST_REQUISITE_FORMAT_WITHOUT_UNIT " +
	    "SYSRES_CONST_REQUISITE_NUMBER_REQUISITE_CODE " +
	    "SYSRES_CONST_REQUISITE_SECTION_ACTIONS " +
	    "SYSRES_CONST_REQUISITE_SECTION_BUTTON " +
	    "SYSRES_CONST_REQUISITE_SECTION_BUTTONS " +
	    "SYSRES_CONST_REQUISITE_SECTION_CARD " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE10 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE11 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE12 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE13 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE14 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE15 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE16 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE17 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE18 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE19 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE2 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE20 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE21 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE22 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE23 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE24 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE3 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE4 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE5 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE6 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE7 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE8 " +
	    "SYSRES_CONST_REQUISITE_SECTION_TABLE9 " +
	    "SYSRES_CONST_REQUISITES_PSEUDOREFERENCE_REQUISITE_NUMBER_REQUISITE_CODE " +
	    "SYSRES_CONST_RIGHT_ALIGNMENT_CODE " +
	    "SYSRES_CONST_ROLES_REFERENCE_CODE " +
	    "SYSRES_CONST_ROUTE_STEP_AFTER_RUS " +
	    "SYSRES_CONST_ROUTE_STEP_AND_CONDITION_RUS " +
	    "SYSRES_CONST_ROUTE_STEP_OR_CONDITION_RUS " +
	    "SYSRES_CONST_ROUTE_TYPE_COMPLEX " +
	    "SYSRES_CONST_ROUTE_TYPE_PARALLEL " +
	    "SYSRES_CONST_ROUTE_TYPE_SERIAL " +
	    "SYSRES_CONST_SBDATASETDESC_NEGATIVE_VALUE " +
	    "SYSRES_CONST_SBDATASETDESC_POSITIVE_VALUE " +
	    "SYSRES_CONST_SBVIEWSDESC_POSITIVE_VALUE " +
	    "SYSRES_CONST_SCRIPT_BLOCK_DESCRIPTION " +
	    "SYSRES_CONST_SEARCH_BY_TEXT_REQUISITE_CODE " +
	    "SYSRES_CONST_SEARCHES_COMPONENT_CONTENT " +
	    "SYSRES_CONST_SEARCHES_CRITERIA_ACTION_NAME " +
	    "SYSRES_CONST_SEARCHES_EDOC_CONTENT " +
	    "SYSRES_CONST_SEARCHES_FOLDER_CONTENT " +
	    "SYSRES_CONST_SEARCHES_JOB_CONTENT " +
	    "SYSRES_CONST_SEARCHES_REFERENCE_CODE " +
	    "SYSRES_CONST_SEARCHES_TASK_CONTENT " +
	    "SYSRES_CONST_SECOND_CHAR " +
	    "SYSRES_CONST_SECTION_REQUISITE_ACTIONS_VALUE " +
	    "SYSRES_CONST_SECTION_REQUISITE_CARD_VALUE " +
	    "SYSRES_CONST_SECTION_REQUISITE_CODE " +
	    "SYSRES_CONST_SECTION_REQUISITE_DETAIL_1_VALUE " +
	    "SYSRES_CONST_SECTION_REQUISITE_DETAIL_2_VALUE " +
	    "SYSRES_CONST_SECTION_REQUISITE_DETAIL_3_VALUE " +
	    "SYSRES_CONST_SECTION_REQUISITE_DETAIL_4_VALUE " +
	    "SYSRES_CONST_SECTION_REQUISITE_DETAIL_5_VALUE " +
	    "SYSRES_CONST_SECTION_REQUISITE_DETAIL_6_VALUE " +
	    "SYSRES_CONST_SELECT_REFERENCE_MODE_NAME " +
	    "SYSRES_CONST_SELECT_TYPE_SELECTABLE " +
	    "SYSRES_CONST_SELECT_TYPE_SELECTABLE_ONLY_CHILD " +
	    "SYSRES_CONST_SELECT_TYPE_SELECTABLE_WITH_CHILD " +
	    "SYSRES_CONST_SELECT_TYPE_UNSLECTABLE " +
	    "SYSRES_CONST_SERVER_TYPE_MAIN " +
	    "SYSRES_CONST_SERVICE_USER_CATEGORY_FIELD_VALUE " +
	    "SYSRES_CONST_SETTINGS_USER_REQUISITE_CODE " +
	    "SYSRES_CONST_SIGNATURE_AND_ENCODE_CERTIFICATE_TYPE_CODE " +
	    "SYSRES_CONST_SIGNATURE_CERTIFICATE_TYPE_CODE " +
	    "SYSRES_CONST_SINGULAR_TITLE_REQUISITE_CODE " +
	    "SYSRES_CONST_SQL_SERVER_AUTHENTIFICATION_FLAG_VALUE_CODE " +
	    "SYSRES_CONST_SQL_SERVER_ENCODE_AUTHENTIFICATION_FLAG_VALUE_CODE " +
	    "SYSRES_CONST_STANDART_ROUTE_REFERENCE_CODE " +
	    "SYSRES_CONST_STANDART_ROUTE_REFERENCE_COMMENT_REQUISITE_CODE " +
	    "SYSRES_CONST_STANDART_ROUTES_GROUPS_REFERENCE_CODE " +
	    "SYSRES_CONST_STATE_REQ_NAME " +
	    "SYSRES_CONST_STATE_REQUISITE_ACTIVE_VALUE " +
	    "SYSRES_CONST_STATE_REQUISITE_CLOSED_VALUE " +
	    "SYSRES_CONST_STATE_REQUISITE_CODE " +
	    "SYSRES_CONST_STATIC_ROLE_TYPE_CODE " +
	    "SYSRES_CONST_STATUS_PLAN_DEFAULT_VALUE " +
	    "SYSRES_CONST_STATUS_VALUE_AUTOCLEANING " +
	    "SYSRES_CONST_STATUS_VALUE_BLUE_SQUARE " +
	    "SYSRES_CONST_STATUS_VALUE_COMPLETE " +
	    "SYSRES_CONST_STATUS_VALUE_GREEN_SQUARE " +
	    "SYSRES_CONST_STATUS_VALUE_ORANGE_SQUARE " +
	    "SYSRES_CONST_STATUS_VALUE_PURPLE_SQUARE " +
	    "SYSRES_CONST_STATUS_VALUE_RED_SQUARE " +
	    "SYSRES_CONST_STATUS_VALUE_SUSPEND " +
	    "SYSRES_CONST_STATUS_VALUE_YELLOW_SQUARE " +
	    "SYSRES_CONST_STDROUTE_SHOW_TO_USERS_REQUISITE_CODE " +
	    "SYSRES_CONST_STORAGE_TYPE_FILE " +
	    "SYSRES_CONST_STORAGE_TYPE_SQL_SERVER " +
	    "SYSRES_CONST_STR_REQUISITE " +
	    "SYSRES_CONST_STRIKEOUT_LIFE_CYCLE_STAGE_DRAW_STYLE " +
	    "SYSRES_CONST_STRING_FORMAT_LEFT_ALIGN_CHAR " +
	    "SYSRES_CONST_STRING_FORMAT_RIGHT_ALIGN_CHAR " +
	    "SYSRES_CONST_STRING_REQUISITE_CODE " +
	    "SYSRES_CONST_STRING_REQUISITE_TYPE " +
	    "SYSRES_CONST_STRING_TYPE_CHAR " +
	    "SYSRES_CONST_SUBSTITUTES_PSEUDOREFERENCE_CODE " +
	    "SYSRES_CONST_SUBTASK_BLOCK_DESCRIPTION " +
	    "SYSRES_CONST_SYSTEM_SETTING_CURRENT_USER_PARAM_VALUE " +
	    "SYSRES_CONST_SYSTEM_SETTING_EMPTY_VALUE_PARAM_VALUE " +
	    "SYSRES_CONST_SYSTEM_VERSION_COMMENT " +
	    "SYSRES_CONST_TASK_ACCESS_TYPE_ALL " +
	    "SYSRES_CONST_TASK_ACCESS_TYPE_ALL_MEMBERS " +
	    "SYSRES_CONST_TASK_ACCESS_TYPE_MANUAL " +
	    "SYSRES_CONST_TASK_ENCODE_TYPE_CERTIFICATION " +
	    "SYSRES_CONST_TASK_ENCODE_TYPE_CERTIFICATION_AND_PASSWORD " +
	    "SYSRES_CONST_TASK_ENCODE_TYPE_NONE " +
	    "SYSRES_CONST_TASK_ENCODE_TYPE_PASSWORD " +
	    "SYSRES_CONST_TASK_ROUTE_ALL_CONDITION " +
	    "SYSRES_CONST_TASK_ROUTE_AND_CONDITION " +
	    "SYSRES_CONST_TASK_ROUTE_OR_CONDITION " +
	    "SYSRES_CONST_TASK_STATE_ABORTED " +
	    "SYSRES_CONST_TASK_STATE_COMPLETE " +
	    "SYSRES_CONST_TASK_STATE_CONTINUED " +
	    "SYSRES_CONST_TASK_STATE_CONTROL " +
	    "SYSRES_CONST_TASK_STATE_INIT " +
	    "SYSRES_CONST_TASK_STATE_WORKING " +
	    "SYSRES_CONST_TASK_TITLE " +
	    "SYSRES_CONST_TASK_TYPES_GROUPS_REFERENCE_CODE " +
	    "SYSRES_CONST_TASK_TYPES_REFERENCE_CODE " +
	    "SYSRES_CONST_TEMPLATES_REFERENCE_CODE " +
	    "SYSRES_CONST_TEST_DATE_REQUISITE_NAME " +
	    "SYSRES_CONST_TEST_DEV_DATABASE_NAME " +
	    "SYSRES_CONST_TEST_DEV_SYSTEM_CODE " +
	    "SYSRES_CONST_TEST_EDMS_DATABASE_NAME " +
	    "SYSRES_CONST_TEST_EDMS_MAIN_CODE " +
	    "SYSRES_CONST_TEST_EDMS_MAIN_DB_NAME " +
	    "SYSRES_CONST_TEST_EDMS_SECOND_CODE " +
	    "SYSRES_CONST_TEST_EDMS_SECOND_DB_NAME " +
	    "SYSRES_CONST_TEST_EDMS_SYSTEM_CODE " +
	    "SYSRES_CONST_TEST_NUMERIC_REQUISITE_NAME " +
	    "SYSRES_CONST_TEXT_REQUISITE " +
	    "SYSRES_CONST_TEXT_REQUISITE_CODE " +
	    "SYSRES_CONST_TEXT_REQUISITE_TYPE " +
	    "SYSRES_CONST_TEXT_TYPE_CHAR " +
	    "SYSRES_CONST_TYPE_CODE_REQUISITE_CODE " +
	    "SYSRES_CONST_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_UNDEFINED_LIFE_CYCLE_STAGE_FONT_COLOR " +
	    "SYSRES_CONST_UNITS_SECTION_ID_REQUISITE_CODE " +
	    "SYSRES_CONST_UNITS_SECTION_REQUISITE_CODE " +
	    "SYSRES_CONST_UNOPERATING_RECORD_FLAG_VALUE_CODE " +
	    "SYSRES_CONST_UNSTORED_DATA_REQUISITE_CODE " +
	    "SYSRES_CONST_UNSTORED_DATA_REQUISITE_NAME " +
	    "SYSRES_CONST_USE_ACCESS_TYPE_CODE " +
	    "SYSRES_CONST_USE_ACCESS_TYPE_NAME " +
	    "SYSRES_CONST_USER_ACCOUNT_TYPE_VALUE_CODE " +
	    "SYSRES_CONST_USER_ADDITIONAL_INFORMATION_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_AND_GROUP_ID_FROM_PSEUDOREFERENCE_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_CATEGORY_NORMAL " +
	    "SYSRES_CONST_USER_CERTIFICATE_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_CERTIFICATE_STATE_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_CERTIFICATE_SUBJECT_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_CERTIFICATE_THUMBPRINT_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_COMMON_CATEGORY " +
	    "SYSRES_CONST_USER_COMMON_CATEGORY_CODE " +
	    "SYSRES_CONST_USER_FULL_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_GROUP_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_LOGIN_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_REMOTE_CONTROLLER_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_REMOTE_SYSTEM_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_RIGHTS_T_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_SERVER_NAME_REQUISITE_CODE " +
	    "SYSRES_CONST_USER_SERVICE_CATEGORY " +
	    "SYSRES_CONST_USER_SERVICE_CATEGORY_CODE " +
	    "SYSRES_CONST_USER_STATUS_ADMINISTRATOR_CODE " +
	    "SYSRES_CONST_USER_STATUS_ADMINISTRATOR_NAME " +
	    "SYSRES_CONST_USER_STATUS_DEVELOPER_CODE " +
	    "SYSRES_CONST_USER_STATUS_DEVELOPER_NAME " +
	    "SYSRES_CONST_USER_STATUS_DISABLED_CODE " +
	    "SYSRES_CONST_USER_STATUS_DISABLED_NAME " +
	    "SYSRES_CONST_USER_STATUS_SYSTEM_DEVELOPER_CODE " +
	    "SYSRES_CONST_USER_STATUS_USER_CODE " +
	    "SYSRES_CONST_USER_STATUS_USER_NAME " +
	    "SYSRES_CONST_USER_STATUS_USER_NAME_DEPRECATED " +
	    "SYSRES_CONST_USER_TYPE_FIELD_VALUE_USER " +
	    "SYSRES_CONST_USER_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_CONTROLLER_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_IS_MAIN_SERVER_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_REFERENCE_CODE " +
	    "SYSRES_CONST_USERS_REGISTRATION_CERTIFICATES_ACTION_NAME " +
	    "SYSRES_CONST_USERS_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_SYSTEM_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_USER_ACCESS_RIGHTS_TYPR_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_USER_AUTHENTICATION_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_USER_COMPONENT_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_USER_GROUP_REQUISITE_CODE " +
	    "SYSRES_CONST_USERS_VIEW_CERTIFICATES_ACTION_NAME " +
	    "SYSRES_CONST_VIEW_DEFAULT_CODE " +
	    "SYSRES_CONST_VIEW_DEFAULT_NAME " +
	    "SYSRES_CONST_VIEWER_REQUISITE_CODE " +
	    "SYSRES_CONST_WAITING_BLOCK_DESCRIPTION " +
	    "SYSRES_CONST_WIZARD_FORM_LABEL_TEST_STRING  " +
	    "SYSRES_CONST_WIZARD_QUERY_PARAM_HEIGHT_ETALON_STRING " +
	    "SYSRES_CONST_WIZARD_REFERENCE_COMMENT_REQUISITE_CODE " +
	    "SYSRES_CONST_WORK_RULES_DESCRIPTION_REQUISITE_CODE " +
	    "SYSRES_CONST_WORK_TIME_CALENDAR_REFERENCE_CODE " +
	    "SYSRES_CONST_WORK_WORKFLOW_HARD_ROUTE_TYPE_VALUE " +
	    "SYSRES_CONST_WORK_WORKFLOW_HARD_ROUTE_TYPE_VALUE_CODE " +
	    "SYSRES_CONST_WORK_WORKFLOW_HARD_ROUTE_TYPE_VALUE_CODE_RUS " +
	    "SYSRES_CONST_WORK_WORKFLOW_SOFT_ROUTE_TYPE_VALUE_CODE_RUS " +
	    "SYSRES_CONST_WORKFLOW_ROUTE_TYPR_HARD " +
	    "SYSRES_CONST_WORKFLOW_ROUTE_TYPR_SOFT " +
	    "SYSRES_CONST_XML_ENCODING " +
	    "SYSRES_CONST_XREC_STAT_REQUISITE_CODE " +
	    "SYSRES_CONST_XRECID_FIELD_NAME " +
	    "SYSRES_CONST_YES " +
	    "SYSRES_CONST_YES_NO_2_REQUISITE_CODE " +
	    "SYSRES_CONST_YES_NO_REQUISITE_CODE " +
	    "SYSRES_CONST_YES_NO_T_REF_TYPE_REQUISITE_CODE " +
	    "SYSRES_CONST_YES_PICK_VALUE " +
	    "SYSRES_CONST_YES_VALUE ";

	  // Base constant
	  const base_constants = "CR FALSE nil NO_VALUE NULL TAB TRUE YES_VALUE ";

	  // Base group name
	  const base_group_name_constants =
	    "ADMINISTRATORS_GROUP_NAME CUSTOMIZERS_GROUP_NAME DEVELOPERS_GROUP_NAME SERVICE_USERS_GROUP_NAME ";

	  // Decision block properties
	  const decision_block_properties_constants =
	    "DECISION_BLOCK_FIRST_OPERAND_PROPERTY DECISION_BLOCK_NAME_PROPERTY DECISION_BLOCK_OPERATION_PROPERTY " +
	    "DECISION_BLOCK_RESULT_TYPE_PROPERTY DECISION_BLOCK_SECOND_OPERAND_PROPERTY ";

	  // File extension
	  const file_extension_constants =
	    "ANY_FILE_EXTENTION COMPRESSED_DOCUMENT_EXTENSION EXTENDED_DOCUMENT_EXTENSION " +
	    "SHORT_COMPRESSED_DOCUMENT_EXTENSION SHORT_EXTENDED_DOCUMENT_EXTENSION ";

	  // Job block properties
	  const job_block_properties_constants =
	    "JOB_BLOCK_ABORT_DEADLINE_PROPERTY " +
	    "JOB_BLOCK_AFTER_FINISH_EVENT " +
	    "JOB_BLOCK_AFTER_QUERY_PARAMETERS_EVENT " +
	    "JOB_BLOCK_ATTACHMENT_PROPERTY " +
	    "JOB_BLOCK_ATTACHMENTS_RIGHTS_GROUP_PROPERTY " +
	    "JOB_BLOCK_ATTACHMENTS_RIGHTS_TYPE_PROPERTY " +
	    "JOB_BLOCK_BEFORE_QUERY_PARAMETERS_EVENT " +
	    "JOB_BLOCK_BEFORE_START_EVENT " +
	    "JOB_BLOCK_CREATED_JOBS_PROPERTY " +
	    "JOB_BLOCK_DEADLINE_PROPERTY " +
	    "JOB_BLOCK_EXECUTION_RESULTS_PROPERTY " +
	    "JOB_BLOCK_IS_PARALLEL_PROPERTY " +
	    "JOB_BLOCK_IS_RELATIVE_ABORT_DEADLINE_PROPERTY " +
	    "JOB_BLOCK_IS_RELATIVE_DEADLINE_PROPERTY " +
	    "JOB_BLOCK_JOB_TEXT_PROPERTY " +
	    "JOB_BLOCK_NAME_PROPERTY " +
	    "JOB_BLOCK_NEED_SIGN_ON_PERFORM_PROPERTY " +
	    "JOB_BLOCK_PERFORMER_PROPERTY " +
	    "JOB_BLOCK_RELATIVE_ABORT_DEADLINE_TYPE_PROPERTY " +
	    "JOB_BLOCK_RELATIVE_DEADLINE_TYPE_PROPERTY " +
	    "JOB_BLOCK_SUBJECT_PROPERTY ";

	  // Language code
	  const language_code_constants = "ENGLISH_LANGUAGE_CODE RUSSIAN_LANGUAGE_CODE ";

	  // Launching external applications
	  const launching_external_applications_constants =
	    "smHidden smMaximized smMinimized smNormal wmNo wmYes ";

	  // Link kind
	  const link_kind_constants =
	    "COMPONENT_TOKEN_LINK_KIND " +
	    "DOCUMENT_LINK_KIND " +
	    "EDOCUMENT_LINK_KIND " +
	    "FOLDER_LINK_KIND " +
	    "JOB_LINK_KIND " +
	    "REFERENCE_LINK_KIND " +
	    "TASK_LINK_KIND ";

	  // Lock type
	  const lock_type_constants =
	    "COMPONENT_TOKEN_LOCK_TYPE EDOCUMENT_VERSION_LOCK_TYPE ";

	  // Monitor block properties
	  const monitor_block_properties_constants =
	    "MONITOR_BLOCK_AFTER_FINISH_EVENT " +
	    "MONITOR_BLOCK_BEFORE_START_EVENT " +
	    "MONITOR_BLOCK_DEADLINE_PROPERTY " +
	    "MONITOR_BLOCK_INTERVAL_PROPERTY " +
	    "MONITOR_BLOCK_INTERVAL_TYPE_PROPERTY " +
	    "MONITOR_BLOCK_IS_RELATIVE_DEADLINE_PROPERTY " +
	    "MONITOR_BLOCK_NAME_PROPERTY " +
	    "MONITOR_BLOCK_RELATIVE_DEADLINE_TYPE_PROPERTY " +
	    "MONITOR_BLOCK_SEARCH_SCRIPT_PROPERTY ";

	  // Notice block properties
	  const notice_block_properties_constants =
	    "NOTICE_BLOCK_AFTER_FINISH_EVENT " +
	    "NOTICE_BLOCK_ATTACHMENT_PROPERTY " +
	    "NOTICE_BLOCK_ATTACHMENTS_RIGHTS_GROUP_PROPERTY " +
	    "NOTICE_BLOCK_ATTACHMENTS_RIGHTS_TYPE_PROPERTY " +
	    "NOTICE_BLOCK_BEFORE_START_EVENT " +
	    "NOTICE_BLOCK_CREATED_NOTICES_PROPERTY " +
	    "NOTICE_BLOCK_DEADLINE_PROPERTY " +
	    "NOTICE_BLOCK_IS_RELATIVE_DEADLINE_PROPERTY " +
	    "NOTICE_BLOCK_NAME_PROPERTY " +
	    "NOTICE_BLOCK_NOTICE_TEXT_PROPERTY " +
	    "NOTICE_BLOCK_PERFORMER_PROPERTY " +
	    "NOTICE_BLOCK_RELATIVE_DEADLINE_TYPE_PROPERTY " +
	    "NOTICE_BLOCK_SUBJECT_PROPERTY ";

	  // Object events
	  const object_events_constants =
	    "dseAfterCancel " +
	    "dseAfterClose " +
	    "dseAfterDelete " +
	    "dseAfterDeleteOutOfTransaction " +
	    "dseAfterInsert " +
	    "dseAfterOpen " +
	    "dseAfterScroll " +
	    "dseAfterUpdate " +
	    "dseAfterUpdateOutOfTransaction " +
	    "dseBeforeCancel " +
	    "dseBeforeClose " +
	    "dseBeforeDelete " +
	    "dseBeforeDetailUpdate " +
	    "dseBeforeInsert " +
	    "dseBeforeOpen " +
	    "dseBeforeUpdate " +
	    "dseOnAnyRequisiteChange " +
	    "dseOnCloseRecord " +
	    "dseOnDeleteError " +
	    "dseOnOpenRecord " +
	    "dseOnPrepareUpdate " +
	    "dseOnUpdateError " +
	    "dseOnUpdateRatifiedRecord " +
	    "dseOnValidDelete " +
	    "dseOnValidUpdate " +
	    "reOnChange " +
	    "reOnChangeValues " +
	    "SELECTION_BEGIN_ROUTE_EVENT " +
	    "SELECTION_END_ROUTE_EVENT ";

	  // Object params
	  const object_params_constants =
	    "CURRENT_PERIOD_IS_REQUIRED " +
	    "PREVIOUS_CARD_TYPE_NAME " +
	    "SHOW_RECORD_PROPERTIES_FORM ";

	  // Other
	  const other_constants =
	    "ACCESS_RIGHTS_SETTING_DIALOG_CODE " +
	    "ADMINISTRATOR_USER_CODE " +
	    "ANALYTIC_REPORT_TYPE " +
	    "asrtHideLocal " +
	    "asrtHideRemote " +
	    "CALCULATED_ROLE_TYPE_CODE " +
	    "COMPONENTS_REFERENCE_DEVELOPER_VIEW_CODE " +
	    "DCTS_TEST_PROTOCOLS_FOLDER_PATH " +
	    "E_EDOC_VERSION_ALREADY_APPROVINGLY_SIGNED " +
	    "E_EDOC_VERSION_ALREADY_APPROVINGLY_SIGNED_BY_USER " +
	    "E_EDOC_VERSION_ALREDY_SIGNED " +
	    "E_EDOC_VERSION_ALREDY_SIGNED_BY_USER " +
	    "EDOC_TYPES_CODE_REQUISITE_FIELD_NAME " +
	    "EDOCUMENTS_ALIAS_NAME " +
	    "FILES_FOLDER_PATH " +
	    "FILTER_OPERANDS_DELIMITER " +
	    "FILTER_OPERATIONS_DELIMITER " +
	    "FORMCARD_NAME " +
	    "FORMLIST_NAME " +
	    "GET_EXTENDED_DOCUMENT_EXTENSION_CREATION_MODE " +
	    "GET_EXTENDED_DOCUMENT_EXTENSION_IMPORT_MODE " +
	    "INTEGRATED_REPORT_TYPE " +
	    "IS_BUILDER_APPLICATION_ROLE " +
	    "IS_BUILDER_APPLICATION_ROLE2 " +
	    "IS_BUILDER_USERS " +
	    "ISBSYSDEV " +
	    "LOG_FOLDER_PATH " +
	    "mbCancel " +
	    "mbNo " +
	    "mbNoToAll " +
	    "mbOK " +
	    "mbYes " +
	    "mbYesToAll " +
	    "MEMORY_DATASET_DESRIPTIONS_FILENAME " +
	    "mrNo " +
	    "mrNoToAll " +
	    "mrYes " +
	    "mrYesToAll " +
	    "MULTIPLE_SELECT_DIALOG_CODE " +
	    "NONOPERATING_RECORD_FLAG_FEMININE " +
	    "NONOPERATING_RECORD_FLAG_MASCULINE " +
	    "OPERATING_RECORD_FLAG_FEMININE " +
	    "OPERATING_RECORD_FLAG_MASCULINE " +
	    "PROFILING_SETTINGS_COMMON_SETTINGS_CODE_VALUE " +
	    "PROGRAM_INITIATED_LOOKUP_ACTION " +
	    "ratDelete " +
	    "ratEdit " +
	    "ratInsert " +
	    "REPORT_TYPE " +
	    "REQUIRED_PICK_VALUES_VARIABLE " +
	    "rmCard " +
	    "rmList " +
	    "SBRTE_PROGID_DEV " +
	    "SBRTE_PROGID_RELEASE " +
	    "STATIC_ROLE_TYPE_CODE " +
	    "SUPPRESS_EMPTY_TEMPLATE_CREATION " +
	    "SYSTEM_USER_CODE " +
	    "UPDATE_DIALOG_DATASET " +
	    "USED_IN_OBJECT_HINT_PARAM " +
	    "USER_INITIATED_LOOKUP_ACTION " +
	    "USER_NAME_FORMAT " +
	    "USER_SELECTION_RESTRICTIONS " +
	    "WORKFLOW_TEST_PROTOCOLS_FOLDER_PATH " +
	    "ELS_SUBTYPE_CONTROL_NAME " +
	    "ELS_FOLDER_KIND_CONTROL_NAME " +
	    "REPEAT_PROCESS_CURRENT_OBJECT_EXCEPTION_NAME ";

	  // Privileges
	  const privileges_constants =
	    "PRIVILEGE_COMPONENT_FULL_ACCESS " +
	    "PRIVILEGE_DEVELOPMENT_EXPORT " +
	    "PRIVILEGE_DEVELOPMENT_IMPORT " +
	    "PRIVILEGE_DOCUMENT_DELETE " +
	    "PRIVILEGE_ESD " +
	    "PRIVILEGE_FOLDER_DELETE " +
	    "PRIVILEGE_MANAGE_ACCESS_RIGHTS " +
	    "PRIVILEGE_MANAGE_REPLICATION " +
	    "PRIVILEGE_MANAGE_SESSION_SERVER " +
	    "PRIVILEGE_OBJECT_FULL_ACCESS " +
	    "PRIVILEGE_OBJECT_VIEW " +
	    "PRIVILEGE_RESERVE_LICENSE " +
	    "PRIVILEGE_SYSTEM_CUSTOMIZE " +
	    "PRIVILEGE_SYSTEM_DEVELOP " +
	    "PRIVILEGE_SYSTEM_INSTALL " +
	    "PRIVILEGE_TASK_DELETE " +
	    "PRIVILEGE_USER_PLUGIN_SETTINGS_CUSTOMIZE " +
	    "PRIVILEGES_PSEUDOREFERENCE_CODE ";

	  // Pseudoreference code
	  const pseudoreference_code_constants =
	    "ACCESS_TYPES_PSEUDOREFERENCE_CODE " +
	    "ALL_AVAILABLE_COMPONENTS_PSEUDOREFERENCE_CODE " +
	    "ALL_AVAILABLE_PRIVILEGES_PSEUDOREFERENCE_CODE " +
	    "ALL_REPLICATE_COMPONENTS_PSEUDOREFERENCE_CODE " +
	    "AVAILABLE_DEVELOPERS_COMPONENTS_PSEUDOREFERENCE_CODE " +
	    "COMPONENTS_PSEUDOREFERENCE_CODE " +
	    "FILTRATER_SETTINGS_CONFLICTS_PSEUDOREFERENCE_CODE " +
	    "GROUPS_PSEUDOREFERENCE_CODE " +
	    "RECEIVE_PROTOCOL_PSEUDOREFERENCE_CODE " +
	    "REFERENCE_REQUISITE_PSEUDOREFERENCE_CODE " +
	    "REFERENCE_REQUISITES_PSEUDOREFERENCE_CODE " +
	    "REFTYPES_PSEUDOREFERENCE_CODE " +
	    "REPLICATION_SEANCES_DIARY_PSEUDOREFERENCE_CODE " +
	    "SEND_PROTOCOL_PSEUDOREFERENCE_CODE " +
	    "SUBSTITUTES_PSEUDOREFERENCE_CODE " +
	    "SYSTEM_SETTINGS_PSEUDOREFERENCE_CODE " +
	    "UNITS_PSEUDOREFERENCE_CODE " +
	    "USERS_PSEUDOREFERENCE_CODE " +
	    "VIEWERS_PSEUDOREFERENCE_CODE ";

	  // Requisite ISBCertificateType values
	  const requisite_ISBCertificateType_values_constants =
	    "CERTIFICATE_TYPE_ENCRYPT " +
	    "CERTIFICATE_TYPE_SIGN " +
	    "CERTIFICATE_TYPE_SIGN_AND_ENCRYPT ";

	  // Requisite ISBEDocStorageType values
	  const requisite_ISBEDocStorageType_values_constants =
	    "STORAGE_TYPE_FILE " +
	    "STORAGE_TYPE_NAS_CIFS " +
	    "STORAGE_TYPE_SAPERION " +
	    "STORAGE_TYPE_SQL_SERVER ";

	  // Requisite CompType2 values
	  const requisite_compType2_values_constants =
	    "COMPTYPE2_REQUISITE_DOCUMENTS_VALUE " +
	    "COMPTYPE2_REQUISITE_TASKS_VALUE " +
	    "COMPTYPE2_REQUISITE_FOLDERS_VALUE " +
	    "COMPTYPE2_REQUISITE_REFERENCES_VALUE ";

	  // Requisite name
	  const requisite_name_constants =
	    "SYSREQ_CODE " +
	    "SYSREQ_COMPTYPE2 " +
	    "SYSREQ_CONST_AVAILABLE_FOR_WEB " +
	    "SYSREQ_CONST_COMMON_CODE " +
	    "SYSREQ_CONST_COMMON_VALUE " +
	    "SYSREQ_CONST_FIRM_CODE " +
	    "SYSREQ_CONST_FIRM_STATUS " +
	    "SYSREQ_CONST_FIRM_VALUE " +
	    "SYSREQ_CONST_SERVER_STATUS " +
	    "SYSREQ_CONTENTS " +
	    "SYSREQ_DATE_OPEN " +
	    "SYSREQ_DATE_CLOSE " +
	    "SYSREQ_DESCRIPTION " +
	    "SYSREQ_DESCRIPTION_LOCALIZE_ID " +
	    "SYSREQ_DOUBLE " +
	    "SYSREQ_EDOC_ACCESS_TYPE " +
	    "SYSREQ_EDOC_AUTHOR " +
	    "SYSREQ_EDOC_CREATED " +
	    "SYSREQ_EDOC_DELEGATE_RIGHTS_REQUISITE_CODE " +
	    "SYSREQ_EDOC_EDITOR " +
	    "SYSREQ_EDOC_ENCODE_TYPE " +
	    "SYSREQ_EDOC_ENCRYPTION_PLUGIN_NAME " +
	    "SYSREQ_EDOC_ENCRYPTION_PLUGIN_VERSION " +
	    "SYSREQ_EDOC_EXPORT_DATE " +
	    "SYSREQ_EDOC_EXPORTER " +
	    "SYSREQ_EDOC_KIND " +
	    "SYSREQ_EDOC_LIFE_STAGE_NAME " +
	    "SYSREQ_EDOC_LOCKED_FOR_SERVER_CODE " +
	    "SYSREQ_EDOC_MODIFIED " +
	    "SYSREQ_EDOC_NAME " +
	    "SYSREQ_EDOC_NOTE " +
	    "SYSREQ_EDOC_QUALIFIED_ID " +
	    "SYSREQ_EDOC_SESSION_KEY " +
	    "SYSREQ_EDOC_SESSION_KEY_ENCRYPTION_PLUGIN_NAME " +
	    "SYSREQ_EDOC_SESSION_KEY_ENCRYPTION_PLUGIN_VERSION " +
	    "SYSREQ_EDOC_SIGNATURE_TYPE " +
	    "SYSREQ_EDOC_SIGNED " +
	    "SYSREQ_EDOC_STORAGE " +
	    "SYSREQ_EDOC_STORAGES_ARCHIVE_STORAGE " +
	    "SYSREQ_EDOC_STORAGES_CHECK_RIGHTS " +
	    "SYSREQ_EDOC_STORAGES_COMPUTER_NAME " +
	    "SYSREQ_EDOC_STORAGES_EDIT_IN_STORAGE " +
	    "SYSREQ_EDOC_STORAGES_EXECUTIVE_STORAGE " +
	    "SYSREQ_EDOC_STORAGES_FUNCTION " +
	    "SYSREQ_EDOC_STORAGES_INITIALIZED " +
	    "SYSREQ_EDOC_STORAGES_LOCAL_PATH " +
	    "SYSREQ_EDOC_STORAGES_SAPERION_DATABASE_NAME " +
	    "SYSREQ_EDOC_STORAGES_SEARCH_BY_TEXT " +
	    "SYSREQ_EDOC_STORAGES_SERVER_NAME " +
	    "SYSREQ_EDOC_STORAGES_SHARED_SOURCE_NAME " +
	    "SYSREQ_EDOC_STORAGES_TYPE " +
	    "SYSREQ_EDOC_TEXT_MODIFIED " +
	    "SYSREQ_EDOC_TYPE_ACT_CODE " +
	    "SYSREQ_EDOC_TYPE_ACT_DESCRIPTION " +
	    "SYSREQ_EDOC_TYPE_ACT_DESCRIPTION_LOCALIZE_ID " +
	    "SYSREQ_EDOC_TYPE_ACT_ON_EXECUTE " +
	    "SYSREQ_EDOC_TYPE_ACT_ON_EXECUTE_EXISTS " +
	    "SYSREQ_EDOC_TYPE_ACT_SECTION " +
	    "SYSREQ_EDOC_TYPE_ADD_PARAMS " +
	    "SYSREQ_EDOC_TYPE_COMMENT " +
	    "SYSREQ_EDOC_TYPE_EVENT_TEXT " +
	    "SYSREQ_EDOC_TYPE_NAME_IN_SINGULAR " +
	    "SYSREQ_EDOC_TYPE_NAME_IN_SINGULAR_LOCALIZE_ID " +
	    "SYSREQ_EDOC_TYPE_NAME_LOCALIZE_ID " +
	    "SYSREQ_EDOC_TYPE_NUMERATION_METHOD " +
	    "SYSREQ_EDOC_TYPE_PSEUDO_REQUISITE_CODE " +
	    "SYSREQ_EDOC_TYPE_REQ_CODE " +
	    "SYSREQ_EDOC_TYPE_REQ_DESCRIPTION " +
	    "SYSREQ_EDOC_TYPE_REQ_DESCRIPTION_LOCALIZE_ID " +
	    "SYSREQ_EDOC_TYPE_REQ_IS_LEADING " +
	    "SYSREQ_EDOC_TYPE_REQ_IS_REQUIRED " +
	    "SYSREQ_EDOC_TYPE_REQ_NUMBER " +
	    "SYSREQ_EDOC_TYPE_REQ_ON_CHANGE " +
	    "SYSREQ_EDOC_TYPE_REQ_ON_CHANGE_EXISTS " +
	    "SYSREQ_EDOC_TYPE_REQ_ON_SELECT " +
	    "SYSREQ_EDOC_TYPE_REQ_ON_SELECT_KIND " +
	    "SYSREQ_EDOC_TYPE_REQ_SECTION " +
	    "SYSREQ_EDOC_TYPE_VIEW_CARD " +
	    "SYSREQ_EDOC_TYPE_VIEW_CODE " +
	    "SYSREQ_EDOC_TYPE_VIEW_COMMENT " +
	    "SYSREQ_EDOC_TYPE_VIEW_IS_MAIN " +
	    "SYSREQ_EDOC_TYPE_VIEW_NAME " +
	    "SYSREQ_EDOC_TYPE_VIEW_NAME_LOCALIZE_ID " +
	    "SYSREQ_EDOC_VERSION_AUTHOR " +
	    "SYSREQ_EDOC_VERSION_CRC " +
	    "SYSREQ_EDOC_VERSION_DATA " +
	    "SYSREQ_EDOC_VERSION_EDITOR " +
	    "SYSREQ_EDOC_VERSION_EXPORT_DATE " +
	    "SYSREQ_EDOC_VERSION_EXPORTER " +
	    "SYSREQ_EDOC_VERSION_HIDDEN " +
	    "SYSREQ_EDOC_VERSION_LIFE_STAGE " +
	    "SYSREQ_EDOC_VERSION_MODIFIED " +
	    "SYSREQ_EDOC_VERSION_NOTE " +
	    "SYSREQ_EDOC_VERSION_SIGNATURE_TYPE " +
	    "SYSREQ_EDOC_VERSION_SIGNED " +
	    "SYSREQ_EDOC_VERSION_SIZE " +
	    "SYSREQ_EDOC_VERSION_SOURCE " +
	    "SYSREQ_EDOC_VERSION_TEXT_MODIFIED " +
	    "SYSREQ_EDOCKIND_DEFAULT_VERSION_STATE_CODE " +
	    "SYSREQ_FOLDER_KIND " +
	    "SYSREQ_FUNC_CATEGORY " +
	    "SYSREQ_FUNC_COMMENT " +
	    "SYSREQ_FUNC_GROUP " +
	    "SYSREQ_FUNC_GROUP_COMMENT " +
	    "SYSREQ_FUNC_GROUP_NUMBER " +
	    "SYSREQ_FUNC_HELP " +
	    "SYSREQ_FUNC_PARAM_DEF_VALUE " +
	    "SYSREQ_FUNC_PARAM_IDENT " +
	    "SYSREQ_FUNC_PARAM_NUMBER " +
	    "SYSREQ_FUNC_PARAM_TYPE " +
	    "SYSREQ_FUNC_TEXT " +
	    "SYSREQ_GROUP_CATEGORY " +
	    "SYSREQ_ID " +
	    "SYSREQ_LAST_UPDATE " +
	    "SYSREQ_LEADER_REFERENCE " +
	    "SYSREQ_LINE_NUMBER " +
	    "SYSREQ_MAIN_RECORD_ID " +
	    "SYSREQ_NAME " +
	    "SYSREQ_NAME_LOCALIZE_ID " +
	    "SYSREQ_NOTE " +
	    "SYSREQ_ORIGINAL_RECORD " +
	    "SYSREQ_OUR_FIRM " +
	    "SYSREQ_PROFILING_SETTINGS_BATCH_LOGING " +
	    "SYSREQ_PROFILING_SETTINGS_BATCH_SIZE " +
	    "SYSREQ_PROFILING_SETTINGS_PROFILING_ENABLED " +
	    "SYSREQ_PROFILING_SETTINGS_SQL_PROFILING_ENABLED " +
	    "SYSREQ_PROFILING_SETTINGS_START_LOGGED " +
	    "SYSREQ_RECORD_STATUS " +
	    "SYSREQ_REF_REQ_FIELD_NAME " +
	    "SYSREQ_REF_REQ_FORMAT " +
	    "SYSREQ_REF_REQ_GENERATED " +
	    "SYSREQ_REF_REQ_LENGTH " +
	    "SYSREQ_REF_REQ_PRECISION " +
	    "SYSREQ_REF_REQ_REFERENCE " +
	    "SYSREQ_REF_REQ_SECTION " +
	    "SYSREQ_REF_REQ_STORED " +
	    "SYSREQ_REF_REQ_TOKENS " +
	    "SYSREQ_REF_REQ_TYPE " +
	    "SYSREQ_REF_REQ_VIEW " +
	    "SYSREQ_REF_TYPE_ACT_CODE " +
	    "SYSREQ_REF_TYPE_ACT_DESCRIPTION " +
	    "SYSREQ_REF_TYPE_ACT_DESCRIPTION_LOCALIZE_ID " +
	    "SYSREQ_REF_TYPE_ACT_ON_EXECUTE " +
	    "SYSREQ_REF_TYPE_ACT_ON_EXECUTE_EXISTS " +
	    "SYSREQ_REF_TYPE_ACT_SECTION " +
	    "SYSREQ_REF_TYPE_ADD_PARAMS " +
	    "SYSREQ_REF_TYPE_COMMENT " +
	    "SYSREQ_REF_TYPE_COMMON_SETTINGS " +
	    "SYSREQ_REF_TYPE_DISPLAY_REQUISITE_NAME " +
	    "SYSREQ_REF_TYPE_EVENT_TEXT " +
	    "SYSREQ_REF_TYPE_MAIN_LEADING_REF " +
	    "SYSREQ_REF_TYPE_NAME_IN_SINGULAR " +
	    "SYSREQ_REF_TYPE_NAME_IN_SINGULAR_LOCALIZE_ID " +
	    "SYSREQ_REF_TYPE_NAME_LOCALIZE_ID " +
	    "SYSREQ_REF_TYPE_NUMERATION_METHOD " +
	    "SYSREQ_REF_TYPE_REQ_CODE " +
	    "SYSREQ_REF_TYPE_REQ_DESCRIPTION " +
	    "SYSREQ_REF_TYPE_REQ_DESCRIPTION_LOCALIZE_ID " +
	    "SYSREQ_REF_TYPE_REQ_IS_CONTROL " +
	    "SYSREQ_REF_TYPE_REQ_IS_FILTER " +
	    "SYSREQ_REF_TYPE_REQ_IS_LEADING " +
	    "SYSREQ_REF_TYPE_REQ_IS_REQUIRED " +
	    "SYSREQ_REF_TYPE_REQ_NUMBER " +
	    "SYSREQ_REF_TYPE_REQ_ON_CHANGE " +
	    "SYSREQ_REF_TYPE_REQ_ON_CHANGE_EXISTS " +
	    "SYSREQ_REF_TYPE_REQ_ON_SELECT " +
	    "SYSREQ_REF_TYPE_REQ_ON_SELECT_KIND " +
	    "SYSREQ_REF_TYPE_REQ_SECTION " +
	    "SYSREQ_REF_TYPE_VIEW_CARD " +
	    "SYSREQ_REF_TYPE_VIEW_CODE " +
	    "SYSREQ_REF_TYPE_VIEW_COMMENT " +
	    "SYSREQ_REF_TYPE_VIEW_IS_MAIN " +
	    "SYSREQ_REF_TYPE_VIEW_NAME " +
	    "SYSREQ_REF_TYPE_VIEW_NAME_LOCALIZE_ID " +
	    "SYSREQ_REFERENCE_TYPE_ID " +
	    "SYSREQ_STATE " +
	    "SYSREQ_STATЕ " +
	    "SYSREQ_SYSTEM_SETTINGS_VALUE " +
	    "SYSREQ_TYPE " +
	    "SYSREQ_UNIT " +
	    "SYSREQ_UNIT_ID " +
	    "SYSREQ_USER_GROUPS_GROUP_FULL_NAME " +
	    "SYSREQ_USER_GROUPS_GROUP_NAME " +
	    "SYSREQ_USER_GROUPS_GROUP_SERVER_NAME " +
	    "SYSREQ_USERS_ACCESS_RIGHTS " +
	    "SYSREQ_USERS_AUTHENTICATION " +
	    "SYSREQ_USERS_CATEGORY " +
	    "SYSREQ_USERS_COMPONENT " +
	    "SYSREQ_USERS_COMPONENT_USER_IS_PUBLIC " +
	    "SYSREQ_USERS_DOMAIN " +
	    "SYSREQ_USERS_FULL_USER_NAME " +
	    "SYSREQ_USERS_GROUP " +
	    "SYSREQ_USERS_IS_MAIN_SERVER " +
	    "SYSREQ_USERS_LOGIN " +
	    "SYSREQ_USERS_REFERENCE_USER_IS_PUBLIC " +
	    "SYSREQ_USERS_STATUS " +
	    "SYSREQ_USERS_USER_CERTIFICATE " +
	    "SYSREQ_USERS_USER_CERTIFICATE_INFO " +
	    "SYSREQ_USERS_USER_CERTIFICATE_PLUGIN_NAME " +
	    "SYSREQ_USERS_USER_CERTIFICATE_PLUGIN_VERSION " +
	    "SYSREQ_USERS_USER_CERTIFICATE_STATE " +
	    "SYSREQ_USERS_USER_CERTIFICATE_SUBJECT_NAME " +
	    "SYSREQ_USERS_USER_CERTIFICATE_THUMBPRINT " +
	    "SYSREQ_USERS_USER_DEFAULT_CERTIFICATE " +
	    "SYSREQ_USERS_USER_DESCRIPTION " +
	    "SYSREQ_USERS_USER_GLOBAL_NAME " +
	    "SYSREQ_USERS_USER_LOGIN " +
	    "SYSREQ_USERS_USER_MAIN_SERVER " +
	    "SYSREQ_USERS_USER_TYPE " +
	    "SYSREQ_WORK_RULES_FOLDER_ID ";

	  // Result
	  const result_constants = "RESULT_VAR_NAME RESULT_VAR_NAME_ENG ";

	  // Rule identification
	  const rule_identification_constants =
	    "AUTO_NUMERATION_RULE_ID " +
	    "CANT_CHANGE_ID_REQUISITE_RULE_ID " +
	    "CANT_CHANGE_OURFIRM_REQUISITE_RULE_ID " +
	    "CHECK_CHANGING_REFERENCE_RECORD_USE_RULE_ID " +
	    "CHECK_CODE_REQUISITE_RULE_ID " +
	    "CHECK_DELETING_REFERENCE_RECORD_USE_RULE_ID " +
	    "CHECK_FILTRATER_CHANGES_RULE_ID " +
	    "CHECK_RECORD_INTERVAL_RULE_ID " +
	    "CHECK_REFERENCE_INTERVAL_RULE_ID " +
	    "CHECK_REQUIRED_DATA_FULLNESS_RULE_ID " +
	    "CHECK_REQUIRED_REQUISITES_FULLNESS_RULE_ID " +
	    "MAKE_RECORD_UNRATIFIED_RULE_ID " +
	    "RESTORE_AUTO_NUMERATION_RULE_ID " +
	    "SET_FIRM_CONTEXT_FROM_RECORD_RULE_ID " +
	    "SET_FIRST_RECORD_IN_LIST_FORM_RULE_ID " +
	    "SET_IDSPS_VALUE_RULE_ID " +
	    "SET_NEXT_CODE_VALUE_RULE_ID " +
	    "SET_OURFIRM_BOUNDS_RULE_ID " +
	    "SET_OURFIRM_REQUISITE_RULE_ID ";

	  // Script block properties
	  const script_block_properties_constants =
	    "SCRIPT_BLOCK_AFTER_FINISH_EVENT " +
	    "SCRIPT_BLOCK_BEFORE_START_EVENT " +
	    "SCRIPT_BLOCK_EXECUTION_RESULTS_PROPERTY " +
	    "SCRIPT_BLOCK_NAME_PROPERTY " +
	    "SCRIPT_BLOCK_SCRIPT_PROPERTY ";

	  // Subtask block properties
	  const subtask_block_properties_constants =
	    "SUBTASK_BLOCK_ABORT_DEADLINE_PROPERTY " +
	    "SUBTASK_BLOCK_AFTER_FINISH_EVENT " +
	    "SUBTASK_BLOCK_ASSIGN_PARAMS_EVENT " +
	    "SUBTASK_BLOCK_ATTACHMENTS_PROPERTY " +
	    "SUBTASK_BLOCK_ATTACHMENTS_RIGHTS_GROUP_PROPERTY " +
	    "SUBTASK_BLOCK_ATTACHMENTS_RIGHTS_TYPE_PROPERTY " +
	    "SUBTASK_BLOCK_BEFORE_START_EVENT " +
	    "SUBTASK_BLOCK_CREATED_TASK_PROPERTY " +
	    "SUBTASK_BLOCK_CREATION_EVENT " +
	    "SUBTASK_BLOCK_DEADLINE_PROPERTY " +
	    "SUBTASK_BLOCK_IMPORTANCE_PROPERTY " +
	    "SUBTASK_BLOCK_INITIATOR_PROPERTY " +
	    "SUBTASK_BLOCK_IS_RELATIVE_ABORT_DEADLINE_PROPERTY " +
	    "SUBTASK_BLOCK_IS_RELATIVE_DEADLINE_PROPERTY " +
	    "SUBTASK_BLOCK_JOBS_TYPE_PROPERTY " +
	    "SUBTASK_BLOCK_NAME_PROPERTY " +
	    "SUBTASK_BLOCK_PARALLEL_ROUTE_PROPERTY " +
	    "SUBTASK_BLOCK_PERFORMERS_PROPERTY " +
	    "SUBTASK_BLOCK_RELATIVE_ABORT_DEADLINE_TYPE_PROPERTY " +
	    "SUBTASK_BLOCK_RELATIVE_DEADLINE_TYPE_PROPERTY " +
	    "SUBTASK_BLOCK_REQUIRE_SIGN_PROPERTY " +
	    "SUBTASK_BLOCK_STANDARD_ROUTE_PROPERTY " +
	    "SUBTASK_BLOCK_START_EVENT " +
	    "SUBTASK_BLOCK_STEP_CONTROL_PROPERTY " +
	    "SUBTASK_BLOCK_SUBJECT_PROPERTY " +
	    "SUBTASK_BLOCK_TASK_CONTROL_PROPERTY " +
	    "SUBTASK_BLOCK_TEXT_PROPERTY " +
	    "SUBTASK_BLOCK_UNLOCK_ATTACHMENTS_ON_STOP_PROPERTY " +
	    "SUBTASK_BLOCK_USE_STANDARD_ROUTE_PROPERTY " +
	    "SUBTASK_BLOCK_WAIT_FOR_TASK_COMPLETE_PROPERTY ";

	  // System component
	  const system_component_constants =
	    "SYSCOMP_CONTROL_JOBS " +
	    "SYSCOMP_FOLDERS " +
	    "SYSCOMP_JOBS " +
	    "SYSCOMP_NOTICES " +
	    "SYSCOMP_TASKS ";

	  // System dialogs
	  const system_dialogs_constants =
	    "SYSDLG_CREATE_EDOCUMENT " +
	    "SYSDLG_CREATE_EDOCUMENT_VERSION " +
	    "SYSDLG_CURRENT_PERIOD " +
	    "SYSDLG_EDIT_FUNCTION_HELP " +
	    "SYSDLG_EDOCUMENT_KINDS_FOR_TEMPLATE " +
	    "SYSDLG_EXPORT_MULTIPLE_EDOCUMENTS " +
	    "SYSDLG_EXPORT_SINGLE_EDOCUMENT " +
	    "SYSDLG_IMPORT_EDOCUMENT " +
	    "SYSDLG_MULTIPLE_SELECT " +
	    "SYSDLG_SETUP_ACCESS_RIGHTS " +
	    "SYSDLG_SETUP_DEFAULT_RIGHTS " +
	    "SYSDLG_SETUP_FILTER_CONDITION " +
	    "SYSDLG_SETUP_SIGN_RIGHTS " +
	    "SYSDLG_SETUP_TASK_OBSERVERS " +
	    "SYSDLG_SETUP_TASK_ROUTE " +
	    "SYSDLG_SETUP_USERS_LIST " +
	    "SYSDLG_SIGN_EDOCUMENT " +
	    "SYSDLG_SIGN_MULTIPLE_EDOCUMENTS ";

	  // System reference names
	  const system_reference_names_constants =
	    "SYSREF_ACCESS_RIGHTS_TYPES " +
	    "SYSREF_ADMINISTRATION_HISTORY " +
	    "SYSREF_ALL_AVAILABLE_COMPONENTS " +
	    "SYSREF_ALL_AVAILABLE_PRIVILEGES " +
	    "SYSREF_ALL_REPLICATING_COMPONENTS " +
	    "SYSREF_AVAILABLE_DEVELOPERS_COMPONENTS " +
	    "SYSREF_CALENDAR_EVENTS " +
	    "SYSREF_COMPONENT_TOKEN_HISTORY " +
	    "SYSREF_COMPONENT_TOKENS " +
	    "SYSREF_COMPONENTS " +
	    "SYSREF_CONSTANTS " +
	    "SYSREF_DATA_RECEIVE_PROTOCOL " +
	    "SYSREF_DATA_SEND_PROTOCOL " +
	    "SYSREF_DIALOGS " +
	    "SYSREF_DIALOGS_REQUISITES " +
	    "SYSREF_EDITORS " +
	    "SYSREF_EDOC_CARDS " +
	    "SYSREF_EDOC_TYPES " +
	    "SYSREF_EDOCUMENT_CARD_REQUISITES " +
	    "SYSREF_EDOCUMENT_CARD_TYPES " +
	    "SYSREF_EDOCUMENT_CARD_TYPES_REFERENCE " +
	    "SYSREF_EDOCUMENT_CARDS " +
	    "SYSREF_EDOCUMENT_HISTORY " +
	    "SYSREF_EDOCUMENT_KINDS " +
	    "SYSREF_EDOCUMENT_REQUISITES " +
	    "SYSREF_EDOCUMENT_SIGNATURES " +
	    "SYSREF_EDOCUMENT_TEMPLATES " +
	    "SYSREF_EDOCUMENT_TEXT_STORAGES " +
	    "SYSREF_EDOCUMENT_VIEWS " +
	    "SYSREF_FILTERER_SETUP_CONFLICTS " +
	    "SYSREF_FILTRATER_SETTING_CONFLICTS " +
	    "SYSREF_FOLDER_HISTORY " +
	    "SYSREF_FOLDERS " +
	    "SYSREF_FUNCTION_GROUPS " +
	    "SYSREF_FUNCTION_PARAMS " +
	    "SYSREF_FUNCTIONS " +
	    "SYSREF_JOB_HISTORY " +
	    "SYSREF_LINKS " +
	    "SYSREF_LOCALIZATION_DICTIONARY " +
	    "SYSREF_LOCALIZATION_LANGUAGES " +
	    "SYSREF_MODULES " +
	    "SYSREF_PRIVILEGES " +
	    "SYSREF_RECORD_HISTORY " +
	    "SYSREF_REFERENCE_REQUISITES " +
	    "SYSREF_REFERENCE_TYPE_VIEWS " +
	    "SYSREF_REFERENCE_TYPES " +
	    "SYSREF_REFERENCES " +
	    "SYSREF_REFERENCES_REQUISITES " +
	    "SYSREF_REMOTE_SERVERS " +
	    "SYSREF_REPLICATION_SESSIONS_LOG " +
	    "SYSREF_REPLICATION_SESSIONS_PROTOCOL " +
	    "SYSREF_REPORTS " +
	    "SYSREF_ROLES " +
	    "SYSREF_ROUTE_BLOCK_GROUPS " +
	    "SYSREF_ROUTE_BLOCKS " +
	    "SYSREF_SCRIPTS " +
	    "SYSREF_SEARCHES " +
	    "SYSREF_SERVER_EVENTS " +
	    "SYSREF_SERVER_EVENTS_HISTORY " +
	    "SYSREF_STANDARD_ROUTE_GROUPS " +
	    "SYSREF_STANDARD_ROUTES " +
	    "SYSREF_STATUSES " +
	    "SYSREF_SYSTEM_SETTINGS " +
	    "SYSREF_TASK_HISTORY " +
	    "SYSREF_TASK_KIND_GROUPS " +
	    "SYSREF_TASK_KINDS " +
	    "SYSREF_TASK_RIGHTS " +
	    "SYSREF_TASK_SIGNATURES " +
	    "SYSREF_TASKS " +
	    "SYSREF_UNITS " +
	    "SYSREF_USER_GROUPS " +
	    "SYSREF_USER_GROUPS_REFERENCE " +
	    "SYSREF_USER_SUBSTITUTION " +
	    "SYSREF_USERS " +
	    "SYSREF_USERS_REFERENCE " +
	    "SYSREF_VIEWERS " +
	    "SYSREF_WORKING_TIME_CALENDARS ";

	  // Table name
	  const table_name_constants =
	    "ACCESS_RIGHTS_TABLE_NAME " +
	    "EDMS_ACCESS_TABLE_NAME " +
	    "EDOC_TYPES_TABLE_NAME ";

	  // Test
	  const test_constants =
	    "TEST_DEV_DB_NAME " +
	    "TEST_DEV_SYSTEM_CODE " +
	    "TEST_EDMS_DB_NAME " +
	    "TEST_EDMS_MAIN_CODE " +
	    "TEST_EDMS_MAIN_DB_NAME " +
	    "TEST_EDMS_SECOND_CODE " +
	    "TEST_EDMS_SECOND_DB_NAME " +
	    "TEST_EDMS_SYSTEM_CODE " +
	    "TEST_ISB5_MAIN_CODE " +
	    "TEST_ISB5_SECOND_CODE " +
	    "TEST_SQL_SERVER_2005_NAME " +
	    "TEST_SQL_SERVER_NAME ";

	  // Using the dialog windows
	  const using_the_dialog_windows_constants =
	    "ATTENTION_CAPTION " +
	    "cbsCommandLinks " +
	    "cbsDefault " +
	    "CONFIRMATION_CAPTION " +
	    "ERROR_CAPTION " +
	    "INFORMATION_CAPTION " +
	    "mrCancel " +
	    "mrOk ";

	  // Using the document
	  const using_the_document_constants =
	    "EDOC_VERSION_ACTIVE_STAGE_CODE " +
	    "EDOC_VERSION_DESIGN_STAGE_CODE " +
	    "EDOC_VERSION_OBSOLETE_STAGE_CODE ";

	  // Using the EA and encryption
	  const using_the_EA_and_encryption_constants =
	    "cpDataEnciphermentEnabled " +
	    "cpDigitalSignatureEnabled " +
	    "cpID " +
	    "cpIssuer " +
	    "cpPluginVersion " +
	    "cpSerial " +
	    "cpSubjectName " +
	    "cpSubjSimpleName " +
	    "cpValidFromDate " +
	    "cpValidToDate ";

	  // Using the ISBL-editor
	  const using_the_ISBL_editor_constants =
	    "ISBL_SYNTAX " + "NO_SYNTAX " + "XML_SYNTAX ";

	  // Wait block properties
	  const wait_block_properties_constants =
	    "WAIT_BLOCK_AFTER_FINISH_EVENT " +
	    "WAIT_BLOCK_BEFORE_START_EVENT " +
	    "WAIT_BLOCK_DEADLINE_PROPERTY " +
	    "WAIT_BLOCK_IS_RELATIVE_DEADLINE_PROPERTY " +
	    "WAIT_BLOCK_NAME_PROPERTY " +
	    "WAIT_BLOCK_RELATIVE_DEADLINE_TYPE_PROPERTY ";

	  // SYSRES Common
	  const sysres_common_constants =
	    "SYSRES_COMMON " +
	    "SYSRES_CONST " +
	    "SYSRES_MBFUNC " +
	    "SYSRES_SBDATA " +
	    "SYSRES_SBGUI " +
	    "SYSRES_SBINTF " +
	    "SYSRES_SBREFDSC " +
	    "SYSRES_SQLERRORS " +
	    "SYSRES_SYSCOMP ";

	  // Константы ==> built_in
	  const CONSTANTS =
	    sysres_constants +
	    base_constants +
	    base_group_name_constants +
	    decision_block_properties_constants +
	    file_extension_constants +
	    job_block_properties_constants +
	    language_code_constants +
	    launching_external_applications_constants +
	    link_kind_constants +
	    lock_type_constants +
	    monitor_block_properties_constants +
	    notice_block_properties_constants +
	    object_events_constants +
	    object_params_constants +
	    other_constants +
	    privileges_constants +
	    pseudoreference_code_constants +
	    requisite_ISBCertificateType_values_constants +
	    requisite_ISBEDocStorageType_values_constants +
	    requisite_compType2_values_constants +
	    requisite_name_constants +
	    result_constants +
	    rule_identification_constants +
	    script_block_properties_constants +
	    subtask_block_properties_constants +
	    system_component_constants +
	    system_dialogs_constants +
	    system_reference_names_constants +
	    table_name_constants +
	    test_constants +
	    using_the_dialog_windows_constants +
	    using_the_document_constants +
	    using_the_EA_and_encryption_constants +
	    using_the_ISBL_editor_constants +
	    wait_block_properties_constants +
	    sysres_common_constants;

	  // enum TAccountType
	  const TAccountType = "atUser atGroup atRole ";

	  // enum TActionEnabledMode
	  const TActionEnabledMode =
	    "aemEnabledAlways " +
	    "aemDisabledAlways " +
	    "aemEnabledOnBrowse " +
	    "aemEnabledOnEdit " +
	    "aemDisabledOnBrowseEmpty ";

	  // enum TAddPosition
	  const TAddPosition = "apBegin apEnd ";

	  // enum TAlignment
	  const TAlignment = "alLeft alRight ";

	  // enum TAreaShowMode
	  const TAreaShowMode =
	    "asmNever " +
	    "asmNoButCustomize " +
	    "asmAsLastTime " +
	    "asmYesButCustomize " +
	    "asmAlways ";

	  // enum TCertificateInvalidationReason
	  const TCertificateInvalidationReason = "cirCommon cirRevoked ";

	  // enum TCertificateType
	  const TCertificateType = "ctSignature ctEncode ctSignatureEncode ";

	  // enum TCheckListBoxItemState
	  const TCheckListBoxItemState = "clbUnchecked clbChecked clbGrayed ";

	  // enum TCloseOnEsc
	  const TCloseOnEsc = "ceISB ceAlways ceNever ";

	  // enum TCompType
	  const TCompType =
	    "ctDocument " +
	    "ctReference " +
	    "ctScript " +
	    "ctUnknown " +
	    "ctReport " +
	    "ctDialog " +
	    "ctFunction " +
	    "ctFolder " +
	    "ctEDocument " +
	    "ctTask " +
	    "ctJob " +
	    "ctNotice " +
	    "ctControlJob ";

	  // enum TConditionFormat
	  const TConditionFormat = "cfInternal cfDisplay ";

	  // enum TConnectionIntent
	  const TConnectionIntent = "ciUnspecified ciWrite ciRead ";

	  // enum TContentKind
	  const TContentKind =
	    "ckFolder " +
	    "ckEDocument " +
	    "ckTask " +
	    "ckJob " +
	    "ckComponentToken " +
	    "ckAny " +
	    "ckReference " +
	    "ckScript " +
	    "ckReport " +
	    "ckDialog ";

	  // enum TControlType
	  const TControlType =
	    "ctISBLEditor " +
	    "ctBevel " +
	    "ctButton " +
	    "ctCheckListBox " +
	    "ctComboBox " +
	    "ctComboEdit " +
	    "ctGrid " +
	    "ctDBCheckBox " +
	    "ctDBComboBox " +
	    "ctDBEdit " +
	    "ctDBEllipsis " +
	    "ctDBMemo " +
	    "ctDBNavigator " +
	    "ctDBRadioGroup " +
	    "ctDBStatusLabel " +
	    "ctEdit " +
	    "ctGroupBox " +
	    "ctInplaceHint " +
	    "ctMemo " +
	    "ctPanel " +
	    "ctListBox " +
	    "ctRadioButton " +
	    "ctRichEdit " +
	    "ctTabSheet " +
	    "ctWebBrowser " +
	    "ctImage " +
	    "ctHyperLink " +
	    "ctLabel " +
	    "ctDBMultiEllipsis " +
	    "ctRibbon " +
	    "ctRichView " +
	    "ctInnerPanel " +
	    "ctPanelGroup " +
	    "ctBitButton ";

	  // enum TCriterionContentType
	  const TCriterionContentType =
	    "cctDate " +
	    "cctInteger " +
	    "cctNumeric " +
	    "cctPick " +
	    "cctReference " +
	    "cctString " +
	    "cctText ";

	  // enum TCultureType
	  const TCultureType = "cltInternal cltPrimary cltGUI ";

	  // enum TDataSetEventType
	  const TDataSetEventType =
	    "dseBeforeOpen " +
	    "dseAfterOpen " +
	    "dseBeforeClose " +
	    "dseAfterClose " +
	    "dseOnValidDelete " +
	    "dseBeforeDelete " +
	    "dseAfterDelete " +
	    "dseAfterDeleteOutOfTransaction " +
	    "dseOnDeleteError " +
	    "dseBeforeInsert " +
	    "dseAfterInsert " +
	    "dseOnValidUpdate " +
	    "dseBeforeUpdate " +
	    "dseOnUpdateRatifiedRecord " +
	    "dseAfterUpdate " +
	    "dseAfterUpdateOutOfTransaction " +
	    "dseOnUpdateError " +
	    "dseAfterScroll " +
	    "dseOnOpenRecord " +
	    "dseOnCloseRecord " +
	    "dseBeforeCancel " +
	    "dseAfterCancel " +
	    "dseOnUpdateDeadlockError " +
	    "dseBeforeDetailUpdate " +
	    "dseOnPrepareUpdate " +
	    "dseOnAnyRequisiteChange ";

	  // enum TDataSetState
	  const TDataSetState = "dssEdit dssInsert dssBrowse dssInActive ";

	  // enum TDateFormatType
	  const TDateFormatType = "dftDate dftShortDate dftDateTime dftTimeStamp ";

	  // enum TDateOffsetType
	  const TDateOffsetType = "dotDays dotHours dotMinutes dotSeconds ";

	  // enum TDateTimeKind
	  const TDateTimeKind = "dtkndLocal dtkndUTC ";

	  // enum TDeaAccessRights
	  const TDeaAccessRights = "arNone arView arEdit arFull ";

	  // enum TDocumentDefaultAction
	  const TDocumentDefaultAction = "ddaView ddaEdit ";

	  // enum TEditMode
	  const TEditMode =
	    "emLock " +
	    "emEdit " +
	    "emSign " +
	    "emExportWithLock " +
	    "emImportWithUnlock " +
	    "emChangeVersionNote " +
	    "emOpenForModify " +
	    "emChangeLifeStage " +
	    "emDelete " +
	    "emCreateVersion " +
	    "emImport " +
	    "emUnlockExportedWithLock " +
	    "emStart " +
	    "emAbort " +
	    "emReInit " +
	    "emMarkAsReaded " +
	    "emMarkAsUnreaded " +
	    "emPerform " +
	    "emAccept " +
	    "emResume " +
	    "emChangeRights " +
	    "emEditRoute " +
	    "emEditObserver " +
	    "emRecoveryFromLocalCopy " +
	    "emChangeWorkAccessType " +
	    "emChangeEncodeTypeToCertificate " +
	    "emChangeEncodeTypeToPassword " +
	    "emChangeEncodeTypeToNone " +
	    "emChangeEncodeTypeToCertificatePassword " +
	    "emChangeStandardRoute " +
	    "emGetText " +
	    "emOpenForView " +
	    "emMoveToStorage " +
	    "emCreateObject " +
	    "emChangeVersionHidden " +
	    "emDeleteVersion " +
	    "emChangeLifeCycleStage " +
	    "emApprovingSign " +
	    "emExport " +
	    "emContinue " +
	    "emLockFromEdit " +
	    "emUnLockForEdit " +
	    "emLockForServer " +
	    "emUnlockFromServer " +
	    "emDelegateAccessRights " +
	    "emReEncode ";

	  // enum TEditorCloseObservType
	  const TEditorCloseObservType = "ecotFile ecotProcess ";

	  // enum TEdmsApplicationAction
	  const TEdmsApplicationAction = "eaGet eaCopy eaCreate eaCreateStandardRoute ";

	  // enum TEDocumentLockType
	  const TEDocumentLockType = "edltAll edltNothing edltQuery ";

	  // enum TEDocumentStepShowMode
	  const TEDocumentStepShowMode = "essmText essmCard ";

	  // enum TEDocumentStepVersionType
	  const TEDocumentStepVersionType = "esvtLast esvtLastActive esvtSpecified ";

	  // enum TEDocumentStorageFunction
	  const TEDocumentStorageFunction = "edsfExecutive edsfArchive ";

	  // enum TEDocumentStorageType
	  const TEDocumentStorageType = "edstSQLServer edstFile ";

	  // enum TEDocumentVersionSourceType
	  const TEDocumentVersionSourceType =
	    "edvstNone edvstEDocumentVersionCopy edvstFile edvstTemplate edvstScannedFile ";

	  // enum TEDocumentVersionState
	  const TEDocumentVersionState = "vsDefault vsDesign vsActive vsObsolete ";

	  // enum TEncodeType
	  const TEncodeType = "etNone etCertificate etPassword etCertificatePassword ";

	  // enum TExceptionCategory
	  const TExceptionCategory = "ecException ecWarning ecInformation ";

	  // enum TExportedSignaturesType
	  const TExportedSignaturesType = "estAll estApprovingOnly ";

	  // enum TExportedVersionType
	  const TExportedVersionType = "evtLast evtLastActive evtQuery ";

	  // enum TFieldDataType
	  const TFieldDataType =
	    "fdtString " +
	    "fdtNumeric " +
	    "fdtInteger " +
	    "fdtDate " +
	    "fdtText " +
	    "fdtUnknown " +
	    "fdtWideString " +
	    "fdtLargeInteger ";

	  // enum TFolderType
	  const TFolderType =
	    "ftInbox " +
	    "ftOutbox " +
	    "ftFavorites " +
	    "ftCommonFolder " +
	    "ftUserFolder " +
	    "ftComponents " +
	    "ftQuickLaunch " +
	    "ftShortcuts " +
	    "ftSearch ";

	  // enum TGridRowHeight
	  const TGridRowHeight = "grhAuto " + "grhX1 " + "grhX2 " + "grhX3 ";

	  // enum THyperlinkType
	  const THyperlinkType = "hltText " + "hltRTF " + "hltHTML ";

	  // enum TImageFileFormat
	  const TImageFileFormat =
	    "iffBMP " +
	    "iffJPEG " +
	    "iffMultiPageTIFF " +
	    "iffSinglePageTIFF " +
	    "iffTIFF " +
	    "iffPNG ";

	  // enum TImageMode
	  const TImageMode = "im8bGrayscale " + "im24bRGB " + "im1bMonochrome ";

	  // enum TImageType
	  const TImageType = "itBMP " + "itJPEG " + "itWMF " + "itPNG ";

	  // enum TInplaceHintKind
	  const TInplaceHintKind =
	    "ikhInformation " + "ikhWarning " + "ikhError " + "ikhNoIcon ";

	  // enum TISBLContext
	  const TISBLContext =
	    "icUnknown " +
	    "icScript " +
	    "icFunction " +
	    "icIntegratedReport " +
	    "icAnalyticReport " +
	    "icDataSetEventHandler " +
	    "icActionHandler " +
	    "icFormEventHandler " +
	    "icLookUpEventHandler " +
	    "icRequisiteChangeEventHandler " +
	    "icBeforeSearchEventHandler " +
	    "icRoleCalculation " +
	    "icSelectRouteEventHandler " +
	    "icBlockPropertyCalculation " +
	    "icBlockQueryParamsEventHandler " +
	    "icChangeSearchResultEventHandler " +
	    "icBlockEventHandler " +
	    "icSubTaskInitEventHandler " +
	    "icEDocDataSetEventHandler " +
	    "icEDocLookUpEventHandler " +
	    "icEDocActionHandler " +
	    "icEDocFormEventHandler " +
	    "icEDocRequisiteChangeEventHandler " +
	    "icStructuredConversionRule " +
	    "icStructuredConversionEventBefore " +
	    "icStructuredConversionEventAfter " +
	    "icWizardEventHandler " +
	    "icWizardFinishEventHandler " +
	    "icWizardStepEventHandler " +
	    "icWizardStepFinishEventHandler " +
	    "icWizardActionEnableEventHandler " +
	    "icWizardActionExecuteEventHandler " +
	    "icCreateJobsHandler " +
	    "icCreateNoticesHandler " +
	    "icBeforeLookUpEventHandler " +
	    "icAfterLookUpEventHandler " +
	    "icTaskAbortEventHandler " +
	    "icWorkflowBlockActionHandler " +
	    "icDialogDataSetEventHandler " +
	    "icDialogActionHandler " +
	    "icDialogLookUpEventHandler " +
	    "icDialogRequisiteChangeEventHandler " +
	    "icDialogFormEventHandler " +
	    "icDialogValidCloseEventHandler " +
	    "icBlockFormEventHandler " +
	    "icTaskFormEventHandler " +
	    "icReferenceMethod " +
	    "icEDocMethod " +
	    "icDialogMethod " +
	    "icProcessMessageHandler ";

	  // enum TItemShow
	  const TItemShow = "isShow " + "isHide " + "isByUserSettings ";

	  // enum TJobKind
	  const TJobKind = "jkJob " + "jkNotice " + "jkControlJob ";

	  // enum TJoinType
	  const TJoinType = "jtInner " + "jtLeft " + "jtRight " + "jtFull " + "jtCross ";

	  // enum TLabelPos
	  const TLabelPos = "lbpAbove " + "lbpBelow " + "lbpLeft " + "lbpRight ";

	  // enum TLicensingType
	  const TLicensingType = "eltPerConnection " + "eltPerUser ";

	  // enum TLifeCycleStageFontColor
	  const TLifeCycleStageFontColor =
	    "sfcUndefined " +
	    "sfcBlack " +
	    "sfcGreen " +
	    "sfcRed " +
	    "sfcBlue " +
	    "sfcOrange " +
	    "sfcLilac ";

	  // enum TLifeCycleStageFontStyle
	  const TLifeCycleStageFontStyle = "sfsItalic " + "sfsStrikeout " + "sfsNormal ";

	  // enum TLockableDevelopmentComponentType
	  const TLockableDevelopmentComponentType =
	    "ldctStandardRoute " +
	    "ldctWizard " +
	    "ldctScript " +
	    "ldctFunction " +
	    "ldctRouteBlock " +
	    "ldctIntegratedReport " +
	    "ldctAnalyticReport " +
	    "ldctReferenceType " +
	    "ldctEDocumentType " +
	    "ldctDialog " +
	    "ldctServerEvents ";

	  // enum TMaxRecordCountRestrictionType
	  const TMaxRecordCountRestrictionType =
	    "mrcrtNone " + "mrcrtUser " + "mrcrtMaximal " + "mrcrtCustom ";

	  // enum TRangeValueType
	  const TRangeValueType =
	    "vtEqual " + "vtGreaterOrEqual " + "vtLessOrEqual " + "vtRange ";

	  // enum TRelativeDate
	  const TRelativeDate =
	    "rdYesterday " +
	    "rdToday " +
	    "rdTomorrow " +
	    "rdThisWeek " +
	    "rdThisMonth " +
	    "rdThisYear " +
	    "rdNextMonth " +
	    "rdNextWeek " +
	    "rdLastWeek " +
	    "rdLastMonth ";

	  // enum TReportDestination
	  const TReportDestination = "rdWindow " + "rdFile " + "rdPrinter ";

	  // enum TReqDataType
	  const TReqDataType =
	    "rdtString " +
	    "rdtNumeric " +
	    "rdtInteger " +
	    "rdtDate " +
	    "rdtReference " +
	    "rdtAccount " +
	    "rdtText " +
	    "rdtPick " +
	    "rdtUnknown " +
	    "rdtLargeInteger " +
	    "rdtDocument ";

	  // enum TRequisiteEventType
	  const TRequisiteEventType = "reOnChange " + "reOnChangeValues ";

	  // enum TSBTimeType
	  const TSBTimeType = "ttGlobal " + "ttLocal " + "ttUser " + "ttSystem ";

	  // enum TSearchShowMode
	  const TSearchShowMode =
	    "ssmBrowse " + "ssmSelect " + "ssmMultiSelect " + "ssmBrowseModal ";

	  // enum TSelectMode
	  const TSelectMode = "smSelect " + "smLike " + "smCard ";

	  // enum TSignatureType
	  const TSignatureType = "stNone " + "stAuthenticating " + "stApproving ";

	  // enum TSignerContentType
	  const TSignerContentType = "sctString " + "sctStream ";

	  // enum TStringsSortType
	  const TStringsSortType = "sstAnsiSort " + "sstNaturalSort ";

	  // enum TStringValueType
	  const TStringValueType = "svtEqual " + "svtContain ";

	  // enum TStructuredObjectAttributeType
	  const TStructuredObjectAttributeType =
	    "soatString " +
	    "soatNumeric " +
	    "soatInteger " +
	    "soatDatetime " +
	    "soatReferenceRecord " +
	    "soatText " +
	    "soatPick " +
	    "soatBoolean " +
	    "soatEDocument " +
	    "soatAccount " +
	    "soatIntegerCollection " +
	    "soatNumericCollection " +
	    "soatStringCollection " +
	    "soatPickCollection " +
	    "soatDatetimeCollection " +
	    "soatBooleanCollection " +
	    "soatReferenceRecordCollection " +
	    "soatEDocumentCollection " +
	    "soatAccountCollection " +
	    "soatContents " +
	    "soatUnknown ";

	  // enum TTaskAbortReason
	  const TTaskAbortReason = "tarAbortByUser " + "tarAbortByWorkflowException ";

	  // enum TTextValueType
	  const TTextValueType = "tvtAllWords " + "tvtExactPhrase " + "tvtAnyWord ";

	  // enum TUserObjectStatus
	  const TUserObjectStatus =
	    "usNone " +
	    "usCompleted " +
	    "usRedSquare " +
	    "usBlueSquare " +
	    "usYellowSquare " +
	    "usGreenSquare " +
	    "usOrangeSquare " +
	    "usPurpleSquare " +
	    "usFollowUp ";

	  // enum TUserType
	  const TUserType =
	    "utUnknown " +
	    "utUser " +
	    "utDeveloper " +
	    "utAdministrator " +
	    "utSystemDeveloper " +
	    "utDisconnected ";

	  // enum TValuesBuildType
	  const TValuesBuildType =
	    "btAnd " + "btDetailAnd " + "btOr " + "btNotOr " + "btOnly ";

	  // enum TViewMode
	  const TViewMode = "vmView " + "vmSelect " + "vmNavigation ";

	  // enum TViewSelectionMode
	  const TViewSelectionMode =
	    "vsmSingle " + "vsmMultiple " + "vsmMultipleCheck " + "vsmNoSelection ";

	  // enum TWizardActionType
	  const TWizardActionType =
	    "wfatPrevious " + "wfatNext " + "wfatCancel " + "wfatFinish ";

	  // enum TWizardFormElementProperty
	  const TWizardFormElementProperty =
	    "wfepUndefined " +
	    "wfepText3 " +
	    "wfepText6 " +
	    "wfepText9 " +
	    "wfepSpinEdit " +
	    "wfepDropDown " +
	    "wfepRadioGroup " +
	    "wfepFlag " +
	    "wfepText12 " +
	    "wfepText15 " +
	    "wfepText18 " +
	    "wfepText21 " +
	    "wfepText24 " +
	    "wfepText27 " +
	    "wfepText30 " +
	    "wfepRadioGroupColumn1 " +
	    "wfepRadioGroupColumn2 " +
	    "wfepRadioGroupColumn3 ";

	  // enum TWizardFormElementType
	  const TWizardFormElementType =
	    "wfetQueryParameter " + "wfetText " + "wfetDelimiter " + "wfetLabel ";

	  // enum TWizardParamType
	  const TWizardParamType =
	    "wptString " +
	    "wptInteger " +
	    "wptNumeric " +
	    "wptBoolean " +
	    "wptDateTime " +
	    "wptPick " +
	    "wptText " +
	    "wptUser " +
	    "wptUserList " +
	    "wptEDocumentInfo " +
	    "wptEDocumentInfoList " +
	    "wptReferenceRecordInfo " +
	    "wptReferenceRecordInfoList " +
	    "wptFolderInfo " +
	    "wptTaskInfo " +
	    "wptContents " +
	    "wptFileName " +
	    "wptDate ";

	  // enum TWizardStepResult
	  const TWizardStepResult =
	    "wsrComplete " +
	    "wsrGoNext " +
	    "wsrGoPrevious " +
	    "wsrCustom " +
	    "wsrCancel " +
	    "wsrGoFinal ";

	  // enum TWizardStepType
	  const TWizardStepType =
	    "wstForm " +
	    "wstEDocument " +
	    "wstTaskCard " +
	    "wstReferenceRecordCard " +
	    "wstFinal ";

	  // enum TWorkAccessType
	  const TWorkAccessType = "waAll " + "waPerformers " + "waManual ";

	  // enum TWorkflowBlockType
	  const TWorkflowBlockType =
	    "wsbStart " +
	    "wsbFinish " +
	    "wsbNotice " +
	    "wsbStep " +
	    "wsbDecision " +
	    "wsbWait " +
	    "wsbMonitor " +
	    "wsbScript " +
	    "wsbConnector " +
	    "wsbSubTask " +
	    "wsbLifeCycleStage " +
	    "wsbPause ";

	  // enum TWorkflowDataType
	  const TWorkflowDataType =
	    "wdtInteger " +
	    "wdtFloat " +
	    "wdtString " +
	    "wdtPick " +
	    "wdtDateTime " +
	    "wdtBoolean " +
	    "wdtTask " +
	    "wdtJob " +
	    "wdtFolder " +
	    "wdtEDocument " +
	    "wdtReferenceRecord " +
	    "wdtUser " +
	    "wdtGroup " +
	    "wdtRole " +
	    "wdtIntegerCollection " +
	    "wdtFloatCollection " +
	    "wdtStringCollection " +
	    "wdtPickCollection " +
	    "wdtDateTimeCollection " +
	    "wdtBooleanCollection " +
	    "wdtTaskCollection " +
	    "wdtJobCollection " +
	    "wdtFolderCollection " +
	    "wdtEDocumentCollection " +
	    "wdtReferenceRecordCollection " +
	    "wdtUserCollection " +
	    "wdtGroupCollection " +
	    "wdtRoleCollection " +
	    "wdtContents " +
	    "wdtUserList " +
	    "wdtSearchDescription " +
	    "wdtDeadLine " +
	    "wdtPickSet " +
	    "wdtAccountCollection ";

	  // enum TWorkImportance
	  const TWorkImportance = "wiLow " + "wiNormal " + "wiHigh ";

	  // enum TWorkRouteType
	  const TWorkRouteType = "wrtSoft " + "wrtHard ";

	  // enum TWorkState
	  const TWorkState =
	    "wsInit " +
	    "wsRunning " +
	    "wsDone " +
	    "wsControlled " +
	    "wsAborted " +
	    "wsContinued ";

	  // enum TWorkTextBuildingMode
	  const TWorkTextBuildingMode =
	    "wtmFull " + "wtmFromCurrent " + "wtmOnlyCurrent ";

	  // Перечисления
	  const ENUMS =
	    TAccountType +
	    TActionEnabledMode +
	    TAddPosition +
	    TAlignment +
	    TAreaShowMode +
	    TCertificateInvalidationReason +
	    TCertificateType +
	    TCheckListBoxItemState +
	    TCloseOnEsc +
	    TCompType +
	    TConditionFormat +
	    TConnectionIntent +
	    TContentKind +
	    TControlType +
	    TCriterionContentType +
	    TCultureType +
	    TDataSetEventType +
	    TDataSetState +
	    TDateFormatType +
	    TDateOffsetType +
	    TDateTimeKind +
	    TDeaAccessRights +
	    TDocumentDefaultAction +
	    TEditMode +
	    TEditorCloseObservType +
	    TEdmsApplicationAction +
	    TEDocumentLockType +
	    TEDocumentStepShowMode +
	    TEDocumentStepVersionType +
	    TEDocumentStorageFunction +
	    TEDocumentStorageType +
	    TEDocumentVersionSourceType +
	    TEDocumentVersionState +
	    TEncodeType +
	    TExceptionCategory +
	    TExportedSignaturesType +
	    TExportedVersionType +
	    TFieldDataType +
	    TFolderType +
	    TGridRowHeight +
	    THyperlinkType +
	    TImageFileFormat +
	    TImageMode +
	    TImageType +
	    TInplaceHintKind +
	    TISBLContext +
	    TItemShow +
	    TJobKind +
	    TJoinType +
	    TLabelPos +
	    TLicensingType +
	    TLifeCycleStageFontColor +
	    TLifeCycleStageFontStyle +
	    TLockableDevelopmentComponentType +
	    TMaxRecordCountRestrictionType +
	    TRangeValueType +
	    TRelativeDate +
	    TReportDestination +
	    TReqDataType +
	    TRequisiteEventType +
	    TSBTimeType +
	    TSearchShowMode +
	    TSelectMode +
	    TSignatureType +
	    TSignerContentType +
	    TStringsSortType +
	    TStringValueType +
	    TStructuredObjectAttributeType +
	    TTaskAbortReason +
	    TTextValueType +
	    TUserObjectStatus +
	    TUserType +
	    TValuesBuildType +
	    TViewMode +
	    TViewSelectionMode +
	    TWizardActionType +
	    TWizardFormElementProperty +
	    TWizardFormElementType +
	    TWizardParamType +
	    TWizardStepResult +
	    TWizardStepType +
	    TWorkAccessType +
	    TWorkflowBlockType +
	    TWorkflowDataType +
	    TWorkImportance +
	    TWorkRouteType +
	    TWorkState +
	    TWorkTextBuildingMode;

	  // Системные функции ==> SYSFUNCTIONS
	  const system_functions =
	    "AddSubString " +
	    "AdjustLineBreaks " +
	    "AmountInWords " +
	    "Analysis " +
	    "ArrayDimCount " +
	    "ArrayHighBound " +
	    "ArrayLowBound " +
	    "ArrayOf " +
	    "ArrayReDim " +
	    "Assert " +
	    "Assigned " +
	    "BeginOfMonth " +
	    "BeginOfPeriod " +
	    "BuildProfilingOperationAnalysis " +
	    "CallProcedure " +
	    "CanReadFile " +
	    "CArrayElement " +
	    "CDataSetRequisite " +
	    "ChangeDate " +
	    "ChangeReferenceDataset " +
	    "Char " +
	    "CharPos " +
	    "CheckParam " +
	    "CheckParamValue " +
	    "CompareStrings " +
	    "ConstantExists " +
	    "ControlState " +
	    "ConvertDateStr " +
	    "Copy " +
	    "CopyFile " +
	    "CreateArray " +
	    "CreateCachedReference " +
	    "CreateConnection " +
	    "CreateDialog " +
	    "CreateDualListDialog " +
	    "CreateEditor " +
	    "CreateException " +
	    "CreateFile " +
	    "CreateFolderDialog " +
	    "CreateInputDialog " +
	    "CreateLinkFile " +
	    "CreateList " +
	    "CreateLock " +
	    "CreateMemoryDataSet " +
	    "CreateObject " +
	    "CreateOpenDialog " +
	    "CreateProgress " +
	    "CreateQuery " +
	    "CreateReference " +
	    "CreateReport " +
	    "CreateSaveDialog " +
	    "CreateScript " +
	    "CreateSQLPivotFunction " +
	    "CreateStringList " +
	    "CreateTreeListSelectDialog " +
	    "CSelectSQL " +
	    "CSQL " +
	    "CSubString " +
	    "CurrentUserID " +
	    "CurrentUserName " +
	    "CurrentVersion " +
	    "DataSetLocateEx " +
	    "DateDiff " +
	    "DateTimeDiff " +
	    "DateToStr " +
	    "DayOfWeek " +
	    "DeleteFile " +
	    "DirectoryExists " +
	    "DisableCheckAccessRights " +
	    "DisableCheckFullShowingRestriction " +
	    "DisableMassTaskSendingRestrictions " +
	    "DropTable " +
	    "DupeString " +
	    "EditText " +
	    "EnableCheckAccessRights " +
	    "EnableCheckFullShowingRestriction " +
	    "EnableMassTaskSendingRestrictions " +
	    "EndOfMonth " +
	    "EndOfPeriod " +
	    "ExceptionExists " +
	    "ExceptionsOff " +
	    "ExceptionsOn " +
	    "Execute " +
	    "ExecuteProcess " +
	    "Exit " +
	    "ExpandEnvironmentVariables " +
	    "ExtractFileDrive " +
	    "ExtractFileExt " +
	    "ExtractFileName " +
	    "ExtractFilePath " +
	    "ExtractParams " +
	    "FileExists " +
	    "FileSize " +
	    "FindFile " +
	    "FindSubString " +
	    "FirmContext " +
	    "ForceDirectories " +
	    "Format " +
	    "FormatDate " +
	    "FormatNumeric " +
	    "FormatSQLDate " +
	    "FormatString " +
	    "FreeException " +
	    "GetComponent " +
	    "GetComponentLaunchParam " +
	    "GetConstant " +
	    "GetLastException " +
	    "GetReferenceRecord " +
	    "GetRefTypeByRefID " +
	    "GetTableID " +
	    "GetTempFolder " +
	    "IfThen " +
	    "In " +
	    "IndexOf " +
	    "InputDialog " +
	    "InputDialogEx " +
	    "InteractiveMode " +
	    "IsFileLocked " +
	    "IsGraphicFile " +
	    "IsNumeric " +
	    "Length " +
	    "LoadString " +
	    "LoadStringFmt " +
	    "LocalTimeToUTC " +
	    "LowerCase " +
	    "Max " +
	    "MessageBox " +
	    "MessageBoxEx " +
	    "MimeDecodeBinary " +
	    "MimeDecodeString " +
	    "MimeEncodeBinary " +
	    "MimeEncodeString " +
	    "Min " +
	    "MoneyInWords " +
	    "MoveFile " +
	    "NewID " +
	    "Now " +
	    "OpenFile " +
	    "Ord " +
	    "Precision " +
	    "Raise " +
	    "ReadCertificateFromFile " +
	    "ReadFile " +
	    "ReferenceCodeByID " +
	    "ReferenceNumber " +
	    "ReferenceRequisiteMode " +
	    "ReferenceRequisiteValue " +
	    "RegionDateSettings " +
	    "RegionNumberSettings " +
	    "RegionTimeSettings " +
	    "RegRead " +
	    "RegWrite " +
	    "RenameFile " +
	    "Replace " +
	    "Round " +
	    "SelectServerCode " +
	    "SelectSQL " +
	    "ServerDateTime " +
	    "SetConstant " +
	    "SetManagedFolderFieldsState " +
	    "ShowConstantsInputDialog " +
	    "ShowMessage " +
	    "Sleep " +
	    "Split " +
	    "SQL " +
	    "SQL2XLSTAB " +
	    "SQLProfilingSendReport " +
	    "StrToDate " +
	    "SubString " +
	    "SubStringCount " +
	    "SystemSetting " +
	    "Time " +
	    "TimeDiff " +
	    "Today " +
	    "Transliterate " +
	    "Trim " +
	    "UpperCase " +
	    "UserStatus " +
	    "UTCToLocalTime " +
	    "ValidateXML " +
	    "VarIsClear " +
	    "VarIsEmpty " +
	    "VarIsNull " +
	    "WorkTimeDiff " +
	    "WriteFile " +
	    "WriteFileEx " +
	    "WriteObjectHistory " +
	    "Анализ " +
	    "БазаДанных " +
	    "БлокЕсть " +
	    "БлокЕстьРасш " +
	    "БлокИнфо " +
	    "БлокСнять " +
	    "БлокСнятьРасш " +
	    "БлокУстановить " +
	    "Ввод " +
	    "ВводМеню " +
	    "ВедС " +
	    "ВедСпр " +
	    "ВерхняяГраницаМассива " +
	    "ВнешПрогр " +
	    "Восст " +
	    "ВременнаяПапка " +
	    "Время " +
	    "ВыборSQL " +
	    "ВыбратьЗапись " +
	    "ВыделитьСтр " +
	    "Вызвать " +
	    "Выполнить " +
	    "ВыпПрогр " +
	    "ГрафическийФайл " +
	    "ГруппаДополнительно " +
	    "ДатаВремяСерв " +
	    "ДеньНедели " +
	    "ДиалогДаНет " +
	    "ДлинаСтр " +
	    "ДобПодстр " +
	    "ЕПусто " +
	    "ЕслиТо " +
	    "ЕЧисло " +
	    "ЗамПодстр " +
	    "ЗаписьСправочника " +
	    "ЗначПоляСпр " +
	    "ИДТипСпр " +
	    "ИзвлечьДиск " +
	    "ИзвлечьИмяФайла " +
	    "ИзвлечьПуть " +
	    "ИзвлечьРасширение " +
	    "ИзмДат " +
	    "ИзменитьРазмерМассива " +
	    "ИзмеренийМассива " +
	    "ИмяОрг " +
	    "ИмяПоляСпр " +
	    "Индекс " +
	    "ИндикаторЗакрыть " +
	    "ИндикаторОткрыть " +
	    "ИндикаторШаг " +
	    "ИнтерактивныйРежим " +
	    "ИтогТблСпр " +
	    "КодВидВедСпр " +
	    "КодВидСпрПоИД " +
	    "КодПоAnalit " +
	    "КодСимвола " +
	    "КодСпр " +
	    "КолПодстр " +
	    "КолПроп " +
	    "КонМес " +
	    "Конст " +
	    "КонстЕсть " +
	    "КонстЗнач " +
	    "КонТран " +
	    "КопироватьФайл " +
	    "КопияСтр " +
	    "КПериод " +
	    "КСтрТблСпр " +
	    "Макс " +
	    "МаксСтрТблСпр " +
	    "Массив " +
	    "Меню " +
	    "МенюРасш " +
	    "Мин " +
	    "НаборДанныхНайтиРасш " +
	    "НаимВидСпр " +
	    "НаимПоAnalit " +
	    "НаимСпр " +
	    "НастроитьПереводыСтрок " +
	    "НачМес " +
	    "НачТран " +
	    "НижняяГраницаМассива " +
	    "НомерСпр " +
	    "НПериод " +
	    "Окно " +
	    "Окр " +
	    "Окружение " +
	    "ОтлИнфДобавить " +
	    "ОтлИнфУдалить " +
	    "Отчет " +
	    "ОтчетАнал " +
	    "ОтчетИнт " +
	    "ПапкаСуществует " +
	    "Пауза " +
	    "ПВыборSQL " +
	    "ПереименоватьФайл " +
	    "Переменные " +
	    "ПереместитьФайл " +
	    "Подстр " +
	    "ПоискПодстр " +
	    "ПоискСтр " +
	    "ПолучитьИДТаблицы " +
	    "ПользовательДополнительно " +
	    "ПользовательИД " +
	    "ПользовательИмя " +
	    "ПользовательСтатус " +
	    "Прервать " +
	    "ПроверитьПараметр " +
	    "ПроверитьПараметрЗнач " +
	    "ПроверитьУсловие " +
	    "РазбСтр " +
	    "РазнВремя " +
	    "РазнДат " +
	    "РазнДатаВремя " +
	    "РазнРабВремя " +
	    "РегУстВрем " +
	    "РегУстДат " +
	    "РегУстЧсл " +
	    "РедТекст " +
	    "РеестрЗапись " +
	    "РеестрСписокИменПарам " +
	    "РеестрЧтение " +
	    "РеквСпр " +
	    "РеквСпрПр " +
	    "Сегодня " +
	    "Сейчас " +
	    "Сервер " +
	    "СерверПроцессИД " +
	    "СертификатФайлСчитать " +
	    "СжПроб " +
	    "Символ " +
	    "СистемаДиректумКод " +
	    "СистемаИнформация " +
	    "СистемаКод " +
	    "Содержит " +
	    "СоединениеЗакрыть " +
	    "СоединениеОткрыть " +
	    "СоздатьДиалог " +
	    "СоздатьДиалогВыбораИзДвухСписков " +
	    "СоздатьДиалогВыбораПапки " +
	    "СоздатьДиалогОткрытияФайла " +
	    "СоздатьДиалогСохраненияФайла " +
	    "СоздатьЗапрос " +
	    "СоздатьИндикатор " +
	    "СоздатьИсключение " +
	    "СоздатьКэшированныйСправочник " +
	    "СоздатьМассив " +
	    "СоздатьНаборДанных " +
	    "СоздатьОбъект " +
	    "СоздатьОтчет " +
	    "СоздатьПапку " +
	    "СоздатьРедактор " +
	    "СоздатьСоединение " +
	    "СоздатьСписок " +
	    "СоздатьСписокСтрок " +
	    "СоздатьСправочник " +
	    "СоздатьСценарий " +
	    "СоздСпр " +
	    "СостСпр " +
	    "Сохр " +
	    "СохрСпр " +
	    "СписокСистем " +
	    "Спр " +
	    "Справочник " +
	    "СпрБлокЕсть " +
	    "СпрБлокСнять " +
	    "СпрБлокСнятьРасш " +
	    "СпрБлокУстановить " +
	    "СпрИзмНабДан " +
	    "СпрКод " +
	    "СпрНомер " +
	    "СпрОбновить " +
	    "СпрОткрыть " +
	    "СпрОтменить " +
	    "СпрПарам " +
	    "СпрПолеЗнач " +
	    "СпрПолеИмя " +
	    "СпрРекв " +
	    "СпрРеквВведЗн " +
	    "СпрРеквНовые " +
	    "СпрРеквПр " +
	    "СпрРеквПредЗн " +
	    "СпрРеквРежим " +
	    "СпрРеквТипТекст " +
	    "СпрСоздать " +
	    "СпрСост " +
	    "СпрСохранить " +
	    "СпрТблИтог " +
	    "СпрТблСтр " +
	    "СпрТблСтрКол " +
	    "СпрТблСтрМакс " +
	    "СпрТблСтрМин " +
	    "СпрТблСтрПред " +
	    "СпрТблСтрСлед " +
	    "СпрТблСтрСозд " +
	    "СпрТблСтрУд " +
	    "СпрТекПредст " +
	    "СпрУдалить " +
	    "СравнитьСтр " +
	    "СтрВерхРегистр " +
	    "СтрНижнРегистр " +
	    "СтрТблСпр " +
	    "СумПроп " +
	    "Сценарий " +
	    "СценарийПарам " +
	    "ТекВерсия " +
	    "ТекОрг " +
	    "Точн " +
	    "Тран " +
	    "Транслитерация " +
	    "УдалитьТаблицу " +
	    "УдалитьФайл " +
	    "УдСпр " +
	    "УдСтрТблСпр " +
	    "Уст " +
	    "УстановкиКонстант " +
	    "ФайлАтрибутСчитать " +
	    "ФайлАтрибутУстановить " +
	    "ФайлВремя " +
	    "ФайлВремяУстановить " +
	    "ФайлВыбрать " +
	    "ФайлЗанят " +
	    "ФайлЗаписать " +
	    "ФайлИскать " +
	    "ФайлКопировать " +
	    "ФайлМожноЧитать " +
	    "ФайлОткрыть " +
	    "ФайлПереименовать " +
	    "ФайлПерекодировать " +
	    "ФайлПереместить " +
	    "ФайлПросмотреть " +
	    "ФайлРазмер " +
	    "ФайлСоздать " +
	    "ФайлСсылкаСоздать " +
	    "ФайлСуществует " +
	    "ФайлСчитать " +
	    "ФайлУдалить " +
	    "ФмтSQLДат " +
	    "ФмтДат " +
	    "ФмтСтр " +
	    "ФмтЧсл " +
	    "Формат " +
	    "ЦМассивЭлемент " +
	    "ЦНаборДанныхРеквизит " +
	    "ЦПодстр ";

	  // Предопределенные переменные ==> built_in
	  const predefined_variables =
	    "AltState " +
	    "Application " +
	    "CallType " +
	    "ComponentTokens " +
	    "CreatedJobs " +
	    "CreatedNotices " +
	    "ControlState " +
	    "DialogResult " +
	    "Dialogs " +
	    "EDocuments " +
	    "EDocumentVersionSource " +
	    "Folders " +
	    "GlobalIDs " +
	    "Job " +
	    "Jobs " +
	    "InputValue " +
	    "LookUpReference " +
	    "LookUpRequisiteNames " +
	    "LookUpSearch " +
	    "Object " +
	    "ParentComponent " +
	    "Processes " +
	    "References " +
	    "Requisite " +
	    "ReportName " +
	    "Reports " +
	    "Result " +
	    "Scripts " +
	    "Searches " +
	    "SelectedAttachments " +
	    "SelectedItems " +
	    "SelectMode " +
	    "Sender " +
	    "ServerEvents " +
	    "ServiceFactory " +
	    "ShiftState " +
	    "SubTask " +
	    "SystemDialogs " +
	    "Tasks " +
	    "Wizard " +
	    "Wizards " +
	    "Work " +
	    "ВызовСпособ " +
	    "ИмяОтчета " +
	    "РеквЗнач ";

	  // Интерфейсы ==> type
	  const interfaces =
	    "IApplication " +
	    "IAccessRights " +
	    "IAccountRepository " +
	    "IAccountSelectionRestrictions " +
	    "IAction " +
	    "IActionList " +
	    "IAdministrationHistoryDescription " +
	    "IAnchors " +
	    "IApplication " +
	    "IArchiveInfo " +
	    "IAttachment " +
	    "IAttachmentList " +
	    "ICheckListBox " +
	    "ICheckPointedList " +
	    "IColumn " +
	    "IComponent " +
	    "IComponentDescription " +
	    "IComponentToken " +
	    "IComponentTokenFactory " +
	    "IComponentTokenInfo " +
	    "ICompRecordInfo " +
	    "IConnection " +
	    "IContents " +
	    "IControl " +
	    "IControlJob " +
	    "IControlJobInfo " +
	    "IControlList " +
	    "ICrypto " +
	    "ICrypto2 " +
	    "ICustomJob " +
	    "ICustomJobInfo " +
	    "ICustomListBox " +
	    "ICustomObjectWizardStep " +
	    "ICustomWork " +
	    "ICustomWorkInfo " +
	    "IDataSet " +
	    "IDataSetAccessInfo " +
	    "IDataSigner " +
	    "IDateCriterion " +
	    "IDateRequisite " +
	    "IDateRequisiteDescription " +
	    "IDateValue " +
	    "IDeaAccessRights " +
	    "IDeaObjectInfo " +
	    "IDevelopmentComponentLock " +
	    "IDialog " +
	    "IDialogFactory " +
	    "IDialogPickRequisiteItems " +
	    "IDialogsFactory " +
	    "IDICSFactory " +
	    "IDocRequisite " +
	    "IDocumentInfo " +
	    "IDualListDialog " +
	    "IECertificate " +
	    "IECertificateInfo " +
	    "IECertificates " +
	    "IEditControl " +
	    "IEditorForm " +
	    "IEdmsExplorer " +
	    "IEdmsObject " +
	    "IEdmsObjectDescription " +
	    "IEdmsObjectFactory " +
	    "IEdmsObjectInfo " +
	    "IEDocument " +
	    "IEDocumentAccessRights " +
	    "IEDocumentDescription " +
	    "IEDocumentEditor " +
	    "IEDocumentFactory " +
	    "IEDocumentInfo " +
	    "IEDocumentStorage " +
	    "IEDocumentVersion " +
	    "IEDocumentVersionListDialog " +
	    "IEDocumentVersionSource " +
	    "IEDocumentWizardStep " +
	    "IEDocVerSignature " +
	    "IEDocVersionState " +
	    "IEnabledMode " +
	    "IEncodeProvider " +
	    "IEncrypter " +
	    "IEvent " +
	    "IEventList " +
	    "IException " +
	    "IExternalEvents " +
	    "IExternalHandler " +
	    "IFactory " +
	    "IField " +
	    "IFileDialog " +
	    "IFolder " +
	    "IFolderDescription " +
	    "IFolderDialog " +
	    "IFolderFactory " +
	    "IFolderInfo " +
	    "IForEach " +
	    "IForm " +
	    "IFormTitle " +
	    "IFormWizardStep " +
	    "IGlobalIDFactory " +
	    "IGlobalIDInfo " +
	    "IGrid " +
	    "IHasher " +
	    "IHistoryDescription " +
	    "IHyperLinkControl " +
	    "IImageButton " +
	    "IImageControl " +
	    "IInnerPanel " +
	    "IInplaceHint " +
	    "IIntegerCriterion " +
	    "IIntegerList " +
	    "IIntegerRequisite " +
	    "IIntegerValue " +
	    "IISBLEditorForm " +
	    "IJob " +
	    "IJobDescription " +
	    "IJobFactory " +
	    "IJobForm " +
	    "IJobInfo " +
	    "ILabelControl " +
	    "ILargeIntegerCriterion " +
	    "ILargeIntegerRequisite " +
	    "ILargeIntegerValue " +
	    "ILicenseInfo " +
	    "ILifeCycleStage " +
	    "IList " +
	    "IListBox " +
	    "ILocalIDInfo " +
	    "ILocalization " +
	    "ILock " +
	    "IMemoryDataSet " +
	    "IMessagingFactory " +
	    "IMetadataRepository " +
	    "INotice " +
	    "INoticeInfo " +
	    "INumericCriterion " +
	    "INumericRequisite " +
	    "INumericValue " +
	    "IObject " +
	    "IObjectDescription " +
	    "IObjectImporter " +
	    "IObjectInfo " +
	    "IObserver " +
	    "IPanelGroup " +
	    "IPickCriterion " +
	    "IPickProperty " +
	    "IPickRequisite " +
	    "IPickRequisiteDescription " +
	    "IPickRequisiteItem " +
	    "IPickRequisiteItems " +
	    "IPickValue " +
	    "IPrivilege " +
	    "IPrivilegeList " +
	    "IProcess " +
	    "IProcessFactory " +
	    "IProcessMessage " +
	    "IProgress " +
	    "IProperty " +
	    "IPropertyChangeEvent " +
	    "IQuery " +
	    "IReference " +
	    "IReferenceCriterion " +
	    "IReferenceEnabledMode " +
	    "IReferenceFactory " +
	    "IReferenceHistoryDescription " +
	    "IReferenceInfo " +
	    "IReferenceRecordCardWizardStep " +
	    "IReferenceRequisiteDescription " +
	    "IReferencesFactory " +
	    "IReferenceValue " +
	    "IRefRequisite " +
	    "IReport " +
	    "IReportFactory " +
	    "IRequisite " +
	    "IRequisiteDescription " +
	    "IRequisiteDescriptionList " +
	    "IRequisiteFactory " +
	    "IRichEdit " +
	    "IRouteStep " +
	    "IRule " +
	    "IRuleList " +
	    "ISchemeBlock " +
	    "IScript " +
	    "IScriptFactory " +
	    "ISearchCriteria " +
	    "ISearchCriterion " +
	    "ISearchDescription " +
	    "ISearchFactory " +
	    "ISearchFolderInfo " +
	    "ISearchForObjectDescription " +
	    "ISearchResultRestrictions " +
	    "ISecuredContext " +
	    "ISelectDialog " +
	    "IServerEvent " +
	    "IServerEventFactory " +
	    "IServiceDialog " +
	    "IServiceFactory " +
	    "ISignature " +
	    "ISignProvider " +
	    "ISignProvider2 " +
	    "ISignProvider3 " +
	    "ISimpleCriterion " +
	    "IStringCriterion " +
	    "IStringList " +
	    "IStringRequisite " +
	    "IStringRequisiteDescription " +
	    "IStringValue " +
	    "ISystemDialogsFactory " +
	    "ISystemInfo " +
	    "ITabSheet " +
	    "ITask " +
	    "ITaskAbortReasonInfo " +
	    "ITaskCardWizardStep " +
	    "ITaskDescription " +
	    "ITaskFactory " +
	    "ITaskInfo " +
	    "ITaskRoute " +
	    "ITextCriterion " +
	    "ITextRequisite " +
	    "ITextValue " +
	    "ITreeListSelectDialog " +
	    "IUser " +
	    "IUserList " +
	    "IValue " +
	    "IView " +
	    "IWebBrowserControl " +
	    "IWizard " +
	    "IWizardAction " +
	    "IWizardFactory " +
	    "IWizardFormElement " +
	    "IWizardParam " +
	    "IWizardPickParam " +
	    "IWizardReferenceParam " +
	    "IWizardStep " +
	    "IWorkAccessRights " +
	    "IWorkDescription " +
	    "IWorkflowAskableParam " +
	    "IWorkflowAskableParams " +
	    "IWorkflowBlock " +
	    "IWorkflowBlockResult " +
	    "IWorkflowEnabledMode " +
	    "IWorkflowParam " +
	    "IWorkflowPickParam " +
	    "IWorkflowReferenceParam " +
	    "IWorkState " +
	    "IWorkTreeCustomNode " +
	    "IWorkTreeJobNode " +
	    "IWorkTreeTaskNode " +
	    "IXMLEditorForm " +
	    "SBCrypto ";

	  // built_in : встроенные или библиотечные объекты (константы, перечисления)
	  const BUILTIN = CONSTANTS + ENUMS;

	  // class: встроенные наборы значений, системные объекты, фабрики
	  const CLASS = predefined_variables;

	  // literal : примитивные типы
	  const LITERAL = "null true false nil ";

	  // number : числа
	  const NUMBERS = {
	    className: "number",
	    begin: hljs.NUMBER_RE,
	    relevance: 0
	  };

	  // string : строки
	  const STRINGS = {
	    className: "string",
	    variants: [
	      {
	        begin: '"',
	        end: '"'
	      },
	      {
	        begin: "'",
	        end: "'"
	      }
	    ]
	  };

	  // Токены
	  const DOCTAGS = {
	    className: "doctag",
	    begin: "\\b(?:TODO|DONE|BEGIN|END|STUB|CHG|FIXME|NOTE|BUG|XXX)\\b",
	    relevance: 0
	  };

	  // Однострочный комментарий
	  const ISBL_LINE_COMMENT_MODE = {
	    className: "comment",
	    begin: "//",
	    end: "$",
	    relevance: 0,
	    contains: [
	      hljs.PHRASAL_WORDS_MODE,
	      DOCTAGS
	    ]
	  };

	  // Многострочный комментарий
	  const ISBL_BLOCK_COMMENT_MODE = {
	    className: "comment",
	    begin: "/\\*",
	    end: "\\*/",
	    relevance: 0,
	    contains: [
	      hljs.PHRASAL_WORDS_MODE,
	      DOCTAGS
	    ]
	  };

	  // comment : комментарии
	  const COMMENTS = {
	    variants: [
	      ISBL_LINE_COMMENT_MODE,
	      ISBL_BLOCK_COMMENT_MODE
	    ]
	  };

	  // keywords : ключевые слова
	  const KEYWORDS = {
	    $pattern: UNDERSCORE_IDENT_RE,
	    keyword: KEYWORD,
	    built_in: BUILTIN,
	    class: CLASS,
	    literal: LITERAL
	  };

	  // methods : методы
	  const METHODS = {
	    begin: "\\.\\s*" + hljs.UNDERSCORE_IDENT_RE,
	    keywords: KEYWORDS,
	    relevance: 0
	  };

	  // type : встроенные типы
	  const TYPES = {
	    className: "type",
	    begin: ":[ \\t]*(" + interfaces.trim().replace(/\s/g, "|") + ")",
	    end: "[ \\t]*=",
	    excludeEnd: true
	  };

	  // variables : переменные
	  const VARIABLES = {
	    className: "variable",
	    keywords: KEYWORDS,
	    begin: UNDERSCORE_IDENT_RE,
	    relevance: 0,
	    contains: [
	      TYPES,
	      METHODS
	    ]
	  };

	  // Имена функций
	  const FUNCTION_TITLE = FUNCTION_NAME_IDENT_RE + "\\(";

	  const TITLE_MODE = {
	    className: "title",
	    keywords: {
	      $pattern: UNDERSCORE_IDENT_RE,
	      built_in: system_functions
	    },
	    begin: FUNCTION_TITLE,
	    end: "\\(",
	    returnBegin: true,
	    excludeEnd: true
	  };

	  // function : функции
	  const FUNCTIONS = {
	    className: "function",
	    begin: FUNCTION_TITLE,
	    end: "\\)$",
	    returnBegin: true,
	    keywords: KEYWORDS,
	    illegal: "[\\[\\]\\|\\$\\?%,~#@]",
	    contains: [
	      TITLE_MODE,
	      METHODS,
	      VARIABLES,
	      STRINGS,
	      NUMBERS,
	      COMMENTS
	    ]
	  };

	  return {
	    name: 'ISBL',
	    case_insensitive: true,
	    keywords: KEYWORDS,
	    illegal: "\\$|\\?|%|,|;$|~|#|@|</",
	    contains: [
	      FUNCTIONS,
	      TYPES,
	      METHODS,
	      VARIABLES,
	      STRINGS,
	      NUMBERS,
	      COMMENTS
	    ]
	  };
	}

	isbl_1 = isbl;
	return isbl_1;
}

var java_1;
var hasRequiredJava;

function requireJava () {
	if (hasRequiredJava) return java_1;
	hasRequiredJava = 1;
	// https://docs.oracle.com/javase/specs/jls/se15/html/jls-3.html#jls-3.10
	var decimalDigits = '[0-9](_*[0-9])*';
	var frac = `\\.(${decimalDigits})`;
	var hexDigits = '[0-9a-fA-F](_*[0-9a-fA-F])*';
	var NUMERIC = {
	  className: 'number',
	  variants: [
	    // DecimalFloatingPointLiteral
	    // including ExponentPart
	    { begin: `(\\b(${decimalDigits})((${frac})|\\.)?|(${frac}))` +
	      `[eE][+-]?(${decimalDigits})[fFdD]?\\b` },
	    // excluding ExponentPart
	    { begin: `\\b(${decimalDigits})((${frac})[fFdD]?\\b|\\.([fFdD]\\b)?)` },
	    { begin: `(${frac})[fFdD]?\\b` },
	    { begin: `\\b(${decimalDigits})[fFdD]\\b` },

	    // HexadecimalFloatingPointLiteral
	    { begin: `\\b0[xX]((${hexDigits})\\.?|(${hexDigits})?\\.(${hexDigits}))` +
	      `[pP][+-]?(${decimalDigits})[fFdD]?\\b` },

	    // DecimalIntegerLiteral
	    { begin: '\\b(0|[1-9](_*[0-9])*)[lL]?\\b' },

	    // HexIntegerLiteral
	    { begin: `\\b0[xX](${hexDigits})[lL]?\\b` },

	    // OctalIntegerLiteral
	    { begin: '\\b0(_*[0-7])*[lL]?\\b' },

	    // BinaryIntegerLiteral
	    { begin: '\\b0[bB][01](_*[01])*[lL]?\\b' },
	  ],
	  relevance: 0
	};

	/*
	Language: Java
	Author: Vsevolod Solovyov <vsevolod.solovyov@gmail.com>
	Category: common, enterprise
	Website: https://www.java.com/
	*/

	function java(hljs) {
	  var JAVA_IDENT_RE = '[\u00C0-\u02B8a-zA-Z_$][\u00C0-\u02B8a-zA-Z_$0-9]*';
	  var GENERIC_IDENT_RE = JAVA_IDENT_RE + '(<' + JAVA_IDENT_RE + '(\\s*,\\s*' + JAVA_IDENT_RE + ')*>)?';
	  var KEYWORDS = 'false synchronized int abstract float private char boolean var static null if const ' +
	    'for true while long strictfp finally protected import native final void ' +
	    'enum else break transient catch instanceof byte super volatile case assert short ' +
	    'package default double public try this switch continue throws protected public private ' +
	    'module requires exports do';

	  var ANNOTATION = {
	    className: 'meta',
	    begin: '@' + JAVA_IDENT_RE,
	    contains: [
	      {
	        begin: /\(/,
	        end: /\)/,
	        contains: ["self"] // allow nested () inside our annotation
	      },
	    ]
	  };
	  const NUMBER = NUMERIC;

	  return {
	    name: 'Java',
	    aliases: ['jsp'],
	    keywords: KEYWORDS,
	    illegal: /<\/|#/,
	    contains: [
	      hljs.COMMENT(
	        '/\\*\\*',
	        '\\*/',
	        {
	          relevance: 0,
	          contains: [
	            {
	              // eat up @'s in emails to prevent them to be recognized as doctags
	              begin: /\w+@/, relevance: 0
	            },
	            {
	              className: 'doctag',
	              begin: '@[A-Za-z]+'
	            }
	          ]
	        }
	      ),
	      // relevance boost
	      {
	        begin: /import java\.[a-z]+\./,
	        keywords: "import",
	        relevance: 2
	      },
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'class',
	        beginKeywords: 'class interface enum', end: /[{;=]/, excludeEnd: true,
	        // TODO: can this be removed somehow?
	        // an extra boost because Java is more popular than other languages with
	        // this same syntax feature (this is just to preserve our tests passing
	        // for now)
	        relevance: 1,
	        keywords: 'class interface enum',
	        illegal: /[:"\[\]]/,
	        contains: [
	          { beginKeywords: 'extends implements' },
	          hljs.UNDERSCORE_TITLE_MODE
	        ]
	      },
	      {
	        // Expression keywords prevent 'keyword Name(...)' from being
	        // recognized as a function definition
	        beginKeywords: 'new throw return else',
	        relevance: 0
	      },
	      {
	        className: 'class',
	        begin: 'record\\s+' + hljs.UNDERSCORE_IDENT_RE + '\\s*\\(',
	        returnBegin: true,
	        excludeEnd: true,
	        end: /[{;=]/,
	        keywords: KEYWORDS,
	        contains: [
	          { beginKeywords: "record" },
	          {
	            begin: hljs.UNDERSCORE_IDENT_RE + '\\s*\\(',
	            returnBegin: true,
	            relevance: 0,
	            contains: [hljs.UNDERSCORE_TITLE_MODE]
	          },
	          {
	            className: 'params',
	            begin: /\(/, end: /\)/,
	            keywords: KEYWORDS,
	            relevance: 0,
	            contains: [
	              hljs.C_BLOCK_COMMENT_MODE
	            ]
	          },
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE
	        ]
	      },
	      {
	        className: 'function',
	        begin: '(' + GENERIC_IDENT_RE + '\\s+)+' + hljs.UNDERSCORE_IDENT_RE + '\\s*\\(', returnBegin: true, end: /[{;=]/,
	        excludeEnd: true,
	        keywords: KEYWORDS,
	        contains: [
	          {
	            begin: hljs.UNDERSCORE_IDENT_RE + '\\s*\\(', returnBegin: true,
	            relevance: 0,
	            contains: [hljs.UNDERSCORE_TITLE_MODE]
	          },
	          {
	            className: 'params',
	            begin: /\(/, end: /\)/,
	            keywords: KEYWORDS,
	            relevance: 0,
	            contains: [
	              ANNOTATION,
	              hljs.APOS_STRING_MODE,
	              hljs.QUOTE_STRING_MODE,
	              NUMBER,
	              hljs.C_BLOCK_COMMENT_MODE
	            ]
	          },
	          hljs.C_LINE_COMMENT_MODE,
	          hljs.C_BLOCK_COMMENT_MODE
	        ]
	      },
	      NUMBER,
	      ANNOTATION
	    ]
	  };
	}

	java_1 = java;
	return java_1;
}

var javascript_1;
var hasRequiredJavascript;

function requireJavascript () {
	if (hasRequiredJavascript) return javascript_1;
	hasRequiredJavascript = 1;
	const IDENT_RE = '[A-Za-z$_][0-9A-Za-z$_]*';
	const KEYWORDS = [
	  "as", // for exports
	  "in",
	  "of",
	  "if",
	  "for",
	  "while",
	  "finally",
	  "var",
	  "new",
	  "function",
	  "do",
	  "return",
	  "void",
	  "else",
	  "break",
	  "catch",
	  "instanceof",
	  "with",
	  "throw",
	  "case",
	  "default",
	  "try",
	  "switch",
	  "continue",
	  "typeof",
	  "delete",
	  "let",
	  "yield",
	  "const",
	  "class",
	  // JS handles these with a special rule
	  // "get",
	  // "set",
	  "debugger",
	  "async",
	  "await",
	  "static",
	  "import",
	  "from",
	  "export",
	  "extends"
	];
	const LITERALS = [
	  "true",
	  "false",
	  "null",
	  "undefined",
	  "NaN",
	  "Infinity"
	];

	const TYPES = [
	  "Intl",
	  "DataView",
	  "Number",
	  "Math",
	  "Date",
	  "String",
	  "RegExp",
	  "Object",
	  "Function",
	  "Boolean",
	  "Error",
	  "Symbol",
	  "Set",
	  "Map",
	  "WeakSet",
	  "WeakMap",
	  "Proxy",
	  "Reflect",
	  "JSON",
	  "Promise",
	  "Float64Array",
	  "Int16Array",
	  "Int32Array",
	  "Int8Array",
	  "Uint16Array",
	  "Uint32Array",
	  "Float32Array",
	  "Array",
	  "Uint8Array",
	  "Uint8ClampedArray",
	  "ArrayBuffer",
	  "BigInt64Array",
	  "BigUint64Array",
	  "BigInt"
	];

	const ERROR_TYPES = [
	  "EvalError",
	  "InternalError",
	  "RangeError",
	  "ReferenceError",
	  "SyntaxError",
	  "TypeError",
	  "URIError"
	];

	const BUILT_IN_GLOBALS = [
	  "setInterval",
	  "setTimeout",
	  "clearInterval",
	  "clearTimeout",

	  "require",
	  "exports",

	  "eval",
	  "isFinite",
	  "isNaN",
	  "parseFloat",
	  "parseInt",
	  "decodeURI",
	  "decodeURIComponent",
	  "encodeURI",
	  "encodeURIComponent",
	  "escape",
	  "unescape"
	];

	const BUILT_IN_VARIABLES = [
	  "arguments",
	  "this",
	  "super",
	  "console",
	  "window",
	  "document",
	  "localStorage",
	  "module",
	  "global" // Node.js
	];

	const BUILT_INS = [].concat(
	  BUILT_IN_GLOBALS,
	  BUILT_IN_VARIABLES,
	  TYPES,
	  ERROR_TYPES
	);

	/**
	 * @param {string} value
	 * @returns {RegExp}
	 * */

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function lookahead(re) {
	  return concat('(?=', re, ')');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: JavaScript
	Description: JavaScript (JS) is a lightweight, interpreted, or just-in-time compiled programming language with first-class functions.
	Category: common, scripting
	Website: https://developer.mozilla.org/en-US/docs/Web/JavaScript
	*/

	/** @type LanguageFn */
	function javascript(hljs) {
	  /**
	   * Takes a string like "<Booger" and checks to see
	   * if we can find a matching "</Booger" later in the
	   * content.
	   * @param {RegExpMatchArray} match
	   * @param {{after:number}} param1
	   */
	  const hasClosingTag = (match, { after }) => {
	    const tag = "</" + match[0].slice(1);
	    const pos = match.input.indexOf(tag, after);
	    return pos !== -1;
	  };

	  const IDENT_RE$1 = IDENT_RE;
	  const FRAGMENT = {
	    begin: '<>',
	    end: '</>'
	  };
	  const XML_TAG = {
	    begin: /<[A-Za-z0-9\\._:-]+/,
	    end: /\/[A-Za-z0-9\\._:-]+>|\/>/,
	    /**
	     * @param {RegExpMatchArray} match
	     * @param {CallbackResponse} response
	     */
	    isTrulyOpeningTag: (match, response) => {
	      const afterMatchIndex = match[0].length + match.index;
	      const nextChar = match.input[afterMatchIndex];
	      // nested type?
	      // HTML should not include another raw `<` inside a tag
	      // But a type might: `<Array<Array<number>>`, etc.
	      if (nextChar === "<") {
	        response.ignoreMatch();
	        return;
	      }
	      // <something>
	      // This is now either a tag or a type.
	      if (nextChar === ">") {
	        // if we cannot find a matching closing tag, then we
	        // will ignore it
	        if (!hasClosingTag(match, { after: afterMatchIndex })) {
	          response.ignoreMatch();
	        }
	      }
	    }
	  };
	  const KEYWORDS$1 = {
	    $pattern: IDENT_RE,
	    keyword: KEYWORDS,
	    literal: LITERALS,
	    built_in: BUILT_INS
	  };

	  // https://tc39.es/ecma262/#sec-literals-numeric-literals
	  const decimalDigits = '[0-9](_?[0-9])*';
	  const frac = `\\.(${decimalDigits})`;
	  // DecimalIntegerLiteral, including Annex B NonOctalDecimalIntegerLiteral
	  // https://tc39.es/ecma262/#sec-additional-syntax-numeric-literals
	  const decimalInteger = `0|[1-9](_?[0-9])*|0[0-7]*[89][0-9]*`;
	  const NUMBER = {
	    className: 'number',
	    variants: [
	      // DecimalLiteral
	      { begin: `(\\b(${decimalInteger})((${frac})|\\.)?|(${frac}))` +
	        `[eE][+-]?(${decimalDigits})\\b` },
	      { begin: `\\b(${decimalInteger})\\b((${frac})\\b|\\.)?|(${frac})\\b` },

	      // DecimalBigIntegerLiteral
	      { begin: `\\b(0|[1-9](_?[0-9])*)n\\b` },

	      // NonDecimalIntegerLiteral
	      { begin: "\\b0[xX][0-9a-fA-F](_?[0-9a-fA-F])*n?\\b" },
	      { begin: "\\b0[bB][0-1](_?[0-1])*n?\\b" },
	      { begin: "\\b0[oO][0-7](_?[0-7])*n?\\b" },

	      // LegacyOctalIntegerLiteral (does not include underscore separators)
	      // https://tc39.es/ecma262/#sec-additional-syntax-numeric-literals
	      { begin: "\\b0[0-7]+n?\\b" },
	    ],
	    relevance: 0
	  };

	  const SUBST = {
	    className: 'subst',
	    begin: '\\$\\{',
	    end: '\\}',
	    keywords: KEYWORDS$1,
	    contains: [] // defined later
	  };
	  const HTML_TEMPLATE = {
	    begin: 'html`',
	    end: '',
	    starts: {
	      end: '`',
	      returnEnd: false,
	      contains: [
	        hljs.BACKSLASH_ESCAPE,
	        SUBST
	      ],
	      subLanguage: 'xml'
	    }
	  };
	  const CSS_TEMPLATE = {
	    begin: 'css`',
	    end: '',
	    starts: {
	      end: '`',
	      returnEnd: false,
	      contains: [
	        hljs.BACKSLASH_ESCAPE,
	        SUBST
	      ],
	      subLanguage: 'css'
	    }
	  };
	  const TEMPLATE_STRING = {
	    className: 'string',
	    begin: '`',
	    end: '`',
	    contains: [
	      hljs.BACKSLASH_ESCAPE,
	      SUBST
	    ]
	  };
	  const JSDOC_COMMENT = hljs.COMMENT(
	    /\/\*\*(?!\/)/,
	    '\\*/',
	    {
	      relevance: 0,
	      contains: [
	        {
	          className: 'doctag',
	          begin: '@[A-Za-z]+',
	          contains: [
	            {
	              className: 'type',
	              begin: '\\{',
	              end: '\\}',
	              relevance: 0
	            },
	            {
	              className: 'variable',
	              begin: IDENT_RE$1 + '(?=\\s*(-)|$)',
	              endsParent: true,
	              relevance: 0
	            },
	            // eat spaces (not newlines) so we can find
	            // types or variables
	            {
	              begin: /(?=[^\n])\s/,
	              relevance: 0
	            }
	          ]
	        }
	      ]
	    }
	  );
	  const COMMENT = {
	    className: "comment",
	    variants: [
	      JSDOC_COMMENT,
	      hljs.C_BLOCK_COMMENT_MODE,
	      hljs.C_LINE_COMMENT_MODE
	    ]
	  };
	  const SUBST_INTERNALS = [
	    hljs.APOS_STRING_MODE,
	    hljs.QUOTE_STRING_MODE,
	    HTML_TEMPLATE,
	    CSS_TEMPLATE,
	    TEMPLATE_STRING,
	    NUMBER,
	    hljs.REGEXP_MODE
	  ];
	  SUBST.contains = SUBST_INTERNALS
	    .concat({
	      // we need to pair up {} inside our subst to prevent
	      // it from ending too early by matching another }
	      begin: /\{/,
	      end: /\}/,
	      keywords: KEYWORDS$1,
	      contains: [
	        "self"
	      ].concat(SUBST_INTERNALS)
	    });
	  const SUBST_AND_COMMENTS = [].concat(COMMENT, SUBST.contains);
	  const PARAMS_CONTAINS = SUBST_AND_COMMENTS.concat([
	    // eat recursive parens in sub expressions
	    {
	      begin: /\(/,
	      end: /\)/,
	      keywords: KEYWORDS$1,
	      contains: ["self"].concat(SUBST_AND_COMMENTS)
	    }
	  ]);
	  const PARAMS = {
	    className: 'params',
	    begin: /\(/,
	    end: /\)/,
	    excludeBegin: true,
	    excludeEnd: true,
	    keywords: KEYWORDS$1,
	    contains: PARAMS_CONTAINS
	  };

	  return {
	    name: 'Javascript',
	    aliases: ['js', 'jsx', 'mjs', 'cjs'],
	    keywords: KEYWORDS$1,
	    // this will be extended by TypeScript
	    exports: { PARAMS_CONTAINS },
	    illegal: /#(?![$_A-z])/,
	    contains: [
	      hljs.SHEBANG({
	        label: "shebang",
	        binary: "node",
	        relevance: 5
	      }),
	      {
	        label: "use_strict",
	        className: 'meta',
	        relevance: 10,
	        begin: /^\s*['"]use (strict|asm)['"]/
	      },
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      HTML_TEMPLATE,
	      CSS_TEMPLATE,
	      TEMPLATE_STRING,
	      COMMENT,
	      NUMBER,
	      { // object attr container
	        begin: concat(/[{,\n]\s*/,
	          // we need to look ahead to make sure that we actually have an
	          // attribute coming up so we don't steal a comma from a potential
	          // "value" container
	          //
	          // NOTE: this might not work how you think.  We don't actually always
	          // enter this mode and stay.  Instead it might merely match `,
	          // <comments up next>` and then immediately end after the , because it
	          // fails to find any actual attrs. But this still does the job because
	          // it prevents the value contain rule from grabbing this instead and
	          // prevening this rule from firing when we actually DO have keys.
	          lookahead(concat(
	            // we also need to allow for multiple possible comments inbetween
	            // the first key:value pairing
	            /(((\/\/.*$)|(\/\*(\*[^/]|[^*])*\*\/))\s*)*/,
	            IDENT_RE$1 + '\\s*:'))),
	        relevance: 0,
	        contains: [
	          {
	            className: 'attr',
	            begin: IDENT_RE$1 + lookahead('\\s*:'),
	            relevance: 0
	          }
	        ]
	      },
	      { // "value" container
	        begin: '(' + hljs.RE_STARTERS_RE + '|\\b(case|return|throw)\\b)\\s*',
	        keywords: 'return throw case',
	        contains: [
	          COMMENT,
	          hljs.REGEXP_MODE,
	          {
	            className: 'function',
	            // we have to count the parens to make sure we actually have the
	            // correct bounding ( ) before the =>.  There could be any number of
	            // sub-expressions inside also surrounded by parens.
	            begin: '(\\(' +
	            '[^()]*(\\(' +
	            '[^()]*(\\(' +
	            '[^()]*' +
	            '\\)[^()]*)*' +
	            '\\)[^()]*)*' +
	            '\\)|' + hljs.UNDERSCORE_IDENT_RE + ')\\s*=>',
	            returnBegin: true,
	            end: '\\s*=>',
	            contains: [
	              {
	                className: 'params',
	                variants: [
	                  {
	                    begin: hljs.UNDERSCORE_IDENT_RE,
	                    relevance: 0
	                  },
	                  {
	                    className: null,
	                    begin: /\(\s*\)/,
	                    skip: true
	                  },
	                  {
	                    begin: /\(/,
	                    end: /\)/,
	                    excludeBegin: true,
	                    excludeEnd: true,
	                    keywords: KEYWORDS$1,
	                    contains: PARAMS_CONTAINS
	                  }
	                ]
	              }
	            ]
	          },
	          { // could be a comma delimited list of params to a function call
	            begin: /,/, relevance: 0
	          },
	          {
	            className: '',
	            begin: /\s/,
	            end: /\s*/,
	            skip: true
	          },
	          { // JSX
	            variants: [
	              { begin: FRAGMENT.begin, end: FRAGMENT.end },
	              {
	                begin: XML_TAG.begin,
	                // we carefully check the opening tag to see if it truly
	                // is a tag and not a false positive
	                'on:begin': XML_TAG.isTrulyOpeningTag,
	                end: XML_TAG.end
	              }
	            ],
	            subLanguage: 'xml',
	            contains: [
	              {
	                begin: XML_TAG.begin,
	                end: XML_TAG.end,
	                skip: true,
	                contains: ['self']
	              }
	            ]
	          }
	        ],
	        relevance: 0
	      },
	      {
	        className: 'function',
	        beginKeywords: 'function',
	        end: /[{;]/,
	        excludeEnd: true,
	        keywords: KEYWORDS$1,
	        contains: [
	          'self',
	          hljs.inherit(hljs.TITLE_MODE, { begin: IDENT_RE$1 }),
	          PARAMS
	        ],
	        illegal: /%/
	      },
	      {
	        // prevent this from getting swallowed up by function
	        // since they appear "function like"
	        beginKeywords: "while if switch catch for"
	      },
	      {
	        className: 'function',
	        // we have to count the parens to make sure we actually have the correct
	        // bounding ( ).  There could be any number of sub-expressions inside
	        // also surrounded by parens.
	        begin: hljs.UNDERSCORE_IDENT_RE +
	          '\\(' + // first parens
	          '[^()]*(\\(' +
	            '[^()]*(\\(' +
	              '[^()]*' +
	            '\\)[^()]*)*' +
	          '\\)[^()]*)*' +
	          '\\)\\s*\\{', // end parens
	        returnBegin:true,
	        contains: [
	          PARAMS,
	          hljs.inherit(hljs.TITLE_MODE, { begin: IDENT_RE$1 }),
	        ]
	      },
	      // hack: prevents detection of keywords in some circumstances
	      // .keyword()
	      // $keyword = x
	      {
	        variants: [
	          { begin: '\\.' + IDENT_RE$1 },
	          { begin: '\\$' + IDENT_RE$1 }
	        ],
	        relevance: 0
	      },
	      { // ES6 class
	        className: 'class',
	        beginKeywords: 'class',
	        end: /[{;=]/,
	        excludeEnd: true,
	        illegal: /[:"[\]]/,
	        contains: [
	          { beginKeywords: 'extends' },
	          hljs.UNDERSCORE_TITLE_MODE
	        ]
	      },
	      {
	        begin: /\b(?=constructor)/,
	        end: /[{;]/,
	        excludeEnd: true,
	        contains: [
	          hljs.inherit(hljs.TITLE_MODE, { begin: IDENT_RE$1 }),
	          'self',
	          PARAMS
	        ]
	      },
	      {
	        begin: '(get|set)\\s+(?=' + IDENT_RE$1 + '\\()',
	        end: /\{/,
	        keywords: "get set",
	        contains: [
	          hljs.inherit(hljs.TITLE_MODE, { begin: IDENT_RE$1 }),
	          { begin: /\(\)/ }, // eat to avoid empty params
	          PARAMS
	        ]
	      },
	      {
	        begin: /\$[(.]/ // relevance booster for a pattern common to JS libs: `$(something)` and `$.something`
	      }
	    ]
	  };
	}

	javascript_1 = javascript;
	return javascript_1;
}

/*
 Language: JBoss CLI
 Author: Raphaël Parrëe <rparree@edc4it.com>
 Description: language definition jboss cli
 Website: https://docs.jboss.org/author/display/WFLY/Command+Line+Interface
 Category: config
 */

var jbossCli_1;
var hasRequiredJbossCli;

function requireJbossCli () {
	if (hasRequiredJbossCli) return jbossCli_1;
	hasRequiredJbossCli = 1;
	function jbossCli(hljs) {
	  const PARAM = {
	    begin: /[\w-]+ *=/,
	    returnBegin: true,
	    relevance: 0,
	    contains: [
	      {
	        className: 'attr',
	        begin: /[\w-]+/
	      }
	    ]
	  };
	  const PARAMSBLOCK = {
	    className: 'params',
	    begin: /\(/,
	    end: /\)/,
	    contains: [PARAM],
	    relevance: 0
	  };
	  const OPERATION = {
	    className: 'function',
	    begin: /:[\w\-.]+/,
	    relevance: 0
	  };
	  const PATH = {
	    className: 'string',
	    begin: /\B([\/.])[\w\-.\/=]+/
	  };
	  const COMMAND_PARAMS = {
	    className: 'params',
	    begin: /--[\w\-=\/]+/
	  };
	  return {
	    name: 'JBoss CLI',
	    aliases: ['wildfly-cli'],
	    keywords: {
	      $pattern: '[a-z\-]+',
	      keyword: 'alias batch cd clear command connect connection-factory connection-info data-source deploy ' +
	      'deployment-info deployment-overlay echo echo-dmr help history if jdbc-driver-info jms-queue|20 jms-topic|20 ls ' +
	      'patch pwd quit read-attribute read-operation reload rollout-plan run-batch set shutdown try unalias ' +
	      'undeploy unset version xa-data-source', // module
	      literal: 'true false'
	    },
	    contains: [
	      hljs.HASH_COMMENT_MODE,
	      hljs.QUOTE_STRING_MODE,
	      COMMAND_PARAMS,
	      OPERATION,
	      PATH,
	      PARAMSBLOCK
	    ]
	  };
	}

	jbossCli_1 = jbossCli;
	return jbossCli_1;
}

/*
Language: JSON
Description: JSON (JavaScript Object Notation) is a lightweight data-interchange format.
Author: Ivan Sagalaev <maniac@softwaremaniacs.org>
Website: http://www.json.org
Category: common, protocols
*/

var json_1;
var hasRequiredJson;

function requireJson () {
	if (hasRequiredJson) return json_1;
	hasRequiredJson = 1;
	function json(hljs) {
	  const LITERALS = {
	    literal: 'true false null'
	  };
	  const ALLOWED_COMMENTS = [
	    hljs.C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE
	  ];
	  const TYPES = [
	    hljs.QUOTE_STRING_MODE,
	    hljs.C_NUMBER_MODE
	  ];
	  const VALUE_CONTAINER = {
	    end: ',',
	    endsWithParent: true,
	    excludeEnd: true,
	    contains: TYPES,
	    keywords: LITERALS
	  };
	  const OBJECT = {
	    begin: /\{/,
	    end: /\}/,
	    contains: [
	      {
	        className: 'attr',
	        begin: /"/,
	        end: /"/,
	        contains: [hljs.BACKSLASH_ESCAPE],
	        illegal: '\\n'
	      },
	      hljs.inherit(VALUE_CONTAINER, {
	        begin: /:/
	      })
	    ].concat(ALLOWED_COMMENTS),
	    illegal: '\\S'
	  };
	  const ARRAY = {
	    begin: '\\[',
	    end: '\\]',
	    contains: [hljs.inherit(VALUE_CONTAINER)], // inherit is a workaround for a bug that makes shared modes with endsWithParent compile only the ending of one of the parents
	    illegal: '\\S'
	  };
	  TYPES.push(OBJECT, ARRAY);
	  ALLOWED_COMMENTS.forEach(function(rule) {
	    TYPES.push(rule);
	  });
	  return {
	    name: 'JSON',
	    contains: TYPES,
	    keywords: LITERALS,
	    illegal: '\\S'
	  };
	}

	json_1 = json;
	return json_1;
}

/*
Language: Julia
Description: Julia is a high-level, high-performance, dynamic programming language.
Author: Kenta Sato <bicycle1885@gmail.com>
Contributors: Alex Arslan <ararslan@comcast.net>, Fredrik Ekre <ekrefredrik@gmail.com>
Website: https://julialang.org
*/

var julia_1;
var hasRequiredJulia;

function requireJulia () {
	if (hasRequiredJulia) return julia_1;
	hasRequiredJulia = 1;
	function julia(hljs) {
	  // Since there are numerous special names in Julia, it is too much trouble
	  // to maintain them by hand. Hence these names (i.e. keywords, literals and
	  // built-ins) are automatically generated from Julia 1.5.2 itself through
	  // the following scripts for each.

	  // ref: https://docs.julialang.org/en/v1/manual/variables/#Allowed-Variable-Names
	  var VARIABLE_NAME_RE = '[A-Za-z_\\u00A1-\\uFFFF][A-Za-z_0-9\\u00A1-\\uFFFF]*';

	  // # keyword generator, multi-word keywords handled manually below (Julia 1.5.2)
	  // import REPL.REPLCompletions
	  // res = String["in", "isa", "where"]
	  // for kw in collect(x.keyword for x in REPLCompletions.complete_keyword(""))
	  //     if !(contains(kw, " ") || kw == "struct")
	  //         push!(res, kw)
	  //     end
	  // end
	  // sort!(unique!(res))
	  // foreach(x -> println("\'", x, "\',"), res)
	  var KEYWORD_LIST = [
	    'baremodule',
	    'begin',
	    'break',
	    'catch',
	    'ccall',
	    'const',
	    'continue',
	    'do',
	    'else',
	    'elseif',
	    'end',
	    'export',
	    'false',
	    'finally',
	    'for',
	    'function',
	    'global',
	    'if',
	    'import',
	    'in',
	    'isa',
	    'let',
	    'local',
	    'macro',
	    'module',
	    'quote',
	    'return',
	    'true',
	    'try',
	    'using',
	    'where',
	    'while',
	  ];

	  // # literal generator (Julia 1.5.2)
	  // import REPL.REPLCompletions
	  // res = String["true", "false"]
	  // for compl in filter!(x -> isa(x, REPLCompletions.ModuleCompletion) && (x.parent === Base || x.parent === Core),
	  //                     REPLCompletions.completions("", 0)[1])
	  //     try
	  //         v = eval(Symbol(compl.mod))
	  //         if !(v isa Function || v isa Type || v isa TypeVar || v isa Module || v isa Colon)
	  //             push!(res, compl.mod)
	  //         end
	  //     catch e
	  //     end
	  // end
	  // sort!(unique!(res))
	  // foreach(x -> println("\'", x, "\',"), res)
	  var LITERAL_LIST = [
	    'ARGS',
	    'C_NULL',
	    'DEPOT_PATH',
	    'ENDIAN_BOM',
	    'ENV',
	    'Inf',
	    'Inf16',
	    'Inf32',
	    'Inf64',
	    'InsertionSort',
	    'LOAD_PATH',
	    'MergeSort',
	    'NaN',
	    'NaN16',
	    'NaN32',
	    'NaN64',
	    'PROGRAM_FILE',
	    'QuickSort',
	    'RoundDown',
	    'RoundFromZero',
	    'RoundNearest',
	    'RoundNearestTiesAway',
	    'RoundNearestTiesUp',
	    'RoundToZero',
	    'RoundUp',
	    'VERSION|0',
	    'devnull',
	    'false',
	    'im',
	    'missing',
	    'nothing',
	    'pi',
	    'stderr',
	    'stdin',
	    'stdout',
	    'true',
	    'undef',
	    'π',
	    'ℯ',
	  ];

	  // # built_in generator (Julia 1.5.2)
	  // import REPL.REPLCompletions
	  // res = String[]
	  // for compl in filter!(x -> isa(x, REPLCompletions.ModuleCompletion) && (x.parent === Base || x.parent === Core),
	  //                     REPLCompletions.completions("", 0)[1])
	  //     try
	  //         v = eval(Symbol(compl.mod))
	  //         if (v isa Type || v isa TypeVar) && (compl.mod != "=>")
	  //             push!(res, compl.mod)
	  //         end
	  //     catch e
	  //     end
	  // end
	  // sort!(unique!(res))
	  // foreach(x -> println("\'", x, "\',"), res)
	  var BUILT_IN_LIST = [
	    'AbstractArray',
	    'AbstractChannel',
	    'AbstractChar',
	    'AbstractDict',
	    'AbstractDisplay',
	    'AbstractFloat',
	    'AbstractIrrational',
	    'AbstractMatrix',
	    'AbstractRange',
	    'AbstractSet',
	    'AbstractString',
	    'AbstractUnitRange',
	    'AbstractVecOrMat',
	    'AbstractVector',
	    'Any',
	    'ArgumentError',
	    'Array',
	    'AssertionError',
	    'BigFloat',
	    'BigInt',
	    'BitArray',
	    'BitMatrix',
	    'BitSet',
	    'BitVector',
	    'Bool',
	    'BoundsError',
	    'CapturedException',
	    'CartesianIndex',
	    'CartesianIndices',
	    'Cchar',
	    'Cdouble',
	    'Cfloat',
	    'Channel',
	    'Char',
	    'Cint',
	    'Cintmax_t',
	    'Clong',
	    'Clonglong',
	    'Cmd',
	    'Colon',
	    'Complex',
	    'ComplexF16',
	    'ComplexF32',
	    'ComplexF64',
	    'CompositeException',
	    'Condition',
	    'Cptrdiff_t',
	    'Cshort',
	    'Csize_t',
	    'Cssize_t',
	    'Cstring',
	    'Cuchar',
	    'Cuint',
	    'Cuintmax_t',
	    'Culong',
	    'Culonglong',
	    'Cushort',
	    'Cvoid',
	    'Cwchar_t',
	    'Cwstring',
	    'DataType',
	    'DenseArray',
	    'DenseMatrix',
	    'DenseVecOrMat',
	    'DenseVector',
	    'Dict',
	    'DimensionMismatch',
	    'Dims',
	    'DivideError',
	    'DomainError',
	    'EOFError',
	    'Enum',
	    'ErrorException',
	    'Exception',
	    'ExponentialBackOff',
	    'Expr',
	    'Float16',
	    'Float32',
	    'Float64',
	    'Function',
	    'GlobalRef',
	    'HTML',
	    'IO',
	    'IOBuffer',
	    'IOContext',
	    'IOStream',
	    'IdDict',
	    'IndexCartesian',
	    'IndexLinear',
	    'IndexStyle',
	    'InexactError',
	    'InitError',
	    'Int',
	    'Int128',
	    'Int16',
	    'Int32',
	    'Int64',
	    'Int8',
	    'Integer',
	    'InterruptException',
	    'InvalidStateException',
	    'Irrational',
	    'KeyError',
	    'LinRange',
	    'LineNumberNode',
	    'LinearIndices',
	    'LoadError',
	    'MIME',
	    'Matrix',
	    'Method',
	    'MethodError',
	    'Missing',
	    'MissingException',
	    'Module',
	    'NTuple',
	    'NamedTuple',
	    'Nothing',
	    'Number',
	    'OrdinalRange',
	    'OutOfMemoryError',
	    'OverflowError',
	    'Pair',
	    'PartialQuickSort',
	    'PermutedDimsArray',
	    'Pipe',
	    'ProcessFailedException',
	    'Ptr',
	    'QuoteNode',
	    'Rational',
	    'RawFD',
	    'ReadOnlyMemoryError',
	    'Real',
	    'ReentrantLock',
	    'Ref',
	    'Regex',
	    'RegexMatch',
	    'RoundingMode',
	    'SegmentationFault',
	    'Set',
	    'Signed',
	    'Some',
	    'StackOverflowError',
	    'StepRange',
	    'StepRangeLen',
	    'StridedArray',
	    'StridedMatrix',
	    'StridedVecOrMat',
	    'StridedVector',
	    'String',
	    'StringIndexError',
	    'SubArray',
	    'SubString',
	    'SubstitutionString',
	    'Symbol',
	    'SystemError',
	    'Task',
	    'TaskFailedException',
	    'Text',
	    'TextDisplay',
	    'Timer',
	    'Tuple',
	    'Type',
	    'TypeError',
	    'TypeVar',
	    'UInt',
	    'UInt128',
	    'UInt16',
	    'UInt32',
	    'UInt64',
	    'UInt8',
	    'UndefInitializer',
	    'UndefKeywordError',
	    'UndefRefError',
	    'UndefVarError',
	    'Union',
	    'UnionAll',
	    'UnitRange',
	    'Unsigned',
	    'Val',
	    'Vararg',
	    'VecElement',
	    'VecOrMat',
	    'Vector',
	    'VersionNumber',
	    'WeakKeyDict',
	    'WeakRef',
	  ];

	  var KEYWORDS = {
	    $pattern: VARIABLE_NAME_RE,
	    keyword: KEYWORD_LIST,
	    literal: LITERAL_LIST,
	    built_in: BUILT_IN_LIST,
	  };

	  // placeholder for recursive self-reference
	  var DEFAULT = {
	    keywords: KEYWORDS, illegal: /<\//
	  };

	  // ref: https://docs.julialang.org/en/v1/manual/integers-and-floating-point-numbers/
	  var NUMBER = {
	    className: 'number',
	    // supported numeric literals:
	    //  * binary literal (e.g. 0x10)
	    //  * octal literal (e.g. 0o76543210)
	    //  * hexadecimal literal (e.g. 0xfedcba876543210)
	    //  * hexadecimal floating point literal (e.g. 0x1p0, 0x1.2p2)
	    //  * decimal literal (e.g. 9876543210, 100_000_000)
	    //  * floating pointe literal (e.g. 1.2, 1.2f, .2, 1., 1.2e10, 1.2e-10)
	    begin: /(\b0x[\d_]*(\.[\d_]*)?|0x\.\d[\d_]*)p[-+]?\d+|\b0[box][a-fA-F0-9][a-fA-F0-9_]*|(\b\d[\d_]*(\.[\d_]*)?|\.\d[\d_]*)([eEfF][-+]?\d+)?/,
	    relevance: 0
	  };

	  var CHAR = {
	    className: 'string',
	    begin: /'(.|\\[xXuU][a-zA-Z0-9]+)'/
	  };

	  var INTERPOLATION = {
	    className: 'subst',
	    begin: /\$\(/, end: /\)/,
	    keywords: KEYWORDS
	  };

	  var INTERPOLATED_VARIABLE = {
	    className: 'variable',
	    begin: '\\$' + VARIABLE_NAME_RE
	  };

	  // TODO: neatly escape normal code in string literal
	  var STRING = {
	    className: 'string',
	    contains: [hljs.BACKSLASH_ESCAPE, INTERPOLATION, INTERPOLATED_VARIABLE],
	    variants: [
	      { begin: /\w*"""/, end: /"""\w*/, relevance: 10 },
	      { begin: /\w*"/, end: /"\w*/ }
	    ]
	  };

	  var COMMAND = {
	    className: 'string',
	    contains: [hljs.BACKSLASH_ESCAPE, INTERPOLATION, INTERPOLATED_VARIABLE],
	    begin: '`', end: '`'
	  };

	  var MACROCALL = {
	    className: 'meta',
	    begin: '@' + VARIABLE_NAME_RE
	  };

	  var COMMENT = {
	    className: 'comment',
	    variants: [
	      { begin: '#=', end: '=#', relevance: 10 },
	      { begin: '#', end: '$' }
	    ]
	  };

	  DEFAULT.name = 'Julia';
	  DEFAULT.contains = [
	    NUMBER,
	    CHAR,
	    STRING,
	    COMMAND,
	    MACROCALL,
	    COMMENT,
	    hljs.HASH_COMMENT_MODE,
	    {
	      className: 'keyword',
	      begin:
	        '\\b(((abstract|primitive)\\s+)type|(mutable\\s+)?struct)\\b'
	    },
	    {begin: /<:/}  // relevance booster
	  ];
	  INTERPOLATION.contains = DEFAULT.contains;

	  return DEFAULT;
	}

	julia_1 = julia;
	return julia_1;
}

/*
Language: Julia REPL
Description: Julia REPL sessions
Author: Morten Piibeleht <morten.piibeleht@gmail.com>
Website: https://julialang.org
Requires: julia.js

The Julia REPL code blocks look something like the following:

  julia> function foo(x)
             x + 1
         end
  foo (generic function with 1 method)

They start on a new line with "julia>". Usually there should also be a space after this, but
we also allow the code to start right after the > character. The code may run over multiple
lines, but the additional lines must start with six spaces (i.e. be indented to match
"julia>"). The rest of the code is assumed to be output from the executed code and will be
left un-highlighted.

Using simply spaces to identify line continuations may get a false-positive if the output
also prints out six spaces, but such cases should be rare.
*/

var juliaRepl_1;
var hasRequiredJuliaRepl;

function requireJuliaRepl () {
	if (hasRequiredJuliaRepl) return juliaRepl_1;
	hasRequiredJuliaRepl = 1;
	function juliaRepl(hljs) {
	  return {
	    name: 'Julia REPL',
	    contains: [
	      {
	        className: 'meta',
	        begin: /^julia>/,
	        relevance: 10,
	        starts: {
	          // end the highlighting if we are on a new line and the line does not have at
	          // least six spaces in the beginning
	          end: /^(?![ ]{6})/,
	          subLanguage: 'julia'
	      },
	      // jldoctest Markdown blocks are used in the Julia manual and package docs indicate
	      // code snippets that should be verified when the documentation is built. They can be
	      // either REPL-like or script-like, but are usually REPL-like and therefore we apply
	      // julia-repl highlighting to them. More information can be found in Documenter's
	      // manual: https://juliadocs.github.io/Documenter.jl/latest/man/doctests.html
	      aliases: ['jldoctest']
	      }
	    ]
	  }
	}

	juliaRepl_1 = juliaRepl;
	return juliaRepl_1;
}

var kotlin_1;
var hasRequiredKotlin;

function requireKotlin () {
	if (hasRequiredKotlin) return kotlin_1;
	hasRequiredKotlin = 1;
	// https://docs.oracle.com/javase/specs/jls/se15/html/jls-3.html#jls-3.10
	var decimalDigits = '[0-9](_*[0-9])*';
	var frac = `\\.(${decimalDigits})`;
	var hexDigits = '[0-9a-fA-F](_*[0-9a-fA-F])*';
	var NUMERIC = {
	  className: 'number',
	  variants: [
	    // DecimalFloatingPointLiteral
	    // including ExponentPart
	    { begin: `(\\b(${decimalDigits})((${frac})|\\.)?|(${frac}))` +
	      `[eE][+-]?(${decimalDigits})[fFdD]?\\b` },
	    // excluding ExponentPart
	    { begin: `\\b(${decimalDigits})((${frac})[fFdD]?\\b|\\.([fFdD]\\b)?)` },
	    { begin: `(${frac})[fFdD]?\\b` },
	    { begin: `\\b(${decimalDigits})[fFdD]\\b` },

	    // HexadecimalFloatingPointLiteral
	    { begin: `\\b0[xX]((${hexDigits})\\.?|(${hexDigits})?\\.(${hexDigits}))` +
	      `[pP][+-]?(${decimalDigits})[fFdD]?\\b` },

	    // DecimalIntegerLiteral
	    { begin: '\\b(0|[1-9](_*[0-9])*)[lL]?\\b' },

	    // HexIntegerLiteral
	    { begin: `\\b0[xX](${hexDigits})[lL]?\\b` },

	    // OctalIntegerLiteral
	    { begin: '\\b0(_*[0-7])*[lL]?\\b' },

	    // BinaryIntegerLiteral
	    { begin: '\\b0[bB][01](_*[01])*[lL]?\\b' },
	  ],
	  relevance: 0
	};

	/*
	 Language: Kotlin
	 Description: Kotlin is an OSS statically typed programming language that targets the JVM, Android, JavaScript and Native.
	 Author: Sergey Mashkov <cy6erGn0m@gmail.com>
	 Website: https://kotlinlang.org
	 Category: common
	 */

	function kotlin(hljs) {
	  const KEYWORDS = {
	    keyword:
	      'abstract as val var vararg get set class object open private protected public noinline ' +
	      'crossinline dynamic final enum if else do while for when throw try catch finally ' +
	      'import package is in fun override companion reified inline lateinit init ' +
	      'interface annotation data sealed internal infix operator out by constructor super ' +
	      'tailrec where const inner suspend typealias external expect actual',
	    built_in:
	      'Byte Short Char Int Long Boolean Float Double Void Unit Nothing',
	    literal:
	      'true false null'
	  };
	  const KEYWORDS_WITH_LABEL = {
	    className: 'keyword',
	    begin: /\b(break|continue|return|this)\b/,
	    starts: {
	      contains: [
	        {
	          className: 'symbol',
	          begin: /@\w+/
	        }
	      ]
	    }
	  };
	  const LABEL = {
	    className: 'symbol',
	    begin: hljs.UNDERSCORE_IDENT_RE + '@'
	  };

	  // for string templates
	  const SUBST = {
	    className: 'subst',
	    begin: /\$\{/,
	    end: /\}/,
	    contains: [ hljs.C_NUMBER_MODE ]
	  };
	  const VARIABLE = {
	    className: 'variable',
	    begin: '\\$' + hljs.UNDERSCORE_IDENT_RE
	  };
	  const STRING = {
	    className: 'string',
	    variants: [
	      {
	        begin: '"""',
	        end: '"""(?=[^"])',
	        contains: [
	          VARIABLE,
	          SUBST
	        ]
	      },
	      // Can't use built-in modes easily, as we want to use STRING in the meta
	      // context as 'meta-string' and there's no syntax to remove explicitly set
	      // classNames in built-in modes.
	      {
	        begin: '\'',
	        end: '\'',
	        illegal: /\n/,
	        contains: [ hljs.BACKSLASH_ESCAPE ]
	      },
	      {
	        begin: '"',
	        end: '"',
	        illegal: /\n/,
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          VARIABLE,
	          SUBST
	        ]
	      }
	    ]
	  };
	  SUBST.contains.push(STRING);

	  const ANNOTATION_USE_SITE = {
	    className: 'meta',
	    begin: '@(?:file|property|field|get|set|receiver|param|setparam|delegate)\\s*:(?:\\s*' + hljs.UNDERSCORE_IDENT_RE + ')?'
	  };
	  const ANNOTATION = {
	    className: 'meta',
	    begin: '@' + hljs.UNDERSCORE_IDENT_RE,
	    contains: [
	      {
	        begin: /\(/,
	        end: /\)/,
	        contains: [
	          hljs.inherit(STRING, {
	            className: 'meta-string'
	          })
	        ]
	      }
	    ]
	  };

	  // https://kotlinlang.org/docs/reference/whatsnew11.html#underscores-in-numeric-literals
	  // According to the doc above, the number mode of kotlin is the same as java 8,
	  // so the code below is copied from java.js
	  const KOTLIN_NUMBER_MODE = NUMERIC;
	  const KOTLIN_NESTED_COMMENT = hljs.COMMENT(
	    '/\\*', '\\*/',
	    {
	      contains: [ hljs.C_BLOCK_COMMENT_MODE ]
	    }
	  );
	  const KOTLIN_PAREN_TYPE = {
	    variants: [
	      {
	        className: 'type',
	        begin: hljs.UNDERSCORE_IDENT_RE
	      },
	      {
	        begin: /\(/,
	        end: /\)/,
	        contains: [] // defined later
	      }
	    ]
	  };
	  const KOTLIN_PAREN_TYPE2 = KOTLIN_PAREN_TYPE;
	  KOTLIN_PAREN_TYPE2.variants[1].contains = [ KOTLIN_PAREN_TYPE ];
	  KOTLIN_PAREN_TYPE.variants[1].contains = [ KOTLIN_PAREN_TYPE2 ];

	  return {
	    name: 'Kotlin',
	    aliases: [ 'kt', 'kts' ],
	    keywords: KEYWORDS,
	    contains: [
	      hljs.COMMENT(
	        '/\\*\\*',
	        '\\*/',
	        {
	          relevance: 0,
	          contains: [
	            {
	              className: 'doctag',
	              begin: '@[A-Za-z]+'
	            }
	          ]
	        }
	      ),
	      hljs.C_LINE_COMMENT_MODE,
	      KOTLIN_NESTED_COMMENT,
	      KEYWORDS_WITH_LABEL,
	      LABEL,
	      ANNOTATION_USE_SITE,
	      ANNOTATION,
	      {
	        className: 'function',
	        beginKeywords: 'fun',
	        end: '[(]|$',
	        returnBegin: true,
	        excludeEnd: true,
	        keywords: KEYWORDS,
	        relevance: 5,
	        contains: [
	          {
	            begin: hljs.UNDERSCORE_IDENT_RE + '\\s*\\(',
	            returnBegin: true,
	            relevance: 0,
	            contains: [ hljs.UNDERSCORE_TITLE_MODE ]
	          },
	          {
	            className: 'type',
	            begin: /</,
	            end: />/,
	            keywords: 'reified',
	            relevance: 0
	          },
	          {
	            className: 'params',
	            begin: /\(/,
	            end: /\)/,
	            endsParent: true,
	            keywords: KEYWORDS,
	            relevance: 0,
	            contains: [
	              {
	                begin: /:/,
	                end: /[=,\/]/,
	                endsWithParent: true,
	                contains: [
	                  KOTLIN_PAREN_TYPE,
	                  hljs.C_LINE_COMMENT_MODE,
	                  KOTLIN_NESTED_COMMENT
	                ],
	                relevance: 0
	              },
	              hljs.C_LINE_COMMENT_MODE,
	              KOTLIN_NESTED_COMMENT,
	              ANNOTATION_USE_SITE,
	              ANNOTATION,
	              STRING,
	              hljs.C_NUMBER_MODE
	            ]
	          },
	          KOTLIN_NESTED_COMMENT
	        ]
	      },
	      {
	        className: 'class',
	        beginKeywords: 'class interface trait', // remove 'trait' when removed from KEYWORDS
	        end: /[:\{(]|$/,
	        excludeEnd: true,
	        illegal: 'extends implements',
	        contains: [
	          {
	            beginKeywords: 'public protected internal private constructor'
	          },
	          hljs.UNDERSCORE_TITLE_MODE,
	          {
	            className: 'type',
	            begin: /</,
	            end: />/,
	            excludeBegin: true,
	            excludeEnd: true,
	            relevance: 0
	          },
	          {
	            className: 'type',
	            begin: /[,:]\s*/,
	            end: /[<\(,]|$/,
	            excludeBegin: true,
	            returnEnd: true
	          },
	          ANNOTATION_USE_SITE,
	          ANNOTATION
	        ]
	      },
	      STRING,
	      {
	        className: 'meta',
	        begin: "^#!/usr/bin/env",
	        end: '$',
	        illegal: '\n'
	      },
	      KOTLIN_NUMBER_MODE
	    ]
	  };
	}

	kotlin_1 = kotlin;
	return kotlin_1;
}

/*
Language: Lasso
Author: Eric Knibbe <eric@lassosoft.com>
Description: Lasso is a language and server platform for database-driven web applications. This definition handles Lasso 9 syntax and LassoScript for Lasso 8.6 and earlier.
Website: http://www.lassosoft.com/What-Is-Lasso
*/

var lasso_1;
var hasRequiredLasso;

function requireLasso () {
	if (hasRequiredLasso) return lasso_1;
	hasRequiredLasso = 1;
	function lasso(hljs) {
	  const LASSO_IDENT_RE = '[a-zA-Z_][\\w.]*';
	  const LASSO_ANGLE_RE = '<\\?(lasso(script)?|=)';
	  const LASSO_CLOSE_RE = '\\]|\\?>';
	  const LASSO_KEYWORDS = {
	    $pattern: LASSO_IDENT_RE + '|&[lg]t;',
	    literal:
	      'true false none minimal full all void and or not ' +
	      'bw nbw ew new cn ncn lt lte gt gte eq neq rx nrx ft',
	    built_in:
	      'array date decimal duration integer map pair string tag xml null ' +
	      'boolean bytes keyword list locale queue set stack staticarray ' +
	      'local var variable global data self inherited currentcapture givenblock',
	    keyword:
	      'cache database_names database_schemanames database_tablenames ' +
	      'define_tag define_type email_batch encode_set html_comment handle ' +
	      'handle_error header if inline iterate ljax_target link ' +
	      'link_currentaction link_currentgroup link_currentrecord link_detail ' +
	      'link_firstgroup link_firstrecord link_lastgroup link_lastrecord ' +
	      'link_nextgroup link_nextrecord link_prevgroup link_prevrecord log ' +
	      'loop namespace_using output_none portal private protect records ' +
	      'referer referrer repeating resultset rows search_args ' +
	      'search_arguments select sort_args sort_arguments thread_atomic ' +
	      'value_list while abort case else fail_if fail_ifnot fail if_empty ' +
	      'if_false if_null if_true loop_abort loop_continue loop_count params ' +
	      'params_up return return_value run_children soap_definetag ' +
	      'soap_lastrequest soap_lastresponse tag_name ascending average by ' +
	      'define descending do equals frozen group handle_failure import in ' +
	      'into join let match max min on order parent protected provide public ' +
	      'require returnhome skip split_thread sum take thread to trait type ' +
	      'where with yield yieldhome'
	  };
	  const HTML_COMMENT = hljs.COMMENT(
	    '<!--',
	    '-->',
	    {
	      relevance: 0
	    }
	  );
	  const LASSO_NOPROCESS = {
	    className: 'meta',
	    begin: '\\[noprocess\\]',
	    starts: {
	      end: '\\[/noprocess\\]',
	      returnEnd: true,
	      contains: [HTML_COMMENT]
	    }
	  };
	  const LASSO_START = {
	    className: 'meta',
	    begin: '\\[/noprocess|' + LASSO_ANGLE_RE
	  };
	  const LASSO_DATAMEMBER = {
	    className: 'symbol',
	    begin: '\'' + LASSO_IDENT_RE + '\''
	  };
	  const LASSO_CODE = [
	    hljs.C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    hljs.inherit(hljs.C_NUMBER_MODE, {
	      begin: hljs.C_NUMBER_RE + '|(-?infinity|NaN)\\b'
	    }),
	    hljs.inherit(hljs.APOS_STRING_MODE, {
	      illegal: null
	    }),
	    hljs.inherit(hljs.QUOTE_STRING_MODE, {
	      illegal: null
	    }),
	    {
	      className: 'string',
	      begin: '`',
	      end: '`'
	    },
	    { // variables
	      variants: [
	        {
	          begin: '[#$]' + LASSO_IDENT_RE
	        },
	        {
	          begin: '#',
	          end: '\\d+',
	          illegal: '\\W'
	        }
	      ]
	    },
	    {
	      className: 'type',
	      begin: '::\\s*',
	      end: LASSO_IDENT_RE,
	      illegal: '\\W'
	    },
	    {
	      className: 'params',
	      variants: [
	        {
	          begin: '-(?!infinity)' + LASSO_IDENT_RE,
	          relevance: 0
	        },
	        {
	          begin: '(\\.\\.\\.)'
	        }
	      ]
	    },
	    {
	      begin: /(->|\.)\s*/,
	      relevance: 0,
	      contains: [LASSO_DATAMEMBER]
	    },
	    {
	      className: 'class',
	      beginKeywords: 'define',
	      returnEnd: true,
	      end: '\\(|=>',
	      contains: [
	        hljs.inherit(hljs.TITLE_MODE, {
	          begin: LASSO_IDENT_RE + '(=(?!>))?|[-+*/%](?!>)'
	        })
	      ]
	    }
	  ];
	  return {
	    name: 'Lasso',
	    aliases: [
	      'ls',
	      'lassoscript'
	    ],
	    case_insensitive: true,
	    keywords: LASSO_KEYWORDS,
	    contains: [
	      {
	        className: 'meta',
	        begin: LASSO_CLOSE_RE,
	        relevance: 0,
	        starts: { // markup
	          end: '\\[|' + LASSO_ANGLE_RE,
	          returnEnd: true,
	          relevance: 0,
	          contains: [HTML_COMMENT]
	        }
	      },
	      LASSO_NOPROCESS,
	      LASSO_START,
	      {
	        className: 'meta',
	        begin: '\\[no_square_brackets',
	        starts: {
	          end: '\\[/no_square_brackets\\]', // not implemented in the language
	          keywords: LASSO_KEYWORDS,
	          contains: [
	            {
	              className: 'meta',
	              begin: LASSO_CLOSE_RE,
	              relevance: 0,
	              starts: {
	                end: '\\[noprocess\\]|' + LASSO_ANGLE_RE,
	                returnEnd: true,
	                contains: [HTML_COMMENT]
	              }
	            },
	            LASSO_NOPROCESS,
	            LASSO_START
	          ].concat(LASSO_CODE)
	        }
	      },
	      {
	        className: 'meta',
	        begin: '\\[',
	        relevance: 0
	      },
	      {
	        className: 'meta',
	        begin: '^#!',
	        end: 'lasso9$',
	        relevance: 10
	      }
	    ].concat(LASSO_CODE)
	  };
	}

	lasso_1 = lasso;
	return lasso_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var latex_1;
var hasRequiredLatex;

function requireLatex () {
	if (hasRequiredLatex) return latex_1;
	hasRequiredLatex = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/*
	Language: LaTeX
	Author: Benedikt Wilde <bwilde@posteo.de>
	Website: https://www.latex-project.org
	Category: markup
	*/

	/** @type LanguageFn */
	function latex(hljs) {
	  const KNOWN_CONTROL_WORDS = either(...[
	      '(?:NeedsTeXFormat|RequirePackage|GetIdInfo)',
	      'Provides(?:Expl)?(?:Package|Class|File)',
	      '(?:DeclareOption|ProcessOptions)',
	      '(?:documentclass|usepackage|input|include)',
	      'makeat(?:letter|other)',
	      'ExplSyntax(?:On|Off)',
	      '(?:new|renew|provide)?command',
	      '(?:re)newenvironment',
	      '(?:New|Renew|Provide|Declare)(?:Expandable)?DocumentCommand',
	      '(?:New|Renew|Provide|Declare)DocumentEnvironment',
	      '(?:(?:e|g|x)?def|let)',
	      '(?:begin|end)',
	      '(?:part|chapter|(?:sub){0,2}section|(?:sub)?paragraph)',
	      'caption',
	      '(?:label|(?:eq|page|name)?ref|(?:paren|foot|super)?cite)',
	      '(?:alpha|beta|[Gg]amma|[Dd]elta|(?:var)?epsilon|zeta|eta|[Tt]heta|vartheta)',
	      '(?:iota|(?:var)?kappa|[Ll]ambda|mu|nu|[Xx]i|[Pp]i|varpi|(?:var)rho)',
	      '(?:[Ss]igma|varsigma|tau|[Uu]psilon|[Pp]hi|varphi|chi|[Pp]si|[Oo]mega)',
	      '(?:frac|sum|prod|lim|infty|times|sqrt|leq|geq|left|right|middle|[bB]igg?)',
	      '(?:[lr]angle|q?quad|[lcvdi]?dots|d?dot|hat|tilde|bar)'
	    ].map(word => word + '(?![a-zA-Z@:_])'));
	  const L3_REGEX = new RegExp([
	      // A function \module_function_name:signature or \__module_function_name:signature,
	      // where both module and function_name need at least two characters and
	      // function_name may contain single underscores.
	      '(?:__)?[a-zA-Z]{2,}_[a-zA-Z](?:_?[a-zA-Z])+:[a-zA-Z]*',
	      // A variable \scope_module_and_name_type or \scope__module_ane_name_type,
	      // where scope is one of l, g or c, type needs at least two characters
	      // and module_and_name may contain single underscores.
	      '[lgc]__?[a-zA-Z](?:_?[a-zA-Z])*_[a-zA-Z]{2,}',
	      // A quark \q_the_name or \q__the_name or
	      // scan mark \s_the_name or \s__vthe_name,
	      // where variable_name needs at least two characters and
	      // may contain single underscores.
	      '[qs]__?[a-zA-Z](?:_?[a-zA-Z])+',
	      // Other LaTeX3 macro names that are not covered by the three rules above.
	      'use(?:_i)?:[a-zA-Z]*',
	      '(?:else|fi|or):',
	      '(?:if|cs|exp):w',
	      '(?:hbox|vbox):n',
	      '::[a-zA-Z]_unbraced',
	      '::[a-zA-Z:]'
	    ].map(pattern => pattern + '(?![a-zA-Z:_])').join('|'));
	  const L2_VARIANTS = [
	    {begin: /[a-zA-Z@]+/}, // control word
	    {begin: /[^a-zA-Z@]?/} // control symbol
	  ];
	  const DOUBLE_CARET_VARIANTS = [
	    {begin: /\^{6}[0-9a-f]{6}/},
	    {begin: /\^{5}[0-9a-f]{5}/},
	    {begin: /\^{4}[0-9a-f]{4}/},
	    {begin: /\^{3}[0-9a-f]{3}/},
	    {begin: /\^{2}[0-9a-f]{2}/},
	    {begin: /\^{2}[\u0000-\u007f]/}
	  ];
	  const CONTROL_SEQUENCE = {
	    className: 'keyword',
	    begin: /\\/,
	    relevance: 0,
	    contains: [
	      {
	        endsParent: true,
	        begin: KNOWN_CONTROL_WORDS
	      },
	      {
	        endsParent: true,
	        begin: L3_REGEX
	      },
	      {
	        endsParent: true,
	        variants: DOUBLE_CARET_VARIANTS
	      },
	      {
	        endsParent: true,
	        relevance: 0,
	        variants: L2_VARIANTS
	      }
	    ]
	  };
	  const MACRO_PARAM = {
	    className: 'params',
	    relevance: 0,
	    begin: /#+\d?/
	  };
	  const DOUBLE_CARET_CHAR = {
	    // relevance: 1
	    variants: DOUBLE_CARET_VARIANTS
	  };
	  const SPECIAL_CATCODE = {
	    className: 'built_in',
	    relevance: 0,
	    begin: /[$&^_]/
	  };
	  const MAGIC_COMMENT = {
	    className: 'meta',
	    begin: '% !TeX',
	    end: '$',
	    relevance: 10
	  };
	  const COMMENT = hljs.COMMENT(
	    '%',
	    '$',
	    {
	      relevance: 0
	    }
	  );
	  const EVERYTHING_BUT_VERBATIM = [
	    CONTROL_SEQUENCE,
	    MACRO_PARAM,
	    DOUBLE_CARET_CHAR,
	    SPECIAL_CATCODE,
	    MAGIC_COMMENT,
	    COMMENT
	  ];
	  const BRACE_GROUP_NO_VERBATIM = {
	    begin: /\{/, end: /\}/,
	    relevance: 0,
	    contains: ['self', ...EVERYTHING_BUT_VERBATIM]
	  };
	  const ARGUMENT_BRACES = hljs.inherit(
	    BRACE_GROUP_NO_VERBATIM,
	    {
	      relevance: 0,
	      endsParent: true,
	      contains: [BRACE_GROUP_NO_VERBATIM, ...EVERYTHING_BUT_VERBATIM]
	    }
	  );
	  const ARGUMENT_BRACKETS = {
	    begin: /\[/,
	      end: /\]/,
	    endsParent: true,
	    relevance: 0,
	    contains: [BRACE_GROUP_NO_VERBATIM, ...EVERYTHING_BUT_VERBATIM]
	  };
	  const SPACE_GOBBLER = {
	    begin: /\s+/,
	    relevance: 0
	  };
	  const ARGUMENT_M = [ARGUMENT_BRACES];
	  const ARGUMENT_O = [ARGUMENT_BRACKETS];
	  const ARGUMENT_AND_THEN = function(arg, starts_mode) {
	    return {
	      contains: [SPACE_GOBBLER],
	      starts: {
	        relevance: 0,
	        contains: arg,
	        starts: starts_mode
	      }
	    };
	  };
	  const CSNAME = function(csname, starts_mode) {
	    return {
	        begin: '\\\\' + csname + '(?![a-zA-Z@:_])',
	        keywords: {$pattern: /\\[a-zA-Z]+/, keyword: '\\' + csname},
	        relevance: 0,
	        contains: [SPACE_GOBBLER],
	        starts: starts_mode
	      };
	  };
	  const BEGIN_ENV = function(envname, starts_mode) {
	    return hljs.inherit(
	      {
	        begin: '\\\\begin(?=[ \t]*(\\r?\\n[ \t]*)?\\{' + envname + '\\})',
	        keywords: {$pattern: /\\[a-zA-Z]+/, keyword: '\\begin'},
	        relevance: 0,
	      },
	      ARGUMENT_AND_THEN(ARGUMENT_M, starts_mode)
	    );
	  };
	  const VERBATIM_DELIMITED_EQUAL = (innerName = "string") => {
	    return hljs.END_SAME_AS_BEGIN({
	      className: innerName,
	      begin: /(.|\r?\n)/,
	      end: /(.|\r?\n)/,
	      excludeBegin: true,
	      excludeEnd: true,
	      endsParent: true
	    });
	  };
	  const VERBATIM_DELIMITED_ENV = function(envname) {
	    return {
	      className: 'string',
	      end: '(?=\\\\end\\{' + envname + '\\})'
	    };
	  };

	  const VERBATIM_DELIMITED_BRACES = (innerName = "string") => {
	    return {
	      relevance: 0,
	      begin: /\{/,
	      starts: {
	        endsParent: true,
	        contains: [
	          {
	            className: innerName,
	            end: /(?=\})/,
	            endsParent:true,
	            contains: [
	              {
	                begin: /\{/,
	                end: /\}/,
	                relevance: 0,
	                contains: ["self"]
	              }
	            ],
	          }
	        ]
	      }
	    };
	  };
	  const VERBATIM = [
	    ...['verb', 'lstinline'].map(csname => CSNAME(csname, {contains: [VERBATIM_DELIMITED_EQUAL()]})),
	    CSNAME('mint', ARGUMENT_AND_THEN(ARGUMENT_M, {contains: [VERBATIM_DELIMITED_EQUAL()]})),
	    CSNAME('mintinline', ARGUMENT_AND_THEN(ARGUMENT_M, {contains: [VERBATIM_DELIMITED_BRACES(), VERBATIM_DELIMITED_EQUAL()]})),
	    CSNAME('url', {contains: [VERBATIM_DELIMITED_BRACES("link"), VERBATIM_DELIMITED_BRACES("link")]}),
	    CSNAME('hyperref', {contains: [VERBATIM_DELIMITED_BRACES("link")]}),
	    CSNAME('href', ARGUMENT_AND_THEN(ARGUMENT_O, {contains: [VERBATIM_DELIMITED_BRACES("link")]})),
	    ...[].concat(...['', '\\*'].map(suffix => [
	      BEGIN_ENV('verbatim' + suffix, VERBATIM_DELIMITED_ENV('verbatim' + suffix)),
	      BEGIN_ENV('filecontents' + suffix,  ARGUMENT_AND_THEN(ARGUMENT_M, VERBATIM_DELIMITED_ENV('filecontents' + suffix))),
	      ...['', 'B', 'L'].map(prefix =>
	        BEGIN_ENV(prefix + 'Verbatim' + suffix, ARGUMENT_AND_THEN(ARGUMENT_O, VERBATIM_DELIMITED_ENV(prefix + 'Verbatim' + suffix)))
	      )
	    ])),
	    BEGIN_ENV('minted', ARGUMENT_AND_THEN(ARGUMENT_O, ARGUMENT_AND_THEN(ARGUMENT_M, VERBATIM_DELIMITED_ENV('minted')))),
	  ];

	  return {
	    name: 'LaTeX',
	    aliases: ['tex'],
	    contains: [
	      ...VERBATIM,
	      ...EVERYTHING_BUT_VERBATIM
	    ]
	  };
	}

	latex_1 = latex;
	return latex_1;
}

/*
Language: LDIF
Contributors: Jacob Childress <jacobc@gmail.com>
Category: enterprise, config
Website: https://en.wikipedia.org/wiki/LDAP_Data_Interchange_Format
*/

var ldif_1;
var hasRequiredLdif;

function requireLdif () {
	if (hasRequiredLdif) return ldif_1;
	hasRequiredLdif = 1;
	function ldif(hljs) {
	  return {
	    name: 'LDIF',
	    contains: [
	      {
	        className: 'attribute',
	        begin: '^dn',
	        end: ': ',
	        excludeEnd: true,
	        starts: {
	          end: '$',
	          relevance: 0
	        },
	        relevance: 10
	      },
	      {
	        className: 'attribute',
	        begin: '^\\w',
	        end: ': ',
	        excludeEnd: true,
	        starts: {
	          end: '$',
	          relevance: 0
	        }
	      },
	      {
	        className: 'literal',
	        begin: '^-',
	        end: '$'
	      },
	      hljs.HASH_COMMENT_MODE
	    ]
	  };
	}

	ldif_1 = ldif;
	return ldif_1;
}

/*
Language: Leaf
Author: Hale Chan <halechan@qq.com>
Description: Based on the Leaf reference from https://vapor.github.io/documentation/guide/leaf.html.
*/

var leaf_1;
var hasRequiredLeaf;

function requireLeaf () {
	if (hasRequiredLeaf) return leaf_1;
	hasRequiredLeaf = 1;
	function leaf(hljs) {
	  return {
	    name: 'Leaf',
	    contains: [
	      {
	        className: 'function',
	        begin: '#+' + '[A-Za-z_0-9]*' + '\\(',
	        end: / \{/,
	        returnBegin: true,
	        excludeEnd: true,
	        contains: [
	          {
	            className: 'keyword',
	            begin: '#+'
	          },
	          {
	            className: 'title',
	            begin: '[A-Za-z_][A-Za-z_0-9]*'
	          },
	          {
	            className: 'params',
	            begin: '\\(',
	            end: '\\)',
	            endsParent: true,
	            contains: [
	              {
	                className: 'string',
	                begin: '"',
	                end: '"'
	              },
	              {
	                className: 'variable',
	                begin: '[A-Za-z_][A-Za-z_0-9]*'
	              }
	            ]
	          }
	        ]
	      }
	    ]
	  };
	}

	leaf_1 = leaf;
	return leaf_1;
}

var less_1;
var hasRequiredLess;

function requireLess () {
	if (hasRequiredLess) return less_1;
	hasRequiredLess = 1;
	const MODES = (hljs) => {
	  return {
	    IMPORTANT: {
	      className: 'meta',
	      begin: '!important'
	    },
	    HEXCOLOR: {
	      className: 'number',
	      begin: '#([a-fA-F0-9]{6}|[a-fA-F0-9]{3})'
	    },
	    ATTRIBUTE_SELECTOR_MODE: {
	      className: 'selector-attr',
	      begin: /\[/,
	      end: /\]/,
	      illegal: '$',
	      contains: [
	        hljs.APOS_STRING_MODE,
	        hljs.QUOTE_STRING_MODE
	      ]
	    }
	  };
	};

	const TAGS = [
	  'a',
	  'abbr',
	  'address',
	  'article',
	  'aside',
	  'audio',
	  'b',
	  'blockquote',
	  'body',
	  'button',
	  'canvas',
	  'caption',
	  'cite',
	  'code',
	  'dd',
	  'del',
	  'details',
	  'dfn',
	  'div',
	  'dl',
	  'dt',
	  'em',
	  'fieldset',
	  'figcaption',
	  'figure',
	  'footer',
	  'form',
	  'h1',
	  'h2',
	  'h3',
	  'h4',
	  'h5',
	  'h6',
	  'header',
	  'hgroup',
	  'html',
	  'i',
	  'iframe',
	  'img',
	  'input',
	  'ins',
	  'kbd',
	  'label',
	  'legend',
	  'li',
	  'main',
	  'mark',
	  'menu',
	  'nav',
	  'object',
	  'ol',
	  'p',
	  'q',
	  'quote',
	  'samp',
	  'section',
	  'span',
	  'strong',
	  'summary',
	  'sup',
	  'table',
	  'tbody',
	  'td',
	  'textarea',
	  'tfoot',
	  'th',
	  'thead',
	  'time',
	  'tr',
	  'ul',
	  'var',
	  'video'
	];

	const MEDIA_FEATURES = [
	  'any-hover',
	  'any-pointer',
	  'aspect-ratio',
	  'color',
	  'color-gamut',
	  'color-index',
	  'device-aspect-ratio',
	  'device-height',
	  'device-width',
	  'display-mode',
	  'forced-colors',
	  'grid',
	  'height',
	  'hover',
	  'inverted-colors',
	  'monochrome',
	  'orientation',
	  'overflow-block',
	  'overflow-inline',
	  'pointer',
	  'prefers-color-scheme',
	  'prefers-contrast',
	  'prefers-reduced-motion',
	  'prefers-reduced-transparency',
	  'resolution',
	  'scan',
	  'scripting',
	  'update',
	  'width',
	  // TODO: find a better solution?
	  'min-width',
	  'max-width',
	  'min-height',
	  'max-height'
	];

	// https://developer.mozilla.org/en-US/docs/Web/CSS/Pseudo-classes
	const PSEUDO_CLASSES = [
	  'active',
	  'any-link',
	  'blank',
	  'checked',
	  'current',
	  'default',
	  'defined',
	  'dir', // dir()
	  'disabled',
	  'drop',
	  'empty',
	  'enabled',
	  'first',
	  'first-child',
	  'first-of-type',
	  'fullscreen',
	  'future',
	  'focus',
	  'focus-visible',
	  'focus-within',
	  'has', // has()
	  'host', // host or host()
	  'host-context', // host-context()
	  'hover',
	  'indeterminate',
	  'in-range',
	  'invalid',
	  'is', // is()
	  'lang', // lang()
	  'last-child',
	  'last-of-type',
	  'left',
	  'link',
	  'local-link',
	  'not', // not()
	  'nth-child', // nth-child()
	  'nth-col', // nth-col()
	  'nth-last-child', // nth-last-child()
	  'nth-last-col', // nth-last-col()
	  'nth-last-of-type', //nth-last-of-type()
	  'nth-of-type', //nth-of-type()
	  'only-child',
	  'only-of-type',
	  'optional',
	  'out-of-range',
	  'past',
	  'placeholder-shown',
	  'read-only',
	  'read-write',
	  'required',
	  'right',
	  'root',
	  'scope',
	  'target',
	  'target-within',
	  'user-invalid',
	  'valid',
	  'visited',
	  'where' // where()
	];

	// https://developer.mozilla.org/en-US/docs/Web/CSS/Pseudo-elements
	const PSEUDO_ELEMENTS = [
	  'after',
	  'backdrop',
	  'before',
	  'cue',
	  'cue-region',
	  'first-letter',
	  'first-line',
	  'grammar-error',
	  'marker',
	  'part',
	  'placeholder',
	  'selection',
	  'slotted',
	  'spelling-error'
	];

	const ATTRIBUTES = [
	  'align-content',
	  'align-items',
	  'align-self',
	  'animation',
	  'animation-delay',
	  'animation-direction',
	  'animation-duration',
	  'animation-fill-mode',
	  'animation-iteration-count',
	  'animation-name',
	  'animation-play-state',
	  'animation-timing-function',
	  'auto',
	  'backface-visibility',
	  'background',
	  'background-attachment',
	  'background-clip',
	  'background-color',
	  'background-image',
	  'background-origin',
	  'background-position',
	  'background-repeat',
	  'background-size',
	  'border',
	  'border-bottom',
	  'border-bottom-color',
	  'border-bottom-left-radius',
	  'border-bottom-right-radius',
	  'border-bottom-style',
	  'border-bottom-width',
	  'border-collapse',
	  'border-color',
	  'border-image',
	  'border-image-outset',
	  'border-image-repeat',
	  'border-image-slice',
	  'border-image-source',
	  'border-image-width',
	  'border-left',
	  'border-left-color',
	  'border-left-style',
	  'border-left-width',
	  'border-radius',
	  'border-right',
	  'border-right-color',
	  'border-right-style',
	  'border-right-width',
	  'border-spacing',
	  'border-style',
	  'border-top',
	  'border-top-color',
	  'border-top-left-radius',
	  'border-top-right-radius',
	  'border-top-style',
	  'border-top-width',
	  'border-width',
	  'bottom',
	  'box-decoration-break',
	  'box-shadow',
	  'box-sizing',
	  'break-after',
	  'break-before',
	  'break-inside',
	  'caption-side',
	  'clear',
	  'clip',
	  'clip-path',
	  'color',
	  'column-count',
	  'column-fill',
	  'column-gap',
	  'column-rule',
	  'column-rule-color',
	  'column-rule-style',
	  'column-rule-width',
	  'column-span',
	  'column-width',
	  'columns',
	  'content',
	  'counter-increment',
	  'counter-reset',
	  'cursor',
	  'direction',
	  'display',
	  'empty-cells',
	  'filter',
	  'flex',
	  'flex-basis',
	  'flex-direction',
	  'flex-flow',
	  'flex-grow',
	  'flex-shrink',
	  'flex-wrap',
	  'float',
	  'font',
	  'font-display',
	  'font-family',
	  'font-feature-settings',
	  'font-kerning',
	  'font-language-override',
	  'font-size',
	  'font-size-adjust',
	  'font-smoothing',
	  'font-stretch',
	  'font-style',
	  'font-variant',
	  'font-variant-ligatures',
	  'font-variation-settings',
	  'font-weight',
	  'height',
	  'hyphens',
	  'icon',
	  'image-orientation',
	  'image-rendering',
	  'image-resolution',
	  'ime-mode',
	  'inherit',
	  'initial',
	  'justify-content',
	  'left',
	  'letter-spacing',
	  'line-height',
	  'list-style',
	  'list-style-image',
	  'list-style-position',
	  'list-style-type',
	  'margin',
	  'margin-bottom',
	  'margin-left',
	  'margin-right',
	  'margin-top',
	  'marks',
	  'mask',
	  'max-height',
	  'max-width',
	  'min-height',
	  'min-width',
	  'nav-down',
	  'nav-index',
	  'nav-left',
	  'nav-right',
	  'nav-up',
	  'none',
	  'normal',
	  'object-fit',
	  'object-position',
	  'opacity',
	  'order',
	  'orphans',
	  'outline',
	  'outline-color',
	  'outline-offset',
	  'outline-style',
	  'outline-width',
	  'overflow',
	  'overflow-wrap',
	  'overflow-x',
	  'overflow-y',
	  'padding',
	  'padding-bottom',
	  'padding-left',
	  'padding-right',
	  'padding-top',
	  'page-break-after',
	  'page-break-before',
	  'page-break-inside',
	  'perspective',
	  'perspective-origin',
	  'pointer-events',
	  'position',
	  'quotes',
	  'resize',
	  'right',
	  'src', // @font-face
	  'tab-size',
	  'table-layout',
	  'text-align',
	  'text-align-last',
	  'text-decoration',
	  'text-decoration-color',
	  'text-decoration-line',
	  'text-decoration-style',
	  'text-indent',
	  'text-overflow',
	  'text-rendering',
	  'text-shadow',
	  'text-transform',
	  'text-underline-position',
	  'top',
	  'transform',
	  'transform-origin',
	  'transform-style',
	  'transition',
	  'transition-delay',
	  'transition-duration',
	  'transition-property',
	  'transition-timing-function',
	  'unicode-bidi',
	  'vertical-align',
	  'visibility',
	  'white-space',
	  'widows',
	  'width',
	  'word-break',
	  'word-spacing',
	  'word-wrap',
	  'z-index'
	  // reverse makes sure longer attributes `font-weight` are matched fully
	  // instead of getting false positives on say `font`
	].reverse();

	// some grammars use them all as a single group
	const PSEUDO_SELECTORS = PSEUDO_CLASSES.concat(PSEUDO_ELEMENTS);

	/*
	Language: Less
	Description: It's CSS, with just a little more.
	Author:   Max Mikhailov <seven.phases.max@gmail.com>
	Website: http://lesscss.org
	Category: common, css
	*/

	/** @type LanguageFn */
	function less(hljs) {
	  const modes = MODES(hljs);
	  const PSEUDO_SELECTORS$1 = PSEUDO_SELECTORS;

	  const AT_MODIFIERS = "and or not only";
	  const IDENT_RE = '[\\w-]+'; // yes, Less identifiers may begin with a digit
	  const INTERP_IDENT_RE = '(' + IDENT_RE + '|@\\{' + IDENT_RE + '\\})';

	  /* Generic Modes */

	  const RULES = []; const VALUE_MODES = []; // forward def. for recursive modes

	  const STRING_MODE = function(c) {
	    return {
	    // Less strings are not multiline (also include '~' for more consistent coloring of "escaped" strings)
	      className: 'string',
	      begin: '~?' + c + '.*?' + c
	    };
	  };

	  const IDENT_MODE = function(name, begin, relevance) {
	    return {
	      className: name,
	      begin: begin,
	      relevance: relevance
	    };
	  };

	  const AT_KEYWORDS = {
	    $pattern: /[a-z-]+/,
	    keyword: AT_MODIFIERS,
	    attribute: MEDIA_FEATURES.join(" ")
	  };

	  const PARENS_MODE = {
	    // used only to properly balance nested parens inside mixin call, def. arg list
	    begin: '\\(',
	    end: '\\)',
	    contains: VALUE_MODES,
	    keywords: AT_KEYWORDS,
	    relevance: 0
	  };

	  // generic Less highlighter (used almost everywhere except selectors):
	  VALUE_MODES.push(
	    hljs.C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    STRING_MODE("'"),
	    STRING_MODE('"'),
	    hljs.CSS_NUMBER_MODE, // fixme: it does not include dot for numbers like .5em :(
	    {
	      begin: '(url|data-uri)\\(',
	      starts: {
	        className: 'string',
	        end: '[\\)\\n]',
	        excludeEnd: true
	      }
	    },
	    modes.HEXCOLOR,
	    PARENS_MODE,
	    IDENT_MODE('variable', '@@?' + IDENT_RE, 10),
	    IDENT_MODE('variable', '@\\{' + IDENT_RE + '\\}'),
	    IDENT_MODE('built_in', '~?`[^`]*?`'), // inline javascript (or whatever host language) *multiline* string
	    { // @media features (it’s here to not duplicate things in AT_RULE_MODE with extra PARENS_MODE overriding):
	      className: 'attribute',
	      begin: IDENT_RE + '\\s*:',
	      end: ':',
	      returnBegin: true,
	      excludeEnd: true
	    },
	    modes.IMPORTANT
	  );

	  const VALUE_WITH_RULESETS = VALUE_MODES.concat({
	    begin: /\{/,
	    end: /\}/,
	    contains: RULES
	  });

	  const MIXIN_GUARD_MODE = {
	    beginKeywords: 'when',
	    endsWithParent: true,
	    contains: [
	      {
	        beginKeywords: 'and not'
	      }
	    ].concat(VALUE_MODES) // using this form to override VALUE’s 'function' match
	  };

	  /* Rule-Level Modes */

	  const RULE_MODE = {
	    begin: INTERP_IDENT_RE + '\\s*:',
	    returnBegin: true,
	    end: /[;}]/,
	    relevance: 0,
	    contains: [
	      {
	        begin: /-(webkit|moz|ms|o)-/
	      },
	      {
	        className: 'attribute',
	        begin: '\\b(' + ATTRIBUTES.join('|') + ')\\b',
	        end: /(?=:)/,
	        starts: {
	          endsWithParent: true,
	          illegal: '[<=$]',
	          relevance: 0,
	          contains: VALUE_MODES
	        }
	      }
	    ]
	  };

	  const AT_RULE_MODE = {
	    className: 'keyword',
	    begin: '@(import|media|charset|font-face|(-[a-z]+-)?keyframes|supports|document|namespace|page|viewport|host)\\b',
	    starts: {
	      end: '[;{}]',
	      keywords: AT_KEYWORDS,
	      returnEnd: true,
	      contains: VALUE_MODES,
	      relevance: 0
	    }
	  };

	  // variable definitions and calls
	  const VAR_RULE_MODE = {
	    className: 'variable',
	    variants: [
	      // using more strict pattern for higher relevance to increase chances of Less detection.
	      // this is *the only* Less specific statement used in most of the sources, so...
	      // (we’ll still often loose to the css-parser unless there's '//' comment,
	      // simply because 1 variable just can't beat 99 properties :)
	      {
	        begin: '@' + IDENT_RE + '\\s*:',
	        relevance: 15
	      },
	      {
	        begin: '@' + IDENT_RE
	      }
	    ],
	    starts: {
	      end: '[;}]',
	      returnEnd: true,
	      contains: VALUE_WITH_RULESETS
	    }
	  };

	  const SELECTOR_MODE = {
	    // first parse unambiguous selectors (i.e. those not starting with tag)
	    // then fall into the scary lookahead-discriminator variant.
	    // this mode also handles mixin definitions and calls
	    variants: [
	      {
	        begin: '[\\.#:&\\[>]',
	        end: '[;{}]' // mixin calls end with ';'
	      },
	      {
	        begin: INTERP_IDENT_RE,
	        end: /\{/
	      }
	    ],
	    returnBegin: true,
	    returnEnd: true,
	    illegal: '[<=\'$"]',
	    relevance: 0,
	    contains: [
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE,
	      MIXIN_GUARD_MODE,
	      IDENT_MODE('keyword', 'all\\b'),
	      IDENT_MODE('variable', '@\\{' + IDENT_RE + '\\}'), // otherwise it’s identified as tag
	      {
	        begin: '\\b(' + TAGS.join('|') + ')\\b',
	        className: 'selector-tag'
	      },
	      IDENT_MODE('selector-tag', INTERP_IDENT_RE + '%?', 0), // '%' for more consistent coloring of @keyframes "tags"
	      IDENT_MODE('selector-id', '#' + INTERP_IDENT_RE),
	      IDENT_MODE('selector-class', '\\.' + INTERP_IDENT_RE, 0),
	      IDENT_MODE('selector-tag', '&', 0),
	      modes.ATTRIBUTE_SELECTOR_MODE,
	      {
	        className: 'selector-pseudo',
	        begin: ':(' + PSEUDO_CLASSES.join('|') + ')'
	      },
	      {
	        className: 'selector-pseudo',
	        begin: '::(' + PSEUDO_ELEMENTS.join('|') + ')'
	      },
	      {
	        begin: '\\(',
	        end: '\\)',
	        contains: VALUE_WITH_RULESETS
	      }, // argument list of parametric mixins
	      {
	        begin: '!important'
	      } // eat !important after mixin call or it will be colored as tag
	    ]
	  };

	  const PSEUDO_SELECTOR_MODE = {
	    begin: IDENT_RE + ':(:)?' + `(${PSEUDO_SELECTORS$1.join('|')})`,
	    returnBegin: true,
	    contains: [ SELECTOR_MODE ]
	  };

	  RULES.push(
	    hljs.C_LINE_COMMENT_MODE,
	    hljs.C_BLOCK_COMMENT_MODE,
	    AT_RULE_MODE,
	    VAR_RULE_MODE,
	    PSEUDO_SELECTOR_MODE,
	    RULE_MODE,
	    SELECTOR_MODE
	  );

	  return {
	    name: 'Less',
	    case_insensitive: true,
	    illegal: '[=>\'/<($"]',
	    contains: RULES
	  };
	}

	less_1 = less;
	return less_1;
}

/*
Language: Lisp
Description: Generic lisp syntax
Author: Vasily Polovnyov <vast@whiteants.net>
Category: lisp
*/

var lisp_1;
var hasRequiredLisp;

function requireLisp () {
	if (hasRequiredLisp) return lisp_1;
	hasRequiredLisp = 1;
	function lisp(hljs) {
	  var LISP_IDENT_RE = '[a-zA-Z_\\-+\\*\\/<=>&#][a-zA-Z0-9_\\-+*\\/<=>&#!]*';
	  var MEC_RE = '\\|[^]*?\\|';
	  var LISP_SIMPLE_NUMBER_RE = '(-|\\+)?\\d+(\\.\\d+|\\/\\d+)?((d|e|f|l|s|D|E|F|L|S)(\\+|-)?\\d+)?';
	  var LITERAL = {
	    className: 'literal',
	    begin: '\\b(t{1}|nil)\\b'
	  };
	  var NUMBER = {
	    className: 'number',
	    variants: [
	      {begin: LISP_SIMPLE_NUMBER_RE, relevance: 0},
	      {begin: '#(b|B)[0-1]+(/[0-1]+)?'},
	      {begin: '#(o|O)[0-7]+(/[0-7]+)?'},
	      {begin: '#(x|X)[0-9a-fA-F]+(/[0-9a-fA-F]+)?'},
	      {begin: '#(c|C)\\(' + LISP_SIMPLE_NUMBER_RE + ' +' + LISP_SIMPLE_NUMBER_RE, end: '\\)'}
	    ]
	  };
	  var STRING = hljs.inherit(hljs.QUOTE_STRING_MODE, {illegal: null});
	  var COMMENT = hljs.COMMENT(
	    ';', '$',
	    {
	      relevance: 0
	    }
	  );
	  var VARIABLE = {
	    begin: '\\*', end: '\\*'
	  };
	  var KEYWORD = {
	    className: 'symbol',
	    begin: '[:&]' + LISP_IDENT_RE
	  };
	  var IDENT = {
	    begin: LISP_IDENT_RE,
	    relevance: 0
	  };
	  var MEC = {
	    begin: MEC_RE
	  };
	  var QUOTED_LIST = {
	    begin: '\\(', end: '\\)',
	    contains: ['self', LITERAL, STRING, NUMBER, IDENT]
	  };
	  var QUOTED = {
	    contains: [NUMBER, STRING, VARIABLE, KEYWORD, QUOTED_LIST, IDENT],
	    variants: [
	      {
	        begin: '[\'`]\\(', end: '\\)'
	      },
	      {
	        begin: '\\(quote ', end: '\\)',
	        keywords: {name: 'quote'}
	      },
	      {
	        begin: '\'' + MEC_RE
	      }
	    ]
	  };
	  var QUOTED_ATOM = {
	    variants: [
	      {begin: '\'' + LISP_IDENT_RE},
	      {begin: '#\'' + LISP_IDENT_RE + '(::' + LISP_IDENT_RE + ')*'}
	    ]
	  };
	  var LIST = {
	    begin: '\\(\\s*', end: '\\)'
	  };
	  var BODY = {
	    endsWithParent: true,
	    relevance: 0
	  };
	  LIST.contains = [
	    {
	      className: 'name',
	      variants: [
	        {
	          begin: LISP_IDENT_RE,
	          relevance: 0,
	        },
	        {begin: MEC_RE}
	      ]
	    },
	    BODY
	  ];
	  BODY.contains = [QUOTED, QUOTED_ATOM, LIST, LITERAL, NUMBER, STRING, COMMENT, VARIABLE, KEYWORD, MEC, IDENT];

	  return {
	    name: 'Lisp',
	    illegal: /\S/,
	    contains: [
	      NUMBER,
	      hljs.SHEBANG(),
	      LITERAL,
	      STRING,
	      COMMENT,
	      QUOTED,
	      QUOTED_ATOM,
	      LIST,
	      IDENT
	    ]
	  };
	}

	lisp_1 = lisp;
	return lisp_1;
}

/*
Language: LiveCode
Author: Ralf Bitter <rabit@revigniter.com>
Description: Language definition for LiveCode server accounting for revIgniter (a web application framework) characteristics.
Version: 1.1
Date: 2019-04-17
Category: enterprise
*/

var livecodeserver_1;
var hasRequiredLivecodeserver;

function requireLivecodeserver () {
	if (hasRequiredLivecodeserver) return livecodeserver_1;
	hasRequiredLivecodeserver = 1;
	function livecodeserver(hljs) {
	  const VARIABLE = {
	    className: 'variable',
	    variants: [
	      {
	        begin: '\\b([gtps][A-Z]{1}[a-zA-Z0-9]*)(\\[.+\\])?(?:\\s*?)'
	      },
	      {
	        begin: '\\$_[A-Z]+'
	      }
	    ],
	    relevance: 0
	  };
	  const COMMENT_MODES = [
	    hljs.C_BLOCK_COMMENT_MODE,
	    hljs.HASH_COMMENT_MODE,
	    hljs.COMMENT('--', '$'),
	    hljs.COMMENT('[^:]//', '$')
	  ];
	  const TITLE1 = hljs.inherit(hljs.TITLE_MODE, {
	    variants: [
	      {
	        begin: '\\b_*rig[A-Z][A-Za-z0-9_\\-]*'
	      },
	      {
	        begin: '\\b_[a-z0-9\\-]+'
	      }
	    ]
	  });
	  const TITLE2 = hljs.inherit(hljs.TITLE_MODE, {
	    begin: '\\b([A-Za-z0-9_\\-]+)\\b'
	  });
	  return {
	    name: 'LiveCode',
	    case_insensitive: false,
	    keywords: {
	      keyword:
	        '$_COOKIE $_FILES $_GET $_GET_BINARY $_GET_RAW $_POST $_POST_BINARY $_POST_RAW $_SESSION $_SERVER ' +
	        'codepoint codepoints segment segments codeunit codeunits sentence sentences trueWord trueWords paragraph ' +
	        'after byte bytes english the until http forever descending using line real8 with seventh ' +
	        'for stdout finally element word words fourth before black ninth sixth characters chars stderr ' +
	        'uInt1 uInt1s uInt2 uInt2s stdin string lines relative rel any fifth items from middle mid ' +
	        'at else of catch then third it file milliseconds seconds second secs sec int1 int1s int4 ' +
	        'int4s internet int2 int2s normal text item last long detailed effective uInt4 uInt4s repeat ' +
	        'end repeat URL in try into switch to words https token binfile each tenth as ticks tick ' +
	        'system real4 by dateItems without char character ascending eighth whole dateTime numeric short ' +
	        'first ftp integer abbreviated abbr abbrev private case while if ' +
	        'div mod wrap and or bitAnd bitNot bitOr bitXor among not in a an within ' +
	        'contains ends with begins the keys of keys',
	      literal:
	        'SIX TEN FORMFEED NINE ZERO NONE SPACE FOUR FALSE COLON CRLF PI COMMA ENDOFFILE EOF EIGHT FIVE ' +
	        'QUOTE EMPTY ONE TRUE RETURN CR LINEFEED RIGHT BACKSLASH NULL SEVEN TAB THREE TWO ' +
	        'six ten formfeed nine zero none space four false colon crlf pi comma endoffile eof eight five ' +
	        'quote empty one true return cr linefeed right backslash null seven tab three two ' +
	        'RIVERSION RISTATE FILE_READ_MODE FILE_WRITE_MODE FILE_WRITE_MODE DIR_WRITE_MODE FILE_READ_UMASK ' +
	        'FILE_WRITE_UMASK DIR_READ_UMASK DIR_WRITE_UMASK',
	      built_in:
	        'put abs acos aliasReference annuity arrayDecode arrayEncode asin atan atan2 average avg avgDev base64Decode ' +
	        'base64Encode baseConvert binaryDecode binaryEncode byteOffset byteToNum cachedURL cachedURLs charToNum ' +
	        'cipherNames codepointOffset codepointProperty codepointToNum codeunitOffset commandNames compound compress ' +
	        'constantNames cos date dateFormat decompress difference directories ' +
	        'diskSpace DNSServers exp exp1 exp2 exp10 extents files flushEvents folders format functionNames geometricMean global ' +
	        'globals hasMemory harmonicMean hostAddress hostAddressToName hostName hostNameToAddress isNumber ISOToMac itemOffset ' +
	        'keys len length libURLErrorData libUrlFormData libURLftpCommand libURLLastHTTPHeaders libURLLastRHHeaders ' +
	        'libUrlMultipartFormAddPart libUrlMultipartFormData libURLVersion lineOffset ln ln1 localNames log log2 log10 ' +
	        'longFilePath lower macToISO matchChunk matchText matrixMultiply max md5Digest median merge messageAuthenticationCode messageDigest millisec ' +
	        'millisecs millisecond milliseconds min monthNames nativeCharToNum normalizeText num number numToByte numToChar ' +
	        'numToCodepoint numToNativeChar offset open openfiles openProcesses openProcessIDs openSockets ' +
	        'paragraphOffset paramCount param params peerAddress pendingMessages platform popStdDev populationStandardDeviation ' +
	        'populationVariance popVariance processID random randomBytes replaceText result revCreateXMLTree revCreateXMLTreeFromFile ' +
	        'revCurrentRecord revCurrentRecordIsFirst revCurrentRecordIsLast revDatabaseColumnCount revDatabaseColumnIsNull ' +
	        'revDatabaseColumnLengths revDatabaseColumnNames revDatabaseColumnNamed revDatabaseColumnNumbered ' +
	        'revDatabaseColumnTypes revDatabaseConnectResult revDatabaseCursors revDatabaseID revDatabaseTableNames ' +
	        'revDatabaseType revDataFromQuery revdb_closeCursor revdb_columnbynumber revdb_columncount revdb_columnisnull ' +
	        'revdb_columnlengths revdb_columnnames revdb_columntypes revdb_commit revdb_connect revdb_connections ' +
	        'revdb_connectionerr revdb_currentrecord revdb_cursorconnection revdb_cursorerr revdb_cursors revdb_dbtype ' +
	        'revdb_disconnect revdb_execute revdb_iseof revdb_isbof revdb_movefirst revdb_movelast revdb_movenext ' +
	        'revdb_moveprev revdb_query revdb_querylist revdb_recordcount revdb_rollback revdb_tablenames ' +
	        'revGetDatabaseDriverPath revNumberOfRecords revOpenDatabase revOpenDatabases revQueryDatabase ' +
	        'revQueryDatabaseBlob revQueryResult revQueryIsAtStart revQueryIsAtEnd revUnixFromMacPath revXMLAttribute ' +
	        'revXMLAttributes revXMLAttributeValues revXMLChildContents revXMLChildNames revXMLCreateTreeFromFileWithNamespaces ' +
	        'revXMLCreateTreeWithNamespaces revXMLDataFromXPathQuery revXMLEvaluateXPath revXMLFirstChild revXMLMatchingNode ' +
	        'revXMLNextSibling revXMLNodeContents revXMLNumberOfChildren revXMLParent revXMLPreviousSibling ' +
	        'revXMLRootNode revXMLRPC_CreateRequest revXMLRPC_Documents revXMLRPC_Error ' +
	        'revXMLRPC_GetHost revXMLRPC_GetMethod revXMLRPC_GetParam revXMLText revXMLRPC_Execute ' +
	        'revXMLRPC_GetParamCount revXMLRPC_GetParamNode revXMLRPC_GetParamType revXMLRPC_GetPath revXMLRPC_GetPort ' +
	        'revXMLRPC_GetProtocol revXMLRPC_GetRequest revXMLRPC_GetResponse revXMLRPC_GetSocket revXMLTree ' +
	        'revXMLTrees revXMLValidateDTD revZipDescribeItem revZipEnumerateItems revZipOpenArchives round sampVariance ' +
	        'sec secs seconds sentenceOffset sha1Digest shell shortFilePath sin specialFolderPath sqrt standardDeviation statRound ' +
	        'stdDev sum sysError systemVersion tan tempName textDecode textEncode tick ticks time to tokenOffset toLower toUpper ' +
	        'transpose truewordOffset trunc uniDecode uniEncode upper URLDecode URLEncode URLStatus uuid value variableNames ' +
	        'variance version waitDepth weekdayNames wordOffset xsltApplyStylesheet xsltApplyStylesheetFromFile xsltLoadStylesheet ' +
	        'xsltLoadStylesheetFromFile add breakpoint cancel clear local variable file word line folder directory URL close socket process ' +
	        'combine constant convert create new alias folder directory decrypt delete variable word line folder ' +
	        'directory URL dispatch divide do encrypt filter get include intersect kill libURLDownloadToFile ' +
	        'libURLFollowHttpRedirects libURLftpUpload libURLftpUploadFile libURLresetAll libUrlSetAuthCallback libURLSetDriver ' +
	        'libURLSetCustomHTTPHeaders libUrlSetExpect100 libURLSetFTPListCommand libURLSetFTPMode libURLSetFTPStopTime ' +
	        'libURLSetStatusCallback load extension loadedExtensions multiply socket prepare process post seek rel relative read from process rename ' +
	        'replace require resetAll resolve revAddXMLNode revAppendXML revCloseCursor revCloseDatabase revCommitDatabase ' +
	        'revCopyFile revCopyFolder revCopyXMLNode revDeleteFolder revDeleteXMLNode revDeleteAllXMLTrees ' +
	        'revDeleteXMLTree revExecuteSQL revGoURL revInsertXMLNode revMoveFolder revMoveToFirstRecord revMoveToLastRecord ' +
	        'revMoveToNextRecord revMoveToPreviousRecord revMoveToRecord revMoveXMLNode revPutIntoXMLNode revRollBackDatabase ' +
	        'revSetDatabaseDriverPath revSetXMLAttribute revXMLRPC_AddParam revXMLRPC_DeleteAllDocuments revXMLAddDTD ' +
	        'revXMLRPC_Free revXMLRPC_FreeAll revXMLRPC_DeleteDocument revXMLRPC_DeleteParam revXMLRPC_SetHost ' +
	        'revXMLRPC_SetMethod revXMLRPC_SetPort revXMLRPC_SetProtocol revXMLRPC_SetSocket revZipAddItemWithData ' +
	        'revZipAddItemWithFile revZipAddUncompressedItemWithData revZipAddUncompressedItemWithFile revZipCancel ' +
	        'revZipCloseArchive revZipDeleteItem revZipExtractItemToFile revZipExtractItemToVariable revZipSetProgressCallback ' +
	        'revZipRenameItem revZipReplaceItemWithData revZipReplaceItemWithFile revZipOpenArchive send set sort split start stop ' +
	        'subtract symmetric union unload vectorDotProduct wait write'
	    },
	    contains: [
	      VARIABLE,
	      {
	        className: 'keyword',
	        begin: '\\bend\\sif\\b'
	      },
	      {
	        className: 'function',
	        beginKeywords: 'function',
	        end: '$',
	        contains: [
	          VARIABLE,
	          TITLE2,
	          hljs.APOS_STRING_MODE,
	          hljs.QUOTE_STRING_MODE,
	          hljs.BINARY_NUMBER_MODE,
	          hljs.C_NUMBER_MODE,
	          TITLE1
	        ]
	      },
	      {
	        className: 'function',
	        begin: '\\bend\\s+',
	        end: '$',
	        keywords: 'end',
	        contains: [
	          TITLE2,
	          TITLE1
	        ],
	        relevance: 0
	      },
	      {
	        beginKeywords: 'command on',
	        end: '$',
	        contains: [
	          VARIABLE,
	          TITLE2,
	          hljs.APOS_STRING_MODE,
	          hljs.QUOTE_STRING_MODE,
	          hljs.BINARY_NUMBER_MODE,
	          hljs.C_NUMBER_MODE,
	          TITLE1
	        ]
	      },
	      {
	        className: 'meta',
	        variants: [
	          {
	            begin: '<\\?(rev|lc|livecode)',
	            relevance: 10
	          },
	          {
	            begin: '<\\?'
	          },
	          {
	            begin: '\\?>'
	          }
	        ]
	      },
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      hljs.BINARY_NUMBER_MODE,
	      hljs.C_NUMBER_MODE,
	      TITLE1
	    ].concat(COMMENT_MODES),
	    illegal: ';$|^\\[|^=|&|\\{'
	  };
	}

	livecodeserver_1 = livecodeserver;
	return livecodeserver_1;
}

var livescript_1;
var hasRequiredLivescript;

function requireLivescript () {
	if (hasRequiredLivescript) return livescript_1;
	hasRequiredLivescript = 1;
	const KEYWORDS = [
	  "as", // for exports
	  "in",
	  "of",
	  "if",
	  "for",
	  "while",
	  "finally",
	  "var",
	  "new",
	  "function",
	  "do",
	  "return",
	  "void",
	  "else",
	  "break",
	  "catch",
	  "instanceof",
	  "with",
	  "throw",
	  "case",
	  "default",
	  "try",
	  "switch",
	  "continue",
	  "typeof",
	  "delete",
	  "let",
	  "yield",
	  "const",
	  "class",
	  // JS handles these with a special rule
	  // "get",
	  // "set",
	  "debugger",
	  "async",
	  "await",
	  "static",
	  "import",
	  "from",
	  "export",
	  "extends"
	];
	const LITERALS = [
	  "true",
	  "false",
	  "null",
	  "undefined",
	  "NaN",
	  "Infinity"
	];

	const TYPES = [
	  "Intl",
	  "DataView",
	  "Number",
	  "Math",
	  "Date",
	  "String",
	  "RegExp",
	  "Object",
	  "Function",
	  "Boolean",
	  "Error",
	  "Symbol",
	  "Set",
	  "Map",
	  "WeakSet",
	  "WeakMap",
	  "Proxy",
	  "Reflect",
	  "JSON",
	  "Promise",
	  "Float64Array",
	  "Int16Array",
	  "Int32Array",
	  "Int8Array",
	  "Uint16Array",
	  "Uint32Array",
	  "Float32Array",
	  "Array",
	  "Uint8Array",
	  "Uint8ClampedArray",
	  "ArrayBuffer",
	  "BigInt64Array",
	  "BigUint64Array",
	  "BigInt"
	];

	const ERROR_TYPES = [
	  "EvalError",
	  "InternalError",
	  "RangeError",
	  "ReferenceError",
	  "SyntaxError",
	  "TypeError",
	  "URIError"
	];

	const BUILT_IN_GLOBALS = [
	  "setInterval",
	  "setTimeout",
	  "clearInterval",
	  "clearTimeout",

	  "require",
	  "exports",

	  "eval",
	  "isFinite",
	  "isNaN",
	  "parseFloat",
	  "parseInt",
	  "decodeURI",
	  "decodeURIComponent",
	  "encodeURI",
	  "encodeURIComponent",
	  "escape",
	  "unescape"
	];

	const BUILT_IN_VARIABLES = [
	  "arguments",
	  "this",
	  "super",
	  "console",
	  "window",
	  "document",
	  "localStorage",
	  "module",
	  "global" // Node.js
	];

	const BUILT_INS = [].concat(
	  BUILT_IN_GLOBALS,
	  BUILT_IN_VARIABLES,
	  TYPES,
	  ERROR_TYPES
	);

	/*
	Language: LiveScript
	Author: Taneli Vatanen <taneli.vatanen@gmail.com>
	Contributors: Jen Evers-Corvina <jen@sevvie.net>
	Origin: coffeescript.js
	Description: LiveScript is a programming language that transcompiles to JavaScript. For info about language see http://livescript.net/
	Website: https://livescript.net
	Category: scripting
	*/

	function livescript(hljs) {
	  const LIVESCRIPT_BUILT_INS = [
	    'npm',
	    'print'
	  ];
	  const LIVESCRIPT_LITERALS = [
	    'yes',
	    'no',
	    'on',
	    'off',
	    'it',
	    'that',
	    'void'
	  ];
	  const LIVESCRIPT_KEYWORDS = [
	    'then',
	    'unless',
	    'until',
	    'loop',
	    'of',
	    'by',
	    'when',
	    'and',
	    'or',
	    'is',
	    'isnt',
	    'not',
	    'it',
	    'that',
	    'otherwise',
	    'from',
	    'to',
	    'til',
	    'fallthrough',
	    'case',
	    'enum',
	    'native',
	    'list',
	    'map',
	    '__hasProp',
	    '__extends',
	    '__slice',
	    '__bind',
	    '__indexOf'
	  ];
	  const KEYWORDS$1 = {
	    keyword: KEYWORDS.concat(LIVESCRIPT_KEYWORDS),
	    literal: LITERALS.concat(LIVESCRIPT_LITERALS),
	    built_in: BUILT_INS.concat(LIVESCRIPT_BUILT_INS)
	  };
	  const JS_IDENT_RE = '[A-Za-z$_](?:-[0-9A-Za-z$_]|[0-9A-Za-z$_])*';
	  const TITLE = hljs.inherit(hljs.TITLE_MODE, {
	    begin: JS_IDENT_RE
	  });
	  const SUBST = {
	    className: 'subst',
	    begin: /#\{/,
	    end: /\}/,
	    keywords: KEYWORDS$1
	  };
	  const SUBST_SIMPLE = {
	    className: 'subst',
	    begin: /#[A-Za-z$_]/,
	    end: /(?:-[0-9A-Za-z$_]|[0-9A-Za-z$_])*/,
	    keywords: KEYWORDS$1
	  };
	  const EXPRESSIONS = [
	    hljs.BINARY_NUMBER_MODE,
	    {
	      className: 'number',
	      begin: '(\\b0[xX][a-fA-F0-9_]+)|(\\b\\d(\\d|_\\d)*(\\.(\\d(\\d|_\\d)*)?)?(_*[eE]([-+]\\d(_\\d|\\d)*)?)?[_a-z]*)',
	      relevance: 0,
	      starts: {
	        end: '(\\s*/)?',
	        relevance: 0
	      } // a number tries to eat the following slash to prevent treating it as a regexp
	    },
	    {
	      className: 'string',
	      variants: [
	        {
	          begin: /'''/,
	          end: /'''/,
	          contains: [hljs.BACKSLASH_ESCAPE]
	        },
	        {
	          begin: /'/,
	          end: /'/,
	          contains: [hljs.BACKSLASH_ESCAPE]
	        },
	        {
	          begin: /"""/,
	          end: /"""/,
	          contains: [
	            hljs.BACKSLASH_ESCAPE,
	            SUBST,
	            SUBST_SIMPLE
	          ]
	        },
	        {
	          begin: /"/,
	          end: /"/,
	          contains: [
	            hljs.BACKSLASH_ESCAPE,
	            SUBST,
	            SUBST_SIMPLE
	          ]
	        },
	        {
	          begin: /\\/,
	          end: /(\s|$)/,
	          excludeEnd: true
	        }
	      ]
	    },
	    {
	      className: 'regexp',
	      variants: [
	        {
	          begin: '//',
	          end: '//[gim]*',
	          contains: [
	            SUBST,
	            hljs.HASH_COMMENT_MODE
	          ]
	        },
	        {
	          // regex can't start with space to parse x / 2 / 3 as two divisions
	          // regex can't start with *, and it supports an "illegal" in the main mode
	          begin: /\/(?![ *])(\\.|[^\\\n])*?\/[gim]*(?=\W)/
	        }
	      ]
	    },
	    {
	      begin: '@' + JS_IDENT_RE
	    },
	    {
	      begin: '``',
	      end: '``',
	      excludeBegin: true,
	      excludeEnd: true,
	      subLanguage: 'javascript'
	    }
	  ];
	  SUBST.contains = EXPRESSIONS;

	  const PARAMS = {
	    className: 'params',
	    begin: '\\(',
	    returnBegin: true,
	    /* We need another contained nameless mode to not have every nested
	    pair of parens to be called "params" */
	    contains: [
	      {
	        begin: /\(/,
	        end: /\)/,
	        keywords: KEYWORDS$1,
	        contains: ['self'].concat(EXPRESSIONS)
	      }
	    ]
	  };

	  const SYMBOLS = {
	    begin: '(#=>|=>|\\|>>|-?->|!->)'
	  };

	  return {
	    name: 'LiveScript',
	    aliases: ['ls'],
	    keywords: KEYWORDS$1,
	    illegal: /\/\*/,
	    contains: EXPRESSIONS.concat([
	      hljs.COMMENT('\\/\\*', '\\*\\/'),
	      hljs.HASH_COMMENT_MODE,
	      SYMBOLS, // relevance booster
	      {
	        className: 'function',
	        contains: [
	          TITLE,
	          PARAMS
	        ],
	        returnBegin: true,
	        variants: [
	          {
	            begin: '(' + JS_IDENT_RE + '\\s*(?:=|:=)\\s*)?(\\(.*\\)\\s*)?\\B->\\*?',
	            end: '->\\*?'
	          },
	          {
	            begin: '(' + JS_IDENT_RE + '\\s*(?:=|:=)\\s*)?!?(\\(.*\\)\\s*)?\\B[-~]{1,2}>\\*?',
	            end: '[-~]{1,2}>\\*?'
	          },
	          {
	            begin: '(' + JS_IDENT_RE + '\\s*(?:=|:=)\\s*)?(\\(.*\\)\\s*)?\\B!?[-~]{1,2}>\\*?',
	            end: '!?[-~]{1,2}>\\*?'
	          }
	        ]
	      },
	      {
	        className: 'class',
	        beginKeywords: 'class',
	        end: '$',
	        illegal: /[:="\[\]]/,
	        contains: [
	          {
	            beginKeywords: 'extends',
	            endsWithParent: true,
	            illegal: /[:="\[\]]/,
	            contains: [TITLE]
	          },
	          TITLE
	        ]
	      },
	      {
	        begin: JS_IDENT_RE + ':',
	        end: ':',
	        returnBegin: true,
	        returnEnd: true,
	        relevance: 0
	      }
	    ])
	  };
	}

	livescript_1 = livescript;
	return livescript_1;
}

/**
 * @param {string} value
 * @returns {RegExp}
 * */

var llvm_1;
var hasRequiredLlvm;

function requireLlvm () {
	if (hasRequiredLlvm) return llvm_1;
	hasRequiredLlvm = 1;
	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/*
	Language: LLVM IR
	Author: Michael Rodler <contact@f0rki.at>
	Description: language used as intermediate representation in the LLVM compiler framework
	Website: https://llvm.org/docs/LangRef.html
	Category: assembler
	Audit: 2020
	*/

	/** @type LanguageFn */
	function llvm(hljs) {
	  const IDENT_RE = /([-a-zA-Z$._][\w$.-]*)/;
	  const TYPE = {
	    className: 'type',
	    begin: /\bi\d+(?=\s|\b)/
	  };
	  const OPERATOR = {
	    className: 'operator',
	    relevance: 0,
	    begin: /=/
	  };
	  const PUNCTUATION = {
	    className: 'punctuation',
	    relevance: 0,
	    begin: /,/
	  };
	  const NUMBER = {
	    className: 'number',
	    variants: [
	        { begin: /0[xX][a-fA-F0-9]+/ },
	        { begin: /-?\d+(?:[.]\d+)?(?:[eE][-+]?\d+(?:[.]\d+)?)?/ }
	    ],
	    relevance: 0
	  };
	  const LABEL = {
	    className: 'symbol',
	    variants: [
	        { begin: /^\s*[a-z]+:/ }, // labels
	    ],
	    relevance: 0
	  };
	  const VARIABLE = {
	    className: 'variable',
	    variants: [
	      { begin: concat(/%/, IDENT_RE) },
	      { begin: /%\d+/ },
	      { begin: /#\d+/ },
	    ]
	  };
	  const FUNCTION = {
	    className: 'title',
	    variants: [
	      { begin: concat(/@/, IDENT_RE) },
	      { begin: /@\d+/ },
	      { begin: concat(/!/, IDENT_RE) },
	      { begin: concat(/!\d+/, IDENT_RE) },
	      // https://llvm.org/docs/LangRef.html#namedmetadatastructure
	      // obviously a single digit can also be used in this fashion
	      { begin: /!\d+/ }
	    ]
	  };

	  return {
	    name: 'LLVM IR',
	    // TODO: split into different categories of keywords
	    keywords:
	      'begin end true false declare define global ' +
	      'constant private linker_private internal ' +
	      'available_externally linkonce linkonce_odr weak ' +
	      'weak_odr appending dllimport dllexport common ' +
	      'default hidden protected extern_weak external ' +
	      'thread_local zeroinitializer undef null to tail ' +
	      'target triple datalayout volatile nuw nsw nnan ' +
	      'ninf nsz arcp fast exact inbounds align ' +
	      'addrspace section alias module asm sideeffect ' +
	      'gc dbg linker_private_weak attributes blockaddress ' +
	      'initialexec localdynamic localexec prefix unnamed_addr ' +
	      'ccc fastcc coldcc x86_stdcallcc x86_fastcallcc ' +
	      'arm_apcscc arm_aapcscc arm_aapcs_vfpcc ptx_device ' +
	      'ptx_kernel intel_ocl_bicc msp430_intrcc spir_func ' +
	      'spir_kernel x86_64_sysvcc x86_64_win64cc x86_thiscallcc ' +
	      'cc c signext zeroext inreg sret nounwind ' +
	      'noreturn noalias nocapture byval nest readnone ' +
	      'readonly inlinehint noinline alwaysinline optsize ssp ' +
	      'sspreq noredzone noimplicitfloat naked builtin cold ' +
	      'nobuiltin noduplicate nonlazybind optnone returns_twice ' +
	      'sanitize_address sanitize_memory sanitize_thread sspstrong ' +
	      'uwtable returned type opaque eq ne slt sgt ' +
	      'sle sge ult ugt ule uge oeq one olt ogt ' +
	      'ole oge ord uno ueq une x acq_rel acquire ' +
	      'alignstack atomic catch cleanup filter inteldialect ' +
	      'max min monotonic nand personality release seq_cst ' +
	      'singlethread umax umin unordered xchg add fadd ' +
	      'sub fsub mul fmul udiv sdiv fdiv urem srem ' +
	      'frem shl lshr ashr and or xor icmp fcmp ' +
	      'phi call trunc zext sext fptrunc fpext uitofp ' +
	      'sitofp fptoui fptosi inttoptr ptrtoint bitcast ' +
	      'addrspacecast select va_arg ret br switch invoke ' +
	      'unwind unreachable indirectbr landingpad resume ' +
	      'malloc alloca free load store getelementptr ' +
	      'extractelement insertelement shufflevector getresult ' +
	      'extractvalue insertvalue atomicrmw cmpxchg fence ' +
	      'argmemonly double',
	    contains: [
	      TYPE,
	      // this matches "empty comments"...
	      // ...because it's far more likely this is a statement terminator in
	      // another language than an actual comment
	      hljs.COMMENT(/;\s*$/, null, { relevance: 0 }),
	      hljs.COMMENT(/;/, /$/),
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'string',
	        variants: [
	          // Double-quoted string
	          { begin: /"/, end: /[^\\]"/ },
	        ]
	      },
	      FUNCTION,
	      PUNCTUATION,
	      OPERATOR,
	      VARIABLE,
	      LABEL,
	      NUMBER
	    ]
	  };
	}

	llvm_1 = llvm;
	return llvm_1;
}

/*
Language: LSL (Linden Scripting Language)
Description: The Linden Scripting Language is used in Second Life by Linden Labs.
Author: Builder's Brewery <buildersbrewery@gmail.com>
Website: http://wiki.secondlife.com/wiki/LSL_Portal
Category: scripting
*/

var lsl_1;
var hasRequiredLsl;

function requireLsl () {
	if (hasRequiredLsl) return lsl_1;
	hasRequiredLsl = 1;
	function lsl(hljs) {

	    var LSL_STRING_ESCAPE_CHARS = {
	        className: 'subst',
	        begin: /\\[tn"\\]/
	    };

	    var LSL_STRINGS = {
	        className: 'string',
	        begin: '"',
	        end: '"',
	        contains: [
	            LSL_STRING_ESCAPE_CHARS
	        ]
	    };

	    var LSL_NUMBERS = {
	        className: 'number',
	        relevance:0,
	        begin: hljs.C_NUMBER_RE
	    };

	    var LSL_CONSTANTS = {
	        className: 'literal',
	        variants: [
	            {
	                begin: '\\b(PI|TWO_PI|PI_BY_TWO|DEG_TO_RAD|RAD_TO_DEG|SQRT2)\\b'
	            },
	            {
	                begin: '\\b(XP_ERROR_(EXPERIENCES_DISABLED|EXPERIENCE_(DISABLED|SUSPENDED)|INVALID_(EXPERIENCE|PARAMETERS)|KEY_NOT_FOUND|MATURITY_EXCEEDED|NONE|NOT_(FOUND|PERMITTED(_LAND)?)|NO_EXPERIENCE|QUOTA_EXCEEDED|RETRY_UPDATE|STORAGE_EXCEPTION|STORE_DISABLED|THROTTLED|UNKNOWN_ERROR)|JSON_APPEND|STATUS_(PHYSICS|ROTATE_[XYZ]|PHANTOM|SANDBOX|BLOCK_GRAB(_OBJECT)?|(DIE|RETURN)_AT_EDGE|CAST_SHADOWS|OK|MALFORMED_PARAMS|TYPE_MISMATCH|BOUNDS_ERROR|NOT_(FOUND|SUPPORTED)|INTERNAL_ERROR|WHITELIST_FAILED)|AGENT(_(BY_(LEGACY_|USER)NAME|FLYING|ATTACHMENTS|SCRIPTED|MOUSELOOK|SITTING|ON_OBJECT|AWAY|WALKING|IN_AIR|TYPING|CROUCHING|BUSY|ALWAYS_RUN|AUTOPILOT|LIST_(PARCEL(_OWNER)?|REGION)))?|CAMERA_(PITCH|DISTANCE|BEHINDNESS_(ANGLE|LAG)|(FOCUS|POSITION)(_(THRESHOLD|LOCKED|LAG))?|FOCUS_OFFSET|ACTIVE)|ANIM_ON|LOOP|REVERSE|PING_PONG|SMOOTH|ROTATE|SCALE|ALL_SIDES|LINK_(ROOT|SET|ALL_(OTHERS|CHILDREN)|THIS)|ACTIVE|PASS(IVE|_(ALWAYS|IF_NOT_HANDLED|NEVER))|SCRIPTED|CONTROL_(FWD|BACK|(ROT_)?(LEFT|RIGHT)|UP|DOWN|(ML_)?LBUTTON)|PERMISSION_(RETURN_OBJECTS|DEBIT|OVERRIDE_ANIMATIONS|SILENT_ESTATE_MANAGEMENT|TAKE_CONTROLS|TRIGGER_ANIMATION|ATTACH|CHANGE_LINKS|(CONTROL|TRACK)_CAMERA|TELEPORT)|INVENTORY_(TEXTURE|SOUND|OBJECT|SCRIPT|LANDMARK|CLOTHING|NOTECARD|BODYPART|ANIMATION|GESTURE|ALL|NONE)|CHANGED_(INVENTORY|COLOR|SHAPE|SCALE|TEXTURE|LINK|ALLOWED_DROP|OWNER|REGION(_START)?|TELEPORT|MEDIA)|OBJECT_(CLICK_ACTION|HOVER_HEIGHT|LAST_OWNER_ID|(PHYSICS|SERVER|STREAMING)_COST|UNKNOWN_DETAIL|CHARACTER_TIME|PHANTOM|PHYSICS|TEMP_(ATTACHED|ON_REZ)|NAME|DESC|POS|PRIM_(COUNT|EQUIVALENCE)|RETURN_(PARCEL(_OWNER)?|REGION)|REZZER_KEY|ROO?T|VELOCITY|OMEGA|OWNER|GROUP(_TAG)?|CREATOR|ATTACHED_(POINT|SLOTS_AVAILABLE)|RENDER_WEIGHT|(BODY_SHAPE|PATHFINDING)_TYPE|(RUNNING|TOTAL)_SCRIPT_COUNT|TOTAL_INVENTORY_COUNT|SCRIPT_(MEMORY|TIME))|TYPE_(INTEGER|FLOAT|STRING|KEY|VECTOR|ROTATION|INVALID)|(DEBUG|PUBLIC)_CHANNEL|ATTACH_(AVATAR_CENTER|CHEST|HEAD|BACK|PELVIS|MOUTH|CHIN|NECK|NOSE|BELLY|[LR](SHOULDER|HAND|FOOT|EAR|EYE|[UL](ARM|LEG)|HIP)|(LEFT|RIGHT)_PEC|HUD_(CENTER_[12]|TOP_(RIGHT|CENTER|LEFT)|BOTTOM(_(RIGHT|LEFT))?)|[LR]HAND_RING1|TAIL_(BASE|TIP)|[LR]WING|FACE_(JAW|[LR]EAR|[LR]EYE|TOUNGE)|GROIN|HIND_[LR]FOOT)|LAND_(LEVEL|RAISE|LOWER|SMOOTH|NOISE|REVERT)|DATA_(ONLINE|NAME|BORN|SIM_(POS|STATUS|RATING)|PAYINFO)|PAYMENT_INFO_(ON_FILE|USED)|REMOTE_DATA_(CHANNEL|REQUEST|REPLY)|PSYS_(PART_(BF_(ZERO|ONE(_MINUS_(DEST_COLOR|SOURCE_(ALPHA|COLOR)))?|DEST_COLOR|SOURCE_(ALPHA|COLOR))|BLEND_FUNC_(DEST|SOURCE)|FLAGS|(START|END)_(COLOR|ALPHA|SCALE|GLOW)|MAX_AGE|(RIBBON|WIND|INTERP_(COLOR|SCALE)|BOUNCE|FOLLOW_(SRC|VELOCITY)|TARGET_(POS|LINEAR)|EMISSIVE)_MASK)|SRC_(MAX_AGE|PATTERN|ANGLE_(BEGIN|END)|BURST_(RATE|PART_COUNT|RADIUS|SPEED_(MIN|MAX))|ACCEL|TEXTURE|TARGET_KEY|OMEGA|PATTERN_(DROP|EXPLODE|ANGLE(_CONE(_EMPTY)?)?)))|VEHICLE_(REFERENCE_FRAME|TYPE_(NONE|SLED|CAR|BOAT|AIRPLANE|BALLOON)|(LINEAR|ANGULAR)_(FRICTION_TIMESCALE|MOTOR_DIRECTION)|LINEAR_MOTOR_OFFSET|HOVER_(HEIGHT|EFFICIENCY|TIMESCALE)|BUOYANCY|(LINEAR|ANGULAR)_(DEFLECTION_(EFFICIENCY|TIMESCALE)|MOTOR_(DECAY_)?TIMESCALE)|VERTICAL_ATTRACTION_(EFFICIENCY|TIMESCALE)|BANKING_(EFFICIENCY|MIX|TIMESCALE)|FLAG_(NO_DEFLECTION_UP|LIMIT_(ROLL_ONLY|MOTOR_UP)|HOVER_((WATER|TERRAIN|UP)_ONLY|GLOBAL_HEIGHT)|MOUSELOOK_(STEER|BANK)|CAMERA_DECOUPLED))|PRIM_(ALLOW_UNSIT|ALPHA_MODE(_(BLEND|EMISSIVE|MASK|NONE))?|NORMAL|SPECULAR|TYPE(_(BOX|CYLINDER|PRISM|SPHERE|TORUS|TUBE|RING|SCULPT))?|HOLE_(DEFAULT|CIRCLE|SQUARE|TRIANGLE)|MATERIAL(_(STONE|METAL|GLASS|WOOD|FLESH|PLASTIC|RUBBER))?|SHINY_(NONE|LOW|MEDIUM|HIGH)|BUMP_(NONE|BRIGHT|DARK|WOOD|BARK|BRICKS|CHECKER|CONCRETE|TILE|STONE|DISKS|GRAVEL|BLOBS|SIDING|LARGETILE|STUCCO|SUCTION|WEAVE)|TEXGEN_(DEFAULT|PLANAR)|SCRIPTED_SIT_ONLY|SCULPT_(TYPE_(SPHERE|TORUS|PLANE|CYLINDER|MASK)|FLAG_(MIRROR|INVERT))|PHYSICS(_(SHAPE_(CONVEX|NONE|PRIM|TYPE)))?|(POS|ROT)_LOCAL|SLICE|TEXT|FLEXIBLE|POINT_LIGHT|TEMP_ON_REZ|PHANTOM|POSITION|SIT_TARGET|SIZE|ROTATION|TEXTURE|NAME|OMEGA|DESC|LINK_TARGET|COLOR|BUMP_SHINY|FULLBRIGHT|TEXGEN|GLOW|MEDIA_(ALT_IMAGE_ENABLE|CONTROLS|(CURRENT|HOME)_URL|AUTO_(LOOP|PLAY|SCALE|ZOOM)|FIRST_CLICK_INTERACT|(WIDTH|HEIGHT)_PIXELS|WHITELIST(_ENABLE)?|PERMS_(INTERACT|CONTROL)|PARAM_MAX|CONTROLS_(STANDARD|MINI)|PERM_(NONE|OWNER|GROUP|ANYONE)|MAX_(URL_LENGTH|WHITELIST_(SIZE|COUNT)|(WIDTH|HEIGHT)_PIXELS)))|MASK_(BASE|OWNER|GROUP|EVERYONE|NEXT)|PERM_(TRANSFER|MODIFY|COPY|MOVE|ALL)|PARCEL_(MEDIA_COMMAND_(STOP|PAUSE|PLAY|LOOP|TEXTURE|URL|TIME|AGENT|UNLOAD|AUTO_ALIGN|TYPE|SIZE|DESC|LOOP_SET)|FLAG_(ALLOW_(FLY|(GROUP_)?SCRIPTS|LANDMARK|TERRAFORM|DAMAGE|CREATE_(GROUP_)?OBJECTS)|USE_(ACCESS_(GROUP|LIST)|BAN_LIST|LAND_PASS_LIST)|LOCAL_SOUND_ONLY|RESTRICT_PUSHOBJECT|ALLOW_(GROUP|ALL)_OBJECT_ENTRY)|COUNT_(TOTAL|OWNER|GROUP|OTHER|SELECTED|TEMP)|DETAILS_(NAME|DESC|OWNER|GROUP|AREA|ID|SEE_AVATARS))|LIST_STAT_(MAX|MIN|MEAN|MEDIAN|STD_DEV|SUM(_SQUARES)?|NUM_COUNT|GEOMETRIC_MEAN|RANGE)|PAY_(HIDE|DEFAULT)|REGION_FLAG_(ALLOW_DAMAGE|FIXED_SUN|BLOCK_TERRAFORM|SANDBOX|DISABLE_(COLLISIONS|PHYSICS)|BLOCK_FLY|ALLOW_DIRECT_TELEPORT|RESTRICT_PUSHOBJECT)|HTTP_(METHOD|MIMETYPE|BODY_(MAXLENGTH|TRUNCATED)|CUSTOM_HEADER|PRAGMA_NO_CACHE|VERBOSE_THROTTLE|VERIFY_CERT)|SIT_(INVALID_(AGENT|LINK_OBJECT)|NO(T_EXPERIENCE|_(ACCESS|EXPERIENCE_PERMISSION|SIT_TARGET)))|STRING_(TRIM(_(HEAD|TAIL))?)|CLICK_ACTION_(NONE|TOUCH|SIT|BUY|PAY|OPEN(_MEDIA)?|PLAY|ZOOM)|TOUCH_INVALID_FACE|PROFILE_(NONE|SCRIPT_MEMORY)|RC_(DATA_FLAGS|DETECT_PHANTOM|GET_(LINK_NUM|NORMAL|ROOT_KEY)|MAX_HITS|REJECT_(TYPES|AGENTS|(NON)?PHYSICAL|LAND))|RCERR_(CAST_TIME_EXCEEDED|SIM_PERF_LOW|UNKNOWN)|ESTATE_ACCESS_(ALLOWED_(AGENT|GROUP)_(ADD|REMOVE)|BANNED_AGENT_(ADD|REMOVE))|DENSITY|FRICTION|RESTITUTION|GRAVITY_MULTIPLIER|KFM_(COMMAND|CMD_(PLAY|STOP|PAUSE)|MODE|FORWARD|LOOP|PING_PONG|REVERSE|DATA|ROTATION|TRANSLATION)|ERR_(GENERIC|PARCEL_PERMISSIONS|MALFORMED_PARAMS|RUNTIME_PERMISSIONS|THROTTLED)|CHARACTER_(CMD_((SMOOTH_)?STOP|JUMP)|DESIRED_(TURN_)?SPEED|RADIUS|STAY_WITHIN_PARCEL|LENGTH|ORIENTATION|ACCOUNT_FOR_SKIPPED_FRAMES|AVOIDANCE_MODE|TYPE(_([ABCD]|NONE))?|MAX_(DECEL|TURN_RADIUS|(ACCEL|SPEED)))|PURSUIT_(OFFSET|FUZZ_FACTOR|GOAL_TOLERANCE|INTERCEPT)|REQUIRE_LINE_OF_SIGHT|FORCE_DIRECT_PATH|VERTICAL|HORIZONTAL|AVOID_(CHARACTERS|DYNAMIC_OBSTACLES|NONE)|PU_(EVADE_(HIDDEN|SPOTTED)|FAILURE_(DYNAMIC_PATHFINDING_DISABLED|INVALID_(GOAL|START)|NO_(NAVMESH|VALID_DESTINATION)|OTHER|TARGET_GONE|(PARCEL_)?UNREACHABLE)|(GOAL|SLOWDOWN_DISTANCE)_REACHED)|TRAVERSAL_TYPE(_(FAST|NONE|SLOW))?|CONTENT_TYPE_(ATOM|FORM|HTML|JSON|LLSD|RSS|TEXT|XHTML|XML)|GCNP_(RADIUS|STATIC)|(PATROL|WANDER)_PAUSE_AT_WAYPOINTS|OPT_(AVATAR|CHARACTER|EXCLUSION_VOLUME|LEGACY_LINKSET|MATERIAL_VOLUME|OTHER|STATIC_OBSTACLE|WALKABLE)|SIM_STAT_PCT_CHARS_STEPPED)\\b'
	            },
	            {
	                begin: '\\b(FALSE|TRUE)\\b'
	            },
	            {
	                begin: '\\b(ZERO_ROTATION)\\b'
	            },
	            {
	                begin: '\\b(EOF|JSON_(ARRAY|DELETE|FALSE|INVALID|NULL|NUMBER|OBJECT|STRING|TRUE)|NULL_KEY|TEXTURE_(BLANK|DEFAULT|MEDIA|PLYWOOD|TRANSPARENT)|URL_REQUEST_(GRANTED|DENIED))\\b'
	            },
	            {
	                begin: '\\b(ZERO_VECTOR|TOUCH_INVALID_(TEXCOORD|VECTOR))\\b'
	            }
	        ]
	    };

	    var LSL_FUNCTIONS = {
	        className: 'built_in',
	        begin: '\\b(ll(AgentInExperience|(Create|DataSize|Delete|KeyCount|Keys|Read|Update)KeyValue|GetExperience(Details|ErrorMessage)|ReturnObjectsBy(ID|Owner)|Json(2List|[GS]etValue|ValueType)|Sin|Cos|Tan|Atan2|Sqrt|Pow|Abs|Fabs|Frand|Floor|Ceil|Round|Vec(Mag|Norm|Dist)|Rot(Between|2(Euler|Fwd|Left|Up))|(Euler|Axes)2Rot|Whisper|(Region|Owner)?Say|Shout|Listen(Control|Remove)?|Sensor(Repeat|Remove)?|Detected(Name|Key|Owner|Type|Pos|Vel|Grab|Rot|Group|LinkNumber)|Die|Ground|Wind|([GS]et)(AnimationOverride|MemoryLimit|PrimMediaParams|ParcelMusicURL|Object(Desc|Name)|PhysicsMaterial|Status|Scale|Color|Alpha|Texture|Pos|Rot|Force|Torque)|ResetAnimationOverride|(Scale|Offset|Rotate)Texture|(Rot)?Target(Remove)?|(Stop)?MoveToTarget|Apply(Rotational)?Impulse|Set(KeyframedMotion|ContentType|RegionPos|(Angular)?Velocity|Buoyancy|HoverHeight|ForceAndTorque|TimerEvent|ScriptState|Damage|TextureAnim|Sound(Queueing|Radius)|Vehicle(Type|(Float|Vector|Rotation)Param)|(Touch|Sit)?Text|Camera(Eye|At)Offset|PrimitiveParams|ClickAction|Link(Alpha|Color|PrimitiveParams(Fast)?|Texture(Anim)?|Camera|Media)|RemoteScriptAccessPin|PayPrice|LocalRot)|ScaleByFactor|Get((Max|Min)ScaleFactor|ClosestNavPoint|StaticPath|SimStats|Env|PrimitiveParams|Link(PrimitiveParams|Number(OfSides)?|Key|Name|Media)|HTTPHeader|FreeURLs|Object(Details|PermMask|PrimCount)|Parcel(MaxPrims|Details|Prim(Count|Owners))|Attached(List)?|(SPMax|Free|Used)Memory|Region(Name|TimeDilation|FPS|Corner|AgentCount)|Root(Position|Rotation)|UnixTime|(Parcel|Region)Flags|(Wall|GMT)clock|SimulatorHostname|BoundingBox|GeometricCenter|Creator|NumberOf(Prims|NotecardLines|Sides)|Animation(List)?|(Camera|Local)(Pos|Rot)|Vel|Accel|Omega|Time(stamp|OfDay)|(Object|CenterOf)?Mass|MassMKS|Energy|Owner|(Owner)?Key|SunDirection|Texture(Offset|Scale|Rot)|Inventory(Number|Name|Key|Type|Creator|PermMask)|Permissions(Key)?|StartParameter|List(Length|EntryType)|Date|Agent(Size|Info|Language|List)|LandOwnerAt|NotecardLine|Script(Name|State))|(Get|Reset|GetAndReset)Time|PlaySound(Slave)?|LoopSound(Master|Slave)?|(Trigger|Stop|Preload)Sound|((Get|Delete)Sub|Insert)String|To(Upper|Lower)|Give(InventoryList|Money)|RezObject|(Stop)?LookAt|Sleep|CollisionFilter|(Take|Release)Controls|DetachFromAvatar|AttachToAvatar(Temp)?|InstantMessage|(GetNext)?Email|StopHover|MinEventDelay|RotLookAt|String(Length|Trim)|(Start|Stop)Animation|TargetOmega|Request(Experience)?Permissions|(Create|Break)Link|BreakAllLinks|(Give|Remove)Inventory|Water|PassTouches|Request(Agent|Inventory)Data|TeleportAgent(Home|GlobalCoords)?|ModifyLand|CollisionSound|ResetScript|MessageLinked|PushObject|PassCollisions|AxisAngle2Rot|Rot2(Axis|Angle)|A(cos|sin)|AngleBetween|AllowInventoryDrop|SubStringIndex|List2(CSV|Integer|Json|Float|String|Key|Vector|Rot|List(Strided)?)|DeleteSubList|List(Statistics|Sort|Randomize|(Insert|Find|Replace)List)|EdgeOfWorld|AdjustSoundVolume|Key2Name|TriggerSoundLimited|EjectFromLand|(CSV|ParseString)2List|OverMyLand|SameGroup|UnSit|Ground(Slope|Normal|Contour)|GroundRepel|(Set|Remove)VehicleFlags|SitOnLink|(AvatarOn)?(Link)?SitTarget|Script(Danger|Profiler)|Dialog|VolumeDetect|ResetOtherScript|RemoteLoadScriptPin|(Open|Close)RemoteDataChannel|SendRemoteData|RemoteDataReply|(Integer|String)ToBase64|XorBase64|Log(10)?|Base64To(String|Integer)|ParseStringKeepNulls|RezAtRoot|RequestSimulatorData|ForceMouselook|(Load|Release|(E|Une)scape)URL|ParcelMedia(CommandList|Query)|ModPow|MapDestination|(RemoveFrom|AddTo|Reset)Land(Pass|Ban)List|(Set|Clear)CameraParams|HTTP(Request|Response)|TextBox|DetectedTouch(UV|Face|Pos|(N|Bin)ormal|ST)|(MD5|SHA1|DumpList2)String|Request(Secure)?URL|Clear(Prim|Link)Media|(Link)?ParticleSystem|(Get|Request)(Username|DisplayName)|RegionSayTo|CastRay|GenerateKey|TransferLindenDollars|ManageEstateAccess|(Create|Delete)Character|ExecCharacterCmd|Evade|FleeFrom|NavigateTo|PatrolPoints|Pursue|UpdateCharacter|WanderWithin))\\b'
	    };

	    return {
	        name: 'LSL (Linden Scripting Language)',
	        illegal: ':',
	        contains: [
	            LSL_STRINGS,
	            {
	                className: 'comment',
	                variants: [
	                    hljs.COMMENT('//', '$'),
	                    hljs.COMMENT('/\\*', '\\*/')
	                ],
	                relevance: 0
	            },
	            LSL_NUMBERS,
	            {
	                className: 'section',
	                variants: [
	                    {
	                        begin: '\\b(state|default)\\b'
	                    },
	                    {
	                        begin: '\\b(state_(entry|exit)|touch(_(start|end))?|(land_)?collision(_(start|end))?|timer|listen|(no_)?sensor|control|(not_)?at_(rot_)?target|money|email|experience_permissions(_denied)?|run_time_permissions|changed|attach|dataserver|moving_(start|end)|link_message|(on|object)_rez|remote_data|http_re(sponse|quest)|path_update|transaction_result)\\b'
	                    }
	                ]
	            },
	            LSL_FUNCTIONS,
	            LSL_CONSTANTS,
	            {
	                className: 'type',
	                begin: '\\b(integer|float|string|key|vector|quaternion|rotation|list)\\b'
	            }
	        ]
	    };
	}

	lsl_1 = lsl;
	return lsl_1;
}

/*
Language: Lua
Description: Lua is a powerful, efficient, lightweight, embeddable scripting language.
Author: Andrew Fedorov <dmmdrs@mail.ru>
Category: common, scripting
Website: https://www.lua.org
*/

var lua_1;
var hasRequiredLua;

function requireLua () {
	if (hasRequiredLua) return lua_1;
	hasRequiredLua = 1;
	function lua(hljs) {
	  const OPENING_LONG_BRACKET = '\\[=*\\[';
	  const CLOSING_LONG_BRACKET = '\\]=*\\]';
	  const LONG_BRACKETS = {
	    begin: OPENING_LONG_BRACKET,
	    end: CLOSING_LONG_BRACKET,
	    contains: ['self']
	  };
	  const COMMENTS = [
	    hljs.COMMENT('--(?!' + OPENING_LONG_BRACKET + ')', '$'),
	    hljs.COMMENT(
	      '--' + OPENING_LONG_BRACKET,
	      CLOSING_LONG_BRACKET,
	      {
	        contains: [LONG_BRACKETS],
	        relevance: 10
	      }
	    )
	  ];
	  return {
	    name: 'Lua',
	    keywords: {
	      $pattern: hljs.UNDERSCORE_IDENT_RE,
	      literal: "true false nil",
	      keyword: "and break do else elseif end for goto if in local not or repeat return then until while",
	      built_in:
	        // Metatags and globals:
	        '_G _ENV _VERSION __index __newindex __mode __call __metatable __tostring __len ' +
	        '__gc __add __sub __mul __div __mod __pow __concat __unm __eq __lt __le assert ' +
	        // Standard methods and properties:
	        'collectgarbage dofile error getfenv getmetatable ipairs load loadfile loadstring ' +
	        'module next pairs pcall print rawequal rawget rawset require select setfenv ' +
	        'setmetatable tonumber tostring type unpack xpcall arg self ' +
	        // Library methods and properties (one line per library):
	        'coroutine resume yield status wrap create running debug getupvalue ' +
	        'debug sethook getmetatable gethook setmetatable setlocal traceback setfenv getinfo setupvalue getlocal getregistry getfenv ' +
	        'io lines write close flush open output type read stderr stdin input stdout popen tmpfile ' +
	        'math log max acos huge ldexp pi cos tanh pow deg tan cosh sinh random randomseed frexp ceil floor rad abs sqrt modf asin min mod fmod log10 atan2 exp sin atan ' +
	        'os exit setlocale date getenv difftime remove time clock tmpname rename execute package preload loadlib loaded loaders cpath config path seeall ' +
	        'string sub upper len gfind rep find match char dump gmatch reverse byte format gsub lower ' +
	        'table setn insert getn foreachi maxn foreach concat sort remove'
	    },
	    contains: COMMENTS.concat([
	      {
	        className: 'function',
	        beginKeywords: 'function',
	        end: '\\)',
	        contains: [
	          hljs.inherit(hljs.TITLE_MODE, {
	            begin: '([_a-zA-Z]\\w*\\.)*([_a-zA-Z]\\w*:)?[_a-zA-Z]\\w*'
	          }),
	          {
	            className: 'params',
	            begin: '\\(',
	            endsWithParent: true,
	            contains: COMMENTS
	          }
	        ].concat(COMMENTS)
	      },
	      hljs.C_NUMBER_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'string',
	        begin: OPENING_LONG_BRACKET,
	        end: CLOSING_LONG_BRACKET,
	        contains: [LONG_BRACKETS],
	        relevance: 5
	      }
	    ])
	  };
	}

	lua_1 = lua;
	return lua_1;
}

/*
Language: Makefile
Author: Ivan Sagalaev <maniac@softwaremaniacs.org>
Contributors: Joël Porquet <joel@porquet.org>
Website: https://www.gnu.org/software/make/manual/html_node/Introduction.html
Category: common
*/

var makefile_1;
var hasRequiredMakefile;

function requireMakefile () {
	if (hasRequiredMakefile) return makefile_1;
	hasRequiredMakefile = 1;
	function makefile(hljs) {
	  /* Variables: simple (eg $(var)) and special (eg $@) */
	  const VARIABLE = {
	    className: 'variable',
	    variants: [
	      {
	        begin: '\\$\\(' + hljs.UNDERSCORE_IDENT_RE + '\\)',
	        contains: [ hljs.BACKSLASH_ESCAPE ]
	      },
	      {
	        begin: /\$[@%<?\^\+\*]/
	      }
	    ]
	  };
	  /* Quoted string with variables inside */
	  const QUOTE_STRING = {
	    className: 'string',
	    begin: /"/,
	    end: /"/,
	    contains: [
	      hljs.BACKSLASH_ESCAPE,
	      VARIABLE
	    ]
	  };
	  /* Function: $(func arg,...) */
	  const FUNC = {
	    className: 'variable',
	    begin: /\$\([\w-]+\s/,
	    end: /\)/,
	    keywords: {
	      built_in:
	        'subst patsubst strip findstring filter filter-out sort ' +
	        'word wordlist firstword lastword dir notdir suffix basename ' +
	        'addsuffix addprefix join wildcard realpath abspath error warning ' +
	        'shell origin flavor foreach if or and call eval file value'
	    },
	    contains: [ VARIABLE ]
	  };
	  /* Variable assignment */
	  const ASSIGNMENT = {
	    begin: '^' + hljs.UNDERSCORE_IDENT_RE + '\\s*(?=[:+?]?=)'
	  };
	  /* Meta targets (.PHONY) */
	  const META = {
	    className: 'meta',
	    begin: /^\.PHONY:/,
	    end: /$/,
	    keywords: {
	      $pattern: /[\.\w]+/,
	      'meta-keyword': '.PHONY'
	    }
	  };
	  /* Targets */
	  const TARGET = {
	    className: 'section',
	    begin: /^[^\s]+:/,
	    end: /$/,
	    contains: [ VARIABLE ]
	  };
	  return {
	    name: 'Makefile',
	    aliases: [
	      'mk',
	      'mak',
	      'make',
	    ],
	    keywords: {
	      $pattern: /[\w-]+/,
	      keyword: 'define endef undefine ifdef ifndef ifeq ifneq else endif ' +
	      'include -include sinclude override export unexport private vpath'
	    },
	    contains: [
	      hljs.HASH_COMMENT_MODE,
	      VARIABLE,
	      QUOTE_STRING,
	      FUNC,
	      ASSIGNMENT,
	      META,
	      TARGET
	    ]
	  };
	}

	makefile_1 = makefile;
	return makefile_1;
}

var mathematica_1;
var hasRequiredMathematica;

function requireMathematica () {
	if (hasRequiredMathematica) return mathematica_1;
	hasRequiredMathematica = 1;
	const SYSTEM_SYMBOLS = [
	  "AASTriangle",
	  "AbelianGroup",
	  "Abort",
	  "AbortKernels",
	  "AbortProtect",
	  "AbortScheduledTask",
	  "Above",
	  "Abs",
	  "AbsArg",
	  "AbsArgPlot",
	  "Absolute",
	  "AbsoluteCorrelation",
	  "AbsoluteCorrelationFunction",
	  "AbsoluteCurrentValue",
	  "AbsoluteDashing",
	  "AbsoluteFileName",
	  "AbsoluteOptions",
	  "AbsolutePointSize",
	  "AbsoluteThickness",
	  "AbsoluteTime",
	  "AbsoluteTiming",
	  "AcceptanceThreshold",
	  "AccountingForm",
	  "Accumulate",
	  "Accuracy",
	  "AccuracyGoal",
	  "ActionDelay",
	  "ActionMenu",
	  "ActionMenuBox",
	  "ActionMenuBoxOptions",
	  "Activate",
	  "Active",
	  "ActiveClassification",
	  "ActiveClassificationObject",
	  "ActiveItem",
	  "ActivePrediction",
	  "ActivePredictionObject",
	  "ActiveStyle",
	  "AcyclicGraphQ",
	  "AddOnHelpPath",
	  "AddSides",
	  "AddTo",
	  "AddToSearchIndex",
	  "AddUsers",
	  "AdjacencyGraph",
	  "AdjacencyList",
	  "AdjacencyMatrix",
	  "AdjacentMeshCells",
	  "AdjustmentBox",
	  "AdjustmentBoxOptions",
	  "AdjustTimeSeriesForecast",
	  "AdministrativeDivisionData",
	  "AffineHalfSpace",
	  "AffineSpace",
	  "AffineStateSpaceModel",
	  "AffineTransform",
	  "After",
	  "AggregatedEntityClass",
	  "AggregationLayer",
	  "AircraftData",
	  "AirportData",
	  "AirPressureData",
	  "AirTemperatureData",
	  "AiryAi",
	  "AiryAiPrime",
	  "AiryAiZero",
	  "AiryBi",
	  "AiryBiPrime",
	  "AiryBiZero",
	  "AlgebraicIntegerQ",
	  "AlgebraicNumber",
	  "AlgebraicNumberDenominator",
	  "AlgebraicNumberNorm",
	  "AlgebraicNumberPolynomial",
	  "AlgebraicNumberTrace",
	  "AlgebraicRules",
	  "AlgebraicRulesData",
	  "Algebraics",
	  "AlgebraicUnitQ",
	  "Alignment",
	  "AlignmentMarker",
	  "AlignmentPoint",
	  "All",
	  "AllowAdultContent",
	  "AllowedCloudExtraParameters",
	  "AllowedCloudParameterExtensions",
	  "AllowedDimensions",
	  "AllowedFrequencyRange",
	  "AllowedHeads",
	  "AllowGroupClose",
	  "AllowIncomplete",
	  "AllowInlineCells",
	  "AllowKernelInitialization",
	  "AllowLooseGrammar",
	  "AllowReverseGroupClose",
	  "AllowScriptLevelChange",
	  "AllowVersionUpdate",
	  "AllTrue",
	  "Alphabet",
	  "AlphabeticOrder",
	  "AlphabeticSort",
	  "AlphaChannel",
	  "AlternateImage",
	  "AlternatingFactorial",
	  "AlternatingGroup",
	  "AlternativeHypothesis",
	  "Alternatives",
	  "AltitudeMethod",
	  "AmbientLight",
	  "AmbiguityFunction",
	  "AmbiguityList",
	  "Analytic",
	  "AnatomyData",
	  "AnatomyForm",
	  "AnatomyPlot3D",
	  "AnatomySkinStyle",
	  "AnatomyStyling",
	  "AnchoredSearch",
	  "And",
	  "AndersonDarlingTest",
	  "AngerJ",
	  "AngleBisector",
	  "AngleBracket",
	  "AnglePath",
	  "AnglePath3D",
	  "AngleVector",
	  "AngularGauge",
	  "Animate",
	  "AnimationCycleOffset",
	  "AnimationCycleRepetitions",
	  "AnimationDirection",
	  "AnimationDisplayTime",
	  "AnimationRate",
	  "AnimationRepetitions",
	  "AnimationRunning",
	  "AnimationRunTime",
	  "AnimationTimeIndex",
	  "Animator",
	  "AnimatorBox",
	  "AnimatorBoxOptions",
	  "AnimatorElements",
	  "Annotate",
	  "Annotation",
	  "AnnotationDelete",
	  "AnnotationKeys",
	  "AnnotationRules",
	  "AnnotationValue",
	  "Annuity",
	  "AnnuityDue",
	  "Annulus",
	  "AnomalyDetection",
	  "AnomalyDetector",
	  "AnomalyDetectorFunction",
	  "Anonymous",
	  "Antialiasing",
	  "AntihermitianMatrixQ",
	  "Antisymmetric",
	  "AntisymmetricMatrixQ",
	  "Antonyms",
	  "AnyOrder",
	  "AnySubset",
	  "AnyTrue",
	  "Apart",
	  "ApartSquareFree",
	  "APIFunction",
	  "Appearance",
	  "AppearanceElements",
	  "AppearanceRules",
	  "AppellF1",
	  "Append",
	  "AppendCheck",
	  "AppendLayer",
	  "AppendTo",
	  "Apply",
	  "ApplySides",
	  "ArcCos",
	  "ArcCosh",
	  "ArcCot",
	  "ArcCoth",
	  "ArcCsc",
	  "ArcCsch",
	  "ArcCurvature",
	  "ARCHProcess",
	  "ArcLength",
	  "ArcSec",
	  "ArcSech",
	  "ArcSin",
	  "ArcSinDistribution",
	  "ArcSinh",
	  "ArcTan",
	  "ArcTanh",
	  "Area",
	  "Arg",
	  "ArgMax",
	  "ArgMin",
	  "ArgumentCountQ",
	  "ARIMAProcess",
	  "ArithmeticGeometricMean",
	  "ARMAProcess",
	  "Around",
	  "AroundReplace",
	  "ARProcess",
	  "Array",
	  "ArrayComponents",
	  "ArrayDepth",
	  "ArrayFilter",
	  "ArrayFlatten",
	  "ArrayMesh",
	  "ArrayPad",
	  "ArrayPlot",
	  "ArrayQ",
	  "ArrayResample",
	  "ArrayReshape",
	  "ArrayRules",
	  "Arrays",
	  "Arrow",
	  "Arrow3DBox",
	  "ArrowBox",
	  "Arrowheads",
	  "ASATriangle",
	  "Ask",
	  "AskAppend",
	  "AskConfirm",
	  "AskDisplay",
	  "AskedQ",
	  "AskedValue",
	  "AskFunction",
	  "AskState",
	  "AskTemplateDisplay",
	  "AspectRatio",
	  "AspectRatioFixed",
	  "Assert",
	  "AssociateTo",
	  "Association",
	  "AssociationFormat",
	  "AssociationMap",
	  "AssociationQ",
	  "AssociationThread",
	  "AssumeDeterministic",
	  "Assuming",
	  "Assumptions",
	  "AstronomicalData",
	  "Asymptotic",
	  "AsymptoticDSolveValue",
	  "AsymptoticEqual",
	  "AsymptoticEquivalent",
	  "AsymptoticGreater",
	  "AsymptoticGreaterEqual",
	  "AsymptoticIntegrate",
	  "AsymptoticLess",
	  "AsymptoticLessEqual",
	  "AsymptoticOutputTracker",
	  "AsymptoticProduct",
	  "AsymptoticRSolveValue",
	  "AsymptoticSolve",
	  "AsymptoticSum",
	  "Asynchronous",
	  "AsynchronousTaskObject",
	  "AsynchronousTasks",
	  "Atom",
	  "AtomCoordinates",
	  "AtomCount",
	  "AtomDiagramCoordinates",
	  "AtomList",
	  "AtomQ",
	  "AttentionLayer",
	  "Attributes",
	  "Audio",
	  "AudioAmplify",
	  "AudioAnnotate",
	  "AudioAnnotationLookup",
	  "AudioBlockMap",
	  "AudioCapture",
	  "AudioChannelAssignment",
	  "AudioChannelCombine",
	  "AudioChannelMix",
	  "AudioChannels",
	  "AudioChannelSeparate",
	  "AudioData",
	  "AudioDelay",
	  "AudioDelete",
	  "AudioDevice",
	  "AudioDistance",
	  "AudioEncoding",
	  "AudioFade",
	  "AudioFrequencyShift",
	  "AudioGenerator",
	  "AudioIdentify",
	  "AudioInputDevice",
	  "AudioInsert",
	  "AudioInstanceQ",
	  "AudioIntervals",
	  "AudioJoin",
	  "AudioLabel",
	  "AudioLength",
	  "AudioLocalMeasurements",
	  "AudioLooping",
	  "AudioLoudness",
	  "AudioMeasurements",
	  "AudioNormalize",
	  "AudioOutputDevice",
	  "AudioOverlay",
	  "AudioPad",
	  "AudioPan",
	  "AudioPartition",
	  "AudioPause",
	  "AudioPitchShift",
	  "AudioPlay",
	  "AudioPlot",
	  "AudioQ",
	  "AudioRecord",
	  "AudioReplace",
	  "AudioResample",
	  "AudioReverb",
	  "AudioReverse",
	  "AudioSampleRate",
	  "AudioSpectralMap",
	  "AudioSpectralTransformation",
	  "AudioSplit",
	  "AudioStop",
	  "AudioStream",
	  "AudioStreams",
	  "AudioTimeStretch",
	  "AudioTracks",
	  "AudioTrim",
	  "AudioType",
	  "AugmentedPolyhedron",
	  "AugmentedSymmetricPolynomial",
	  "Authenticate",
	  "Authentication",
	  "AuthenticationDialog",
	  "AutoAction",
	  "Autocomplete",
	  "AutocompletionFunction",
	  "AutoCopy",
	  "AutocorrelationTest",
	  "AutoDelete",
	  "AutoEvaluateEvents",
	  "AutoGeneratedPackage",
	  "AutoIndent",
	  "AutoIndentSpacings",
	  "AutoItalicWords",
	  "AutoloadPath",
	  "AutoMatch",
	  "Automatic",
	  "AutomaticImageSize",
	  "AutoMultiplicationSymbol",
	  "AutoNumberFormatting",
	  "AutoOpenNotebooks",
	  "AutoOpenPalettes",
	  "AutoQuoteCharacters",
	  "AutoRefreshed",
	  "AutoRemove",
	  "AutorunSequencing",
	  "AutoScaling",
	  "AutoScroll",
	  "AutoSpacing",
	  "AutoStyleOptions",
	  "AutoStyleWords",
	  "AutoSubmitting",
	  "Axes",
	  "AxesEdge",
	  "AxesLabel",
	  "AxesOrigin",
	  "AxesStyle",
	  "AxiomaticTheory",
	  "Axis",
	  "BabyMonsterGroupB",
	  "Back",
	  "Background",
	  "BackgroundAppearance",
	  "BackgroundTasksSettings",
	  "Backslash",
	  "Backsubstitution",
	  "Backward",
	  "Ball",
	  "Band",
	  "BandpassFilter",
	  "BandstopFilter",
	  "BarabasiAlbertGraphDistribution",
	  "BarChart",
	  "BarChart3D",
	  "BarcodeImage",
	  "BarcodeRecognize",
	  "BaringhausHenzeTest",
	  "BarLegend",
	  "BarlowProschanImportance",
	  "BarnesG",
	  "BarOrigin",
	  "BarSpacing",
	  "BartlettHannWindow",
	  "BartlettWindow",
	  "BaseDecode",
	  "BaseEncode",
	  "BaseForm",
	  "Baseline",
	  "BaselinePosition",
	  "BaseStyle",
	  "BasicRecurrentLayer",
	  "BatchNormalizationLayer",
	  "BatchSize",
	  "BatesDistribution",
	  "BattleLemarieWavelet",
	  "BayesianMaximization",
	  "BayesianMaximizationObject",
	  "BayesianMinimization",
	  "BayesianMinimizationObject",
	  "Because",
	  "BeckmannDistribution",
	  "Beep",
	  "Before",
	  "Begin",
	  "BeginDialogPacket",
	  "BeginFrontEndInteractionPacket",
	  "BeginPackage",
	  "BellB",
	  "BellY",
	  "Below",
	  "BenfordDistribution",
	  "BeniniDistribution",
	  "BenktanderGibratDistribution",
	  "BenktanderWeibullDistribution",
	  "BernoulliB",
	  "BernoulliDistribution",
	  "BernoulliGraphDistribution",
	  "BernoulliProcess",
	  "BernsteinBasis",
	  "BesselFilterModel",
	  "BesselI",
	  "BesselJ",
	  "BesselJZero",
	  "BesselK",
	  "BesselY",
	  "BesselYZero",
	  "Beta",
	  "BetaBinomialDistribution",
	  "BetaDistribution",
	  "BetaNegativeBinomialDistribution",
	  "BetaPrimeDistribution",
	  "BetaRegularized",
	  "Between",
	  "BetweennessCentrality",
	  "BeveledPolyhedron",
	  "BezierCurve",
	  "BezierCurve3DBox",
	  "BezierCurve3DBoxOptions",
	  "BezierCurveBox",
	  "BezierCurveBoxOptions",
	  "BezierFunction",
	  "BilateralFilter",
	  "Binarize",
	  "BinaryDeserialize",
	  "BinaryDistance",
	  "BinaryFormat",
	  "BinaryImageQ",
	  "BinaryRead",
	  "BinaryReadList",
	  "BinarySerialize",
	  "BinaryWrite",
	  "BinCounts",
	  "BinLists",
	  "Binomial",
	  "BinomialDistribution",
	  "BinomialProcess",
	  "BinormalDistribution",
	  "BiorthogonalSplineWavelet",
	  "BipartiteGraphQ",
	  "BiquadraticFilterModel",
	  "BirnbaumImportance",
	  "BirnbaumSaundersDistribution",
	  "BitAnd",
	  "BitClear",
	  "BitGet",
	  "BitLength",
	  "BitNot",
	  "BitOr",
	  "BitSet",
	  "BitShiftLeft",
	  "BitShiftRight",
	  "BitXor",
	  "BiweightLocation",
	  "BiweightMidvariance",
	  "Black",
	  "BlackmanHarrisWindow",
	  "BlackmanNuttallWindow",
	  "BlackmanWindow",
	  "Blank",
	  "BlankForm",
	  "BlankNullSequence",
	  "BlankSequence",
	  "Blend",
	  "Block",
	  "BlockchainAddressData",
	  "BlockchainBase",
	  "BlockchainBlockData",
	  "BlockchainContractValue",
	  "BlockchainData",
	  "BlockchainGet",
	  "BlockchainKeyEncode",
	  "BlockchainPut",
	  "BlockchainTokenData",
	  "BlockchainTransaction",
	  "BlockchainTransactionData",
	  "BlockchainTransactionSign",
	  "BlockchainTransactionSubmit",
	  "BlockMap",
	  "BlockRandom",
	  "BlomqvistBeta",
	  "BlomqvistBetaTest",
	  "Blue",
	  "Blur",
	  "BodePlot",
	  "BohmanWindow",
	  "Bold",
	  "Bond",
	  "BondCount",
	  "BondList",
	  "BondQ",
	  "Bookmarks",
	  "Boole",
	  "BooleanConsecutiveFunction",
	  "BooleanConvert",
	  "BooleanCountingFunction",
	  "BooleanFunction",
	  "BooleanGraph",
	  "BooleanMaxterms",
	  "BooleanMinimize",
	  "BooleanMinterms",
	  "BooleanQ",
	  "BooleanRegion",
	  "Booleans",
	  "BooleanStrings",
	  "BooleanTable",
	  "BooleanVariables",
	  "BorderDimensions",
	  "BorelTannerDistribution",
	  "Bottom",
	  "BottomHatTransform",
	  "BoundaryDiscretizeGraphics",
	  "BoundaryDiscretizeRegion",
	  "BoundaryMesh",
	  "BoundaryMeshRegion",
	  "BoundaryMeshRegionQ",
	  "BoundaryStyle",
	  "BoundedRegionQ",
	  "BoundingRegion",
	  "Bounds",
	  "Box",
	  "BoxBaselineShift",
	  "BoxData",
	  "BoxDimensions",
	  "Boxed",
	  "Boxes",
	  "BoxForm",
	  "BoxFormFormatTypes",
	  "BoxFrame",
	  "BoxID",
	  "BoxMargins",
	  "BoxMatrix",
	  "BoxObject",
	  "BoxRatios",
	  "BoxRotation",
	  "BoxRotationPoint",
	  "BoxStyle",
	  "BoxWhiskerChart",
	  "Bra",
	  "BracketingBar",
	  "BraKet",
	  "BrayCurtisDistance",
	  "BreadthFirstScan",
	  "Break",
	  "BridgeData",
	  "BrightnessEqualize",
	  "BroadcastStationData",
	  "Brown",
	  "BrownForsytheTest",
	  "BrownianBridgeProcess",
	  "BrowserCategory",
	  "BSplineBasis",
	  "BSplineCurve",
	  "BSplineCurve3DBox",
	  "BSplineCurve3DBoxOptions",
	  "BSplineCurveBox",
	  "BSplineCurveBoxOptions",
	  "BSplineFunction",
	  "BSplineSurface",
	  "BSplineSurface3DBox",
	  "BSplineSurface3DBoxOptions",
	  "BubbleChart",
	  "BubbleChart3D",
	  "BubbleScale",
	  "BubbleSizes",
	  "BuildingData",
	  "BulletGauge",
	  "BusinessDayQ",
	  "ButterflyGraph",
	  "ButterworthFilterModel",
	  "Button",
	  "ButtonBar",
	  "ButtonBox",
	  "ButtonBoxOptions",
	  "ButtonCell",
	  "ButtonContents",
	  "ButtonData",
	  "ButtonEvaluator",
	  "ButtonExpandable",
	  "ButtonFrame",
	  "ButtonFunction",
	  "ButtonMargins",
	  "ButtonMinHeight",
	  "ButtonNote",
	  "ButtonNotebook",
	  "ButtonSource",
	  "ButtonStyle",
	  "ButtonStyleMenuListing",
	  "Byte",
	  "ByteArray",
	  "ByteArrayFormat",
	  "ByteArrayQ",
	  "ByteArrayToString",
	  "ByteCount",
	  "ByteOrdering",
	  "C",
	  "CachedValue",
	  "CacheGraphics",
	  "CachePersistence",
	  "CalendarConvert",
	  "CalendarData",
	  "CalendarType",
	  "Callout",
	  "CalloutMarker",
	  "CalloutStyle",
	  "CallPacket",
	  "CanberraDistance",
	  "Cancel",
	  "CancelButton",
	  "CandlestickChart",
	  "CanonicalGraph",
	  "CanonicalizePolygon",
	  "CanonicalizePolyhedron",
	  "CanonicalName",
	  "CanonicalWarpingCorrespondence",
	  "CanonicalWarpingDistance",
	  "CantorMesh",
	  "CantorStaircase",
	  "Cap",
	  "CapForm",
	  "CapitalDifferentialD",
	  "Capitalize",
	  "CapsuleShape",
	  "CaptureRunning",
	  "CardinalBSplineBasis",
	  "CarlemanLinearize",
	  "CarmichaelLambda",
	  "CaseOrdering",
	  "Cases",
	  "CaseSensitive",
	  "Cashflow",
	  "Casoratian",
	  "Catalan",
	  "CatalanNumber",
	  "Catch",
	  "CategoricalDistribution",
	  "Catenate",
	  "CatenateLayer",
	  "CauchyDistribution",
	  "CauchyWindow",
	  "CayleyGraph",
	  "CDF",
	  "CDFDeploy",
	  "CDFInformation",
	  "CDFWavelet",
	  "Ceiling",
	  "CelestialSystem",
	  "Cell",
	  "CellAutoOverwrite",
	  "CellBaseline",
	  "CellBoundingBox",
	  "CellBracketOptions",
	  "CellChangeTimes",
	  "CellContents",
	  "CellContext",
	  "CellDingbat",
	  "CellDynamicExpression",
	  "CellEditDuplicate",
	  "CellElementsBoundingBox",
	  "CellElementSpacings",
	  "CellEpilog",
	  "CellEvaluationDuplicate",
	  "CellEvaluationFunction",
	  "CellEvaluationLanguage",
	  "CellEventActions",
	  "CellFrame",
	  "CellFrameColor",
	  "CellFrameLabelMargins",
	  "CellFrameLabels",
	  "CellFrameMargins",
	  "CellGroup",
	  "CellGroupData",
	  "CellGrouping",
	  "CellGroupingRules",
	  "CellHorizontalScrolling",
	  "CellID",
	  "CellLabel",
	  "CellLabelAutoDelete",
	  "CellLabelMargins",
	  "CellLabelPositioning",
	  "CellLabelStyle",
	  "CellLabelTemplate",
	  "CellMargins",
	  "CellObject",
	  "CellOpen",
	  "CellPrint",
	  "CellProlog",
	  "Cells",
	  "CellSize",
	  "CellStyle",
	  "CellTags",
	  "CellularAutomaton",
	  "CensoredDistribution",
	  "Censoring",
	  "Center",
	  "CenterArray",
	  "CenterDot",
	  "CentralFeature",
	  "CentralMoment",
	  "CentralMomentGeneratingFunction",
	  "Cepstrogram",
	  "CepstrogramArray",
	  "CepstrumArray",
	  "CForm",
	  "ChampernowneNumber",
	  "ChangeOptions",
	  "ChannelBase",
	  "ChannelBrokerAction",
	  "ChannelDatabin",
	  "ChannelHistoryLength",
	  "ChannelListen",
	  "ChannelListener",
	  "ChannelListeners",
	  "ChannelListenerWait",
	  "ChannelObject",
	  "ChannelPreSendFunction",
	  "ChannelReceiverFunction",
	  "ChannelSend",
	  "ChannelSubscribers",
	  "ChanVeseBinarize",
	  "Character",
	  "CharacterCounts",
	  "CharacterEncoding",
	  "CharacterEncodingsPath",
	  "CharacteristicFunction",
	  "CharacteristicPolynomial",
	  "CharacterName",
	  "CharacterNormalize",
	  "CharacterRange",
	  "Characters",
	  "ChartBaseStyle",
	  "ChartElementData",
	  "ChartElementDataFunction",
	  "ChartElementFunction",
	  "ChartElements",
	  "ChartLabels",
	  "ChartLayout",
	  "ChartLegends",
	  "ChartStyle",
	  "Chebyshev1FilterModel",
	  "Chebyshev2FilterModel",
	  "ChebyshevDistance",
	  "ChebyshevT",
	  "ChebyshevU",
	  "Check",
	  "CheckAbort",
	  "CheckAll",
	  "Checkbox",
	  "CheckboxBar",
	  "CheckboxBox",
	  "CheckboxBoxOptions",
	  "ChemicalData",
	  "ChessboardDistance",
	  "ChiDistribution",
	  "ChineseRemainder",
	  "ChiSquareDistribution",
	  "ChoiceButtons",
	  "ChoiceDialog",
	  "CholeskyDecomposition",
	  "Chop",
	  "ChromaticityPlot",
	  "ChromaticityPlot3D",
	  "ChromaticPolynomial",
	  "Circle",
	  "CircleBox",
	  "CircleDot",
	  "CircleMinus",
	  "CirclePlus",
	  "CirclePoints",
	  "CircleThrough",
	  "CircleTimes",
	  "CirculantGraph",
	  "CircularOrthogonalMatrixDistribution",
	  "CircularQuaternionMatrixDistribution",
	  "CircularRealMatrixDistribution",
	  "CircularSymplecticMatrixDistribution",
	  "CircularUnitaryMatrixDistribution",
	  "Circumsphere",
	  "CityData",
	  "ClassifierFunction",
	  "ClassifierInformation",
	  "ClassifierMeasurements",
	  "ClassifierMeasurementsObject",
	  "Classify",
	  "ClassPriors",
	  "Clear",
	  "ClearAll",
	  "ClearAttributes",
	  "ClearCookies",
	  "ClearPermissions",
	  "ClearSystemCache",
	  "ClebschGordan",
	  "ClickPane",
	  "Clip",
	  "ClipboardNotebook",
	  "ClipFill",
	  "ClippingStyle",
	  "ClipPlanes",
	  "ClipPlanesStyle",
	  "ClipRange",
	  "Clock",
	  "ClockGauge",
	  "ClockwiseContourIntegral",
	  "Close",
	  "Closed",
	  "CloseKernels",
	  "ClosenessCentrality",
	  "Closing",
	  "ClosingAutoSave",
	  "ClosingEvent",
	  "ClosingSaveDialog",
	  "CloudAccountData",
	  "CloudBase",
	  "CloudConnect",
	  "CloudConnections",
	  "CloudDeploy",
	  "CloudDirectory",
	  "CloudDisconnect",
	  "CloudEvaluate",
	  "CloudExport",
	  "CloudExpression",
	  "CloudExpressions",
	  "CloudFunction",
	  "CloudGet",
	  "CloudImport",
	  "CloudLoggingData",
	  "CloudObject",
	  "CloudObjectInformation",
	  "CloudObjectInformationData",
	  "CloudObjectNameFormat",
	  "CloudObjects",
	  "CloudObjectURLType",
	  "CloudPublish",
	  "CloudPut",
	  "CloudRenderingMethod",
	  "CloudSave",
	  "CloudShare",
	  "CloudSubmit",
	  "CloudSymbol",
	  "CloudUnshare",
	  "CloudUserID",
	  "ClusterClassify",
	  "ClusterDissimilarityFunction",
	  "ClusteringComponents",
	  "ClusteringTree",
	  "CMYKColor",
	  "Coarse",
	  "CodeAssistOptions",
	  "Coefficient",
	  "CoefficientArrays",
	  "CoefficientDomain",
	  "CoefficientList",
	  "CoefficientRules",
	  "CoifletWavelet",
	  "Collect",
	  "Colon",
	  "ColonForm",
	  "ColorBalance",
	  "ColorCombine",
	  "ColorConvert",
	  "ColorCoverage",
	  "ColorData",
	  "ColorDataFunction",
	  "ColorDetect",
	  "ColorDistance",
	  "ColorFunction",
	  "ColorFunctionScaling",
	  "Colorize",
	  "ColorNegate",
	  "ColorOutput",
	  "ColorProfileData",
	  "ColorQ",
	  "ColorQuantize",
	  "ColorReplace",
	  "ColorRules",
	  "ColorSelectorSettings",
	  "ColorSeparate",
	  "ColorSetter",
	  "ColorSetterBox",
	  "ColorSetterBoxOptions",
	  "ColorSlider",
	  "ColorsNear",
	  "ColorSpace",
	  "ColorToneMapping",
	  "Column",
	  "ColumnAlignments",
	  "ColumnBackgrounds",
	  "ColumnForm",
	  "ColumnLines",
	  "ColumnsEqual",
	  "ColumnSpacings",
	  "ColumnWidths",
	  "CombinedEntityClass",
	  "CombinerFunction",
	  "CometData",
	  "CommonDefaultFormatTypes",
	  "Commonest",
	  "CommonestFilter",
	  "CommonName",
	  "CommonUnits",
	  "CommunityBoundaryStyle",
	  "CommunityGraphPlot",
	  "CommunityLabels",
	  "CommunityRegionStyle",
	  "CompanyData",
	  "CompatibleUnitQ",
	  "CompilationOptions",
	  "CompilationTarget",
	  "Compile",
	  "Compiled",
	  "CompiledCodeFunction",
	  "CompiledFunction",
	  "CompilerOptions",
	  "Complement",
	  "ComplementedEntityClass",
	  "CompleteGraph",
	  "CompleteGraphQ",
	  "CompleteKaryTree",
	  "CompletionsListPacket",
	  "Complex",
	  "ComplexContourPlot",
	  "Complexes",
	  "ComplexExpand",
	  "ComplexInfinity",
	  "ComplexityFunction",
	  "ComplexListPlot",
	  "ComplexPlot",
	  "ComplexPlot3D",
	  "ComplexRegionPlot",
	  "ComplexStreamPlot",
	  "ComplexVectorPlot",
	  "ComponentMeasurements",
	  "ComponentwiseContextMenu",
	  "Compose",
	  "ComposeList",
	  "ComposeSeries",
	  "CompositeQ",
	  "Composition",
	  "CompoundElement",
	  "CompoundExpression",
	  "CompoundPoissonDistribution",
	  "CompoundPoissonProcess",
	  "CompoundRenewalProcess",
	  "Compress",
	  "CompressedData",
	  "CompressionLevel",
	  "ComputeUncertainty",
	  "Condition",
	  "ConditionalExpression",
	  "Conditioned",
	  "Cone",
	  "ConeBox",
	  "ConfidenceLevel",
	  "ConfidenceRange",
	  "ConfidenceTransform",
	  "ConfigurationPath",
	  "ConformAudio",
	  "ConformImages",
	  "Congruent",
	  "ConicHullRegion",
	  "ConicHullRegion3DBox",
	  "ConicHullRegionBox",
	  "ConicOptimization",
	  "Conjugate",
	  "ConjugateTranspose",
	  "Conjunction",
	  "Connect",
	  "ConnectedComponents",
	  "ConnectedGraphComponents",
	  "ConnectedGraphQ",
	  "ConnectedMeshComponents",
	  "ConnectedMoleculeComponents",
	  "ConnectedMoleculeQ",
	  "ConnectionSettings",
	  "ConnectLibraryCallbackFunction",
	  "ConnectSystemModelComponents",
	  "ConnesWindow",
	  "ConoverTest",
	  "ConsoleMessage",
	  "ConsoleMessagePacket",
	  "Constant",
	  "ConstantArray",
	  "ConstantArrayLayer",
	  "ConstantImage",
	  "ConstantPlusLayer",
	  "ConstantRegionQ",
	  "Constants",
	  "ConstantTimesLayer",
	  "ConstellationData",
	  "ConstrainedMax",
	  "ConstrainedMin",
	  "Construct",
	  "Containing",
	  "ContainsAll",
	  "ContainsAny",
	  "ContainsExactly",
	  "ContainsNone",
	  "ContainsOnly",
	  "ContentFieldOptions",
	  "ContentLocationFunction",
	  "ContentObject",
	  "ContentPadding",
	  "ContentsBoundingBox",
	  "ContentSelectable",
	  "ContentSize",
	  "Context",
	  "ContextMenu",
	  "Contexts",
	  "ContextToFileName",
	  "Continuation",
	  "Continue",
	  "ContinuedFraction",
	  "ContinuedFractionK",
	  "ContinuousAction",
	  "ContinuousMarkovProcess",
	  "ContinuousTask",
	  "ContinuousTimeModelQ",
	  "ContinuousWaveletData",
	  "ContinuousWaveletTransform",
	  "ContourDetect",
	  "ContourGraphics",
	  "ContourIntegral",
	  "ContourLabels",
	  "ContourLines",
	  "ContourPlot",
	  "ContourPlot3D",
	  "Contours",
	  "ContourShading",
	  "ContourSmoothing",
	  "ContourStyle",
	  "ContraharmonicMean",
	  "ContrastiveLossLayer",
	  "Control",
	  "ControlActive",
	  "ControlAlignment",
	  "ControlGroupContentsBox",
	  "ControllabilityGramian",
	  "ControllabilityMatrix",
	  "ControllableDecomposition",
	  "ControllableModelQ",
	  "ControllerDuration",
	  "ControllerInformation",
	  "ControllerInformationData",
	  "ControllerLinking",
	  "ControllerManipulate",
	  "ControllerMethod",
	  "ControllerPath",
	  "ControllerState",
	  "ControlPlacement",
	  "ControlsRendering",
	  "ControlType",
	  "Convergents",
	  "ConversionOptions",
	  "ConversionRules",
	  "ConvertToBitmapPacket",
	  "ConvertToPostScript",
	  "ConvertToPostScriptPacket",
	  "ConvexHullMesh",
	  "ConvexPolygonQ",
	  "ConvexPolyhedronQ",
	  "ConvolutionLayer",
	  "Convolve",
	  "ConwayGroupCo1",
	  "ConwayGroupCo2",
	  "ConwayGroupCo3",
	  "CookieFunction",
	  "Cookies",
	  "CoordinateBoundingBox",
	  "CoordinateBoundingBoxArray",
	  "CoordinateBounds",
	  "CoordinateBoundsArray",
	  "CoordinateChartData",
	  "CoordinatesToolOptions",
	  "CoordinateTransform",
	  "CoordinateTransformData",
	  "CoprimeQ",
	  "Coproduct",
	  "CopulaDistribution",
	  "Copyable",
	  "CopyDatabin",
	  "CopyDirectory",
	  "CopyFile",
	  "CopyTag",
	  "CopyToClipboard",
	  "CornerFilter",
	  "CornerNeighbors",
	  "Correlation",
	  "CorrelationDistance",
	  "CorrelationFunction",
	  "CorrelationTest",
	  "Cos",
	  "Cosh",
	  "CoshIntegral",
	  "CosineDistance",
	  "CosineWindow",
	  "CosIntegral",
	  "Cot",
	  "Coth",
	  "Count",
	  "CountDistinct",
	  "CountDistinctBy",
	  "CounterAssignments",
	  "CounterBox",
	  "CounterBoxOptions",
	  "CounterClockwiseContourIntegral",
	  "CounterEvaluator",
	  "CounterFunction",
	  "CounterIncrements",
	  "CounterStyle",
	  "CounterStyleMenuListing",
	  "CountRoots",
	  "CountryData",
	  "Counts",
	  "CountsBy",
	  "Covariance",
	  "CovarianceEstimatorFunction",
	  "CovarianceFunction",
	  "CoxianDistribution",
	  "CoxIngersollRossProcess",
	  "CoxModel",
	  "CoxModelFit",
	  "CramerVonMisesTest",
	  "CreateArchive",
	  "CreateCellID",
	  "CreateChannel",
	  "CreateCloudExpression",
	  "CreateDatabin",
	  "CreateDataStructure",
	  "CreateDataSystemModel",
	  "CreateDialog",
	  "CreateDirectory",
	  "CreateDocument",
	  "CreateFile",
	  "CreateIntermediateDirectories",
	  "CreateManagedLibraryExpression",
	  "CreateNotebook",
	  "CreatePacletArchive",
	  "CreatePalette",
	  "CreatePalettePacket",
	  "CreatePermissionsGroup",
	  "CreateScheduledTask",
	  "CreateSearchIndex",
	  "CreateSystemModel",
	  "CreateTemporary",
	  "CreateUUID",
	  "CreateWindow",
	  "CriterionFunction",
	  "CriticalityFailureImportance",
	  "CriticalitySuccessImportance",
	  "CriticalSection",
	  "Cross",
	  "CrossEntropyLossLayer",
	  "CrossingCount",
	  "CrossingDetect",
	  "CrossingPolygon",
	  "CrossMatrix",
	  "Csc",
	  "Csch",
	  "CTCLossLayer",
	  "Cube",
	  "CubeRoot",
	  "Cubics",
	  "Cuboid",
	  "CuboidBox",
	  "Cumulant",
	  "CumulantGeneratingFunction",
	  "Cup",
	  "CupCap",
	  "Curl",
	  "CurlyDoubleQuote",
	  "CurlyQuote",
	  "CurrencyConvert",
	  "CurrentDate",
	  "CurrentImage",
	  "CurrentlySpeakingPacket",
	  "CurrentNotebookImage",
	  "CurrentScreenImage",
	  "CurrentValue",
	  "Curry",
	  "CurryApplied",
	  "CurvatureFlowFilter",
	  "CurveClosed",
	  "Cyan",
	  "CycleGraph",
	  "CycleIndexPolynomial",
	  "Cycles",
	  "CyclicGroup",
	  "Cyclotomic",
	  "Cylinder",
	  "CylinderBox",
	  "CylindricalDecomposition",
	  "D",
	  "DagumDistribution",
	  "DamData",
	  "DamerauLevenshteinDistance",
	  "DampingFactor",
	  "Darker",
	  "Dashed",
	  "Dashing",
	  "DatabaseConnect",
	  "DatabaseDisconnect",
	  "DatabaseReference",
	  "Databin",
	  "DatabinAdd",
	  "DatabinRemove",
	  "Databins",
	  "DatabinUpload",
	  "DataCompression",
	  "DataDistribution",
	  "DataRange",
	  "DataReversed",
	  "Dataset",
	  "DatasetDisplayPanel",
	  "DataStructure",
	  "DataStructureQ",
	  "Date",
	  "DateBounds",
	  "Dated",
	  "DateDelimiters",
	  "DateDifference",
	  "DatedUnit",
	  "DateFormat",
	  "DateFunction",
	  "DateHistogram",
	  "DateInterval",
	  "DateList",
	  "DateListLogPlot",
	  "DateListPlot",
	  "DateListStepPlot",
	  "DateObject",
	  "DateObjectQ",
	  "DateOverlapsQ",
	  "DatePattern",
	  "DatePlus",
	  "DateRange",
	  "DateReduction",
	  "DateString",
	  "DateTicksFormat",
	  "DateValue",
	  "DateWithinQ",
	  "DaubechiesWavelet",
	  "DavisDistribution",
	  "DawsonF",
	  "DayCount",
	  "DayCountConvention",
	  "DayHemisphere",
	  "DaylightQ",
	  "DayMatchQ",
	  "DayName",
	  "DayNightTerminator",
	  "DayPlus",
	  "DayRange",
	  "DayRound",
	  "DeBruijnGraph",
	  "DeBruijnSequence",
	  "Debug",
	  "DebugTag",
	  "Decapitalize",
	  "Decimal",
	  "DecimalForm",
	  "DeclareKnownSymbols",
	  "DeclarePackage",
	  "Decompose",
	  "DeconvolutionLayer",
	  "Decrement",
	  "Decrypt",
	  "DecryptFile",
	  "DedekindEta",
	  "DeepSpaceProbeData",
	  "Default",
	  "DefaultAxesStyle",
	  "DefaultBaseStyle",
	  "DefaultBoxStyle",
	  "DefaultButton",
	  "DefaultColor",
	  "DefaultControlPlacement",
	  "DefaultDuplicateCellStyle",
	  "DefaultDuration",
	  "DefaultElement",
	  "DefaultFaceGridsStyle",
	  "DefaultFieldHintStyle",
	  "DefaultFont",
	  "DefaultFontProperties",
	  "DefaultFormatType",
	  "DefaultFormatTypeForStyle",
	  "DefaultFrameStyle",
	  "DefaultFrameTicksStyle",
	  "DefaultGridLinesStyle",
	  "DefaultInlineFormatType",
	  "DefaultInputFormatType",
	  "DefaultLabelStyle",
	  "DefaultMenuStyle",
	  "DefaultNaturalLanguage",
	  "DefaultNewCellStyle",
	  "DefaultNewInlineCellStyle",
	  "DefaultNotebook",
	  "DefaultOptions",
	  "DefaultOutputFormatType",
	  "DefaultPrintPrecision",
	  "DefaultStyle",
	  "DefaultStyleDefinitions",
	  "DefaultTextFormatType",
	  "DefaultTextInlineFormatType",
	  "DefaultTicksStyle",
	  "DefaultTooltipStyle",
	  "DefaultValue",
	  "DefaultValues",
	  "Defer",
	  "DefineExternal",
	  "DefineInputStreamMethod",
	  "DefineOutputStreamMethod",
	  "DefineResourceFunction",
	  "Definition",
	  "Degree",
	  "DegreeCentrality",
	  "DegreeGraphDistribution",
	  "DegreeLexicographic",
	  "DegreeReverseLexicographic",
	  "DEigensystem",
	  "DEigenvalues",
	  "Deinitialization",
	  "Del",
	  "DelaunayMesh",
	  "Delayed",
	  "Deletable",
	  "Delete",
	  "DeleteAnomalies",
	  "DeleteBorderComponents",
	  "DeleteCases",
	  "DeleteChannel",
	  "DeleteCloudExpression",
	  "DeleteContents",
	  "DeleteDirectory",
	  "DeleteDuplicates",
	  "DeleteDuplicatesBy",
	  "DeleteFile",
	  "DeleteMissing",
	  "DeleteObject",
	  "DeletePermissionsKey",
	  "DeleteSearchIndex",
	  "DeleteSmallComponents",
	  "DeleteStopwords",
	  "DeleteWithContents",
	  "DeletionWarning",
	  "DelimitedArray",
	  "DelimitedSequence",
	  "Delimiter",
	  "DelimiterFlashTime",
	  "DelimiterMatching",
	  "Delimiters",
	  "DeliveryFunction",
	  "Dendrogram",
	  "Denominator",
	  "DensityGraphics",
	  "DensityHistogram",
	  "DensityPlot",
	  "DensityPlot3D",
	  "DependentVariables",
	  "Deploy",
	  "Deployed",
	  "Depth",
	  "DepthFirstScan",
	  "Derivative",
	  "DerivativeFilter",
	  "DerivedKey",
	  "DescriptorStateSpace",
	  "DesignMatrix",
	  "DestroyAfterEvaluation",
	  "Det",
	  "DeviceClose",
	  "DeviceConfigure",
	  "DeviceExecute",
	  "DeviceExecuteAsynchronous",
	  "DeviceObject",
	  "DeviceOpen",
	  "DeviceOpenQ",
	  "DeviceRead",
	  "DeviceReadBuffer",
	  "DeviceReadLatest",
	  "DeviceReadList",
	  "DeviceReadTimeSeries",
	  "Devices",
	  "DeviceStreams",
	  "DeviceWrite",
	  "DeviceWriteBuffer",
	  "DGaussianWavelet",
	  "DiacriticalPositioning",
	  "Diagonal",
	  "DiagonalizableMatrixQ",
	  "DiagonalMatrix",
	  "DiagonalMatrixQ",
	  "Dialog",
	  "DialogIndent",
	  "DialogInput",
	  "DialogLevel",
	  "DialogNotebook",
	  "DialogProlog",
	  "DialogReturn",
	  "DialogSymbols",
	  "Diamond",
	  "DiamondMatrix",
	  "DiceDissimilarity",
	  "DictionaryLookup",
	  "DictionaryWordQ",
	  "DifferenceDelta",
	  "DifferenceOrder",
	  "DifferenceQuotient",
	  "DifferenceRoot",
	  "DifferenceRootReduce",
	  "Differences",
	  "DifferentialD",
	  "DifferentialRoot",
	  "DifferentialRootReduce",
	  "DifferentiatorFilter",
	  "DigitalSignature",
	  "DigitBlock",
	  "DigitBlockMinimum",
	  "DigitCharacter",
	  "DigitCount",
	  "DigitQ",
	  "DihedralAngle",
	  "DihedralGroup",
	  "Dilation",
	  "DimensionalCombinations",
	  "DimensionalMeshComponents",
	  "DimensionReduce",
	  "DimensionReducerFunction",
	  "DimensionReduction",
	  "Dimensions",
	  "DiracComb",
	  "DiracDelta",
	  "DirectedEdge",
	  "DirectedEdges",
	  "DirectedGraph",
	  "DirectedGraphQ",
	  "DirectedInfinity",
	  "Direction",
	  "Directive",
	  "Directory",
	  "DirectoryName",
	  "DirectoryQ",
	  "DirectoryStack",
	  "DirichletBeta",
	  "DirichletCharacter",
	  "DirichletCondition",
	  "DirichletConvolve",
	  "DirichletDistribution",
	  "DirichletEta",
	  "DirichletL",
	  "DirichletLambda",
	  "DirichletTransform",
	  "DirichletWindow",
	  "DisableConsolePrintPacket",
	  "DisableFormatting",
	  "DiscreteAsymptotic",
	  "DiscreteChirpZTransform",
	  "DiscreteConvolve",
	  "DiscreteDelta",
	  "DiscreteHadamardTransform",
	  "DiscreteIndicator",
	  "DiscreteLimit",
	  "DiscreteLQEstimatorGains",
	  "DiscreteLQRegulatorGains",
	  "DiscreteLyapunovSolve",
	  "DiscreteMarkovProcess",
	  "DiscreteMaxLimit",
	  "DiscreteMinLimit",
	  "DiscretePlot",
	  "DiscretePlot3D",
	  "DiscreteRatio",
	  "DiscreteRiccatiSolve",
	  "DiscreteShift",
	  "DiscreteTimeModelQ",
	  "DiscreteUniformDistribution",
	  "DiscreteVariables",
	  "DiscreteWaveletData",
	  "DiscreteWaveletPacketTransform",
	  "DiscreteWaveletTransform",
	  "DiscretizeGraphics",
	  "DiscretizeRegion",
	  "Discriminant",
	  "DisjointQ",
	  "Disjunction",
	  "Disk",
	  "DiskBox",
	  "DiskMatrix",
	  "DiskSegment",
	  "Dispatch",
	  "DispatchQ",
	  "DispersionEstimatorFunction",
	  "Display",
	  "DisplayAllSteps",
	  "DisplayEndPacket",
	  "DisplayFlushImagePacket",
	  "DisplayForm",
	  "DisplayFunction",
	  "DisplayPacket",
	  "DisplayRules",
	  "DisplaySetSizePacket",
	  "DisplayString",
	  "DisplayTemporary",
	  "DisplayWith",
	  "DisplayWithRef",
	  "DisplayWithVariable",
	  "DistanceFunction",
	  "DistanceMatrix",
	  "DistanceTransform",
	  "Distribute",
	  "Distributed",
	  "DistributedContexts",
	  "DistributeDefinitions",
	  "DistributionChart",
	  "DistributionDomain",
	  "DistributionFitTest",
	  "DistributionParameterAssumptions",
	  "DistributionParameterQ",
	  "Dithering",
	  "Div",
	  "Divergence",
	  "Divide",
	  "DivideBy",
	  "Dividers",
	  "DivideSides",
	  "Divisible",
	  "Divisors",
	  "DivisorSigma",
	  "DivisorSum",
	  "DMSList",
	  "DMSString",
	  "Do",
	  "DockedCells",
	  "DocumentGenerator",
	  "DocumentGeneratorInformation",
	  "DocumentGeneratorInformationData",
	  "DocumentGenerators",
	  "DocumentNotebook",
	  "DocumentWeightingRules",
	  "Dodecahedron",
	  "DomainRegistrationInformation",
	  "DominantColors",
	  "DOSTextFormat",
	  "Dot",
	  "DotDashed",
	  "DotEqual",
	  "DotLayer",
	  "DotPlusLayer",
	  "Dotted",
	  "DoubleBracketingBar",
	  "DoubleContourIntegral",
	  "DoubleDownArrow",
	  "DoubleLeftArrow",
	  "DoubleLeftRightArrow",
	  "DoubleLeftTee",
	  "DoubleLongLeftArrow",
	  "DoubleLongLeftRightArrow",
	  "DoubleLongRightArrow",
	  "DoubleRightArrow",
	  "DoubleRightTee",
	  "DoubleUpArrow",
	  "DoubleUpDownArrow",
	  "DoubleVerticalBar",
	  "DoublyInfinite",
	  "Down",
	  "DownArrow",
	  "DownArrowBar",
	  "DownArrowUpArrow",
	  "DownLeftRightVector",
	  "DownLeftTeeVector",
	  "DownLeftVector",
	  "DownLeftVectorBar",
	  "DownRightTeeVector",
	  "DownRightVector",
	  "DownRightVectorBar",
	  "Downsample",
	  "DownTee",
	  "DownTeeArrow",
	  "DownValues",
	  "DragAndDrop",
	  "DrawEdges",
	  "DrawFrontFaces",
	  "DrawHighlighted",
	  "Drop",
	  "DropoutLayer",
	  "DSolve",
	  "DSolveValue",
	  "Dt",
	  "DualLinearProgramming",
	  "DualPolyhedron",
	  "DualSystemsModel",
	  "DumpGet",
	  "DumpSave",
	  "DuplicateFreeQ",
	  "Duration",
	  "Dynamic",
	  "DynamicBox",
	  "DynamicBoxOptions",
	  "DynamicEvaluationTimeout",
	  "DynamicGeoGraphics",
	  "DynamicImage",
	  "DynamicLocation",
	  "DynamicModule",
	  "DynamicModuleBox",
	  "DynamicModuleBoxOptions",
	  "DynamicModuleParent",
	  "DynamicModuleValues",
	  "DynamicName",
	  "DynamicNamespace",
	  "DynamicReference",
	  "DynamicSetting",
	  "DynamicUpdating",
	  "DynamicWrapper",
	  "DynamicWrapperBox",
	  "DynamicWrapperBoxOptions",
	  "E",
	  "EarthImpactData",
	  "EarthquakeData",
	  "EccentricityCentrality",
	  "Echo",
	  "EchoFunction",
	  "EclipseType",
	  "EdgeAdd",
	  "EdgeBetweennessCentrality",
	  "EdgeCapacity",
	  "EdgeCapForm",
	  "EdgeColor",
	  "EdgeConnectivity",
	  "EdgeContract",
	  "EdgeCost",
	  "EdgeCount",
	  "EdgeCoverQ",
	  "EdgeCycleMatrix",
	  "EdgeDashing",
	  "EdgeDelete",
	  "EdgeDetect",
	  "EdgeForm",
	  "EdgeIndex",
	  "EdgeJoinForm",
	  "EdgeLabeling",
	  "EdgeLabels",
	  "EdgeLabelStyle",
	  "EdgeList",
	  "EdgeOpacity",
	  "EdgeQ",
	  "EdgeRenderingFunction",
	  "EdgeRules",
	  "EdgeShapeFunction",
	  "EdgeStyle",
	  "EdgeTaggedGraph",
	  "EdgeTaggedGraphQ",
	  "EdgeTags",
	  "EdgeThickness",
	  "EdgeWeight",
	  "EdgeWeightedGraphQ",
	  "Editable",
	  "EditButtonSettings",
	  "EditCellTagsSettings",
	  "EditDistance",
	  "EffectiveInterest",
	  "Eigensystem",
	  "Eigenvalues",
	  "EigenvectorCentrality",
	  "Eigenvectors",
	  "Element",
	  "ElementData",
	  "ElementwiseLayer",
	  "ElidedForms",
	  "Eliminate",
	  "EliminationOrder",
	  "Ellipsoid",
	  "EllipticE",
	  "EllipticExp",
	  "EllipticExpPrime",
	  "EllipticF",
	  "EllipticFilterModel",
	  "EllipticK",
	  "EllipticLog",
	  "EllipticNomeQ",
	  "EllipticPi",
	  "EllipticReducedHalfPeriods",
	  "EllipticTheta",
	  "EllipticThetaPrime",
	  "EmbedCode",
	  "EmbeddedHTML",
	  "EmbeddedService",
	  "EmbeddingLayer",
	  "EmbeddingObject",
	  "EmitSound",
	  "EmphasizeSyntaxErrors",
	  "EmpiricalDistribution",
	  "Empty",
	  "EmptyGraphQ",
	  "EmptyRegion",
	  "EnableConsolePrintPacket",
	  "Enabled",
	  "Encode",
	  "Encrypt",
	  "EncryptedObject",
	  "EncryptFile",
	  "End",
	  "EndAdd",
	  "EndDialogPacket",
	  "EndFrontEndInteractionPacket",
	  "EndOfBuffer",
	  "EndOfFile",
	  "EndOfLine",
	  "EndOfString",
	  "EndPackage",
	  "EngineEnvironment",
	  "EngineeringForm",
	  "Enter",
	  "EnterExpressionPacket",
	  "EnterTextPacket",
	  "Entity",
	  "EntityClass",
	  "EntityClassList",
	  "EntityCopies",
	  "EntityFunction",
	  "EntityGroup",
	  "EntityInstance",
	  "EntityList",
	  "EntityPrefetch",
	  "EntityProperties",
	  "EntityProperty",
	  "EntityPropertyClass",
	  "EntityRegister",
	  "EntityStore",
	  "EntityStores",
	  "EntityTypeName",
	  "EntityUnregister",
	  "EntityValue",
	  "Entropy",
	  "EntropyFilter",
	  "Environment",
	  "Epilog",
	  "EpilogFunction",
	  "Equal",
	  "EqualColumns",
	  "EqualRows",
	  "EqualTilde",
	  "EqualTo",
	  "EquatedTo",
	  "Equilibrium",
	  "EquirippleFilterKernel",
	  "Equivalent",
	  "Erf",
	  "Erfc",
	  "Erfi",
	  "ErlangB",
	  "ErlangC",
	  "ErlangDistribution",
	  "Erosion",
	  "ErrorBox",
	  "ErrorBoxOptions",
	  "ErrorNorm",
	  "ErrorPacket",
	  "ErrorsDialogSettings",
	  "EscapeRadius",
	  "EstimatedBackground",
	  "EstimatedDistribution",
	  "EstimatedProcess",
	  "EstimatorGains",
	  "EstimatorRegulator",
	  "EuclideanDistance",
	  "EulerAngles",
	  "EulerCharacteristic",
	  "EulerE",
	  "EulerGamma",
	  "EulerianGraphQ",
	  "EulerMatrix",
	  "EulerPhi",
	  "Evaluatable",
	  "Evaluate",
	  "Evaluated",
	  "EvaluatePacket",
	  "EvaluateScheduledTask",
	  "EvaluationBox",
	  "EvaluationCell",
	  "EvaluationCompletionAction",
	  "EvaluationData",
	  "EvaluationElements",
	  "EvaluationEnvironment",
	  "EvaluationMode",
	  "EvaluationMonitor",
	  "EvaluationNotebook",
	  "EvaluationObject",
	  "EvaluationOrder",
	  "Evaluator",
	  "EvaluatorNames",
	  "EvenQ",
	  "EventData",
	  "EventEvaluator",
	  "EventHandler",
	  "EventHandlerTag",
	  "EventLabels",
	  "EventSeries",
	  "ExactBlackmanWindow",
	  "ExactNumberQ",
	  "ExactRootIsolation",
	  "ExampleData",
	  "Except",
	  "ExcludedForms",
	  "ExcludedLines",
	  "ExcludedPhysicalQuantities",
	  "ExcludePods",
	  "Exclusions",
	  "ExclusionsStyle",
	  "Exists",
	  "Exit",
	  "ExitDialog",
	  "ExoplanetData",
	  "Exp",
	  "Expand",
	  "ExpandAll",
	  "ExpandDenominator",
	  "ExpandFileName",
	  "ExpandNumerator",
	  "Expectation",
	  "ExpectationE",
	  "ExpectedValue",
	  "ExpGammaDistribution",
	  "ExpIntegralE",
	  "ExpIntegralEi",
	  "ExpirationDate",
	  "Exponent",
	  "ExponentFunction",
	  "ExponentialDistribution",
	  "ExponentialFamily",
	  "ExponentialGeneratingFunction",
	  "ExponentialMovingAverage",
	  "ExponentialPowerDistribution",
	  "ExponentPosition",
	  "ExponentStep",
	  "Export",
	  "ExportAutoReplacements",
	  "ExportByteArray",
	  "ExportForm",
	  "ExportPacket",
	  "ExportString",
	  "Expression",
	  "ExpressionCell",
	  "ExpressionGraph",
	  "ExpressionPacket",
	  "ExpressionUUID",
	  "ExpToTrig",
	  "ExtendedEntityClass",
	  "ExtendedGCD",
	  "Extension",
	  "ExtentElementFunction",
	  "ExtentMarkers",
	  "ExtentSize",
	  "ExternalBundle",
	  "ExternalCall",
	  "ExternalDataCharacterEncoding",
	  "ExternalEvaluate",
	  "ExternalFunction",
	  "ExternalFunctionName",
	  "ExternalIdentifier",
	  "ExternalObject",
	  "ExternalOptions",
	  "ExternalSessionObject",
	  "ExternalSessions",
	  "ExternalStorageBase",
	  "ExternalStorageDownload",
	  "ExternalStorageGet",
	  "ExternalStorageObject",
	  "ExternalStoragePut",
	  "ExternalStorageUpload",
	  "ExternalTypeSignature",
	  "ExternalValue",
	  "Extract",
	  "ExtractArchive",
	  "ExtractLayer",
	  "ExtractPacletArchive",
	  "ExtremeValueDistribution",
	  "FaceAlign",
	  "FaceForm",
	  "FaceGrids",
	  "FaceGridsStyle",
	  "FacialFeatures",
	  "Factor",
	  "FactorComplete",
	  "Factorial",
	  "Factorial2",
	  "FactorialMoment",
	  "FactorialMomentGeneratingFunction",
	  "FactorialPower",
	  "FactorInteger",
	  "FactorList",
	  "FactorSquareFree",
	  "FactorSquareFreeList",
	  "FactorTerms",
	  "FactorTermsList",
	  "Fail",
	  "Failure",
	  "FailureAction",
	  "FailureDistribution",
	  "FailureQ",
	  "False",
	  "FareySequence",
	  "FARIMAProcess",
	  "FeatureDistance",
	  "FeatureExtract",
	  "FeatureExtraction",
	  "FeatureExtractor",
	  "FeatureExtractorFunction",
	  "FeatureNames",
	  "FeatureNearest",
	  "FeatureSpacePlot",
	  "FeatureSpacePlot3D",
	  "FeatureTypes",
	  "FEDisableConsolePrintPacket",
	  "FeedbackLinearize",
	  "FeedbackSector",
	  "FeedbackSectorStyle",
	  "FeedbackType",
	  "FEEnableConsolePrintPacket",
	  "FetalGrowthData",
	  "Fibonacci",
	  "Fibonorial",
	  "FieldCompletionFunction",
	  "FieldHint",
	  "FieldHintStyle",
	  "FieldMasked",
	  "FieldSize",
	  "File",
	  "FileBaseName",
	  "FileByteCount",
	  "FileConvert",
	  "FileDate",
	  "FileExistsQ",
	  "FileExtension",
	  "FileFormat",
	  "FileHandler",
	  "FileHash",
	  "FileInformation",
	  "FileName",
	  "FileNameDepth",
	  "FileNameDialogSettings",
	  "FileNameDrop",
	  "FileNameForms",
	  "FileNameJoin",
	  "FileNames",
	  "FileNameSetter",
	  "FileNameSplit",
	  "FileNameTake",
	  "FilePrint",
	  "FileSize",
	  "FileSystemMap",
	  "FileSystemScan",
	  "FileTemplate",
	  "FileTemplateApply",
	  "FileType",
	  "FilledCurve",
	  "FilledCurveBox",
	  "FilledCurveBoxOptions",
	  "Filling",
	  "FillingStyle",
	  "FillingTransform",
	  "FilteredEntityClass",
	  "FilterRules",
	  "FinancialBond",
	  "FinancialData",
	  "FinancialDerivative",
	  "FinancialIndicator",
	  "Find",
	  "FindAnomalies",
	  "FindArgMax",
	  "FindArgMin",
	  "FindChannels",
	  "FindClique",
	  "FindClusters",
	  "FindCookies",
	  "FindCurvePath",
	  "FindCycle",
	  "FindDevices",
	  "FindDistribution",
	  "FindDistributionParameters",
	  "FindDivisions",
	  "FindEdgeCover",
	  "FindEdgeCut",
	  "FindEdgeIndependentPaths",
	  "FindEquationalProof",
	  "FindEulerianCycle",
	  "FindExternalEvaluators",
	  "FindFaces",
	  "FindFile",
	  "FindFit",
	  "FindFormula",
	  "FindFundamentalCycles",
	  "FindGeneratingFunction",
	  "FindGeoLocation",
	  "FindGeometricConjectures",
	  "FindGeometricTransform",
	  "FindGraphCommunities",
	  "FindGraphIsomorphism",
	  "FindGraphPartition",
	  "FindHamiltonianCycle",
	  "FindHamiltonianPath",
	  "FindHiddenMarkovStates",
	  "FindImageText",
	  "FindIndependentEdgeSet",
	  "FindIndependentVertexSet",
	  "FindInstance",
	  "FindIntegerNullVector",
	  "FindKClan",
	  "FindKClique",
	  "FindKClub",
	  "FindKPlex",
	  "FindLibrary",
	  "FindLinearRecurrence",
	  "FindList",
	  "FindMatchingColor",
	  "FindMaximum",
	  "FindMaximumCut",
	  "FindMaximumFlow",
	  "FindMaxValue",
	  "FindMeshDefects",
	  "FindMinimum",
	  "FindMinimumCostFlow",
	  "FindMinimumCut",
	  "FindMinValue",
	  "FindMoleculeSubstructure",
	  "FindPath",
	  "FindPeaks",
	  "FindPermutation",
	  "FindPostmanTour",
	  "FindProcessParameters",
	  "FindRepeat",
	  "FindRoot",
	  "FindSequenceFunction",
	  "FindSettings",
	  "FindShortestPath",
	  "FindShortestTour",
	  "FindSpanningTree",
	  "FindSystemModelEquilibrium",
	  "FindTextualAnswer",
	  "FindThreshold",
	  "FindTransientRepeat",
	  "FindVertexCover",
	  "FindVertexCut",
	  "FindVertexIndependentPaths",
	  "Fine",
	  "FinishDynamic",
	  "FiniteAbelianGroupCount",
	  "FiniteGroupCount",
	  "FiniteGroupData",
	  "First",
	  "FirstCase",
	  "FirstPassageTimeDistribution",
	  "FirstPosition",
	  "FischerGroupFi22",
	  "FischerGroupFi23",
	  "FischerGroupFi24Prime",
	  "FisherHypergeometricDistribution",
	  "FisherRatioTest",
	  "FisherZDistribution",
	  "Fit",
	  "FitAll",
	  "FitRegularization",
	  "FittedModel",
	  "FixedOrder",
	  "FixedPoint",
	  "FixedPointList",
	  "FlashSelection",
	  "Flat",
	  "Flatten",
	  "FlattenAt",
	  "FlattenLayer",
	  "FlatTopWindow",
	  "FlipView",
	  "Floor",
	  "FlowPolynomial",
	  "FlushPrintOutputPacket",
	  "Fold",
	  "FoldList",
	  "FoldPair",
	  "FoldPairList",
	  "FollowRedirects",
	  "Font",
	  "FontColor",
	  "FontFamily",
	  "FontForm",
	  "FontName",
	  "FontOpacity",
	  "FontPostScriptName",
	  "FontProperties",
	  "FontReencoding",
	  "FontSize",
	  "FontSlant",
	  "FontSubstitutions",
	  "FontTracking",
	  "FontVariations",
	  "FontWeight",
	  "For",
	  "ForAll",
	  "ForceVersionInstall",
	  "Format",
	  "FormatRules",
	  "FormatType",
	  "FormatTypeAutoConvert",
	  "FormatValues",
	  "FormBox",
	  "FormBoxOptions",
	  "FormControl",
	  "FormFunction",
	  "FormLayoutFunction",
	  "FormObject",
	  "FormPage",
	  "FormTheme",
	  "FormulaData",
	  "FormulaLookup",
	  "FortranForm",
	  "Forward",
	  "ForwardBackward",
	  "Fourier",
	  "FourierCoefficient",
	  "FourierCosCoefficient",
	  "FourierCosSeries",
	  "FourierCosTransform",
	  "FourierDCT",
	  "FourierDCTFilter",
	  "FourierDCTMatrix",
	  "FourierDST",
	  "FourierDSTMatrix",
	  "FourierMatrix",
	  "FourierParameters",
	  "FourierSequenceTransform",
	  "FourierSeries",
	  "FourierSinCoefficient",
	  "FourierSinSeries",
	  "FourierSinTransform",
	  "FourierTransform",
	  "FourierTrigSeries",
	  "FractionalBrownianMotionProcess",
	  "FractionalGaussianNoiseProcess",
	  "FractionalPart",
	  "FractionBox",
	  "FractionBoxOptions",
	  "FractionLine",
	  "Frame",
	  "FrameBox",
	  "FrameBoxOptions",
	  "Framed",
	  "FrameInset",
	  "FrameLabel",
	  "Frameless",
	  "FrameMargins",
	  "FrameRate",
	  "FrameStyle",
	  "FrameTicks",
	  "FrameTicksStyle",
	  "FRatioDistribution",
	  "FrechetDistribution",
	  "FreeQ",
	  "FrenetSerretSystem",
	  "FrequencySamplingFilterKernel",
	  "FresnelC",
	  "FresnelF",
	  "FresnelG",
	  "FresnelS",
	  "Friday",
	  "FrobeniusNumber",
	  "FrobeniusSolve",
	  "FromAbsoluteTime",
	  "FromCharacterCode",
	  "FromCoefficientRules",
	  "FromContinuedFraction",
	  "FromDate",
	  "FromDigits",
	  "FromDMS",
	  "FromEntity",
	  "FromJulianDate",
	  "FromLetterNumber",
	  "FromPolarCoordinates",
	  "FromRomanNumeral",
	  "FromSphericalCoordinates",
	  "FromUnixTime",
	  "Front",
	  "FrontEndDynamicExpression",
	  "FrontEndEventActions",
	  "FrontEndExecute",
	  "FrontEndObject",
	  "FrontEndResource",
	  "FrontEndResourceString",
	  "FrontEndStackSize",
	  "FrontEndToken",
	  "FrontEndTokenExecute",
	  "FrontEndValueCache",
	  "FrontEndVersion",
	  "FrontFaceColor",
	  "FrontFaceOpacity",
	  "Full",
	  "FullAxes",
	  "FullDefinition",
	  "FullForm",
	  "FullGraphics",
	  "FullInformationOutputRegulator",
	  "FullOptions",
	  "FullRegion",
	  "FullSimplify",
	  "Function",
	  "FunctionCompile",
	  "FunctionCompileExport",
	  "FunctionCompileExportByteArray",
	  "FunctionCompileExportLibrary",
	  "FunctionCompileExportString",
	  "FunctionDomain",
	  "FunctionExpand",
	  "FunctionInterpolation",
	  "FunctionPeriod",
	  "FunctionRange",
	  "FunctionSpace",
	  "FussellVeselyImportance",
	  "GaborFilter",
	  "GaborMatrix",
	  "GaborWavelet",
	  "GainMargins",
	  "GainPhaseMargins",
	  "GalaxyData",
	  "GalleryView",
	  "Gamma",
	  "GammaDistribution",
	  "GammaRegularized",
	  "GapPenalty",
	  "GARCHProcess",
	  "GatedRecurrentLayer",
	  "Gather",
	  "GatherBy",
	  "GaugeFaceElementFunction",
	  "GaugeFaceStyle",
	  "GaugeFrameElementFunction",
	  "GaugeFrameSize",
	  "GaugeFrameStyle",
	  "GaugeLabels",
	  "GaugeMarkers",
	  "GaugeStyle",
	  "GaussianFilter",
	  "GaussianIntegers",
	  "GaussianMatrix",
	  "GaussianOrthogonalMatrixDistribution",
	  "GaussianSymplecticMatrixDistribution",
	  "GaussianUnitaryMatrixDistribution",
	  "GaussianWindow",
	  "GCD",
	  "GegenbauerC",
	  "General",
	  "GeneralizedLinearModelFit",
	  "GenerateAsymmetricKeyPair",
	  "GenerateConditions",
	  "GeneratedCell",
	  "GeneratedDocumentBinding",
	  "GenerateDerivedKey",
	  "GenerateDigitalSignature",
	  "GenerateDocument",
	  "GeneratedParameters",
	  "GeneratedQuantityMagnitudes",
	  "GenerateFileSignature",
	  "GenerateHTTPResponse",
	  "GenerateSecuredAuthenticationKey",
	  "GenerateSymmetricKey",
	  "GeneratingFunction",
	  "GeneratorDescription",
	  "GeneratorHistoryLength",
	  "GeneratorOutputType",
	  "Generic",
	  "GenericCylindricalDecomposition",
	  "GenomeData",
	  "GenomeLookup",
	  "GeoAntipode",
	  "GeoArea",
	  "GeoArraySize",
	  "GeoBackground",
	  "GeoBoundingBox",
	  "GeoBounds",
	  "GeoBoundsRegion",
	  "GeoBubbleChart",
	  "GeoCenter",
	  "GeoCircle",
	  "GeoContourPlot",
	  "GeoDensityPlot",
	  "GeodesicClosing",
	  "GeodesicDilation",
	  "GeodesicErosion",
	  "GeodesicOpening",
	  "GeoDestination",
	  "GeodesyData",
	  "GeoDirection",
	  "GeoDisk",
	  "GeoDisplacement",
	  "GeoDistance",
	  "GeoDistanceList",
	  "GeoElevationData",
	  "GeoEntities",
	  "GeoGraphics",
	  "GeogravityModelData",
	  "GeoGridDirectionDifference",
	  "GeoGridLines",
	  "GeoGridLinesStyle",
	  "GeoGridPosition",
	  "GeoGridRange",
	  "GeoGridRangePadding",
	  "GeoGridUnitArea",
	  "GeoGridUnitDistance",
	  "GeoGridVector",
	  "GeoGroup",
	  "GeoHemisphere",
	  "GeoHemisphereBoundary",
	  "GeoHistogram",
	  "GeoIdentify",
	  "GeoImage",
	  "GeoLabels",
	  "GeoLength",
	  "GeoListPlot",
	  "GeoLocation",
	  "GeologicalPeriodData",
	  "GeomagneticModelData",
	  "GeoMarker",
	  "GeometricAssertion",
	  "GeometricBrownianMotionProcess",
	  "GeometricDistribution",
	  "GeometricMean",
	  "GeometricMeanFilter",
	  "GeometricOptimization",
	  "GeometricScene",
	  "GeometricTransformation",
	  "GeometricTransformation3DBox",
	  "GeometricTransformation3DBoxOptions",
	  "GeometricTransformationBox",
	  "GeometricTransformationBoxOptions",
	  "GeoModel",
	  "GeoNearest",
	  "GeoPath",
	  "GeoPosition",
	  "GeoPositionENU",
	  "GeoPositionXYZ",
	  "GeoProjection",
	  "GeoProjectionData",
	  "GeoRange",
	  "GeoRangePadding",
	  "GeoRegionValuePlot",
	  "GeoResolution",
	  "GeoScaleBar",
	  "GeoServer",
	  "GeoSmoothHistogram",
	  "GeoStreamPlot",
	  "GeoStyling",
	  "GeoStylingImageFunction",
	  "GeoVariant",
	  "GeoVector",
	  "GeoVectorENU",
	  "GeoVectorPlot",
	  "GeoVectorXYZ",
	  "GeoVisibleRegion",
	  "GeoVisibleRegionBoundary",
	  "GeoWithinQ",
	  "GeoZoomLevel",
	  "GestureHandler",
	  "GestureHandlerTag",
	  "Get",
	  "GetBoundingBoxSizePacket",
	  "GetContext",
	  "GetEnvironment",
	  "GetFileName",
	  "GetFrontEndOptionsDataPacket",
	  "GetLinebreakInformationPacket",
	  "GetMenusPacket",
	  "GetPageBreakInformationPacket",
	  "Glaisher",
	  "GlobalClusteringCoefficient",
	  "GlobalPreferences",
	  "GlobalSession",
	  "Glow",
	  "GoldenAngle",
	  "GoldenRatio",
	  "GompertzMakehamDistribution",
	  "GoochShading",
	  "GoodmanKruskalGamma",
	  "GoodmanKruskalGammaTest",
	  "Goto",
	  "Grad",
	  "Gradient",
	  "GradientFilter",
	  "GradientOrientationFilter",
	  "GrammarApply",
	  "GrammarRules",
	  "GrammarToken",
	  "Graph",
	  "Graph3D",
	  "GraphAssortativity",
	  "GraphAutomorphismGroup",
	  "GraphCenter",
	  "GraphComplement",
	  "GraphData",
	  "GraphDensity",
	  "GraphDiameter",
	  "GraphDifference",
	  "GraphDisjointUnion",
	  "GraphDistance",
	  "GraphDistanceMatrix",
	  "GraphElementData",
	  "GraphEmbedding",
	  "GraphHighlight",
	  "GraphHighlightStyle",
	  "GraphHub",
	  "Graphics",
	  "Graphics3D",
	  "Graphics3DBox",
	  "Graphics3DBoxOptions",
	  "GraphicsArray",
	  "GraphicsBaseline",
	  "GraphicsBox",
	  "GraphicsBoxOptions",
	  "GraphicsColor",
	  "GraphicsColumn",
	  "GraphicsComplex",
	  "GraphicsComplex3DBox",
	  "GraphicsComplex3DBoxOptions",
	  "GraphicsComplexBox",
	  "GraphicsComplexBoxOptions",
	  "GraphicsContents",
	  "GraphicsData",
	  "GraphicsGrid",
	  "GraphicsGridBox",
	  "GraphicsGroup",
	  "GraphicsGroup3DBox",
	  "GraphicsGroup3DBoxOptions",
	  "GraphicsGroupBox",
	  "GraphicsGroupBoxOptions",
	  "GraphicsGrouping",
	  "GraphicsHighlightColor",
	  "GraphicsRow",
	  "GraphicsSpacing",
	  "GraphicsStyle",
	  "GraphIntersection",
	  "GraphLayout",
	  "GraphLinkEfficiency",
	  "GraphPeriphery",
	  "GraphPlot",
	  "GraphPlot3D",
	  "GraphPower",
	  "GraphPropertyDistribution",
	  "GraphQ",
	  "GraphRadius",
	  "GraphReciprocity",
	  "GraphRoot",
	  "GraphStyle",
	  "GraphUnion",
	  "Gray",
	  "GrayLevel",
	  "Greater",
	  "GreaterEqual",
	  "GreaterEqualLess",
	  "GreaterEqualThan",
	  "GreaterFullEqual",
	  "GreaterGreater",
	  "GreaterLess",
	  "GreaterSlantEqual",
	  "GreaterThan",
	  "GreaterTilde",
	  "Green",
	  "GreenFunction",
	  "Grid",
	  "GridBaseline",
	  "GridBox",
	  "GridBoxAlignment",
	  "GridBoxBackground",
	  "GridBoxDividers",
	  "GridBoxFrame",
	  "GridBoxItemSize",
	  "GridBoxItemStyle",
	  "GridBoxOptions",
	  "GridBoxSpacings",
	  "GridCreationSettings",
	  "GridDefaultElement",
	  "GridElementStyleOptions",
	  "GridFrame",
	  "GridFrameMargins",
	  "GridGraph",
	  "GridLines",
	  "GridLinesStyle",
	  "GroebnerBasis",
	  "GroupActionBase",
	  "GroupBy",
	  "GroupCentralizer",
	  "GroupElementFromWord",
	  "GroupElementPosition",
	  "GroupElementQ",
	  "GroupElements",
	  "GroupElementToWord",
	  "GroupGenerators",
	  "Groupings",
	  "GroupMultiplicationTable",
	  "GroupOrbits",
	  "GroupOrder",
	  "GroupPageBreakWithin",
	  "GroupSetwiseStabilizer",
	  "GroupStabilizer",
	  "GroupStabilizerChain",
	  "GroupTogetherGrouping",
	  "GroupTogetherNestedGrouping",
	  "GrowCutComponents",
	  "Gudermannian",
	  "GuidedFilter",
	  "GumbelDistribution",
	  "HaarWavelet",
	  "HadamardMatrix",
	  "HalfLine",
	  "HalfNormalDistribution",
	  "HalfPlane",
	  "HalfSpace",
	  "HalftoneShading",
	  "HamiltonianGraphQ",
	  "HammingDistance",
	  "HammingWindow",
	  "HandlerFunctions",
	  "HandlerFunctionsKeys",
	  "HankelH1",
	  "HankelH2",
	  "HankelMatrix",
	  "HankelTransform",
	  "HannPoissonWindow",
	  "HannWindow",
	  "HaradaNortonGroupHN",
	  "HararyGraph",
	  "HarmonicMean",
	  "HarmonicMeanFilter",
	  "HarmonicNumber",
	  "Hash",
	  "HatchFilling",
	  "HatchShading",
	  "Haversine",
	  "HazardFunction",
	  "Head",
	  "HeadCompose",
	  "HeaderAlignment",
	  "HeaderBackground",
	  "HeaderDisplayFunction",
	  "HeaderLines",
	  "HeaderSize",
	  "HeaderStyle",
	  "Heads",
	  "HeavisideLambda",
	  "HeavisidePi",
	  "HeavisideTheta",
	  "HeldGroupHe",
	  "HeldPart",
	  "HelpBrowserLookup",
	  "HelpBrowserNotebook",
	  "HelpBrowserSettings",
	  "Here",
	  "HermiteDecomposition",
	  "HermiteH",
	  "HermitianMatrixQ",
	  "HessenbergDecomposition",
	  "Hessian",
	  "HeunB",
	  "HeunBPrime",
	  "HeunC",
	  "HeunCPrime",
	  "HeunD",
	  "HeunDPrime",
	  "HeunG",
	  "HeunGPrime",
	  "HeunT",
	  "HeunTPrime",
	  "HexadecimalCharacter",
	  "Hexahedron",
	  "HexahedronBox",
	  "HexahedronBoxOptions",
	  "HiddenItems",
	  "HiddenMarkovProcess",
	  "HiddenSurface",
	  "Highlighted",
	  "HighlightGraph",
	  "HighlightImage",
	  "HighlightMesh",
	  "HighpassFilter",
	  "HigmanSimsGroupHS",
	  "HilbertCurve",
	  "HilbertFilter",
	  "HilbertMatrix",
	  "Histogram",
	  "Histogram3D",
	  "HistogramDistribution",
	  "HistogramList",
	  "HistogramTransform",
	  "HistogramTransformInterpolation",
	  "HistoricalPeriodData",
	  "HitMissTransform",
	  "HITSCentrality",
	  "HjorthDistribution",
	  "HodgeDual",
	  "HoeffdingD",
	  "HoeffdingDTest",
	  "Hold",
	  "HoldAll",
	  "HoldAllComplete",
	  "HoldComplete",
	  "HoldFirst",
	  "HoldForm",
	  "HoldPattern",
	  "HoldRest",
	  "HolidayCalendar",
	  "HomeDirectory",
	  "HomePage",
	  "Horizontal",
	  "HorizontalForm",
	  "HorizontalGauge",
	  "HorizontalScrollPosition",
	  "HornerForm",
	  "HostLookup",
	  "HotellingTSquareDistribution",
	  "HoytDistribution",
	  "HTMLSave",
	  "HTTPErrorResponse",
	  "HTTPRedirect",
	  "HTTPRequest",
	  "HTTPRequestData",
	  "HTTPResponse",
	  "Hue",
	  "HumanGrowthData",
	  "HumpDownHump",
	  "HumpEqual",
	  "HurwitzLerchPhi",
	  "HurwitzZeta",
	  "HyperbolicDistribution",
	  "HypercubeGraph",
	  "HyperexponentialDistribution",
	  "Hyperfactorial",
	  "Hypergeometric0F1",
	  "Hypergeometric0F1Regularized",
	  "Hypergeometric1F1",
	  "Hypergeometric1F1Regularized",
	  "Hypergeometric2F1",
	  "Hypergeometric2F1Regularized",
	  "HypergeometricDistribution",
	  "HypergeometricPFQ",
	  "HypergeometricPFQRegularized",
	  "HypergeometricU",
	  "Hyperlink",
	  "HyperlinkAction",
	  "HyperlinkCreationSettings",
	  "Hyperplane",
	  "Hyphenation",
	  "HyphenationOptions",
	  "HypoexponentialDistribution",
	  "HypothesisTestData",
	  "I",
	  "IconData",
	  "Iconize",
	  "IconizedObject",
	  "IconRules",
	  "Icosahedron",
	  "Identity",
	  "IdentityMatrix",
	  "If",
	  "IgnoreCase",
	  "IgnoreDiacritics",
	  "IgnorePunctuation",
	  "IgnoreSpellCheck",
	  "IgnoringInactive",
	  "Im",
	  "Image",
	  "Image3D",
	  "Image3DProjection",
	  "Image3DSlices",
	  "ImageAccumulate",
	  "ImageAdd",
	  "ImageAdjust",
	  "ImageAlign",
	  "ImageApply",
	  "ImageApplyIndexed",
	  "ImageAspectRatio",
	  "ImageAssemble",
	  "ImageAugmentationLayer",
	  "ImageBoundingBoxes",
	  "ImageCache",
	  "ImageCacheValid",
	  "ImageCapture",
	  "ImageCaptureFunction",
	  "ImageCases",
	  "ImageChannels",
	  "ImageClip",
	  "ImageCollage",
	  "ImageColorSpace",
	  "ImageCompose",
	  "ImageContainsQ",
	  "ImageContents",
	  "ImageConvolve",
	  "ImageCooccurrence",
	  "ImageCorners",
	  "ImageCorrelate",
	  "ImageCorrespondingPoints",
	  "ImageCrop",
	  "ImageData",
	  "ImageDeconvolve",
	  "ImageDemosaic",
	  "ImageDifference",
	  "ImageDimensions",
	  "ImageDisplacements",
	  "ImageDistance",
	  "ImageEffect",
	  "ImageExposureCombine",
	  "ImageFeatureTrack",
	  "ImageFileApply",
	  "ImageFileFilter",
	  "ImageFileScan",
	  "ImageFilter",
	  "ImageFocusCombine",
	  "ImageForestingComponents",
	  "ImageFormattingWidth",
	  "ImageForwardTransformation",
	  "ImageGraphics",
	  "ImageHistogram",
	  "ImageIdentify",
	  "ImageInstanceQ",
	  "ImageKeypoints",
	  "ImageLabels",
	  "ImageLegends",
	  "ImageLevels",
	  "ImageLines",
	  "ImageMargins",
	  "ImageMarker",
	  "ImageMarkers",
	  "ImageMeasurements",
	  "ImageMesh",
	  "ImageMultiply",
	  "ImageOffset",
	  "ImagePad",
	  "ImagePadding",
	  "ImagePartition",
	  "ImagePeriodogram",
	  "ImagePerspectiveTransformation",
	  "ImagePosition",
	  "ImagePreviewFunction",
	  "ImagePyramid",
	  "ImagePyramidApply",
	  "ImageQ",
	  "ImageRangeCache",
	  "ImageRecolor",
	  "ImageReflect",
	  "ImageRegion",
	  "ImageResize",
	  "ImageResolution",
	  "ImageRestyle",
	  "ImageRotate",
	  "ImageRotated",
	  "ImageSaliencyFilter",
	  "ImageScaled",
	  "ImageScan",
	  "ImageSize",
	  "ImageSizeAction",
	  "ImageSizeCache",
	  "ImageSizeMultipliers",
	  "ImageSizeRaw",
	  "ImageSubtract",
	  "ImageTake",
	  "ImageTransformation",
	  "ImageTrim",
	  "ImageType",
	  "ImageValue",
	  "ImageValuePositions",
	  "ImagingDevice",
	  "ImplicitRegion",
	  "Implies",
	  "Import",
	  "ImportAutoReplacements",
	  "ImportByteArray",
	  "ImportOptions",
	  "ImportString",
	  "ImprovementImportance",
	  "In",
	  "Inactivate",
	  "Inactive",
	  "IncidenceGraph",
	  "IncidenceList",
	  "IncidenceMatrix",
	  "IncludeAromaticBonds",
	  "IncludeConstantBasis",
	  "IncludeDefinitions",
	  "IncludeDirectories",
	  "IncludeFileExtension",
	  "IncludeGeneratorTasks",
	  "IncludeHydrogens",
	  "IncludeInflections",
	  "IncludeMetaInformation",
	  "IncludePods",
	  "IncludeQuantities",
	  "IncludeRelatedTables",
	  "IncludeSingularTerm",
	  "IncludeWindowTimes",
	  "Increment",
	  "IndefiniteMatrixQ",
	  "Indent",
	  "IndentingNewlineSpacings",
	  "IndentMaxFraction",
	  "IndependenceTest",
	  "IndependentEdgeSetQ",
	  "IndependentPhysicalQuantity",
	  "IndependentUnit",
	  "IndependentUnitDimension",
	  "IndependentVertexSetQ",
	  "Indeterminate",
	  "IndeterminateThreshold",
	  "IndexCreationOptions",
	  "Indexed",
	  "IndexEdgeTaggedGraph",
	  "IndexGraph",
	  "IndexTag",
	  "Inequality",
	  "InexactNumberQ",
	  "InexactNumbers",
	  "InfiniteFuture",
	  "InfiniteLine",
	  "InfinitePast",
	  "InfinitePlane",
	  "Infinity",
	  "Infix",
	  "InflationAdjust",
	  "InflationMethod",
	  "Information",
	  "InformationData",
	  "InformationDataGrid",
	  "Inherited",
	  "InheritScope",
	  "InhomogeneousPoissonProcess",
	  "InitialEvaluationHistory",
	  "Initialization",
	  "InitializationCell",
	  "InitializationCellEvaluation",
	  "InitializationCellWarning",
	  "InitializationObjects",
	  "InitializationValue",
	  "Initialize",
	  "InitialSeeding",
	  "InlineCounterAssignments",
	  "InlineCounterIncrements",
	  "InlineRules",
	  "Inner",
	  "InnerPolygon",
	  "InnerPolyhedron",
	  "Inpaint",
	  "Input",
	  "InputAliases",
	  "InputAssumptions",
	  "InputAutoReplacements",
	  "InputField",
	  "InputFieldBox",
	  "InputFieldBoxOptions",
	  "InputForm",
	  "InputGrouping",
	  "InputNamePacket",
	  "InputNotebook",
	  "InputPacket",
	  "InputSettings",
	  "InputStream",
	  "InputString",
	  "InputStringPacket",
	  "InputToBoxFormPacket",
	  "Insert",
	  "InsertionFunction",
	  "InsertionPointObject",
	  "InsertLinebreaks",
	  "InsertResults",
	  "Inset",
	  "Inset3DBox",
	  "Inset3DBoxOptions",
	  "InsetBox",
	  "InsetBoxOptions",
	  "Insphere",
	  "Install",
	  "InstallService",
	  "InstanceNormalizationLayer",
	  "InString",
	  "Integer",
	  "IntegerDigits",
	  "IntegerExponent",
	  "IntegerLength",
	  "IntegerName",
	  "IntegerPart",
	  "IntegerPartitions",
	  "IntegerQ",
	  "IntegerReverse",
	  "Integers",
	  "IntegerString",
	  "Integral",
	  "Integrate",
	  "Interactive",
	  "InteractiveTradingChart",
	  "Interlaced",
	  "Interleaving",
	  "InternallyBalancedDecomposition",
	  "InterpolatingFunction",
	  "InterpolatingPolynomial",
	  "Interpolation",
	  "InterpolationOrder",
	  "InterpolationPoints",
	  "InterpolationPrecision",
	  "Interpretation",
	  "InterpretationBox",
	  "InterpretationBoxOptions",
	  "InterpretationFunction",
	  "Interpreter",
	  "InterpretTemplate",
	  "InterquartileRange",
	  "Interrupt",
	  "InterruptSettings",
	  "IntersectedEntityClass",
	  "IntersectingQ",
	  "Intersection",
	  "Interval",
	  "IntervalIntersection",
	  "IntervalMarkers",
	  "IntervalMarkersStyle",
	  "IntervalMemberQ",
	  "IntervalSlider",
	  "IntervalUnion",
	  "Into",
	  "Inverse",
	  "InverseBetaRegularized",
	  "InverseCDF",
	  "InverseChiSquareDistribution",
	  "InverseContinuousWaveletTransform",
	  "InverseDistanceTransform",
	  "InverseEllipticNomeQ",
	  "InverseErf",
	  "InverseErfc",
	  "InverseFourier",
	  "InverseFourierCosTransform",
	  "InverseFourierSequenceTransform",
	  "InverseFourierSinTransform",
	  "InverseFourierTransform",
	  "InverseFunction",
	  "InverseFunctions",
	  "InverseGammaDistribution",
	  "InverseGammaRegularized",
	  "InverseGaussianDistribution",
	  "InverseGudermannian",
	  "InverseHankelTransform",
	  "InverseHaversine",
	  "InverseImagePyramid",
	  "InverseJacobiCD",
	  "InverseJacobiCN",
	  "InverseJacobiCS",
	  "InverseJacobiDC",
	  "InverseJacobiDN",
	  "InverseJacobiDS",
	  "InverseJacobiNC",
	  "InverseJacobiND",
	  "InverseJacobiNS",
	  "InverseJacobiSC",
	  "InverseJacobiSD",
	  "InverseJacobiSN",
	  "InverseLaplaceTransform",
	  "InverseMellinTransform",
	  "InversePermutation",
	  "InverseRadon",
	  "InverseRadonTransform",
	  "InverseSeries",
	  "InverseShortTimeFourier",
	  "InverseSpectrogram",
	  "InverseSurvivalFunction",
	  "InverseTransformedRegion",
	  "InverseWaveletTransform",
	  "InverseWeierstrassP",
	  "InverseWishartMatrixDistribution",
	  "InverseZTransform",
	  "Invisible",
	  "InvisibleApplication",
	  "InvisibleTimes",
	  "IPAddress",
	  "IrreduciblePolynomialQ",
	  "IslandData",
	  "IsolatingInterval",
	  "IsomorphicGraphQ",
	  "IsotopeData",
	  "Italic",
	  "Item",
	  "ItemAspectRatio",
	  "ItemBox",
	  "ItemBoxOptions",
	  "ItemDisplayFunction",
	  "ItemSize",
	  "ItemStyle",
	  "ItoProcess",
	  "JaccardDissimilarity",
	  "JacobiAmplitude",
	  "Jacobian",
	  "JacobiCD",
	  "JacobiCN",
	  "JacobiCS",
	  "JacobiDC",
	  "JacobiDN",
	  "JacobiDS",
	  "JacobiNC",
	  "JacobiND",
	  "JacobiNS",
	  "JacobiP",
	  "JacobiSC",
	  "JacobiSD",
	  "JacobiSN",
	  "JacobiSymbol",
	  "JacobiZeta",
	  "JankoGroupJ1",
	  "JankoGroupJ2",
	  "JankoGroupJ3",
	  "JankoGroupJ4",
	  "JarqueBeraALMTest",
	  "JohnsonDistribution",
	  "Join",
	  "JoinAcross",
	  "Joined",
	  "JoinedCurve",
	  "JoinedCurveBox",
	  "JoinedCurveBoxOptions",
	  "JoinForm",
	  "JordanDecomposition",
	  "JordanModelDecomposition",
	  "JulianDate",
	  "JuliaSetBoettcher",
	  "JuliaSetIterationCount",
	  "JuliaSetPlot",
	  "JuliaSetPoints",
	  "K",
	  "KagiChart",
	  "KaiserBesselWindow",
	  "KaiserWindow",
	  "KalmanEstimator",
	  "KalmanFilter",
	  "KarhunenLoeveDecomposition",
	  "KaryTree",
	  "KatzCentrality",
	  "KCoreComponents",
	  "KDistribution",
	  "KEdgeConnectedComponents",
	  "KEdgeConnectedGraphQ",
	  "KeepExistingVersion",
	  "KelvinBei",
	  "KelvinBer",
	  "KelvinKei",
	  "KelvinKer",
	  "KendallTau",
	  "KendallTauTest",
	  "KernelExecute",
	  "KernelFunction",
	  "KernelMixtureDistribution",
	  "KernelObject",
	  "Kernels",
	  "Ket",
	  "Key",
	  "KeyCollisionFunction",
	  "KeyComplement",
	  "KeyDrop",
	  "KeyDropFrom",
	  "KeyExistsQ",
	  "KeyFreeQ",
	  "KeyIntersection",
	  "KeyMap",
	  "KeyMemberQ",
	  "KeypointStrength",
	  "Keys",
	  "KeySelect",
	  "KeySort",
	  "KeySortBy",
	  "KeyTake",
	  "KeyUnion",
	  "KeyValueMap",
	  "KeyValuePattern",
	  "Khinchin",
	  "KillProcess",
	  "KirchhoffGraph",
	  "KirchhoffMatrix",
	  "KleinInvariantJ",
	  "KnapsackSolve",
	  "KnightTourGraph",
	  "KnotData",
	  "KnownUnitQ",
	  "KochCurve",
	  "KolmogorovSmirnovTest",
	  "KroneckerDelta",
	  "KroneckerModelDecomposition",
	  "KroneckerProduct",
	  "KroneckerSymbol",
	  "KuiperTest",
	  "KumaraswamyDistribution",
	  "Kurtosis",
	  "KuwaharaFilter",
	  "KVertexConnectedComponents",
	  "KVertexConnectedGraphQ",
	  "LABColor",
	  "Label",
	  "Labeled",
	  "LabeledSlider",
	  "LabelingFunction",
	  "LabelingSize",
	  "LabelStyle",
	  "LabelVisibility",
	  "LaguerreL",
	  "LakeData",
	  "LambdaComponents",
	  "LambertW",
	  "LaminaData",
	  "LanczosWindow",
	  "LandauDistribution",
	  "Language",
	  "LanguageCategory",
	  "LanguageData",
	  "LanguageIdentify",
	  "LanguageOptions",
	  "LaplaceDistribution",
	  "LaplaceTransform",
	  "Laplacian",
	  "LaplacianFilter",
	  "LaplacianGaussianFilter",
	  "Large",
	  "Larger",
	  "Last",
	  "Latitude",
	  "LatitudeLongitude",
	  "LatticeData",
	  "LatticeReduce",
	  "Launch",
	  "LaunchKernels",
	  "LayeredGraphPlot",
	  "LayerSizeFunction",
	  "LayoutInformation",
	  "LCHColor",
	  "LCM",
	  "LeaderSize",
	  "LeafCount",
	  "LeapYearQ",
	  "LearnDistribution",
	  "LearnedDistribution",
	  "LearningRate",
	  "LearningRateMultipliers",
	  "LeastSquares",
	  "LeastSquaresFilterKernel",
	  "Left",
	  "LeftArrow",
	  "LeftArrowBar",
	  "LeftArrowRightArrow",
	  "LeftDownTeeVector",
	  "LeftDownVector",
	  "LeftDownVectorBar",
	  "LeftRightArrow",
	  "LeftRightVector",
	  "LeftTee",
	  "LeftTeeArrow",
	  "LeftTeeVector",
	  "LeftTriangle",
	  "LeftTriangleBar",
	  "LeftTriangleEqual",
	  "LeftUpDownVector",
	  "LeftUpTeeVector",
	  "LeftUpVector",
	  "LeftUpVectorBar",
	  "LeftVector",
	  "LeftVectorBar",
	  "LegendAppearance",
	  "Legended",
	  "LegendFunction",
	  "LegendLabel",
	  "LegendLayout",
	  "LegendMargins",
	  "LegendMarkers",
	  "LegendMarkerSize",
	  "LegendreP",
	  "LegendreQ",
	  "LegendreType",
	  "Length",
	  "LengthWhile",
	  "LerchPhi",
	  "Less",
	  "LessEqual",
	  "LessEqualGreater",
	  "LessEqualThan",
	  "LessFullEqual",
	  "LessGreater",
	  "LessLess",
	  "LessSlantEqual",
	  "LessThan",
	  "LessTilde",
	  "LetterCharacter",
	  "LetterCounts",
	  "LetterNumber",
	  "LetterQ",
	  "Level",
	  "LeveneTest",
	  "LeviCivitaTensor",
	  "LevyDistribution",
	  "Lexicographic",
	  "LibraryDataType",
	  "LibraryFunction",
	  "LibraryFunctionError",
	  "LibraryFunctionInformation",
	  "LibraryFunctionLoad",
	  "LibraryFunctionUnload",
	  "LibraryLoad",
	  "LibraryUnload",
	  "LicenseID",
	  "LiftingFilterData",
	  "LiftingWaveletTransform",
	  "LightBlue",
	  "LightBrown",
	  "LightCyan",
	  "Lighter",
	  "LightGray",
	  "LightGreen",
	  "Lighting",
	  "LightingAngle",
	  "LightMagenta",
	  "LightOrange",
	  "LightPink",
	  "LightPurple",
	  "LightRed",
	  "LightSources",
	  "LightYellow",
	  "Likelihood",
	  "Limit",
	  "LimitsPositioning",
	  "LimitsPositioningTokens",
	  "LindleyDistribution",
	  "Line",
	  "Line3DBox",
	  "Line3DBoxOptions",
	  "LinearFilter",
	  "LinearFractionalOptimization",
	  "LinearFractionalTransform",
	  "LinearGradientImage",
	  "LinearizingTransformationData",
	  "LinearLayer",
	  "LinearModelFit",
	  "LinearOffsetFunction",
	  "LinearOptimization",
	  "LinearProgramming",
	  "LinearRecurrence",
	  "LinearSolve",
	  "LinearSolveFunction",
	  "LineBox",
	  "LineBoxOptions",
	  "LineBreak",
	  "LinebreakAdjustments",
	  "LineBreakChart",
	  "LinebreakSemicolonWeighting",
	  "LineBreakWithin",
	  "LineColor",
	  "LineGraph",
	  "LineIndent",
	  "LineIndentMaxFraction",
	  "LineIntegralConvolutionPlot",
	  "LineIntegralConvolutionScale",
	  "LineLegend",
	  "LineOpacity",
	  "LineSpacing",
	  "LineWrapParts",
	  "LinkActivate",
	  "LinkClose",
	  "LinkConnect",
	  "LinkConnectedQ",
	  "LinkCreate",
	  "LinkError",
	  "LinkFlush",
	  "LinkFunction",
	  "LinkHost",
	  "LinkInterrupt",
	  "LinkLaunch",
	  "LinkMode",
	  "LinkObject",
	  "LinkOpen",
	  "LinkOptions",
	  "LinkPatterns",
	  "LinkProtocol",
	  "LinkRankCentrality",
	  "LinkRead",
	  "LinkReadHeld",
	  "LinkReadyQ",
	  "Links",
	  "LinkService",
	  "LinkWrite",
	  "LinkWriteHeld",
	  "LiouvilleLambda",
	  "List",
	  "Listable",
	  "ListAnimate",
	  "ListContourPlot",
	  "ListContourPlot3D",
	  "ListConvolve",
	  "ListCorrelate",
	  "ListCurvePathPlot",
	  "ListDeconvolve",
	  "ListDensityPlot",
	  "ListDensityPlot3D",
	  "Listen",
	  "ListFormat",
	  "ListFourierSequenceTransform",
	  "ListInterpolation",
	  "ListLineIntegralConvolutionPlot",
	  "ListLinePlot",
	  "ListLogLinearPlot",
	  "ListLogLogPlot",
	  "ListLogPlot",
	  "ListPicker",
	  "ListPickerBox",
	  "ListPickerBoxBackground",
	  "ListPickerBoxOptions",
	  "ListPlay",
	  "ListPlot",
	  "ListPlot3D",
	  "ListPointPlot3D",
	  "ListPolarPlot",
	  "ListQ",
	  "ListSliceContourPlot3D",
	  "ListSliceDensityPlot3D",
	  "ListSliceVectorPlot3D",
	  "ListStepPlot",
	  "ListStreamDensityPlot",
	  "ListStreamPlot",
	  "ListSurfacePlot3D",
	  "ListVectorDensityPlot",
	  "ListVectorPlot",
	  "ListVectorPlot3D",
	  "ListZTransform",
	  "Literal",
	  "LiteralSearch",
	  "LocalAdaptiveBinarize",
	  "LocalCache",
	  "LocalClusteringCoefficient",
	  "LocalizeDefinitions",
	  "LocalizeVariables",
	  "LocalObject",
	  "LocalObjects",
	  "LocalResponseNormalizationLayer",
	  "LocalSubmit",
	  "LocalSymbol",
	  "LocalTime",
	  "LocalTimeZone",
	  "LocationEquivalenceTest",
	  "LocationTest",
	  "Locator",
	  "LocatorAutoCreate",
	  "LocatorBox",
	  "LocatorBoxOptions",
	  "LocatorCentering",
	  "LocatorPane",
	  "LocatorPaneBox",
	  "LocatorPaneBoxOptions",
	  "LocatorRegion",
	  "Locked",
	  "Log",
	  "Log10",
	  "Log2",
	  "LogBarnesG",
	  "LogGamma",
	  "LogGammaDistribution",
	  "LogicalExpand",
	  "LogIntegral",
	  "LogisticDistribution",
	  "LogisticSigmoid",
	  "LogitModelFit",
	  "LogLikelihood",
	  "LogLinearPlot",
	  "LogLogisticDistribution",
	  "LogLogPlot",
	  "LogMultinormalDistribution",
	  "LogNormalDistribution",
	  "LogPlot",
	  "LogRankTest",
	  "LogSeriesDistribution",
	  "LongEqual",
	  "Longest",
	  "LongestCommonSequence",
	  "LongestCommonSequencePositions",
	  "LongestCommonSubsequence",
	  "LongestCommonSubsequencePositions",
	  "LongestMatch",
	  "LongestOrderedSequence",
	  "LongForm",
	  "Longitude",
	  "LongLeftArrow",
	  "LongLeftRightArrow",
	  "LongRightArrow",
	  "LongShortTermMemoryLayer",
	  "Lookup",
	  "Loopback",
	  "LoopFreeGraphQ",
	  "Looping",
	  "LossFunction",
	  "LowerCaseQ",
	  "LowerLeftArrow",
	  "LowerRightArrow",
	  "LowerTriangularize",
	  "LowerTriangularMatrixQ",
	  "LowpassFilter",
	  "LQEstimatorGains",
	  "LQGRegulator",
	  "LQOutputRegulatorGains",
	  "LQRegulatorGains",
	  "LUBackSubstitution",
	  "LucasL",
	  "LuccioSamiComponents",
	  "LUDecomposition",
	  "LunarEclipse",
	  "LUVColor",
	  "LyapunovSolve",
	  "LyonsGroupLy",
	  "MachineID",
	  "MachineName",
	  "MachineNumberQ",
	  "MachinePrecision",
	  "MacintoshSystemPageSetup",
	  "Magenta",
	  "Magnification",
	  "Magnify",
	  "MailAddressValidation",
	  "MailExecute",
	  "MailFolder",
	  "MailItem",
	  "MailReceiverFunction",
	  "MailResponseFunction",
	  "MailSearch",
	  "MailServerConnect",
	  "MailServerConnection",
	  "MailSettings",
	  "MainSolve",
	  "MaintainDynamicCaches",
	  "Majority",
	  "MakeBoxes",
	  "MakeExpression",
	  "MakeRules",
	  "ManagedLibraryExpressionID",
	  "ManagedLibraryExpressionQ",
	  "MandelbrotSetBoettcher",
	  "MandelbrotSetDistance",
	  "MandelbrotSetIterationCount",
	  "MandelbrotSetMemberQ",
	  "MandelbrotSetPlot",
	  "MangoldtLambda",
	  "ManhattanDistance",
	  "Manipulate",
	  "Manipulator",
	  "MannedSpaceMissionData",
	  "MannWhitneyTest",
	  "MantissaExponent",
	  "Manual",
	  "Map",
	  "MapAll",
	  "MapAt",
	  "MapIndexed",
	  "MAProcess",
	  "MapThread",
	  "MarchenkoPasturDistribution",
	  "MarcumQ",
	  "MardiaCombinedTest",
	  "MardiaKurtosisTest",
	  "MardiaSkewnessTest",
	  "MarginalDistribution",
	  "MarkovProcessProperties",
	  "Masking",
	  "MatchingDissimilarity",
	  "MatchLocalNameQ",
	  "MatchLocalNames",
	  "MatchQ",
	  "Material",
	  "MathematicalFunctionData",
	  "MathematicaNotation",
	  "MathieuC",
	  "MathieuCharacteristicA",
	  "MathieuCharacteristicB",
	  "MathieuCharacteristicExponent",
	  "MathieuCPrime",
	  "MathieuGroupM11",
	  "MathieuGroupM12",
	  "MathieuGroupM22",
	  "MathieuGroupM23",
	  "MathieuGroupM24",
	  "MathieuS",
	  "MathieuSPrime",
	  "MathMLForm",
	  "MathMLText",
	  "Matrices",
	  "MatrixExp",
	  "MatrixForm",
	  "MatrixFunction",
	  "MatrixLog",
	  "MatrixNormalDistribution",
	  "MatrixPlot",
	  "MatrixPower",
	  "MatrixPropertyDistribution",
	  "MatrixQ",
	  "MatrixRank",
	  "MatrixTDistribution",
	  "Max",
	  "MaxBend",
	  "MaxCellMeasure",
	  "MaxColorDistance",
	  "MaxDate",
	  "MaxDetect",
	  "MaxDuration",
	  "MaxExtraBandwidths",
	  "MaxExtraConditions",
	  "MaxFeatureDisplacement",
	  "MaxFeatures",
	  "MaxFilter",
	  "MaximalBy",
	  "Maximize",
	  "MaxItems",
	  "MaxIterations",
	  "MaxLimit",
	  "MaxMemoryUsed",
	  "MaxMixtureKernels",
	  "MaxOverlapFraction",
	  "MaxPlotPoints",
	  "MaxPoints",
	  "MaxRecursion",
	  "MaxStableDistribution",
	  "MaxStepFraction",
	  "MaxSteps",
	  "MaxStepSize",
	  "MaxTrainingRounds",
	  "MaxValue",
	  "MaxwellDistribution",
	  "MaxWordGap",
	  "McLaughlinGroupMcL",
	  "Mean",
	  "MeanAbsoluteLossLayer",
	  "MeanAround",
	  "MeanClusteringCoefficient",
	  "MeanDegreeConnectivity",
	  "MeanDeviation",
	  "MeanFilter",
	  "MeanGraphDistance",
	  "MeanNeighborDegree",
	  "MeanShift",
	  "MeanShiftFilter",
	  "MeanSquaredLossLayer",
	  "Median",
	  "MedianDeviation",
	  "MedianFilter",
	  "MedicalTestData",
	  "Medium",
	  "MeijerG",
	  "MeijerGReduce",
	  "MeixnerDistribution",
	  "MellinConvolve",
	  "MellinTransform",
	  "MemberQ",
	  "MemoryAvailable",
	  "MemoryConstrained",
	  "MemoryConstraint",
	  "MemoryInUse",
	  "MengerMesh",
	  "Menu",
	  "MenuAppearance",
	  "MenuCommandKey",
	  "MenuEvaluator",
	  "MenuItem",
	  "MenuList",
	  "MenuPacket",
	  "MenuSortingValue",
	  "MenuStyle",
	  "MenuView",
	  "Merge",
	  "MergeDifferences",
	  "MergingFunction",
	  "MersennePrimeExponent",
	  "MersennePrimeExponentQ",
	  "Mesh",
	  "MeshCellCentroid",
	  "MeshCellCount",
	  "MeshCellHighlight",
	  "MeshCellIndex",
	  "MeshCellLabel",
	  "MeshCellMarker",
	  "MeshCellMeasure",
	  "MeshCellQuality",
	  "MeshCells",
	  "MeshCellShapeFunction",
	  "MeshCellStyle",
	  "MeshConnectivityGraph",
	  "MeshCoordinates",
	  "MeshFunctions",
	  "MeshPrimitives",
	  "MeshQualityGoal",
	  "MeshRange",
	  "MeshRefinementFunction",
	  "MeshRegion",
	  "MeshRegionQ",
	  "MeshShading",
	  "MeshStyle",
	  "Message",
	  "MessageDialog",
	  "MessageList",
	  "MessageName",
	  "MessageObject",
	  "MessageOptions",
	  "MessagePacket",
	  "Messages",
	  "MessagesNotebook",
	  "MetaCharacters",
	  "MetaInformation",
	  "MeteorShowerData",
	  "Method",
	  "MethodOptions",
	  "MexicanHatWavelet",
	  "MeyerWavelet",
	  "Midpoint",
	  "Min",
	  "MinColorDistance",
	  "MinDate",
	  "MinDetect",
	  "MineralData",
	  "MinFilter",
	  "MinimalBy",
	  "MinimalPolynomial",
	  "MinimalStateSpaceModel",
	  "Minimize",
	  "MinimumTimeIncrement",
	  "MinIntervalSize",
	  "MinkowskiQuestionMark",
	  "MinLimit",
	  "MinMax",
	  "MinorPlanetData",
	  "Minors",
	  "MinRecursion",
	  "MinSize",
	  "MinStableDistribution",
	  "Minus",
	  "MinusPlus",
	  "MinValue",
	  "Missing",
	  "MissingBehavior",
	  "MissingDataMethod",
	  "MissingDataRules",
	  "MissingQ",
	  "MissingString",
	  "MissingStyle",
	  "MissingValuePattern",
	  "MittagLefflerE",
	  "MixedFractionParts",
	  "MixedGraphQ",
	  "MixedMagnitude",
	  "MixedRadix",
	  "MixedRadixQuantity",
	  "MixedUnit",
	  "MixtureDistribution",
	  "Mod",
	  "Modal",
	  "Mode",
	  "Modular",
	  "ModularInverse",
	  "ModularLambda",
	  "Module",
	  "Modulus",
	  "MoebiusMu",
	  "Molecule",
	  "MoleculeContainsQ",
	  "MoleculeEquivalentQ",
	  "MoleculeGraph",
	  "MoleculeModify",
	  "MoleculePattern",
	  "MoleculePlot",
	  "MoleculePlot3D",
	  "MoleculeProperty",
	  "MoleculeQ",
	  "MoleculeRecognize",
	  "MoleculeValue",
	  "Moment",
	  "Momentary",
	  "MomentConvert",
	  "MomentEvaluate",
	  "MomentGeneratingFunction",
	  "MomentOfInertia",
	  "Monday",
	  "Monitor",
	  "MonomialList",
	  "MonomialOrder",
	  "MonsterGroupM",
	  "MoonPhase",
	  "MoonPosition",
	  "MorletWavelet",
	  "MorphologicalBinarize",
	  "MorphologicalBranchPoints",
	  "MorphologicalComponents",
	  "MorphologicalEulerNumber",
	  "MorphologicalGraph",
	  "MorphologicalPerimeter",
	  "MorphologicalTransform",
	  "MortalityData",
	  "Most",
	  "MountainData",
	  "MouseAnnotation",
	  "MouseAppearance",
	  "MouseAppearanceTag",
	  "MouseButtons",
	  "Mouseover",
	  "MousePointerNote",
	  "MousePosition",
	  "MovieData",
	  "MovingAverage",
	  "MovingMap",
	  "MovingMedian",
	  "MoyalDistribution",
	  "Multicolumn",
	  "MultiedgeStyle",
	  "MultigraphQ",
	  "MultilaunchWarning",
	  "MultiLetterItalics",
	  "MultiLetterStyle",
	  "MultilineFunction",
	  "Multinomial",
	  "MultinomialDistribution",
	  "MultinormalDistribution",
	  "MultiplicativeOrder",
	  "Multiplicity",
	  "MultiplySides",
	  "Multiselection",
	  "MultivariateHypergeometricDistribution",
	  "MultivariatePoissonDistribution",
	  "MultivariateTDistribution",
	  "N",
	  "NakagamiDistribution",
	  "NameQ",
	  "Names",
	  "NamespaceBox",
	  "NamespaceBoxOptions",
	  "Nand",
	  "NArgMax",
	  "NArgMin",
	  "NBernoulliB",
	  "NBodySimulation",
	  "NBodySimulationData",
	  "NCache",
	  "NDEigensystem",
	  "NDEigenvalues",
	  "NDSolve",
	  "NDSolveValue",
	  "Nearest",
	  "NearestFunction",
	  "NearestMeshCells",
	  "NearestNeighborGraph",
	  "NearestTo",
	  "NebulaData",
	  "NeedCurrentFrontEndPackagePacket",
	  "NeedCurrentFrontEndSymbolsPacket",
	  "NeedlemanWunschSimilarity",
	  "Needs",
	  "Negative",
	  "NegativeBinomialDistribution",
	  "NegativeDefiniteMatrixQ",
	  "NegativeIntegers",
	  "NegativeMultinomialDistribution",
	  "NegativeRationals",
	  "NegativeReals",
	  "NegativeSemidefiniteMatrixQ",
	  "NeighborhoodData",
	  "NeighborhoodGraph",
	  "Nest",
	  "NestedGreaterGreater",
	  "NestedLessLess",
	  "NestedScriptRules",
	  "NestGraph",
	  "NestList",
	  "NestWhile",
	  "NestWhileList",
	  "NetAppend",
	  "NetBidirectionalOperator",
	  "NetChain",
	  "NetDecoder",
	  "NetDelete",
	  "NetDrop",
	  "NetEncoder",
	  "NetEvaluationMode",
	  "NetExtract",
	  "NetFlatten",
	  "NetFoldOperator",
	  "NetGANOperator",
	  "NetGraph",
	  "NetInformation",
	  "NetInitialize",
	  "NetInsert",
	  "NetInsertSharedArrays",
	  "NetJoin",
	  "NetMapOperator",
	  "NetMapThreadOperator",
	  "NetMeasurements",
	  "NetModel",
	  "NetNestOperator",
	  "NetPairEmbeddingOperator",
	  "NetPort",
	  "NetPortGradient",
	  "NetPrepend",
	  "NetRename",
	  "NetReplace",
	  "NetReplacePart",
	  "NetSharedArray",
	  "NetStateObject",
	  "NetTake",
	  "NetTrain",
	  "NetTrainResultsObject",
	  "NetworkPacketCapture",
	  "NetworkPacketRecording",
	  "NetworkPacketRecordingDuring",
	  "NetworkPacketTrace",
	  "NeumannValue",
	  "NevilleThetaC",
	  "NevilleThetaD",
	  "NevilleThetaN",
	  "NevilleThetaS",
	  "NewPrimitiveStyle",
	  "NExpectation",
	  "Next",
	  "NextCell",
	  "NextDate",
	  "NextPrime",
	  "NextScheduledTaskTime",
	  "NHoldAll",
	  "NHoldFirst",
	  "NHoldRest",
	  "NicholsGridLines",
	  "NicholsPlot",
	  "NightHemisphere",
	  "NIntegrate",
	  "NMaximize",
	  "NMaxValue",
	  "NMinimize",
	  "NMinValue",
	  "NominalVariables",
	  "NonAssociative",
	  "NoncentralBetaDistribution",
	  "NoncentralChiSquareDistribution",
	  "NoncentralFRatioDistribution",
	  "NoncentralStudentTDistribution",
	  "NonCommutativeMultiply",
	  "NonConstants",
	  "NondimensionalizationTransform",
	  "None",
	  "NoneTrue",
	  "NonlinearModelFit",
	  "NonlinearStateSpaceModel",
	  "NonlocalMeansFilter",
	  "NonNegative",
	  "NonNegativeIntegers",
	  "NonNegativeRationals",
	  "NonNegativeReals",
	  "NonPositive",
	  "NonPositiveIntegers",
	  "NonPositiveRationals",
	  "NonPositiveReals",
	  "Nor",
	  "NorlundB",
	  "Norm",
	  "Normal",
	  "NormalDistribution",
	  "NormalGrouping",
	  "NormalizationLayer",
	  "Normalize",
	  "Normalized",
	  "NormalizedSquaredEuclideanDistance",
	  "NormalMatrixQ",
	  "NormalsFunction",
	  "NormFunction",
	  "Not",
	  "NotCongruent",
	  "NotCupCap",
	  "NotDoubleVerticalBar",
	  "Notebook",
	  "NotebookApply",
	  "NotebookAutoSave",
	  "NotebookClose",
	  "NotebookConvertSettings",
	  "NotebookCreate",
	  "NotebookCreateReturnObject",
	  "NotebookDefault",
	  "NotebookDelete",
	  "NotebookDirectory",
	  "NotebookDynamicExpression",
	  "NotebookEvaluate",
	  "NotebookEventActions",
	  "NotebookFileName",
	  "NotebookFind",
	  "NotebookFindReturnObject",
	  "NotebookGet",
	  "NotebookGetLayoutInformationPacket",
	  "NotebookGetMisspellingsPacket",
	  "NotebookImport",
	  "NotebookInformation",
	  "NotebookInterfaceObject",
	  "NotebookLocate",
	  "NotebookObject",
	  "NotebookOpen",
	  "NotebookOpenReturnObject",
	  "NotebookPath",
	  "NotebookPrint",
	  "NotebookPut",
	  "NotebookPutReturnObject",
	  "NotebookRead",
	  "NotebookResetGeneratedCells",
	  "Notebooks",
	  "NotebookSave",
	  "NotebookSaveAs",
	  "NotebookSelection",
	  "NotebookSetupLayoutInformationPacket",
	  "NotebooksMenu",
	  "NotebookTemplate",
	  "NotebookWrite",
	  "NotElement",
	  "NotEqualTilde",
	  "NotExists",
	  "NotGreater",
	  "NotGreaterEqual",
	  "NotGreaterFullEqual",
	  "NotGreaterGreater",
	  "NotGreaterLess",
	  "NotGreaterSlantEqual",
	  "NotGreaterTilde",
	  "Nothing",
	  "NotHumpDownHump",
	  "NotHumpEqual",
	  "NotificationFunction",
	  "NotLeftTriangle",
	  "NotLeftTriangleBar",
	  "NotLeftTriangleEqual",
	  "NotLess",
	  "NotLessEqual",
	  "NotLessFullEqual",
	  "NotLessGreater",
	  "NotLessLess",
	  "NotLessSlantEqual",
	  "NotLessTilde",
	  "NotNestedGreaterGreater",
	  "NotNestedLessLess",
	  "NotPrecedes",
	  "NotPrecedesEqual",
	  "NotPrecedesSlantEqual",
	  "NotPrecedesTilde",
	  "NotReverseElement",
	  "NotRightTriangle",
	  "NotRightTriangleBar",
	  "NotRightTriangleEqual",
	  "NotSquareSubset",
	  "NotSquareSubsetEqual",
	  "NotSquareSuperset",
	  "NotSquareSupersetEqual",
	  "NotSubset",
	  "NotSubsetEqual",
	  "NotSucceeds",
	  "NotSucceedsEqual",
	  "NotSucceedsSlantEqual",
	  "NotSucceedsTilde",
	  "NotSuperset",
	  "NotSupersetEqual",
	  "NotTilde",
	  "NotTildeEqual",
	  "NotTildeFullEqual",
	  "NotTildeTilde",
	  "NotVerticalBar",
	  "Now",
	  "NoWhitespace",
	  "NProbability",
	  "NProduct",
	  "NProductFactors",
	  "NRoots",
	  "NSolve",
	  "NSum",
	  "NSumTerms",
	  "NuclearExplosionData",
	  "NuclearReactorData",
	  "Null",
	  "NullRecords",
	  "NullSpace",
	  "NullWords",
	  "Number",
	  "NumberCompose",
	  "NumberDecompose",
	  "NumberExpand",
	  "NumberFieldClassNumber",
	  "NumberFieldDiscriminant",
	  "NumberFieldFundamentalUnits",
	  "NumberFieldIntegralBasis",
	  "NumberFieldNormRepresentatives",
	  "NumberFieldRegulator",
	  "NumberFieldRootsOfUnity",
	  "NumberFieldSignature",
	  "NumberForm",
	  "NumberFormat",
	  "NumberLinePlot",
	  "NumberMarks",
	  "NumberMultiplier",
	  "NumberPadding",
	  "NumberPoint",
	  "NumberQ",
	  "NumberSeparator",
	  "NumberSigns",
	  "NumberString",
	  "Numerator",
	  "NumeratorDenominator",
	  "NumericalOrder",
	  "NumericalSort",
	  "NumericArray",
	  "NumericArrayQ",
	  "NumericArrayType",
	  "NumericFunction",
	  "NumericQ",
	  "NuttallWindow",
	  "NValues",
	  "NyquistGridLines",
	  "NyquistPlot",
	  "O",
	  "ObservabilityGramian",
	  "ObservabilityMatrix",
	  "ObservableDecomposition",
	  "ObservableModelQ",
	  "OceanData",
	  "Octahedron",
	  "OddQ",
	  "Off",
	  "Offset",
	  "OLEData",
	  "On",
	  "ONanGroupON",
	  "Once",
	  "OneIdentity",
	  "Opacity",
	  "OpacityFunction",
	  "OpacityFunctionScaling",
	  "Open",
	  "OpenAppend",
	  "Opener",
	  "OpenerBox",
	  "OpenerBoxOptions",
	  "OpenerView",
	  "OpenFunctionInspectorPacket",
	  "Opening",
	  "OpenRead",
	  "OpenSpecialOptions",
	  "OpenTemporary",
	  "OpenWrite",
	  "Operate",
	  "OperatingSystem",
	  "OperatorApplied",
	  "OptimumFlowData",
	  "Optional",
	  "OptionalElement",
	  "OptionInspectorSettings",
	  "OptionQ",
	  "Options",
	  "OptionsPacket",
	  "OptionsPattern",
	  "OptionValue",
	  "OptionValueBox",
	  "OptionValueBoxOptions",
	  "Or",
	  "Orange",
	  "Order",
	  "OrderDistribution",
	  "OrderedQ",
	  "Ordering",
	  "OrderingBy",
	  "OrderingLayer",
	  "Orderless",
	  "OrderlessPatternSequence",
	  "OrnsteinUhlenbeckProcess",
	  "Orthogonalize",
	  "OrthogonalMatrixQ",
	  "Out",
	  "Outer",
	  "OuterPolygon",
	  "OuterPolyhedron",
	  "OutputAutoOverwrite",
	  "OutputControllabilityMatrix",
	  "OutputControllableModelQ",
	  "OutputForm",
	  "OutputFormData",
	  "OutputGrouping",
	  "OutputMathEditExpression",
	  "OutputNamePacket",
	  "OutputResponse",
	  "OutputSizeLimit",
	  "OutputStream",
	  "Over",
	  "OverBar",
	  "OverDot",
	  "Overflow",
	  "OverHat",
	  "Overlaps",
	  "Overlay",
	  "OverlayBox",
	  "OverlayBoxOptions",
	  "Overscript",
	  "OverscriptBox",
	  "OverscriptBoxOptions",
	  "OverTilde",
	  "OverVector",
	  "OverwriteTarget",
	  "OwenT",
	  "OwnValues",
	  "Package",
	  "PackingMethod",
	  "PackPaclet",
	  "PacletDataRebuild",
	  "PacletDirectoryAdd",
	  "PacletDirectoryLoad",
	  "PacletDirectoryRemove",
	  "PacletDirectoryUnload",
	  "PacletDisable",
	  "PacletEnable",
	  "PacletFind",
	  "PacletFindRemote",
	  "PacletInformation",
	  "PacletInstall",
	  "PacletInstallSubmit",
	  "PacletNewerQ",
	  "PacletObject",
	  "PacletObjectQ",
	  "PacletSite",
	  "PacletSiteObject",
	  "PacletSiteRegister",
	  "PacletSites",
	  "PacletSiteUnregister",
	  "PacletSiteUpdate",
	  "PacletUninstall",
	  "PacletUpdate",
	  "PaddedForm",
	  "Padding",
	  "PaddingLayer",
	  "PaddingSize",
	  "PadeApproximant",
	  "PadLeft",
	  "PadRight",
	  "PageBreakAbove",
	  "PageBreakBelow",
	  "PageBreakWithin",
	  "PageFooterLines",
	  "PageFooters",
	  "PageHeaderLines",
	  "PageHeaders",
	  "PageHeight",
	  "PageRankCentrality",
	  "PageTheme",
	  "PageWidth",
	  "Pagination",
	  "PairedBarChart",
	  "PairedHistogram",
	  "PairedSmoothHistogram",
	  "PairedTTest",
	  "PairedZTest",
	  "PaletteNotebook",
	  "PalettePath",
	  "PalindromeQ",
	  "Pane",
	  "PaneBox",
	  "PaneBoxOptions",
	  "Panel",
	  "PanelBox",
	  "PanelBoxOptions",
	  "Paneled",
	  "PaneSelector",
	  "PaneSelectorBox",
	  "PaneSelectorBoxOptions",
	  "PaperWidth",
	  "ParabolicCylinderD",
	  "ParagraphIndent",
	  "ParagraphSpacing",
	  "ParallelArray",
	  "ParallelCombine",
	  "ParallelDo",
	  "Parallelepiped",
	  "ParallelEvaluate",
	  "Parallelization",
	  "Parallelize",
	  "ParallelMap",
	  "ParallelNeeds",
	  "Parallelogram",
	  "ParallelProduct",
	  "ParallelSubmit",
	  "ParallelSum",
	  "ParallelTable",
	  "ParallelTry",
	  "Parameter",
	  "ParameterEstimator",
	  "ParameterMixtureDistribution",
	  "ParameterVariables",
	  "ParametricFunction",
	  "ParametricNDSolve",
	  "ParametricNDSolveValue",
	  "ParametricPlot",
	  "ParametricPlot3D",
	  "ParametricRampLayer",
	  "ParametricRegion",
	  "ParentBox",
	  "ParentCell",
	  "ParentConnect",
	  "ParentDirectory",
	  "ParentForm",
	  "Parenthesize",
	  "ParentList",
	  "ParentNotebook",
	  "ParetoDistribution",
	  "ParetoPickandsDistribution",
	  "ParkData",
	  "Part",
	  "PartBehavior",
	  "PartialCorrelationFunction",
	  "PartialD",
	  "ParticleAcceleratorData",
	  "ParticleData",
	  "Partition",
	  "PartitionGranularity",
	  "PartitionsP",
	  "PartitionsQ",
	  "PartLayer",
	  "PartOfSpeech",
	  "PartProtection",
	  "ParzenWindow",
	  "PascalDistribution",
	  "PassEventsDown",
	  "PassEventsUp",
	  "Paste",
	  "PasteAutoQuoteCharacters",
	  "PasteBoxFormInlineCells",
	  "PasteButton",
	  "Path",
	  "PathGraph",
	  "PathGraphQ",
	  "Pattern",
	  "PatternFilling",
	  "PatternSequence",
	  "PatternTest",
	  "PauliMatrix",
	  "PaulWavelet",
	  "Pause",
	  "PausedTime",
	  "PDF",
	  "PeakDetect",
	  "PeanoCurve",
	  "PearsonChiSquareTest",
	  "PearsonCorrelationTest",
	  "PearsonDistribution",
	  "PercentForm",
	  "PerfectNumber",
	  "PerfectNumberQ",
	  "PerformanceGoal",
	  "Perimeter",
	  "PeriodicBoundaryCondition",
	  "PeriodicInterpolation",
	  "Periodogram",
	  "PeriodogramArray",
	  "Permanent",
	  "Permissions",
	  "PermissionsGroup",
	  "PermissionsGroupMemberQ",
	  "PermissionsGroups",
	  "PermissionsKey",
	  "PermissionsKeys",
	  "PermutationCycles",
	  "PermutationCyclesQ",
	  "PermutationGroup",
	  "PermutationLength",
	  "PermutationList",
	  "PermutationListQ",
	  "PermutationMax",
	  "PermutationMin",
	  "PermutationOrder",
	  "PermutationPower",
	  "PermutationProduct",
	  "PermutationReplace",
	  "Permutations",
	  "PermutationSupport",
	  "Permute",
	  "PeronaMalikFilter",
	  "Perpendicular",
	  "PerpendicularBisector",
	  "PersistenceLocation",
	  "PersistenceTime",
	  "PersistentObject",
	  "PersistentObjects",
	  "PersistentValue",
	  "PersonData",
	  "PERTDistribution",
	  "PetersenGraph",
	  "PhaseMargins",
	  "PhaseRange",
	  "PhysicalSystemData",
	  "Pi",
	  "Pick",
	  "PIDData",
	  "PIDDerivativeFilter",
	  "PIDFeedforward",
	  "PIDTune",
	  "Piecewise",
	  "PiecewiseExpand",
	  "PieChart",
	  "PieChart3D",
	  "PillaiTrace",
	  "PillaiTraceTest",
	  "PingTime",
	  "Pink",
	  "PitchRecognize",
	  "Pivoting",
	  "PixelConstrained",
	  "PixelValue",
	  "PixelValuePositions",
	  "Placed",
	  "Placeholder",
	  "PlaceholderReplace",
	  "Plain",
	  "PlanarAngle",
	  "PlanarGraph",
	  "PlanarGraphQ",
	  "PlanckRadiationLaw",
	  "PlaneCurveData",
	  "PlanetaryMoonData",
	  "PlanetData",
	  "PlantData",
	  "Play",
	  "PlayRange",
	  "Plot",
	  "Plot3D",
	  "Plot3Matrix",
	  "PlotDivision",
	  "PlotJoined",
	  "PlotLabel",
	  "PlotLabels",
	  "PlotLayout",
	  "PlotLegends",
	  "PlotMarkers",
	  "PlotPoints",
	  "PlotRange",
	  "PlotRangeClipping",
	  "PlotRangeClipPlanesStyle",
	  "PlotRangePadding",
	  "PlotRegion",
	  "PlotStyle",
	  "PlotTheme",
	  "Pluralize",
	  "Plus",
	  "PlusMinus",
	  "Pochhammer",
	  "PodStates",
	  "PodWidth",
	  "Point",
	  "Point3DBox",
	  "Point3DBoxOptions",
	  "PointBox",
	  "PointBoxOptions",
	  "PointFigureChart",
	  "PointLegend",
	  "PointSize",
	  "PoissonConsulDistribution",
	  "PoissonDistribution",
	  "PoissonProcess",
	  "PoissonWindow",
	  "PolarAxes",
	  "PolarAxesOrigin",
	  "PolarGridLines",
	  "PolarPlot",
	  "PolarTicks",
	  "PoleZeroMarkers",
	  "PolyaAeppliDistribution",
	  "PolyGamma",
	  "Polygon",
	  "Polygon3DBox",
	  "Polygon3DBoxOptions",
	  "PolygonalNumber",
	  "PolygonAngle",
	  "PolygonBox",
	  "PolygonBoxOptions",
	  "PolygonCoordinates",
	  "PolygonDecomposition",
	  "PolygonHoleScale",
	  "PolygonIntersections",
	  "PolygonScale",
	  "Polyhedron",
	  "PolyhedronAngle",
	  "PolyhedronCoordinates",
	  "PolyhedronData",
	  "PolyhedronDecomposition",
	  "PolyhedronGenus",
	  "PolyLog",
	  "PolynomialExtendedGCD",
	  "PolynomialForm",
	  "PolynomialGCD",
	  "PolynomialLCM",
	  "PolynomialMod",
	  "PolynomialQ",
	  "PolynomialQuotient",
	  "PolynomialQuotientRemainder",
	  "PolynomialReduce",
	  "PolynomialRemainder",
	  "Polynomials",
	  "PoolingLayer",
	  "PopupMenu",
	  "PopupMenuBox",
	  "PopupMenuBoxOptions",
	  "PopupView",
	  "PopupWindow",
	  "Position",
	  "PositionIndex",
	  "Positive",
	  "PositiveDefiniteMatrixQ",
	  "PositiveIntegers",
	  "PositiveRationals",
	  "PositiveReals",
	  "PositiveSemidefiniteMatrixQ",
	  "PossibleZeroQ",
	  "Postfix",
	  "PostScript",
	  "Power",
	  "PowerDistribution",
	  "PowerExpand",
	  "PowerMod",
	  "PowerModList",
	  "PowerRange",
	  "PowerSpectralDensity",
	  "PowersRepresentations",
	  "PowerSymmetricPolynomial",
	  "Precedence",
	  "PrecedenceForm",
	  "Precedes",
	  "PrecedesEqual",
	  "PrecedesSlantEqual",
	  "PrecedesTilde",
	  "Precision",
	  "PrecisionGoal",
	  "PreDecrement",
	  "Predict",
	  "PredictionRoot",
	  "PredictorFunction",
	  "PredictorInformation",
	  "PredictorMeasurements",
	  "PredictorMeasurementsObject",
	  "PreemptProtect",
	  "PreferencesPath",
	  "Prefix",
	  "PreIncrement",
	  "Prepend",
	  "PrependLayer",
	  "PrependTo",
	  "PreprocessingRules",
	  "PreserveColor",
	  "PreserveImageOptions",
	  "Previous",
	  "PreviousCell",
	  "PreviousDate",
	  "PriceGraphDistribution",
	  "PrimaryPlaceholder",
	  "Prime",
	  "PrimeNu",
	  "PrimeOmega",
	  "PrimePi",
	  "PrimePowerQ",
	  "PrimeQ",
	  "Primes",
	  "PrimeZetaP",
	  "PrimitivePolynomialQ",
	  "PrimitiveRoot",
	  "PrimitiveRootList",
	  "PrincipalComponents",
	  "PrincipalValue",
	  "Print",
	  "PrintableASCIIQ",
	  "PrintAction",
	  "PrintForm",
	  "PrintingCopies",
	  "PrintingOptions",
	  "PrintingPageRange",
	  "PrintingStartingPageNumber",
	  "PrintingStyleEnvironment",
	  "Printout3D",
	  "Printout3DPreviewer",
	  "PrintPrecision",
	  "PrintTemporary",
	  "Prism",
	  "PrismBox",
	  "PrismBoxOptions",
	  "PrivateCellOptions",
	  "PrivateEvaluationOptions",
	  "PrivateFontOptions",
	  "PrivateFrontEndOptions",
	  "PrivateKey",
	  "PrivateNotebookOptions",
	  "PrivatePaths",
	  "Probability",
	  "ProbabilityDistribution",
	  "ProbabilityPlot",
	  "ProbabilityPr",
	  "ProbabilityScalePlot",
	  "ProbitModelFit",
	  "ProcessConnection",
	  "ProcessDirectory",
	  "ProcessEnvironment",
	  "Processes",
	  "ProcessEstimator",
	  "ProcessInformation",
	  "ProcessObject",
	  "ProcessParameterAssumptions",
	  "ProcessParameterQ",
	  "ProcessStateDomain",
	  "ProcessStatus",
	  "ProcessTimeDomain",
	  "Product",
	  "ProductDistribution",
	  "ProductLog",
	  "ProgressIndicator",
	  "ProgressIndicatorBox",
	  "ProgressIndicatorBoxOptions",
	  "Projection",
	  "Prolog",
	  "PromptForm",
	  "ProofObject",
	  "Properties",
	  "Property",
	  "PropertyList",
	  "PropertyValue",
	  "Proportion",
	  "Proportional",
	  "Protect",
	  "Protected",
	  "ProteinData",
	  "Pruning",
	  "PseudoInverse",
	  "PsychrometricPropertyData",
	  "PublicKey",
	  "PublisherID",
	  "PulsarData",
	  "PunctuationCharacter",
	  "Purple",
	  "Put",
	  "PutAppend",
	  "Pyramid",
	  "PyramidBox",
	  "PyramidBoxOptions",
	  "QBinomial",
	  "QFactorial",
	  "QGamma",
	  "QHypergeometricPFQ",
	  "QnDispersion",
	  "QPochhammer",
	  "QPolyGamma",
	  "QRDecomposition",
	  "QuadraticIrrationalQ",
	  "QuadraticOptimization",
	  "Quantile",
	  "QuantilePlot",
	  "Quantity",
	  "QuantityArray",
	  "QuantityDistribution",
	  "QuantityForm",
	  "QuantityMagnitude",
	  "QuantityQ",
	  "QuantityUnit",
	  "QuantityVariable",
	  "QuantityVariableCanonicalUnit",
	  "QuantityVariableDimensions",
	  "QuantityVariableIdentifier",
	  "QuantityVariablePhysicalQuantity",
	  "Quartics",
	  "QuartileDeviation",
	  "Quartiles",
	  "QuartileSkewness",
	  "Query",
	  "QueueingNetworkProcess",
	  "QueueingProcess",
	  "QueueProperties",
	  "Quiet",
	  "Quit",
	  "Quotient",
	  "QuotientRemainder",
	  "RadialGradientImage",
	  "RadialityCentrality",
	  "RadicalBox",
	  "RadicalBoxOptions",
	  "RadioButton",
	  "RadioButtonBar",
	  "RadioButtonBox",
	  "RadioButtonBoxOptions",
	  "Radon",
	  "RadonTransform",
	  "RamanujanTau",
	  "RamanujanTauL",
	  "RamanujanTauTheta",
	  "RamanujanTauZ",
	  "Ramp",
	  "Random",
	  "RandomChoice",
	  "RandomColor",
	  "RandomComplex",
	  "RandomEntity",
	  "RandomFunction",
	  "RandomGeoPosition",
	  "RandomGraph",
	  "RandomImage",
	  "RandomInstance",
	  "RandomInteger",
	  "RandomPermutation",
	  "RandomPoint",
	  "RandomPolygon",
	  "RandomPolyhedron",
	  "RandomPrime",
	  "RandomReal",
	  "RandomSample",
	  "RandomSeed",
	  "RandomSeeding",
	  "RandomVariate",
	  "RandomWalkProcess",
	  "RandomWord",
	  "Range",
	  "RangeFilter",
	  "RangeSpecification",
	  "RankedMax",
	  "RankedMin",
	  "RarerProbability",
	  "Raster",
	  "Raster3D",
	  "Raster3DBox",
	  "Raster3DBoxOptions",
	  "RasterArray",
	  "RasterBox",
	  "RasterBoxOptions",
	  "Rasterize",
	  "RasterSize",
	  "Rational",
	  "RationalFunctions",
	  "Rationalize",
	  "Rationals",
	  "Ratios",
	  "RawArray",
	  "RawBoxes",
	  "RawData",
	  "RawMedium",
	  "RayleighDistribution",
	  "Re",
	  "Read",
	  "ReadByteArray",
	  "ReadLine",
	  "ReadList",
	  "ReadProtected",
	  "ReadString",
	  "Real",
	  "RealAbs",
	  "RealBlockDiagonalForm",
	  "RealDigits",
	  "RealExponent",
	  "Reals",
	  "RealSign",
	  "Reap",
	  "RebuildPacletData",
	  "RecognitionPrior",
	  "RecognitionThreshold",
	  "Record",
	  "RecordLists",
	  "RecordSeparators",
	  "Rectangle",
	  "RectangleBox",
	  "RectangleBoxOptions",
	  "RectangleChart",
	  "RectangleChart3D",
	  "RectangularRepeatingElement",
	  "RecurrenceFilter",
	  "RecurrenceTable",
	  "RecurringDigitsForm",
	  "Red",
	  "Reduce",
	  "RefBox",
	  "ReferenceLineStyle",
	  "ReferenceMarkers",
	  "ReferenceMarkerStyle",
	  "Refine",
	  "ReflectionMatrix",
	  "ReflectionTransform",
	  "Refresh",
	  "RefreshRate",
	  "Region",
	  "RegionBinarize",
	  "RegionBoundary",
	  "RegionBoundaryStyle",
	  "RegionBounds",
	  "RegionCentroid",
	  "RegionDifference",
	  "RegionDimension",
	  "RegionDisjoint",
	  "RegionDistance",
	  "RegionDistanceFunction",
	  "RegionEmbeddingDimension",
	  "RegionEqual",
	  "RegionFillingStyle",
	  "RegionFunction",
	  "RegionImage",
	  "RegionIntersection",
	  "RegionMeasure",
	  "RegionMember",
	  "RegionMemberFunction",
	  "RegionMoment",
	  "RegionNearest",
	  "RegionNearestFunction",
	  "RegionPlot",
	  "RegionPlot3D",
	  "RegionProduct",
	  "RegionQ",
	  "RegionResize",
	  "RegionSize",
	  "RegionSymmetricDifference",
	  "RegionUnion",
	  "RegionWithin",
	  "RegisterExternalEvaluator",
	  "RegularExpression",
	  "Regularization",
	  "RegularlySampledQ",
	  "RegularPolygon",
	  "ReIm",
	  "ReImLabels",
	  "ReImPlot",
	  "ReImStyle",
	  "Reinstall",
	  "RelationalDatabase",
	  "RelationGraph",
	  "Release",
	  "ReleaseHold",
	  "ReliabilityDistribution",
	  "ReliefImage",
	  "ReliefPlot",
	  "RemoteAuthorizationCaching",
	  "RemoteConnect",
	  "RemoteConnectionObject",
	  "RemoteFile",
	  "RemoteRun",
	  "RemoteRunProcess",
	  "Remove",
	  "RemoveAlphaChannel",
	  "RemoveAsynchronousTask",
	  "RemoveAudioStream",
	  "RemoveBackground",
	  "RemoveChannelListener",
	  "RemoveChannelSubscribers",
	  "Removed",
	  "RemoveDiacritics",
	  "RemoveInputStreamMethod",
	  "RemoveOutputStreamMethod",
	  "RemoveProperty",
	  "RemoveScheduledTask",
	  "RemoveUsers",
	  "RemoveVideoStream",
	  "RenameDirectory",
	  "RenameFile",
	  "RenderAll",
	  "RenderingOptions",
	  "RenewalProcess",
	  "RenkoChart",
	  "RepairMesh",
	  "Repeated",
	  "RepeatedNull",
	  "RepeatedString",
	  "RepeatedTiming",
	  "RepeatingElement",
	  "Replace",
	  "ReplaceAll",
	  "ReplaceHeldPart",
	  "ReplaceImageValue",
	  "ReplaceList",
	  "ReplacePart",
	  "ReplacePixelValue",
	  "ReplaceRepeated",
	  "ReplicateLayer",
	  "RequiredPhysicalQuantities",
	  "Resampling",
	  "ResamplingAlgorithmData",
	  "ResamplingMethod",
	  "Rescale",
	  "RescalingTransform",
	  "ResetDirectory",
	  "ResetMenusPacket",
	  "ResetScheduledTask",
	  "ReshapeLayer",
	  "Residue",
	  "ResizeLayer",
	  "Resolve",
	  "ResourceAcquire",
	  "ResourceData",
	  "ResourceFunction",
	  "ResourceObject",
	  "ResourceRegister",
	  "ResourceRemove",
	  "ResourceSearch",
	  "ResourceSubmissionObject",
	  "ResourceSubmit",
	  "ResourceSystemBase",
	  "ResourceSystemPath",
	  "ResourceUpdate",
	  "ResourceVersion",
	  "ResponseForm",
	  "Rest",
	  "RestartInterval",
	  "Restricted",
	  "Resultant",
	  "ResumePacket",
	  "Return",
	  "ReturnEntersInput",
	  "ReturnExpressionPacket",
	  "ReturnInputFormPacket",
	  "ReturnPacket",
	  "ReturnReceiptFunction",
	  "ReturnTextPacket",
	  "Reverse",
	  "ReverseApplied",
	  "ReverseBiorthogonalSplineWavelet",
	  "ReverseElement",
	  "ReverseEquilibrium",
	  "ReverseGraph",
	  "ReverseSort",
	  "ReverseSortBy",
	  "ReverseUpEquilibrium",
	  "RevolutionAxis",
	  "RevolutionPlot3D",
	  "RGBColor",
	  "RiccatiSolve",
	  "RiceDistribution",
	  "RidgeFilter",
	  "RiemannR",
	  "RiemannSiegelTheta",
	  "RiemannSiegelZ",
	  "RiemannXi",
	  "Riffle",
	  "Right",
	  "RightArrow",
	  "RightArrowBar",
	  "RightArrowLeftArrow",
	  "RightComposition",
	  "RightCosetRepresentative",
	  "RightDownTeeVector",
	  "RightDownVector",
	  "RightDownVectorBar",
	  "RightTee",
	  "RightTeeArrow",
	  "RightTeeVector",
	  "RightTriangle",
	  "RightTriangleBar",
	  "RightTriangleEqual",
	  "RightUpDownVector",
	  "RightUpTeeVector",
	  "RightUpVector",
	  "RightUpVectorBar",
	  "RightVector",
	  "RightVectorBar",
	  "RiskAchievementImportance",
	  "RiskReductionImportance",
	  "RogersTanimotoDissimilarity",
	  "RollPitchYawAngles",
	  "RollPitchYawMatrix",
	  "RomanNumeral",
	  "Root",
	  "RootApproximant",
	  "RootIntervals",
	  "RootLocusPlot",
	  "RootMeanSquare",
	  "RootOfUnityQ",
	  "RootReduce",
	  "Roots",
	  "RootSum",
	  "Rotate",
	  "RotateLabel",
	  "RotateLeft",
	  "RotateRight",
	  "RotationAction",
	  "RotationBox",
	  "RotationBoxOptions",
	  "RotationMatrix",
	  "RotationTransform",
	  "Round",
	  "RoundImplies",
	  "RoundingRadius",
	  "Row",
	  "RowAlignments",
	  "RowBackgrounds",
	  "RowBox",
	  "RowHeights",
	  "RowLines",
	  "RowMinHeight",
	  "RowReduce",
	  "RowsEqual",
	  "RowSpacings",
	  "RSolve",
	  "RSolveValue",
	  "RudinShapiro",
	  "RudvalisGroupRu",
	  "Rule",
	  "RuleCondition",
	  "RuleDelayed",
	  "RuleForm",
	  "RulePlot",
	  "RulerUnits",
	  "Run",
	  "RunProcess",
	  "RunScheduledTask",
	  "RunThrough",
	  "RuntimeAttributes",
	  "RuntimeOptions",
	  "RussellRaoDissimilarity",
	  "SameQ",
	  "SameTest",
	  "SameTestProperties",
	  "SampledEntityClass",
	  "SampleDepth",
	  "SampledSoundFunction",
	  "SampledSoundList",
	  "SampleRate",
	  "SamplingPeriod",
	  "SARIMAProcess",
	  "SARMAProcess",
	  "SASTriangle",
	  "SatelliteData",
	  "SatisfiabilityCount",
	  "SatisfiabilityInstances",
	  "SatisfiableQ",
	  "Saturday",
	  "Save",
	  "Saveable",
	  "SaveAutoDelete",
	  "SaveConnection",
	  "SaveDefinitions",
	  "SavitzkyGolayMatrix",
	  "SawtoothWave",
	  "Scale",
	  "Scaled",
	  "ScaleDivisions",
	  "ScaledMousePosition",
	  "ScaleOrigin",
	  "ScalePadding",
	  "ScaleRanges",
	  "ScaleRangeStyle",
	  "ScalingFunctions",
	  "ScalingMatrix",
	  "ScalingTransform",
	  "Scan",
	  "ScheduledTask",
	  "ScheduledTaskActiveQ",
	  "ScheduledTaskInformation",
	  "ScheduledTaskInformationData",
	  "ScheduledTaskObject",
	  "ScheduledTasks",
	  "SchurDecomposition",
	  "ScientificForm",
	  "ScientificNotationThreshold",
	  "ScorerGi",
	  "ScorerGiPrime",
	  "ScorerHi",
	  "ScorerHiPrime",
	  "ScreenRectangle",
	  "ScreenStyleEnvironment",
	  "ScriptBaselineShifts",
	  "ScriptForm",
	  "ScriptLevel",
	  "ScriptMinSize",
	  "ScriptRules",
	  "ScriptSizeMultipliers",
	  "Scrollbars",
	  "ScrollingOptions",
	  "ScrollPosition",
	  "SearchAdjustment",
	  "SearchIndexObject",
	  "SearchIndices",
	  "SearchQueryString",
	  "SearchResultObject",
	  "Sec",
	  "Sech",
	  "SechDistribution",
	  "SecondOrderConeOptimization",
	  "SectionGrouping",
	  "SectorChart",
	  "SectorChart3D",
	  "SectorOrigin",
	  "SectorSpacing",
	  "SecuredAuthenticationKey",
	  "SecuredAuthenticationKeys",
	  "SeedRandom",
	  "Select",
	  "Selectable",
	  "SelectComponents",
	  "SelectedCells",
	  "SelectedNotebook",
	  "SelectFirst",
	  "Selection",
	  "SelectionAnimate",
	  "SelectionCell",
	  "SelectionCellCreateCell",
	  "SelectionCellDefaultStyle",
	  "SelectionCellParentStyle",
	  "SelectionCreateCell",
	  "SelectionDebuggerTag",
	  "SelectionDuplicateCell",
	  "SelectionEvaluate",
	  "SelectionEvaluateCreateCell",
	  "SelectionMove",
	  "SelectionPlaceholder",
	  "SelectionSetStyle",
	  "SelectWithContents",
	  "SelfLoops",
	  "SelfLoopStyle",
	  "SemanticImport",
	  "SemanticImportString",
	  "SemanticInterpretation",
	  "SemialgebraicComponentInstances",
	  "SemidefiniteOptimization",
	  "SendMail",
	  "SendMessage",
	  "Sequence",
	  "SequenceAlignment",
	  "SequenceAttentionLayer",
	  "SequenceCases",
	  "SequenceCount",
	  "SequenceFold",
	  "SequenceFoldList",
	  "SequenceForm",
	  "SequenceHold",
	  "SequenceLastLayer",
	  "SequenceMostLayer",
	  "SequencePosition",
	  "SequencePredict",
	  "SequencePredictorFunction",
	  "SequenceReplace",
	  "SequenceRestLayer",
	  "SequenceReverseLayer",
	  "SequenceSplit",
	  "Series",
	  "SeriesCoefficient",
	  "SeriesData",
	  "SeriesTermGoal",
	  "ServiceConnect",
	  "ServiceDisconnect",
	  "ServiceExecute",
	  "ServiceObject",
	  "ServiceRequest",
	  "ServiceResponse",
	  "ServiceSubmit",
	  "SessionSubmit",
	  "SessionTime",
	  "Set",
	  "SetAccuracy",
	  "SetAlphaChannel",
	  "SetAttributes",
	  "Setbacks",
	  "SetBoxFormNamesPacket",
	  "SetCloudDirectory",
	  "SetCookies",
	  "SetDelayed",
	  "SetDirectory",
	  "SetEnvironment",
	  "SetEvaluationNotebook",
	  "SetFileDate",
	  "SetFileLoadingContext",
	  "SetNotebookStatusLine",
	  "SetOptions",
	  "SetOptionsPacket",
	  "SetPermissions",
	  "SetPrecision",
	  "SetProperty",
	  "SetSecuredAuthenticationKey",
	  "SetSelectedNotebook",
	  "SetSharedFunction",
	  "SetSharedVariable",
	  "SetSpeechParametersPacket",
	  "SetStreamPosition",
	  "SetSystemModel",
	  "SetSystemOptions",
	  "Setter",
	  "SetterBar",
	  "SetterBox",
	  "SetterBoxOptions",
	  "Setting",
	  "SetUsers",
	  "SetValue",
	  "Shading",
	  "Shallow",
	  "ShannonWavelet",
	  "ShapiroWilkTest",
	  "Share",
	  "SharingList",
	  "Sharpen",
	  "ShearingMatrix",
	  "ShearingTransform",
	  "ShellRegion",
	  "ShenCastanMatrix",
	  "ShiftedGompertzDistribution",
	  "ShiftRegisterSequence",
	  "Short",
	  "ShortDownArrow",
	  "Shortest",
	  "ShortestMatch",
	  "ShortestPathFunction",
	  "ShortLeftArrow",
	  "ShortRightArrow",
	  "ShortTimeFourier",
	  "ShortTimeFourierData",
	  "ShortUpArrow",
	  "Show",
	  "ShowAutoConvert",
	  "ShowAutoSpellCheck",
	  "ShowAutoStyles",
	  "ShowCellBracket",
	  "ShowCellLabel",
	  "ShowCellTags",
	  "ShowClosedCellArea",
	  "ShowCodeAssist",
	  "ShowContents",
	  "ShowControls",
	  "ShowCursorTracker",
	  "ShowGroupOpenCloseIcon",
	  "ShowGroupOpener",
	  "ShowInvisibleCharacters",
	  "ShowPageBreaks",
	  "ShowPredictiveInterface",
	  "ShowSelection",
	  "ShowShortBoxForm",
	  "ShowSpecialCharacters",
	  "ShowStringCharacters",
	  "ShowSyntaxStyles",
	  "ShrinkingDelay",
	  "ShrinkWrapBoundingBox",
	  "SiderealTime",
	  "SiegelTheta",
	  "SiegelTukeyTest",
	  "SierpinskiCurve",
	  "SierpinskiMesh",
	  "Sign",
	  "Signature",
	  "SignedRankTest",
	  "SignedRegionDistance",
	  "SignificanceLevel",
	  "SignPadding",
	  "SignTest",
	  "SimilarityRules",
	  "SimpleGraph",
	  "SimpleGraphQ",
	  "SimplePolygonQ",
	  "SimplePolyhedronQ",
	  "Simplex",
	  "Simplify",
	  "Sin",
	  "Sinc",
	  "SinghMaddalaDistribution",
	  "SingleEvaluation",
	  "SingleLetterItalics",
	  "SingleLetterStyle",
	  "SingularValueDecomposition",
	  "SingularValueList",
	  "SingularValuePlot",
	  "SingularValues",
	  "Sinh",
	  "SinhIntegral",
	  "SinIntegral",
	  "SixJSymbol",
	  "Skeleton",
	  "SkeletonTransform",
	  "SkellamDistribution",
	  "Skewness",
	  "SkewNormalDistribution",
	  "SkinStyle",
	  "Skip",
	  "SliceContourPlot3D",
	  "SliceDensityPlot3D",
	  "SliceDistribution",
	  "SliceVectorPlot3D",
	  "Slider",
	  "Slider2D",
	  "Slider2DBox",
	  "Slider2DBoxOptions",
	  "SliderBox",
	  "SliderBoxOptions",
	  "SlideView",
	  "Slot",
	  "SlotSequence",
	  "Small",
	  "SmallCircle",
	  "Smaller",
	  "SmithDecomposition",
	  "SmithDelayCompensator",
	  "SmithWatermanSimilarity",
	  "SmoothDensityHistogram",
	  "SmoothHistogram",
	  "SmoothHistogram3D",
	  "SmoothKernelDistribution",
	  "SnDispersion",
	  "Snippet",
	  "SnubPolyhedron",
	  "SocialMediaData",
	  "Socket",
	  "SocketConnect",
	  "SocketListen",
	  "SocketListener",
	  "SocketObject",
	  "SocketOpen",
	  "SocketReadMessage",
	  "SocketReadyQ",
	  "Sockets",
	  "SocketWaitAll",
	  "SocketWaitNext",
	  "SoftmaxLayer",
	  "SokalSneathDissimilarity",
	  "SolarEclipse",
	  "SolarSystemFeatureData",
	  "SolidAngle",
	  "SolidData",
	  "SolidRegionQ",
	  "Solve",
	  "SolveAlways",
	  "SolveDelayed",
	  "Sort",
	  "SortBy",
	  "SortedBy",
	  "SortedEntityClass",
	  "Sound",
	  "SoundAndGraphics",
	  "SoundNote",
	  "SoundVolume",
	  "SourceLink",
	  "Sow",
	  "Space",
	  "SpaceCurveData",
	  "SpaceForm",
	  "Spacer",
	  "Spacings",
	  "Span",
	  "SpanAdjustments",
	  "SpanCharacterRounding",
	  "SpanFromAbove",
	  "SpanFromBoth",
	  "SpanFromLeft",
	  "SpanLineThickness",
	  "SpanMaxSize",
	  "SpanMinSize",
	  "SpanningCharacters",
	  "SpanSymmetric",
	  "SparseArray",
	  "SpatialGraphDistribution",
	  "SpatialMedian",
	  "SpatialTransformationLayer",
	  "Speak",
	  "SpeakerMatchQ",
	  "SpeakTextPacket",
	  "SpearmanRankTest",
	  "SpearmanRho",
	  "SpeciesData",
	  "SpecificityGoal",
	  "SpectralLineData",
	  "Spectrogram",
	  "SpectrogramArray",
	  "Specularity",
	  "SpeechCases",
	  "SpeechInterpreter",
	  "SpeechRecognize",
	  "SpeechSynthesize",
	  "SpellingCorrection",
	  "SpellingCorrectionList",
	  "SpellingDictionaries",
	  "SpellingDictionariesPath",
	  "SpellingOptions",
	  "SpellingSuggestionsPacket",
	  "Sphere",
	  "SphereBox",
	  "SpherePoints",
	  "SphericalBesselJ",
	  "SphericalBesselY",
	  "SphericalHankelH1",
	  "SphericalHankelH2",
	  "SphericalHarmonicY",
	  "SphericalPlot3D",
	  "SphericalRegion",
	  "SphericalShell",
	  "SpheroidalEigenvalue",
	  "SpheroidalJoiningFactor",
	  "SpheroidalPS",
	  "SpheroidalPSPrime",
	  "SpheroidalQS",
	  "SpheroidalQSPrime",
	  "SpheroidalRadialFactor",
	  "SpheroidalS1",
	  "SpheroidalS1Prime",
	  "SpheroidalS2",
	  "SpheroidalS2Prime",
	  "Splice",
	  "SplicedDistribution",
	  "SplineClosed",
	  "SplineDegree",
	  "SplineKnots",
	  "SplineWeights",
	  "Split",
	  "SplitBy",
	  "SpokenString",
	  "Sqrt",
	  "SqrtBox",
	  "SqrtBoxOptions",
	  "Square",
	  "SquaredEuclideanDistance",
	  "SquareFreeQ",
	  "SquareIntersection",
	  "SquareMatrixQ",
	  "SquareRepeatingElement",
	  "SquaresR",
	  "SquareSubset",
	  "SquareSubsetEqual",
	  "SquareSuperset",
	  "SquareSupersetEqual",
	  "SquareUnion",
	  "SquareWave",
	  "SSSTriangle",
	  "StabilityMargins",
	  "StabilityMarginsStyle",
	  "StableDistribution",
	  "Stack",
	  "StackBegin",
	  "StackComplete",
	  "StackedDateListPlot",
	  "StackedListPlot",
	  "StackInhibit",
	  "StadiumShape",
	  "StandardAtmosphereData",
	  "StandardDeviation",
	  "StandardDeviationFilter",
	  "StandardForm",
	  "Standardize",
	  "Standardized",
	  "StandardOceanData",
	  "StandbyDistribution",
	  "Star",
	  "StarClusterData",
	  "StarData",
	  "StarGraph",
	  "StartAsynchronousTask",
	  "StartExternalSession",
	  "StartingStepSize",
	  "StartOfLine",
	  "StartOfString",
	  "StartProcess",
	  "StartScheduledTask",
	  "StartupSound",
	  "StartWebSession",
	  "StateDimensions",
	  "StateFeedbackGains",
	  "StateOutputEstimator",
	  "StateResponse",
	  "StateSpaceModel",
	  "StateSpaceRealization",
	  "StateSpaceTransform",
	  "StateTransformationLinearize",
	  "StationaryDistribution",
	  "StationaryWaveletPacketTransform",
	  "StationaryWaveletTransform",
	  "StatusArea",
	  "StatusCentrality",
	  "StepMonitor",
	  "StereochemistryElements",
	  "StieltjesGamma",
	  "StippleShading",
	  "StirlingS1",
	  "StirlingS2",
	  "StopAsynchronousTask",
	  "StoppingPowerData",
	  "StopScheduledTask",
	  "StrataVariables",
	  "StratonovichProcess",
	  "StreamColorFunction",
	  "StreamColorFunctionScaling",
	  "StreamDensityPlot",
	  "StreamMarkers",
	  "StreamPlot",
	  "StreamPoints",
	  "StreamPosition",
	  "Streams",
	  "StreamScale",
	  "StreamStyle",
	  "String",
	  "StringBreak",
	  "StringByteCount",
	  "StringCases",
	  "StringContainsQ",
	  "StringCount",
	  "StringDelete",
	  "StringDrop",
	  "StringEndsQ",
	  "StringExpression",
	  "StringExtract",
	  "StringForm",
	  "StringFormat",
	  "StringFreeQ",
	  "StringInsert",
	  "StringJoin",
	  "StringLength",
	  "StringMatchQ",
	  "StringPadLeft",
	  "StringPadRight",
	  "StringPart",
	  "StringPartition",
	  "StringPosition",
	  "StringQ",
	  "StringRepeat",
	  "StringReplace",
	  "StringReplaceList",
	  "StringReplacePart",
	  "StringReverse",
	  "StringRiffle",
	  "StringRotateLeft",
	  "StringRotateRight",
	  "StringSkeleton",
	  "StringSplit",
	  "StringStartsQ",
	  "StringTake",
	  "StringTemplate",
	  "StringToByteArray",
	  "StringToStream",
	  "StringTrim",
	  "StripBoxes",
	  "StripOnInput",
	  "StripWrapperBoxes",
	  "StrokeForm",
	  "StructuralImportance",
	  "StructuredArray",
	  "StructuredArrayHeadQ",
	  "StructuredSelection",
	  "StruveH",
	  "StruveL",
	  "Stub",
	  "StudentTDistribution",
	  "Style",
	  "StyleBox",
	  "StyleBoxAutoDelete",
	  "StyleData",
	  "StyleDefinitions",
	  "StyleForm",
	  "StyleHints",
	  "StyleKeyMapping",
	  "StyleMenuListing",
	  "StyleNameDialogSettings",
	  "StyleNames",
	  "StylePrint",
	  "StyleSheetPath",
	  "Subdivide",
	  "Subfactorial",
	  "Subgraph",
	  "SubMinus",
	  "SubPlus",
	  "SubresultantPolynomialRemainders",
	  "SubresultantPolynomials",
	  "Subresultants",
	  "Subscript",
	  "SubscriptBox",
	  "SubscriptBoxOptions",
	  "Subscripted",
	  "Subsequences",
	  "Subset",
	  "SubsetCases",
	  "SubsetCount",
	  "SubsetEqual",
	  "SubsetMap",
	  "SubsetPosition",
	  "SubsetQ",
	  "SubsetReplace",
	  "Subsets",
	  "SubStar",
	  "SubstitutionSystem",
	  "Subsuperscript",
	  "SubsuperscriptBox",
	  "SubsuperscriptBoxOptions",
	  "SubtitleEncoding",
	  "SubtitleTracks",
	  "Subtract",
	  "SubtractFrom",
	  "SubtractSides",
	  "SubValues",
	  "Succeeds",
	  "SucceedsEqual",
	  "SucceedsSlantEqual",
	  "SucceedsTilde",
	  "Success",
	  "SuchThat",
	  "Sum",
	  "SumConvergence",
	  "SummationLayer",
	  "Sunday",
	  "SunPosition",
	  "Sunrise",
	  "Sunset",
	  "SuperDagger",
	  "SuperMinus",
	  "SupernovaData",
	  "SuperPlus",
	  "Superscript",
	  "SuperscriptBox",
	  "SuperscriptBoxOptions",
	  "Superset",
	  "SupersetEqual",
	  "SuperStar",
	  "Surd",
	  "SurdForm",
	  "SurfaceAppearance",
	  "SurfaceArea",
	  "SurfaceColor",
	  "SurfaceData",
	  "SurfaceGraphics",
	  "SurvivalDistribution",
	  "SurvivalFunction",
	  "SurvivalModel",
	  "SurvivalModelFit",
	  "SuspendPacket",
	  "SuzukiDistribution",
	  "SuzukiGroupSuz",
	  "SwatchLegend",
	  "Switch",
	  "Symbol",
	  "SymbolName",
	  "SymletWavelet",
	  "Symmetric",
	  "SymmetricGroup",
	  "SymmetricKey",
	  "SymmetricMatrixQ",
	  "SymmetricPolynomial",
	  "SymmetricReduction",
	  "Symmetrize",
	  "SymmetrizedArray",
	  "SymmetrizedArrayRules",
	  "SymmetrizedDependentComponents",
	  "SymmetrizedIndependentComponents",
	  "SymmetrizedReplacePart",
	  "SynchronousInitialization",
	  "SynchronousUpdating",
	  "Synonyms",
	  "Syntax",
	  "SyntaxForm",
	  "SyntaxInformation",
	  "SyntaxLength",
	  "SyntaxPacket",
	  "SyntaxQ",
	  "SynthesizeMissingValues",
	  "SystemCredential",
	  "SystemCredentialData",
	  "SystemCredentialKey",
	  "SystemCredentialKeys",
	  "SystemCredentialStoreObject",
	  "SystemDialogInput",
	  "SystemException",
	  "SystemGet",
	  "SystemHelpPath",
	  "SystemInformation",
	  "SystemInformationData",
	  "SystemInstall",
	  "SystemModel",
	  "SystemModeler",
	  "SystemModelExamples",
	  "SystemModelLinearize",
	  "SystemModelParametricSimulate",
	  "SystemModelPlot",
	  "SystemModelProgressReporting",
	  "SystemModelReliability",
	  "SystemModels",
	  "SystemModelSimulate",
	  "SystemModelSimulateSensitivity",
	  "SystemModelSimulationData",
	  "SystemOpen",
	  "SystemOptions",
	  "SystemProcessData",
	  "SystemProcesses",
	  "SystemsConnectionsModel",
	  "SystemsModelDelay",
	  "SystemsModelDelayApproximate",
	  "SystemsModelDelete",
	  "SystemsModelDimensions",
	  "SystemsModelExtract",
	  "SystemsModelFeedbackConnect",
	  "SystemsModelLabels",
	  "SystemsModelLinearity",
	  "SystemsModelMerge",
	  "SystemsModelOrder",
	  "SystemsModelParallelConnect",
	  "SystemsModelSeriesConnect",
	  "SystemsModelStateFeedbackConnect",
	  "SystemsModelVectorRelativeOrders",
	  "SystemStub",
	  "SystemTest",
	  "Tab",
	  "TabFilling",
	  "Table",
	  "TableAlignments",
	  "TableDepth",
	  "TableDirections",
	  "TableForm",
	  "TableHeadings",
	  "TableSpacing",
	  "TableView",
	  "TableViewBox",
	  "TableViewBoxBackground",
	  "TableViewBoxItemSize",
	  "TableViewBoxOptions",
	  "TabSpacings",
	  "TabView",
	  "TabViewBox",
	  "TabViewBoxOptions",
	  "TagBox",
	  "TagBoxNote",
	  "TagBoxOptions",
	  "TaggingRules",
	  "TagSet",
	  "TagSetDelayed",
	  "TagStyle",
	  "TagUnset",
	  "Take",
	  "TakeDrop",
	  "TakeLargest",
	  "TakeLargestBy",
	  "TakeList",
	  "TakeSmallest",
	  "TakeSmallestBy",
	  "TakeWhile",
	  "Tally",
	  "Tan",
	  "Tanh",
	  "TargetDevice",
	  "TargetFunctions",
	  "TargetSystem",
	  "TargetUnits",
	  "TaskAbort",
	  "TaskExecute",
	  "TaskObject",
	  "TaskRemove",
	  "TaskResume",
	  "Tasks",
	  "TaskSuspend",
	  "TaskWait",
	  "TautologyQ",
	  "TelegraphProcess",
	  "TemplateApply",
	  "TemplateArgBox",
	  "TemplateBox",
	  "TemplateBoxOptions",
	  "TemplateEvaluate",
	  "TemplateExpression",
	  "TemplateIf",
	  "TemplateObject",
	  "TemplateSequence",
	  "TemplateSlot",
	  "TemplateSlotSequence",
	  "TemplateUnevaluated",
	  "TemplateVerbatim",
	  "TemplateWith",
	  "TemporalData",
	  "TemporalRegularity",
	  "Temporary",
	  "TemporaryVariable",
	  "TensorContract",
	  "TensorDimensions",
	  "TensorExpand",
	  "TensorProduct",
	  "TensorQ",
	  "TensorRank",
	  "TensorReduce",
	  "TensorSymmetry",
	  "TensorTranspose",
	  "TensorWedge",
	  "TestID",
	  "TestReport",
	  "TestReportObject",
	  "TestResultObject",
	  "Tetrahedron",
	  "TetrahedronBox",
	  "TetrahedronBoxOptions",
	  "TeXForm",
	  "TeXSave",
	  "Text",
	  "Text3DBox",
	  "Text3DBoxOptions",
	  "TextAlignment",
	  "TextBand",
	  "TextBoundingBox",
	  "TextBox",
	  "TextCases",
	  "TextCell",
	  "TextClipboardType",
	  "TextContents",
	  "TextData",
	  "TextElement",
	  "TextForm",
	  "TextGrid",
	  "TextJustification",
	  "TextLine",
	  "TextPacket",
	  "TextParagraph",
	  "TextPosition",
	  "TextRecognize",
	  "TextSearch",
	  "TextSearchReport",
	  "TextSentences",
	  "TextString",
	  "TextStructure",
	  "TextStyle",
	  "TextTranslation",
	  "Texture",
	  "TextureCoordinateFunction",
	  "TextureCoordinateScaling",
	  "TextWords",
	  "Therefore",
	  "ThermodynamicData",
	  "ThermometerGauge",
	  "Thick",
	  "Thickness",
	  "Thin",
	  "Thinning",
	  "ThisLink",
	  "ThompsonGroupTh",
	  "Thread",
	  "ThreadingLayer",
	  "ThreeJSymbol",
	  "Threshold",
	  "Through",
	  "Throw",
	  "ThueMorse",
	  "Thumbnail",
	  "Thursday",
	  "Ticks",
	  "TicksStyle",
	  "TideData",
	  "Tilde",
	  "TildeEqual",
	  "TildeFullEqual",
	  "TildeTilde",
	  "TimeConstrained",
	  "TimeConstraint",
	  "TimeDirection",
	  "TimeFormat",
	  "TimeGoal",
	  "TimelinePlot",
	  "TimeObject",
	  "TimeObjectQ",
	  "TimeRemaining",
	  "Times",
	  "TimesBy",
	  "TimeSeries",
	  "TimeSeriesAggregate",
	  "TimeSeriesForecast",
	  "TimeSeriesInsert",
	  "TimeSeriesInvertibility",
	  "TimeSeriesMap",
	  "TimeSeriesMapThread",
	  "TimeSeriesModel",
	  "TimeSeriesModelFit",
	  "TimeSeriesResample",
	  "TimeSeriesRescale",
	  "TimeSeriesShift",
	  "TimeSeriesThread",
	  "TimeSeriesWindow",
	  "TimeUsed",
	  "TimeValue",
	  "TimeWarpingCorrespondence",
	  "TimeWarpingDistance",
	  "TimeZone",
	  "TimeZoneConvert",
	  "TimeZoneOffset",
	  "Timing",
	  "Tiny",
	  "TitleGrouping",
	  "TitsGroupT",
	  "ToBoxes",
	  "ToCharacterCode",
	  "ToColor",
	  "ToContinuousTimeModel",
	  "ToDate",
	  "Today",
	  "ToDiscreteTimeModel",
	  "ToEntity",
	  "ToeplitzMatrix",
	  "ToExpression",
	  "ToFileName",
	  "Together",
	  "Toggle",
	  "ToggleFalse",
	  "Toggler",
	  "TogglerBar",
	  "TogglerBox",
	  "TogglerBoxOptions",
	  "ToHeldExpression",
	  "ToInvertibleTimeSeries",
	  "TokenWords",
	  "Tolerance",
	  "ToLowerCase",
	  "Tomorrow",
	  "ToNumberField",
	  "TooBig",
	  "Tooltip",
	  "TooltipBox",
	  "TooltipBoxOptions",
	  "TooltipDelay",
	  "TooltipStyle",
	  "ToonShading",
	  "Top",
	  "TopHatTransform",
	  "ToPolarCoordinates",
	  "TopologicalSort",
	  "ToRadicals",
	  "ToRules",
	  "ToSphericalCoordinates",
	  "ToString",
	  "Total",
	  "TotalHeight",
	  "TotalLayer",
	  "TotalVariationFilter",
	  "TotalWidth",
	  "TouchPosition",
	  "TouchscreenAutoZoom",
	  "TouchscreenControlPlacement",
	  "ToUpperCase",
	  "Tr",
	  "Trace",
	  "TraceAbove",
	  "TraceAction",
	  "TraceBackward",
	  "TraceDepth",
	  "TraceDialog",
	  "TraceForward",
	  "TraceInternal",
	  "TraceLevel",
	  "TraceOff",
	  "TraceOn",
	  "TraceOriginal",
	  "TracePrint",
	  "TraceScan",
	  "TrackedSymbols",
	  "TrackingFunction",
	  "TracyWidomDistribution",
	  "TradingChart",
	  "TraditionalForm",
	  "TraditionalFunctionNotation",
	  "TraditionalNotation",
	  "TraditionalOrder",
	  "TrainingProgressCheckpointing",
	  "TrainingProgressFunction",
	  "TrainingProgressMeasurements",
	  "TrainingProgressReporting",
	  "TrainingStoppingCriterion",
	  "TrainingUpdateSchedule",
	  "TransferFunctionCancel",
	  "TransferFunctionExpand",
	  "TransferFunctionFactor",
	  "TransferFunctionModel",
	  "TransferFunctionPoles",
	  "TransferFunctionTransform",
	  "TransferFunctionZeros",
	  "TransformationClass",
	  "TransformationFunction",
	  "TransformationFunctions",
	  "TransformationMatrix",
	  "TransformedDistribution",
	  "TransformedField",
	  "TransformedProcess",
	  "TransformedRegion",
	  "TransitionDirection",
	  "TransitionDuration",
	  "TransitionEffect",
	  "TransitiveClosureGraph",
	  "TransitiveReductionGraph",
	  "Translate",
	  "TranslationOptions",
	  "TranslationTransform",
	  "Transliterate",
	  "Transparent",
	  "TransparentColor",
	  "Transpose",
	  "TransposeLayer",
	  "TrapSelection",
	  "TravelDirections",
	  "TravelDirectionsData",
	  "TravelDistance",
	  "TravelDistanceList",
	  "TravelMethod",
	  "TravelTime",
	  "TreeForm",
	  "TreeGraph",
	  "TreeGraphQ",
	  "TreePlot",
	  "TrendStyle",
	  "Triangle",
	  "TriangleCenter",
	  "TriangleConstruct",
	  "TriangleMeasurement",
	  "TriangleWave",
	  "TriangularDistribution",
	  "TriangulateMesh",
	  "Trig",
	  "TrigExpand",
	  "TrigFactor",
	  "TrigFactorList",
	  "Trigger",
	  "TrigReduce",
	  "TrigToExp",
	  "TrimmedMean",
	  "TrimmedVariance",
	  "TropicalStormData",
	  "True",
	  "TrueQ",
	  "TruncatedDistribution",
	  "TruncatedPolyhedron",
	  "TsallisQExponentialDistribution",
	  "TsallisQGaussianDistribution",
	  "TTest",
	  "Tube",
	  "TubeBezierCurveBox",
	  "TubeBezierCurveBoxOptions",
	  "TubeBox",
	  "TubeBoxOptions",
	  "TubeBSplineCurveBox",
	  "TubeBSplineCurveBoxOptions",
	  "Tuesday",
	  "TukeyLambdaDistribution",
	  "TukeyWindow",
	  "TunnelData",
	  "Tuples",
	  "TuranGraph",
	  "TuringMachine",
	  "TuttePolynomial",
	  "TwoWayRule",
	  "Typed",
	  "TypeSpecifier",
	  "UnateQ",
	  "Uncompress",
	  "UnconstrainedParameters",
	  "Undefined",
	  "UnderBar",
	  "Underflow",
	  "Underlined",
	  "Underoverscript",
	  "UnderoverscriptBox",
	  "UnderoverscriptBoxOptions",
	  "Underscript",
	  "UnderscriptBox",
	  "UnderscriptBoxOptions",
	  "UnderseaFeatureData",
	  "UndirectedEdge",
	  "UndirectedGraph",
	  "UndirectedGraphQ",
	  "UndoOptions",
	  "UndoTrackedVariables",
	  "Unequal",
	  "UnequalTo",
	  "Unevaluated",
	  "UniformDistribution",
	  "UniformGraphDistribution",
	  "UniformPolyhedron",
	  "UniformSumDistribution",
	  "Uninstall",
	  "Union",
	  "UnionedEntityClass",
	  "UnionPlus",
	  "Unique",
	  "UnitaryMatrixQ",
	  "UnitBox",
	  "UnitConvert",
	  "UnitDimensions",
	  "Unitize",
	  "UnitRootTest",
	  "UnitSimplify",
	  "UnitStep",
	  "UnitSystem",
	  "UnitTriangle",
	  "UnitVector",
	  "UnitVectorLayer",
	  "UnityDimensions",
	  "UniverseModelData",
	  "UniversityData",
	  "UnixTime",
	  "Unprotect",
	  "UnregisterExternalEvaluator",
	  "UnsameQ",
	  "UnsavedVariables",
	  "Unset",
	  "UnsetShared",
	  "UntrackedVariables",
	  "Up",
	  "UpArrow",
	  "UpArrowBar",
	  "UpArrowDownArrow",
	  "Update",
	  "UpdateDynamicObjects",
	  "UpdateDynamicObjectsSynchronous",
	  "UpdateInterval",
	  "UpdatePacletSites",
	  "UpdateSearchIndex",
	  "UpDownArrow",
	  "UpEquilibrium",
	  "UpperCaseQ",
	  "UpperLeftArrow",
	  "UpperRightArrow",
	  "UpperTriangularize",
	  "UpperTriangularMatrixQ",
	  "Upsample",
	  "UpSet",
	  "UpSetDelayed",
	  "UpTee",
	  "UpTeeArrow",
	  "UpTo",
	  "UpValues",
	  "URL",
	  "URLBuild",
	  "URLDecode",
	  "URLDispatcher",
	  "URLDownload",
	  "URLDownloadSubmit",
	  "URLEncode",
	  "URLExecute",
	  "URLExpand",
	  "URLFetch",
	  "URLFetchAsynchronous",
	  "URLParse",
	  "URLQueryDecode",
	  "URLQueryEncode",
	  "URLRead",
	  "URLResponseTime",
	  "URLSave",
	  "URLSaveAsynchronous",
	  "URLShorten",
	  "URLSubmit",
	  "UseGraphicsRange",
	  "UserDefinedWavelet",
	  "Using",
	  "UsingFrontEnd",
	  "UtilityFunction",
	  "V2Get",
	  "ValenceErrorHandling",
	  "ValidationLength",
	  "ValidationSet",
	  "Value",
	  "ValueBox",
	  "ValueBoxOptions",
	  "ValueDimensions",
	  "ValueForm",
	  "ValuePreprocessingFunction",
	  "ValueQ",
	  "Values",
	  "ValuesData",
	  "Variables",
	  "Variance",
	  "VarianceEquivalenceTest",
	  "VarianceEstimatorFunction",
	  "VarianceGammaDistribution",
	  "VarianceTest",
	  "VectorAngle",
	  "VectorAround",
	  "VectorAspectRatio",
	  "VectorColorFunction",
	  "VectorColorFunctionScaling",
	  "VectorDensityPlot",
	  "VectorGlyphData",
	  "VectorGreater",
	  "VectorGreaterEqual",
	  "VectorLess",
	  "VectorLessEqual",
	  "VectorMarkers",
	  "VectorPlot",
	  "VectorPlot3D",
	  "VectorPoints",
	  "VectorQ",
	  "VectorRange",
	  "Vectors",
	  "VectorScale",
	  "VectorScaling",
	  "VectorSizes",
	  "VectorStyle",
	  "Vee",
	  "Verbatim",
	  "Verbose",
	  "VerboseConvertToPostScriptPacket",
	  "VerificationTest",
	  "VerifyConvergence",
	  "VerifyDerivedKey",
	  "VerifyDigitalSignature",
	  "VerifyFileSignature",
	  "VerifyInterpretation",
	  "VerifySecurityCertificates",
	  "VerifySolutions",
	  "VerifyTestAssumptions",
	  "Version",
	  "VersionedPreferences",
	  "VersionNumber",
	  "VertexAdd",
	  "VertexCapacity",
	  "VertexColors",
	  "VertexComponent",
	  "VertexConnectivity",
	  "VertexContract",
	  "VertexCoordinateRules",
	  "VertexCoordinates",
	  "VertexCorrelationSimilarity",
	  "VertexCosineSimilarity",
	  "VertexCount",
	  "VertexCoverQ",
	  "VertexDataCoordinates",
	  "VertexDegree",
	  "VertexDelete",
	  "VertexDiceSimilarity",
	  "VertexEccentricity",
	  "VertexInComponent",
	  "VertexInDegree",
	  "VertexIndex",
	  "VertexJaccardSimilarity",
	  "VertexLabeling",
	  "VertexLabels",
	  "VertexLabelStyle",
	  "VertexList",
	  "VertexNormals",
	  "VertexOutComponent",
	  "VertexOutDegree",
	  "VertexQ",
	  "VertexRenderingFunction",
	  "VertexReplace",
	  "VertexShape",
	  "VertexShapeFunction",
	  "VertexSize",
	  "VertexStyle",
	  "VertexTextureCoordinates",
	  "VertexWeight",
	  "VertexWeightedGraphQ",
	  "Vertical",
	  "VerticalBar",
	  "VerticalForm",
	  "VerticalGauge",
	  "VerticalSeparator",
	  "VerticalSlider",
	  "VerticalTilde",
	  "Video",
	  "VideoEncoding",
	  "VideoExtractFrames",
	  "VideoFrameList",
	  "VideoFrameMap",
	  "VideoPause",
	  "VideoPlay",
	  "VideoQ",
	  "VideoStop",
	  "VideoStream",
	  "VideoStreams",
	  "VideoTimeSeries",
	  "VideoTracks",
	  "VideoTrim",
	  "ViewAngle",
	  "ViewCenter",
	  "ViewMatrix",
	  "ViewPoint",
	  "ViewPointSelectorSettings",
	  "ViewPort",
	  "ViewProjection",
	  "ViewRange",
	  "ViewVector",
	  "ViewVertical",
	  "VirtualGroupData",
	  "Visible",
	  "VisibleCell",
	  "VoiceStyleData",
	  "VoigtDistribution",
	  "VolcanoData",
	  "Volume",
	  "VonMisesDistribution",
	  "VoronoiMesh",
	  "WaitAll",
	  "WaitAsynchronousTask",
	  "WaitNext",
	  "WaitUntil",
	  "WakebyDistribution",
	  "WalleniusHypergeometricDistribution",
	  "WaringYuleDistribution",
	  "WarpingCorrespondence",
	  "WarpingDistance",
	  "WatershedComponents",
	  "WatsonUSquareTest",
	  "WattsStrogatzGraphDistribution",
	  "WaveletBestBasis",
	  "WaveletFilterCoefficients",
	  "WaveletImagePlot",
	  "WaveletListPlot",
	  "WaveletMapIndexed",
	  "WaveletMatrixPlot",
	  "WaveletPhi",
	  "WaveletPsi",
	  "WaveletScale",
	  "WaveletScalogram",
	  "WaveletThreshold",
	  "WeaklyConnectedComponents",
	  "WeaklyConnectedGraphComponents",
	  "WeaklyConnectedGraphQ",
	  "WeakStationarity",
	  "WeatherData",
	  "WeatherForecastData",
	  "WebAudioSearch",
	  "WebElementObject",
	  "WeberE",
	  "WebExecute",
	  "WebImage",
	  "WebImageSearch",
	  "WebSearch",
	  "WebSessionObject",
	  "WebSessions",
	  "WebWindowObject",
	  "Wedge",
	  "Wednesday",
	  "WeibullDistribution",
	  "WeierstrassE1",
	  "WeierstrassE2",
	  "WeierstrassE3",
	  "WeierstrassEta1",
	  "WeierstrassEta2",
	  "WeierstrassEta3",
	  "WeierstrassHalfPeriods",
	  "WeierstrassHalfPeriodW1",
	  "WeierstrassHalfPeriodW2",
	  "WeierstrassHalfPeriodW3",
	  "WeierstrassInvariantG2",
	  "WeierstrassInvariantG3",
	  "WeierstrassInvariants",
	  "WeierstrassP",
	  "WeierstrassPPrime",
	  "WeierstrassSigma",
	  "WeierstrassZeta",
	  "WeightedAdjacencyGraph",
	  "WeightedAdjacencyMatrix",
	  "WeightedData",
	  "WeightedGraphQ",
	  "Weights",
	  "WelchWindow",
	  "WheelGraph",
	  "WhenEvent",
	  "Which",
	  "While",
	  "White",
	  "WhiteNoiseProcess",
	  "WhitePoint",
	  "Whitespace",
	  "WhitespaceCharacter",
	  "WhittakerM",
	  "WhittakerW",
	  "WienerFilter",
	  "WienerProcess",
	  "WignerD",
	  "WignerSemicircleDistribution",
	  "WikidataData",
	  "WikidataSearch",
	  "WikipediaData",
	  "WikipediaSearch",
	  "WilksW",
	  "WilksWTest",
	  "WindDirectionData",
	  "WindingCount",
	  "WindingPolygon",
	  "WindowClickSelect",
	  "WindowElements",
	  "WindowFloating",
	  "WindowFrame",
	  "WindowFrameElements",
	  "WindowMargins",
	  "WindowMovable",
	  "WindowOpacity",
	  "WindowPersistentStyles",
	  "WindowSelected",
	  "WindowSize",
	  "WindowStatusArea",
	  "WindowTitle",
	  "WindowToolbars",
	  "WindowWidth",
	  "WindSpeedData",
	  "WindVectorData",
	  "WinsorizedMean",
	  "WinsorizedVariance",
	  "WishartMatrixDistribution",
	  "With",
	  "WolframAlpha",
	  "WolframAlphaDate",
	  "WolframAlphaQuantity",
	  "WolframAlphaResult",
	  "WolframLanguageData",
	  "Word",
	  "WordBoundary",
	  "WordCharacter",
	  "WordCloud",
	  "WordCount",
	  "WordCounts",
	  "WordData",
	  "WordDefinition",
	  "WordFrequency",
	  "WordFrequencyData",
	  "WordList",
	  "WordOrientation",
	  "WordSearch",
	  "WordSelectionFunction",
	  "WordSeparators",
	  "WordSpacings",
	  "WordStem",
	  "WordTranslation",
	  "WorkingPrecision",
	  "WrapAround",
	  "Write",
	  "WriteLine",
	  "WriteString",
	  "Wronskian",
	  "XMLElement",
	  "XMLObject",
	  "XMLTemplate",
	  "Xnor",
	  "Xor",
	  "XYZColor",
	  "Yellow",
	  "Yesterday",
	  "YuleDissimilarity",
	  "ZernikeR",
	  "ZeroSymmetric",
	  "ZeroTest",
	  "ZeroWidthTimes",
	  "Zeta",
	  "ZetaZero",
	  "ZIPCodeData",
	  "ZipfDistribution",
	  "ZoomCenter",
	  "ZoomFactor",
	  "ZTest",
	  "ZTransform",
	  "$Aborted",
	  "$ActivationGroupID",
	  "$ActivationKey",
	  "$ActivationUserRegistered",
	  "$AddOnsDirectory",
	  "$AllowDataUpdates",
	  "$AllowExternalChannelFunctions",
	  "$AllowInternet",
	  "$AssertFunction",
	  "$Assumptions",
	  "$AsynchronousTask",
	  "$AudioDecoders",
	  "$AudioEncoders",
	  "$AudioInputDevices",
	  "$AudioOutputDevices",
	  "$BaseDirectory",
	  "$BasePacletsDirectory",
	  "$BatchInput",
	  "$BatchOutput",
	  "$BlockchainBase",
	  "$BoxForms",
	  "$ByteOrdering",
	  "$CacheBaseDirectory",
	  "$Canceled",
	  "$ChannelBase",
	  "$CharacterEncoding",
	  "$CharacterEncodings",
	  "$CloudAccountName",
	  "$CloudBase",
	  "$CloudConnected",
	  "$CloudConnection",
	  "$CloudCreditsAvailable",
	  "$CloudEvaluation",
	  "$CloudExpressionBase",
	  "$CloudObjectNameFormat",
	  "$CloudObjectURLType",
	  "$CloudRootDirectory",
	  "$CloudSymbolBase",
	  "$CloudUserID",
	  "$CloudUserUUID",
	  "$CloudVersion",
	  "$CloudVersionNumber",
	  "$CloudWolframEngineVersionNumber",
	  "$CommandLine",
	  "$CompilationTarget",
	  "$ConditionHold",
	  "$ConfiguredKernels",
	  "$Context",
	  "$ContextPath",
	  "$ControlActiveSetting",
	  "$Cookies",
	  "$CookieStore",
	  "$CreationDate",
	  "$CurrentLink",
	  "$CurrentTask",
	  "$CurrentWebSession",
	  "$DataStructures",
	  "$DateStringFormat",
	  "$DefaultAudioInputDevice",
	  "$DefaultAudioOutputDevice",
	  "$DefaultFont",
	  "$DefaultFrontEnd",
	  "$DefaultImagingDevice",
	  "$DefaultLocalBase",
	  "$DefaultMailbox",
	  "$DefaultNetworkInterface",
	  "$DefaultPath",
	  "$DefaultProxyRules",
	  "$DefaultSystemCredentialStore",
	  "$Display",
	  "$DisplayFunction",
	  "$DistributedContexts",
	  "$DynamicEvaluation",
	  "$Echo",
	  "$EmbedCodeEnvironments",
	  "$EmbeddableServices",
	  "$EntityStores",
	  "$Epilog",
	  "$EvaluationCloudBase",
	  "$EvaluationCloudObject",
	  "$EvaluationEnvironment",
	  "$ExportFormats",
	  "$ExternalIdentifierTypes",
	  "$ExternalStorageBase",
	  "$Failed",
	  "$FinancialDataSource",
	  "$FontFamilies",
	  "$FormatType",
	  "$FrontEnd",
	  "$FrontEndSession",
	  "$GeoEntityTypes",
	  "$GeoLocation",
	  "$GeoLocationCity",
	  "$GeoLocationCountry",
	  "$GeoLocationPrecision",
	  "$GeoLocationSource",
	  "$HistoryLength",
	  "$HomeDirectory",
	  "$HTMLExportRules",
	  "$HTTPCookies",
	  "$HTTPRequest",
	  "$IgnoreEOF",
	  "$ImageFormattingWidth",
	  "$ImageResolution",
	  "$ImagingDevice",
	  "$ImagingDevices",
	  "$ImportFormats",
	  "$IncomingMailSettings",
	  "$InitialDirectory",
	  "$Initialization",
	  "$InitializationContexts",
	  "$Input",
	  "$InputFileName",
	  "$InputStreamMethods",
	  "$Inspector",
	  "$InstallationDate",
	  "$InstallationDirectory",
	  "$InterfaceEnvironment",
	  "$InterpreterTypes",
	  "$IterationLimit",
	  "$KernelCount",
	  "$KernelID",
	  "$Language",
	  "$LaunchDirectory",
	  "$LibraryPath",
	  "$LicenseExpirationDate",
	  "$LicenseID",
	  "$LicenseProcesses",
	  "$LicenseServer",
	  "$LicenseSubprocesses",
	  "$LicenseType",
	  "$Line",
	  "$Linked",
	  "$LinkSupported",
	  "$LoadedFiles",
	  "$LocalBase",
	  "$LocalSymbolBase",
	  "$MachineAddresses",
	  "$MachineDomain",
	  "$MachineDomains",
	  "$MachineEpsilon",
	  "$MachineID",
	  "$MachineName",
	  "$MachinePrecision",
	  "$MachineType",
	  "$MaxExtraPrecision",
	  "$MaxLicenseProcesses",
	  "$MaxLicenseSubprocesses",
	  "$MaxMachineNumber",
	  "$MaxNumber",
	  "$MaxPiecewiseCases",
	  "$MaxPrecision",
	  "$MaxRootDegree",
	  "$MessageGroups",
	  "$MessageList",
	  "$MessagePrePrint",
	  "$Messages",
	  "$MinMachineNumber",
	  "$MinNumber",
	  "$MinorReleaseNumber",
	  "$MinPrecision",
	  "$MobilePhone",
	  "$ModuleNumber",
	  "$NetworkConnected",
	  "$NetworkInterfaces",
	  "$NetworkLicense",
	  "$NewMessage",
	  "$NewSymbol",
	  "$NotebookInlineStorageLimit",
	  "$Notebooks",
	  "$NoValue",
	  "$NumberMarks",
	  "$Off",
	  "$OperatingSystem",
	  "$Output",
	  "$OutputForms",
	  "$OutputSizeLimit",
	  "$OutputStreamMethods",
	  "$Packages",
	  "$ParentLink",
	  "$ParentProcessID",
	  "$PasswordFile",
	  "$PatchLevelID",
	  "$Path",
	  "$PathnameSeparator",
	  "$PerformanceGoal",
	  "$Permissions",
	  "$PermissionsGroupBase",
	  "$PersistenceBase",
	  "$PersistencePath",
	  "$PipeSupported",
	  "$PlotTheme",
	  "$Post",
	  "$Pre",
	  "$PreferencesDirectory",
	  "$PreInitialization",
	  "$PrePrint",
	  "$PreRead",
	  "$PrintForms",
	  "$PrintLiteral",
	  "$Printout3DPreviewer",
	  "$ProcessID",
	  "$ProcessorCount",
	  "$ProcessorType",
	  "$ProductInformation",
	  "$ProgramName",
	  "$PublisherID",
	  "$RandomState",
	  "$RecursionLimit",
	  "$RegisteredDeviceClasses",
	  "$RegisteredUserName",
	  "$ReleaseNumber",
	  "$RequesterAddress",
	  "$RequesterWolframID",
	  "$RequesterWolframUUID",
	  "$RootDirectory",
	  "$ScheduledTask",
	  "$ScriptCommandLine",
	  "$ScriptInputString",
	  "$SecuredAuthenticationKeyTokens",
	  "$ServiceCreditsAvailable",
	  "$Services",
	  "$SessionID",
	  "$SetParentLink",
	  "$SharedFunctions",
	  "$SharedVariables",
	  "$SoundDisplay",
	  "$SoundDisplayFunction",
	  "$SourceLink",
	  "$SSHAuthentication",
	  "$SubtitleDecoders",
	  "$SubtitleEncoders",
	  "$SummaryBoxDataSizeLimit",
	  "$SuppressInputFormHeads",
	  "$SynchronousEvaluation",
	  "$SyntaxHandler",
	  "$System",
	  "$SystemCharacterEncoding",
	  "$SystemCredentialStore",
	  "$SystemID",
	  "$SystemMemory",
	  "$SystemShell",
	  "$SystemTimeZone",
	  "$SystemWordLength",
	  "$TemplatePath",
	  "$TemporaryDirectory",
	  "$TemporaryPrefix",
	  "$TestFileName",
	  "$TextStyle",
	  "$TimedOut",
	  "$TimeUnit",
	  "$TimeZone",
	  "$TimeZoneEntity",
	  "$TopDirectory",
	  "$TraceOff",
	  "$TraceOn",
	  "$TracePattern",
	  "$TracePostAction",
	  "$TracePreAction",
	  "$UnitSystem",
	  "$Urgent",
	  "$UserAddOnsDirectory",
	  "$UserAgentLanguages",
	  "$UserAgentMachine",
	  "$UserAgentName",
	  "$UserAgentOperatingSystem",
	  "$UserAgentString",
	  "$UserAgentVersion",
	  "$UserBaseDirectory",
	  "$UserBasePacletsDirectory",
	  "$UserDocumentsDirectory",
	  "$Username",
	  "$UserName",
	  "$UserURLBase",
	  "$Version",
	  "$VersionNumber",
	  "$VideoDecoders",
	  "$VideoEncoders",
	  "$VoiceStyles",
	  "$WolframDocumentsDirectory",
	  "$WolframID",
	  "$WolframUUID"
	];

	/**
	 * @param {string} value
	 * @returns {RegExp}
	 * */

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function source(re) {
	  if (!re) return null;
	  if (typeof re === "string") return re;

	  return re.source;
	}

	/**
	 * @param {RegExp | string } re
	 * @returns {string}
	 */
	function optional(re) {
	  return concat('(', re, ')?');
	}

	/**
	 * @param {...(RegExp | string) } args
	 * @returns {string}
	 */
	function concat(...args) {
	  const joined = args.map((x) => source(x)).join("");
	  return joined;
	}

	/**
	 * Any of the passed expresssions may match
	 *
	 * Creates a huge this | this | that | that match
	 * @param {(RegExp | string)[] } args
	 * @returns {string}
	 */
	function either(...args) {
	  const joined = '(' + args.map((x) => source(x)).join("|") + ")";
	  return joined;
	}

	/*
	Language: Wolfram Language
	Description: The Wolfram Language is the programming language used in Wolfram Mathematica, a modern technical computing system spanning most areas of technical computing.
	Authors: Patrick Scheibe <patrick@halirutan.de>, Robert Jacobson <robertjacobson@acm.org>
	Website: https://www.wolfram.com/mathematica/
	Category: scientific
	*/

	/** @type LanguageFn */
	function mathematica(hljs) {
	  /*
	  This rather scary looking matching of Mathematica numbers is carefully explained by Robert Jacobson here:
	  https://wltools.github.io/LanguageSpec/Specification/Syntax/Number-representations/
	   */
	  const BASE_RE = /([2-9]|[1-2]\d|[3][0-5])\^\^/;
	  const BASE_DIGITS_RE = /(\w*\.\w+|\w+\.\w*|\w+)/;
	  const NUMBER_RE = /(\d*\.\d+|\d+\.\d*|\d+)/;
	  const BASE_NUMBER_RE = either(concat(BASE_RE, BASE_DIGITS_RE), NUMBER_RE);

	  const ACCURACY_RE = /``[+-]?(\d*\.\d+|\d+\.\d*|\d+)/;
	  const PRECISION_RE = /`([+-]?(\d*\.\d+|\d+\.\d*|\d+))?/;
	  const APPROXIMATE_NUMBER_RE = either(ACCURACY_RE, PRECISION_RE);

	  const SCIENTIFIC_NOTATION_RE = /\*\^[+-]?\d+/;

	  const MATHEMATICA_NUMBER_RE = concat(
	    BASE_NUMBER_RE,
	    optional(APPROXIMATE_NUMBER_RE),
	    optional(SCIENTIFIC_NOTATION_RE)
	  );

	  const NUMBERS = {
	    className: 'number',
	    relevance: 0,
	    begin: MATHEMATICA_NUMBER_RE
	  };

	  const SYMBOL_RE = /[a-zA-Z$][a-zA-Z0-9$]*/;
	  const SYSTEM_SYMBOLS_SET = new Set(SYSTEM_SYMBOLS);
	  /** @type {Mode} */
	  const SYMBOLS = {
	    variants: [
	      {
	        className: 'builtin-symbol',
	        begin: SYMBOL_RE,
	        // for performance out of fear of regex.either(...Mathematica.SYSTEM_SYMBOLS)
	        "on:begin": (match, response) => {
	          if (!SYSTEM_SYMBOLS_SET.has(match[0])) response.ignoreMatch();
	        }
	      },
	      {
	        className: 'symbol',
	        relevance: 0,
	        begin: SYMBOL_RE
	      }
	    ]
	  };

	  const NAMED_CHARACTER = {
	    className: 'named-character',
	    begin: /\\\[[$a-zA-Z][$a-zA-Z0-9]+\]/
	  };

	  const OPERATORS = {
	    className: 'operator',
	    relevance: 0,
	    begin: /[+\-*/,;.:@~=><&|_`'^?!%]+/
	  };
	  const PATTERNS = {
	    className: 'pattern',
	    relevance: 0,
	    begin: /([a-zA-Z$][a-zA-Z0-9$]*)?_+([a-zA-Z$][a-zA-Z0-9$]*)?/
	  };

	  const SLOTS = {
	    className: 'slot',
	    relevance: 0,
	    begin: /#[a-zA-Z$][a-zA-Z0-9$]*|#+[0-9]?/
	  };

	  const BRACES = {
	    className: 'brace',
	    relevance: 0,
	    begin: /[[\](){}]/
	  };

	  const MESSAGES = {
	    className: 'message-name',
	    relevance: 0,
	    begin: concat("::", SYMBOL_RE)
	  };

	  return {
	    name: 'Mathematica',
	    aliases: [
	      'mma',
	      'wl'
	    ],
	    classNameAliases: {
	      brace: 'punctuation',
	      pattern: 'type',
	      slot: 'type',
	      symbol: 'variable',
	      'named-character': 'variable',
	      'builtin-symbol': 'built_in',
	      'message-name': 'string'
	    },
	    contains: [
	      hljs.COMMENT(/\(\*/, /\*\)/, {
	        contains: [ 'self' ]
	      }),
	      PATTERNS,
	      SLOTS,
	      MESSAGES,
	      SYMBOLS,
	      NAMED_CHARACTER,
	      hljs.QUOTE_STRING_MODE,
	      NUMBERS,
	      OPERATORS,
	      BRACES
	    ]
	  };
	}

	mathematica_1 = mathematica;
	return mathematica_1;
}

/*
Language: Matlab
Author: Denis Bardadym <bardadymchik@gmail.com>
Contributors: Eugene Nizhibitsky <nizhibitsky@ya.ru>, Egor Rogov <e.rogov@postgrespro.ru>
Website: https://www.mathworks.com/products/matlab.html
Category: scientific
*/

var matlab_1;
var hasRequiredMatlab;

function requireMatlab () {
	if (hasRequiredMatlab) return matlab_1;
	hasRequiredMatlab = 1;
	/*
	  Formal syntax is not published, helpful link:
	  https://github.com/kornilova-l/matlab-IntelliJ-plugin/blob/master/src/main/grammar/Matlab.bnf
	*/
	function matlab(hljs) {

	  var TRANSPOSE_RE = '(\'|\\.\')+';
	  var TRANSPOSE = {
	    relevance: 0,
	    contains: [
	      { begin: TRANSPOSE_RE }
	    ]
	  };

	  return {
	    name: 'Matlab',
	    keywords: {
	      keyword:
	        'arguments break case catch classdef continue else elseif end enumeration events for function ' +
	        'global if methods otherwise parfor persistent properties return spmd switch try while',
	      built_in:
	        'sin sind sinh asin asind asinh cos cosd cosh acos acosd acosh tan tand tanh atan ' +
	        'atand atan2 atanh sec secd sech asec asecd asech csc cscd csch acsc acscd acsch cot ' +
	        'cotd coth acot acotd acoth hypot exp expm1 log log1p log10 log2 pow2 realpow reallog ' +
	        'realsqrt sqrt nthroot nextpow2 abs angle complex conj imag real unwrap isreal ' +
	        'cplxpair fix floor ceil round mod rem sign airy besselj bessely besselh besseli ' +
	        'besselk beta betainc betaln ellipj ellipke erf erfc erfcx erfinv expint gamma ' +
	        'gammainc gammaln psi legendre cross dot factor isprime primes gcd lcm rat rats perms ' +
	        'nchoosek factorial cart2sph cart2pol pol2cart sph2cart hsv2rgb rgb2hsv zeros ones ' +
	        'eye repmat rand randn linspace logspace freqspace meshgrid accumarray size length ' +
	        'ndims numel disp isempty isequal isequalwithequalnans cat reshape diag blkdiag tril ' +
	        'triu fliplr flipud flipdim rot90 find sub2ind ind2sub bsxfun ndgrid permute ipermute ' +
	        'shiftdim circshift squeeze isscalar isvector ans eps realmax realmin pi i|0 inf nan ' +
	        'isnan isinf isfinite j|0 why compan gallery hadamard hankel hilb invhilb magic pascal ' +
	        'rosser toeplitz vander wilkinson max min nanmax nanmin mean nanmean type table ' +
	        'readtable writetable sortrows sort figure plot plot3 scatter scatter3 cellfun ' +
	        'legend intersect ismember procrustes hold num2cell '
	    },
	    illegal: '(//|"|#|/\\*|\\s+/\\w+)',
	    contains: [
	      {
	        className: 'function',
	        beginKeywords: 'function', end: '$',
	        contains: [
	          hljs.UNDERSCORE_TITLE_MODE,
	          {
	            className: 'params',
	            variants: [
	              {begin: '\\(', end: '\\)'},
	              {begin: '\\[', end: '\\]'}
	            ]
	          }
	        ]
	      },
	      {
	        className: 'built_in',
	        begin: /true|false/,
	        relevance: 0,
	        starts: TRANSPOSE
	      },
	      {
	        begin: '[a-zA-Z][a-zA-Z_0-9]*' + TRANSPOSE_RE,
	        relevance: 0
	      },
	      {
	        className: 'number',
	        begin: hljs.C_NUMBER_RE,
	        relevance: 0,
	        starts: TRANSPOSE
	      },
	      {
	        className: 'string',
	        begin: '\'', end: '\'',
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          {begin: '\'\''}]
	      },
	      {
	        begin: /\]|\}|\)/,
	        relevance: 0,
	        starts: TRANSPOSE
	      },
	      {
	        className: 'string',
	        begin: '"', end: '"',
	        contains: [
	          hljs.BACKSLASH_ESCAPE,
	          {begin: '""'}
	        ],
	        starts: TRANSPOSE
	      },
	      hljs.COMMENT('^\\s*%\\{\\s*$', '^\\s*%\\}\\s*$'),
	      hljs.COMMENT('%', '$')
	    ]
	  };
	}

	matlab_1 = matlab;
	return matlab_1;
}

/*
Language: Maxima
Author: Robert Dodier <robert.dodier@gmail.com>
Website: http://maxima.sourceforge.net
Category: scientific
*/

var maxima_1;
var hasRequiredMaxima;

function requireMaxima () {
	if (hasRequiredMaxima) return maxima_1;
	hasRequiredMaxima = 1;
	function maxima(hljs) {
	  const KEYWORDS =
	    'if then else elseif for thru do while unless step in and or not';
	  const LITERALS =
	    'true false unknown inf minf ind und %e %i %pi %phi %gamma';
	  const BUILTIN_FUNCTIONS =
	    ' abasep abs absint absolute_real_time acos acosh acot acoth acsc acsch activate' +
	    ' addcol add_edge add_edges addmatrices addrow add_vertex add_vertices adjacency_matrix' +
	    ' adjoin adjoint af agd airy airy_ai airy_bi airy_dai airy_dbi algsys alg_type' +
	    ' alias allroots alphacharp alphanumericp amortization %and annuity_fv' +
	    ' annuity_pv antid antidiff AntiDifference append appendfile apply apply1 apply2' +
	    ' applyb1 apropos args arit_amortization arithmetic arithsum array arrayapply' +
	    ' arrayinfo arraymake arraysetapply ascii asec asech asin asinh askinteger' +
	    ' asksign assoc assoc_legendre_p assoc_legendre_q assume assume_external_byte_order' +
	    ' asympa at atan atan2 atanh atensimp atom atvalue augcoefmatrix augmented_lagrangian_method' +
	    ' av average_degree backtrace bars barsplot barsplot_description base64 base64_decode' +
	    ' bashindices batch batchload bc2 bdvac belln benefit_cost bern bernpoly bernstein_approx' +
	    ' bernstein_expand bernstein_poly bessel bessel_i bessel_j bessel_k bessel_simplify' +
	    ' bessel_y beta beta_incomplete beta_incomplete_generalized beta_incomplete_regularized' +
	    ' bezout bfallroots bffac bf_find_root bf_fmin_cobyla bfhzeta bfloat bfloatp' +
	    ' bfpsi bfpsi0 bfzeta biconnected_components bimetric binomial bipartition' +
	    ' block blockmatrixp bode_gain bode_phase bothcoef box boxplot boxplot_description' +
	    ' break bug_report build_info|10 buildq build_sample burn cabs canform canten' +
	    ' cardinality carg cartan cartesian_product catch cauchy_matrix cbffac cdf_bernoulli' +
	    ' cdf_beta cdf_binomial cdf_cauchy cdf_chi2 cdf_continuous_uniform cdf_discrete_uniform' +
	    ' cdf_exp cdf_f cdf_gamma cdf_general_finite_discrete cdf_geometric cdf_gumbel' +
	    ' cdf_hypergeometric cdf_laplace cdf_logistic cdf_lognormal cdf_negative_binomial' +
	    ' cdf_noncentral_chi2 cdf_noncentral_student_t cdf_normal cdf_pareto cdf_poisson' +
	    ' cdf_rank_sum cdf_rayleigh cdf_signed_rank cdf_student_t cdf_weibull cdisplay' +
	    ' ceiling central_moment cequal cequalignore cf cfdisrep cfexpand cgeodesic' +
	    ' cgreaterp cgreaterpignore changename changevar chaosgame charat charfun charfun2' +
	    ' charlist charp charpoly chdir chebyshev_t chebyshev_u checkdiv check_overlaps' +
	    ' chinese cholesky christof chromatic_index chromatic_number cint circulant_graph' +
	    ' clear_edge_weight clear_rules clear_vertex_label clebsch_gordan clebsch_graph' +
	    ' clessp clesspignore close closefile cmetric coeff coefmatrix cograd col collapse' +
	    ' collectterms columnop columnspace columnswap columnvector combination combine' +
	    ' comp2pui compare compfile compile compile_file complement_graph complete_bipartite_graph' +
	    ' complete_graph complex_number_p components compose_functions concan concat' +
	    ' conjugate conmetderiv connected_components connect_vertices cons constant' +
	    ' constantp constituent constvalue cont2part content continuous_freq contortion' +
	    ' contour_plot contract contract_edge contragrad contrib_ode convert coord' +
	    ' copy copy_file copy_graph copylist copymatrix cor cos cosh cot coth cov cov1' +
	    ' covdiff covect covers crc24sum create_graph create_list csc csch csetup cspline' +
	    ' ctaylor ct_coordsys ctransform ctranspose cube_graph cuboctahedron_graph' +
	    ' cunlisp cv cycle_digraph cycle_graph cylindrical days360 dblint deactivate' +
	    ' declare declare_constvalue declare_dimensions declare_fundamental_dimensions' +
	    ' declare_fundamental_units declare_qty declare_translated declare_unit_conversion' +
	    ' declare_units declare_weights decsym defcon define define_alt_display define_variable' +
	    ' defint defmatch defrule defstruct deftaylor degree_sequence del delete deleten' +
	    ' delta demo demoivre denom depends derivdegree derivlist describe desolve' +
	    ' determinant dfloat dgauss_a dgauss_b dgeev dgemm dgeqrf dgesv dgesvd diag' +
	    ' diagmatrix diag_matrix diagmatrixp diameter diff digitcharp dimacs_export' +
	    ' dimacs_import dimension dimensionless dimensions dimensions_as_list direct' +
	    ' directory discrete_freq disjoin disjointp disolate disp dispcon dispform' +
	    ' dispfun dispJordan display disprule dispterms distrib divide divisors divsum' +
	    ' dkummer_m dkummer_u dlange dodecahedron_graph dotproduct dotsimp dpart' +
	    ' draw draw2d draw3d drawdf draw_file draw_graph dscalar echelon edge_coloring' +
	    ' edge_connectivity edges eigens_by_jacobi eigenvalues eigenvectors eighth' +
	    ' einstein eivals eivects elapsed_real_time elapsed_run_time ele2comp ele2polynome' +
	    ' ele2pui elem elementp elevation_grid elim elim_allbut eliminate eliminate_using' +
	    ' ellipse elliptic_e elliptic_ec elliptic_eu elliptic_f elliptic_kc elliptic_pi' +
	    ' ematrix empty_graph emptyp endcons entermatrix entertensor entier equal equalp' +
	    ' equiv_classes erf erfc erf_generalized erfi errcatch error errormsg errors' +
	    ' euler ev eval_string evenp every evolution evolution2d evundiff example exp' +
	    ' expand expandwrt expandwrt_factored expint expintegral_chi expintegral_ci' +
	    ' expintegral_e expintegral_e1 expintegral_ei expintegral_e_simplify expintegral_li' +
	    ' expintegral_shi expintegral_si explicit explose exponentialize express expt' +
	    ' exsec extdiff extract_linear_equations extremal_subset ezgcd %f f90 facsum' +
	    ' factcomb factor factorfacsum factorial factorout factorsum facts fast_central_elements' +
	    ' fast_linsolve fasttimes featurep fernfale fft fib fibtophi fifth filename_merge' +
	    ' file_search file_type fillarray findde find_root find_root_abs find_root_error' +
	    ' find_root_rel first fix flatten flength float floatnump floor flower_snark' +
	    ' flush flush1deriv flushd flushnd flush_output fmin_cobyla forget fortran' +
	    ' fourcos fourexpand fourier fourier_elim fourint fourintcos fourintsin foursimp' +
	    ' foursin fourth fposition frame_bracket freeof freshline fresnel_c fresnel_s' +
	    ' from_adjacency_matrix frucht_graph full_listify fullmap fullmapl fullratsimp' +
	    ' fullratsubst fullsetify funcsolve fundamental_dimensions fundamental_units' +
	    ' fundef funmake funp fv g0 g1 gamma gamma_greek gamma_incomplete gamma_incomplete_generalized' +
	    ' gamma_incomplete_regularized gauss gauss_a gauss_b gaussprob gcd gcdex gcdivide' +
	    ' gcfac gcfactor gd generalized_lambert_w genfact gen_laguerre genmatrix gensym' +
	    ' geo_amortization geo_annuity_fv geo_annuity_pv geomap geometric geometric_mean' +
	    ' geosum get getcurrentdirectory get_edge_weight getenv get_lu_factors get_output_stream_string' +
	    ' get_pixel get_plot_option get_tex_environment get_tex_environment_default' +
	    ' get_vertex_label gfactor gfactorsum ggf girth global_variances gn gnuplot_close' +
	    ' gnuplot_replot gnuplot_reset gnuplot_restart gnuplot_start go Gosper GosperSum' +
	    ' gr2d gr3d gradef gramschmidt graph6_decode graph6_encode graph6_export graph6_import' +
	    ' graph_center graph_charpoly graph_eigenvalues graph_flow graph_order graph_periphery' +
	    ' graph_product graph_size graph_union great_rhombicosidodecahedron_graph great_rhombicuboctahedron_graph' +
	    ' grid_graph grind grobner_basis grotzch_graph hamilton_cycle hamilton_path' +
	    ' hankel hankel_1 hankel_2 harmonic harmonic_mean hav heawood_graph hermite' +
	    ' hessian hgfred hilbertmap hilbert_matrix hipow histogram histogram_description' +
	    ' hodge horner hypergeometric i0 i1 %ibes ic1 ic2 ic_convert ichr1 ichr2 icosahedron_graph' +
	    ' icosidodecahedron_graph icurvature ident identfor identity idiff idim idummy' +
	    ' ieqn %if ifactors iframes ifs igcdex igeodesic_coords ilt image imagpart' +
	    ' imetric implicit implicit_derivative implicit_plot indexed_tensor indices' +
	    ' induced_subgraph inferencep inference_result infix info_display init_atensor' +
	    ' init_ctensor in_neighbors innerproduct inpart inprod inrt integerp integer_partitions' +
	    ' integrate intersect intersection intervalp intopois intosum invariant1 invariant2' +
	    ' inverse_fft inverse_jacobi_cd inverse_jacobi_cn inverse_jacobi_cs inverse_jacobi_dc' +
	    ' inverse_jacobi_dn inverse_jacobi_ds inverse_jacobi_nc inverse_jacobi_nd inverse_jacobi_ns' +
	    ' inverse_jacobi_sc inverse_jacobi_sd inverse_jacobi_sn invert invert_by_adjoint' +
	    ' invert_by_lu inv_mod irr is is_biconnected is_bipartite is_connected is_digraph' +
	    ' is_edge_in_graph is_graph is_graph_or_digraph ishow is_isomorphic isolate' +
	    ' isomorphism is_planar isqrt isreal_p is_sconnected is_tree is_vertex_in_graph' +
	    ' items_inference %j j0 j1 jacobi jacobian jacobi_cd jacobi_cn jacobi_cs jacobi_dc' +
	    ' jacobi_dn jacobi_ds jacobi_nc jacobi_nd jacobi_ns jacobi_p jacobi_sc jacobi_sd' +
	    ' jacobi_sn JF jn join jordan julia julia_set julia_sin %k kdels kdelta kill' +
	    ' killcontext kostka kron_delta kronecker_product kummer_m kummer_u kurtosis' +
	    ' kurtosis_bernoulli kurtosis_beta kurtosis_binomial kurtosis_chi2 kurtosis_continuous_uniform' +
	    ' kurtosis_discrete_uniform kurtosis_exp kurtosis_f kurtosis_gamma kurtosis_general_finite_discrete' +
	    ' kurtosis_geometric kurtosis_gumbel kurtosis_hypergeometric kurtosis_laplace' +
	    ' kurtosis_logistic kurtosis_lognormal kurtosis_negative_binomial kurtosis_noncentral_chi2' +
	    ' kurtosis_noncentral_student_t kurtosis_normal kurtosis_pareto kurtosis_poisson' +
	    ' kurtosis_rayleigh kurtosis_student_t kurtosis_weibull label labels lagrange' +
	    ' laguerre lambda lambert_w laplace laplacian_matrix last lbfgs lc2kdt lcharp' +
	    ' lc_l lcm lc_u ldefint ldisp ldisplay legendre_p legendre_q leinstein length' +
	    ' let letrules letsimp levi_civita lfreeof lgtreillis lhs li liediff limit' +
	    ' Lindstedt linear linearinterpol linear_program linear_regression line_graph' +
	    ' linsolve listarray list_correlations listify list_matrix_entries list_nc_monomials' +
	    ' listoftens listofvars listp lmax lmin load loadfile local locate_matrix_entry' +
	    ' log logcontract log_gamma lopow lorentz_gauge lowercasep lpart lratsubst' +
	    ' lreduce lriemann lsquares_estimates lsquares_estimates_approximate lsquares_estimates_exact' +
	    ' lsquares_mse lsquares_residual_mse lsquares_residuals lsum ltreillis lu_backsub' +
	    ' lucas lu_factor %m macroexpand macroexpand1 make_array makebox makefact makegamma' +
	    ' make_graph make_level_picture makelist makeOrders make_poly_continent make_poly_country' +
	    ' make_polygon make_random_state make_rgb_picture makeset make_string_input_stream' +
	    ' make_string_output_stream make_transform mandelbrot mandelbrot_set map mapatom' +
	    ' maplist matchdeclare matchfix mat_cond mat_fullunblocker mat_function mathml_display' +
	    ' mat_norm matrix matrixmap matrixp matrix_size mattrace mat_trace mat_unblocker' +
	    ' max max_clique max_degree max_flow maximize_lp max_independent_set max_matching' +
	    ' maybe md5sum mean mean_bernoulli mean_beta mean_binomial mean_chi2 mean_continuous_uniform' +
	    ' mean_deviation mean_discrete_uniform mean_exp mean_f mean_gamma mean_general_finite_discrete' +
	    ' mean_geometric mean_gumbel mean_hypergeometric mean_laplace mean_logistic' +
	    ' mean_lognormal mean_negative_binomial mean_noncentral_chi2 mean_noncentral_student_t' +
	    ' mean_normal mean_pareto mean_poisson mean_rayleigh mean_student_t mean_weibull' +
	    ' median median_deviation member mesh metricexpandall mgf1_sha1 min min_degree' +
	    ' min_edge_cut minfactorial minimalPoly minimize_lp minimum_spanning_tree minor' +
	    ' minpack_lsquares minpack_solve min_vertex_cover min_vertex_cut mkdir mnewton' +
	    ' mod mode_declare mode_identity ModeMatrix moebius mon2schur mono monomial_dimensions' +
	    ' multibernstein_poly multi_display_for_texinfo multi_elem multinomial multinomial_coeff' +
	    ' multi_orbit multiplot_mode multi_pui multsym multthru mycielski_graph nary' +
	    ' natural_unit nc_degree ncexpt ncharpoly negative_picture neighbors new newcontext' +
	    ' newdet new_graph newline newton new_variable next_prime nicedummies niceindices' +
	    ' ninth nofix nonarray noncentral_moment nonmetricity nonnegintegerp nonscalarp' +
	    ' nonzeroandfreeof notequal nounify nptetrad npv nroots nterms ntermst' +
	    ' nthroot nullity nullspace num numbered_boundaries numberp number_to_octets' +
	    ' num_distinct_partitions numerval numfactor num_partitions nusum nzeta nzetai' +
	    ' nzetar octets_to_number octets_to_oid odd_girth oddp ode2 ode_check odelin' +
	    ' oid_to_octets op opena opena_binary openr openr_binary openw openw_binary' +
	    ' operatorp opsubst optimize %or orbit orbits ordergreat ordergreatp orderless' +
	    ' orderlessp orthogonal_complement orthopoly_recur orthopoly_weight outermap' +
	    ' out_neighbors outofpois pade parabolic_cylinder_d parametric parametric_surface' +
	    ' parg parGosper parse_string parse_timedate part part2cont partfrac partition' +
	    ' partition_set partpol path_digraph path_graph pathname_directory pathname_name' +
	    ' pathname_type pdf_bernoulli pdf_beta pdf_binomial pdf_cauchy pdf_chi2 pdf_continuous_uniform' +
	    ' pdf_discrete_uniform pdf_exp pdf_f pdf_gamma pdf_general_finite_discrete' +
	    ' pdf_geometric pdf_gumbel pdf_hypergeometric pdf_laplace pdf_logistic pdf_lognormal' +
	    ' pdf_negative_binomial pdf_noncentral_chi2 pdf_noncentral_student_t pdf_normal' +
	    ' pdf_pareto pdf_poisson pdf_rank_sum pdf_rayleigh pdf_signed_rank pdf_student_t' +
	    ' pdf_weibull pearson_skewness permanent permut permutation permutations petersen_graph' +
	    ' petrov pickapart picture_equalp picturep piechart piechart_description planar_embedding' +
	    ' playback plog plot2d plot3d plotdf ploteq plsquares pochhammer points poisdiff' +
	    ' poisexpt poisint poismap poisplus poissimp poissubst poistimes poistrim polar' +
	    ' polarform polartorect polar_to_xy poly_add poly_buchberger poly_buchberger_criterion' +
	    ' poly_colon_ideal poly_content polydecomp poly_depends_p poly_elimination_ideal' +
	    ' poly_exact_divide poly_expand poly_expt poly_gcd polygon poly_grobner poly_grobner_equal' +
	    ' poly_grobner_member poly_grobner_subsetp poly_ideal_intersection poly_ideal_polysaturation' +
	    ' poly_ideal_polysaturation1 poly_ideal_saturation poly_ideal_saturation1 poly_lcm' +
	    ' poly_minimization polymod poly_multiply polynome2ele polynomialp poly_normal_form' +
	    ' poly_normalize poly_normalize_list poly_polysaturation_extension poly_primitive_part' +
	    ' poly_pseudo_divide poly_reduced_grobner poly_reduction poly_saturation_extension' +
	    ' poly_s_polynomial poly_subtract polytocompanion pop postfix potential power_mod' +
	    ' powerseries powerset prefix prev_prime primep primes principal_components' +
	    ' print printf printfile print_graph printpois printprops prodrac product properties' +
	    ' propvars psi psubst ptriangularize pui pui2comp pui2ele pui2polynome pui_direct' +
	    ' puireduc push put pv qput qrange qty quad_control quad_qag quad_qagi quad_qagp' +
	    ' quad_qags quad_qawc quad_qawf quad_qawo quad_qaws quadrilateral quantile' +
	    ' quantile_bernoulli quantile_beta quantile_binomial quantile_cauchy quantile_chi2' +
	    ' quantile_continuous_uniform quantile_discrete_uniform quantile_exp quantile_f' +
	    ' quantile_gamma quantile_general_finite_discrete quantile_geometric quantile_gumbel' +
	    ' quantile_hypergeometric quantile_laplace quantile_logistic quantile_lognormal' +
	    ' quantile_negative_binomial quantile_noncentral_chi2 quantile_noncentral_student_t' +
	    ' quantile_normal quantile_pareto quantile_poisson quantile_rayleigh quantile_student_t' +
	    ' quantile_weibull quartile_skewness quit qunit quotient racah_v racah_w radcan' +
	    ' radius random random_bernoulli random_beta random_binomial random_bipartite_graph' +
	    ' random_cauchy random_chi2 random_continuous_uniform random_digraph random_discrete_uniform' +
	    ' random_exp random_f random_gamma random_general_finite_discrete random_geometric' +
	    ' random_graph random_graph1 random_gumbel random_hypergeometric random_laplace' +
	    ' random_logistic random_lognormal random_negative_binomial random_network' +
	    ' random_noncentral_chi2 random_noncentral_student_t random_normal random_pareto' +
	    ' random_permutation random_poisson random_rayleigh random_regular_graph random_student_t' +
	    ' random_tournament random_tree random_weibull range rank rat ratcoef ratdenom' +
	    ' ratdiff ratdisrep ratexpand ratinterpol rational rationalize ratnumer ratnump' +
	    ' ratp ratsimp ratsubst ratvars ratweight read read_array read_binary_array' +
	    ' read_binary_list read_binary_matrix readbyte readchar read_hashed_array readline' +
	    ' read_list read_matrix read_nested_list readonly read_xpm real_imagpart_to_conjugate' +
	    ' realpart realroots rearray rectangle rectform rectform_log_if_constant recttopolar' +
	    ' rediff reduce_consts reduce_order region region_boundaries region_boundaries_plus' +
	    ' rem remainder remarray rembox remcomps remcon remcoord remfun remfunction' +
	    ' remlet remove remove_constvalue remove_dimensions remove_edge remove_fundamental_dimensions' +
	    ' remove_fundamental_units remove_plot_option remove_vertex rempart remrule' +
	    ' remsym remvalue rename rename_file reset reset_displays residue resolvante' +
	    ' resolvante_alternee1 resolvante_bipartite resolvante_diedrale resolvante_klein' +
	    ' resolvante_klein3 resolvante_produit_sym resolvante_unitaire resolvante_vierer' +
	    ' rest resultant return reveal reverse revert revert2 rgb2level rhs ricci riemann' +
	    ' rinvariant risch rk rmdir rncombine romberg room rootscontract round row' +
	    ' rowop rowswap rreduce run_testsuite %s save saving scalarp scaled_bessel_i' +
	    ' scaled_bessel_i0 scaled_bessel_i1 scalefactors scanmap scatterplot scatterplot_description' +
	    ' scene schur2comp sconcat scopy scsimp scurvature sdowncase sec sech second' +
	    ' sequal sequalignore set_alt_display setdifference set_draw_defaults set_edge_weight' +
	    ' setelmx setequalp setify setp set_partitions set_plot_option set_prompt set_random_state' +
	    ' set_tex_environment set_tex_environment_default setunits setup_autoload set_up_dot_simplifications' +
	    ' set_vertex_label seventh sexplode sf sha1sum sha256sum shortest_path shortest_weighted_path' +
	    ' show showcomps showratvars sierpinskiale sierpinskimap sign signum similaritytransform' +
	    ' simp_inequality simplify_sum simplode simpmetderiv simtran sin sinh sinsert' +
	    ' sinvertcase sixth skewness skewness_bernoulli skewness_beta skewness_binomial' +
	    ' skewness_chi2 skewness_continuous_uniform skewness_discrete_uniform skewness_exp' +
	    ' skewness_f skewness_gamma skewness_general_finite_discrete skewness_geometric' +
	    ' skewness_gumbel skewness_hypergeometric skewness_laplace skewness_logistic' +
	    ' skewness_lognormal skewness_negative_binomial skewness_noncentral_chi2 skewness_noncentral_student_t' +
	    ' skewness_normal skewness_pareto skewness_poisson skewness_rayleigh skewness_student_t' +
	    ' skewness_weibull slength smake small_rhombicosidodecahedron_graph small_rhombicuboctahedron_graph' +
	    ' smax smin smismatch snowmap snub_cube_graph snub_dodecahedron_graph solve' +
	    ' solve_rec solve_rec_rat some somrac sort sparse6_decode sparse6_encode sparse6_export' +
	    ' sparse6_import specint spherical spherical_bessel_j spherical_bessel_y spherical_hankel1' +
	    ' spherical_hankel2 spherical_harmonic spherical_to_xyz splice split sposition' +
	    ' sprint sqfr sqrt sqrtdenest sremove sremovefirst sreverse ssearch ssort sstatus' +
	    ' ssubst ssubstfirst staircase standardize standardize_inverse_trig starplot' +
	    ' starplot_description status std std1 std_bernoulli std_beta std_binomial' +
	    ' std_chi2 std_continuous_uniform std_discrete_uniform std_exp std_f std_gamma' +
	    ' std_general_finite_discrete std_geometric std_gumbel std_hypergeometric std_laplace' +
	    ' std_logistic std_lognormal std_negative_binomial std_noncentral_chi2 std_noncentral_student_t' +
	    ' std_normal std_pareto std_poisson std_rayleigh std_student_t std_weibull' +
	    ' stemplot stirling stirling1 stirling2 strim striml strimr string stringout' +
	    ' stringp strong_components struve_h struve_l sublis sublist sublist_indices' +
	    ' submatrix subsample subset subsetp subst substinpart subst_parallel substpart' +
	    ' substring subvar subvarp sum sumcontract summand_to_rec supcase supcontext' +
	    ' symbolp symmdifference symmetricp system take_channel take_inference tan' +
	    ' tanh taylor taylorinfo taylorp taylor_simplifier taytorat tcl_output tcontract' +
	    ' tellrat tellsimp tellsimpafter tentex tenth test_mean test_means_difference' +
	    ' test_normality test_proportion test_proportions_difference test_rank_sum' +
	    ' test_sign test_signed_rank test_variance test_variance_ratio tex tex1 tex_display' +
	    ' texput %th third throw time timedate timer timer_info tldefint tlimit todd_coxeter' +
	    ' toeplitz tokens to_lisp topological_sort to_poly to_poly_solve totaldisrep' +
	    ' totalfourier totient tpartpol trace tracematrix trace_options transform_sample' +
	    ' translate translate_file transpose treefale tree_reduce treillis treinat' +
	    ' triangle triangularize trigexpand trigrat trigreduce trigsimp trunc truncate' +
	    ' truncated_cube_graph truncated_dodecahedron_graph truncated_icosahedron_graph' +
	    ' truncated_tetrahedron_graph tr_warnings_get tube tutte_graph ueivects uforget' +
	    ' ultraspherical underlying_graph undiff union unique uniteigenvectors unitp' +
	    ' units unit_step unitvector unorder unsum untellrat untimer' +
	    ' untrace uppercasep uricci uriemann uvect vandermonde_matrix var var1 var_bernoulli' +
	    ' var_beta var_binomial var_chi2 var_continuous_uniform var_discrete_uniform' +
	    ' var_exp var_f var_gamma var_general_finite_discrete var_geometric var_gumbel' +
	    ' var_hypergeometric var_laplace var_logistic var_lognormal var_negative_binomial' +
	    ' var_noncentral_chi2 var_noncentral_student_t var_normal var_pareto var_poisson' +
	    ' var_rayleigh var_student_t var_weibull vector vectorpotential vectorsimp' +
	    ' verbify vers vertex_coloring vertex_connectivity vertex_degree vertex_distance' +
	    ' vertex_eccentricity vertex_in_degree vertex_out_degree vertices vertices_to_cycle' +
	    ' vertices_to_path %w weyl wheel_graph wiener_index wigner_3j wigner_6j' +
	    ' wigner_9j with_stdout write_binary_data writebyte write_data writefile wronskian' +
	    ' xreduce xthru %y Zeilberger zeroequiv zerofor zeromatrix zeromatrixp zeta' +
	    ' zgeev zheev zlange zn_add_table zn_carmichael_lambda zn_characteristic_factors' +
	    ' zn_determinant zn_factor_generators zn_invert_by_lu zn_log zn_mult_table' +
	    ' absboxchar activecontexts adapt_depth additive adim aform algebraic' +
	    ' algepsilon algexact aliases allbut all_dotsimp_denoms allocation allsym alphabetic' +
	    ' animation antisymmetric arrays askexp assume_pos assume_pos_pred assumescalar' +
	    ' asymbol atomgrad atrig1 axes axis_3d axis_bottom axis_left axis_right axis_top' +
	    ' azimuth background background_color backsubst berlefact bernstein_explicit' +
	    ' besselexpand beta_args_sum_to_integer beta_expand bftorat bftrunc bindtest' +
	    ' border boundaries_array box boxchar breakup %c capping cauchysum cbrange' +
	    ' cbtics center cflength cframe_flag cnonmet_flag color color_bar color_bar_tics' +
	    ' colorbox columns commutative complex cone context contexts contour contour_levels' +
	    ' cosnpiflag ctaypov ctaypt ctayswitch ctayvar ct_coords ctorsion_flag ctrgsimp' +
	    ' cube current_let_rule_package cylinder data_file_name debugmode decreasing' +
	    ' default_let_rule_package delay dependencies derivabbrev derivsubst detout' +
	    ' diagmetric diff dim dimensions dispflag display2d|10 display_format_internal' +
	    ' distribute_over doallmxops domain domxexpt domxmxops domxnctimes dontfactor' +
	    ' doscmxops doscmxplus dot0nscsimp dot0simp dot1simp dotassoc dotconstrules' +
	    ' dotdistrib dotexptsimp dotident dotscrules draw_graph_program draw_realpart' +
	    ' edge_color edge_coloring edge_partition edge_type edge_width %edispflag' +
	    ' elevation %emode endphi endtheta engineering_format_floats enhanced3d %enumer' +
	    ' epsilon_lp erfflag erf_representation errormsg error_size error_syms error_type' +
	    ' %e_to_numlog eval even evenfun evflag evfun ev_point expandwrt_denom expintexpand' +
	    ' expintrep expon expop exptdispflag exptisolate exptsubst facexpand facsum_combine' +
	    ' factlim factorflag factorial_expand factors_only fb feature features' +
	    ' file_name file_output_append file_search_demo file_search_lisp file_search_maxima|10' +
	    ' file_search_tests file_search_usage file_type_lisp file_type_maxima|10 fill_color' +
	    ' fill_density filled_func fixed_vertices flipflag float2bf font font_size' +
	    ' fortindent fortspaces fpprec fpprintprec functions gamma_expand gammalim' +
	    ' gdet genindex gensumnum GGFCFMAX GGFINFINITY globalsolve gnuplot_command' +
	    ' gnuplot_curve_styles gnuplot_curve_titles gnuplot_default_term_command gnuplot_dumb_term_command' +
	    ' gnuplot_file_args gnuplot_file_name gnuplot_out_file gnuplot_pdf_term_command' +
	    ' gnuplot_pm3d gnuplot_png_term_command gnuplot_postamble gnuplot_preamble' +
	    ' gnuplot_ps_term_command gnuplot_svg_term_command gnuplot_term gnuplot_view_args' +
	    ' Gosper_in_Zeilberger gradefs grid grid2d grind halfangles head_angle head_both' +
	    ' head_length head_type height hypergeometric_representation %iargs ibase' +
	    ' icc1 icc2 icounter idummyx ieqnprint ifb ifc1 ifc2 ifg ifgi ifr iframe_bracket_form' +
	    ' ifri igeowedge_flag ikt1 ikt2 imaginary inchar increasing infeval' +
	    ' infinity inflag infolists inm inmc1 inmc2 intanalysis integer integervalued' +
	    ' integrate_use_rootsof integration_constant integration_constant_counter interpolate_color' +
	    ' intfaclim ip_grid ip_grid_in irrational isolate_wrt_times iterations itr' +
	    ' julia_parameter %k1 %k2 keepfloat key key_pos kinvariant kt label label_alignment' +
	    ' label_orientation labels lassociative lbfgs_ncorrections lbfgs_nfeval_max' +
	    ' leftjust legend letrat let_rule_packages lfg lg lhospitallim limsubst linear' +
	    ' linear_solver linechar linel|10 linenum line_type linewidth line_width linsolve_params' +
	    ' linsolvewarn lispdisp listarith listconstvars listdummyvars lmxchar load_pathname' +
	    ' loadprint logabs logarc logcb logconcoeffp logexpand lognegint logsimp logx' +
	    ' logx_secondary logy logy_secondary logz lriem m1pbranch macroexpansion macros' +
	    ' mainvar manual_demo maperror mapprint matrix_element_add matrix_element_mult' +
	    ' matrix_element_transpose maxapplydepth maxapplyheight maxima_tempdir|10 maxima_userdir|10' +
	    ' maxnegex MAX_ORD maxposex maxpsifracdenom maxpsifracnum maxpsinegint maxpsiposint' +
	    ' maxtayorder mesh_lines_color method mod_big_prime mode_check_errorp' +
	    ' mode_checkp mode_check_warnp mod_test mod_threshold modular_linear_solver' +
	    ' modulus multiplicative multiplicities myoptions nary negdistrib negsumdispflag' +
	    ' newline newtonepsilon newtonmaxiter nextlayerfactor niceindicespref nm nmc' +
	    ' noeval nolabels nonegative_lp noninteger nonscalar noun noundisp nouns np' +
	    ' npi nticks ntrig numer numer_pbranch obase odd oddfun opacity opproperties' +
	    ' opsubst optimprefix optionset orientation origin orthopoly_returns_intervals' +
	    ' outative outchar packagefile palette partswitch pdf_file pfeformat phiresolution' +
	    ' %piargs piece pivot_count_sx pivot_max_sx plot_format plot_options plot_realpart' +
	    ' png_file pochhammer_max_index points pointsize point_size points_joined point_type' +
	    ' poislim poisson poly_coefficient_ring poly_elimination_order polyfactor poly_grobner_algorithm' +
	    ' poly_grobner_debug poly_monomial_order poly_primary_elimination_order poly_return_term_list' +
	    ' poly_secondary_elimination_order poly_top_reduction_only posfun position' +
	    ' powerdisp pred prederror primep_number_of_tests product_use_gamma program' +
	    ' programmode promote_float_to_bigfloat prompt proportional_axes props psexpand' +
	    ' ps_file radexpand radius radsubstflag rassociative ratalgdenom ratchristof' +
	    ' ratdenomdivide rateinstein ratepsilon ratfac rational ratmx ratprint ratriemann' +
	    ' ratsimpexpons ratvarswitch ratweights ratweyl ratwtlvl real realonly redraw' +
	    ' refcheck resolution restart resultant ric riem rmxchar %rnum_list rombergabs' +
	    ' rombergit rombergmin rombergtol rootsconmode rootsepsilon run_viewer same_xy' +
	    ' same_xyz savedef savefactors scalar scalarmatrixp scale scale_lp setcheck' +
	    ' setcheckbreak setval show_edge_color show_edges show_edge_type show_edge_width' +
	    ' show_id show_label showtime show_vertex_color show_vertex_size show_vertex_type' +
	    ' show_vertices show_weight simp simplified_output simplify_products simpproduct' +
	    ' simpsum sinnpiflag solvedecomposes solveexplicit solvefactors solvenullwarn' +
	    ' solveradcan solvetrigwarn space sparse sphere spring_embedding_depth sqrtdispflag' +
	    ' stardisp startphi starttheta stats_numer stringdisp structures style sublis_apply_lambda' +
	    ' subnumsimp sumexpand sumsplitfact surface surface_hide svg_file symmetric' +
	    ' tab taylordepth taylor_logexpand taylor_order_coefficients taylor_truncate_polynomials' +
	    ' tensorkill terminal testsuite_files thetaresolution timer_devalue title tlimswitch' +
	    ' tr track transcompile transform transform_xy translate_fast_arrays transparent' +
	    ' transrun tr_array_as_ref tr_bound_function_applyp tr_file_tty_messagesp tr_float_can_branch_complex' +
	    ' tr_function_call_default trigexpandplus trigexpandtimes triginverses trigsign' +
	    ' trivial_solutions tr_numer tr_optimize_max_loop tr_semicompile tr_state_vars' +
	    ' tr_warn_bad_function_calls tr_warn_fexpr tr_warn_meval tr_warn_mode' +
	    ' tr_warn_undeclared tr_warn_undefined_variable tstep ttyoff tube_extremes' +
	    ' ufg ug %unitexpand unit_vectors uric uriem use_fast_arrays user_preamble' +
	    ' usersetunits values vect_cross verbose vertex_color vertex_coloring vertex_partition' +
	    ' vertex_size vertex_type view warnings weyl width windowname windowtitle wired_surface' +
	    ' wireframe xaxis xaxis_color xaxis_secondary xaxis_type xaxis_width xlabel' +
	    ' xlabel_secondary xlength xrange xrange_secondary xtics xtics_axis xtics_rotate' +
	    ' xtics_rotate_secondary xtics_secondary xtics_secondary_axis xu_grid x_voxel' +
	    ' xy_file xyplane xy_scale yaxis yaxis_color yaxis_secondary yaxis_type yaxis_width' +
	    ' ylabel ylabel_secondary ylength yrange yrange_secondary ytics ytics_axis' +
	    ' ytics_rotate ytics_rotate_secondary ytics_secondary ytics_secondary_axis' +
	    ' yv_grid y_voxel yx_ratio zaxis zaxis_color zaxis_type zaxis_width zeroa zerob' +
	    ' zerobern zeta%pi zlabel zlabel_rotate zlength zmin zn_primroot_limit zn_primroot_pretest';
	  const SYMBOLS = '_ __ %|0 %%|0';

	  return {
	    name: 'Maxima',
	    keywords: {
	      $pattern: '[A-Za-z_%][0-9A-Za-z_%]*',
	      keyword: KEYWORDS,
	      literal: LITERALS,
	      built_in: BUILTIN_FUNCTIONS,
	      symbol: SYMBOLS
	    },
	    contains: [
	      {
	        className: 'comment',
	        begin: '/\\*',
	        end: '\\*/',
	        contains: [ 'self' ]
	      },
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'number',
	        relevance: 0,
	        variants: [
	          {
	            // float number w/ exponent
	            // hmm, I wonder if we ought to include other exponent markers?
	            begin: '\\b(\\d+|\\d+\\.|\\.\\d+|\\d+\\.\\d+)[Ee][-+]?\\d+\\b'
	          },
	          {
	            // bigfloat number
	            begin: '\\b(\\d+|\\d+\\.|\\.\\d+|\\d+\\.\\d+)[Bb][-+]?\\d+\\b',
	            relevance: 10
	          },
	          {
	            // float number w/out exponent
	            // Doesn't seem to recognize floats which start with '.'
	            begin: '\\b(\\.\\d+|\\d+\\.\\d+)\\b'
	          },
	          {
	            // integer in base up to 36
	            // Doesn't seem to recognize integers which end with '.'
	            begin: '\\b(\\d+|0[0-9A-Za-z]+)\\.?\\b'
	          }
	        ]
	      }
	    ],
	    illegal: /@/
	  };
	}

	maxima_1 = maxima;
	return maxima_1;
}

/*
Language: MEL
Description: Maya Embedded Language
Author: Shuen-Huei Guan <drake.guan@gmail.com>
Website: http://www.autodesk.com/products/autodesk-maya/overview
Category: graphics
*/

var mel_1;
var hasRequiredMel;

function requireMel () {
	if (hasRequiredMel) return mel_1;
	hasRequiredMel = 1;
	function mel(hljs) {
	  return {
	    name: 'MEL',
	    keywords:
	      'int float string vector matrix if else switch case default while do for in break ' +
	      'continue global proc return about abs addAttr addAttributeEditorNodeHelp addDynamic ' +
	      'addNewShelfTab addPP addPanelCategory addPrefixToName advanceToNextDrivenKey ' +
	      'affectedNet affects aimConstraint air alias aliasAttr align alignCtx alignCurve ' +
	      'alignSurface allViewFit ambientLight angle angleBetween animCone animCurveEditor ' +
	      'animDisplay animView annotate appendStringArray applicationName applyAttrPreset ' +
	      'applyTake arcLenDimContext arcLengthDimension arclen arrayMapper art3dPaintCtx ' +
	      'artAttrCtx artAttrPaintVertexCtx artAttrSkinPaintCtx artAttrTool artBuildPaintMenu ' +
	      'artFluidAttrCtx artPuttyCtx artSelectCtx artSetPaintCtx artUserPaintCtx assignCommand ' +
	      'assignInputDevice assignViewportFactories attachCurve attachDeviceAttr attachSurface ' +
	      'attrColorSliderGrp attrCompatibility attrControlGrp attrEnumOptionMenu ' +
	      'attrEnumOptionMenuGrp attrFieldGrp attrFieldSliderGrp attrNavigationControlGrp ' +
	      'attrPresetEditWin attributeExists attributeInfo attributeMenu attributeQuery ' +
	      'autoKeyframe autoPlace bakeClip bakeFluidShading bakePartialHistory bakeResults ' +
	      'bakeSimulation basename basenameEx batchRender bessel bevel bevelPlus binMembership ' +
	      'bindSkin blend2 blendShape blendShapeEditor blendShapePanel blendTwoAttr blindDataType ' +
	      'boneLattice boundary boxDollyCtx boxZoomCtx bufferCurve buildBookmarkMenu ' +
	      'buildKeyframeMenu button buttonManip CBG cacheFile cacheFileCombine cacheFileMerge ' +
	      'cacheFileTrack camera cameraView canCreateManip canvas capitalizeString catch ' +
	      'catchQuiet ceil changeSubdivComponentDisplayLevel changeSubdivRegion channelBox ' +
	      'character characterMap characterOutlineEditor characterize chdir checkBox checkBoxGrp ' +
	      'checkDefaultRenderGlobals choice circle circularFillet clamp clear clearCache clip ' +
	      'clipEditor clipEditorCurrentTimeCtx clipSchedule clipSchedulerOutliner clipTrimBefore ' +
	      'closeCurve closeSurface cluster cmdFileOutput cmdScrollFieldExecuter ' +
	      'cmdScrollFieldReporter cmdShell coarsenSubdivSelectionList collision color ' +
	      'colorAtPoint colorEditor colorIndex colorIndexSliderGrp colorSliderButtonGrp ' +
	      'colorSliderGrp columnLayout commandEcho commandLine commandPort compactHairSystem ' +
	      'componentEditor compositingInterop computePolysetVolume condition cone confirmDialog ' +
	      'connectAttr connectControl connectDynamic connectJoint connectionInfo constrain ' +
	      'constrainValue constructionHistory container containsMultibyte contextInfo control ' +
	      'convertFromOldLayers convertIffToPsd convertLightmap convertSolidTx convertTessellation ' +
	      'convertUnit copyArray copyFlexor copyKey copySkinWeights cos cpButton cpCache ' +
	      'cpClothSet cpCollision cpConstraint cpConvClothToMesh cpForces cpGetSolverAttr cpPanel ' +
	      'cpProperty cpRigidCollisionFilter cpSeam cpSetEdit cpSetSolverAttr cpSolver ' +
	      'cpSolverTypes cpTool cpUpdateClothUVs createDisplayLayer createDrawCtx createEditor ' +
	      'createLayeredPsdFile createMotionField createNewShelf createNode createRenderLayer ' +
	      'createSubdivRegion cross crossProduct ctxAbort ctxCompletion ctxEditMode ctxTraverse ' +
	      'currentCtx currentTime currentTimeCtx currentUnit curve curveAddPtCtx ' +
	      'curveCVCtx curveEPCtx curveEditorCtx curveIntersect curveMoveEPCtx curveOnSurface ' +
	      'curveSketchCtx cutKey cycleCheck cylinder dagPose date defaultLightListCheckBox ' +
	      'defaultNavigation defineDataServer defineVirtualDevice deformer deg_to_rad delete ' +
	      'deleteAttr deleteShadingGroupsAndMaterials deleteShelfTab deleteUI deleteUnusedBrushes ' +
	      'delrandstr detachCurve detachDeviceAttr detachSurface deviceEditor devicePanel dgInfo ' +
	      'dgdirty dgeval dgtimer dimWhen directKeyCtx directionalLight dirmap dirname disable ' +
	      'disconnectAttr disconnectJoint diskCache displacementToPoly displayAffected ' +
	      'displayColor displayCull displayLevelOfDetail displayPref displayRGBColor ' +
	      'displaySmoothness displayStats displayString displaySurface distanceDimContext ' +
	      'distanceDimension doBlur dolly dollyCtx dopeSheetEditor dot dotProduct ' +
	      'doubleProfileBirailSurface drag dragAttrContext draggerContext dropoffLocator ' +
	      'duplicate duplicateCurve duplicateSurface dynCache dynControl dynExport dynExpression ' +
	      'dynGlobals dynPaintEditor dynParticleCtx dynPref dynRelEdPanel dynRelEditor ' +
	      'dynamicLoad editAttrLimits editDisplayLayerGlobals editDisplayLayerMembers ' +
	      'editRenderLayerAdjustment editRenderLayerGlobals editRenderLayerMembers editor ' +
	      'editorTemplate effector emit emitter enableDevice encodeString endString endsWith env ' +
	      'equivalent equivalentTol erf error eval evalDeferred evalEcho event ' +
	      'exactWorldBoundingBox exclusiveLightCheckBox exec executeForEachObject exists exp ' +
	      'expression expressionEditorListen extendCurve extendSurface extrude fcheck fclose feof ' +
	      'fflush fgetline fgetword file fileBrowserDialog fileDialog fileExtension fileInfo ' +
	      'filetest filletCurve filter filterCurve filterExpand filterStudioImport ' +
	      'findAllIntersections findAnimCurves findKeyframe findMenuItem findRelatedSkinCluster ' +
	      'finder firstParentOf fitBspline flexor floatEq floatField floatFieldGrp floatScrollBar ' +
	      'floatSlider floatSlider2 floatSliderButtonGrp floatSliderGrp floor flow fluidCacheInfo ' +
	      'fluidEmitter fluidVoxelInfo flushUndo fmod fontDialog fopen formLayout format fprint ' +
	      'frameLayout fread freeFormFillet frewind fromNativePath fwrite gamma gauss ' +
	      'geometryConstraint getApplicationVersionAsFloat getAttr getClassification ' +
	      'getDefaultBrush getFileList getFluidAttr getInputDeviceRange getMayaPanelTypes ' +
	      'getModifiers getPanel getParticleAttr getPluginResource getenv getpid glRender ' +
	      'glRenderEditor globalStitch gmatch goal gotoBindPose grabColor gradientControl ' +
	      'gradientControlNoAttr graphDollyCtx graphSelectContext graphTrackCtx gravity grid ' +
	      'gridLayout group groupObjectsByName HfAddAttractorToAS HfAssignAS HfBuildEqualMap ' +
	      'HfBuildFurFiles HfBuildFurImages HfCancelAFR HfConnectASToHF HfCreateAttractor ' +
	      'HfDeleteAS HfEditAS HfPerformCreateAS HfRemoveAttractorFromAS HfSelectAttached ' +
	      'HfSelectAttractors HfUnAssignAS hardenPointCurve hardware hardwareRenderPanel ' +
	      'headsUpDisplay headsUpMessage help helpLine hermite hide hilite hitTest hotBox hotkey ' +
	      'hotkeyCheck hsv_to_rgb hudButton hudSlider hudSliderButton hwReflectionMap hwRender ' +
	      'hwRenderLoad hyperGraph hyperPanel hyperShade hypot iconTextButton iconTextCheckBox ' +
	      'iconTextRadioButton iconTextRadioCollection iconTextScrollList iconTextStaticLabel ' +
	      'ikHandle ikHandleCtx ikHandleDisplayScale ikSolver ikSplineHandleCtx ikSystem ' +
	      'ikSystemInfo ikfkDisplayMethod illustratorCurves image imfPlugins inheritTransform ' +
	      'insertJoint insertJointCtx insertKeyCtx insertKnotCurve insertKnotSurface instance ' +
	      'instanceable instancer intField intFieldGrp intScrollBar intSlider intSliderGrp ' +
	      'interToUI internalVar intersect iprEngine isAnimCurve isConnected isDirty isParentOf ' +
	      'isSameObject isTrue isValidObjectName isValidString isValidUiName isolateSelect ' +
	      'itemFilter itemFilterAttr itemFilterRender itemFilterType joint jointCluster jointCtx ' +
	      'jointDisplayScale jointLattice keyTangent keyframe keyframeOutliner ' +
	      'keyframeRegionCurrentTimeCtx keyframeRegionDirectKeyCtx keyframeRegionDollyCtx ' +
	      'keyframeRegionInsertKeyCtx keyframeRegionMoveKeyCtx keyframeRegionScaleKeyCtx ' +
	      'keyframeRegionSelectKeyCtx keyframeRegionSetKeyCtx keyframeRegionTrackCtx ' +
	      'keyframeStats lassoContext lattice latticeDeformKeyCtx launch launchImageEditor ' +
	      'layerButton layeredShaderPort layeredTexturePort layout layoutDialog lightList ' +
	      'lightListEditor lightListPanel lightlink lineIntersection linearPrecision linstep ' +
	      'listAnimatable listAttr listCameras listConnections listDeviceAttachments listHistory ' +
	      'listInputDeviceAxes listInputDeviceButtons listInputDevices listMenuAnnotation ' +
	      'listNodeTypes listPanelCategories listRelatives listSets listTransforms ' +
	      'listUnselected listerEditor loadFluid loadNewShelf loadPlugin ' +
	      'loadPluginLanguageResources loadPrefObjects localizedPanelLabel lockNode loft log ' +
	      'longNameOf lookThru ls lsThroughFilter lsType lsUI Mayatomr mag makeIdentity makeLive ' +
	      'makePaintable makeRoll makeSingleSurface makeTubeOn makebot manipMoveContext ' +
	      'manipMoveLimitsCtx manipOptions manipRotateContext manipRotateLimitsCtx ' +
	      'manipScaleContext manipScaleLimitsCtx marker match max memory menu menuBarLayout ' +
	      'menuEditor menuItem menuItemToShelf menuSet menuSetPref messageLine min minimizeApp ' +
	      'mirrorJoint modelCurrentTimeCtx modelEditor modelPanel mouse movIn movOut move ' +
	      'moveIKtoFK moveKeyCtx moveVertexAlongDirection multiProfileBirailSurface mute ' +
	      'nParticle nameCommand nameField namespace namespaceInfo newPanelItems newton nodeCast ' +
	      'nodeIconButton nodeOutliner nodePreset nodeType noise nonLinear normalConstraint ' +
	      'normalize nurbsBoolean nurbsCopyUVSet nurbsCube nurbsEditUV nurbsPlane nurbsSelect ' +
	      'nurbsSquare nurbsToPoly nurbsToPolygonsPref nurbsToSubdiv nurbsToSubdivPref ' +
	      'nurbsUVSet nurbsViewDirectionVector objExists objectCenter objectLayer objectType ' +
	      'objectTypeUI obsoleteProc oceanNurbsPreviewPlane offsetCurve offsetCurveOnSurface ' +
	      'offsetSurface openGLExtension openMayaPref optionMenu optionMenuGrp optionVar orbit ' +
	      'orbitCtx orientConstraint outlinerEditor outlinerPanel overrideModifier ' +
	      'paintEffectsDisplay pairBlend palettePort paneLayout panel panelConfiguration ' +
	      'panelHistory paramDimContext paramDimension paramLocator parent parentConstraint ' +
	      'particle particleExists particleInstancer particleRenderInfo partition pasteKey ' +
	      'pathAnimation pause pclose percent performanceOptions pfxstrokes pickWalk picture ' +
	      'pixelMove planarSrf plane play playbackOptions playblast plugAttr plugNode pluginInfo ' +
	      'pluginResourceUtil pointConstraint pointCurveConstraint pointLight pointMatrixMult ' +
	      'pointOnCurve pointOnSurface pointPosition poleVectorConstraint polyAppend ' +
	      'polyAppendFacetCtx polyAppendVertex polyAutoProjection polyAverageNormal ' +
	      'polyAverageVertex polyBevel polyBlendColor polyBlindData polyBoolOp polyBridgeEdge ' +
	      'polyCacheMonitor polyCheck polyChipOff polyClipboard polyCloseBorder polyCollapseEdge ' +
	      'polyCollapseFacet polyColorBlindData polyColorDel polyColorPerVertex polyColorSet ' +
	      'polyCompare polyCone polyCopyUV polyCrease polyCreaseCtx polyCreateFacet ' +
	      'polyCreateFacetCtx polyCube polyCut polyCutCtx polyCylinder polyCylindricalProjection ' +
	      'polyDelEdge polyDelFacet polyDelVertex polyDuplicateAndConnect polyDuplicateEdge ' +
	      'polyEditUV polyEditUVShell polyEvaluate polyExtrudeEdge polyExtrudeFacet ' +
	      'polyExtrudeVertex polyFlipEdge polyFlipUV polyForceUV polyGeoSampler polyHelix ' +
	      'polyInfo polyInstallAction polyLayoutUV polyListComponentConversion polyMapCut ' +
	      'polyMapDel polyMapSew polyMapSewMove polyMergeEdge polyMergeEdgeCtx polyMergeFacet ' +
	      'polyMergeFacetCtx polyMergeUV polyMergeVertex polyMirrorFace polyMoveEdge ' +
	      'polyMoveFacet polyMoveFacetUV polyMoveUV polyMoveVertex polyNormal polyNormalPerVertex ' +
	      'polyNormalizeUV polyOptUvs polyOptions polyOutput polyPipe polyPlanarProjection ' +
	      'polyPlane polyPlatonicSolid polyPoke polyPrimitive polyPrism polyProjection ' +
	      'polyPyramid polyQuad polyQueryBlindData polyReduce polySelect polySelectConstraint ' +
	      'polySelectConstraintMonitor polySelectCtx polySelectEditCtx polySeparate ' +
	      'polySetToFaceNormal polySewEdge polyShortestPathCtx polySmooth polySoftEdge ' +
	      'polySphere polySphericalProjection polySplit polySplitCtx polySplitEdge polySplitRing ' +
	      'polySplitVertex polyStraightenUVBorder polySubdivideEdge polySubdivideFacet ' +
	      'polyToSubdiv polyTorus polyTransfer polyTriangulate polyUVSet polyUnite polyWedgeFace ' +
	      'popen popupMenu pose pow preloadRefEd print progressBar progressWindow projFileViewer ' +
	      'projectCurve projectTangent projectionContext projectionManip promptDialog propModCtx ' +
	      'propMove psdChannelOutliner psdEditTextureFile psdExport psdTextureFile putenv pwd ' +
	      'python querySubdiv quit rad_to_deg radial radioButton radioButtonGrp radioCollection ' +
	      'radioMenuItemCollection rampColorPort rand randomizeFollicles randstate rangeControl ' +
	      'readTake rebuildCurve rebuildSurface recordAttr recordDevice redo reference ' +
	      'referenceEdit referenceQuery refineSubdivSelectionList refresh refreshAE ' +
	      'registerPluginResource rehash reloadImage removeJoint removeMultiInstance ' +
	      'removePanelCategory rename renameAttr renameSelectionList renameUI render ' +
	      'renderGlobalsNode renderInfo renderLayerButton renderLayerParent ' +
	      'renderLayerPostProcess renderLayerUnparent renderManip renderPartition ' +
	      'renderQualityNode renderSettings renderThumbnailUpdate renderWindowEditor ' +
	      'renderWindowSelectContext renderer reorder reorderDeformers requires reroot ' +
	      'resampleFluid resetAE resetPfxToPolyCamera resetTool resolutionNode retarget ' +
	      'reverseCurve reverseSurface revolve rgb_to_hsv rigidBody rigidSolver roll rollCtx ' +
	      'rootOf rot rotate rotationInterpolation roundConstantRadius rowColumnLayout rowLayout ' +
	      'runTimeCommand runup sampleImage saveAllShelves saveAttrPreset saveFluid saveImage ' +
	      'saveInitialState saveMenu savePrefObjects savePrefs saveShelf saveToolSettings scale ' +
	      'scaleBrushBrightness scaleComponents scaleConstraint scaleKey scaleKeyCtx sceneEditor ' +
	      'sceneUIReplacement scmh scriptCtx scriptEditorInfo scriptJob scriptNode scriptTable ' +
	      'scriptToShelf scriptedPanel scriptedPanelType scrollField scrollLayout sculpt ' +
	      'searchPathArray seed selLoadSettings select selectContext selectCurveCV selectKey ' +
	      'selectKeyCtx selectKeyframeRegionCtx selectMode selectPref selectPriority selectType ' +
	      'selectedNodes selectionConnection separator setAttr setAttrEnumResource ' +
	      'setAttrMapping setAttrNiceNameResource setConstraintRestPosition ' +
	      'setDefaultShadingGroup setDrivenKeyframe setDynamic setEditCtx setEditor setFluidAttr ' +
	      'setFocus setInfinity setInputDeviceMapping setKeyCtx setKeyPath setKeyframe ' +
	      'setKeyframeBlendshapeTargetWts setMenuMode setNodeNiceNameResource setNodeTypeFlag ' +
	      'setParent setParticleAttr setPfxToPolyCamera setPluginResource setProject ' +
	      'setStampDensity setStartupMessage setState setToolTo setUITemplate setXformManip sets ' +
	      'shadingConnection shadingGeometryRelCtx shadingLightRelCtx shadingNetworkCompare ' +
	      'shadingNode shapeCompare shelfButton shelfLayout shelfTabLayout shellField ' +
	      'shortNameOf showHelp showHidden showManipCtx showSelectionInTitle ' +
	      'showShadingGroupAttrEditor showWindow sign simplify sin singleProfileBirailSurface ' +
	      'size sizeBytes skinCluster skinPercent smoothCurve smoothTangentSurface smoothstep ' +
	      'snap2to2 snapKey snapMode snapTogetherCtx snapshot soft softMod softModCtx sort sound ' +
	      'soundControl source spaceLocator sphere sphrand spotLight spotLightPreviewPort ' +
	      'spreadSheetEditor spring sqrt squareSurface srtContext stackTrace startString ' +
	      'startsWith stitchAndExplodeShell stitchSurface stitchSurfacePoints strcmp ' +
	      'stringArrayCatenate stringArrayContains stringArrayCount stringArrayInsertAtIndex ' +
	      'stringArrayIntersector stringArrayRemove stringArrayRemoveAtIndex ' +
	      'stringArrayRemoveDuplicates stringArrayRemoveExact stringArrayToString ' +
	      'stringToStringArray strip stripPrefixFromName stroke subdAutoProjection ' +
	      'subdCleanTopology subdCollapse subdDuplicateAndConnect subdEditUV ' +
	      'subdListComponentConversion subdMapCut subdMapSewMove subdMatchTopology subdMirror ' +
	      'subdToBlind subdToPoly subdTransferUVsToCache subdiv subdivCrease ' +
	      'subdivDisplaySmoothness substitute substituteAllString substituteGeometry substring ' +
	      'surface surfaceSampler surfaceShaderList swatchDisplayPort switchTable symbolButton ' +
	      'symbolCheckBox sysFile system tabLayout tan tangentConstraint texLatticeDeformContext ' +
	      'texManipContext texMoveContext texMoveUVShellContext texRotateContext texScaleContext ' +
	      'texSelectContext texSelectShortestPathCtx texSmudgeUVContext texWinToolCtx text ' +
	      'textCurves textField textFieldButtonGrp textFieldGrp textManip textScrollList ' +
	      'textToShelf textureDisplacePlane textureHairColor texturePlacementContext ' +
	      'textureWindow threadCount threePointArcCtx timeControl timePort timerX toNativePath ' +
	      'toggle toggleAxis toggleWindowVisibility tokenize tokenizeList tolerance tolower ' +
	      'toolButton toolCollection toolDropped toolHasOptions toolPropertyWindow torus toupper ' +
	      'trace track trackCtx transferAttributes transformCompare transformLimits translator ' +
	      'trim trunc truncateFluidCache truncateHairCache tumble tumbleCtx turbulence ' +
	      'twoPointArcCtx uiRes uiTemplate unassignInputDevice undo undoInfo ungroup uniform unit ' +
	      'unloadPlugin untangleUV untitledFileName untrim upAxis updateAE userCtx uvLink ' +
	      'uvSnapshot validateShelfName vectorize view2dToolCtx viewCamera viewClipPlane ' +
	      'viewFit viewHeadOn viewLookAt viewManip viewPlace viewSet visor volumeAxis vortex ' +
	      'waitCursor warning webBrowser webBrowserPrefs whatIs window windowPref wire ' +
	      'wireContext workspace wrinkle wrinkleContext writeTake xbmLangPathList xform',
	    illegal: '</',
	    contains: [
	      hljs.C_NUMBER_MODE,
	      hljs.APOS_STRING_MODE,
	      hljs.QUOTE_STRING_MODE,
	      {
	        className: 'string',
	        begin: '`',
	        end: '`',
	        contains: [ hljs.BACKSLASH_ESCAPE ]
	      },
	      { // eats variables
	        begin: /[$%@](\^\w\b|#\w+|[^\s\w{]|\{\w+\}|\w+)/
	      },
	      hljs.C_LINE_COMMENT_MODE,
	      hljs.C_BLOCK_COMMENT_MODE
	    ]
	  };
	}

	mel_1 = mel;
	return mel_1;
}

/*
Language: Mercury
Author: mucaho <mkucko@gmail.com>
Description: Mercury is a logic/functional programming language which combines the clarity and expressiveness of declarative programming with advanced static analysis and error detection features.
Website: https://www.mercurylang.org
*/

var mercury_1;
var hasRequiredMercury;

function requireMercury () {
	if (hasRequiredMercury) return mercury_1;
	hasRequiredMercury = 1;
	function mercury(hljs) {
	  const KEYWORDS = {
	    keyword:
	      'module use_module import_module include_module end_module initialise ' +
	      'mutable initialize finalize finalise interface implementation pred ' +
	      'mode func type inst solver any_pred any_func is semidet det nondet ' +
	      'multi erroneous failure cc_nondet cc_multi typeclass instance where ' +
	      'pragma promise external trace atomic or_else require_complete_switch ' +
	      'require_det require_semidet require_multi require_nondet ' +
	      'require_cc_multi require_cc_nondet require_erroneous require_failure',
	    meta:
	      // pragma
	      'inline no_inline type_spec source_file fact_table obsolete memo ' +
	      'loop_check minimal_model terminates does_not_terminate ' +
	      'check_termination promise_equivalent_clauses ' +
	      // preprocessor
	      'foreign_proc foreign_decl foreign_code foreign_type ' +
	      'foreign_import_module foreign_export_enum foreign_export ' +
	      'foreign_enum may_call_mercury will_not_call_mercury thread_safe ' +
	      'not_thread_safe maybe_thread_safe promise_pure promise_semipure ' +
	      'tabled_for_io local untrailed trailed attach_to_io_state ' +
	      'can_pass_as_mercury_type stable will_not_throw_exception ' +
	      'may_modify_trail will_not_modify_trail may_duplicate ' +
	      'may_not_duplicate affects_liveness does_not_affect_liveness ' +
	      'doesnt_affect_liveness no_sharing unknown_sharing sharing',
	    built_in:
	      'some all not if then else true fail false try catch catch_any ' +
	      'semidet_true semidet_false semidet_fail impure_true impure semipure'
	  };

	  const COMMENT = hljs.COMMENT('%', '$');

	  const NUMCODE = {
	    className: 'number',
	    begin: "0'.\\|0[box][0-9a-fA-F]*"
	  };

	  const ATOM = hljs.inherit(hljs.APOS_STRING_MODE, {
	    relevance: 0
	  });
	  const STRING = hljs.inherit(hljs.QUOTE_STRING_MODE, {
	    relevance: 0
	  });
	  const STRING_FMT = {
	    className: 'subst',
	    begin: '\\\\[abfnrtv]\\|\\\\x[0-9a-fA-F]*\\\\\\|%[-+# *.0-9]*[dioxXucsfeEgGp]',
	    relevance: 0
	  };
	  STRING.contains = STRING.contains.slice(); // we need our own copy of contains
	  STRING.contains.push(STRING_FMT);

	  const IMPLICATION = {
	    className: 'built_in',
	    variants: [
	      {
	        begin: '<=>'
	      },
	      {
	        begin: '<=',
	        relevance: 0
	      },
	      {
	        begin: '=>',
	        relevance: 0
	      },
	      {
	        begin: '/\\\\'
	      },
	      {
	        begin: '\\\\/'
	      }
	    ]
	  };

	  const HEAD_BODY_CONJUNCTION = {
	    className: 'built_in',
	    variants: [
	      {
	        begin: ':-\\|-->'
	      },
	      {
	        begin: '=',
	        relevance: 0
	      }
	    ]
	  };

	  return {
	    name: 'Mercury',
	    aliases: [
	      'm',
	      'moo'
	    ],
	    keywords: KEYWORDS,
	    contains: [
	      IMPLICATION,
	      HEAD_BODY_CONJUNCTION,
	      COMMENT,
	      hljs.C_BLOCK_COMMENT_MODE,
	      NUMCODE,
	      hljs.NUMBER_MODE,
	      ATOM,
	      STRING,
	      { // relevance booster
	        begin: /:-/
	      },
	      { // relevance booster
	        begin: /\.$/
	      }
	    ]
	  };
	}

	mercury_1 = mercury;
	return mercury_1;
}

/*
Language: MIPS Assembly
Author: Nebuleon Fumika <nebuleon.fumika@gmail.com>
Description: MIPS Assembly (up to MIPS32R2)
Website: https://en.wikipedia.org/wiki/MIPS_architecture
Category: assembler
*/

var mipsasm_1;
var hasRequiredMipsasm;

function requireMipsasm () {
	if (hasRequiredMipsasm) return mipsasm_1;
	hasRequiredMipsasm = 1;
	function mipsasm(hljs) {
	  // local labels: %?[FB]?[AT]?\d{1,2}\w+
	  return {
	    name: 'MIPS Assembly',
	    case_insensitive: true,
	    aliases: [ 'mips' ],
	    keywords: {
	      $pattern: '\\.?' + hljs.IDENT_RE,
	      meta:
	        // GNU preprocs
	        '.2byte .4byte .align .ascii .asciz .balign .byte .code .data .else .end .endif .endm .endr .equ .err .exitm .extern .global .hword .if .ifdef .ifndef .include .irp .long .macro .rept .req .section .set .skip .space .text .word .ltorg ',
	      built_in:
	        '$0 $1 $2 $3 $4 $5 $6 $7 $8 $9 $10 $11 $12 $13 $14 $15 ' + // integer registers
	        '$16 $17 $18 $19 $20 $21 $22 $23 $24 $25 $26 $27 $28 $29 $30 $31 ' + // integer registers
	        'zero at v0 v1 a0 a1 a2 a3 a4 a5 a6 a7 ' + // integer register aliases
	        't0 t1 t2 t3 t4 t5 t6 t7 t8 t9 s0 s1 s2 s3 s4 s5 s6 s7 s8 ' + // integer register aliases
	        'k0 k1 gp sp fp ra ' + // integer register aliases
	        '$f0 $f1 $f2 $f2 $f4 $f5 $f6 $f7 $f8 $f9 $f10 $f11 $f12 $f13 $f14 $f15 ' + // floating-point registers
	        '$f16 $f17 $f18 $f19 $f20 $f21 $f22 $f23 $f24 $f25 $f26 $f27 $f28 $f29 $f30 $f31 ' + // floating-point registers
	        'Context Random EntryLo0 EntryLo1 Context PageMask Wired EntryHi ' + // Coprocessor 0 registers
	        'HWREna BadVAddr Count Compare SR IntCtl SRSCtl SRSMap Cause EPC PRId ' + // Coprocessor 0 registers
	        'EBase Config Config1 Config2 Config3 LLAddr Debug DEPC DESAVE CacheErr ' + // Coprocessor 0 registers
	        'ECC ErrorEPC TagLo DataLo TagHi DataHi WatchLo WatchHi PerfCtl PerfCnt ' // Coprocessor 0 registers
	    },
	    contains: [
	      {
	        className: 'keyword',
	        begin: '\\b(' + // mnemonics
	            // 32-bit integer instructions
	            'addi?u?|andi?|b(al)?|beql?|bgez(al)?l?|bgtzl?|blezl?|bltz(al)?l?|' +
	            'bnel?|cl[oz]|divu?|ext|ins|j(al)?|jalr(\\.hb)?|jr(\\.hb)?|lbu?|lhu?|' +
	            'll|lui|lw[lr]?|maddu?|mfhi|m