import { AltMatcher, ManyMatcher, OpMatcher, SeqMatcher, SymMatcher, } from './matchers';
import { BeginMatcher, EndMatcher, VoidMatcher, } from './matchers/anchor-matcher';
import { CommentMatcher } from './matchers/comment-matcher';
import { NumMatcher } from './matchers/num-matcher';
import { StrContentMatcher, StrNodeMatcher, StrTplMatcher, } from './matchers/str-matcher';
import { TreeMatcher } from './matchers/tree-matcher';
import { coerceCommentOptions, coerceManyOptions, coerceNumOptions, coerceOpOptions, coerceStrOptions, coerceSymOptions, coerceTreeOptions, } from './options';
import { isRegex } from './regex';
class TerminalBuilder {
    handler(fn) {
        const voidMatcher = new VoidBuilder(fn);
        const builder = new SeqBuilder(this, voidMatcher);
        return builder;
    }
}
class AbstractBuilder extends TerminalBuilder {
    sym(arg1, arg2) {
        const opts = coerceSymOptions(arg1, arg2);
        const builder = new SymBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    op(arg1, arg2) {
        const opts = coerceOpOptions(arg1, arg2);
        const builder = new OpBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    comment(arg1, arg2) {
        const opts = coerceCommentOptions(arg1, arg2);
        const builder = new CommentBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    num(arg1, arg2) {
        const opts = coerceNumOptions(arg1, arg2);
        const builder = new NumBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    many(arg1, arg2, arg3) {
        const opts = coerceManyOptions(arg1, arg2, arg3);
        const builder = new ManyBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    opt(innerBuilder) {
        const opts = coerceManyOptions(innerBuilder, 0, 1);
        const builder = new ManyBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    alt(...alts) {
        const builder = new AltBuilder(alts);
        return new SeqBuilder(this, builder);
    }
    tree(arg1) {
        const opts = coerceTreeOptions(arg1);
        const builder = new TreeBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    str(arg1, arg2) {
        const opts = coerceStrOptions(arg1, arg2);
        const builder = new StrBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    end() {
        return new EndBuilder(this);
    }
    join(other) {
        const builder = new SeqBuilder(this, other);
        return builder;
    }
}
// Anchors
export class BeginBuilder extends AbstractBuilder {
    build() {
        return new BeginMatcher();
    }
}
export class EndBuilder extends TerminalBuilder {
    constructor(builder) {
        super();
        this.builder = builder;
    }
    build() {
        const matcher = this.builder.build();
        const matchers = matcher instanceof SeqMatcher ? matcher.seq : [matcher];
        matchers.push(new EndMatcher());
        return new SeqMatcher({ matchers });
    }
}
export function begin() {
    return new BeginBuilder();
}
export class VoidBuilder extends AbstractBuilder {
    constructor(fn) {
        super();
        this.fn = fn;
    }
    build() {
        return new VoidMatcher(this.fn);
    }
}
export function handler(fn) {
    return new VoidBuilder(fn);
}
// Sequence
export class SeqBuilder extends AbstractBuilder {
    constructor(prev, next) {
        super();
        const prevSeq = prev instanceof SeqBuilder
            ? prev.builders
            : [prev];
        const nextSeq = next instanceof SeqBuilder
            ? next.builders
            : [next];
        this.builders = [...prevSeq, ...nextSeq];
    }
    build() {
        const matchers = this.builders.map((builder) => builder.build());
        return new SeqMatcher({ matchers });
    }
}
export function join(first, second, ...others) {
    const seq = new SeqBuilder(first, second);
    if (!others.length) {
        return seq;
    }
    return others.reduce((res, query) => {
        return res.join(query);
    }, seq);
}
// Symbols
export class SymBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        return new SymMatcher(this.opts);
    }
}
export function sym(arg1, arg2) {
    const opts = coerceSymOptions(arg1, arg2);
    return new SymBuilder(opts);
}
// Operators
export class OpBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        return new OpMatcher(this.opts);
    }
}
export function op(arg1, arg2) {
    const opts = coerceOpOptions(arg1, arg2);
    return new OpBuilder(opts);
}
// Comments
export class CommentBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        return new CommentMatcher(this.opts);
    }
}
export function comment(arg1, arg2) {
    const opts = coerceCommentOptions(arg1, arg2);
    return new CommentBuilder(opts);
}
// Numbers
export class NumBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        return new NumMatcher(this.opts);
    }
}
export function num(arg1, arg2) {
    const opts = coerceNumOptions(arg1, arg2);
    return new NumBuilder(opts);
}
// Repetitive patterns
export class ManyBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        const matcher = this.opts.builder.build();
        return new ManyMatcher({ ...this.opts, matcher });
    }
}
export function many(arg1, arg2, arg3) {
    const opts = coerceManyOptions(arg1, arg2, arg3);
    return new ManyBuilder(opts);
}
export function opt(builder) {
    const opts = coerceManyOptions(builder, 0, 1);
    return new ManyBuilder(opts);
}
// Alternatives
export class AltBuilder extends AbstractBuilder {
    constructor(builders) {
        super();
        this.builders = builders;
    }
    build() {
        const matchers = this.builders.map((alt) => alt.build());
        return new AltMatcher({ matchers });
    }
}
export function alt(...builders) {
    return new AltBuilder(builders);
}
// Trees
export class TreeBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        const builderOpts = this.opts;
        const matcher = builderOpts.search ? builderOpts.search.build() : null;
        const matcherOpts = { ...builderOpts, matcher };
        return new TreeMatcher(matcherOpts);
    }
}
export function tree(arg1) {
    const opts = coerceTreeOptions(arg1);
    return new TreeBuilder(opts);
}
// Strings
export class StrBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        var _a, _b, _c, _d, _e, _f;
        if (this.opts.type === 'str-content') {
            return new StrNodeMatcher({
                matchers: [
                    new StrContentMatcher({
                        value: (_a = this.opts.match) !== null && _a !== void 0 ? _a : null,
                        handler: (_b = this.opts.handler) !== null && _b !== void 0 ? _b : null,
                    }),
                ],
                preHandler: null,
                postHandler: null,
            });
        }
        if (this.opts.match) {
            const matchers = [];
            this.opts.match.forEach((m) => {
                if (typeof m === 'string' || isRegex(m)) {
                    const contentMatcher = new StrContentMatcher({
                        value: m,
                        handler: null,
                    });
                    matchers.push(contentMatcher);
                }
                else if (m instanceof StrBuilder) {
                    const childStrMatcher = m.build();
                    if (childStrMatcher.matchers) {
                        matchers.push(...childStrMatcher.matchers);
                    }
                }
                else if (m instanceof StrTplMatcher) {
                    matchers.push(m);
                }
                else {
                    const tplMatcher = new StrTplMatcher({
                        matcher: m.build(),
                        preHandler: null,
                        postHandler: null,
                    });
                    matchers.push(tplMatcher);
                }
            });
            return new StrNodeMatcher({
                matchers: matchers,
                preHandler: (_c = this.opts.preHandler) !== null && _c !== void 0 ? _c : null,
                postHandler: (_d = this.opts.postHandler) !== null && _d !== void 0 ? _d : null,
            });
        }
        return new StrNodeMatcher({
            matchers: null,
            preHandler: (_e = this.opts.preHandler) !== null && _e !== void 0 ? _e : null,
            postHandler: (_f = this.opts.postHandler) !== null && _f !== void 0 ? _f : null,
        });
    }
}
export function str(arg1, arg2) {
    const opts = coerceStrOptions(arg1, arg2);
    return new StrBuilder(opts);
}
export function buildRoot(builder) {
    const matcher = builder.build();
    return matcher instanceof TreeMatcher && matcher.type === 'root-tree'
        ? matcher
        : new TreeMatcher({ matcher, type: 'root-tree' });
}
//# sourceMappingURL=builder.js.map