"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildRoot = exports.str = exports.StrBuilder = exports.tree = exports.TreeBuilder = exports.alt = exports.AltBuilder = exports.opt = exports.many = exports.ManyBuilder = exports.num = exports.NumBuilder = exports.comment = exports.CommentBuilder = exports.op = exports.OpBuilder = exports.sym = exports.SymBuilder = exports.join = exports.SeqBuilder = exports.handler = exports.VoidBuilder = exports.begin = exports.EndBuilder = exports.BeginBuilder = void 0;
const matchers_1 = require("./matchers");
const anchor_matcher_1 = require("./matchers/anchor-matcher");
const comment_matcher_1 = require("./matchers/comment-matcher");
const num_matcher_1 = require("./matchers/num-matcher");
const str_matcher_1 = require("./matchers/str-matcher");
const tree_matcher_1 = require("./matchers/tree-matcher");
const options_1 = require("./options");
const regex_1 = require("./regex");
class TerminalBuilder {
    handler(fn) {
        const voidMatcher = new VoidBuilder(fn);
        const builder = new SeqBuilder(this, voidMatcher);
        return builder;
    }
}
class AbstractBuilder extends TerminalBuilder {
    sym(arg1, arg2) {
        const opts = (0, options_1.coerceSymOptions)(arg1, arg2);
        const builder = new SymBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    op(arg1, arg2) {
        const opts = (0, options_1.coerceOpOptions)(arg1, arg2);
        const builder = new OpBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    comment(arg1, arg2) {
        const opts = (0, options_1.coerceCommentOptions)(arg1, arg2);
        const builder = new CommentBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    num(arg1, arg2) {
        const opts = (0, options_1.coerceNumOptions)(arg1, arg2);
        const builder = new NumBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    many(arg1, arg2, arg3) {
        const opts = (0, options_1.coerceManyOptions)(arg1, arg2, arg3);
        const builder = new ManyBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    opt(innerBuilder) {
        const opts = (0, options_1.coerceManyOptions)(innerBuilder, 0, 1);
        const builder = new ManyBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    alt(...alts) {
        const builder = new AltBuilder(alts);
        return new SeqBuilder(this, builder);
    }
    tree(arg1) {
        const opts = (0, options_1.coerceTreeOptions)(arg1);
        const builder = new TreeBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    str(arg1, arg2) {
        const opts = (0, options_1.coerceStrOptions)(arg1, arg2);
        const builder = new StrBuilder(opts);
        return new SeqBuilder(this, builder);
    }
    end() {
        return new EndBuilder(this);
    }
    join(other) {
        const builder = new SeqBuilder(this, other);
        return builder;
    }
}
// Anchors
class BeginBuilder extends AbstractBuilder {
    build() {
        return new anchor_matcher_1.BeginMatcher();
    }
}
exports.BeginBuilder = BeginBuilder;
class EndBuilder extends TerminalBuilder {
    constructor(builder) {
        super();
        this.builder = builder;
    }
    build() {
        const matcher = this.builder.build();
        const matchers = matcher instanceof matchers_1.SeqMatcher ? matcher.seq : [matcher];
        matchers.push(new anchor_matcher_1.EndMatcher());
        return new matchers_1.SeqMatcher({ matchers });
    }
}
exports.EndBuilder = EndBuilder;
function begin() {
    return new BeginBuilder();
}
exports.begin = begin;
class VoidBuilder extends AbstractBuilder {
    constructor(fn) {
        super();
        this.fn = fn;
    }
    build() {
        return new anchor_matcher_1.VoidMatcher(this.fn);
    }
}
exports.VoidBuilder = VoidBuilder;
function handler(fn) {
    return new VoidBuilder(fn);
}
exports.handler = handler;
// Sequence
class SeqBuilder extends AbstractBuilder {
    constructor(prev, next) {
        super();
        const prevSeq = prev instanceof SeqBuilder
            ? prev.builders
            : [prev];
        const nextSeq = next instanceof SeqBuilder
            ? next.builders
            : [next];
        this.builders = [...prevSeq, ...nextSeq];
    }
    build() {
        const matchers = this.builders.map((builder) => builder.build());
        return new matchers_1.SeqMatcher({ matchers });
    }
}
exports.SeqBuilder = SeqBuilder;
function join(first, second, ...others) {
    const seq = new SeqBuilder(first, second);
    if (!others.length) {
        return seq;
    }
    return others.reduce((res, query) => {
        return res.join(query);
    }, seq);
}
exports.join = join;
// Symbols
class SymBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        return new matchers_1.SymMatcher(this.opts);
    }
}
exports.SymBuilder = SymBuilder;
function sym(arg1, arg2) {
    const opts = (0, options_1.coerceSymOptions)(arg1, arg2);
    return new SymBuilder(opts);
}
exports.sym = sym;
// Operators
class OpBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        return new matchers_1.OpMatcher(this.opts);
    }
}
exports.OpBuilder = OpBuilder;
function op(arg1, arg2) {
    const opts = (0, options_1.coerceOpOptions)(arg1, arg2);
    return new OpBuilder(opts);
}
exports.op = op;
// Comments
class CommentBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        return new comment_matcher_1.CommentMatcher(this.opts);
    }
}
exports.CommentBuilder = CommentBuilder;
function comment(arg1, arg2) {
    const opts = (0, options_1.coerceCommentOptions)(arg1, arg2);
    return new CommentBuilder(opts);
}
exports.comment = comment;
// Numbers
class NumBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        return new num_matcher_1.NumMatcher(this.opts);
    }
}
exports.NumBuilder = NumBuilder;
function num(arg1, arg2) {
    const opts = (0, options_1.coerceNumOptions)(arg1, arg2);
    return new NumBuilder(opts);
}
exports.num = num;
// Repetitive patterns
class ManyBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        const matcher = this.opts.builder.build();
        return new matchers_1.ManyMatcher({ ...this.opts, matcher });
    }
}
exports.ManyBuilder = ManyBuilder;
function many(arg1, arg2, arg3) {
    const opts = (0, options_1.coerceManyOptions)(arg1, arg2, arg3);
    return new ManyBuilder(opts);
}
exports.many = many;
function opt(builder) {
    const opts = (0, options_1.coerceManyOptions)(builder, 0, 1);
    return new ManyBuilder(opts);
}
exports.opt = opt;
// Alternatives
class AltBuilder extends AbstractBuilder {
    constructor(builders) {
        super();
        this.builders = builders;
    }
    build() {
        const matchers = this.builders.map((alt) => alt.build());
        return new matchers_1.AltMatcher({ matchers });
    }
}
exports.AltBuilder = AltBuilder;
function alt(...builders) {
    return new AltBuilder(builders);
}
exports.alt = alt;
// Trees
class TreeBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        const builderOpts = this.opts;
        const matcher = builderOpts.search ? builderOpts.search.build() : null;
        const matcherOpts = { ...builderOpts, matcher };
        return new tree_matcher_1.TreeMatcher(matcherOpts);
    }
}
exports.TreeBuilder = TreeBuilder;
function tree(arg1) {
    const opts = (0, options_1.coerceTreeOptions)(arg1);
    return new TreeBuilder(opts);
}
exports.tree = tree;
// Strings
class StrBuilder extends AbstractBuilder {
    constructor(opts) {
        super();
        this.opts = opts;
    }
    build() {
        var _a, _b, _c, _d, _e, _f;
        if (this.opts.type === 'str-content') {
            return new str_matcher_1.StrNodeMatcher({
                matchers: [
                    new str_matcher_1.StrContentMatcher({
                        value: (_a = this.opts.match) !== null && _a !== void 0 ? _a : null,
                        handler: (_b = this.opts.handler) !== null && _b !== void 0 ? _b : null,
                    }),
                ],
                preHandler: null,
                postHandler: null,
            });
        }
        if (this.opts.match) {
            const matchers = [];
            this.opts.match.forEach((m) => {
                if (typeof m === 'string' || (0, regex_1.isRegex)(m)) {
                    const contentMatcher = new str_matcher_1.StrContentMatcher({
                        value: m,
                        handler: null,
                    });
                    matchers.push(contentMatcher);
                }
                else if (m instanceof StrBuilder) {
                    const childStrMatcher = m.build();
                    if (childStrMatcher.matchers) {
                        matchers.push(...childStrMatcher.matchers);
                    }
                }
                else if (m instanceof str_matcher_1.StrTplMatcher) {
                    matchers.push(m);
                }
                else {
                    const tplMatcher = new str_matcher_1.StrTplMatcher({
                        matcher: m.build(),
                        preHandler: null,
                        postHandler: null,
                    });
                    matchers.push(tplMatcher);
                }
            });
            return new str_matcher_1.StrNodeMatcher({
                matchers: matchers,
                preHandler: (_c = this.opts.preHandler) !== null && _c !== void 0 ? _c : null,
                postHandler: (_d = this.opts.postHandler) !== null && _d !== void 0 ? _d : null,
            });
        }
        return new str_matcher_1.StrNodeMatcher({
            matchers: null,
            preHandler: (_e = this.opts.preHandler) !== null && _e !== void 0 ? _e : null,
            postHandler: (_f = this.opts.postHandler) !== null && _f !== void 0 ? _f : null,
        });
    }
}
exports.StrBuilder = StrBuilder;
function str(arg1, arg2) {
    const opts = (0, options_1.coerceStrOptions)(arg1, arg2);
    return new StrBuilder(opts);
}
exports.str = str;
function buildRoot(builder) {
    const matcher = builder.build();
    return matcher instanceof tree_matcher_1.TreeMatcher && matcher.type === 'root-tree'
        ? matcher
        : new tree_matcher_1.TreeMatcher({ matcher, type: 'root-tree' });
}
exports.buildRoot = buildRoot;
//# sourceMappingURL=builder.js.map