"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultTaskPool = getDefaultTaskPool;
exports.getTaskPoolForConfiguration = getTaskPoolForConfiguration;
exports.convertToZipWorker = convertToZipWorker;
exports.makeArchiveFromDirectory = makeArchiveFromDirectory;
exports.convertToZip = convertToZip;
exports.extractArchiveTo = extractArchiveTo;
const tslib_1 = require("tslib");
const core_1 = require("@yarnpkg/core");
const fslib_1 = require("@yarnpkg/fslib");
const libzip_1 = require("@yarnpkg/libzip");
const stream_1 = require("stream");
const tar_1 = tslib_1.__importDefault(require("tar"));
const TaskPool_1 = require("./TaskPool");
const miscUtils = tslib_1.__importStar(require("./miscUtils"));
const worker_zip_1 = require("./worker-zip");
function createTaskPool(poolMode, poolSize) {
    switch (poolMode) {
        case `async`:
            return new TaskPool_1.AsyncPool(convertToZipWorker, { poolSize });
        case `workers`:
            return new TaskPool_1.WorkerPool((0, worker_zip_1.getContent)(), { poolSize });
        default: {
            throw new Error(`Assertion failed: Unknown value ${poolMode} for taskPoolMode`);
        }
    }
}
let defaultWorkerPool;
function getDefaultTaskPool() {
    if (typeof defaultWorkerPool === `undefined`)
        defaultWorkerPool = createTaskPool(`workers`, core_1.nodeUtils.availableParallelism());
    return defaultWorkerPool;
}
const workerPools = new WeakMap();
function getTaskPoolForConfiguration(configuration) {
    if (typeof configuration === `undefined`)
        return getDefaultTaskPool();
    return miscUtils.getFactoryWithDefault(workerPools, configuration, () => {
        const poolMode = configuration.get(`taskPoolMode`);
        const poolSize = configuration.get(`taskPoolConcurrency`);
        switch (poolMode) {
            case `async`:
                return new TaskPool_1.AsyncPool(convertToZipWorker, { poolSize });
            case `workers`:
                return new TaskPool_1.WorkerPool((0, worker_zip_1.getContent)(), { poolSize });
            default: {
                throw new Error(`Assertion failed: Unknown value ${poolMode} for taskPoolMode`);
            }
        }
    });
}
async function convertToZipWorker(data) {
    const { tmpFile, tgz, compressionLevel, extractBufferOpts } = data;
    const zipFs = new libzip_1.ZipFS(tmpFile, { create: true, level: compressionLevel, stats: fslib_1.statUtils.makeDefaultStats() });
    // Buffers sent through Node are turned into regular Uint8Arrays
    const tgzBuffer = Buffer.from(tgz.buffer, tgz.byteOffset, tgz.byteLength);
    await extractArchiveTo(tgzBuffer, zipFs, extractBufferOpts);
    zipFs.saveAndClose();
    return tmpFile;
}
async function makeArchiveFromDirectory(source, { baseFs = new fslib_1.NodeFS(), prefixPath = fslib_1.PortablePath.root, compressionLevel, inMemory = false } = {}) {
    let zipFs;
    if (inMemory) {
        zipFs = new libzip_1.ZipFS(null, { level: compressionLevel });
    }
    else {
        const tmpFolder = await fslib_1.xfs.mktempPromise();
        const tmpFile = fslib_1.ppath.join(tmpFolder, `archive.zip`);
        zipFs = new libzip_1.ZipFS(tmpFile, { create: true, level: compressionLevel });
    }
    const target = fslib_1.ppath.resolve(fslib_1.PortablePath.root, prefixPath);
    await zipFs.copyPromise(target, source, { baseFs, stableTime: true, stableSort: true });
    return zipFs;
}
async function convertToZip(tgz, opts = {}) {
    const tmpFolder = await fslib_1.xfs.mktempPromise();
    const tmpFile = fslib_1.ppath.join(tmpFolder, `archive.zip`);
    const compressionLevel = opts.compressionLevel
        ?? opts.configuration?.get(`compressionLevel`)
        ?? `mixed`;
    const extractBufferOpts = {
        prefixPath: opts.prefixPath,
        stripComponents: opts.stripComponents,
    };
    const taskPool = opts.taskPool ?? getTaskPoolForConfiguration(opts.configuration);
    await taskPool.run({ tmpFile, tgz, compressionLevel, extractBufferOpts });
    return new libzip_1.ZipFS(tmpFile, { level: opts.compressionLevel });
}
async function* parseTar(tgz) {
    // @ts-expect-error - Types are wrong about what this function returns
    const parser = new tar_1.default.Parse();
    const passthrough = new stream_1.PassThrough({ objectMode: true, autoDestroy: true, emitClose: true });
    parser.on(`entry`, (entry) => {
        passthrough.write(entry);
    });
    parser.on(`error`, error => {
        passthrough.destroy(error);
    });
    parser.on(`close`, () => {
        if (!passthrough.destroyed) {
            passthrough.end();
        }
    });
    parser.end(tgz);
    for await (const entry of passthrough) {
        const it = entry;
        yield it;
        it.resume();
    }
}
async function extractArchiveTo(tgz, targetFs, { stripComponents = 0, prefixPath = fslib_1.PortablePath.dot } = {}) {
    function ignore(entry) {
        // Disallow absolute paths; might be malicious (ex: /etc/passwd)
        if (entry.path[0] === `/`)
            return true;
        const parts = entry.path.split(/\//g);
        // We also ignore paths that could lead to escaping outside the archive
        if (parts.some((part) => part === `..`))
            return true;
        if (parts.length <= stripComponents)
            return true;
        return false;
    }
    for await (const entry of parseTar(tgz)) {
        if (ignore(entry))
            continue;
        const parts = fslib_1.ppath.normalize(fslib_1.npath.toPortablePath(entry.path)).replace(/\/$/, ``).split(/\//g);
        if (parts.length <= stripComponents)
            continue;
        const slicePath = parts.slice(stripComponents).join(`/`);
        const mappedPath = fslib_1.ppath.join(prefixPath, slicePath);
        let mode = 0o644;
        // If a single executable bit is set, normalize so that all are
        if (entry.type === `Directory` || ((entry.mode ?? 0) & 0o111) !== 0)
            mode |= 0o111;
        switch (entry.type) {
            case `Directory`:
                {
                    targetFs.mkdirpSync(fslib_1.ppath.dirname(mappedPath), { chmod: 0o755, utimes: [fslib_1.constants.SAFE_TIME, fslib_1.constants.SAFE_TIME] });
                    targetFs.mkdirSync(mappedPath, { mode });
                    targetFs.utimesSync(mappedPath, fslib_1.constants.SAFE_TIME, fslib_1.constants.SAFE_TIME);
                }
                break;
            case `OldFile`:
            case `File`:
                {
                    targetFs.mkdirpSync(fslib_1.ppath.dirname(mappedPath), { chmod: 0o755, utimes: [fslib_1.constants.SAFE_TIME, fslib_1.constants.SAFE_TIME] });
                    targetFs.writeFileSync(mappedPath, await miscUtils.bufferStream(entry), { mode });
                    targetFs.utimesSync(mappedPath, fslib_1.constants.SAFE_TIME, fslib_1.constants.SAFE_TIME);
                }
                break;
            case `SymbolicLink`:
                {
                    targetFs.mkdirpSync(fslib_1.ppath.dirname(mappedPath), { chmod: 0o755, utimes: [fslib_1.constants.SAFE_TIME, fslib_1.constants.SAFE_TIME] });
                    targetFs.symlinkSync(entry.linkpath, mappedPath);
                    targetFs.lutimesSync(mappedPath, fslib_1.constants.SAFE_TIME, fslib_1.constants.SAFE_TIME);
                }
                break;
        }
    }
    return targetFs;
}
