'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var compare = require('@thi.ng/compare');
var equiv = require('@thi.ng/equiv');
var checks = require('@thi.ng/checks');
var errors = require('@thi.ng/errors');
var api = require('@thi.ng/api');
var random = require('@thi.ng/random');

const binarySearch = (buf, x, key = (x) => x, cmp = compare.compare, low = 0, high = buf.length - 1) => {
    const kx = key(x);
    while (low <= high) {
        const mid = (low + high) >>> 1;
        const c = cmp(key(buf[mid]), kx);
        if (c < 0) {
            low = mid + 1;
        }
        else if (c > 0) {
            high = mid - 1;
        }
        else {
            return mid;
        }
    }
    return -low - 1;
};
const binarySearchNumeric = (buf, x, cmp = compare.compareNumAsc, low = 0, high = buf.length - 1) => {
    while (low <= high) {
        const mid = (low + high) >>> 1;
        const c = cmp(buf[mid], x);
        if (c < 0) {
            low = mid + 1;
        }
        else if (c > 0) {
            high = mid - 1;
        }
        else {
            return mid;
        }
    }
    return -low - 1;
};
const binarySearch2 = (buf, x) => {
    let idx = buf[1] <= x ? 1 : 0;
    return buf[idx] === x ? idx : buf[0] < x ? -idx - 2 : -1;
};
const binarySearch4 = (buf, x) => {
    let idx = buf[2] <= x ? 2 : 0;
    idx |= buf[idx + 1] <= x ? 1 : 0;
    return buf[idx] === x ? idx : buf[0] < x ? -idx - 2 : -1;
};
const binarySearch8 = (buf, x) => {
    let idx = buf[4] <= x ? 4 : 0;
    idx |= buf[idx + 2] <= x ? 2 : 0;
    idx |= buf[idx + 1] <= x ? 1 : 0;
    return buf[idx] === x ? idx : buf[0] < x ? -idx - 2 : -1;
};
const binarySearch16 = (buf, x) => {
    let idx = buf[8] <= x ? 8 : 0;
    idx |= buf[idx + 4] <= x ? 4 : 0;
    idx |= buf[idx + 2] <= x ? 2 : 0;
    idx |= buf[idx + 1] <= x ? 1 : 0;
    return buf[idx] === x ? idx : buf[0] < x ? -idx - 2 : -1;
};
const binarySearch32 = (buf, x) => {
    let idx = buf[16] <= x ? 16 : 0;
    idx |= buf[idx + 4] <= x ? 8 : 0;
    idx |= buf[idx + 4] <= x ? 4 : 0;
    idx |= buf[idx + 2] <= x ? 2 : 0;
    idx |= buf[idx + 1] <= x ? 1 : 0;
    return buf[idx] === x ? idx : buf[0] < x ? -idx - 2 : -1;
};
const bsLT = (i) => (i < 0 ? -i - 2 : i - 1);
const bsLE = (i) => (i < 0 ? -i - 2 : i);
const bsGT = (i, n) => ((i = i < 0 ? -i - 1 : i + 1), i < n ? i : -1);
const bsGE = (i, n) => ((i = i < 0 ? -i - 1 : i), i < n ? i : -1);
const bsEQ = (i) => (i < 0 ? -1 : i);

const bisect = (src, i = src.length >>> 1) => [
    src.slice(0, i),
    src.slice(i),
];
const bisectWith = (src, pred) => {
    const i = src.findIndex(pred);
    return i >= 0 ? bisect(src, i) : [src, []];
};

const endsWith = (buf, needle, equiv$1 = equiv.equiv) => {
    let i = buf.length;
    let j = needle.length;
    if (i < j)
        return false;
    while ((--i, --j >= 0 && equiv$1(buf[i], needle[j]))) { }
    return j < 0;
};

const ensureIterable = (x) => {
    (x == null || !x[Symbol.iterator]) &&
        errors.illegalArgs(`value is not iterable: ${x}`);
    return x;
};

const ensureArray = (x) => checks.isArray(x) ? x : [...ensureIterable(x)];
const ensureArrayLike = (x) => checks.isArrayLike(x) ? x : [...ensureIterable(x)];

const find = (buf, x, equiv$1 = equiv.equiv) => {
    const i = findIndex(buf, x, equiv$1);
    return i !== -1 ? buf[i] : undefined;
};
const findIndex = (buf, x, equiv$1 = equiv.equiv) => {
    for (let i = buf.length; --i >= 0;) {
        if (equiv$1(x, buf[i]))
            return i;
    }
    return -1;
};

const fillRange = (buf, index = 0, start = 0, end = buf.length, step = end > start ? 1 : -1) => {
    if (step > 0) {
        for (; start < end; start += step)
            buf[index++] = start;
    }
    else {
        for (; start > end; start += step)
            buf[index++] = start;
    }
    return buf;
};

const fuzzyMatch = (domain, query, equiv$1 = equiv.equiv) => {
    const nd = domain.length;
    const nq = query.length;
    if (nq > nd) {
        return false;
    }
    if (nq === nd) {
        return equiv$1(query, domain);
    }
    next: for (let i = 0, j = 0; i < nq; i++) {
        const q = query[i];
        while (j < nd) {
            if (equiv$1(domain[j++], q)) {
                continue next;
            }
        }
        return false;
    }
    return true;
};

const isSorted = (arr, cmp = compare.compare, start = 0, end = arr.length) => {
    let prev = arr[start];
    while (++start < end) {
        const curr = arr[start];
        if (cmp(prev, curr) > 0)
            return false;
        prev = curr;
    }
    return true;
};

const insert = (buf, x, i, k = Infinity) => i < 0 || i >= k || k < 1 ? buf : insertUnsafe(buf, x, i, k);
const insertUnsafe = (buf, x, i, k = Infinity) => {
    let j = buf.length < k ? buf.length + 1 : k;
    for (; --j > i;)
        buf[j] = buf[j - 1];
    buf[i] = x;
    return buf;
};

const into = (dest, src, max = Infinity) => {
    for (let x of src) {
        if (--max < 0)
            break;
        dest.push(x);
    }
    return dest;
};

function* arrayIterator(buf, start = 0, end) {
    if (!buf)
        return;
    start = start;
    end === undefined && (end = buf.length);
    const step = start <= end ? 1 : -1;
    for (; start !== end; start += step) {
        yield buf[start];
    }
}

const eqStrict = (a, b) => a === b;
const levenshtein = (a, b, maxDist = Infinity, equiv = eqStrict) => {
    if (a === b) {
        return 0;
    }
    if (a.length > b.length) {
        const tmp = a;
        a = b;
        b = tmp;
    }
    let la = a.length;
    let lb = b.length;
    while (la > 0 && equiv(a[~-la], b[~-lb])) {
        la--;
        lb--;
    }
    let offset = 0;
    while (offset < la && equiv(a[offset], b[offset])) {
        offset++;
    }
    la -= offset;
    lb -= offset;
    if (la === 0 || lb < 3) {
        return lb;
    }
    let x = 0;
    let y;
    let minDist;
    let d0;
    let d1;
    let d2;
    let d3;
    let dd;
    let dy;
    let ay;
    let bx0;
    let bx1;
    let bx2;
    let bx3;
    const _min = (d0, d1, d2, bx, ay) => {
        return d0 < d1 || d2 < d1
            ? d0 > d2
                ? d2 + 1
                : d0 + 1
            : equiv(ay, bx)
                ? d1
                : d1 + 1;
    };
    const vector = [];
    for (y = 0; y < la; y++) {
        vector.push(y + 1, a[offset + y]);
    }
    const len = vector.length - 1;
    const lb3 = lb - 3;
    for (; x < lb3;) {
        bx0 = b[offset + (d0 = x)];
        bx1 = b[offset + (d1 = x + 1)];
        bx2 = b[offset + (d2 = x + 2)];
        bx3 = b[offset + (d3 = x + 3)];
        dd = x += 4;
        minDist = Infinity;
        for (y = 0; y < len; y += 2) {
            dy = vector[y];
            ay = vector[y + 1];
            d0 = _min(dy, d0, d1, bx0, ay);
            d1 = _min(d0, d1, d2, bx1, ay);
            d2 = _min(d1, d2, d3, bx2, ay);
            dd = _min(d2, d3, dd, bx3, ay);
            dd < minDist && (minDist = dd);
            vector[y] = dd;
            d3 = d2;
            d2 = d1;
            d1 = d0;
            d0 = dy;
        }
        if (minDist > maxDist)
            return Infinity;
    }
    for (; x < lb;) {
        bx0 = b[offset + (d0 = x)];
        dd = ++x;
        minDist = Infinity;
        for (y = 0; y < len; y += 2) {
            dy = vector[y];
            vector[y] = dd = _min(dy, d0, dd, bx0, vector[y + 1]);
            dd < minDist && (minDist = dd);
            d0 = dy;
        }
        if (minDist > maxDist)
            return Infinity;
    }
    return dd;
};
const normalizedLevenshtein = (a, b, maxDist = Infinity, equiv = eqStrict) => {
    const n = Math.max(a.length, b.length);
    return n > 0 ? levenshtein(a, b, maxDist, equiv) / n : 0;
};

const first = (buf) => buf[0];
const peek = (buf) => buf[buf.length - 1];

const swap = (arr, x, y) => {
    const t = arr[x];
    arr[x] = arr[y];
    arr[y] = t;
};
const multiSwap = (...xs) => {
    const [b, c, d] = xs;
    const n = xs.length;
    switch (n) {
        case 0:
            return swap;
        case 1:
            return (a, x, y) => {
                swap(a, x, y);
                swap(b, x, y);
            };
        case 2:
            return (a, x, y) => {
                swap(a, x, y);
                swap(b, x, y);
                swap(c, x, y);
            };
        case 3:
            return (a, x, y) => {
                swap(a, x, y);
                swap(b, x, y);
                swap(c, x, y);
                swap(d, x, y);
            };
        default:
            return (a, x, y) => {
                swap(a, x, y);
                for (let i = n; --i >= 0;)
                    swap(xs[i], x, y);
            };
    }
};

function quickSort(arr, _cmp = compare.compare, _swap = swap, start = 0, end = arr.length - 1) {
    if (start < end) {
        const pivot = arr[start + ((end - start) >> 1)];
        let s = start - 1;
        let e = end + 1;
        while (true) {
            do {
                s++;
            } while (_cmp(arr[s], pivot) < 0);
            do {
                e--;
            } while (_cmp(arr[e], pivot) > 0);
            if (s >= e)
                break;
            _swap(arr, s, e);
        }
        quickSort(arr, _cmp, _swap, start, e);
        quickSort(arr, _cmp, _swap, e + 1, end);
    }
    return arr;
}

const shuffleRange = (buf, start = 0, end = buf.length, rnd = random.SYSTEM) => {
    api.assert(start >= 0 && end >= start && end <= buf.length, `illegal range ${start}..${end}`);
    let n = end - start;
    const l = n;
    if (l > 1) {
        while (--n >= 0) {
            const a = (start + rnd.float(l)) | 0;
            const b = (start + rnd.float(l)) | 0;
            const t = buf[a];
            buf[a] = buf[b];
            buf[b] = t;
        }
    }
    return buf;
};
const shuffle = (buf, n = buf.length, rnd = random.SYSTEM) => shuffleRange(buf, 0, n, rnd);

const sortByCachedKey = (src, key, cmp = compare.compare) => {
    const keys = checks.isFunction(key) ? src.map(key) : key;
    api.assert(keys.length === src.length, `keys.length != src.length`);
    quickSort(keys, cmp, multiSwap(src));
    return src;
};

const startsWith = (buf, needle, equiv$1 = equiv.equiv) => {
    let i = buf.length;
    let j = needle.length;
    if (i < j)
        return false;
    while (-j >= 0 && equiv$1(buf[j], needle[j])) { }
    return j < 0;
};

const swizzle = (order) => {
    const [a, b, c, d, e, f, g, h] = order;
    switch (order.length) {
        case 0:
            return () => [];
        case 1:
            return (x) => [x[a]];
        case 2:
            return (x) => [x[a], x[b]];
        case 3:
            return (x) => [x[a], x[b], x[c]];
        case 4:
            return (x) => [x[a], x[b], x[c], x[d]];
        case 5:
            return (x) => [x[a], x[b], x[c], x[d], x[e]];
        case 6:
            return (x) => [x[a], x[b], x[c], x[d], x[e], x[f]];
        case 7:
            return (x) => [x[a], x[b], x[c], x[d], x[e], x[f], x[g]];
        case 8:
            return (x) => [x[a], x[b], x[c], x[d], x[e], x[f], x[g], x[h]];
        default:
            return (x) => {
                const res = [];
                for (let i = order.length; --i >= 0;) {
                    res[i] = x[order[i]];
                }
                return res;
            };
    }
};

exports.arrayIterator = arrayIterator;
exports.binarySearch = binarySearch;
exports.binarySearch16 = binarySearch16;
exports.binarySearch2 = binarySearch2;
exports.binarySearch32 = binarySearch32;
exports.binarySearch4 = binarySearch4;
exports.binarySearch8 = binarySearch8;
exports.binarySearchNumeric = binarySearchNumeric;
exports.bisect = bisect;
exports.bisectWith = bisectWith;
exports.bsEQ = bsEQ;
exports.bsGE = bsGE;
exports.bsGT = bsGT;
exports.bsLE = bsLE;
exports.bsLT = bsLT;
exports.endsWith = endsWith;
exports.ensureArray = ensureArray;
exports.ensureArrayLike = ensureArrayLike;
exports.ensureIterable = ensureIterable;
exports.fillRange = fillRange;
exports.find = find;
exports.findIndex = findIndex;
exports.first = first;
exports.fuzzyMatch = fuzzyMatch;
exports.insert = insert;
exports.insertUnsafe = insertUnsafe;
exports.into = into;
exports.isSorted = isSorted;
exports.levenshtein = levenshtein;
exports.multiSwap = multiSwap;
exports.normalizedLevenshtein = normalizedLevenshtein;
exports.peek = peek;
exports.quickSort = quickSort;
exports.shuffle = shuffle;
exports.shuffleRange = shuffleRange;
exports.sortByCachedKey = sortByCachedKey;
exports.startsWith = startsWith;
exports.swap = swap;
exports.swizzle = swizzle;
