"use strict";

var _interopRequireDefault = require("@babel/runtime-corejs3/helpers/interopRequireDefault").default;

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parse = parse;

var _concat = _interopRequireDefault(require("@babel/runtime-corejs3/core-js-stable/instance/concat"));

var _from = _interopRequireDefault(require("@babel/runtime-corejs3/core-js-stable/array/from"));

var _toConsumableArray2 = _interopRequireDefault(require("@babel/runtime-corejs3/helpers/toConsumableArray"));

var _classCallCheck2 = _interopRequireDefault(require("@babel/runtime-corejs3/helpers/classCallCheck"));

var _createClass2 = _interopRequireDefault(require("@babel/runtime-corejs3/helpers/createClass"));

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime-corejs3/helpers/defineProperty"));

var _error = require("./error");

/**
 * Parses a data exported by Ruby's `Marshal.load`.
 * @param buf A binary data to parse
 * @returns the decoded value
 * @throws {MarshalError} when the data contains an invalid format.
 */
function parse(buf) {
  return new Parser(buf).read();
}

var Parser = /*#__PURE__*/function () {
  function Parser(buf) {
    var index = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    (0, _classCallCheck2.default)(this, Parser);
    (0, _defineProperty2.default)(this, "symbols", []);
    (0, _defineProperty2.default)(this, "objects", []);
    this.buf = buf;
    this.index = index;
  }

  (0, _createClass2.default)(Parser, [{
    key: "read",
    value: function read() {
      this.symbols = [];
      this.objects = [];
      var major = this.readByte();
      var minor = this.readByte();

      if (major !== 4 || minor > 8) {
        var _context;

        throw new _error.MarshalError((0, _concat.default)(_context = "incompatible marshal file format (can't be read): format version 4.8 required; ".concat(major, ".")).call(_context, minor, " given"));
      }

      return this.readAny();
    }
  }, {
    key: "readAny",
    value: function readAny() {
      var tag = this.readByte();

      switch (tag) {
        // '"': an instance of String
        case 0x22:
          return this.entry(this.readString());
        // '/': an instance of Regexp

        case 0x2f:
          {
            var source = this.readString(); // Discard flags

            this.readByte();
            return this.entry(new RegExp(source));
          }
        // '0': nil

        case 0x30:
          return null;
        // ':': an instance of Symbol

        case 0x3a:
          {
            var sym = this.readString();
            this.symbols.push(sym);
            return sym;
          }
        // ';': symbol reference

        case 0x3b:
          {
            var symref = this.readInt();

            if (symref < 0 || symref >= this.symbols.length) {
              throw new _error.MarshalError("bad symbol");
            }

            return this.symbols[symref];
          }
        // '@': an object link

        case 0x40:
          {
            var objref = this.readInt();

            if (objref < 0 || objref >= this.objects.length) {
              throw new _error.MarshalError("dump format error (unlinked)");
            }

            return this.objects[objref];
          }
        // 'C': an instance of String/Regexp/Array/Hash subclass

        case 0x43:
          {
            // Discard class name
            this.readAny();
            return this.readAny();
          }
        // 'F': false

        case 0x46:
          return false;
        // 'I': add instance variables

        case 0x49:
          {
            var obj = this.readAny();
            var length = this.readInt();

            for (var i = 0; i < length; i++) {
              // Discard instance variables
              this.readAny();
              this.readAny();
            }

            return obj;
          }
        // 'M': a module or class (old format)

        case 0x4d:
          {
            // Discard class/module name
            this.readBytes();
            return this.entry({});
          }
        // 'S': an instance of a struct

        case 0x53:
          {
            // Discard class name
            this.readAny();

            var _length = this.readLength("struct");

            var struct = this.entry({});

            for (var _i = 0; _i < _length; _i++) {
              var key = this.readAny();
              var value = this.readAny(); // Discard non-String keys

              if (typeof key === "number" || typeof key === "string") {
                Object.defineProperty(struct, key, {
                  value: value,
                  writable: true,
                  configurable: true,
                  enumerable: true
                });
              }
            }

            return struct;
          }
        // 'T': true

        case 0x54:
          return true;
        // 'U': custom format (marshal_dump)

        case 0x55:
          {
            // Discard class name
            this.readAny();

            var _obj = this.entry({}); // Discard data


            this.readAny();
            return _obj;
          }
        // '[': an instance of Array

        case 0x5b:
          {
            var _length2 = this.readLength("array");

            var arr = this.entry([]);

            for (var _i2 = 0; _i2 < _length2; _i2++) {
              arr.push(this.readAny());
            }

            return arr;
          }
        // 'c': a class

        case 0x63:
          {
            // Discard class name
            this.readBytes();
            return this.entry({});
          }
        // 'd': TYPE_DATA

        case 0x64:
          throw new _error.MarshalError("unimplemented: TYPE_DATA");
        // 'e': TYPE_EXTENDED

        case 0x65:
          throw new _error.MarshalError("unimplemented: TYPE_EXTENDED");
        // 'f': an instance of Float

        case 0x66:
          {
            var s = this.readString();
            var f = s === "inf" ? Infinity : s === "-inf" ? -Infinity : s === "nan" ? NaN : parseFloat(s);
            return this.entry(f);
          }
        // 'i': an instance of Integer (small)

        case 0x69:
          return this.readInt();
        // 'l': an instance of Integer (large)

        case 0x6c:
          {
            var signChar = this.readByte();

            var _length3 = this.readLength("string") * 2;

            var sum = 0;
            var magnitude = signChar === 0x2d ? -1 : 1;

            for (var _i3 = 0; _i3 < _length3; _i3++) {
              sum += this.readByte() * magnitude;
              magnitude *= 256;
            }

            return this.entry(sum);
          }
        // 'm': a module

        case 0x6d:
          {
            // Discard module name
            this.readBytes();
            return this.entry({});
          }
        // 'o': a plain object

        case 0x6f:
          {
            // Discard class name
            this.readAny();

            var _obj2 = this.entry({});

            var _length4 = this.readInt();

            for (var _i4 = 0; _i4 < _length4; _i4++) {
              // Discard instance variables
              this.readAny();
              this.readAny();
            }

            return _obj2;
          }
        // 'u': old custom format (_dump)

        case 0x75:
          {
            // Discard class name
            this.readAny(); // Discard data

            this.readBytes();
            return this.entry({});
          }
        // '{': an instance of Hash (without default value)

        case 0x7b:
          {
            var _length5 = this.readLength("hash");

            var hash = this.entry({});

            for (var _i5 = 0; _i5 < _length5; _i5++) {
              var _key = this.readAny();

              var _value = this.readAny(); // Discard non-String keys


              if (typeof _key === "number" || typeof _key === "string") {
                Object.defineProperty(hash, _key, {
                  value: _value,
                  writable: true,
                  configurable: true,
                  enumerable: true
                });
              }
            }

            return hash;
          }
        // '}': an instance of Hash (with default value)

        case 0x7d:
          {
            var _length6 = this.readLength("hash");

            var _hash = this.entry({});

            for (var _i6 = 0; _i6 < _length6; _i6++) {
              var _key2 = this.readAny();

              var _value2 = this.readAny(); // Discard non-String keys


              if (typeof _key2 === "number" || typeof _key2 === "string") {
                Object.defineProperty(_hash, _key2, {
                  value: _value2,
                  writable: true,
                  configurable: true,
                  enumerable: true
                });
              }
            }

            _hash["__ruby_default"] = this.readAny();
            return _hash;
          }

        default:
          throw new _error.MarshalError("dump format error(0x".concat(tag.toString(16), ")"));
      }
    }
  }, {
    key: "readLength",
    value: function readLength(msg) {
      var length = this.readInt();

      if (length < 0) {
        throw new _error.MarshalError("negative ".concat(msg, " size (or size too big)"));
      }

      return length;
    }
  }, {
    key: "readInt",
    value: function readInt() {
      var tag = this.readByte();

      if (tag === 0) {
        return 0;
      }

      if (tag >= 5 && tag < 128) {
        return tag - 5;
      }

      if (tag >= 128 && tag <= 251) {
        return tag - 251;
      }

      var length = tag < 128 ? tag : 256 - tag;
      var sum = 0;
      var magnitude = 1;

      for (var i = 0; i < length; i++) {
        sum += magnitude * this.readByte();
        magnitude *= 256;
      }

      if (tag >= 128) {
        sum -= magnitude;
      }

      return sum;
    }
  }, {
    key: "readByte",
    value: function readByte() {
      if (this.index >= this.buf.byteLength) {
        throw new _error.MarshalError("marshal data too short");
      }

      var byte = this.buf[this.index];
      this.index++;
      return byte;
    }
  }, {
    key: "readString",
    value: function readString() {
      var bytes = this.readBytes();

      if (typeof TextDecoder !== "function" && bytes.every(function (x) {
        return x < 128;
      })) {
        return String.fromCharCode.apply(String, (0, _toConsumableArray2.default)((0, _from.default)(bytes)));
      }

      return new TextDecoder().decode(bytes);
    }
  }, {
    key: "readBytes",
    value: function readBytes() {
      var length = this.readLength("string");

      if (this.index + length > this.buf.byteLength) {
        throw new _error.MarshalError("marshal data too short");
      }

      var bytes = this.buf.subarray(this.index, this.index + length);
      this.index += length;
      return bytes;
    }
  }, {
    key: "entry",
    value: function entry(obj) {
      this.objects.push(obj);
      return obj;
    }
  }]);
  return Parser;
}();