"use strict";
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.wrapTerraformExpression = exports.replaceQuotes = void 0;
/**
 * This function is used to escape quotes within a string.
 *  Note: This is a really naive implmentation that will not be able to work with
 *    nested templates. However, nested templates require parsing, so we will
 *    only handle strings without nested templates for the moment
 * @param input The string to escape
 */
function replaceQuotes(input) {
    const fullyQuotedMatch = /^(\s*")(.*)("\s*)$/.exec(input);
    const hasTemplate = /(?<!\$)\$\{/.test(input);
    if (fullyQuotedMatch !== null && !hasTemplate) {
        return `${fullyQuotedMatch[1]}${fullyQuotedMatch[2].replace(/"/g, `\\"`)}${fullyQuotedMatch[3]}`;
    }
    return input;
}
exports.replaceQuotes = replaceQuotes;
/**
 * This function is used to wrap a string in quotes conditionally.
 *  - If the string is already quoted, it will be returned as-is.
 *  - If the string contains a newline, it will be wrapped in a heredoc.
 *  - If the string is a number, it will be returned as-is.
 *  - If the string is a boolean, it will be returned as-is.
 *  - If the string is a list or map, it will be returned as-is.
 *
 * @param input The string to wrap
 * @returns The wrapped string, and the offset of the cursor within the string
 *
 * @example
 * wrapTerraformExpression("foo") // { wrap: '"foo"', wrapOffset: 1 }
 * wrapTerraformExpression("foo\nbar") // { wrap: '<<EOE\nfoo\nbar\nEOE\n', wrapOffset: 5 }
 * wrapTerraformExpression("22") // { wrap: '22', wrapOffset: 0 }
 * wrapTerraformExpression("true") // { wrap: 'true', wrapOffset: 0 }
 * wrapTerraformExpression("[22, 10]") // { wrap: '[22, 10]', wrapOffset: 0 }
 * wrapTerraformExpression("{ foo = 22 }") // { wrap: '{ foo = 22 }', wrapOffset: 0 }
 */
function wrapTerraformExpression(input) {
    input = replaceQuotes(input);
    if (!isNaN(Number(input))) {
        return { wrap: input, wrapOffset: 0 };
    }
    if (input === "true" || input === "false") {
        return { wrap: input, wrapOffset: 0 };
    }
    if (input.startsWith(`"`)) {
        if (input.indexOf("\n") >= 0) {
            const trimWrapped = input.substring(1, input.length - 1);
            return {
                wrap: `<<EOE\n${trimWrapped}\nEOE\n`,
                wrapOffset: 5,
            };
        }
        return { wrap: input, wrapOffset: 0 };
    }
    if (input.startsWith("[") || input.startsWith("{")) {
        return { wrap: input, wrapOffset: 0 };
    }
    if (input.startsWith("<<")) {
        // For Heredocs, we need to ensure that the string ends with an empty newline as
        // the HCL parser doesn't find the ending token otherwise
        if (!input.endsWith("\n")) {
            return { wrap: input + "\n", wrapOffset: 0 };
        }
        return { wrap: input, wrapOffset: 0 };
    }
    return { wrap: `"${input}"`, wrapOffset: 1 };
}
exports.wrapTerraformExpression = wrapTerraformExpression;
//# sourceMappingURL=data:application/json;base64,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