/**
 * All platform names supported by this library.
 *
 * @public
 */
declare const platformNames: readonly ["noarch", "linux-32", "linux-64", "linux-aarch64", "linux-armv6l", "linux-armv7l", "linux-ppc64le", "linux-ppc64", "linux-ppc", "linux-s390x", "linux-riscv32", "linux-riscv64", "osx-64", "osx-arm64", "win-32", "win-64", "win-arm64", "emscripten-wasm32", "wasi-wasm32", "zos-z"];
/**
 * A type that represents a valid platform.
 *
 * @public
 */
type Platform = (typeof platformNames)[number];
/**
 * A type guard that identifies if an input value is a `Platform`
 *
 * @public
 */
declare function isPlatform(maybePlatform: unknown): maybePlatform is Platform;
/**
 * All architecture names supported by this library.
 *
 * @public
 */
declare const archNames: readonly ["x86", "x86_64", "aarch64", "arm64", "armv6l", "armv7l", "ppc64le", "ppc64", "ppc", "s390x", "riscv32", "riscv64", "wasm32", "z"];
/**
 * A type that represents a valid architecture.
 *
 * @public
 */
type Arch = (typeof archNames)[number];
/**
 * A type guard that identifies if an input value is an `Arch`
 *
 * @public
 */
declare function isArch(maybeArch: unknown): maybeArch is Platform;
/**
 * Returns the architecture of a certain platform
 *
 * @param platform - The platform
 * @public
 */
declare function platformArch(platform: Platform): Arch | null;

/* tslint:disable */
/* eslint-disable */
/**
 * Solve a set of specs with the given channels and platforms.
 * @public
 * @param specs - Matchspecs of packages that must be included.
 * @param channels - The channels to request for repodata of packages
 * @param platforms - The platforms to solve for
 */
declare function simpleSolve(specs: string[], channels: string[], platforms: Platform[]): Promise<SolvedPackage[]>;


/**
 * Noarch packages are packages that are not architecture specific and therefore
 * only have to be built once. A `NoArchType` is either specific to an
 * architecture or not.
 *
 * @public
 */
declare type NoArchType = undefined | true | "python" | "generic";


/** A unique symbol used for branding `PackageName` types. */
declare const PACKAGE_NAME_BRAND: unique symbol;

/**
 * A **branded type** representing a validated package name.
 *
 * - This type is **enforced at runtime** using `isPackageName()`.
 * - Ensures that a package name conforms to the expected format.
 *
 * @example
 *
 * ```ts
 * const pkg: PackageName = "valid-package" as PackageName;
 * ```
 *
 * @public
 */
declare type PackageName = string & {
    [PACKAGE_NAME_BRAND]: void;
};

/**
 * Defines the allowed characters for any package name.
 *
 * Allowed characters:
 *
 * - Lowercase letters (`a-z`)
 * - Uppercase letters (`A-Z`)
 * - Digits (`0-9`)
 * - Underscore (`_`)
 * - Dash (`-`)
 * - Dot (`.`)
 *
 * @public
 */
declare type PackageNameChar =
    | "a"
    | "b"
    | "c"
    | "d"
    | "e"
    | "f"
    | "g"
    | "h"
    | "i"
    | "j"
    | "k"
    | "l"
    | "m"
    | "n"
    | "o"
    | "p"
    | "q"
    | "r"
    | "s"
    | "t"
    | "u"
    | "v"
    | "w"
    | "x"
    | "y"
    | "z"
    | "A"
    | "B"
    | "C"
    | "D"
    | "E"
    | "F"
    | "G"
    | "H"
    | "I"
    | "J"
    | "K"
    | "L"
    | "M"
    | "N"
    | "O"
    | "P"
    | "Q"
    | "R"
    | "S"
    | "T"
    | "U"
    | "V"
    | "W"
    | "X"
    | "Y"
    | "Z"
    | "0"
    | "1"
    | "2"
    | "3"
    | "4"
    | "5"
    | "6"
    | "7"
    | "8"
    | "9"
    | "_"
    | "-"
    | ".";

/**
 * Ensures that a string is a valid package name.
 *
 * - If `S` contains only valid characters and is not empty, it resolves to `S`.
 * - Otherwise, it resolves to `never`.
 *
 * @example
 *
 * ```ts
 * type Valid = PackageNameLiteral<"valid-name">; // 'valid-name'
 * type Invalid = PackageNameLiteral<"invalid!">; // never
 * type Empty = PackageNameLiteral<"">; // never
 * ```
 *
 * @public
 */
declare type PackageNameLiteral<S extends string> =
    ContainsOnlyPackageNameChars<S> & NonEmptyString<S>;

/**
 * A type that accepts:
 *
 * - A **PackageName** (a runtime-validated string).
 * - A **string literal** that satisfies `PackageNameLiteral<S>`.
 *
 * This is useful for functions that accept both validated runtime values and
 * compile-time checked literals.
 *
 * @example
 *
 * ```ts
 * function processPackage(name: PackageNameOrLiteral) { ... }
 *
 * processPackage("valid-package"); // ✅ Allowed (checked at compile-time)
 * processPackage("invalid!"); // ❌ Compile-time error
 * ```
 *
 * @param S - The input string type.
 * @public
 */
declare type PackageNameOrLiteral<S extends string = string> =
    | PackageName
    | (S extends PackageNameLiteral<S> ? S : never);

/**
 * Checks whether a string consists only of valid package name characters.
 *
 * - If `S` contains only allowed characters, it resolves to `S`.
 * - Otherwise, it resolves to `never`.
 *
 * @example
 *
 * ```ts
 * type Valid = ContainsOnlyPackageNameChars<"valid-name">; // 'valid-name'
 * type Invalid = ContainsOnlyPackageNameChars<"invalid!">; // never
 * ```
 *
 * @public
 */
declare type ContainsOnlyPackageNameChars<S extends string> =
    S extends ""
        ? ""
        : S extends `${infer First}${infer Rest}`
          ? First extends PackageNameChar
              ? ContainsOnlyPackageNameChars<Rest> extends never
                  ? never
                  : S
              : never
          : never;

/**
 * Ensures that a string is non-empty.
 *
 * - If `T` is an empty string, it resolves to `never`.
 * - Otherwise, it resolves to `T`.
 *
 * @example
 *
 * ```ts
 * type Valid = NonEmptyString<"hello">; // 'hello'
 * type Invalid = NonEmptyString<"">; // never
 * ```
 *
 * @public
 */
declare type NonEmptyString<T extends string> = T extends "" ? never : T;


/**
 * A package record is a JSON object that contains metadata about a package.
 *
 * @public
 */
declare type PackageRecordJson = {
    name: string;
    version: string;
    build: string;
    build_number: number;
    subdir: Platform;
    arch?: Arch;
    constrains?: string[];
    depends?: string[];
    features?: string;
    license?: string;
    license_family?: string;
    md5?: string;
    legacy_bz2_md5?: string;
    legacy_bz2_size?: number;
    sha256?: string;
    platform?: string;
    python_site_packages_path?: string;
    size?: number;
    noarch?: NoArchType;
    timestamp?: number;
    track_features?: string;
};



/**
 * Defines how strict a parser should be when parsing an object from a string.
 *
 * @public
 */
type ParseStrictness = "strict" | "lenient";
/**
 * A single record in the Conda repodata. A single record refers to a single
 * binary distribution of a package on a Conda channel.
 *
 * @public
 */
declare class PackageRecord {
  free(): void;
  /**
   * Constructs a new instance from the json representation of a
   * PackageRecord.
   */
  constructor(json: PackageRecordJson);
  /**
   * Convert this instance to the canonical json representation of a
   * PackageRecord.
   */
  toJson(): PackageRecordJson;
  /**
   * Optionally the architecture the package supports. This is almost
   * always the second part of the `subdir` field. Except for `64` which
   * maps to `x86_64` and `32` which maps to `x86`. This will be undefined if
   * the package is `noarch`.
   */
  arch: Arch | undefined;
  /**
   * The build string of the package.
   */
  build: string;
  /**
   * The build number of the package.
   */
  buildNumber: number;
  /**
   * Additional constraints on packages. `constrains` are different from
   * `depends` in that packages specified in `depends` must be installed
   * next to this package, whereas packages specified in `constrains` are
   * not required to be installed, but if they are installed they must follow
   * these constraints.
   */
  constrains: string[];
  /**
   * Specification of packages this package depends on
   */
  depends: string[];
  /**
   * Features are a deprecated way to specify different feature sets for the
   * conda solver. This is not supported anymore and should not be used.
   * Instead, `mutex` packages should be used to specify
   * mutually exclusive features.
   */
  features: string | undefined;
  /**
   * The specific license of the package
   */
  license: string | undefined;
  /**
   * The license family
   */
  licenseFamily: string | undefined;
  /**
   * The name of the package
   */
  name: PackageName;
  /**
   * Optionally a MD5 hash of the package archive encoded as a hex string
   */
  md5: string | undefined;
  /**
   * Deprecated md5 hash
   * @deprecated Use `md5` instead
   */
  legacyBz2Md5: string | undefined;
  /**
   * Deprecated size of the package archive.
   * @deprecated Use `size` instead
   */
  legacyBz2Size: number | undefined;
  /**
   * Optionally a Sha256 hash of the package archive encoded as a hex string
   */
  sha256: string | undefined;
  /**
   * Optionally the platform the package supports.
   * Note that this does not match the `Platform` type, but is only the first
   * part of the platform (e.g. `linux`, `osx`, `win`, ...).
   * The `subdir` field contains the `Platform` enum.
   */
  platform: string | undefined;
  /**
   * Optionally a path within the environment of the site-packages directory.
   * This field is only present for python interpreter packages.
   */
  pythonSitePackagesPath: string | undefined;
  /**
   * The size of the package archive in bytes
   */
  size: number | undefined;
  /**
   * The subdirectory where the package can be found
   */
  subdir: Platform;
  /**
   * If this package is independent of architecture this field specifies in
   * what way. See [`NoArchType`] for more information.
   */
  noarch: NoArchType;
  /**
   * The date this entry was created.
   */
  timestamp: Date | undefined;
  /**
   * Track features are nowadays only used to downweight packages (ie. give
   * them less priority). To that effect, the package is downweighted
   * by the number of track_features.
   */
  trackFeatures: string[];
  /**
   * The version of the package
   */
  version: VersionWithSource;
}
/**
 * A package that has been solved by the solver.
 * @public
 */
declare class SolvedPackage {
  private constructor();
  free(): void;
  url: string;
  packageName: string;
  build: string;
  buildNumber: bigint;
  get repoName(): string | undefined;
  set repoName(value: string | null | undefined);
  filename: string;
  version: string;
}
/**
 * This class implements an order relation between version strings. Version
 * strings can contain the usual alphanumeric characters (A-Za-z0-9), separated
 * into segments by dots and underscores. Empty segments (i.e. two consecutive
 * dots, a leading/trailing underscore) are not permitted. An optional epoch
 * number - an integer followed by '!' - can precede the actual version string
 * (this is useful to indicate a change in the versioning scheme itself).
 * Version comparison is case-insensitive.
 *
 * @public
 */
declare class Version {
  free(): void;
  /**
   * Constructs a new Version object from a string representation.
   * @param version - The string representation of the version.
   */
  constructor(version: string);
  /**
   * Returns the string representation of the version.
   *
   * An attempt is made to return the version in the same format as the input
   * string but this is not guaranteed.
   */
  toString(): string;
  /**
   * Returns the major and minor part of the version if the version does not
   * represent a typical major minor version. If any of the parts are not a
   * single number, undefined is returned.
   */
  asMajorMinor(): [number, number] | undefined;
  /**
   * Returns true if this version starts with the other version. This is
   * defined as the other version being a prefix of this version.
   * @param other - The version to use for the comparison
   */
  startsWith(other: Version): boolean;
  /**
   * Returns true if this version is compatible with the other version.
   * @param other - The version to use for the comparison
   */
  compatibleWith(other: Version): boolean;
  /**
   * Pop the last `n` segments from the version.
   * @param n - The number of segments to pop
   */
  popSegments(n: number): Version | undefined;
  /**
   * Extend the version to the given length by adding zeros. If the version
   * is already at the specified length or longer the original version
   * will be returned.
   * @param length - The length to extend to
   */
  extendToLength(length: number): Version;
  /**
   * Returns a new version with the segments from start to end (exclusive).
   *
   * Returns undefined if the start or end index is out of bounds.
   * @param start - The start index
   * @param end - The end index
   */
  withSegments(start: number, end: number): Version | undefined;
  /**
   * Returns the version without the local part. E.g. `1+2.3` becomes `1`.
   */
  stripLocal(): Version;
  /**
   * Returns a new version where the major segment of this version has been
   * bumped.
   */
  bumpMajor(): Version;
  /**
   * Returns a new version where the minor segment of this version has been
   * bumped.
   */
  bumpMinor(): Version;
  /**
   * Returns a new version where the patch segment of this version has been
   * bumped.
   */
  bumpPatch(): Version;
  /**
   * Returns a new version where the last segment of this version has been
   * bumped.
   */
  bumpLast(): Version;
  /**
   * Returns a new version where the given segment of this version has been
   * bumped.
   * @param index - The index of the segment to bump
   */
  bumpSegment(index: number): Version;
  /**
   * Returns a new version where the last segment is an "alpha" segment (ie.
   * `.0a0`)
   */
  withAlpha(): Version;
  /**
   * Compares this version with another version. Returns `true` if the
   * versions are considered equal.
   *
   * Note that two version strings can be considered equal even if they are
   * not exactly the same. For example, `1.0` and `1` are considered equal.
   * @param other - The version to compare with
   */
  equals(other: Version): boolean;
  /**
   * Compare two versions.
   *
   * Returns `-1` if this instance should be ordered before `other`, `0` if
   * this version and `other` are considered equal, `1` if this version
   * should be ordered after `other`.
   * @param other - The version to compare with
   */
  compare(other: Version): number;
  /**
   * The epoch part of the version. E.g. `1` in `1!2.3`.
   */
  readonly epoch: number | undefined;
  /**
   * `true` if the version has a local part. E.g. `2.3` in `1+2.3`.
   */
  readonly hasLocal: boolean;
  /**
   * `true` if the version is considered a development version.
   *
   * A development version is a version that contains the string `dev` in the
   * version string.
   */
  readonly isDev: boolean;
  /**
   * The number of segments in the version.
   */
  readonly length: number;
}
/**
 * Represents a version specification in the conda ecosystem.
 *
 * @public
 */
declare class VersionSpec {
  free(): void;
  /**
   * Constructs a new VersionSpec object from a string representation.
   * @param version_spec - The string representation of the version spec.
   * @param parse_strictness - The strictness of the parser.
   */
  constructor(version_spec: string, parse_strictness?: ParseStrictness | null);
  /**
   * Returns the string representation of the version spec.
   */
  toString(): string;
  /**
   * Returns true if the version matches this version spec.
   * @param version - The version to match
   */
  matches(version: Version): boolean;
}
/**
 * Holds a version and the string it was created from. This is useful if you
 * want to retain the original string the version was created from. This might
 * be useful in cases where you have multiple strings that are represented by
 * the same `Version` but you still want to be able to distinguish them.
 *
 * The string `1.0` and `1.01` represent the same version. When you print the
 * parsed version though it will come out as `1.0`. You loose the original
 * representation. This struct stores the original source string.
 *
 * @public
 */
declare class VersionWithSource {
  free(): void;
  constructor(source: string);
  readonly version: Version;
  readonly source: string;
}

/**
 * A **branded type** representing a **normalized** package name.
 *
 * - A `NormalizedPackageName` is always **lowercase**.
 * - It extends `PackageName`, ensuring that it still follows package name rules.
 * - Can be obtained by calling `normalizePackageName()`.
 *
 * @example
 *
 * ```ts
 * const normalized: NormalizedPackageName =
 *     "valid-package" as NormalizedPackageName;
 * ```
 *
 * @public
 */
type NormalizedPackageName = Lowercase<PackageName>;
/**
 * A type that accepts:
 *
 * - A **NormalizedPackageName** (a runtime-validated string).
 * - A **string literal** that satisfies `Lowercase<PackageNameLiteral<S>>`.
 *
 * This is useful for functions that accept both validated runtime values and
 * compile-time checked literals.
 *
 * @example
 *
 * ```ts
 * function processNormalizedPackage(name: NormalizedPackageNameOrLiteral) { ... }
 *
 * processNormalizedPackage("valid-package"); // ✅ Allowed (checked at compile-time)
 * processNormalizedPackage("Invalid-Package"); // ❌ Compile-time error
 * ```
 *
 * @param S - The input string type.
 * @public
 */
type NormalizedPackageNameOrLiteral<S extends string = string> = NormalizedPackageName | (S extends Lowercase<PackageNameLiteral<S>> ? S : never);
/**
 * **Normalizes a package name to lowercase.**
 *
 * - If given a **string literal**, it is validated at compile time.
 * - If given a **runtime-validated** `PackageName`, it is accepted directly.
 * - Returns a `NormalizedPackageName` with all characters converted to lowercase.
 *
 * @example
 *
 * ```ts
 * const normalized = normalizePackageName("Valid-Package"); // "valid-package"
 * ```
 *
 * @param name - The package name to normalize.
 * @returns The normalized package name.
 * @public
 */
declare function normalizePackageName<T extends string>(name: PackageNameOrLiteral<T>): NormalizedPackageName;
/**
 * **Checks if a string is a valid `PackageName`.**
 *
 * - Returns `true` if `input` matches the allowed package name format.
 * - If `true`, TypeScript narrows the type to `PackageName<string>`.
 *
 * @example
 *
 * ```ts
 * if (isPackageName(userInput)) {
 *     const validName: PackageName = userInput;
 * }
 * ```
 *
 * @param input - The string to validate.
 * @returns `true` if valid, otherwise `false`.
 * @public
 */
declare function isPackageName(input: string): input is PackageName;
/**
 * **Checks if a string is a valid `NormalizedPackageName`.**
 *
 * - A normalized package name must be **lowercase**.
 * - If `true`, TypeScript narrows the type to `NormalizedPackageName<string>`.
 *
 * @example
 *
 * ```ts
 * if (isNormalizedPackageName(userInput)) {
 *     const validNormalizedName: NormalizedPackageName = userInput;
 * }
 * ```
 *
 * @param input - The string to validate.
 * @returns `true` if valid, otherwise `false`.
 * @public
 */
declare function isNormalizedPackageName(input: string): input is NormalizedPackageName;
/**
 * Parses a string and returns it as a `PackageName` if it is valid.
 *
 * @param input - The string to parse.
 * @returns The parsed `PackageName`.
 * @throws Will throw an error if the input is not a valid package name.
 */
declare function parsePackageName(input: string): PackageName;

export { type Arch, type ContainsOnlyPackageNameChars, type NoArchType, type NonEmptyString, type NormalizedPackageName, type NormalizedPackageNameOrLiteral, type PackageName, type PackageNameChar, type PackageNameLiteral, type PackageNameOrLiteral, PackageRecord, type PackageRecordJson, type ParseStrictness, type Platform, SolvedPackage, Version, VersionSpec, VersionWithSource, archNames, isArch, isNormalizedPackageName, isPackageName, isPlatform, normalizePackageName, parsePackageName, platformArch, platformNames, simpleSolve };
