"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.tryReuseAutoclosedPr = tryReuseAutoclosedPr;
const luxon_1 = require("luxon");
const global_1 = require("../../../../config/global");
const logger_1 = require("../../../../logger");
const platform_1 = require("../../../../modules/platform");
const REOPEN_THRESHOLD_MILLIS = 1000 * 60 * 60 * 24 * 7;
async function tryReuseAutoclosedPr(branchName) {
    if (!platform_1.platform.tryReuseAutoclosedPr) {
        return null;
    }
    const autoclosedPr = await platform_1.platform.findPr({ branchName, state: 'closed' });
    if (!autoclosedPr) {
        return null;
    }
    if (!autoclosedPr.title.endsWith(' - autoclosed')) {
        return null;
    }
    const closedAt = autoclosedPr.closedAt;
    if (!closedAt) {
        return null;
    }
    const closedMillisAgo = luxon_1.DateTime.fromISO(closedAt)
        .diffNow()
        .negate()
        .toMillis();
    if (closedMillisAgo > REOPEN_THRESHOLD_MILLIS) {
        logger_1.logger.debug(`Found autoclosed PR ${autoclosedPr.number} but it is too old to reopen`);
        return null;
    }
    logger_1.logger.debug({ number: autoclosedPr.number }, 'Found autoclosed PR for branch');
    if (global_1.GlobalConfig.get('dryRun')) {
        logger_1.logger.info('DRY-RUN: Would try to reopen autoclosed PR');
        return null;
    }
    try {
        const pr = await platform_1.platform.tryReuseAutoclosedPr(autoclosedPr);
        return pr;
    }
    catch (err) {
        logger_1.logger.debug({ err }, `Error trying to reuse existing PR with branch=${branchName}`);
        return null;
    }
}
//# sourceMappingURL=pr-reuse.js.map