"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareLabels = prepareLabels;
exports.getChangedLabels = getChangedLabels;
exports.areLabelsModified = areLabelsModified;
exports.shouldUpdateLabels = shouldUpdateLabels;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const dequal_1 = require("dequal");
const logger_1 = require("../../../../logger");
const platform_1 = require("../../../../modules/platform");
const template = tslib_1.__importStar(require("../../../../util/template"));
/**
 * Filter labels that go over the maximum char limit, based on platform limits.
 */
function trimLabel(label, limit) {
    const trimmed = label.trim();
    if (trimmed.length <= limit) {
        return trimmed;
    }
    return trimmed.slice(0, limit).trim();
}
function prepareLabels(config) {
    const labelCharLimit = platform_1.platform.labelCharLimit?.() ?? 50;
    const labels = config.labels ?? [];
    const addLabels = config.addLabels ?? [];
    return [...new Set([...labels, ...addLabels])]
        .filter(is_1.default.nonEmptyStringAndNotWhitespace)
        .map((label) => template.compile(label, config))
        .filter(is_1.default.nonEmptyStringAndNotWhitespace)
        .map((label) => trimLabel(label, labelCharLimit))
        .sort();
}
/**
 * Determine changed labels between old and new label arrays.
 *
 * This function takes two arrays of labels, 'oldLabels' and 'newLabels', and calculates the labels
 * that need to be added and removed to transition from 'oldLabels' to 'newLabels'.
 */
function getChangedLabels(oldLabels, newLabels) {
    const labelsToAdd = newLabels?.filter((l) => !oldLabels?.includes(l));
    const labelsToRemove = oldLabels?.filter((l) => !newLabels?.includes(l));
    return [labelsToAdd, labelsToRemove];
}
/**
 * Check if labels in the PR have been modified.
 *
 * This function compares two arrays of labels, 'prInitialLabels' and 'prCurrentLabels',
 * to determine if they are different, indicating that labels in the PR have been modified.
 */
function areLabelsModified(prInitialLabels, prCurrentLabels) {
    const modified = !(0, dequal_1.dequal)(prInitialLabels.sort(), prCurrentLabels.sort());
    if (modified) {
        logger_1.logger.debug({ prInitialLabels, prCurrentLabels }, 'PR labels have been modified by user, skipping labels update');
    }
    return modified;
}
/**
 * Determine if labels should be updated in the Pull Request.
 */
function shouldUpdateLabels(prInitialLabels, prCurrentLabels, configuredLabels) {
    // If the 'labelsInDebugData' field is undefined
    // it means the PR was created before the update-labels logic was merged, and labels should not be updated.
    //  Reference: https://github.com/renovatebot/renovate/pull/25340
    if (!is_1.default.array(prInitialLabels)) {
        return false;
    }
    // If the labels are unchanged, they should not be updated
    if ((0, dequal_1.dequal)((configuredLabels ?? []).sort(), prInitialLabels.sort())) {
        return false;
    }
    // If the labels in the PR have been modified by the user, they should not be updated
    if (areLabelsModified(prInitialLabels, prCurrentLabels ?? [])) {
        logger_1.logger.debug('Labels have been modified by user - skipping labels update.');
        return false;
    }
    logger_1.logger.debug('Labels have been changed in repo config- updating labels.');
    return true;
}
//# sourceMappingURL=labels.js.map