"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubChangeLogSource = void 0;
const tslib_1 = require("tslib");
const node_url_1 = tslib_1.__importDefault(require("node:url"));
const global_1 = require("../../../../../../config/global");
const logger_1 = require("../../../../../../logger");
const hostRules = tslib_1.__importStar(require("../../../../../../util/host-rules"));
const source_1 = require("../source");
class GitHubChangeLogSource extends source_1.ChangeLogSource {
    constructor() {
        super('github', 'github-tags');
    }
    getAPIBaseUrl(config) {
        const baseUrl = this.getBaseUrl(config);
        return baseUrl.startsWith('https://github.com/')
            ? 'https://api.github.com/'
            : baseUrl + 'api/v3/';
    }
    getCompareURL(baseUrl, repository, prevHead, nextHead) {
        return `${baseUrl}${repository}/compare/${prevHead}...${nextHead}`;
    }
    shouldSkipPackage(config) {
        if (config.sourceUrl === 'https://github.com/DefinitelyTyped/DefinitelyTyped') {
            logger_1.logger.trace('No release notes for @types');
            return true;
        }
        return false;
    }
    hasValidToken(config) {
        const sourceUrl = config.sourceUrl;
        const parsedUrl = node_url_1.default.parse(sourceUrl);
        const host = parsedUrl.host;
        const manager = config.manager;
        const packageName = config.packageName;
        const url = sourceUrl.startsWith('https://github.com/')
            ? 'https://api.github.com/'
            : sourceUrl;
        const { token } = hostRules.find({
            hostType: 'github',
            url,
            readOnly: true,
        });
        // istanbul ignore if
        if (host && !token) {
            if (host.endsWith('.github.com') || host === 'github.com') {
                if (!global_1.GlobalConfig.get('githubTokenWarn')) {
                    logger_1.logger.debug({ manager, packageName, sourceUrl }, 'GitHub token warning has been suppressed. Skipping release notes retrieval');
                    return { isValid: false };
                }
                logger_1.logger.warn({ manager, packageName, sourceUrl }, 'No github.com token has been configured. Skipping release notes retrieval');
                return { isValid: false, error: 'MissingGithubToken' };
            }
            logger_1.logger.debug({ manager, packageName, sourceUrl }, 'Repository URL does not match any known github hosts - skipping changelog retrieval');
            return { isValid: false };
        }
        return { isValid: true };
    }
}
exports.GitHubChangeLogSource = GitHubChangeLogSource;
//# sourceMappingURL=source.js.map