"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.slugifyUrl = slugifyUrl;
exports.compareChangelogFilePath = compareChangelogFilePath;
const tslib_1 = require("tslib");
const slugify_1 = tslib_1.__importDefault(require("slugify"));
const regex_1 = require("../../../../../util/regex");
function slugifyUrl(url) {
    const r = (0, regex_1.regEx)(/[:/.]+/g);
    return (0, slugify_1.default)(url.replace(r, ' '));
}
/**
 * A comparator function to sort changelog files by preference,
 * prioritizing markdown files and plain text files over other file types.
 *
 * Avoid selecting files like `CHANGELOG.json` when `CHANGELOG.md` is available.
 * @see https://github.com/renovatebot/renovate/issues/25830
 *
 * @example
 * ```
 * const changelogFiles = [
 * 'CHANGELOG.json',
 * 'CHANGELOG',
 * 'CHANGELOG.md',
 * ].sort(compareChangelogFilePath);
 *
 * console.log(changelogFiles); // =>
 * [
 *  'CHANGELOG.md',
 *  'CHANGELOG',
 *  'CHANGELOG.json',
 * ]
 * ```
 *
 * @param a path to changelog file
 * @param b path to changelog file
 */
function compareChangelogFilePath(a, b) {
    const preferedChangelogRegexList = [
        /\.(?:md|markdown|mkd)$/i,
        /\.(?:txt|text)$/i,
    ];
    const aPreferedIndex = preferedChangelogRegexList.findIndex((f) => f.test(a));
    const bPreferedIndex = preferedChangelogRegexList.findIndex((f) => f.test(b));
    if (aPreferedIndex === bPreferedIndex) {
        return a.localeCompare(b);
    }
    else if (aPreferedIndex >= 0 && bPreferedIndex >= 0) {
        return aPreferedIndex - bPreferedIndex;
    }
    return aPreferedIndex >= 0 ? -1 : 1;
}
//# sourceMappingURL=common.js.map