"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPrUpdatesTable = getPrUpdatesTable;
const tslib_1 = require("tslib");
const logger_1 = require("../../../../../logger");
const regex_1 = require("../../../../../util/regex");
const template = tslib_1.__importStar(require("../../../../../util/template"));
function getRowDefinition(prBodyColumns, upgrade) {
    const res = [];
    if (upgrade.prBodyDefinitions) {
        for (const header of prBodyColumns) {
            const value = upgrade.prBodyDefinitions[header];
            res.push({ header, value });
        }
    }
    return res;
}
function getNonEmptyColumns(prBodyColumns, rows) {
    const res = [];
    for (const header of prBodyColumns) {
        for (const row of rows) {
            if (row[header]?.length) {
                if (!res.includes(header)) {
                    res.push(header);
                }
            }
        }
    }
    return res;
}
function getPrUpdatesTable(config) {
    if (config.prBodyColumns === undefined) {
        logger_1.logger.warn('getPrUpdatesTable - prBodyColumns is undefined');
        return '';
    }
    const tableKeyValuePairs = {};
    for (const upgrade of config.upgrades) {
        if (upgrade) {
            // Create a key based on the properties which are significant in the updates table
            const key = `${upgrade.depName ?? ''}_${upgrade.depType ?? ''}_${upgrade.newValue ?? ''}_${upgrade.newVersion ?? ''}_${upgrade.currentValue ?? ''}_${upgrade.currentVersion ?? ''}_${upgrade.updateType}`;
            const res = {};
            const rowDefinition = getRowDefinition(config.prBodyColumns, upgrade);
            for (const column of rowDefinition) {
                const { header, value } = column;
                try {
                    // istanbul ignore else
                    if (value) {
                        res[header] = template
                            .compile(value, upgrade)
                            .replace((0, regex_1.regEx)(/``/g), '');
                    }
                    else {
                        res[header] = '';
                    }
                }
                catch (err) /* istanbul ignore next */ {
                    logger_1.logger.warn({ header, value, err }, 'Handlebars compilation error');
                }
            }
            if (tableKeyValuePairs[key]) {
                // compare the duplicate upgrades as per their table values
                // and select one with better values
                tableKeyValuePairs[key] = compareTableValues(tableKeyValuePairs[key], res, config.prBodyColumns);
            }
            else {
                tableKeyValuePairs[key] = res;
            }
        }
    }
    const tableValues = Object.values(tableKeyValuePairs);
    const tableColumns = getNonEmptyColumns(config.prBodyColumns, tableValues);
    let res = '\n\nThis PR contains the following updates:\n\n';
    res += '| ' + tableColumns.join(' | ') + ' |\n';
    res += '|' + tableColumns.map(() => '---|').join('') + '\n';
    const rows = [];
    for (const row of tableValues) {
        let val = '|';
        for (const column of tableColumns) {
            const content = row[column]
                ? row[column]
                    .replace((0, regex_1.regEx)(/^@/), '@&#8203;')
                    .replace((0, regex_1.regEx)(/\|/g), '\\|')
                : '';
            val += ` ${content} |`;
        }
        val += '\n';
        rows.push(val);
    }
    const uniqueRows = [...new Set(rows)];
    res += uniqueRows.join('');
    res += '\n\n';
    return res;
}
// return the row with better table values
function compareTableValues(a, b, prBodyColumns) {
    let score = 0;
    for (const header of prBodyColumns) {
        if (!b[header] && !a[header]) {
            continue;
        }
        if (!b[header]) {
            score--;
            continue;
        }
        if (!a[header]) {
            score++;
            continue;
        }
        if (a[header] !== b[header]) {
            if (a[header].length < b[header].length) {
                score++;
            }
            else {
                score--;
            }
        }
    }
    return score > 0 ? b : a;
}
//# sourceMappingURL=updates-table.js.map