"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPrConfigDescription = getPrConfigDescription;
const tslib_1 = require("tslib");
const croner_1 = require("croner");
const cronstrue_1 = tslib_1.__importDefault(require("cronstrue"));
const emoji_1 = require("../../../../../util/emoji");
const string_1 = require("../../../../../util/string");
function getPrConfigDescription(config) {
    let prBody = `\n\n---\n\n### Configuration\n\n`;
    prBody += (0, emoji_1.emojify)(`:date: **Schedule**: `);
    prBody +=
        'Branch creation - ' + scheduleToString(config.schedule, config.timezone);
    prBody +=
        ', Automerge - ' +
            scheduleToString(config.automergeSchedule, config.timezone) +
            '.';
    prBody += '\n\n';
    prBody += (0, emoji_1.emojify)(':vertical_traffic_light: **Automerge**: ');
    if (config.automerge) {
        prBody += 'Enabled.';
    }
    else if (config.automergedPreviously) {
        prBody += 'Disabled because a matching PR was automerged previously.';
    }
    else {
        prBody +=
            'Disabled by config. Please merge this manually once you are satisfied.';
    }
    prBody += '\n\n';
    prBody += (0, emoji_1.emojify)(':recycle: **Rebasing**: ');
    if (config.rebaseWhen === 'behind-base-branch') {
        prBody += 'Whenever PR is behind base branch';
    }
    else if (config.rebaseWhen === 'never' || config.stopUpdating) {
        prBody += 'Never';
    }
    else {
        prBody += 'Whenever PR becomes conflicted';
    }
    prBody += `, or you tick the rebase/retry checkbox.\n\n`;
    if (config.recreateClosed) {
        prBody += (0, emoji_1.emojify)(`:ghost: **Immortal**: This PR will be recreated if closed unmerged. Get [config help](${config.productLinks?.help}) if that's undesired.\n\n`);
    }
    else {
        prBody += (0, emoji_1.emojify)(`:no_bell: **Ignore**: Close this PR and you won't be reminded about ${config.upgrades.length === 1 ? 'this update' : 'these updates'} again.\n\n`);
    }
    return prBody;
}
function scheduleToString(schedule, timezone) {
    let scheduleString = '';
    if (schedule && schedule[0] !== 'at any time') {
        scheduleString =
            getReadableCronSchedule(schedule) ?? `"${String(schedule)}"`;
        if (timezone) {
            scheduleString += ` in timezone ${timezone}`;
        }
        else {
            scheduleString += ` (UTC)`;
        }
    }
    else {
        scheduleString += 'At any time (no schedule defined)';
    }
    return scheduleString;
}
/**
 * Return human-readable cron schedule summary if the schedule is a valid cron
 * else return null
 */
function getReadableCronSchedule(scheduleText) {
    // assuming if one schedule is cron the others in the array will be cron too
    try {
        new croner_1.CronPattern(scheduleText[0]); // validate cron
        return scheduleText
            .map((cron) => (0, string_1.capitalize)(cronstrue_1.default
            .toString(cron, {
            throwExceptionOnParseError: false,
        })
            .replace('Every minute, ', '')) + ` ( ${cron} )`)
            .join(', ');
    }
    catch {
        return null;
    }
}
//# sourceMappingURL=config-description.js.map