"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkAutoMerge = checkAutoMerge;
// TODO #22198
const global_1 = require("../../../../config/global");
const logger_1 = require("../../../../logger");
const platform_1 = require("../../../../modules/platform");
const comment_1 = require("../../../../modules/platform/comment");
const scm_1 = require("../../../../modules/platform/scm");
const schedule_1 = require("../branch/schedule");
const status_checks_1 = require("../branch/status-checks");
async function checkAutoMerge(pr, config) {
    logger_1.logger.trace({ config }, 'checkAutoMerge');
    const { branchName, baseBranch, automergeType, automergeStrategy, pruneBranchAfterAutomerge, automergeComment, ignoreTests, rebaseRequested, } = config;
    // Return if PR not ready for automerge
    if (!(0, schedule_1.isScheduledNow)(config, 'automergeSchedule')) {
        logger_1.logger.debug(`PR automerge is off schedule`);
        return {
            automerged: false,
            prAutomergeBlockReason: 'off schedule',
        };
    }
    const isConflicted = config.isConflicted ??
        (await scm_1.scm.isBranchConflicted(baseBranch, branchName));
    if (isConflicted) {
        logger_1.logger.debug('PR is conflicted');
        return {
            automerged: false,
            prAutomergeBlockReason: 'Conflicted',
        };
    }
    if (!ignoreTests && pr.cannotMergeReason) {
        logger_1.logger.debug(`Platform reported that PR is not ready for merge. Reason: [${pr.cannotMergeReason}]`);
        return {
            automerged: false,
            prAutomergeBlockReason: 'PlatformNotReady',
        };
    }
    const branchStatus = await (0, status_checks_1.resolveBranchStatus)(branchName, !!config.internalChecksAsSuccess, config.ignoreTests);
    if (branchStatus !== 'green') {
        logger_1.logger.debug(`PR is not ready for merge (branch status is ${branchStatus})`);
        return {
            automerged: false,
            prAutomergeBlockReason: 'BranchNotGreen',
        };
    }
    if (await scm_1.scm.isBranchModified(branchName, baseBranch)) {
        logger_1.logger.debug('PR is ready for automerge but has been modified');
        return {
            automerged: false,
            prAutomergeBlockReason: 'BranchModified',
        };
    }
    if (automergeType === 'pr-comment') {
        // TODO: types (#22198)
        logger_1.logger.debug(`Applying automerge comment: ${automergeComment}`);
        // istanbul ignore if
        if (global_1.GlobalConfig.get('dryRun')) {
            logger_1.logger.info(`DRY-RUN: Would add PR automerge comment to PR #${pr.number}`);
            return {
                automerged: false,
                prAutomergeBlockReason: 'DryRun',
            };
        }
        if (rebaseRequested) {
            await (0, comment_1.ensureCommentRemoval)({
                type: 'by-content',
                number: pr.number,
                content: automergeComment,
            });
        }
        await (0, comment_1.ensureComment)({
            number: pr.number,
            topic: null,
            content: automergeComment,
        });
        return { automerged: true, branchRemoved: false };
    }
    // Let's merge this
    // istanbul ignore if
    if (global_1.GlobalConfig.get('dryRun')) {
        // TODO: types (#22198)
        logger_1.logger.info(`DRY-RUN: Would merge PR #${pr.number} with strategy "${automergeStrategy}"`);
        return {
            automerged: false,
            prAutomergeBlockReason: 'DryRun',
        };
    }
    // TODO: types (#22198)
    logger_1.logger.debug(`Automerging #${pr.number} with strategy ${automergeStrategy}`);
    const res = await platform_1.platform.mergePr({
        branchName,
        id: pr.number,
        strategy: automergeStrategy,
    });
    if (res) {
        logger_1.logger.info({ pr: pr.number, prTitle: pr.title }, 'PR automerged');
        if (!pruneBranchAfterAutomerge) {
            logger_1.logger.info('Skipping pruning of merged branch');
            return { automerged: true, branchRemoved: false };
        }
        let branchRemoved = false;
        try {
            await scm_1.scm.deleteBranch(branchName);
            branchRemoved = true;
        }
        catch (err) /* istanbul ignore next */ {
            logger_1.logger.warn({ branchName, err }, 'Branch auto-remove failed');
        }
        return { automerged: true, branchRemoved };
    }
    return {
        automerged: false,
        prAutomergeBlockReason: 'PlatformRejection',
    };
}
//# sourceMappingURL=automerge.js.map