"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getReconfigureBranchName = getReconfigureBranchName;
exports.setBranchStatus = setBranchStatus;
exports.getReconfigureConfig = getReconfigureConfig;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const platform_1 = require("../../../modules/platform");
const scm_1 = require("../../../modules/platform/scm");
const common_1 = require("../../../util/common");
const fs_1 = require("../../../util/fs");
const merge_1 = require("../init/merge");
function getReconfigureBranchName(prefix) {
    return `${prefix}reconfigure`;
}
async function setBranchStatus(branchName, description, state, context) {
    if (!is_1.default.nonEmptyString(context)) {
        // already logged this case when validating the status check
        return;
    }
    await platform_1.platform.setBranchStatus({
        branchName,
        context,
        description,
        state,
    });
}
async function getReconfigureConfig(branchName) {
    await scm_1.scm.checkoutBranch(branchName);
    const configFileName = await (0, merge_1.detectConfigFile)();
    if (configFileName === null) {
        logger_1.logger.debug('No config file found in reconfigure branch');
        return {
            ok: false,
            errMessage: 'Validation Failed - No config file found',
        };
    }
    const configFileRaw = await (0, fs_1.readLocalFile)(configFileName, 'utf8');
    if (configFileRaw === null) {
        return {
            ok: false,
            errMessage: 'Validation Failed - Invalid config file',
            configFileName,
        };
    }
    let configFileParsed;
    try {
        configFileParsed = (0, common_1.parseJson)(configFileRaw, configFileName);
        // no need to confirm renovate field in package.json we already do it in `detectConfigFile()`
        if (configFileName === 'package.json') {
            configFileParsed = configFileParsed.renovate;
        }
    }
    catch (err) {
        logger_1.logger.debug({ err }, 'Error while parsing config file');
        return {
            ok: false,
            errMessage: 'Validation Failed - Unparsable config file',
            configFileName,
        };
    }
    return { ok: true, config: configFileParsed, configFileName };
}
//# sourceMappingURL=utils.js.map