"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkReconfigureBranch = checkReconfigureBranch;
const global_1 = require("../../../config/global");
const secrets_1 = require("../../../config/secrets");
const logger_1 = require("../../../logger");
const platform_1 = require("../../../modules/platform");
const scm_1 = require("../../../modules/platform/scm");
const repository_1 = require("../../../util/cache/repository");
const git_1 = require("../../../util/git");
const inherited_1 = require("../init/inherited");
const merge_1 = require("../init/merge");
const process_1 = require("../process");
const comment_1 = require("./comment");
const reconfigure_cache_1 = require("./reconfigure-cache");
const utils_1 = require("./utils");
const validate_1 = require("./validate");
async function checkReconfigureBranch(config, repoConfig) {
    logger_1.logger.debug('checkReconfigureBranch()');
    if (global_1.GlobalConfig.get('platform') === 'local') {
        logger_1.logger.debug('Not attempting to reconfigure when running with local platform');
        return;
    }
    const context = config.statusCheckNames?.configValidation;
    const reconfigureBranch = (0, utils_1.getReconfigureBranchName)(config.branchPrefix);
    const branchExists = await scm_1.scm.branchExists(reconfigureBranch);
    // this is something the user initiates, skip if no branch exists
    if (!branchExists) {
        logger_1.logger.debug('No reconfigure branch found');
        (0, reconfigure_cache_1.deleteReconfigureBranchCache)(); // in order to remove cache when the branch has been deleted
        return;
    }
    const existingPr = await platform_1.platform.findPr({
        branchName: reconfigureBranch,
        state: 'open',
        includeOtherAuthors: true,
        targetBranch: config.defaultBranch,
    });
    const branchSha = (0, git_1.getBranchCommit)(reconfigureBranch);
    const cache = (0, repository_1.getCache)();
    const reconfigureCache = cache.reconfigureBranchCache;
    // only use valid cached information
    if (reconfigureCache?.reconfigureBranchSha === branchSha) {
        if (!existingPr || reconfigureCache.extractResult) {
            logger_1.logger.debug('Skipping validation check as branch sha is unchanged');
            return;
        }
    }
    const result = await (0, utils_1.getReconfigureConfig)(reconfigureBranch);
    if (!result.ok) {
        await (0, utils_1.setBranchStatus)(reconfigureBranch, result.errMessage, 'red', context);
        (0, reconfigure_cache_1.setReconfigureBranchCache)(branchSha, false);
        await scm_1.scm.checkoutBranch(config.defaultBranch);
        return;
    }
    const isValidConfig = await (0, validate_1.validateReconfigureBranch)(config, result.config, result.configFileName, existingPr);
    if (!isValidConfig) {
        logger_1.logger.debug('Found errors in reconfigure config. Skipping dependencies extraction');
        await scm_1.scm.checkoutBranch(config.defaultBranch);
        return;
    }
    if (!existingPr) {
        logger_1.logger.debug('No reconfigure pr found. Skipping dependencies extraction');
        await scm_1.scm.checkoutBranch(config.defaultBranch);
        return;
    }
    let extractResult;
    // Recompute config similar to repo config processing
    // Get non-global config from file config
    // Merge it with inherited and static repo config
    // Finally, merge the reconfigure config
    let newConfig = global_1.GlobalConfig.set((0, secrets_1.applySecretsAndVariablesToConfig)({
        config: repoConfig,
    }));
    newConfig.baseBranch = config.defaultBranch;
    newConfig.repoIsOnboarded = true;
    newConfig.errors = [];
    newConfig.warnings = [];
    try {
        newConfig = await (0, inherited_1.mergeInheritedConfig)(newConfig);
        newConfig = await (0, merge_1.mergeRenovateConfig)(newConfig, reconfigureBranch);
        await scm_1.scm.checkoutBranch(config.defaultBranch);
        extractResult = await (0, process_1.extractDependencies)(newConfig, false);
    }
    catch (err) {
        logger_1.logger.debug({ err }, 'Error while extracting dependencies using the reconfigure config');
        (0, reconfigure_cache_1.setReconfigureBranchCache)(branchSha, true);
        await scm_1.scm.checkoutBranch(config.defaultBranch); // being cautious
        return;
    }
    let commentEnsured = false;
    if (extractResult) {
        commentEnsured = await (0, comment_1.ensureReconfigurePrComment)(newConfig, extractResult.packageFiles, extractResult.branches, reconfigureBranch, existingPr);
    }
    // if comment is not added or updated
    // do not store extractResult in cache so that we re-process the reconfigure branch on next run and do not skip
    // istanbul ignore if: should rarely happen
    if (!commentEnsured) {
        extractResult = undefined;
    }
    await (0, utils_1.setBranchStatus)(reconfigureBranch, 'Validation Successful', 'green', context);
    (0, reconfigure_cache_1.setReconfigureBranchCache)(branchSha, true, extractResult);
    await scm_1.scm.checkoutBranch(config.defaultBranch); //being cautious
}
//# sourceMappingURL=index.js.map