"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.calculateMostRecentTimestamp = calculateMostRecentTimestamp;
const luxon_1 = require("luxon");
const logger_1 = require("../../../../logger");
const timestamp_1 = require("../../../../util/timestamp");
/**
 * Calculates the `mostRecentTimestamp` value for a set of releases.
 *
 * This function determines the highest release (a release with the highest version)
 * and checks if its timestamp is also the highest among all releases.
 * If so, it assigns that timestamp as the `mostRecentTimestamp` value in the result.
 * This helps identify if the package was abandoned.
 *
 * The function skips setting `mostRecentTimestamp` if:
 * - No releases could be determined as the highest (e.g. for invalid versions)
 * - The highest release is deprecated
 * - A lower version has a more recent timestamp than the highest version
 *
 * @returns The `ReleaseResult` value, potentially updated with a `mostRecentTimestamp` timestamp
 */
function calculateMostRecentTimestamp(versioningApi, releaseResult) {
    const { lookupName } = releaseResult;
    let highestRelease;
    for (const release of releaseResult.releases) {
        if (!highestRelease) {
            if (versioningApi.isVersion(release.version)) {
                highestRelease = release;
            }
            continue;
        }
        try {
            if (versioningApi.isGreaterThan(release.version, highestRelease.version)) {
                highestRelease = release;
                continue;
            }
        }
        catch {
            logger_1.logger.trace({ lookupName }, 'Error calculating "mostRecentTimestamp" value');
        }
    }
    if (!highestRelease) {
        logger_1.logger.trace({ lookupName }, 'Could not determine the highest release to calculate "mostRecentTimestamp" value');
        return releaseResult;
    }
    if (highestRelease.isDeprecated) {
        logger_1.logger.trace({ lookupName }, 'Highest release is deprecated, skip calculating "mostRecentTimestamp" value');
        return releaseResult;
    }
    const highestReleaseTimestamp = (0, timestamp_1.asTimestamp)(highestRelease.releaseTimestamp);
    if (highestReleaseTimestamp) {
        const highestReleaseDatetime = luxon_1.DateTime.fromISO(highestReleaseTimestamp);
        const higherTimestampExists = releaseResult.releases.some((release) => {
            const releaseTimestamp = (0, timestamp_1.asTimestamp)(release.releaseTimestamp);
            if (!releaseTimestamp) {
                return false;
            }
            return luxon_1.DateTime.fromISO(releaseTimestamp) > highestReleaseDatetime;
        });
        if (!higherTimestampExists) {
            logger_1.logger.trace({ lookupName }, 'Using "mostRecentTimestamp" value because it is the highest timestamp of the highest release version');
            releaseResult.mostRecentTimestamp = highestReleaseTimestamp;
            return releaseResult;
        }
    }
    logger_1.logger.trace({ lookupName }, 'Skip using "mostRecentTimestamp" value because the higher timestamp exists for lower version');
    return releaseResult;
}
//# sourceMappingURL=timestamps.js.map