"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateUpdate = generateUpdate;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../../logger");
const date_1 = require("../../../../util/date");
const merge_confidence_1 = require("../../../../util/merge-confidence");
const update_type_1 = require("./update-type");
async function generateUpdate(config, currentValue, versioningApi, rangeStrategy, currentVersion, bucket, release) {
    const newVersion = release.version;
    const update = {
        bucket,
        newVersion,
        newValue: null,
    };
    // istanbul ignore if
    if (release.checksumUrl !== undefined) {
        update.checksumUrl = release.checksumUrl;
    }
    // istanbul ignore if
    if (release.downloadUrl !== undefined) {
        update.downloadUrl = release.downloadUrl;
    }
    // istanbul ignore if
    if (release.newDigest !== undefined) {
        update.newDigest = release.newDigest;
    }
    // istanbul ignore if
    if (release.releaseTimestamp) {
        update.releaseTimestamp = release.releaseTimestamp;
        update.newVersionAgeInDays = (0, date_1.getElapsedDays)(release.releaseTimestamp);
    }
    // istanbul ignore if
    if (release.registryUrl !== undefined) {
        /**
         * This means:
         *  - registry strategy is set to merge
         *  - releases were fetched from multiple registry urls
         */
        update.registryUrl = release.registryUrl;
    }
    if (currentValue) {
        try {
            update.newValue = versioningApi.getNewValue({
                currentValue,
                rangeStrategy,
                currentVersion,
                newVersion,
            });
        }
        catch (err) /* istanbul ignore next */ {
            logger_1.logger.warn({ err, currentValue, rangeStrategy, currentVersion, newVersion }, 'getNewValue error');
            update.newValue = currentValue;
        }
    }
    else {
        update.newValue = currentValue;
    }
    update.newMajor = versioningApi.getMajor(newVersion);
    update.newMinor = versioningApi.getMinor(newVersion);
    update.newPatch = versioningApi.getPatch(newVersion);
    // istanbul ignore if
    if (!update.updateType && !currentVersion) {
        logger_1.logger.debug({ update }, 'Update has no currentVersion');
        update.newValue = currentValue;
        return update;
    }
    update.updateType =
        update.updateType ??
            (0, update_type_1.getUpdateType)(config, versioningApi, currentVersion, newVersion);
    if (versioningApi.isBreaking) {
        // This versioning scheme has breaking awareness
        update.isBreaking = versioningApi.isBreaking(currentVersion, newVersion);
    }
    else {
        // This versioning scheme does not have breaking awareness - assume only major updates are breaking
        // Updates from, or to, unstable releases should be treated as breaking too.
        // But we should not add that as default behavior until we stop treating non-LTS versions as unstable first
        update.isBreaking = update.updateType === 'major';
    }
    const { datasource, packageName, packageRules } = config;
    if (packageRules?.some((pr) => is_1.default.nonEmptyArray(pr.matchConfidence))) {
        update.mergeConfidenceLevel = await (0, merge_confidence_1.getMergeConfidenceLevel)(datasource, packageName, currentVersion, newVersion, update.updateType);
    }
    if (!versioningApi.isVersion(update.newValue)) {
        update.isRange = true;
    }
    if (rangeStrategy === 'update-lockfile' && currentValue === update.newValue) {
        update.isLockfileUpdate = true;
    }
    if (rangeStrategy === 'bump' &&
        // TODO #22198
        versioningApi.matches(newVersion, currentValue)) {
        update.isBump = true;
    }
    return update;
}
//# sourceMappingURL=generate.js.map