"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filterVersions = filterVersions;
const tslib_1 = require("tslib");
const semver_1 = tslib_1.__importDefault(require("semver"));
const error_messages_1 = require("../../../../constants/error-messages");
const logger_1 = require("../../../../logger");
const npmVersioning = tslib_1.__importStar(require("../../../../modules/versioning/npm"));
const pep440 = tslib_1.__importStar(require("../../../../modules/versioning/pep440"));
const poetryVersioning = tslib_1.__importStar(require("../../../../modules/versioning/poetry"));
const string_match_1 = require("../../../../util/string-match");
function isReleaseStable(release, versioningApi) {
    if (!versioningApi.isStable(release.version)) {
        return false;
    }
    if (release.isStable === false) {
        return false;
    }
    return true;
}
function filterVersions(config, currentVersion, latestVersion, releases, versioningApi) {
    const { ignoreUnstable, ignoreDeprecated, respectLatest, allowedVersions } = config;
    // istanbul ignore if: shouldn't happen
    if (!currentVersion) {
        return [];
    }
    // Leave only versions greater than current
    let filteredReleases = releases.filter((r) => versioningApi.isVersion(r.version) &&
        versioningApi.isGreaterThan(r.version, currentVersion));
    const currentRelease = releases.find((r) => versioningApi.isValid(r.version) &&
        versioningApi.isVersion(r.version) &&
        versioningApi.isValid(currentVersion) &&
        versioningApi.isVersion(currentVersion) &&
        versioningApi.equals(r.version, currentVersion));
    // Don't upgrade from non-deprecated to deprecated
    if (ignoreDeprecated && currentRelease && !currentRelease.isDeprecated) {
        filteredReleases = filteredReleases.filter((r) => {
            if (r.isDeprecated) {
                logger_1.logger.trace(`Skipping ${config.depName}@${r.version} because it is deprecated`);
                return false;
            }
            return true;
        });
    }
    if (allowedVersions) {
        const isAllowedPred = (0, string_match_1.getRegexPredicate)(allowedVersions);
        if (isAllowedPred) {
            filteredReleases = filteredReleases.filter(({ version }) => isAllowedPred(version));
        }
        else if (versioningApi.isValid(allowedVersions)) {
            filteredReleases = filteredReleases.filter((r) => versioningApi.matches(r.version, allowedVersions));
        }
        else if (config.versioning !== npmVersioning.id &&
            semver_1.default.validRange(allowedVersions)) {
            logger_1.logger.debug({ depName: config.depName }, 'Falling back to npm semver syntax for allowedVersions');
            filteredReleases = filteredReleases.filter((r) => semver_1.default.satisfies(semver_1.default.valid(r.version)
                ? r.version
                : /* istanbul ignore next: not reachable, but it's safer to preserve it */ semver_1.default.coerce(r.version), allowedVersions));
        }
        else if (config.versioning === poetryVersioning.id &&
            pep440.isValid(allowedVersions)) {
            logger_1.logger.debug({ depName: config.depName }, 'Falling back to pypi syntax for allowedVersions');
            filteredReleases = filteredReleases.filter((r) => pep440.matches(r.version, allowedVersions));
        }
        else {
            const error = new Error(error_messages_1.CONFIG_VALIDATION);
            error.validationSource = 'config';
            error.validationError = 'Invalid `allowedVersions`';
            error.validationMessage =
                'The following allowedVersions does not parse as a valid version or range: ' +
                    JSON.stringify(allowedVersions);
            throw error;
        }
    }
    if (config.followTag) {
        return filteredReleases;
    }
    if (respectLatest &&
        latestVersion &&
        !versioningApi.isGreaterThan(currentVersion, latestVersion)) {
        filteredReleases = filteredReleases.filter((r) => !versioningApi.isGreaterThan(r.version, latestVersion));
    }
    if (!ignoreUnstable) {
        return filteredReleases;
    }
    if (currentRelease && isReleaseStable(currentRelease, versioningApi)) {
        return filteredReleases.filter((r) => isReleaseStable(r, versioningApi));
    }
    const currentMajor = versioningApi.getMajor(currentVersion);
    const currentMinor = versioningApi.getMinor(currentVersion);
    const currentPatch = versioningApi.getPatch(currentVersion);
    return filteredReleases.filter((r) => {
        if (isReleaseStable(r, versioningApi)) {
            return true;
        }
        const major = versioningApi.getMajor(r.version);
        if (major !== currentMajor) {
            return false;
        }
        if (versioningApi.allowUnstableMajorUpgrades) {
            return true;
        }
        const minor = versioningApi.getMinor(r.version);
        const patch = versioningApi.getPatch(r.version);
        return minor === currentMinor && patch === currentPatch;
    });
}
//# sourceMappingURL=filter.js.map