"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filterInternalChecks = filterInternalChecks;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const config_1 = require("../../../../config");
const logger_1 = require("../../../../logger");
const postprocess_release_1 = require("../../../../modules/datasource/postprocess-release");
const date_1 = require("../../../../util/date");
const merge_confidence_1 = require("../../../../util/merge-confidence");
const number_1 = require("../../../../util/number");
const package_rules_1 = require("../../../../util/package-rules");
const pretty_time_1 = require("../../../../util/pretty-time");
const update_type_1 = require("./update-type");
async function filterInternalChecks(config, versioningApi, bucket, sortedReleases) {
    const { currentVersion, datasource, depName, internalChecksFilter } = config;
    let release = undefined;
    let pendingChecks = false;
    let pendingReleases = [];
    if (internalChecksFilter === 'none') {
        // Don't care if minimumReleaseAge or minimumConfidence are unmet
        release = sortedReleases.pop();
    }
    else {
        // iterate through releases from highest to lowest, looking for the first which will pass checks if present
        for (let candidateRelease of sortedReleases.reverse()) {
            // merge the release data into dependency config
            let releaseConfig = (0, config_1.mergeChildConfig)(config, candidateRelease);
            // calculate updateType and then apply it
            releaseConfig.updateType = (0, update_type_1.getUpdateType)(releaseConfig, versioningApi, 
            // TODO #22198
            currentVersion, candidateRelease.version);
            releaseConfig = (0, config_1.mergeChildConfig)(releaseConfig, releaseConfig[releaseConfig.updateType]);
            // Apply packageRules in case any apply to updateType
            releaseConfig = await (0, package_rules_1.applyPackageRules)(releaseConfig, 'update-type');
            const updatedCandidateRelease = await (0, postprocess_release_1.postprocessRelease)(releaseConfig, candidateRelease);
            if (!updatedCandidateRelease) {
                continue;
            }
            candidateRelease = updatedCandidateRelease;
            // Now check for a minimumReleaseAge config
            const { minimumConfidence, minimumReleaseAge, updateType } = releaseConfig;
            if (is_1.default.nonEmptyString(minimumReleaseAge) &&
                candidateRelease.releaseTimestamp) {
                if ((0, date_1.getElapsedMs)(candidateRelease.releaseTimestamp) <
                    (0, number_1.coerceNumber)((0, pretty_time_1.toMs)(minimumReleaseAge), 0)) {
                    // Skip it if it doesn't pass checks
                    logger_1.logger.trace({ depName, check: 'minimumReleaseAge' }, `Release ${candidateRelease.version} is pending status checks`);
                    pendingReleases.unshift(candidateRelease);
                    continue;
                }
            }
            // TODO #22198
            if ((0, merge_confidence_1.isActiveConfidenceLevel)(minimumConfidence)) {
                const confidenceLevel = (await (0, merge_confidence_1.getMergeConfidenceLevel)(datasource, depName, currentVersion, candidateRelease.version, updateType)) ?? 'neutral';
                // TODO #22198
                if (!(0, merge_confidence_1.satisfiesConfidenceLevel)(confidenceLevel, minimumConfidence)) {
                    logger_1.logger.trace({ depName, check: 'minimumConfidence' }, `Release ${candidateRelease.version} is pending status checks`);
                    pendingReleases.unshift(candidateRelease);
                    continue;
                }
            }
            // If we get to here, then the release is OK and we can stop iterating
            release = candidateRelease;
            break;
        }
        if (!release) {
            if (pendingReleases.length) {
                // If all releases were pending then just take the highest
                logger_1.logger.trace({ depName, bucket }, 'All releases are pending - using latest');
                release = pendingReleases.pop();
                // None are pending anymore because we took the latest, so empty the array
                pendingReleases = [];
                if (internalChecksFilter === 'strict') {
                    pendingChecks = true;
                }
            }
        }
    }
    return { release, pendingChecks, pendingReleases };
}
//# sourceMappingURL=filter-checks.js.map