"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.calculateLibYears = calculateLibYears;
const luxon_1 = require("luxon");
const reporting_1 = require("../../../instrumentation/reporting");
const logger_1 = require("../../../logger");
function calculateLibYears(config, packageFiles) {
    if (!packageFiles) {
        return;
    }
    const allDeps = [];
    for (const [manager, files] of Object.entries(packageFiles)) {
        for (const file of files) {
            for (const dep of file.deps) {
                const depInfo = {
                    depName: dep.depName,
                    manager,
                    file: file.packageFile,
                    datasource: dep.datasource,
                    version: (dep.currentVersion ?? dep.currentValue),
                };
                if (!dep.updates?.length) {
                    allDeps.push(depInfo);
                    continue;
                }
                depInfo.outdated = true;
                if (!dep.currentVersionTimestamp) {
                    logger_1.logger.once.debug(`No currentVersionTimestamp for ${dep.depName}`);
                    allDeps.push(depInfo);
                    continue;
                }
                // timestamps are in ISO format
                const currentVersionDate = luxon_1.DateTime.fromISO(dep.currentVersionTimestamp);
                for (const update of dep.updates) {
                    if (!update.releaseTimestamp) {
                        logger_1.logger.once.debug(`No releaseTimestamp for ${dep.depName} update to ${update.newVersion}`);
                        continue;
                    }
                    const releaseDate = luxon_1.DateTime.fromISO(update.releaseTimestamp);
                    const libYears = releaseDate.diff(currentVersionDate, 'years').years;
                    if (libYears >= 0) {
                        update.libYears = libYears;
                    }
                }
                // Set the highest libYears for the dep
                const depLibYears = Math.max(...dep.updates.map((update) => update.libYears ?? 0), 0);
                depInfo.libYear = depLibYears;
                allDeps.push(depInfo);
            }
        }
    }
    const libYearsWithStatus = getLibYears(allDeps);
    logger_1.logger.debug(libYearsWithStatus, 'Repository libYears');
    (0, reporting_1.addLibYears)(config, libYearsWithStatus);
}
function getLibYears(allDeps) {
    const [totalDepsCount, outdatedDepsCount, totalLibYears] = getCounts(allDeps);
    const managerLibYears = getManagerLibYears(allDeps);
    return {
        libYears: {
            managers: managerLibYears,
            total: totalLibYears,
        },
        dependencyStatus: {
            outdated: outdatedDepsCount,
            total: totalDepsCount,
        },
    };
}
function getManagerLibYears(deps) {
    /** {manager : {depKey: libYear }} */
    const managerLibYears = {};
    for (const dep of deps) {
        const depKey = `${dep.depName}@${dep.version}@${dep.datasource}`;
        const manager = dep.manager;
        managerLibYears[manager] ??= {};
        if (dep.libYear) {
            if (!managerLibYears[manager][depKey]) {
                managerLibYears[manager][depKey] = dep.libYear;
            }
        }
    }
    const res = {};
    for (const [manager, deps] of Object.entries(managerLibYears)) {
        const managerLibYear = Object.values(deps).reduce((sum, curr) => {
            return sum + curr;
        }, 0);
        res[manager] = managerLibYear;
    }
    return res;
}
function getCounts(deps) {
    const distinctDeps = new Set();
    let totalDepsCount = 0, outdatedDepsCount = 0, totalLibYears = 0;
    for (const dep of deps) {
        const depKey = `${dep.depName}@${dep.version}@${dep.datasource}`;
        if (!distinctDeps.has(depKey)) {
            if (dep.outdated) {
                outdatedDepsCount++;
            }
            if (dep.libYear) {
                totalLibYears += dep.libYear;
            }
            totalDepsCount++;
            distinctDeps.add(depKey);
        }
    }
    return [totalDepsCount, outdatedDepsCount, totalLibYears];
}
//# sourceMappingURL=libyear.js.map