"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setOnboardingCache = setOnboardingCache;
exports.deleteOnboardingCache = deleteOnboardingCache;
exports.hasOnboardingBranchChanged = hasOnboardingBranchChanged;
exports.isOnboardingBranchModified = isOnboardingBranchModified;
exports.getOnboardingFileNameFromCache = getOnboardingFileNameFromCache;
exports.getOnboardingConfigFromCache = getOnboardingConfigFromCache;
exports.setOnboardingConfigDetails = setOnboardingConfigDetails;
exports.isOnboardingBranchConflicted = isOnboardingBranchConflicted;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../../logger");
const scm_1 = require("../../../../modules/platform/scm");
const repository_1 = require("../../../../util/cache/repository");
const git_1 = require("../../../../util/git");
function setOnboardingCache(defaultBranchSha, onboardingBranchSha, isConflicted, isModified) {
    // do not update cache if commit is null/undefined
    if (!(is_1.default.nonEmptyString(defaultBranchSha) &&
        is_1.default.nonEmptyString(onboardingBranchSha))) {
        logger_1.logger.debug('Onboarding cache not updated');
        return;
    }
    const cache = (0, repository_1.getCache)();
    const onboardingCache = {
        defaultBranchSha,
        onboardingBranchSha,
        isConflicted,
        isModified,
    };
    if (cache.onboardingBranchCache) {
        logger_1.logger.debug({ onboardingCache }, 'Update Onboarding Cache');
    }
    else {
        logger_1.logger.debug({ onboardingCache }, 'Create Onboarding Cache');
    }
    cache.onboardingBranchCache = onboardingCache;
}
function deleteOnboardingCache() {
    const cache = (0, repository_1.getCache)();
    if (cache?.onboardingBranchCache) {
        logger_1.logger.debug('Delete Onboarding Cache');
        delete cache.onboardingBranchCache;
    }
}
// checks if onboarding branch has been modified since last run
// return true if cache isn't present
function hasOnboardingBranchChanged(onboardingBranch) {
    const cache = (0, repository_1.getCache)();
    const onboardingSha = (0, git_1.getBranchCommit)(onboardingBranch);
    if (cache.onboardingBranchCache) {
        return onboardingSha !== cache.onboardingBranchCache.onboardingBranchSha;
    }
    return true;
}
// checks if onboarding branch has been modified by user
// once set to true it stays true as we do not rebase onboarding branches anymore (this feature will be added in future though)
async function isOnboardingBranchModified(onboardingBranch, defaultBranch) {
    const cache = (0, repository_1.getCache)();
    const onboardingCache = cache.onboardingBranchCache;
    const onboardingSha = (0, git_1.getBranchCommit)(onboardingBranch);
    let isModified = false;
    if (onboardingCache &&
        onboardingSha === onboardingCache.onboardingBranchSha &&
        !is_1.default.undefined(onboardingCache.isModified)) {
        return onboardingCache.isModified;
    }
    else {
        isModified = await scm_1.scm.isBranchModified(onboardingBranch, defaultBranch);
    }
    return isModified;
}
function getOnboardingFileNameFromCache() {
    const cache = (0, repository_1.getCache)();
    return cache.onboardingBranchCache?.configFileName;
}
function getOnboardingConfigFromCache() {
    const cache = (0, repository_1.getCache)();
    return cache.onboardingBranchCache?.configFileParsed;
}
function setOnboardingConfigDetails(configFileName, configFileParsed) {
    const cache = (0, repository_1.getCache)();
    if (cache.onboardingBranchCache) {
        cache.onboardingBranchCache.configFileName = configFileName;
        cache.onboardingBranchCache.configFileParsed = configFileParsed;
    }
}
async function isOnboardingBranchConflicted(defaultBranch, onboardingBranch) {
    const cache = (0, repository_1.getCache)();
    const onboardingCache = cache.onboardingBranchCache;
    const onboardingSha = (0, git_1.getBranchCommit)(onboardingBranch);
    const defaultBranchSha = (0, git_1.getBranchCommit)(defaultBranch);
    let isConflicted = false;
    if (onboardingCache &&
        defaultBranchSha === onboardingCache.defaultBranchSha &&
        onboardingSha === onboardingCache.onboardingBranchSha &&
        !is_1.default.undefined(onboardingCache.isConflicted)) {
        return onboardingCache.isConflicted;
    }
    else {
        isConflicted = await scm_1.scm.isBranchConflicted(defaultBranch, onboardingBranch);
    }
    return isConflicted;
}
//# sourceMappingURL=onboarding-branch-cache.js.map