"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOnboardingConfig = getOnboardingConfig;
exports.getOnboardingConfigContents = getOnboardingConfigContents;
const global_1 = require("../../../../config/global");
const local_1 = require("../../../../config/presets/local");
const util_1 = require("../../../../config/presets/util");
const logger_1 = require("../../../../logger");
const clone_1 = require("../../../../util/clone");
const json_writer_1 = require("../../../../util/json-writer");
async function getOnboardingConfig(config) {
    let onboardingConfig = (0, clone_1.clone)(config.onboardingConfig);
    // TODO #22198 fix types
    const foundPreset = await searchDefaultOnboardingPreset(config.repository);
    if (foundPreset) {
        logger_1.logger.debug(`Found preset ${foundPreset} - using it in onboarding config`);
        onboardingConfig = {
            $schema: 'https://docs.renovatebot.com/renovate-schema.json',
            extends: [foundPreset],
        };
    }
    else {
        // Organization preset did not exist
        logger_1.logger.debug('No default org/owner preset found, so the default onboarding config will be used instead.');
    }
    logger_1.logger.debug({ config: onboardingConfig }, 'onboarding config');
    return onboardingConfig;
}
async function searchDefaultOnboardingPreset(repository) {
    let foundPreset;
    logger_1.logger.debug('Checking for a default Renovate preset which can be used.');
    const repoPathParts = repository.split('/');
    for (let index = repoPathParts.length - 1; index >= 1 && !foundPreset; index--) {
        const groupName = repoPathParts.slice(0, index).join('/');
        // Check for group/renovate-config
        try {
            const repo = `${groupName}/renovate-config`;
            const preset = `local>${repo}`;
            logger_1.logger.debug(`Checking for preset: ${preset}`);
            if (await (0, local_1.getPreset)({ repo })) {
                foundPreset = preset;
            }
        }
        catch (err) {
            if (err.message !== util_1.PRESET_DEP_NOT_FOUND &&
                !err.message.startsWith('Unsupported platform')) {
                logger_1.logger.warn({ err }, 'Unknown error fetching default owner preset');
            }
        }
    }
    if (!foundPreset) {
        // Check for org/.{{platform}}
        const orgName = repoPathParts[0];
        // TODO: types (#22198)
        const platform = global_1.GlobalConfig.get('platform');
        try {
            const repo = `${orgName}/.${platform}`;
            const presetName = 'renovate-config';
            const orgPresetName = `local>${repo}:${presetName}`;
            logger_1.logger.debug(`Checking for preset: ${orgPresetName}`);
            if (await (0, local_1.getPreset)({
                repo,
                presetName,
            })) {
                foundPreset = orgPresetName;
            }
        }
        catch (err) {
            if (err.message !== util_1.PRESET_DEP_NOT_FOUND &&
                !err.message.startsWith('Unsupported platform')) {
                logger_1.logger.warn({ err }, 'Unknown error fetching default owner preset');
            }
        }
    }
    return foundPreset;
}
async function getOnboardingConfigContents(config, fileName) {
    const codeFormat = await json_writer_1.EditorConfig.getCodeFormat(fileName);
    const jsonWriter = new json_writer_1.JSONWriter(codeFormat);
    const onboardingConfig = await getOnboardingConfig(config);
    return jsonWriter.write(onboardingConfig);
}
//# sourceMappingURL=config.js.map