"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeInheritedConfig = mergeInheritedConfig;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const dequal_1 = require("dequal");
const config_1 = require("../../../config");
const decrypt_1 = require("../../../config/decrypt");
const parse_1 = require("../../../config/parse");
const presets_1 = require("../../../config/presets");
const secrets_1 = require("../../../config/secrets");
const validation_1 = require("../../../config/validation");
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const platform_1 = require("../../../modules/platform");
const hostRules = tslib_1.__importStar(require("../../../util/host-rules"));
const queue = tslib_1.__importStar(require("../../../util/http/queue"));
const throttle = tslib_1.__importStar(require("../../../util/http/throttle"));
const template = tslib_1.__importStar(require("../../../util/template"));
async function mergeInheritedConfig(config) {
    // typescript doesn't know that repo is defined
    if (!config.repository || !config.inheritConfig) {
        return config;
    }
    if (!is_1.default.string(config.inheritConfigRepoName) ||
        !is_1.default.string(config.inheritConfigFileName)) {
        // Config validation should prevent this error
        logger_1.logger.error({
            inheritConfigRepoName: config.inheritConfigRepoName,
            inheritConfigFileName: config.inheritConfigFileName,
        }, 'Invalid inherited config.');
        return config;
    }
    const templateConfig = {
        topLevelOrg: config.topLevelOrg,
        parentOrg: config.parentOrg,
        repository: config.repository,
    };
    const inheritConfigRepoName = template.compile(config.inheritConfigRepoName, templateConfig, false);
    logger_1.logger.trace({ templateConfig, inheritConfigRepoName }, 'Compiled inheritConfigRepoName result.');
    logger_1.logger.debug(`Checking for inherited config file ${config.inheritConfigFileName} in repo ${inheritConfigRepoName}.`);
    let configFileRaw = null;
    try {
        configFileRaw = await platform_1.platform.getRawFile(config.inheritConfigFileName, inheritConfigRepoName);
    }
    catch (err) {
        if (config.inheritConfigStrict) {
            logger_1.logger.debug({ err }, 'Error getting inherited config.');
            throw new Error(error_messages_1.CONFIG_INHERIT_NOT_FOUND);
        }
        logger_1.logger.trace({ err }, `Error getting inherited config.`);
    }
    if (!configFileRaw) {
        logger_1.logger.debug(`No inherited config found in ${inheritConfigRepoName}.`);
        return config;
    }
    const parseResult = (0, parse_1.parseFileConfig)(config.inheritConfigFileName, configFileRaw);
    if (!parseResult.success) {
        logger_1.logger.debug({ parseResult }, 'Error parsing inherited config.');
        throw new Error(error_messages_1.CONFIG_INHERIT_PARSE_ERROR);
    }
    const inheritedConfig = parseResult.parsedContents;
    logger_1.logger.debug({ config: inheritedConfig }, `Inherited config`);
    const res = await (0, validation_1.validateConfig)('inherit', inheritedConfig);
    if (res.errors.length) {
        logger_1.logger.warn({ errors: res.errors }, 'Found errors in inherited configuration.');
        throw new Error(error_messages_1.CONFIG_VALIDATION);
    }
    if (res.warnings.length) {
        logger_1.logger.warn({ warnings: res.warnings }, 'Found warnings in inherited configuration.');
    }
    let decryptedConfig = await (0, decrypt_1.decryptConfig)(inheritedConfig, config.repository);
    let filteredConfig = (0, config_1.removeGlobalConfig)(decryptedConfig, true);
    if (!(0, dequal_1.dequal)(decryptedConfig, filteredConfig)) {
        logger_1.logger.debug({ inheritedConfig: decryptedConfig, filteredConfig }, 'Removed global config from inherited config.');
    }
    if (is_1.default.nullOrUndefined(filteredConfig.extends)) {
        filteredConfig = (0, secrets_1.applySecretsAndVariablesToConfig)({
            config: filteredConfig,
            secrets: config.secrets ?? {},
            variables: config.variables ?? {},
        });
        setInheritedHostRules(filteredConfig);
        return (0, config_1.mergeChildConfig)(config, filteredConfig);
    }
    logger_1.logger.debug('Resolving presets found in inherited config');
    const resolvedConfig = await (0, presets_1.resolveConfigPresets)(filteredConfig, config, config.ignorePresets);
    logger_1.logger.trace({ config: resolvedConfig }, 'Resolved inherited config');
    const validationRes = await (0, validation_1.validateConfig)('inherit', resolvedConfig);
    if (validationRes.errors.length) {
        logger_1.logger.warn({ errors: validationRes.errors }, 'Found errors in presets inside the inherited configuration.');
        throw new Error(error_messages_1.CONFIG_VALIDATION);
    }
    if (validationRes.warnings.length) {
        logger_1.logger.warn({ warnings: validationRes.warnings }, 'Found warnings in presets inside the inherited configuration.');
    }
    // decrypt again, as resolved presets could contain encrypted values
    decryptedConfig = await (0, decrypt_1.decryptConfig)(resolvedConfig, config.repository);
    // remove global config options once again, as resolved presets could have added some
    filteredConfig = (0, config_1.removeGlobalConfig)(decryptedConfig, true);
    if (!(0, dequal_1.dequal)(decryptedConfig, filteredConfig)) {
        logger_1.logger.debug({ inheritedConfig: decryptedConfig, filteredConfig }, 'Removed global config from inherited config presets.');
    }
    filteredConfig = (0, secrets_1.applySecretsAndVariablesToConfig)({
        config: filteredConfig,
        secrets: config.secrets ?? {},
        variables: config.variables ?? {},
    });
    setInheritedHostRules(filteredConfig);
    return (0, config_1.mergeChildConfig)(config, filteredConfig);
}
function setInheritedHostRules(config) {
    if (config.hostRules) {
        logger_1.logger.debug('Setting hostRules from config');
        for (const rule of config.hostRules) {
            try {
                hostRules.add(rule);
            }
            catch (err) {
                // istanbul ignore next
                logger_1.logger.warn({ err, config: rule }, 'Error setting hostRule from config');
            }
        }
        // host rules can change concurrency
        queue.clear();
        throttle.clear();
        delete config.hostRules;
    }
}
//# sourceMappingURL=inherited.js.map