"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.initApis = initApis;
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const platform_1 = require("../../../modules/platform");
const common_1 = require("../onboarding/common");
async function getJsonFile(file) {
    try {
        return await platform_1.platform.getJsonFile(file);
    }
    catch {
        return null;
    }
}
async function validateOptimizeForDisabled(config) {
    if (config.optimizeForDisabled) {
        const renovateConfig = await getJsonFile((0, common_1.getDefaultConfigFileName)(config));
        if (renovateConfig?.enabled === false) {
            throw new Error(error_messages_1.REPOSITORY_DISABLED_BY_CONFIG);
        }
        /*
         * The following is to support a use case within Mend customers where:
         *  - Bot admins configure install the bot into every repo
         *  - Bot admins configure `extends: [':disableRenovate'] in order to skip repos by default
         *  - Repo users can push a `renovate.json` containing `extends: [':enableRenovate']` to re-enable Renovate
         */
        if (config.extends?.includes(':disableRenovate')) {
            logger_1.logger.debug('Global config disables Renovate - checking renovate.json to see if it is re-enabled');
            if (renovateConfig?.extends?.includes(':enableRenovate') ??
                renovateConfig?.ignorePresets?.includes(':disableRenovate') ??
                renovateConfig?.enabled) {
                logger_1.logger.debug('Repository config re-enables Renovate - continuing');
            }
            else {
                logger_1.logger.debug('Repository config does not re-enable Renovate - skipping');
                throw new Error(error_messages_1.REPOSITORY_DISABLED_BY_CONFIG);
            }
        }
    }
}
async function validateIncludeForks(config) {
    if (config.forkProcessing !== 'enabled' && config.isFork) {
        const defaultConfigFile = (0, common_1.getDefaultConfigFileName)(config);
        const repoConfig = await getJsonFile(defaultConfigFile);
        if (!repoConfig) {
            logger_1.logger.debug(`Default config file ${defaultConfigFile} not found in repo`);
            throw new Error(error_messages_1.REPOSITORY_FORKED);
        }
        if (repoConfig.includeForks) {
            logger_1.logger.debug(`Found legacy setting includeForks in ${defaultConfigFile} - continuing`);
            return;
        }
        if (repoConfig.forkProcessing === 'enabled') {
            logger_1.logger.debug(`Found forkProcessing=enabled in ${defaultConfigFile} - continuing`);
            return;
        }
        logger_1.logger.debug({ config: repoConfig }, `Default config file ${defaultConfigFile} found in repo but does not enable forks`);
        throw new Error(error_messages_1.REPOSITORY_FORKED);
    }
}
// TODO: fix types (#22198)
async function getPlatformConfig(config) {
    const platformConfig = await platform_1.platform.initRepo(config);
    return {
        ...config,
        ...platformConfig,
    };
}
// TODO: fix types (#22198)
async function initApis(input) {
    let config = { ...input };
    config = await getPlatformConfig(config);
    await validateOptimizeForDisabled(config);
    await validateIncludeForks(config);
    return config;
}
//# sourceMappingURL=apis.js.map