"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.rebaseMigrationBranch = rebaseMigrationBranch;
exports.jsonStripWhitespaces = jsonStripWhitespaces;
const tslib_1 = require("tslib");
const json5_1 = tslib_1.__importDefault(require("json5"));
const global_1 = require("../../../../config/global");
const logger_1 = require("../../../../logger");
const scm_1 = require("../../../../modules/platform/scm");
const git_1 = require("../../../../util/git");
const stringify_1 = require("../../../../util/stringify");
const common_1 = require("../common");
const commit_message_1 = require("./commit-message");
const migrated_data_1 = require("./migrated-data");
async function rebaseMigrationBranch(config, migratedConfigData) {
    logger_1.logger.debug('Checking if migration branch needs rebasing');
    const baseBranch = config.defaultBranch;
    const branchName = (0, common_1.getMigrationBranchName)(config);
    const configFileName = migratedConfigData.filename;
    let contents = migratedConfigData.content;
    const existingContents = await (0, git_1.getFile)(configFileName, branchName);
    if (jsonStripWhitespaces(contents) === jsonStripWhitespaces(existingContents)) {
        logger_1.logger.debug('Migration branch is up to date');
        return null;
    }
    logger_1.logger.debug('Rebasing migration branch');
    if (global_1.GlobalConfig.get('dryRun')) {
        logger_1.logger.info('DRY-RUN: Would rebase files in migration branch');
        return null;
    }
    const commitMessageFactory = new commit_message_1.ConfigMigrationCommitMessageFactory(config, configFileName);
    const commitMessage = commitMessageFactory.getCommitMessage();
    await scm_1.scm.checkoutBranch(baseBranch);
    contents =
        await migrated_data_1.MigratedDataFactory.applyPrettierFormatting(migratedConfigData);
    return scm_1.scm.commitAndPush({
        baseBranch: config.baseBranch,
        branchName,
        files: [
            {
                type: 'addition',
                path: configFileName,
                contents,
            },
        ],
        message: commitMessage.toString(),
        platformCommit: config.platformCommit,
    });
}
/**
 * @param json a JSON string
 * @return a minimal json string. i.e. does not contain any formatting/whitespaces
 */
function jsonStripWhitespaces(json) {
    if (!json) {
        return null;
    }
    /**
     * JSON.stringify(value, replacer, space):
     * If "space" is anything other than a string or number —
     * for example, is null or not provided — no white space is used.
     *
     * https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/JSON/stringify#parameters
     */
    return ((0, stringify_1.quickStringify)(json5_1.default.parse(json)) ??
        /* istanbul ignore next: should never happen */ null);
}
//# sourceMappingURL=rebase.js.map