"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.counts = void 0;
exports.resetAllLimits = resetAllLimits;
exports.setMaxLimit = setMaxLimit;
exports.incLimitedValue = incLimitedValue;
exports.getCount = getCount;
exports.setCount = setCount;
exports.incCountValue = incCountValue;
exports.calcLimit = calcLimit;
exports.hasMultipleLimits = hasMultipleLimits;
exports.isLimitReached = isLimitReached;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../logger");
const limits = new Map();
function resetAllLimits() {
    limits.clear();
}
function setMaxLimit(key, val) {
    const max = typeof val === 'number' ? Math.max(0, val) : null;
    limits.set(key, { current: 0, max });
    logger_1.logger.debug(`${key} limit = ${max}`);
}
function incLimitedValue(key, incBy = 1) {
    const limit = limits.get(key) ?? { max: null, current: 0 };
    limits.set(key, {
        ...limit,
        current: limit.current + incBy,
    });
}
function handleCommitsLimit() {
    const limit = limits.get('Commits');
    // TODO: fix me?
    // eslint-disable-next-line @typescript-eslint/prefer-optional-chain
    if (!limit || limit.max === null) {
        return false;
    }
    const { max, current } = limit;
    return max - current <= 0;
}
exports.counts = new Map();
function getCount(key) {
    const count = exports.counts.get(key);
    // istanbul ignore if: should not happen
    if (!is_1.default.integer(count)) {
        logger_1.logger.debug(`Could not compute the count of ${key}, returning zero.`);
        return 0;
    }
    return count;
}
function setCount(key, val) {
    exports.counts.set(key, val);
    logger_1.logger.debug(`${key} count = ${val}`);
}
function incCountValue(key, incBy = 1) {
    const count = getCount(key);
    exports.counts.set(key, count + incBy);
}
function handleConcurrentLimits(key, config) {
    const limitKey = key === 'Branches' ? 'branchConcurrentLimit' : 'prConcurrentLimit';
    // calculate the limits for this branch
    const hourlyLimit = calcLimit(config.upgrades, 'prHourlyLimit');
    const hourlyPrCount = getCount('HourlyPRs');
    // if a limit is defined ( >0 ) and limit reached return true ie. limit has been reached
    if (hourlyLimit && hourlyPrCount >= hourlyLimit) {
        return true;
    }
    const limitValue = calcLimit(config.upgrades, limitKey);
    const currentCount = getCount(key);
    if (limitValue && currentCount >= limitValue) {
        return true;
    }
    return false;
}
function calcLimit(upgrades, limitName) {
    logger_1.logger.debug({
        limits: upgrades.map((upg) => {
            return { depName: upg.depName, [limitName]: upg[limitName] };
        }),
    }, `${limitName} of the upgrades present in this branch`);
    if (hasMultipleLimits(upgrades, limitName)) {
        logger_1.logger.once.debug(`Branch has multiple ${limitName} limits. The lowest among these will be selected.`);
    }
    let lowestLimit = Number.MAX_SAFE_INTEGER;
    for (const upgrade of upgrades) {
        let limit = upgrade[limitName];
        // inherit prConcurrentLimit value incase branchConcurrentLimit is null
        if (!is_1.default.number(limit) && limitName === 'branchConcurrentLimit') {
            limit = upgrade.prConcurrentLimit;
        }
        // istanbul ignore if: should never happen as all limits get a default value
        if (is_1.default.undefined(limit)) {
            limit = Number.MAX_SAFE_INTEGER;
        }
        // no limit
        if (limit === 0 || limit === null) {
            logger_1.logger.debug(`${limitName} of this branch is unlimited, because atleast one of the upgrade has it's ${limitName} set to "No limit" ie. 0 or null`);
            return 0;
        }
        // limit is set
        lowestLimit = limit < lowestLimit ? limit : lowestLimit;
    }
    logger_1.logger.debug(`Calculated lowest ${limitName} among the upgrades present in this branch is ${lowestLimit}.`);
    return lowestLimit;
}
function hasMultipleLimits(upgrades, limitName) {
    if (upgrades.length === 1) {
        return false;
    }
    const distinctLimits = new Set();
    for (const upgrade of upgrades) {
        let limitValue = upgrade[limitName];
        // inherit prConcurrentLimit value incase branchConcurrentLimit is null
        if (limitName === 'branchConcurrentLimit' && !is_1.default.number(limitValue)) {
            limitValue = upgrade.prConcurrentLimit;
        }
        // istanbul ignore if: should not happen as the limits are of type number
        if (limitValue === null) {
            limitValue = 0;
        }
        if (!is_1.default.undefined(limitValue) && !distinctLimits.has(limitValue)) {
            distinctLimits.add(limitValue);
        }
    }
    return distinctLimits.size > 1;
}
function isLimitReached(limit, config) {
    if (limit === 'Commits') {
        return handleCommitsLimit();
    }
    if (config) {
        return handleConcurrentLimits(limit, config);
    }
    // istanbul ignore next: should not happen
    throw new Error('Config is required for computing limits for Branches and PullRequests');
}
//# sourceMappingURL=limits.js.map