"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.globalInitialize = globalInitialize;
exports.globalFinalize = globalFinalize;
const tslib_1 = require("tslib");
const node_os_1 = tslib_1.__importDefault(require("node:os"));
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const upath_1 = tslib_1.__importDefault(require("upath"));
const secrets_1 = require("../../config/secrets");
const logger_1 = require("../../logger");
const remap_1 = require("../../logger/remap");
const platform_1 = require("../../modules/platform");
const packageCache = tslib_1.__importStar(require("../../util/cache/package"));
const emoji_1 = require("../../util/emoji");
const git_1 = require("../../util/git");
const hostRules = tslib_1.__importStar(require("../../util/host-rules"));
const rate_limits_1 = require("../../util/http/rate-limits");
const merge_confidence_1 = require("../../util/merge-confidence");
const limits_1 = require("./limits");
async function setDirectories(input) {
    const config = { ...input };
    process.env.TMPDIR = process.env.RENOVATE_TMPDIR ?? node_os_1.default.tmpdir();
    if (config.baseDir) {
        logger_1.logger.debug('Using configured baseDir: ' + config.baseDir);
    }
    else {
        config.baseDir = upath_1.default.join(process.env.TMPDIR, 'renovate');
        logger_1.logger.debug('Using baseDir: ' + config.baseDir);
    }
    await fs_extra_1.default.ensureDir(config.baseDir);
    if (config.cacheDir) {
        logger_1.logger.debug('Using configured cacheDir: ' + config.cacheDir);
    }
    else {
        config.cacheDir = upath_1.default.join(config.baseDir, 'cache');
        logger_1.logger.debug('Using cacheDir: ' + config.cacheDir);
    }
    await fs_extra_1.default.ensureDir(config.cacheDir);
    if (config.binarySource === 'docker' || config.binarySource === 'install') {
        if (config.containerbaseDir) {
            logger_1.logger.debug('Using configured containerbaseDir: ' + config.containerbaseDir);
        }
        else {
            config.containerbaseDir = upath_1.default.join(config.cacheDir, 'containerbase');
            logger_1.logger.debug('Using containerbaseDir: ' + config.containerbaseDir);
        }
        await fs_extra_1.default.ensureDir(config.containerbaseDir);
    }
    return config;
}
function limitCommitsPerRun(config) {
    let limit = config.prCommitsPerRunLimit;
    limit = typeof limit === 'number' && limit > 0 ? limit : null;
    (0, limits_1.setMaxLimit)('Commits', limit);
}
async function checkVersions() {
    const validGitVersion = await (0, git_1.validateGitVersion)();
    if (!validGitVersion) {
        throw new Error('Init: git version needs upgrading');
    }
}
function setGlobalHostRules(config) {
    if (config.hostRules) {
        logger_1.logger.debug('Setting global hostRules');
        (0, secrets_1.applySecretsAndVariablesToConfig)({
            config,
            deleteVariables: false,
            deleteSecrets: false,
        });
        config.hostRules.forEach((rule) => hostRules.add(rule));
    }
}
function configureThirdPartyLibraries(config) {
    // Not using early return style to make clear what's the criterion to set the variables,
    // especially when there is more stuff added here in the future.
    if (!config.useCloudMetadataServices) {
        logger_1.logger.debug('Disabling the use of cloud metadata services');
        process.env.AWS_EC2_METADATA_DISABLED = 'true'; // See https://docs.aws.amazon.com/cli/latest/userguide/cli-configure-envvars.html#envvars-list
        process.env.METADATA_SERVER_DETECTION = 'none'; // See https://cloud.google.com/nodejs/docs/reference/gcp-metadata/latest#environment-variables
    }
}
async function globalInitialize(config_) {
    let config = config_;
    (0, rate_limits_1.setHttpRateLimits)();
    await checkVersions();
    setGlobalHostRules(config);
    config = await (0, platform_1.initPlatform)(config);
    config = await setDirectories(config);
    await packageCache.init(config);
    limitCommitsPerRun(config);
    (0, emoji_1.setEmojiConfig)(config);
    setGlobalHostRules(config);
    configureThirdPartyLibraries(config);
    await (0, merge_confidence_1.initMergeConfidence)(config);
    return config;
}
async function globalFinalize(config) {
    await packageCache.cleanup(config);
    (0, remap_1.resetGlobalLogLevelRemaps)();
}
//# sourceMappingURL=initialize.js.map