"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getRepositoryConfig = getRepositoryConfig;
exports.validatePresets = validatePresets;
exports.start = start;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const bunyan_1 = require("bunyan");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const upath_1 = tslib_1.__importDefault(require("upath"));
const configParser = tslib_1.__importStar(require("../../config"));
const global_1 = require("../../config/global");
const presets_1 = require("../../config/presets");
const secrets_1 = require("../../config/secrets");
const error_messages_1 = require("../../constants/error-messages");
const expose_cjs_1 = require("../../expose.cjs");
const instrumentation_1 = require("../../instrumentation");
const reporting_1 = require("../../instrumentation/reporting");
const logger_1 = require("../../logger");
const remap_1 = require("../../logger/remap");
const env_1 = require("../../util/env");
const hostRules = tslib_1.__importStar(require("../../util/host-rules"));
const queue = tslib_1.__importStar(require("../../util/http/queue"));
const throttle = tslib_1.__importStar(require("../../util/http/throttle"));
const regex_1 = require("../../util/regex");
const sanitize_1 = require("../../util/sanitize");
const repositoryWorker = tslib_1.__importStar(require("../repository"));
const autodiscover_1 = require("./autodiscover");
const parse_1 = require("./config/parse");
const initialize_1 = require("./initialize");
const limits_1 = require("./limits");
async function getRepositoryConfig(globalConfig, repository) {
    const repoConfig = configParser.mergeChildConfig(globalConfig, is_1.default.string(repository) ? { repository } : repository);
    const repoParts = repoConfig.repository.split('/');
    repoParts.pop();
    repoConfig.parentOrg = repoParts.join('/');
    repoConfig.topLevelOrg = repoParts.shift();
    // TODO: types (#22198)
    const platform = global_1.GlobalConfig.get('platform');
    repoConfig.localDir =
        platform === 'local'
            ? process.cwd()
            : upath_1.default.join(repoConfig.baseDir, `./repos/${platform}/${repoConfig.repository}`);
    await fs_extra_1.default.ensureDir(repoConfig.localDir);
    delete repoConfig.baseDir;
    return configParser.filterConfig(repoConfig, 'repository');
}
function getGlobalConfig() {
    return (0, parse_1.parseConfigs)((0, env_1.getEnv)(), process.argv);
}
function haveReachedLimits() {
    if ((0, limits_1.isLimitReached)('Commits')) {
        logger_1.logger.info('Max commits created for this run.');
        return true;
    }
    return false;
}
/* istanbul ignore next */
function checkEnv() {
    const range = expose_cjs_1.pkg.engines.node;
    if (process.release?.name !== 'node' || !process.versions?.node) {
        logger_1.logger.warn({ release: process.release, versions: process.versions }, 'Unknown node environment detected.');
    }
    else if (!semver_1.default.satisfies(process.versions?.node, range)) {
        logger_1.logger.error({ versions: process.versions, range }, 'Unsupported node environment detected. Please update your node version.');
    }
}
async function validatePresets(config) {
    logger_1.logger.debug('validatePresets()');
    try {
        await (0, presets_1.resolveConfigPresets)(config);
    }
    catch (err) /* istanbul ignore next */ {
        logger_1.logger.error({ err }, error_messages_1.CONFIG_PRESETS_INVALID);
        throw new Error(error_messages_1.CONFIG_PRESETS_INVALID);
    }
}
async function start() {
    // istanbul ignore next
    if (regex_1.regexEngineStatus.type === 'available') {
        logger_1.logger.debug('Using RE2 regex engine');
    }
    else if (regex_1.regexEngineStatus.type === 'unavailable') {
        logger_1.logger.warn({ err: regex_1.regexEngineStatus.err }, 'RE2 not usable, falling back to RegExp');
    }
    else if (regex_1.regexEngineStatus.type === 'ignored') {
        logger_1.logger.debug('RE2 regex engine is ignored via RENOVATE_X_IGNORE_RE2');
    }
    let config;
    const env = (0, env_1.getEnv)();
    try {
        if (is_1.default.nonEmptyStringAndNotWhitespace(env.AWS_SECRET_ACCESS_KEY)) {
            (0, sanitize_1.addSecretForSanitizing)(env.AWS_SECRET_ACCESS_KEY, 'global');
        }
        if (is_1.default.nonEmptyStringAndNotWhitespace(env.AWS_SESSION_TOKEN)) {
            (0, sanitize_1.addSecretForSanitizing)(env.AWS_SESSION_TOKEN, 'global');
        }
        await (0, instrumentation_1.instrument)('config', async () => {
            // read global config from file, env and cli args
            config = await getGlobalConfig();
            // Set allowedHeaders and userAgent in case hostRules headers are configured in file config
            global_1.GlobalConfig.set({
                allowedHeaders: config.allowedHeaders,
                userAgent: config.userAgent,
            });
            // initialize all submodules
            config = await (0, initialize_1.globalInitialize)(config);
            // Set platform, endpoint, allowedHeaders and userAgent in case local presets are used
            global_1.GlobalConfig.set({
                allowedHeaders: config.allowedHeaders,
                platform: config.platform,
                endpoint: config.endpoint,
                userAgent: config.userAgent,
            });
            await validatePresets(config);
            (0, remap_1.setGlobalLogLevelRemaps)(config.logLevelRemap);
            checkEnv();
            // validate secrets and variables. Will throw and abort if invalid
            (0, secrets_1.validateConfigSecretsAndVariables)(config);
        });
        // autodiscover repositories (needs to come after platform initialization)
        config = await (0, instrumentation_1.instrument)('discover', () => (0, autodiscover_1.autodiscoverRepositories)(config));
        if (is_1.default.nonEmptyString(config.writeDiscoveredRepos)) {
            const content = JSON.stringify(config.repositories);
            await fs_extra_1.default.writeFile(config.writeDiscoveredRepos, content);
            logger_1.logger.info(`Written discovered repositories to ${config.writeDiscoveredRepos}`);
            return 0;
        }
        // Iterate through repositories sequentially
        for (const repository of config.repositories) {
            if (haveReachedLimits()) {
                break;
            }
            await (0, instrumentation_1.instrument)('repository', async () => {
                const repoConfig = await getRepositoryConfig(config, repository);
                if (repoConfig.hostRules) {
                    logger_1.logger.debug('Reinitializing hostRules for repo');
                    hostRules.clear();
                    repoConfig.hostRules.forEach((rule) => hostRules.add(rule));
                    repoConfig.hostRules = [];
                }
                // host rules can change concurrency
                queue.clear();
                throttle.clear();
                await repositoryWorker.renovateRepository(repoConfig);
                (0, logger_1.setMeta)({});
            }, {
                attributes: {
                    repository: typeof repository === 'string'
                        ? repository
                        : repository.repository,
                },
            });
        }
        (0, reporting_1.finalizeReport)();
        await (0, reporting_1.exportStats)(config);
    }
    catch (err) /* istanbul ignore next */ {
        if (err.message.startsWith('Init: ')) {
            logger_1.logger.fatal({ errorMessage: err.message.substring(6) }, 'Initialization error');
        }
        else {
            logger_1.logger.fatal({ err }, 'Unknown error');
        }
        if (!config) {
            // return early if we can't parse config options
            logger_1.logger.debug(`Missing config`);
            return 2;
        }
    }
    finally {
        await (0, initialize_1.globalFinalize)(config);
        if ((0, logger_1.logLevel)() === 'info') {
            logger_1.logger.info(`Renovate was run at log level "${(0, logger_1.logLevel)()}". Set LOG_LEVEL=debug in environment variables to see extended debug logs.`);
        }
    }
    const loggerErrors = (0, logger_1.getProblems)().filter((p) => p.level >= bunyan_1.ERROR);
    if (loggerErrors.length) {
        logger_1.logger.info({ loggerErrors }, 'Renovate is exiting with a non-zero code due to the following logged errors');
        return 1;
    }
    return 0;
}
//# sourceMappingURL=index.js.map