"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveGlobalExtends = resolveGlobalExtends;
exports.parseConfigs = parseConfigs;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const decrypt_1 = require("../../../../config/decrypt");
const defaultsParser = tslib_1.__importStar(require("../../../../config/defaults"));
const presets_1 = require("../../../../config/presets");
const secrets_1 = require("../../../../config/secrets");
const utils_1 = require("../../../../config/utils");
const error_messages_1 = require("../../../../constants/error-messages");
const logger_1 = require("../../../../logger");
const manager_1 = require("../../../../modules/manager");
const array_1 = require("../../../../util/array");
const env_1 = require("../../../../util/env");
const fs_1 = require("../../../../util/fs");
const sanitize_1 = require("../../../../util/sanitize");
const url_1 = require("../../../../util/url");
const cliParser = tslib_1.__importStar(require("./cli"));
const codespaces = tslib_1.__importStar(require("./codespaces"));
const envParser = tslib_1.__importStar(require("./env"));
const fileParser = tslib_1.__importStar(require("./file"));
const host_rules_from_env_1 = require("./host-rules-from-env");
async function resolveGlobalExtends(globalExtends, ignorePresets) {
    try {
        // Make a "fake" config to pass to resolveConfigPresets and resolve globalPresets
        const config = { extends: globalExtends, ignorePresets };
        const resolvedConfig = await (0, presets_1.resolveConfigPresets)(config);
        return resolvedConfig;
    }
    catch (err) {
        logger_1.logger.error({ err }, 'Error resolving config preset');
        throw new Error(error_messages_1.CONFIG_PRESETS_INVALID);
    }
}
async function parseConfigs(env, argv) {
    logger_1.logger.debug('Parsing configs');
    // Get configs
    const defaultConfig = defaultsParser.getConfig();
    const fileConfig = await fileParser.getConfig(env);
    const cliConfig = cliParser.getConfig(argv);
    const envConfig = await envParser.getConfig(env);
    let config = (0, utils_1.mergeChildConfig)(fileConfig, envConfig);
    config = (0, utils_1.mergeChildConfig)(config, cliConfig);
    config = await codespaces.setConfig(config);
    let resolvedGlobalExtends;
    if (is_1.default.nonEmptyArray(config?.globalExtends)) {
        // resolve global presets immediately
        resolvedGlobalExtends = await resolveGlobalExtends(config.globalExtends, config.ignorePresets);
        config = (0, utils_1.mergeChildConfig)(resolvedGlobalExtends, config);
        delete config.globalExtends;
    }
    const combinedConfig = config;
    config = (0, utils_1.mergeChildConfig)(defaultConfig, config);
    if (config.forceCli) {
        const forcedCli = { ...cliConfig };
        delete forcedCli.token;
        delete forcedCli.hostRules;
        if (config.force) {
            config.force = { ...config.force, ...forcedCli };
        }
        else {
            config.force = forcedCli;
        }
    }
    if (!config.privateKey && config.privateKeyPath) {
        config.privateKey = await (0, fs_1.readSystemFile)(config.privateKeyPath, 'utf8');
        delete config.privateKeyPath;
    }
    if (!config.privateKeyOld && config.privateKeyPathOld) {
        config.privateKeyOld = await (0, fs_1.readSystemFile)(config.privateKeyPathOld, 'utf8');
        delete config.privateKeyPathOld;
    }
    // Add private keys for sanitizing then set and delete them
    (0, sanitize_1.addSecretForSanitizing)(config.privateKey, 'global');
    (0, sanitize_1.addSecretForSanitizing)(config.privateKeyOld, 'global');
    (0, decrypt_1.setPrivateKeys)(config.privateKey, config.privateKeyOld);
    delete config.privateKey;
    delete config.privateKeyOld;
    if (config.logContext) {
        // This only has an effect if logContext was defined via file or CLI, otherwise it would already have been detected in env
        (0, logger_1.setContext)(config.logContext);
    }
    logger_1.logger.trace({ config: defaultConfig }, 'Default config');
    logger_1.logger.debug({ config: fileConfig }, 'File config');
    logger_1.logger.debug({ config: cliConfig }, 'CLI config');
    logger_1.logger.debug({ config: envConfig }, 'Env config');
    logger_1.logger.debug({ config: resolvedGlobalExtends }, 'Resolved global extends');
    logger_1.logger.debug({ config: combinedConfig }, 'Combined config');
    if (config.detectGlobalManagerConfig) {
        logger_1.logger.debug('Detecting global manager config');
        const globalManagerConfig = await (0, manager_1.detectAllGlobalConfig)();
        logger_1.logger.debug({ config: globalManagerConfig }, 'Global manager config');
        config = (0, utils_1.mergeChildConfig)(config, globalManagerConfig);
    }
    if (config.detectHostRulesFromEnv) {
        const hostRules = (0, host_rules_from_env_1.hostRulesFromEnv)(env);
        config.hostRules = [...(0, array_1.coerceArray)(config.hostRules), ...hostRules];
    }
    // Get global config
    logger_1.logger.trace({ config }, 'Full config');
    // Massage endpoint to have a trailing slash
    if (config.endpoint) {
        logger_1.logger.debug('Adding trailing slash to endpoint');
        config.endpoint = (0, url_1.ensureTrailingSlash)(config.endpoint);
    }
    // Massage forkProcessing
    if (!config.autodiscover && config.forkProcessing !== 'disabled') {
        logger_1.logger.debug('Enabling forkProcessing while in non-autodiscover mode');
        config.forkProcessing = 'enabled';
    }
    // Only try deletion if RENOVATE_CONFIG_FILE is set
    await fileParser.deleteNonDefaultConfig(env, !!config.deleteConfigFile);
    // Massage onboardingNoDeps
    if (!config.autodiscover && config.onboardingNoDeps !== 'disabled') {
        logger_1.logger.debug('Enabling onboardingNoDeps while in non-autodiscover mode');
        config.onboardingNoDeps = 'enabled';
    }
    // do not add these secrets to repoSecrets and,
    //  do not delete the secrets/variables object after applying on global config as it needs to be re-used for repo config
    if (is_1.default.nonEmptyObject(config.secrets) ||
        is_1.default.nonEmptyObject(config.variables)) {
        config = (0, secrets_1.applySecretsAndVariablesToConfig)({
            config,
            secrets: config.secrets,
            variables: config.variables,
            deleteSecrets: false,
            deleteVariables: false,
        });
        // adding these secrets to the globalSecrets set so that they can be redacted from logs
        for (const secret of Object.values(config.secrets)) {
            (0, sanitize_1.addSecretForSanitizing)(secret, 'global');
        }
    }
    if (is_1.default.nonEmptyObject(config.customEnvVariables)) {
        (0, env_1.setCustomEnv)(config.customEnvVariables);
    }
    return config;
}
//# sourceMappingURL=index.js.map