"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.autodiscoverRepositories = autodiscoverRepositories;
exports.applyFilters = applyFilters;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../logger");
const platform_1 = require("../../modules/platform");
const string_match_1 = require("../../util/string-match");
// istanbul ignore next
function repoName(value) {
    return String(is_1.default.string(value) ? value : value.repository).toLowerCase();
}
async function autodiscoverRepositories(config) {
    const { autodiscoverFilter } = config;
    if (config.platform === 'local') {
        if (config.repositories?.length) {
            logger_1.logger.debug({ repositories: config.repositories }, 'Found repositories when in local mode');
            throw new Error('Invalid configuration: repositories list not supported when platform=local');
        }
        config.repositories = ['local'];
        return config;
    }
    if (!config.autodiscover) {
        if (!config.repositories?.length) {
            logger_1.logger.warn('No repositories found - did you want to run with flag --autodiscover?');
        }
        return config;
    }
    // Autodiscover list of repositories
    let discovered = await platform_1.platform.getRepos({
        topics: config.autodiscoverTopics,
        sort: config.autodiscoverRepoSort,
        order: config.autodiscoverRepoOrder,
        includeMirrors: config.includeMirrors,
        namespaces: config.autodiscoverNamespaces,
        projects: config.autodiscoverProjects,
    });
    if (!discovered?.length) {
        // Soft fail (no error thrown) if no accessible repositories
        logger_1.logger.debug('No repositories were autodiscovered');
        return config;
    }
    logger_1.logger.debug(`Autodiscovered ${discovered.length} repositories`);
    logger_1.logger.trace({ length: discovered.length, repositories: discovered }, `Autodiscovered repositories`);
    if (autodiscoverFilter) {
        logger_1.logger.debug({ autodiscoverFilter }, 'Applying autodiscoverFilter');
        discovered = applyFilters(discovered, is_1.default.string(autodiscoverFilter) ? [autodiscoverFilter] : autodiscoverFilter);
        if (!discovered.length) {
            // Soft fail (no error thrown) if no accessible repositories match the filter
            logger_1.logger.debug('None of the discovered repositories matched the filter');
            return config;
        }
    }
    logger_1.logger.info({ length: discovered.length, repositories: discovered }, `Autodiscovered repositories`);
    // istanbul ignore if
    if (config.repositories?.length) {
        logger_1.logger.debug('Checking autodiscovered repositories against configured repositories');
        for (const configuredRepo of config.repositories) {
            const repository = repoName(configuredRepo);
            let found = false;
            for (let i = discovered.length - 1; i > -1; i -= 1) {
                if (repository === repoName(discovered[i])) {
                    found = true;
                    logger_1.logger.debug({ repository }, 'Using configured repository settings');
                    // TODO: fix typings
                    discovered[i] = configuredRepo;
                }
            }
            if (!found) {
                logger_1.logger.warn({ repository }, 'Configured repository is in not in autodiscover list');
            }
        }
    }
    return { ...config, repositories: discovered };
}
function applyFilters(repos, filters) {
    return repos.filter((repo) => (0, string_match_1.matchRegexOrGlobList)(repo, filters));
}
//# sourceMappingURL=autodiscover.js.map