"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MaybeTimestamp = exports.Timestamp = void 0;
exports.asTimestamp = asTimestamp;
const luxon_1 = require("luxon");
const zod_1 = require("zod");
const timezoneOffset = new Date().getTimezoneOffset() * 60000;
const millenium = 946684800000; // 2000-01-01T00:00:00.000Z
const tomorrowOffset = 86400000; // 24 * 60 * 60 * 1000;
function isValid(date) {
    if (!date.isValid) {
        return false;
    }
    const tomorrow = luxon_1.DateTime.now().toMillis() + tomorrowOffset; // 24 * 60 * 60 * 1000;
    const ts = date.toMillis();
    return ts > millenium && ts < tomorrow;
}
function asTimestamp(input) {
    if (input instanceof Date) {
        const date = luxon_1.DateTime.fromJSDate(input, { zone: 'UTC' });
        if (isValid(date)) {
            return date.toISO();
        }
        return null;
    }
    if (typeof input === 'number') {
        const millisDate = luxon_1.DateTime.fromMillis(input, { zone: 'UTC' });
        if (isValid(millisDate)) {
            return millisDate.toISO();
        }
        const secondsDate = luxon_1.DateTime.fromSeconds(input, { zone: 'UTC' });
        if (isValid(secondsDate)) {
            return secondsDate.toISO();
        }
        return null;
    }
    if (typeof input === 'string') {
        const isoDate = luxon_1.DateTime.fromISO(input, { zone: 'UTC' });
        if (isValid(isoDate)) {
            return isoDate.toISO();
        }
        const httpDate = luxon_1.DateTime.fromHTTP(input, { zone: 'UTC' });
        if (isValid(httpDate)) {
            return httpDate.toISO();
        }
        const sqlDate = luxon_1.DateTime.fromSQL(input, { zone: 'UTC' });
        if (isValid(sqlDate)) {
            return sqlDate.toISO();
        }
        const numberLikeDate = luxon_1.DateTime.fromFormat(input, 'yyyyMMddHHmmss', {
            zone: 'UTC',
        });
        if (isValid(numberLikeDate)) {
            return numberLikeDate.toISO();
        }
        const numberLikeOffsetDate = luxon_1.DateTime.fromFormat(input, 'yyyyMMddHHmmssZZZ', { zone: 'UTC' });
        if (isValid(numberLikeOffsetDate)) {
            return numberLikeOffsetDate.toISO();
        }
        const fallbackDate = luxon_1.DateTime.fromMillis(Date.parse(input) - timezoneOffset, { zone: 'UTC' });
        if (isValid(fallbackDate)) {
            return fallbackDate.toISO();
        }
        return null;
    }
    return null;
}
exports.Timestamp = zod_1.z.unknown().transform((input, ctx) => {
    const timestamp = asTimestamp(input);
    if (!timestamp) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: 'Invalid timestamp',
        });
        return zod_1.z.NEVER;
    }
    return timestamp;
});
exports.MaybeTimestamp = exports.Timestamp.nullable().catch(null);
//# sourceMappingURL=timestamp.js.map