"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateInterpolatedValues = validateInterpolatedValues;
exports.replaceInterpolatedValuesInObject = replaceInterpolatedValuesInObject;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const error_messages_1 = require("../constants/error-messages");
const logger_1 = require("../logger");
const string_1 = require("./string");
function validateInterpolatedValues(input, options) {
    if (!input) {
        return;
    }
    const { name, nameRegex } = options;
    const validationErrors = [];
    if (is_1.default.plainObject(input)) {
        for (const [key, value] of Object.entries(input)) {
            if (!nameRegex.test(key)) {
                validationErrors.push(`Invalid ${name} name "${key}"`);
            }
            if (!is_1.default.string(value)) {
                validationErrors.push(`${(0, string_1.capitalize)(name)} values must be strings. Found type ${typeof value} for ${name} ${key}`);
            }
        }
    }
    else {
        validationErrors.push(`Config ${name}s must be a plain object. Found: ${typeof input}`);
    }
    if (validationErrors.length) {
        logger_1.logger.error({ validationErrors }, `Invalid ${name}s configured`);
        throw new Error(name === 'secrets' ? error_messages_1.CONFIG_SECRETS_INVALID : error_messages_1.CONFIG_VARIABLES_INVALID);
    }
}
function replaceInterpolatedValuesInString(key, value, input, options) {
    const { name, templateRegex } = options;
    // do nothing if no interpolator template found
    if (!templateRegex.test(value)) {
        return value;
    }
    const disallowedPrefixes = ['branch', 'commit', 'group', 'pr', 'semantic'];
    if (disallowedPrefixes.some((prefix) => key.startsWith(prefix))) {
        const error = new Error(error_messages_1.CONFIG_VALIDATION);
        error.validationSource = 'config';
        error.validationError = `Disallowed ${name} substitution`;
        error.validationMessage = `The field \`${key}\` may not use ${name} substitution`;
        throw error;
    }
    return value.replace(templateRegex, (_, key) => {
        if (input?.[key]) {
            return input[key];
        }
        const error = new Error(error_messages_1.CONFIG_VALIDATION);
        error.validationSource = 'config';
        error.validationError = `Unknown ${name} name`;
        error.validationMessage = `The following ${name} name was not found in config: ${String(key)}`;
        throw error;
    });
}
function replaceInterpolatedValuesInObject(config_, input, options, deleteValues = true) {
    const config = { ...config_ };
    const { name } = options;
    if (deleteValues) {
        delete config[name];
    }
    for (const [key, value] of Object.entries(config)) {
        if (is_1.default.plainObject(value)) {
            config[key] = replaceInterpolatedValuesInObject(value, input, options, deleteValues);
        }
        if (is_1.default.string(value)) {
            config[key] = replaceInterpolatedValuesInString(key, value, input, options);
        }
        if (is_1.default.array(value)) {
            for (const [arrayIndex, arrayItem] of value.entries()) {
                if (is_1.default.plainObject(arrayItem)) {
                    value[arrayIndex] = replaceInterpolatedValuesInObject(arrayItem, input, options, deleteValues);
                }
                else if (is_1.default.string(arrayItem)) {
                    value[arrayIndex] = replaceInterpolatedValuesInString(key, arrayItem, input, options);
                }
            }
        }
    }
    return config;
}
//# sourceMappingURL=interpolator.js.map