"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabHttp = exports.setBaseUrl = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const got_1 = require("got");
const logger_1 = require("../../logger");
const external_host_error_1 = require("../../types/errors/external-host-error");
const env_1 = require("../env");
const url_1 = require("../url");
const http_1 = require("./http");
let baseUrl = 'https://gitlab.com/api/v4/';
const setBaseUrl = (url) => {
    baseUrl = url;
};
exports.setBaseUrl = setBaseUrl;
class GitlabHttp extends http_1.HttpBase {
    get baseUrl() {
        return baseUrl;
    }
    constructor(type = 'gitlab', options) {
        super(type, options);
    }
    async requestJsonUnsafe(method, options) {
        const resolvedUrl = this.resolveUrl(options.url, options.httpOptions);
        const opts = {
            ...options,
            url: resolvedUrl,
        };
        opts.httpOptions ??= {};
        opts.httpOptions.throwHttpErrors = true;
        const result = await super.requestJsonUnsafe(method, opts);
        if (opts.httpOptions.paginate && is_1.default.array(result.body)) {
            delete opts.httpOptions.cacheProvider;
            opts.httpOptions.memCache = false;
            // Check if result is paginated
            try {
                const linkHeader = (0, url_1.parseLinkHeader)(result.headers.link);
                const nextUrl = linkHeader?.next?.url
                    ? (0, url_1.parseUrl)(linkHeader.next.url)
                    : null;
                if (nextUrl) {
                    if ((0, env_1.getEnv)().GITLAB_IGNORE_REPO_URL) {
                        const defaultEndpoint = new URL(baseUrl);
                        nextUrl.protocol = defaultEndpoint.protocol;
                        nextUrl.host = defaultEndpoint.host;
                    }
                    opts.url = nextUrl;
                    const nextResult = await this.requestJsonUnsafe(method, opts);
                    if (is_1.default.array(nextResult.body)) {
                        result.body.push(...nextResult.body);
                    }
                }
            }
            catch (err) {
                logger_1.logger.warn({ err }, 'Pagination error');
            }
        }
        return result;
    }
    handleError(url, _httpOptions, err) {
        if (err instanceof got_1.RequestError && err.response?.statusCode) {
            if (err.response.statusCode === 404) {
                logger_1.logger.trace({ err }, 'GitLab 404');
                logger_1.logger.debug({ url }, 'GitLab API 404');
                throw err;
            }
            logger_1.logger.debug({ err }, 'Gitlab API error');
            if (err.response.statusCode === 429 ||
                (err.response.statusCode >= 500 && err.response.statusCode < 600)) {
                throw new external_host_error_1.ExternalHostError(err, 'gitlab');
            }
        }
        const platformFailureCodes = [
            'EAI_AGAIN',
            'ECONNRESET',
            'ETIMEDOUT',
            'UNABLE_TO_VERIFY_LEAF_SIGNATURE',
        ];
        // TODO: fix test, should be `RequestError`
        if ('code' in err &&
            is_1.default.string(err.code) &&
            platformFailureCodes.includes(err.code)) {
            throw new external_host_error_1.ExternalHostError(err, 'gitlab');
        }
        if (err.name === 'ParseError') {
            throw new external_host_error_1.ExternalHostError(err, 'gitlab');
        }
        throw err;
    }
    calculateRetryDelay(retryObject) {
        const { error, attemptCount, retryOptions } = retryObject;
        if (attemptCount <= retryOptions.limit &&
            error.options.method === 'POST' &&
            error.response?.statusCode === 409 &&
            error.response.rawBody.toString().includes('Resource lock')) {
            const noise = Math.random() * 100;
            return 2 ** (attemptCount - 1) * 1000 + noise;
        }
        return super.calculateRetryDelay(retryObject);
    }
}
exports.GitlabHttp = GitlabHttp;
//# sourceMappingURL=gitlab.js.map