import { DateTime } from 'luxon';
import type { PackageCacheNamespace } from '../../../cache/package/types';
import type { GithubDatasourceItem, GithubGraphqlCacheRecord, GithubGraphqlCacheStrategy } from '../types';
/**
 * Cache strategy handles the caching Github GraphQL items
 * and reconciling them with newly obtained ones from paginated queries.
 */
export declare abstract class AbstractGithubGraphqlCacheStrategy<GithubItem extends GithubDatasourceItem> implements GithubGraphqlCacheStrategy<GithubItem> {
    protected readonly cacheNs: PackageCacheNamespace;
    protected readonly cacheKey: string;
    /**
     * Time period after which a cache record is considered expired.
     */
    protected static readonly cacheTTLDays = 30;
    /**
     * The time which is used during single cache access cycle.
     */
    protected readonly now: DateTime<true>;
    /**
     * Set of all versions which were reconciled
     * during the current cache access cycle.
     */
    private reconciledVersions;
    /**
     * These fields will be persisted.
     */
    private items;
    protected createdAt: DateTime;
    /**
     * This flag indicates whether there is any new or updated items
     */
    protected hasNovelty: boolean;
    /**
     * Loading and persisting data is delegated to the concrete strategy.
     */
    abstract load(): Promise<GithubGraphqlCacheRecord<GithubItem> | undefined>;
    abstract persist(cacheRecord: GithubGraphqlCacheRecord<GithubItem>): Promise<void>;
    constructor(cacheNs: PackageCacheNamespace, cacheKey: string);
    /**
     * Load data previously persisted by this strategy
     * for given `cacheNs` and `cacheKey`.
     */
    private getItems;
    /**
     * If package release exists longer than this cache can exist,
     * we assume it won't updated/removed on the Github side.
     */
    private isStabilized;
    /**
     * Process items received from GraphQL page
     * ordered by `releaseTimestamp` in descending order
     * (fresh versions go first).
     */
    reconcile(items: GithubItem[]): Promise<boolean>;
    /**
     * Handle removed items for packages that are not stabilized
     * and return the list of all items.
     */
    finalizeAndReturn(): Promise<GithubItem[]>;
    private store;
}
